/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimadsp.c 846 2009-03-28 09:40:49Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimadsp.c 846 2009-03-28 09:40:49Z takahiko $");

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <strings.h>
#include <stdbool.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "ptrop.h"
#include "dkimlogger.h"
#include "inetdomain.h"
#include "dnsresolv.h"
#include "dkim.h"
#include "dkimspec.h"
#include "dkimenum.h"
#include "dkimtvobj.h"
#include "dkimparse.h"
#include "dkimadsp.h"

#ifndef MIN
#define MIN(a,b)    ((a) < (b) ? (a) : (b))
#endif

struct DkimAdsp {
    DkimTvobj_MEMBER;
    dkim_adsp_signpractice_t signing_practice;  // adsp-dkim-tag

    bool parsed_dkim;
};

static dkim_stat_t DkimAdsp_parse_dkim(DkimTvobj *base, const DkimTagParseContext *context,
                                       const char **nextp);

static const DkimTvobjFieldMap dkim_adsp_field_tbl[] = {
    {"dkim", DkimAdsp_parse_dkim, true, NULL, offsetof(DkimAdsp, parsed_dkim)},
    {NULL, NULL, false, NULL, 0},   // sentinel
};

/*
 * [draft-ietf-dkim-ssp-09] 4.2.1.
 * adsp-dkim-tag = %x64.6b.69.6d *WSP "=" *WSP
 *                 ("unknown" / "all" / "discardable")
 * @error DSTAT_OK
 * @error DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION
 */
dkim_stat_t
DkimAdsp_parse_dkim(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimAdsp *self = (DkimAdsp *) base;

    if (0 != context->tagno) {  // a "valid ADSP record" must starts with a valid "dkim" tag
        *nextp = context->valuehead;
        DkimLogPermFail(base->policy,
                        "adsp-dkim-tag appeared not at the front of ADSP record: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->signing_practice =
        DkimEnum_lookupSignPracticeByNameSlice(context->valuehead, context->valuetail);
    if (DKIM_ADSP_SIGNPRACTICE_NULL == self->signing_practice) {
        /*
         * [draft-ietf-dkim-ssp-09] 4.2.1.
         * Any other values are treated as "unknown".
         */
        DkimLogInfo(base->policy,
                    "unsupported outbound signing practice (treated as \"unknown\"): dkim=%.*s",
                    (int) (context->valuetail - context->valuehead), context->valuehead);
        self->signing_practice = DKIM_ADSP_SIGNPRACTICE_UNKNOWN;
    }   // end if
    *nextp = context->valuetail;
    return DSTAT_OK;
}   // end function : DkimAdsp_parse_dkim

////////////////////////////////////////////////////////////////////////

/**
 * @param policy
 * @param keyval
 * @param dstat
 * @error DSTAT_OK 成功
 * @error DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION tag=value ペアの構造に文法エラーがある
 * @error DSTAT_PERMFAIL_MISSING_REQUIRED_TAG 必須タグがレコード中で指定されていない
 * @error DSTAT_PERMFAIL_TAG_DUPLICATED 既にパース済みのタグに遭遇した
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 */
DkimAdsp *
DkimAdsp_build(const DkimPolicy *policy, const char *keyval, dkim_stat_t *dstat)
{
    assert(NULL != keyval);

    DkimAdsp *self = (DkimAdsp *) malloc(sizeof(DkimAdsp));
    if (NULL == self) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimAdsp));
    self->policy = policy;
    self->ftbl = dkim_adsp_field_tbl;

    /*
     * [draft-ietf-dkim-ssp-09] 4.1.
     * Note:   ADSP changes the "Tag=Value List" syntax from [RFC4871] to
     *    use WSP rather than FWS in its DNS records.  Domains MUST NOT
     *    publish ADSP records with wildcard names.  Wildcards within a
     *    domain publishing ADSP records pose a particular problem, as
     *    discussed in more detail in Section 6.3.
     */
    dkim_stat_t build_stat = DkimTvobj_build((DkimTvobj *) self, keyval, STRTAIL(keyval), true);
    if (DSTAT_OK != build_stat) {
        SETDEREF(dstat, build_stat);
        DkimAdsp_free(self);
        return NULL;
    }   // end if

    SETDEREF(dstat, DSTAT_OK);
    return self;
}   // end function : DkimAdsp_build

void
DkimAdsp_free(DkimAdsp *self)
{
    assert(NULL != self);
    free(self);
}   // end function : DkimAdsp_free

/**
 * @error DSTAT_OK 適切な ADSP レコードが見つかった.
 * @error DSTAT_INFO_ADSP_NOT_EXIST ADSP レコードは見つからなかった, またはレコードの文法が ADSP レコードに合致しなかった.
 * @error DSTAT_PERMFAIL_MULTIPLE_ADSP_RECORD ADSP レコードが複数存在した
 * @error DSTAT_TMPERR_DNS_LOOKUP_FAILURE DNS エラーが発生した.
 * @error DSTAT_SYSERR_IMPLERROR デフォルト値として設定されている値がパースできない
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー
 */
static DkimAdsp *
DkimAdsp_query(const DkimPolicy *policy, DnsResolver *resolver, const char *domain,
               dkim_stat_t *dstat)
{
    assert(NULL != resolver);
    assert(NULL != domain);

    // ADSP レコードを探す
    DnsTxtResponse *txt_rr = NULL;
    dns_stat_t txtquery_stat = DnsResolver_lookupTxt(resolver, domain, &txt_rr);
    switch (txtquery_stat) {
    case DNS_STAT_NOERROR:;
        // a TXT RR is found
        /*
         * [draft-ietf-dkim-ssp-09] 4.3.
         * If the result of this query is a "NOERROR" response (rcode=0 in
         * [RFC1035]) with an answer which is a single record that is a valid
         * ADSP record, use that record, and the algorithm terminates.
         *
         * If the result of the query is NXDOMAIN or NOERROR with zero
         * records, there is no ADSP record.  If the result of the query
         * contains more than one record, or a record that is not a valid
         * ADSP record, the ADSP result is undefined.
         */
        if (0 == DnsTxtResponse_size(txt_rr)) {
            // TXT レコードがなかった場合
            DnsTxtResponse_free(txt_rr);
            SETDEREF(dstat, DSTAT_INFO_ADSP_NOT_EXIST);
            break;
        } else if (1 < DnsTxtResponse_size(txt_rr)) {
            DnsTxtResponse_free(txt_rr);
            SETDEREF(dstat, DSTAT_PERMFAIL_MULTIPLE_ADSP_RECORD);
            break;
        }   // end if

        dkim_stat_t build_stat;
        const char *txtrecord = DnsTxtResponse_data(txt_rr, 0);
        DkimAdsp *self = DkimAdsp_build(policy, txtrecord, &build_stat);
        if (NULL != self) {
            // parsed as a valid ADSP record
            DnsTxtResponse_free(txt_rr);
            SETDEREF(dstat, DSTAT_OK);
            return self;
        } else if (DSTAT_ISCRITERR(build_stat)) {
            // システムエラーはそのまま伝播させる
            DkimLogSysError
                (policy,
                 "System error has occurred while parsing ADSP record: domain=%s, err=%s, record=%s",
                 domain, DKIM_strerror(build_stat), NNSTR(txtrecord));
            SETDEREF(dstat, build_stat);
        } else if (DSTAT_ISPERMFAIL(build_stat)) {
            /*
             * ADSP レコードに文法エラーがある場合は NODATA と同じ扱い
             * [draft-ietf-dkim-ssp-09] 4.1.
             * Records not
             * in compliance with that syntax or the syntax of individual tags
             * described in Section 4.3 MUST be ignored (considered equivalent to a
             * NODATA result) for purposes of ADSP, although they MAY cause the
             * logging of warning messages via an appropriate system logging
             * mechanism.
             */
            DkimLogDebug(policy, "ADSP record candidate discarded: domain=%s, err=%s, record=%s",
                         domain, DKIM_strerror(build_stat), NNSTR(txtrecord));
            SETDEREF(dstat, DSTAT_INFO_ADSP_NOT_EXIST);
        } else {
            DkimLogNotice(policy, "DkimAdsp_build failed: domain=%s, err=%s, record=%s",
                          domain, DKIM_strerror(build_stat), NNSTR(txtrecord));
            SETDEREF(dstat, DSTAT_INFO_ADSP_NOT_EXIST);
        }   // end if

        // a TXT RR is not a valid ADSP record
        DnsTxtResponse_free(txt_rr);
        break;

    case DNS_STAT_NXDOMAIN:
    case DNS_STAT_NODATA:
        // この階層でレコードが見つからなくてもエラーではない
        DkimLogDebug(policy, "No ADSP records are found in DNS, domain=%s", domain);
        SETDEREF(dstat, DSTAT_INFO_ADSP_NOT_EXIST);
        break;

    case DNS_STAT_FORMERR:
    case DNS_STAT_SERVFAIL:
    case DNS_STAT_NOTIMPL:
    case DNS_STAT_REFUSED:
    case DNS_STAT_YXDOMAIN:
    case DNS_STAT_YXRRSET:
    case DNS_STAT_NXRRSET:
    case DNS_STAT_NOTAUTH:
    case DNS_STAT_NOTZONE:
    case DNS_STAT_RESERVED11:
    case DNS_STAT_RESERVED12:
    case DNS_STAT_RESERVED13:
    case DNS_STAT_RESERVED14:
    case DNS_STAT_RESERVED15:
        DkimLogInfo(policy, "DNS error on ADSP record look-up, domain=%s, err=%s",
                    domain, DnsResolver_getErrorString(resolver));
        SETDEREF(dstat, DSTAT_TMPERR_DNS_LOOKUP_FAILURE);
        break;

    default:
        DkimLogImplError(policy,
                         "DnsResolver_lookupTxt returns unexpected value: value=0x%x, domain=%s",
                         txtquery_stat, domain);
        SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
        break;
    }   // end switch

    return NULL;
}   // end function : DkimAdsp_query

/**
 * Checking whether a given Author Domain is within scope for ADSP.
 * @error DSTAT_OK 指定されたドメインは存在した.
 * @error DSTAT_INFO_ADSP_NXDOMAIN 指定したドメインへの MX レコードのルックアップが NXDOMAIN を返した.
 * @error DSTAT_TMPERR_DNS_LOOKUP_FAILURE DNS エラーが発生した.
 * @error DSTAT_SYSERR_IMPLERROR 実装エラー.
 */
static dkim_stat_t
DkimAdsp_checkDomainExistence(const DkimPolicy *policy, DnsResolver *resolver, const char *domain)
{
    assert(NULL != resolver);
    assert(NULL != domain);

    /*
     * [draft-ietf-dkim-ssp-09] 4.3.
     * The host MUST perform a DNS query for a record corresponding to
     * the Author Domain (with no prefix).  The type of the query can be
     * of any type, since this step is only to determine if the domain
     * itself exists in DNS.  This query MAY be done in parallel with the
     * query to fetch the named ADSP Record.  If the result of this query
     * is that the Author domain does not exist in the DNS (often called
     * an NXDOMAIN error, rcode=3 in [RFC1035]), the algorithm MUST
     * terminate with an error indicating that the domain is out of
     * scope.  Note that a result with rcode=0 but no records (often
     * called NODATA) is not the same as NXDOMAIN.
     *
     * NON-NORMATIVE DISCUSSION: Any resource record type could be
     * used for this query since the existence of a resource record of
     * any type will prevent an "NXDOMAIN" error.  MX is a reasonable
     * choice for this purpose because this record type is thought to
     * be the most common for domains used in e-mail, and will
     * therefore produce a result which can be more readily cached
     * than a negative result.
     */

    DnsMxResponse *mx_rr = NULL;
    dns_stat_t mxquery_stat = DnsResolver_lookupMx(resolver, domain, &mx_rr);
    switch (mxquery_stat) {
    case DNS_STAT_NOERROR:
        DnsMxResponse_free(mx_rr);
        // fall through

    case DNS_STAT_NODATA:
        return DSTAT_OK;

    case DNS_STAT_NXDOMAIN:
        DkimLogPermFail(policy, "The author domain does not exist: domain=%s, err=%s", domain,
                        DnsResolver_getErrorString(resolver));
        return DSTAT_INFO_ADSP_NXDOMAIN;

    case DNS_STAT_FORMERR:
    case DNS_STAT_SERVFAIL:
    case DNS_STAT_NOTIMPL:
    case DNS_STAT_REFUSED:
    case DNS_STAT_YXDOMAIN:
    case DNS_STAT_YXRRSET:
    case DNS_STAT_NXRRSET:
    case DNS_STAT_NOTAUTH:
    case DNS_STAT_NOTZONE:
    case DNS_STAT_RESERVED11:
    case DNS_STAT_RESERVED12:
    case DNS_STAT_RESERVED13:
    case DNS_STAT_RESERVED14:
    case DNS_STAT_RESERVED15:
        DkimLogPermFail(policy, "DNS error on checking author domain existence, domain=%s, err=%s",
                        domain, DnsResolver_getErrorString(resolver));
        return DSTAT_TMPERR_DNS_LOOKUP_FAILURE;

    default:
        DkimLogImplError(policy, "DnsResolver_lookupTxt returns unexpected value: 0x%x",
                         mxquery_stat);
        return DSTAT_SYSERR_IMPLERROR;
    }   // end switch
}   // end function : DkimAdsp_checkDomainExistence

static char *
DkimAdsp_buildAdspDomainName(const DkimPolicy *policy, const char *authordomain, dkim_stat_t *dstat)
{
    size_t dkimdomainlen =
        strlen(authordomain) + sizeof(DKIM_DNS_ADSP_SELECTOR "." DKIM_DNS_NAMESPACE ".");
    char *dkimdomain = (char *) malloc(dkimdomainlen);
    if (NULL == dkimdomain) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if

    int ret =
        snprintf(dkimdomain, dkimdomainlen, DKIM_DNS_ADSP_SELECTOR "." DKIM_DNS_NAMESPACE ".%s",
                 authordomain);
    if ((int) dkimdomainlen <= ret) {
        DkimLogImplError(policy, "buffer too small: bufsize=%u, writelen=%d, domain=%s",
                         dkimdomainlen, ret, authordomain);
        free(dkimdomain);
        SETDEREF(dstat, DSTAT_SYSERR_IMPLERROR);
        return NULL;
    }   // end if
    SETDEREF(dstat, DSTAT_OK);
    return dkimdomain;
}   // end function : DkimAdsp_buildAdspDomainName

/**
 * @error DSTAT_OK 適切な ADSP レコードが見つかった.
 * @error DSTAT_INFO_ADSP_NXDOMAIN 指定したドメインへの MX レコードのルックアップが NXDOMAIN を返した.
 * @error DSTAT_INFO_ADSP_NOT_EXIST ADSP レコードは見つからなかった, またはレコードの文法が ADSP レコードに合致しなかった.
 * @error DSTAT_PERMFAIL_MULTIPLE_ADSP_RECORD ADSP レコードが複数存在した
 * @error DSTAT_TMPERR_DNS_LOOKUP_FAILURE DNS エラーが発生した.
 * @error DSTAT_SYSERR_NORESOURCE メモリ確保エラー.
 * @error DSTAT_SYSERR_IMPLERROR 実装エラー.
 */
DkimAdsp *
DkimAdsp_retrieve(const DkimPolicy *policy, const char *authordomain, DnsResolver *resolver,
                  dkim_stat_t *dstat)
{
    assert(NULL != authordomain);
    assert(NULL != resolver);

    dkim_stat_t retr_stat;
    DkimAdsp *self = NULL;
    char *dkimdomain = NULL;

    // check domain existence with MX record look-up
    retr_stat = DkimAdsp_checkDomainExistence(policy, resolver, authordomain);
    if (DSTAT_OK != retr_stat) {
        SETDEREF(dstat, retr_stat);
        goto finally;
    }   // end if

    // build domain name to look-up an ADSP record
    dkimdomain = DkimAdsp_buildAdspDomainName(policy, authordomain, &retr_stat);
    if (NULL == dkimdomain) {
        SETDEREF(dstat, retr_stat);
        goto finally;
    }   // end if

    // retrieve an ADSP record
    self = DkimAdsp_query(policy, resolver, dkimdomain, dstat);

  finally:
    free(dkimdomain);
    return self;
}   // end function : DkimAdsp_retrieve

////////////////////////////////////////////////////////////////////////
// accessor

dkim_adsp_signpractice_t
DkimAdsp_getSigningPractice(const DkimAdsp *self)
{
    return self->signing_practice;
}   // end function : DkimAdsp_getSigningPractice
