/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimverifysession.c 853 2009-03-30 00:45:29Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimverifysession.c 853 2009-03-30 00:45:29Z takahiko $");

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <limits.h>
#include <sys/param.h>
#include <sys/types.h>
#include <stdbool.h>

#include "intarray.h"
#include "strarray.h"
#include "pstring.h"
#include "ptrop.h"
#include "inetdomain.h"
#include "inetmailbox.h"
#include "mailheaders.h"
#include "dkimlogger.h"
#include "dkim.h"
#include "dkimenum.h"
#include "dkimspec.h"
#include "dkimpubkey.h"
#include "dkimadsp.h"
#include "dkimsignature.h"
#include "dkimdigest.h"
#include "dkimheaders.h"
#include "dkimverifypolicy.h"

typedef struct DkimVerifyFrame {
    /// 検証フレームに固有な検証経過/結果のステータスを保持する.
    dkim_stat_t dstat;
//  /// DkimVerifySession の headers メンバが保持する DKIM-Signature ヘッダへのインデックス
//  size_t headeridx;
    /// DKIM-Signature ヘッダをパースして構築した DkimSignature オブジェクト
    DkimSignature *signature;
    /// DKIM-Signature ヘッダに対応する公開鍵オブジェクト
    DkimPubkey *pubkey;
    /// ダイジェスト値を計算するオブジェクト
    DkimDigest *digest;
    /// DKIM 署名検証スコア (のキャッシュ)
    dkim_score_t score;
} DkimVerifyFrame;

struct DkimVerifySession {
    /// 検証ポリシー
    const DkimVerifyPolicy *vpolicy;
    /// 検証処理全体のステータスを保持する.
    dkim_stat_t vprocstat;

    // DNS resolver
    DnsResolver *resolver;

    /// 元のメールについていた DKIM-Signature ヘッダの数
    /// 設定した検証対象ヘッダの上限を超えた場合 DkimVerifyFrame の数より多い場合がある
    size_t sigheadernum;

    /// 元メールのヘッダを保持する MailHeaders オブジェクトへの参照
    const MailHeaders *headers;
    /// 検証フレームへのポインタの配列
    PtrArray *frame;
    /// ADSP record
    DkimAdsp *adsp;
    /// DKIM ADSP score (cache)
    dkim_adsp_score_t adsp_score;

    // author
    InetMailbox *author;
    size_t authoridx;
    const char *refRawAuthorField;  // リファレンスを保持するだけなので解放はしないこと
    const char *refRawAuthorValue;  // リファレンスを保持するだけなので解放はしないこと
};

static DkimVerifyFrame *DkimVerifyFrame_new(void);
static void DkimVerifyFrame_free(DkimVerifyFrame *frame);

DkimVerifyFrame *
DkimVerifyFrame_new(void)
{
    DkimVerifyFrame *frame = (DkimVerifyFrame *) malloc(sizeof(DkimVerifyFrame));
    if (NULL == frame) {
        return NULL;
    }   // end if
    memset(frame, 0, sizeof(DkimVerifyFrame));

    frame->dstat = DSTAT_OK;
    frame->score = DKIM_SCORE_NULL;

    return frame;
}   // end function : DkimVerifyFrame_new

void
DkimVerifyFrame_free(DkimVerifyFrame *frame)
{
    assert(NULL != frame);

    if (NULL != frame->digest) {
        DkimDigest_free(frame->digest);
    }   // end if
    if (NULL != frame->signature) {
        DkimSignature_free(frame->signature);
    }   // end if
    if (NULL != frame->pubkey) {
        DkimPubkey_free(frame->pubkey);
    }   // end if
    free(frame);
}   // end function : DkimVerifyFrame_free

/**
 * DkimVerifySession オブジェクトの構築
 * @return 空の DkimVerifySession オブジェクト
 */
DkimVerifySession *
DkimVerifySession_new(const DkimVerifyPolicy *vpolicy, DnsResolver *resolver)
{
    assert(NULL != vpolicy);

    DkimVerifySession *self = (DkimVerifySession *) malloc(sizeof(DkimVerifySession));
    if (NULL == self) {
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimVerifySession));

    // 最低限必要なオブジェクトの初期化
//  self->headers = DkimHeaders_new(0);
//  if (NULL == self->headers) {
//      goto cleanup;
//  }   // end if
    self->frame = PtrArray_new(0, (void (*)(void *)) DkimVerifyFrame_free);
    if (NULL == self->frame) {
        goto cleanup;
    }   // end if

    self->sigheadernum = 0;
    self->vpolicy = vpolicy;
    self->adsp_score = DKIM_ADSP_SCORE_NULL;
    self->resolver = resolver;

    return self;

  cleanup:
    DkimVerifySession_free(self);
    return NULL;
}   // end function : DkimVerifySession_new

/**
 * release DkimVerifySession object
 * @param self DkimVerifySession object to release
 */
void
DkimVerifySession_free(DkimVerifySession *self)
{
    assert(NULL != self);

//  if (NULL != self->headers) {
//      DkimHeaders_free(self->headers);
//  }   // end if
    if (NULL != self->frame) {
        PtrArray_free(self->frame);
    }   // end if
    if (NULL != self->adsp) {
        DkimAdsp_free(self->adsp);
    }   // end if
    if (NULL != self->author) {
        InetMailbox_free(self->author);
    }   // end if

    free(self);
}   // end function : DkimVerifySession_free

static dkim_stat_t
DkimVerifySession_setupFrame(DkimVerifySession *self, const char *headerf, const char *headerv)
{
    dkim_stat_t ret;

    // create a verification frame
    DkimVerifyFrame *frame = DkimVerifyFrame_new();
    if (NULL == frame) {
        DkimLogNoResource(self->vpolicy);
        self->vprocstat = DSTAT_SYSERR_NORESOURCE;
        return self->vprocstat;
    }   // end if

    // 不正な署名ヘッダも1つと数えるので, すぐに DkimVerifySession に登録
    if (0 > PtrArray_append(self->frame, frame)) {
        DkimVerifyFrame_free(frame);
        DkimLogNoResource(self->vpolicy);
        self->vprocstat = DSTAT_SYSERR_NORESOURCE;
        return self->vprocstat;
    }   // end if

    // DKIM-Signature ヘッダの parse および検証
    frame->signature =
        DkimSignature_build((const DkimPolicy *) self->vpolicy, headerf, headerv, &ret);
    if (NULL == frame->signature) {
        frame->dstat = ret;
        return frame->dstat;
    }   // end if

    // Author が署名されているか (sig-h-tag に含まれているか) 確認する.
    // sig-h-tag では同名のヘッダが複数回指定され得るが,
    // DkimHeaders_extractAuthor() では, メッセージヘッダ中に複数回出現する ヘッダを Author として選択しないため,
    // sig-h-tag に含まれるかのみを確認すればよい.
    if (!DkimSignature_isHeaderSigned(frame->signature, self->refRawAuthorField)) {
        DkimLogPermFail(self->vpolicy, "sig-h-tag doesn't include author header: author-header=%s",
                        self->refRawAuthorField);
        frame->dstat = DSTAT_PERMFAIL_AUTHOR_NOT_SIGNED;
        return frame->dstat;
    }   // end if

    // 検証ポリシーで期限切れの署名を受け付けないよう指定されている場合は署名の期限を確認する.
    if (!self->vpolicy->accept_expired_signature) {
        frame->dstat = DkimSignature_isExpired(frame->signature);
        if (DSTAT_OK != frame->dstat) {
            return frame->dstat;
        }   // end if
    }   // end if

    // DKIM-Signature ヘッダが正当な場合
    // 署名ヘッダの概略をログに吐けるようメッセージを生成
    DkimLogInfo
        (self->vpolicy,
         "DKIM-Signature[%u]: domain=%s, selector=%s, pubkeyalg=%s, digestalg=%s, hdrcanon=%s, bodycanon=%s",
         (unsigned int) self->sigheadernum,
         InetMailbox_getDomain(DkimSignature_getIdentity(frame->signature)),
         DkimSignature_getSelector(frame->signature),
         DkimEnum_lookupPubkeyAlgByValue(DkimSignature_getPubKeyAlg(frame->signature)),
         DkimEnum_lookupDigestAlgByValue(DkimSignature_getDigestAlg(frame->signature)),
         DkimEnum_lookupCanonAlgByValue(DkimSignature_getHeaderCanonAlg(frame->signature)),
         DkimEnum_lookupCanonAlgByValue(DkimSignature_getBodyCanonAlg(frame->signature)));

    // 公開鍵の取得
    frame->pubkey =
        DkimPubkey_retrieve((const DkimPolicy *) self->vpolicy, frame->signature, self->resolver,
                            &ret);
    if (NULL == frame->pubkey) {
        frame->dstat = ret;
        return frame->dstat;
    }   // end if

    // DkimDigest オブジェクトの生成
    frame->digest =
        DkimDigest_newWithSignature((const DkimPolicy *) self->vpolicy, frame->signature, &ret);
    if (NULL == frame->digest) {
        frame->dstat = ret;
        return frame->dstat;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimVerifySession_setupFrame

/**
 * DKIM の検証対象とするメールのヘッダを登録する.
 * @param headers ヘッダを格納する MailHeaders オブジェクト.
 *                キーはヘッダ名. ": " の左側の部分. ただし ":" は含まない.
 *                値はヘッダの値. ": " の右側の部分. ただし ":" の直後のスペースは含まない (sendmail 8.13 まで, 8.14 以降では仕様が変わる).
 * @return DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS: Author を抽出できなかったので検証処理を中断した.
 */
dkim_stat_t
DkimVerifySession_setHeaders(DkimVerifySession *self, const MailHeaders *headers)
{
    assert(NULL != self);
    assert(NULL != headers);
    assert(NULL == self->headers);

    self->headers = headers;

    // Author の抽出
    // - 署名 (DKIM-Signature) ヘッダの有無に関わらず Author の抽出はおこなう.
    // SPEC: Author の抽出ができない場合は署名の検証処理は一切おこなわない
    dkim_stat_t ext_stat =
        DkimHeaders_extractAuthor((const DkimPolicy *) self->vpolicy, self->headers,
                                  self->vpolicy->author_priority,
                                  &(self->authoridx), &(self->refRawAuthorField),
                                  &(self->refRawAuthorValue), &(self->author));
    if (DSTAT_OK != ext_stat) {
        self->vprocstat = ext_stat;
    }   // end if

    return self->vprocstat;
}   // end function : DkimVerifySession_setHeaders

/**
 * DKIM の検証対象とするメールのヘッダを登録する.
 * @return DSTAT_OK: DKIM 署名ヘッダが見つかり, 検証処理の続行可能.
 *         DSTAT_VERIFIED_NO_SIGNHEADER: DKIM 署名ヘッダがなかったので検証処理の続行は意味がない.
 *         その他: その他のエラー
 */
dkim_stat_t
DkimVerifySession_startBody(DkimVerifySession *self)
{
    assert(NULL != self);

    if (DSTAT_OK != self->vprocstat) {
        // 何もしない
        return DSTAT_OK;
    }   // end if

    assert(NULL != self->author);

    // DKIM-Signature ヘッダの数だけ検証フレームをセットアップ
    size_t headernum = MailHeaders_getCount(self->headers);
    for (size_t headeridx = 0; headeridx < headernum; ++headeridx) {
        const char *headerf, *headerv;
        MailHeaders_get(self->headers, headeridx, &headerf, &headerv);
        if (NULL == headerf || NULL == headerv) {
            continue;
        }   // end if

        if (0 != strcasecmp(DKIM_SIGNHEADER, headerf)) {
            // DKIM-Signature ヘッダでない
            continue;
        }   // end if

        // DKIM-Signature ヘッダを発見
        ++(self->sigheadernum);

        // 一通あたりの署名ヘッダの最大数を超えていないことを確認
        if (self->vpolicy->sign_header_limit < self->sigheadernum) {
            DkimLogInfo(self->vpolicy, "too many signature headers: count=%u, limit=%u",
                        self->sigheadernum, self->vpolicy->sign_header_limit);
            break;
        }   // end if

        dkim_stat_t setup_stat = DkimVerifySession_setupFrame(self, headerf, headerv);
        if (DSTAT_ISCRITERR(setup_stat)) {
            // システムエラーが発生した場合はすぐに抜ける
            self->vprocstat = setup_stat;
            return self->vprocstat;
        }   // end if
    }   // end for

    // DKIM-Signature ヘッダが見つかったか確認する
    size_t framenum = PtrArray_getCount(self->frame);
    if (0 == framenum) {
        self->vprocstat = DSTAT_VERIFIED_NO_SIGNHEADER;
        return self->vprocstat;
    }   // end if

    self->vprocstat = DSTAT_OK;
    return self->vprocstat;
}   // end function : DkimVerifySession_startBody

dkim_stat_t
DkimVerifySession_updateBody(DkimVerifySession *self, const unsigned char *bodyp, size_t len)
{
    assert(NULL != self);

    if (DSTAT_OK != self->vprocstat) {
        // 何もしない
        return DSTAT_OK;
    }   // end if

    // 各検証フレームに対してダイジェストを更新
    size_t framenum = PtrArray_getCount(self->frame);
    for (size_t frameidx = 0; frameidx < framenum; ++frameidx) {
        DkimVerifyFrame *frame = (DkimVerifyFrame *) PtrArray_get(self->frame, frameidx);
        // ここまでの過程でエラーが発生しているものは弾く
        if (DSTAT_OK != frame->dstat) {
            continue;
        }   // end if

        frame->dstat = DkimDigest_updateBody(frame->digest, bodyp, len);
        if (DSTAT_OK != frame->dstat) {
            // EVP_DigestUpdate() は事実上失敗しないので,
            // ここで失敗するのはメモリの確保に失敗した場合くらい
            DkimLogPermFail(self->vpolicy, "body digest update failed for signature no.%u",
                            (unsigned int) frameidx);
            // 他の署名の検証を続行させるため，return しない
        }   // end if
    }   // end if

    return DSTAT_OK;
}   // end function : DkimVerifySession_updateBody

static bool
DkimVerifySession_isAuthorSignature(const InetMailbox *identity, const InetMailbox *author)
{
    /*
     * [draft-ietf-dkim-ssp-09] 2.7.
     * An "author signature" is a Valid Signature that has the same domain
     * name in the DKIM signing identity as the domain name in the Author
     * Address.  If the DKIM signing identity has a Local-part, it is be
     * identical to the Local-part in the Author Address.  Following
     * [RFC5321], Local-part comparisons are case sensitive, but domain
     * comparisons are case insensitive.
     */
    if (!InetDomain_equals(InetMailbox_getDomain(identity), InetMailbox_getDomain(author))) {
        return false;
    }   // end if

    const char *identity_localpart = InetMailbox_getLocalPart(identity);
    if (0 < strlen(identity_localpart)
        && 0 != strcmp(identity_localpart, InetMailbox_getLocalPart(author))) {
        return false;
    }   // end if

    return true;
}   // end function : DkimVerifySession_isAuthorSignature

dkim_stat_t
DkimVerifySession_finishBody(DkimVerifySession *self)
{
    assert(NULL != self);

    if (DSTAT_OK != self->vprocstat) {
        // 何もしない
        return self->vprocstat;
    }   // end if

    size_t framenum = PtrArray_getCount(self->frame);
    for (size_t frameidx = 0; frameidx < framenum; ++frameidx) {
        DkimVerifyFrame *frame = (DkimVerifyFrame *) PtrArray_get(self->frame, frameidx);
        // ここまでの過程でエラーが発生しているものは弾く
        if (DSTAT_OK != frame->dstat) {
            continue;
        }   // end if

        frame->dstat =
            DkimDigest_verifyMessage(frame->digest, self->headers, frame->signature,
                                     DkimPubkey_getPublicKey(frame->pubkey));

        /*
         * [draft-ietf-dkim-ssp-09] 2.7.
         * An "author signature" is a Valid Signature that has the same domain
         * name in the DKIM signing identity as the domain name in the Author
         * Address.  If the DKIM signing identity has a Local-part, it is be
         * identical to the Local-part in the Author Address.  Following
         * [RFC5321], Local-part comparisons are case sensitive, but domain
         * comparisons are case insensitive.
         */
        if (DSTAT_INFO_DIGEST_MATCH == frame->dstat) {
            const InetMailbox *identity = DkimSignature_getIdentity(frame->signature);
            if (DkimVerifySession_isAuthorSignature(identity, self->author)) {
                // Author signature (= First party signature)
                frame->dstat = DSTAT_VERIFIED_AUTHOR;
            } else {
                // Third party signature
                DkimLogDebug(self->vpolicy,
                             "third party signature: identity=%s@%s, author=%s@%s",
                             InetMailbox_getLocalPart(identity),
                             InetMailbox_getDomain(identity),
                             InetMailbox_getLocalPart(self->author),
                             InetMailbox_getDomain(self->author));
                frame->dstat = DSTAT_VERIFIED_THIRDPARTY;
            }   // end if
        }   // end if
    }   // end for

    return DSTAT_OK;
}   // end function : DkimVerifySession_finishBody

/**
 * Author のドメインを元に ADSP レコードを取得し, dkim-adsp スコアを計算する.
 * @param signature_verified 1つ以上の署名の検証に成功している場合は true,
 *                           署名の検証に1つも成功していない場合は false.
 *                           dkim=all の場合のスコアに影響する.
 * @return ADSP スコア, エラーが発生した場合は DKIM_ADSP_SCORE_NULL.
 * @attention author が NULL な状況で呼び出してはならない.
 */
dkim_adsp_score_t
DkimVerifySession_evalAdsp(DkimVerifySession *self)
{
    // return ADSP score cache if available
    if (DKIM_ADSP_SCORE_NULL != self->adsp_score) {
        return self->adsp_score;
    }   // end if

    if (NULL == self->author) {
        // SPEC: Author が存在しない場合は permerror
        return DKIM_ADSP_SCORE_PERMERROR;
    }   // end if

    // 署名の検証に成功しているかを調べる
    bool have_author_signature = false;
    bool have_temporary_error = false;
    size_t framenum = PtrArray_getCount(self->frame);
    for (size_t frameidx = 0; frameidx < framenum; ++frameidx) {
        DkimVerifyFrame *frame = (DkimVerifyFrame *) PtrArray_get(self->frame, frameidx);
        if (DSTAT_VERIFIED_AUTHOR == frame->dstat) {
            have_author_signature = true;
        } else if (DSTAT_ISTMPERR(frame->dstat) || DSTAT_ISSYSERR(frame->dstat)) {
            have_temporary_error = true;
        }   // end if
    }   // end for

    if (have_author_signature) {
        /*
         * [draft-kucherawy-sender-auth-header-20] 2.4.2.
         * pass:  This message had an author signature which validated.  (An
         *    ADSP check is not strictly required to be performed for this
         *    result, since a valid author domain signature satisfies all
         *    possible ADSP policies.)
         */
        return self->adsp_score = DKIM_ADSP_SCORE_PASS;
    } else if (have_temporary_error) {
        // SPEC: システムエラー時の dkim-adsp スコアは "temperror"
        return self->adsp_score = DKIM_ADSP_SCORE_TEMPERROR;
    }   // end if

    // retrieving ADSP record if the message doesn't have an author signature
    if (NULL == self->adsp) {
        dkim_stat_t adsp_stat;
        self->adsp =
            DkimAdsp_retrieve((const DkimPolicy *) self->vpolicy,
                              InetMailbox_getDomain(self->author), self->resolver, &adsp_stat);
        switch (adsp_stat) {
        case DSTAT_OK:
            // do nothing
            break;
        case DSTAT_INFO_ADSP_NXDOMAIN:
            /*
             * 当該ドメインへの DNS クエリが NXDOMAIN エラーになった.
             * [draft-kucherawy-sender-auth-header-20] 2.4.2.
             * nxdomain:  Evaluating the ADSP for the author's DNS domain indicated
             *    that the author's DNS domain does not exist.
             */
            DkimLogInfo(self->vpolicy, "Author domain seems not to exist (NXDOMAIN): domain=%s",
                        InetMailbox_getDomain(self->author));
            return self->adsp_score = DKIM_ADSP_SCORE_NXDOMAIN;
        case DSTAT_INFO_ADSP_NOT_EXIST:
            /*
             * no valid ADSP records are found
             * [draft-kucherawy-sender-auth-header-20] 2.4.2.
             * none:  No DKIM author domain signing practises (ADSP) record was
             *    published.
             */
            DkimLogInfo(self->vpolicy, "no valid DKIM ADSP records are found: domain=%s",
                        InetMailbox_getDomain(self->author));
            return self->adsp_score = DKIM_ADSP_SCORE_NONE;
        case DSTAT_PERMFAIL_MULTIPLE_ADSP_RECORD:
            /*
             * multiple ADSP records are found
             * [draft-kucherawy-sender-auth-header-20] 2.4.2.
             * permerror:  A DKIM policy could not be retrieved due to some error
             *    which is likely not transient in nature, such as a permanent DNS
             *    error.  A later attempt is unlikely to produce a final result.
             */
            DkimLogInfo(self->vpolicy, "multiple DKIM ADSP records are found: domain=%s",
                        InetMailbox_getDomain(self->author));
            return self->adsp_score = DKIM_ADSP_SCORE_PERMERROR;
        case DSTAT_TMPERR_DNS_LOOKUP_FAILURE:
            DkimLogInfo(self->vpolicy,
                        "DNS lookup error has occurred while retrieving the ADSP record: domain=%s",
                        InetMailbox_getDomain(self->author));
            return self->adsp_score = DKIM_ADSP_SCORE_TEMPERROR;
        case DSTAT_SYSERR_NORESOURCE:
            DkimLogSysError(self->vpolicy,
                            "System error occurred while retrieving the ADSP record: domain=%s",
                            InetMailbox_getDomain(self->author));
            return DKIM_ADSP_SCORE_NULL;
        case DSTAT_SYSERR_IMPLERROR:
        default:
            DkimLogImplError
                (self->vpolicy,
                 "unexpected error occurred while retrieving the ADSP record: domain=%s, err=%s",
                 InetMailbox_getDomain(self->author), DKIM_strerror(adsp_stat));
            abort();
        }   // end switch
    }   // end if

    // log ADSP record
    dkim_adsp_signpractice_t outbound_practice = DkimAdsp_getSigningPractice(self->adsp);
    DkimLogDebug(self->vpolicy, "valid DKIM ADSP record is found: domain=%s, practice=%s",
                 InetMailbox_getDomain(self->author),
                 DkimEnum_lookupSignPracticeByValue(outbound_practice));

    // determine ADSP score according to outbound signing practice
    switch (outbound_practice) {
    case DKIM_ADSP_SIGNPRACTICE_ALL:
        /*
         * [draft-ietf-dkim-ssp-09] 4.2.1.
         * all  All mail from the domain is signed with an Author Signature.
         *
         * [draft-kucherawy-sender-auth-header-20] 2.4.2.
         * fail:  No valid author signature was found on the message and the
         *    published ASDP record indicated an "all" policy.
         */
        self->adsp_score = DKIM_ADSP_SCORE_FAIL;
        break;
    case DKIM_ADSP_SIGNPRACTICE_DISCARDABLE:
        /*
         * [draft-ietf-dkim-ssp-09] 4.2.1.
         * discardable  All mail from the domain is signed with an Author
         *    Signature.  Furthermore, if a message arrives without a valid
         *    Author Signature due to modification in transit, submission via
         *    a path without access to a signing key, or any other reason,
         *    the domain encourages the recipient(s) to discard it.
         *
         * [draft-kucherawy-sender-auth-header-20] 2.4.2.
         * discard:  No valid author signature was found on the message and the
         *    published ADSP record indicated a "discardable" policy.
         */
        self->adsp_score = DKIM_ADSP_SCORE_DISCARD;
        break;
    case DKIM_ADSP_SIGNPRACTICE_UNKNOWN:
        /*
         * [draft-ietf-dkim-ssp-09] 4.2.1.
         * unknown  The domain might sign some or all email.
         *
         * [draft-kucherawy-sender-auth-header-20] 2.4.2.
         * unknown:  No valid author signature was found on the message and the
         *    published ADSP was "unknown".
         */
        self->adsp_score = DKIM_ADSP_SCORE_UNKNOWN;
        break;
    case DKIM_ADSP_SIGNPRACTICE_NULL:
    default:
        abort();
    }   // end switch

    return self->adsp_score;
}   // end function : DkimVerifySession_evalAdsp

/**
 * 検証対象となっている DKIM 署名の数を返す.
 * つまり, DKIM 検証フレームの数を返す.
 */
size_t
DkimVerifySession_getVerificationNum(const DkimVerifySession *self)
{
    assert(NULL != self);
    return PtrArray_getCount(self->frame);
}   // end function : DkimVerifySession_getVerificationNum

static dkim_score_t
DkimVerifySession_getVerifyFrameScore(const DkimVerifySession *self, DkimVerifyFrame *frame)
{
    // スコアがキャッシュされている場合はそれを返す.
    if (DKIM_SCORE_NULL != frame->score) {
        return frame->score;
    }   // end if

    // 検証セッション全体でエラーになっていないか確認
    switch (self->vprocstat) {
    case DSTAT_OK:
        // do nothing
        break;
    case DSTAT_PERMFAIL_AUTHOR_AMBIGUOUS:
        /*
         * Author の抽出に失敗した場合は "permerror"
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * permerror:  The message could not be verified due to some error which
         *    is unrecoverable, such as a required header field being absent.  A
         *    later attempt is unlikley to produce a final result.
         */
        return frame->score = DKIM_SCORE_PERMERROR;
    case DSTAT_VERIFIED_NO_SIGNHEADER:
        /*
         * DKIM-Signature ヘッダが見つからない場合は "none"
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * none:  The message was not signed.
         */
        return frame->score = DKIM_SCORE_NONE;
    case DSTAT_SYSERR_NORESOURCE:
    default:
        return frame->score = DKIM_SCORE_TEMPERROR;
    }   // end switch

    if (DSTAT_ISTMPERR(frame->dstat) || DSTAT_ISSYSERR(frame->dstat)) {
        return frame->score = DKIM_SCORE_TEMPERROR;
    }   // end if

    switch (frame->dstat) {
    case DSTAT_VERIFIED_AUTHOR:
        /*
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * pass:  The message was signed, the signature(s) is (were) acceptable
         *    to the verifier, and the signature(s) passed verification tests.
         */
        return frame->score = DKIM_SCORE_PASS;
    case DSTAT_VERIFIED_THIRDPARTY:
        // Third party signature
        DkimLogInfo(self->vpolicy, "third party signature: identity-domain=%s, author-domain=%s",
                    InetMailbox_getDomain(DkimSignature_getIdentity(frame->signature)),
                    InetMailbox_getDomain(self->author));
        // RFC4871 では Author と sig-i-tag の関係には触れていないので,
        // Third party signature でも "dkim=" スコアは "pass".
        return frame->score = DKIM_SCORE_PASS;
    case DSTAT_PERMFAIL_HEADERDIGEST_MISMATCH:
    case DSTAT_PERMFAIL_BODYDIGEST_MISMATCH:
        /*
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * fail:  The message was signed and the signature(s) is (were)
         *    acceptable to the verifier, but it (they) failed the verification
         *    test(s).
         */
        return frame->score = DKIM_SCORE_FAIL;
    default:
        /*
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * neutral:  The message was signed but the signature(s) contained
         *    syntax errors or were not otherwise able to be processed.  This
         *    result SHOULD also be used for other failures not covered
         *    elsewhere in this list.
         */
        return frame->score = DKIM_SCORE_NEUTRAL;
    }   // end switch
}   // end function : DkimVerifySession_getVerifyFrameScore

/**
 * 指定した番号の署名検証結果を返す.
 */
dkim_score_t
DkimVerifySession_getVerifyFrameResult(const DkimVerifySession *self, size_t signo,
                                       const InetMailbox **identity)
{
    size_t framenum = PtrArray_getCount(self->frame);
    DkimVerifyFrame *frame = (DkimVerifyFrame *) PtrArray_get(self->frame, signo);
    dkim_score_t rawscore;

    if (signo < framenum) {
        rawscore = DkimVerifySession_getVerifyFrameScore(self, frame);
    } else if (signo < self->sigheadernum) {
        /*
         * 署名の数が多すぎて検証対象にならなかった場合は "policy"
         * [draft-kucherawy-sender-auth-header-20] 2.4.1.
         * policy:  The message was signed but the signature(s) is (were) not
         *    acceptable to the verifier.
         */
        rawscore = DKIM_SCORE_POLICY;
    } else {
        abort();
    }   // end if

    *identity = frame->signature ? DkimSignature_getIdentity(frame->signature) : NULL;
    return rawscore;
}   // end function : DkimVerifySession_getVerificationResult

/*
 * @attention 必ず DkimVerifySession_eoh() の後によぶこと
 */
dkim_stat_t
DkimVerifySession_setCanonDump(DkimVerifySession *self, const char *basedir, const char *prefix)
{
    assert(NULL != self);

    if (DSTAT_OK != self->vprocstat) {
        // do nothing
        return DSTAT_OK;
    }   // end if

    // 署名対象のヘッダが異なる可能性があるので DKIM-Signature ヘッダの数 (ただし設定数以下) だけダンプする
    size_t framenum = PtrArray_getCount(self->frame);
    for (size_t frameidx = 0; frameidx < framenum; ++frameidx) {
        DkimVerifyFrame *frame = (DkimVerifyFrame *) PtrArray_get(self->frame, frameidx);
        char fnHeader[MAXPATHLEN];
        char fnBody[MAXPATHLEN];

        if (DSTAT_OK != frame->dstat) {
            continue;
        }   // end if
        snprintf(fnHeader, MAXPATHLEN, "%s/%s.%02d.header", basedir, prefix, frameidx);
        snprintf(fnBody, MAXPATHLEN, "%s/%s.%02d.body", basedir, prefix, frameidx);
        (void) DkimDigest_openCanonDump(frame->digest, fnHeader, fnBody);
    }   // end for
    return DSTAT_OK;
}   // end function : DkimVerifySession_setCanonDump

////////////////////////////////////////////////////////////////////////
// accessor

const char *
DkimVerifySession_getAuthorHeaderName(const DkimVerifySession *self)
{
    return self->refRawAuthorField;
}   // end function : DkimVerifySession_getAuthorHeaderName

const InetMailbox *
DkimVerifySession_getAuthorMailbox(const DkimVerifySession *self)
{
    return self->author;
}   // end function : DkimVerifySession_getAuthorMailbox
