/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimwildcard.c 653 2009-02-28 20:35:27Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimwildcard.c 653 2009-02-28 20:35:27Z takahiko $");

#include <assert.h>
#include <stdbool.h>
#include <string.h>

#include "ptrop.h"
#include "xskip.h"
#include "dkimwildcard.h"

static bool
DkimWildcard_matchPubkeyGranularityImpl(const char *pattern_head, const char *pattern_tail,
                                        const char *target_head, const char *target_tail,
                                        bool accept_wildcard)
{
    const char *pattern;
    const char *target;

    for (pattern = pattern_head, target = target_head; pattern < pattern_tail; ++pattern, ++target) {
        if ('*' == *pattern) {
            if (accept_wildcard) {
                // '*' をワイルドカードとして扱う
                ++pattern;
                for (const char *bq = target_tail; target <= bq; --bq) {
                    if (DkimWildcard_matchPubkeyGranularityImpl
                        (pattern, pattern_tail, bq, target_tail, false)) {
                        return true;
                    }   // end if
                }   // end for
                return false;
            } else {
                // '*' を文字として扱う
                if (target_tail <= target || '*' != *target) {
                    return false;
                }   // end if
            }   // end if
        } else if (IS_ATEXT(*pattern) || '.' == *pattern) {
            if (target_tail <= target || *pattern != *target) { // case-sensitive に比較
                return false;
            }   // end if
        } else {
            // atext でも '.' (dot-atom-text に含まれる) でもない
            return false;
        }   // end if
    }   // end for
    return (pattern == pattern_tail && target == target_tail) ? true : false;
}   // end function : DkimWildcard_matchExec

/**
 * DKIM public key granularity に (与えられたローカルパートが) マッチするか調べる.
 *
 * TODO: 大文字小文字の区別はどうするか確認
 *
 * @param pattern パターン
 * @return マッチする場合は true, マッチしない場合は false
 * @attention '*' は pattern 中に1度しか許容されず, 2個目移行の '*' は通常の文字として扱われる.
 *
 * [RFC4871]
 * key-g-tag       = %x67 [FWS] "=" [FWS] key-g-tag-lpart
 * key-g-tag-lpart = [dot-atom-text] ["*" [dot-atom-text] ]
 */
bool
DkimWildcard_matchPubkeyGranularity(const char *pattern_head, const char *pattern_tail,
                                    const char *target_head, const char *target_tail)
{
    // [RFC4871] 3.6.1.
    // An empty "g=" value never matches any addresses.
    if (pattern_head == pattern_tail) {
        return false;
    }   // end if

    return DkimWildcard_matchPubkeyGranularityImpl(pattern_head, pattern_tail, target_head,
                                                   target_tail, true);
}   // end function : DkimWildcard_match
