#include "config.h"

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>

#include "FuseCompress.hpp"
#include "TransformTable.hpp"
#include "CompressedTable.hpp"
#include "CompressedMagic.hpp"

#include <cstdlib>
#include <cstring>
#include <iostream>
#include <vector>

#include <rlog/rlog.h>
#include <rlog/StdioNode.h>
#include <rlog/SyslogNode.h>
#include <rlog/RLogChannel.h>
#include <rlog/RLogNode.h>

using namespace std;
using namespace rlog;

TransformTable	*g_TransformTable;
int		 g_BufferedMemorySize = 100 * 1024;
bool		 g_Debug = false;
const char	*g_Storage;
CompressedTable	 g_CompressedTable;
CompressedMagic	 g_CompressedMagic;

static void print_license(void)
{
	printf("%s version %s\n", PACKAGE_NAME, PACKAGE_VERSION);
	printf("Copyright (C) 2007  Milan Svoboda.\n");
	printf("This is free software; see the source for copying conditions.  There is NO\n");
	printf("warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\n");
}

static void print_help(void)
{
	print_license();

	printf("Usage: %s [OPTIONS] /storage/directory /mount/point\n\n", PACKAGE);

	printf("\t-h                   print this help\n");
	printf("\t-v                   print version\n");
	printf("\t-c lzo/bz2/gz/none   choose compression method\n");
	printf("\t-b size              block size in kilobytes\n");
	printf("\t-o ...               pass arguments to fuse library\n");
	printf("\t-d                   print debug messages\n");
	printf("\t-f ext1,ext2,ext3    force files with listed extensions\n");
	printf("\t                      to be always compressed\n");
	printf("\n");
}

static void init_log(void)
{
	if (g_Debug)
	{
		static StdioNode log(STDERR_FILENO);
		log.subscribeTo(GetGlobalChannel("warning"));
		log.subscribeTo(GetGlobalChannel("error"));
		log.subscribeTo(GetGlobalChannel("debug"));
	}
	else
	{
		static SyslogNode log(PACKAGE_NAME);
		log.subscribeTo(GetGlobalChannel("warning"));
		log.subscribeTo(GetGlobalChannel("error"));
	}
}

int main(int argc, char *argv[])
{
	DIR	*dir;
	int	 next_option;
	char	*compressor = NULL;
	
	const char* const	 short_options = "dhvo:c:b:f:";

	vector<const char *>		fuse;
	FuseCompress		fusecompress;

	fuse.push_back(argv[0]);

	do {
		next_option = getopt(argc, argv, short_options);
		switch (next_option)
		{
			case 'd':
				fuse.push_back("-f");
				g_Debug = true;
				break;

			case 'h':
				print_help();
				exit(EXIT_SUCCESS);

			case 'v':
				print_license();
				exit(EXIT_SUCCESS);

			case 'c':
				compressor = optarg;
				break;

			case 'o':
				fuse.push_back("-o");
				fuse.push_back(optarg);
				break;

			case 'b':
				g_BufferedMemorySize = atoi(optarg) * 1024;
				if (g_BufferedMemorySize <= 0)
				{
					print_help();
					exit(EXIT_FAILURE);
				}
				break;

			case 'f':
				g_CompressedTable.Add(optarg, ",");
				break;

			case -1:
				// No more option arguments available.
				//
				break;

			default:
				print_help();
				exit(EXIT_FAILURE);
		}
	} while (next_option != -1);

	argc -= optind;
	argv += optind;
	
	if (argc != 2)
	{
	    print_help();
	    exit(EXIT_FAILURE);
	}

	// Set up default options for fuse.
	// 
	// Fuse problems:
	// 	kernel_cache - causes sigfaults when trying to run compiled
	// 	               executables from FuseCompressed filesystem
	//
	fuse.push_back("-o");
	fuse.push_back("default_permissions,use_ino");
	
	g_Storage = argv[0];

	fuse.push_back(argv[1]);

	g_TransformTable = new (std::nothrow) TransformTable();
	if (!g_TransformTable)
	{
		cerr << "No memory to allocate object of TransformTable class"
		     << endl;
		abort();
	}

	// Set default transformation as user wanted.
	// 
	if ((compressor) &&
	    (g_TransformTable->setDefault(compressor) == false))
	{
		cerr << "Compressor " << compressor << " not found!" << endl;
		exit(EXIT_FAILURE);
	}

	if ((dir = opendir(g_Storage)) == NULL)
	{
		int errns = errno;

		cerr << "Failed to open storage directory "
		     << "'" << g_Storage << "': " << strerror(errns) << endl;
		exit(EXIT_FAILURE);
	}

	init_log();

	umask(0);
	return fusecompress.Run(dir, fuse.size(), &fuse[0]);
}

