// -*- c++ -*-

/*
 *
 * Copyright (C) 2003 George Staikos <staikos@kde.org>
 *               2004 Dirk Ziegelmeier <dziegel@gmx.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef __KdetvChannel_PLUGIN_H
#define __KdetvChannel_PLUGIN_H

#include <qstring.h>
#include <qptrlist.h>
#include <qdatetime.h>

#include <kdebug.h>

#include "kdetvpluginbase.h"
#include "channel.h"

class ChannelStore;

/**
 * Misc plugin base class
 */
class LIBKDETV_EXPORT KdetvChannelPlugin : public KdetvPluginBase
{
	Q_OBJECT

public:

	KdetvChannelPlugin(Kdetv *ktv, const QString& cfgkey, QObject *parent = 0, const char* name = 0);
	virtual ~KdetvChannelPlugin();


	struct ChannelFileMetaInfo
    {
        ChannelFileMetaInfo()
            : _contributor(QString::null),
              _country(QString::null),
              _region(QString::null),
              _type(QString::null),
              _comment(QString::null),
              _lastUpdate(QDateTime::currentDateTime())
        {
        }

		QString   _contributor, _country, _region, _type, _comment;
        QDateTime _lastUpdate;
        QMap<QString, Channel::PropertyList> _globalControls;
	};

	/**
	 * Flags indicating the level of support provided by the handler.
	 */
	enum Flags {
		FormatRead = 0x1,
		FormatWrite = 0x2,
		FormatReadWrite = FormatRead | FormatWrite
	};

	/**
	 * The name of the format.
	 */
	const QString& formatName() const { return _fmtName; }

    /**
	 * Returns human-readable and localized plugin name for menu entries.
	 */
    const QString& menuName() const { return _menuName; };

	/**
	 * Returns true iff this object can read the specified format. If
	 * no format is specified, then it is assumed that the user wants
	 * to know about the one returned by @ref name() .
	 */
	bool canRead(const QString &fmt = QString::null) const;

	/**
	 * Returns true iff this object can write the specified format.
	 * @see canRead() .
	 */
	bool canWrite(const QString &fmt = QString::null) const;

	/**
	 * Returns true if this object can handle the specified file.
	 * This function should do a quick test about the file, so the proper
	 * format could be easily selected.
	 * The default implementation only checks, if a filename ends with the
	 * name of the format. If thats not appropriate, then you should override
	 * this method.
	 * @p rflags the requested flags
	 */
	virtual bool handlesFile(const QString& filename, int rflags) const;

	/**
	 * Load the specified file into the specified store. Returns true
	 * on success, false otherwise.
	 */
    virtual bool load(ChannelStore *store, KdetvChannelPlugin::ChannelFileMetaInfo *info,
                      const QString &filename, const QString &fmt);
    
	/**
	 * Save the specified store into the specified file. Returns true
	 * on success, false otherwise.
	 */
    virtual bool save(ChannelStore *store, KdetvChannelPlugin::ChannelFileMetaInfo *info,
                      const QString &filename, const QString &fmt);
    
	/**
	 * Load the contents of the specified QIODevice into the specified
	 * store.  Returns true on success, false otherwise.
	 */
	virtual bool load(ChannelStore *store, KdetvChannelPlugin::ChannelFileMetaInfo *info,
                      QIODevice *file, const QString& fmt);

	/**
	 * Save the contents of the specified store into the specified
	 * QIODevice. Returns true on success, false otherwise.
	 */
	virtual bool save( ChannelStore *store, KdetvChannelPlugin::ChannelFileMetaInfo *info,
                       QIODevice *file, const QString& fmt);

private:
	class KdetvChannelPluginPrivate;
	KdetvChannelPluginPrivate *d;

protected:
	QString _fmtName;
    QString _menuName;
	int _flags;
};

typedef QPtrList<KdetvChannelPlugin> KdetvChannelPluginList;

#endif
