/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "boardtabwidget.h"

#include "libkita/boardmanager.h"
#include "libkita/signalcollection.h"

#include <kpopupmenu.h>
#include <kdebug.h>
#include <klocale.h>
#include <krun.h>
#include <kstdaccel.h>
#include <kaction.h>

#include <qapplication.h>
#include <qclipboard.h>


KitaBoardTabWidget::KitaBoardTabWidget( QWidget* parent, const char* name, WFlags f )
        : KitaTabWidgetBase( parent, name, f )
{
    setXMLFile( "boardtabwidgetui.rc" );

    setTabBar( new SubjectTabBar( this ) );

    FavoriteListView* favoriteList = new FavoriteListView( this, "favoriteTab" );
    addTab( favoriteList, i18n( "Favorite" ) );

    setupActions();
}

KitaBoardTabWidget::~KitaBoardTabWidget()
{}

/* public slots */
void KitaBoardTabWidget::loadBoard( const KURL& boardURL, bool withNewTab )
{
    KitaBoardView * view = findView( boardURL );
    QString boardName = Kita::BoardManager::boardName( boardURL );
    if ( !view ) {

        if ( count() == 1 || withNewTab ) view = createView( boardName );
        else {

            QWidget* w = currentPage();
            /* currentPage is FavoriteListView. */
            if ( !isSubjectView( w ) ) w = page( count() - 2 );

            view = static_cast< KitaBoardView *>( w );
            view->init();
            setTabLabel( view, boardName );
        }
    }

    if ( view ) {
        showPage( view );
        view->loadBoard( boardURL );
    }
}

/* create KitaBoardView */ /* private */
KitaBoardView* KitaBoardTabWidget::createView( QString label )
{
    KitaBoardView * view = new KitaBoardView( this );
    if ( view ) {
        insertTab( view, label, count() - 1 );
        connect( view, SIGNAL( closeCurrentTab() ), SLOT( slotCloseCurrentTab() ) );
    }

    return view;
}

/* private */
KitaBoardView* KitaBoardTabWidget::findView( const KURL& boardURL )
{
    int max = count() - 1;
    if ( max <= 0 ) return NULL;
    int i = 0;

    while ( i < max ) {
        KitaBoardView * view = isSubjectView( page ( i ) );
        if ( view && view->boardURL() == boardURL ) return view;
        i++;
    }

    return NULL;
}


/* protected */ /* virtual */
void KitaBoardTabWidget::deleteWidget( QWidget* w )
{
    KitaBoardView * view = isSubjectView( w );
    if ( !view ) return ;
    removePage( view );
    delete view;
}


/* private */
KitaBoardView* KitaBoardTabWidget::isSubjectView( QWidget* w )
{
    KitaBoardView * view = NULL;
    if ( w ) {
        if ( w->isA( "KitaBoardView" ) ) view = static_cast< KitaBoardView* >( w );
    }

    return view;
}

/*--------------------------------*/
/* KitaBoardView actions */

/* private */
void KitaBoardTabWidget::setupActions()
{
    new KAction( i18n( "Find" ),
                 "find",
                 KStdAccel::find(),
                 this,
                 SLOT( slotFocusSearchCombo() ),
                 actionCollection(),
                 "subjectview_find" );

    new KAction( i18n( "Reload" ),
                 "reload",
                 KStdAccel::reload(),
                 this,
                 SLOT( slotReloadButton() ),
                 actionCollection(),
                 "subjectview_reload" );

    new KAction( i18n( "New Thread" ),
                 0,
                 this,
                 SLOT( slotNewThread() ),
                 actionCollection(),
                 "subjectview_newthread" );

    new KAction( i18n( "Show Old Logs" ),
                 0,
                 this,
                 SLOT( slotShowOldLogs() ),
                 actionCollection(),
                 "subjectview_showoldlogs" );

    new KAction( i18n( "Show Title Image" ),
                 0,
                 this,
                 SLOT( slotShowTitleImg() ),
                 actionCollection(),
                 "subjectview_showtitleimg" );
}

/* public slot */
void KitaBoardTabWidget::slotReloadButton()
{
    KitaBoardView * view = isSubjectView( currentPage() );
    if ( view ) view->reloadSubject();
}

/* public slot */
void KitaBoardTabWidget::slotFocusSearchCombo()
{
    KitaBoardView * view = isSubjectView( currentPage() );
    if ( view ) view->slotFocusSearchCombo();
}

/* public slot */
void KitaBoardTabWidget::slotNewThread()
{
    KitaBoardView * view = isSubjectView( currentPage() );
    if ( view ) view->slotCreateNewThread();
}


/* public slot */
void KitaBoardTabWidget::slotShowOldLogs( int idx )
{
    KitaBoardView * view;
    if ( idx == -1 ) view = isSubjectView( currentPage() );
    else view = isSubjectView( page( idx ) );
    if ( view ) view->toggleShowOldLogs();
}

/* public slot */
void KitaBoardTabWidget::slotShowTitleImg( int idx )
{
    KitaBoardView * view;
    if ( idx == -1 ) view = isSubjectView( currentPage() );
    else view = isSubjectView( page( idx ) );
    if ( view ) {
        KURL imgURL = Kita::BoardManager::titleImgURL( view->boardURL() );
        if ( !imgURL.isEmpty() ) {
            emit openURLRequestExt( imgURL, KParts::URLArgs(), QString::null, 1, view->boardURL() );
        }
    }
}


/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/



SubjectTabBar::SubjectTabBar( QWidget* parent, const char* name )
        : KitaTabBarBase( parent, name )
{}

SubjectTabBar::~SubjectTabBar()
{}


/* private */ /* virtual */
void SubjectTabBar::showPopupMenu( int idx, QPoint global )
{
    enum{
        MENU_CLOSE,
        MENU_CLOSEOTHER,
        MENU_CLOSELEFT,
        MENU_CLOSERIGHT,
        MENU_OPENBROWSER,
        MENU_COPYTITLE,
        MENU_SHOWTITLEIMG,
        MENU_SHOWOLDLOGS
    };

    KitaBoardTabWidget* tabwidget = static_cast<KitaBoardTabWidget*>( parentWidget() );
    KActionCollection * collection = tabwidget->actionCollection();

    if ( QString::compare( tabwidget->page( idx ) ->name(), "favoriteTab" ) == 0 ) return ;

    KPopupMenu* popup = new KPopupMenu( this );
    popup->clear();

    popup->insertItem( i18n( "Close this tab" ) , MENU_CLOSE );
    collection->action( "tab_prevtab" ) ->plug( popup );
    collection->action( "tab_nexttab" ) ->plug( popup );
    popup->insertSeparator();

    popup->insertItem( i18n( "Close Other Tabs" ) , MENU_CLOSEOTHER );
    popup->insertItem( i18n( "Close right tabs" ) , MENU_CLOSERIGHT );
    popup->insertItem( i18n( "Close left tabs" ) , MENU_CLOSELEFT );
    collection->action( "tab_closealltab" ) ->plug( popup );

    popup->insertSeparator();
    popup->insertItem( i18n( "Show Old Logs" ) , MENU_SHOWOLDLOGS );
    popup->insertItem( i18n( "Open with Web Browser" ), MENU_OPENBROWSER );
    popup->insertItem( i18n( "Copy title and URL" ), MENU_COPYTITLE );
    popup->insertItem( i18n( "Show Title Image" ) , MENU_SHOWTITLEIMG );

    popup->insertSeparator();
    collection->action( "tab_configkeys" ) ->plug( popup );

    KitaBoardView* subjectView = static_cast<KitaBoardView *>( tabwidget->page( idx ) );
    QClipboard* clipboard = QApplication::clipboard();

    int ret = popup->exec( global );
    delete popup;

    switch ( ret ) {
    case MENU_CLOSE: tabwidget->slotCloseTab( idx ); break;
    case MENU_CLOSEOTHER: tabwidget->slotCloseOtherTab( idx ); break;
    case MENU_CLOSERIGHT: tabwidget->slotCloseRightTab( idx ); break;
    case MENU_CLOSELEFT: tabwidget->slotCloseLeftTab( idx ); break;
    case MENU_SHOWOLDLOGS: tabwidget->slotShowOldLogs( idx ); break;
    case MENU_SHOWTITLEIMG: tabwidget->slotShowTitleImg( idx ); break;
    case MENU_OPENBROWSER:
        KRun::runURL( subjectView->boardURL(), "text/html" );
        break;
    case MENU_COPYTITLE:
        QString cliptxt = Kita::BoardManager::boardName( subjectView->boardURL() ) + "\n" + subjectView->boardURL().prettyURL();
        clipboard->setText( cliptxt , QClipboard::Clipboard );
        clipboard->setText( cliptxt , QClipboard::Selection );
        break;
    }
}



/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/
/*--------------------------------------------------------------*/


KitaBoardDock::KitaBoardDock( KDockManager* dockManager,
                                  const char* name,
                                  const QPixmap &pixmap,
                                  QWidget* parent,
                                  const QString& strCaption,
                                  const QString& strTabPageLabel,
                                  WFlags f )
    : KitaDockWidgetBase( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f ) {}


KitaBoardDock::~KitaBoardDock()
{
    if ( m_boardTab ) delete m_boardTab;
}


/* setup & create KitaBoardDock */ /* public */
KitaBoardTabWidget* KitaBoardDock::setup()
{
    /* create KitaNaviTabWidget */
    m_boardTab = new KitaBoardTabWidget( this, "Subject" );
    setWidget( m_boardTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( switchToSubject() ),
             SLOT( slotShowDock() ) );

    return m_boardTab;
}

/* public slot */
void KitaBoardDock::slotLoadBoard( const KURL& boardURL, bool withNewTab )
{
    slotShowDock();

    if ( m_boardTab ) m_boardTab->loadBoard( boardURL, withNewTab );
}
