/***************************************************************************
*   Copyright (C) 2004 by Kita Developers                                 *
*   ikemo@users.sourceforge.jp                                            *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
***************************************************************************/

#include "libkita/kita_misc.h"
#include "libkita/signalcollection.h"
#include "libkita/datmanager.h"
#include "libkita/boardmanager.h"
#include "writetabwidget.h"
#include "writeview.h"

#include <kdebug.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <klocale.h>
#include <kpopupmenu.h>
#include <kapplication.h>

#include <qmessagebox.h>
#include <qclipboard.h>

/*--------------------------------------------------------------------------------*/


KitaWriteTabWidget::KitaWriteTabWidget( QWidget* parent, const char* name, WFlags f )
        : KitaTabWidgetBase( parent, name, f )
{
    setXMLFile( "writetabwidgetui.rc" );
    setTabBar( new KitaWriteTabBar( this ) );

    connectSignals();
    setupActions();
}


KitaWriteTabWidget::~KitaWriteTabWidget() {}


/* public slot */
void KitaWriteTabWidget::slotShowWriteView( const KURL& url,
        const QString& resStr )
{
    openWriteView( WRITEMODE_NORMAL, url, resStr, QString::null );
}


/* public slot */
void KitaWriteTabWidget::slotCreateNewThread( const KURL& url,
        const QString& resStr,
        const QString& subject )
{
    openWriteView( WRITEMODE_NEWTHREAD, url, resStr, subject );
}


/* private */
void KitaWriteTabWidget::openWriteView( int mode, const KURL& url,
                                        const QString& resStr, const QString& subject )
{
    // TODO: machiBBS kakiko support.
    if ( Kita::BoardManager::type( url ) == Kita::Board_MachiBBS ) {
        QMessageBox::warning( this, QString( "<(_ _)>" ),
                              i18n( "Can't write to machi BBS in this version." ) );
        return ;
    }

    /* view exists */
    KitaWriteView* view = findWriteView( url );
    if ( view ) {

        if ( view->body().length() ) {

            if ( QMessageBox::warning( this, "Kita",
                                       i18n( "Do you want to clear the text?" ),
                                       QMessageBox::Ok,
                                       QMessageBox::Cancel | QMessageBox::Default ) == QMessageBox::Cancel ) return ;
        }

        /* clear */
        view->setMessage( resStr );
        setCurrentPage( indexOf( view ) );
        return ;
    }

    /* create new write view & add it to tab */
    QString threadName;
    KitaWriteView* new_dlg;
    if ( mode == WRITEMODE_NORMAL ) {

        /* write res */
        threadName = Kita::DatManager::threadName( url );
        new_dlg = new KitaWriteView( this, "write dialog", url, WRITEMODE_NORMAL );
    } else {

        /* create new thread */
        threadName = i18n( "New Thread" );
        new_dlg = new KitaWriteView( this, "write dialog", url, WRITEMODE_NEWTHREAD, subject );
    }

    new_dlg->setMessage( resStr );
    connect( new_dlg, SIGNAL( closeCurrentTab() ), SLOT( slotCloseCurrentTab() ) );
    addTab( new_dlg, threadName );
    showPage( new_dlg );
}


/* close view which URL is url. */ /* public slot */
void KitaWriteTabWidget::slotCloseWriteTab( const KURL& url )
{
    KitaWriteView * view = findWriteView( url );
    if ( view ) slotCloseTab( indexOf( view ) );
}


/* private */
void KitaWriteTabWidget::connectSignals()
{
    Kita::SignalCollection * signalCollection = Kita::SignalCollection::getInstance();

    connect( signalCollection, SIGNAL( activateThreadView( const KURL& ) ),
             this, SLOT( slotChangeWriteTab( const KURL& ) ) );
    connect( signalCollection, SIGNAL( closeWriteTab( const KURL& ) ),
             SLOT( slotCloseWriteTab( const KURL& ) ) );
}


/* private */
KitaWriteView* KitaWriteTabWidget::findWriteView( const KURL& url )
{
    KURL datURL = Kita::getDatURL( url );
    if ( datURL.isEmpty() ) return NULL;

    int max = count();
    if ( max == 0 ) return NULL;
    int i = 0;

    while ( i < max ) {
        KitaWriteView * view = isWriteView( page ( i ) );
        if ( view ) {
            if ( view->datURL() == datURL ) return view;
        }
        i++;
    }

    return NULL;
}


/* private */
KitaWriteView* KitaWriteTabWidget::isWriteView( QWidget* w )
{
    KitaWriteView * view = NULL;
    if ( w ) {
        if ( w->isA( "KitaWriteView" ) ) view = static_cast< KitaWriteView* >( w );
    }

    return view;
}



/* when thread view is focused, this slot is called    */
/* See also KitaThreadView::setFocus.                  */ /* private slot */
void KitaWriteTabWidget::slotChangeWriteTab( const KURL& url )
{
    KitaWriteView * view;
    int max = count();
    if ( max == 0 ) return ;

    /* disable all ok buttons. */
    int i = 0;
    while ( i < max ) {
        view = isWriteView( page( i ) );
        if ( view ) view->slotEnableWriting( FALSE );
        i++;
    }

    /* show current url page. */
    view = findWriteView( url );
    if ( view ) {
        static_cast< KitaDockWidgetBase* >( parentWidget() ) ->slotShowDock( FALSE, FALSE );
        if ( currentPage() != view ) setCurrentPage( indexOf( view ) );
        view->slotEnableWriting( TRUE );
    }
}


/* protected */ /* virtual */
void KitaWriteTabWidget::deleteWidget( QWidget* w )
{
    KitaWriteView * view = isWriteView( w );

    if ( view == NULL ) return ;

    if ( view->body().length() ) {
        if ( QMessageBox::warning( this, "Kita",
                                   i18n( "If you close this dialog, you will lose text.\n"
                                         "Do you want to close?" ),
                                   QMessageBox::Ok, QMessageBox::Cancel | QMessageBox::Default )
                == QMessageBox::Cancel ) return ;
    }

    KitaTabWidgetBase::deleteWidget( w );

    if ( count() == 0 ) {
        static_cast< KitaDockWidgetBase* >( parentWidget() ) ->slotHideDock();
    }
}


/*--------------------------------*/
/* KitaWriteTabWidget actions */


/* private */
void KitaWriteTabWidget::setupActions()
{
    new KAction( i18n( "quote clipboard" ),
                 Key_F2,
                 this,
                 SLOT( slotQuoteClipboard() ),
                 actionCollection(),
                 "writeview_quoteclip" );
}


/* public slot */
void KitaWriteTabWidget::slotQuoteClipboard()
{
    KitaWriteView * view = isWriteView( currentPage() );
    if ( view ) {
        QClipboard * clipboard = QApplication::clipboard();
        QString str = clipboard->text( QClipboard::Selection );
        if ( str == QString::null ) str = clipboard->text( QClipboard::Clipboard );
        if ( str != QString::null ) {

            QString msg = "\n> " + str.replace( "\n", "\n> " ) + "\n";
            view->insertMessage( msg );
        }
    }
}


/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/



KitaWriteTabBar::KitaWriteTabBar( QWidget* parent, const char* name ) : KitaTabBarBase( parent, name ) {}

KitaWriteTabBar::~KitaWriteTabBar() {}



/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/


/* Don't forget to call setup later ! */
KitaWriteDock::KitaWriteDock( KDockManager* dockManager,
                              const char* name,
                              const QPixmap &pixmap,
                              QWidget* parent,
                              const QString& strCaption,
                              const QString& strTabPageLabel,
                              WFlags f )
        : KitaDockWidgetBase( dockManager, name, pixmap, parent, strCaption, strTabPageLabel, f ), m_writeTab( 0 ) {}


KitaWriteDock::~KitaWriteDock() {}


KitaWriteTabWidget* KitaWriteDock::setup()
{
    if ( m_writeTab ) return NULL;
    m_writeTab = new KitaWriteTabWidget( this, "Write View" );
    setWidget( m_writeTab );

    Kita::SignalCollection* signalCollection = Kita::SignalCollection::getInstance();
    connect( signalCollection, SIGNAL( showWriteView( const KURL& , const QString& ) ),
             this, SLOT( slotShowWriteView( const KURL& , const QString& ) ) );
    connect( signalCollection, SIGNAL( createNewThread( const KURL& , const QString&, const QString& ) ),
             this, SLOT( slotCreateNewThread( const KURL& , const QString&, const QString& ) ) );
    connect( signalCollection, SIGNAL( switchToWritedock() ),
             SLOT( slotShowDock() ) );

    return m_writeTab;
}


void KitaWriteDock::slotShowWriteView( const KURL& url, const QString& resStr )
{
    if ( !m_writeTab ) return ;

    slotShowDock();
    m_writeTab->slotShowWriteView( url, resStr );
}


/* public slot */
void KitaWriteDock::slotCreateNewThread( const KURL& url,
        const QString& resStr,
        const QString& subject )
{
    if ( !m_writeTab ) return ;

    slotShowDock();
    m_writeTab->slotCreateNewThread( url, resStr, subject );
}
