/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// effect.cpp

#include "effect.h"

#include <qdom.h>


namespace KoverArtist
{


struct TypeStringTab
{
   Effect::Type type;
   const char* name;
};

static const TypeStringTab sTypeStringTab[] =
{ { Effect::NONE, "none" },
  { Effect::OUTLINE, "outline" },
  { Effect::GLOW, "glow" },
  { Effect::SHADOW, "shadow" },
  { Effect::BOX, "box" },
  { Effect::ROUNDED_BOX, "rounded-box" },
  { (Effect::Type)-1, 0 }
};


Effect::Effect()
:mType(Effect::NONE)
,mColor1("#888888")
,mColor2(Qt::black)
,mTransparency(100)
,mSize(10)
{
}


Effect::Effect(Effect::Type aType, const QColor& aColor1, const QColor& aColor2,
               int aSize, int aTransparency)
:mType(aType)
,mColor1(aColor1)
,mColor2(aColor2)
,mTransparency(aTransparency)
,mSize(aSize)
{
}


Effect::Effect(const Effect& o)
:mType(o.mType)
,mColor1(o.mColor1)
,mColor2(o.mColor2)
,mTransparency(o.mTransparency)
,mSize(o.mSize)
{
}


Effect& Effect::operator=(const Effect& o)
{
   if (&o!=this)
   {
      mType = o.mType;
      mColor1 = o.mColor1;
      mColor2 = o.mColor2;
      mTransparency = o.mTransparency;
      mSize = o.mSize;
   }
   return *this;
}


void Effect::clear()
{
   *this = Effect();
}


const char* Effect::typeString(Type aType)
{
   for (int i=0; sTypeStringTab[i].name; ++i)
      if (sTypeStringTab[i].type==aType)
         return sTypeStringTab[i].name;
   return "none";
}


Effect::Type Effect::typeFromString(const char* aName)
{
   for (int i=0; sTypeStringTab[i].name; ++i)
      if (strcasecmp(sTypeStringTab[i].name,aName)==0)
         return sTypeStringTab[i].type;
   return Effect::NONE;
}


void Effect::toDom(QDomDocument&, QDomElement& aElem) const
{
   aElem.setAttribute("type", typeString());
   aElem.setAttribute("color1", mColor1.name());
   aElem.setAttribute("color2", mColor2.name());
   aElem.setAttribute("transparency", mTransparency);
   aElem.setAttribute("size", mSize);
}


bool Effect::fromDom(const QDomElement& aElem)
{
   mType = typeFromString(aElem.attribute("type"));
   mColor1 = QColor(aElem.attribute("color1"));
   mColor2 = QColor(aElem.attribute("color2"));

   mTransparency = aElem.attribute("transparency").toInt();
   if (mTransparency<0) mTransparency = 0;
   else if (mTransparency>255) mTransparency = 255;

   mSize = aElem.attribute("size").toInt();
   if (mSize<0) mSize = 0;
   else if (mSize>100) mSize = 100;

   return true;
}


} //namespace
