/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// effectconfig.cpp

#include "effectconfig.h"
#include "effect.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qslider.h>
#include <qcombobox.h>
#include <qtooltip.h>

#include <kcolorbutton.h>
#include <klocale.h>


EffectConfig::EffectConfig(KoverArtist::Effect* aEffect, QWidget* aParent,
                           const char* aName)
:Inherited(aParent, aName)
,mEffect(aEffect)
{
   QWidget *base = this;
   QHBoxLayout *box = 0;
   QLabel *lbl;

   QVBoxLayout *vbox = new QVBoxLayout(base, 4, 2);

   box = new QHBoxLayout(vbox, 2);

   mCboType = new QComboBox(base, "cbo-type");
   mCboType->insertItem(i18n("No Effect"), int(KoverArtist::Effect::NONE));
   mCboType->insertItem(i18n("Outline"), int(KoverArtist::Effect::OUTLINE));
   mCboType->insertItem(i18n("Glow"), int(KoverArtist::Effect::GLOW));
   mCboType->insertItem(i18n("Shadow"), int(KoverArtist::Effect::SHADOW));
   mCboType->insertItem(i18n("Box"), int(KoverArtist::Effect::BOX));
//    mCboType->insertItem(i18n("Rounded Box"), int(KoverArtist::Effect::ROUNDED_BOX));
   mCboType->setFixedHeight(mCboType->sizeHint().height());
   connect(mCboType, SIGNAL(activated(int)), base, SLOT(setType(int)));
   box->addWidget(mCboType);

   mBtnColor1 = new KColorButton(Qt::white, base, "btn-color1");
   mBtnColor1->setFixedSize(mBtnColor1->sizeHint());
   connect(mBtnColor1, SIGNAL(changed(const QColor&)), SLOT(setColor1(const QColor&)));
   QToolTip::add(mBtnColor1, i18n("Change the primary effect color"));
   box->addWidget(mBtnColor1);

   mBtnColor2 = new KColorButton(Qt::white, base, "btn-color2");
   mBtnColor2->setFixedSize(mBtnColor2->sizeHint());
   connect(mBtnColor2, SIGNAL(changed(const QColor&)), SLOT(setColor2(const QColor&)));
   QToolTip::add(mBtnColor2, i18n("Change the secondary effect color"));
   box->addWidget(mBtnColor2);

   box = new QHBoxLayout(vbox, 2);

   mLblSize = new QLabel(i18n("Size")+":", base);
   mLblSize->setFixedSize(mLblSize->sizeHint());
   box->addWidget(mLblSize);

   mLblSizeVal = new QLabel("999", base);
   mLblSizeVal->setFixedSize(mLblSizeVal->sizeHint());
   box->addWidget(mLblSizeVal);

   box->addSpacing(4);

   mSldSize = new QSlider(0, 40, 5, 0, Qt::Horizontal, base);
   connect(mSldSize, SIGNAL(valueChanged(int)), this, SLOT(setSize(int)));
   connect(mSldSize, SIGNAL(sliderReleased()), this, SLOT(sendChanged()));
   box->addWidget(mSldSize);

   box = new QHBoxLayout(vbox, 2);

   mLblTrans = new QLabel(i18n("Transparency")+":", base);
   mLblTrans->setFixedSize(mLblTrans->sizeHint());
   box->addWidget(mLblTrans);

   mLblTransVal = new QLabel("999", base);
   mLblTransVal->setFixedSize(mLblTransVal->sizeHint());
   box->addWidget(mLblTransVal);

   box->addSpacing(4);

   mSldTrans = new QSlider(0, 255, 32, 0, Qt::Horizontal, base);
   connect(mSldTrans, SIGNAL(valueChanged(int)), this, SLOT(setTrans(int)));
   connect(mSldTrans, SIGNAL(sliderReleased()), this, SLOT(sendChanged()));
   box->addWidget(mSldTrans);

   vbox->addStretch(1000);
   vbox->activate();

   update();
}


EffectConfig::~EffectConfig()
{
}


void EffectConfig::update()
{
   bool upd = isUpdatesEnabled();
   setUpdatesEnabled(false);

   mCboType->setCurrentItem(int(mEffect->type()));
   setType(int(mEffect->type()));

   mBtnColor1->setColor(mEffect->color1());
   mBtnColor2->setColor(mEffect->color2());
   mSldSize->setValue(mEffect->size());
   mLblSizeVal->setText(QString("%1").arg(mEffect->size()));
   mSldTrans->setValue(mEffect->transparency());
   mLblTransVal->setText(QString("%1").arg(mEffect->transparency()));

   setUpdatesEnabled(upd);
}


void EffectConfig::setType(int aType)
{
   bool act = (aType!=0);
   mBtnColor1->setEnabled(act);
   mBtnColor2->setEnabled(act);
   mLblSize->setEnabled(act);
   mLblSizeVal->setEnabled(act);
   mSldSize->setEnabled(act);
   mLblTrans->setEnabled(act);
   mLblTransVal->setEnabled(act);
   mSldTrans->setEnabled(act);

   if (aType!=mEffect->type())
   {
      mEffect->setType((KoverArtist::Effect::Type)aType);
      if (isUpdatesEnabled()) emit changed();
   }
}


void EffectConfig::setColor1(const QColor& c)
{
   if (mEffect->color1()==c) return;
   mEffect->setColor1(c);
   if (isUpdatesEnabled()) emit changed();
}


void EffectConfig::setColor2(const QColor& c)
{
   if (mEffect->color2()==c) return;
   mEffect->setColor2(c);
   if (isUpdatesEnabled()) emit changed();
}


void EffectConfig::setSize(int x)
{
   if (x==mEffect->size()) return;
   mEffect->setSize(x);
   mLblSizeVal->setText(QString("%1").arg(x));
}


void EffectConfig::setTrans(int x)
{
   if (x==mEffect->transparency()) return;
   mEffect->setTransparency(x);
   mLblTransVal->setText(QString("%1").arg(x));
}


void EffectConfig::sendChanged()
{
   if (isUpdatesEnabled()) emit changed();
}

#include "effectconfig.moc"
