# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###




import gobject

try: 
    import dbus
    dbus_version = getattr(dbus, 'version', (0,0,0))
    if dbus_version >= (0,80,0):
        from dbus.mainloop.glib import DBusGMainLoop
        DBusGMainLoop(set_as_default=True)
        dbus.SystemBus()
    else:
        print "DBus version too old"

except: dbus_imported = False
else: dbus_imported=True

import utils
from logger import Logger
from library import ListenDB


DIFFERD_HAL_STARTUP_TIME=2000


class Hal(gobject.GObject,Logger):
    __gsignals__ = {
        "volume-mounted" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str,)),
        "volume-umounted" : (gobject.SIGNAL_RUN_LAST,
                gobject.TYPE_NONE,
                (str,)),
        }

    def __init__(self):
        gobject.GObject.__init__(self)
        self.__volumes = set()

        if dbus_imported:
            try:self.bus = dbus.SystemBus()
            except:self.bus=None

        if dbus_imported and self.bus and utils.dbus_service_available(self.bus,'org.freedesktop.Hal'):
            self.dbus_hal = True
            self.hal_object = self.bus.get_object('org.freedesktop.Hal','/org/freedesktop/Hal/Manager')
            self.hal_interface = dbus.Interface(self.hal_object, 'org.freedesktop.Hal.Manager')

            if ListenDB.isloaded():
                self.__connect_to_db()
            else:
                ListenDB.connect("loaded",self.__connect_to_db)
    
        else:
            self.dbus_hal = False
            self.logdebug("No Hal support")

    def __connect_to_db(self,*args,**kwargs):
        gobject.timeout_add(DIFFERD_HAL_STARTUP_TIME, self.__post_load)

    @utils.threaded
    def __post_load(self):
        self.hal_interface.connect_to_signal('DeviceAdded', self.__device_added)
        self.hal_interface.connect_to_signal('DeviceRemoved', self.__device_removed)
        udis = self.hal_interface.FindDeviceByCapability("volume")
        for udi in udis: self.__device_added(udi,True)

    def umount(self,udi):
        dev_obj = self.bus.get_object('org.freedesktop.Hal', udi)
        dev = dbus.Interface(dev_obj, 'org.freedesktop.Hal.Device.Volume')
        dev.Unmount([])

    def eject(self,udi):
        dev_obj = self.bus.get_object('org.freedesktop.Hal', udi)
        dev = dbus.Interface(dev_obj, 'org.freedesktop.Hal.Device.Volume')
        dev.Eject([])

    def __device_added(self,udi,emit_signal=True):
        dev_obj = self.bus.get_object('org.freedesktop.Hal', udi)
        dev = dbus.Interface(dev_obj, 'org.freedesktop.Hal.Device')
        if dev.QueryCapability('volume'):
            if dev.PropertyExists('volume.ignore') and dev.GetProperty('volume.ignore') : return
            dev.connect_to_signal('PropertyModified', lambda nb, changes: self.__on_property_modified(udi,nb,changes) )
            if dev.GetProperty('volume.is_mounted'):
                self.__volumes.add(udi)
                self.loginfo("mounted volume found: %s",udi)
                if emit_signal: self.emit("volume-mounted",udi)

    def __device_removed(self,udi):
        if udi in self.__volumes:
            self.__volumes.remove(udi)
            self.loginfo("device removed: %s",udi)
            self.emit("volume-umounted",udi)

    def __on_property_modified(self, udi, nb_change, changes):
        for property, _added, _removed in changes:
            if property == "volume.is_mounted":
                value = self.get_property(udi,property)
                if udi in self.__volumes and not value:
                    self.loginfo("device removed: %s",udi)
                    self.__volumes.remove(udi)
                    self.emit("volume-umounted",udi)
                elif udi not in self.__volumes and value:
                    self.loginfo("mounted volume found: %s",udi)
                    self.__volumes.add(udi)
                    self.emit("volume-mounted",udi)

    def get_mounted_devices_udi(self):
        if not self.dbus_hal: return []
        else: return self.__volumes

    def get_mount_point(self,udi):  
        return self.get_property(udi,"volume.mount_point")
    
    def get_device_path(self,udi):  
        return self.get_property(udi,"block.device")

    def get_property(self,udi,property):
        if not self.dbus_hal: return None
        try:
            dev_obj = self.bus.get_object("org.freedesktop.Hal" ,udi)
            dev = dbus.Interface(dev_obj, 'org.freedesktop.Hal.Device')
            return dev.GetProperty(property)
        except:
            return None

    def get_properties(self,udi):
        if not self.dbus_hal: return {}
        try:
            dev_obj = self.bus.get_object("org.freedesktop.Hal" ,udi)
            dev = dbus.Interface(dev_obj, 'org.freedesktop.Hal.Device')
            return dev.GetAllProperties()
        except:
            return {}
    
hal = Hal()

