/*
 * names.c: This here is used to maintain a list of all the people currently
 * on your channel.  Seems to work 
 *
 * Written By Michael Sandrof
 *
 * Copyright (c) 1990 Michael Sandrof.
 * Copyright (c) 1991, 1992 Troy Rollo.
 * Copyright (c) 1992-2000 Matthew R. Green.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * renamed to channels.c and extensively modified by
 * Joshua J. Drake
 */

#include "irc.h"
IRCII_RCSID("@(#)$Id: names.c,v 1.81 2000/08/31 12:10:52 mrg Exp $");

#include "dma.h"
#include "ircaux.h"
#include "channels.h"
#include "window.h"
#include "screen.h"
#include "server.h"
#include "lastlog.h"
#include "list.h"
#include "output.h"
#include "notify.h"
#include "vars.h"

#include "nicks.h"
#include "bans.h"
#include "ckey.h"
#include "friends.h"
#include "ninjacmd.h"

/* channel identifiers */
#define STRING_CHANNEL '+'
#define MULTI_CHANNEL '#'
#define LOCAL_CHANNEL '&'
#define SAFE_CHANNEL '!'

/* from channels.h */
static	u_char	mode_str[] = MODE_STRING;

static	void	free_channel _((Channel **));
static	void	show_channel _((Channel *));
static	void	clear_channel _((Channel *));
static	u_char	*recreate_mode _((Channel *));
static	int	decifer_mode _((u_char *, Channel *, u_char *));

/* ninja extensions */
static	void	clear_channel_nicks _((Channel *));
static	void	clear_channel_bans _((Channel *));
static	void	clear_channel_excepts _((Channel *));
static	void	clear_channel_denys _((Channel *));

/* clear_channel: erases all entries in a nick list for the given channel */
/* in ninja this also removes other channel information */
static	void
clear_channel(chan)
	Channel *chan;
{
   clear_channel_nicks(chan);
   clear_channel_bans(chan);
   clear_channel_excepts(chan);
   clear_channel_denys(chan);
}

/* clear channel nicks */
static	void
clear_channel_nicks(chan)
    	Channel *chan;
{
   Nick *tmp, *next;
   
   for (tmp = chan->nicks; tmp; tmp = next)
     {
	next = tmp->next;
	free_nick(&tmp);
     }
   chan->nicks = (Nick *) 0;
}

/* clear channel bans */
static	void
clear_channel_bans(chan)
    	Channel *chan;
{
   Ban *tmp, *next;
   
   for (tmp = chan->bans; tmp; tmp = next)
     {
	next = tmp->next;
	new_free(&(tmp->ban));
	new_free(&(tmp->owner));
	new_free(&tmp);
     }
   chan->bans = (Ban *) 0;
}

/* clear channel exceptions */
static	void
clear_channel_excepts(chan)
    	Channel *chan;
{
   Except *tmp, *next;
   
   for (tmp = chan->excepts; tmp; tmp = next)
     {
	next = tmp->next;
	new_free(&(tmp->exception));
	new_free(&(tmp->owner));
	new_free(&tmp);
     }
   chan->excepts = (Except *) 0;
}

/* clear channel deny lines */
static	void
clear_channel_denys(chan)
    	Channel *chan;
{
   Deny *tmp, *next;
   
   for (tmp = chan->denys; tmp; tmp = next)
     {
	next = tmp->next;
	new_free(&(tmp->regexp));
	new_free(&(tmp->owner));
	new_free(&tmp);
     }
   chan->denys = (Deny *) 0;
}

/*
 * we need this to deal with !channels.
 */
int
same_channel(channel, chan2)
	Channel *channel;
	u_char	*chan2;
{
	size_t	len, len2;

	/* take the easy way out */
	if (*channel->channel != '!' && *chan2 != '!')
		return (!my_stricmp(channel->channel, chan2));

	/*
	 * OK, so what we have is channel = "!fooo" and chan2 = "!JUNKfoo".
	 */
	len = my_strlen(channel->channel);	
	len2 = my_strlen(chan2);	

	/* bail out on known stuff */
	if (len > len2)
		return (0);
	if (len == len2)
		return (!my_stricmp(channel->channel, chan2));

	/*
	 * replace the channel name if we are the same!
	 */
	if (!my_stricmp(channel->channel + 1, chan2 + 1 + (len2 - len)))
	{
		malloc_strcpy(&channel->channel, chan2);
		return (1);
	}
	return (0);
}

extern	Channel *
lookup_channel(channel, server, do_unlink)
	u_char	*channel;
	int	server;
	int	do_unlink;
{
   Channel	*chan, *last = (Channel *) 0;
   
   if (server == -1)
     server = primary_server;
   if (server < 0 || server >= number_of_servers)
       return (Channel *) 0;
   chan = server_list[server].chan_list;
   while (chan)
     {
	if (chan->server == server && same_channel(chan, channel))
	  {
	     if (do_unlink == CHAN_UNLINK)
	       {
		  if (last)
		    last->next = chan->next;
		  else
		    server_list[server].chan_list = chan->next;
	       }
	     break;
	  }
	last = chan;
	chan = chan->next;
     }
   return chan;
}

/*
 * add_channel: adds the named channel to the channel list.  If the channel
 * is already in the list, its attributes are modified accordingly with the
 * connected and copy parameters.
 */
void
add_channel(channel, server, connected, copy)
	u_char	*channel;
	int	server;
	int	connected;
	Channel *copy;
{
   Channel *new;
   int	do_add = 0;
   /* u_char *tkey; */
   
   /*
    * avoid adding channel "0"
    */
   if (channel[0] == '0'&& channel[1] == '\0')
     return;
   /* also avoid adding non-channels, 
    * if this is triggered, whatever is calling this is buggy 
    */
   if (!is_channel(channel))
     {
	put_error("add_channel() trying to add channel \"%s\".. notify ninjairc@qoop.org",
		  channel);
	return;
     }

   if ((new = lookup_channel(channel, server, CHAN_NOUNLINK)) == (Channel *) 0)
     {
	new = (Channel *) new_malloc(sizeof(Channel));
	/* setting stuff to 0 is pointless because..
	 * dma_Malloc does memset(mem, 0, sizeof(mem))
	 * 
	new->channel = (u_char *) 0;
	new->status = 0;
	new->key = (u_char *) 0;
	new->nicks = (Nick *) 0;
	new->mode = 0;
	new->limit = 0;
	 */
	new->s_mode = empty_string;
	
	/*
	 * not sure if this is needed..
	 * 
	tkey = get_ckey(channel);
	if (tkey)
	  {
	     dma_strcpy(&new->key, tkey);
	 // dont do this, let the server do it!
	     // new->mode |= MODE_KEY;
	  }
	 */
	malloc_strcpy(&new->channel, channel);
	if ((new->window = is_bound(channel, server)) == (Window *) 0)
	  new->window = curr_scr_win;
	do_add = 1;
	add_to_list((List **) &server_list[server].chan_list, (List *) new);
     }
   else
     {
	if (new->connected != CHAN_LIMBO && new->connected != CHAN_JOINING)
	  yell("--- add_channel: add_channel found channel not CHAN_LIMBO/JOINING: %s", new->channel);
     }
   
   if (do_add || (connected == CHAN_JOINED))
     {
#ifdef SHOW_JOIN_TIME
	gettimeofday(&new->join_time, NULL);
#endif
	new->server = server;
	clear_channel(new);
     }
   if (copy)
     {
	new->mode = copy->mode;
	new->limit = copy->limit;
	new->window = copy->window;
	malloc_strcpy(&new->key, copy->key);
     }
   new->connected = connected;
   if ((connected == CHAN_JOINED) && !is_current_channel(channel, server, 0))
     {
	int	flag = 1;
	Window	*tmp, *expected, *possible = (Window *) 0;
	
	expected = new->window;
	
	while ((tmp = traverse_all_windows(&flag)))
	  {
	     if (tmp->server == server)
	       {
		  if (tmp == expected)
		    {	
		       set_channel_by_refnum(tmp->refnum, channel);
		       new->window = tmp;
		       update_all_status();
		       return;
		    }
		  else if (!possible)
		    possible = tmp;
	       }
	  }
	if (possible)
	  {
	     set_channel_by_refnum(possible->refnum, channel);
	     new->window = possible;
	     update_all_status();
	     return;
	  }
	set_channel_by_refnum(0, channel);
	new->window = curr_scr_win;
     }
   update_all_windows();
}

/*
 * add_to_channel: adds the given nickname to the given channel.  If the
 * nickname is already on the channel, nothing happens.  If the channel is
 * not on the channel list, nothing happens (although perhaps the channel
 * should be addded to the list?  but this should never happen)
 * 
 * this will return the channel pointer, if you don't need it cast the call
 * to void.
 */
Channel *
add_to_channel(channel, nick, server, 
	       oper, chop, voice, 
	       user, host, real, 
	       hops, here, serv_str)
	u_char	*channel;
	u_char	*nick;
	int	server;
	int	oper;
	int	chop;
	int	voice;
/* ninja extensions! */
	u_char	*user;
	u_char	*host;
	u_char	*real;
	int	hops;
	int	here;
	u_char	*serv_str;
{
   Nick *new;
   Channel *chan;
   int ischop = chop;
   int hasvoice = voice;
   int isoper = oper;
   
   if (!nick)
     return NULL;
   if ((chan = lookup_channel(channel, server, CHAN_NOUNLINK)))
     {
	/* these two if statements should never
	 * be true with ninja using the WHO command
	 * to cache users instead of names
	 */
	if (*nick == '+')
	  {
	     hasvoice = 1;
	     nick++;
	  }
	if (*nick == '@')
	  {
	     nick++;
	     if (!my_stricmp(nick, get_server_nickname(server))
		 && !(chan->status & CHAN_MODE))
	       {
		  u_char	*mode = recreate_mode(chan);
		  
		  if (*mode)
		    {
		       int	old_server = from_server;
		       
		       from_server = server;
		       send_to_server("MODE %s %s", chan->channel, mode);
		       from_server = old_server;
		    }
		  chan->status |= CHAN_CHOP;
	       }
	     ischop = 1;
	  }
	
	if ((new = (Nick *) remove_from_list((List **) &(chan->nicks), nick)))
	  free_nick(&new);
	new = (Nick *) new_malloc(sizeof(Nick));
	/* dma_Malloc already bzero's data
	new->nick = (u_char *) 0;
	 */
	if (ischop == 1)
	  new->status |= NICK_CHOP;
	if (hasvoice == 1)
	  new->status |= NICK_VOICE;
	if (isoper == 1)
	  new->status |= NICK_OPER;
	if (here == 1)
	  new->status |= NICK_HERE;
	else if (here == 0)
	  new->status |= NICK_AWAY;
	new->hops = hops;
	malloc_strcpy(&(new->nick), nick);
	malloc_strcpy(&(new->user), user);
	malloc_strcpy(&(new->host), host);
	malloc_strcpy(&(new->realname), real);
	malloc_strcpy(&(new->server), serv_str);
	add_to_list((List **) &(chan->nicks), (List *) new);
     }
   if (here != -1) /* don't do this on names output, wait for who output */
     notify_mark(nick, NFY_ONLINE, !NFY_FROM_QRY, user, host, (new->status & NICK_AWAY));
   return chan;
}


/*
 * recreate_mode: converts the bitmap representation of a channels mode into
 * a string
 * 
 * there were some possible overflows in here.. fixed by jjd 9/25/01
 */
static	u_char	*
recreate_mode(chan)
	Channel *chan;
{
   int	mode_pos = 0, mode;
   static	u_char	*s;
   u_char	buffer[BIG_BUFFER_SIZE];
   int len, left;
   
   memset(buffer, 0, sizeof(buffer));	 /* ultra-paranoia */
   s = buffer;
   mode = chan->mode;
   while (mode && (s-buffer) < (sizeof(buffer)-1))
     {
	if (mode % 2)
	  *s++ = mode_str[mode_pos];
	mode /= 2;
	mode_pos++;
     }
   left = sizeof(buffer) - (s - buffer) - 1;
   if (chan->key
       && (chan->mode & MODE_KEY)
       && !get_int_var(HIDE_CHANNEL_KEYS_VAR)
       && left > 2)
     {
	*s++ = ' ';
	len = my_strlen(chan->key);
	my_strncpy(s, chan->key, MIN(left, len));
	s += len;
	left -= len;
     }
   if (chan->limit && left > 1)
     snprintf(CP(s), left, " %d", chan->limit);

   if (chan->s_mode == empty_string)
     chan->s_mode = NULL;
   malloc_strcpy(&chan->s_mode, buffer);
   return chan->s_mode;
}

/*
 * decifer_mode: This will figure out the mode string as returned by mode
 * commands and convert that mode string into a one byte bit map of modes 
 */
static	int
decifer_mode(mode_string, chan, from)
	u_char	*mode_string;
	Channel *chan;
	u_char	*from;
{
   u_char	*limit = 0;
   u_char	*person;
   int	add = 0;
   int	limit_set = 0;
   int	limit_reset = 0;
   u_char	*rest, *the_key;
   Nick *ThisNick;
   u_long	value = 0;
   u_char	*copy, *free_copy = (u_char *) 0;
   int old_status = 0;
   
   dma_strcpy(&free_copy, mode_string);
   copy = free_copy;
   if (!(copy = next_arg(copy, &rest)))
     {
	dma_Free(&free_copy);
	return -1;
     }
   for (; *copy; copy++)
     {
	switch (*copy)
	  {
	   case '+':
	     add = 1;
	     value = 0;
	     break;
	   case '-':
	     add = 0;
	     value = 0;
	     break;
/*
 * Ninja supports 2.8-hybird modes for now
 *
	   case 'a':
	     value = MODE_ANONYMOUS;
	     break;
	   case 'c':
	     value = MODE_COLOURLESS;
	     break;
 */
	   case 'i':
	     value = MODE_INVITE;
	     break;
	   case 'k':
	     value = MODE_KEY;
	     the_key = next_arg(rest, &rest);
	     if (add)
	       malloc_strcpy(&(chan->key), the_key);
	     else
	       new_free(&(chan->key));
	     add_ckey(chan->channel, chan->key);
	     break;	
	   case 'l':
	     value = MODE_LIMIT;
	     if (add)
	       {
		  limit_set = 1;
		  if (!(limit = next_arg(rest, &rest)))
		    limit = empty_string;
		  else if (0 == my_strcmp(limit, zero))
		    limit_reset = 1, limit_set = 0, add = 0;
	       }
	     else
	       limit_reset = 1;
	     break;
	   case 'm':
	     value = MODE_MODERATED;
	     break;
	   case 'o':
	     /* if its me, update my channel status */
	     if ((person = next_arg(rest, &rest)) && !my_stricmp(person, get_server_nickname(from_server))) {
		old_status = chan->status;
		if (add)
		  chan->status |= CHAN_CHOP;
		else
		  chan->status &= ~CHAN_CHOP;
	     }
	     /* the nick */
	     ThisNick = (Nick *) list_lookup((List **) &(chan->nicks), person, !USE_WILDCARDS, !REMOVE_FROM_LIST);
	     if (ThisNick)
	       {
		  if (add)
		    ThisNick->status |= NICK_CHOP;
		  else
		    ThisNick->status &= ~NICK_CHOP;
	       }
	     /* if im getting opped and the channel is sync'd, check friends */
	     if (add 
		 && my_stricmp(person, get_server_nickname(from_server)) == 0
		 && old_status != chan->status
		 && ((chan->status & CHAN_MODE)
		     && (chan->status & CHAN_WHO)))
	       friends_channel_sync(chan);
	     break;
	   case 'n':
	     value = MODE_MSGS;
	     break;
	   case 'p':
	     value = MODE_PRIVATE;
	     break;
/*	     
	   case 'q':
	     value = MODE_QUIET;
	     break;
	   case 'r':
	     value = MODE_REOP;
	     break;
 */
	   case 's':
	     value = MODE_SECRET;
	     break;
	   case 't':
	     value = MODE_TOPIC;
	     break;
	   case 'v':
	     person = next_arg(rest, &rest);
	     ThisNick = (Nick *) list_lookup((List **) &(chan->nicks), person, !USE_WILDCARDS, !REMOVE_FROM_LIST);
	     if (ThisNick)
	       {
		  if (add)
		    ThisNick->status |= NICK_VOICE;
		  else
		    ThisNick->status &= ~NICK_VOICE;
	       }
	     break;
	     /* ban cache */
	   case 'b':
	     person = next_arg(rest, &rest);
	     if (!person || !*person)
	       break;
	     if (add)
	       {
		  u_char tmpstr[32];
		  
		  snprintf(tmpstr, sizeof(tmpstr)-1, "%lu", (unsigned long)time(NULL));
		  tmpstr[sizeof(tmpstr)-1] = '\0';
		  add_ban(chan, person, from, tmpstr);
	       }
	     else
	       {
		  Ban *new;
		  if ((new = (Ban *) remove_from_list((List **) &(chan->bans), person)))
		    {
		       dma_Free(&(new->ban));
		       dma_Free(&(new->owner));
		       dma_Free(&new);
		    }
	       }
	     break;
	     /* exception cache */
	   case 'e':
	     person = next_arg(rest, &rest);
	     if (!person || !*person)
	       break;
	     if (add)
	       {
		  u_char tmpstr[32];
		  
		  snprintf(tmpstr, sizeof(tmpstr)-1, "%lu", (unsigned long)time(NULL));
		  tmpstr[sizeof(tmpstr)-1] = '\0';
		  /* add_exception(chan, person, from, tmpstr);
		   */
	       }
	     else
	       {
		  Except *new;
		  if ((new = (Except *) remove_from_list((List **) &(chan->excepts), person)))
		    {
		       dma_Free(&(new->exception));
		       dma_Free(&(new->owner));
		       dma_Free(&new);
		    }
	       }
	     break;
	     /* deny cache */
	   case 'd':
	     person = next_arg(rest, &rest);
	     if (!person || !*person)
	       break;
	     if (add)
	       {
		  u_char tmpstr[32];
		  
		  snprintf(tmpstr, sizeof(tmpstr)-1, "%lu", (unsigned long)time(NULL));
		  tmpstr[sizeof(tmpstr)-1] = '\0';
		  /* add_deny(chan, person, from, tmpstr);
		   */
	       }
	     else
	       {
		  Deny *new;
		  if ((new = (Deny *) remove_from_list((List **) &(chan->denys), person)))
		    {
		       dma_Free(&(new->regexp));
		       dma_Free(&(new->owner));
		       dma_Free(&new);
		    }
	       }
	     break;
	     /* misc unsupported modes w/arguments */
	   case 'I':
	   case 'O': /* this is a weird special case */
	     (void) next_arg(rest, &rest);
	     break;
/*	     
	   case 'R':
	     value = MODE_REGONLY;
	     break;
 */
	  }
	if (add)
	  chan->mode |= value;
	else
	  chan->mode &= ~value;
     }
   if (limit_set)
     limit_set = my_atoi(limit);
   else if (limit_reset)
     limit_set = 0;
   else
     limit_set = -1;
   dma_Free(&free_copy);
   return limit_set;
}

/*
 * get_channel_mode: returns the current mode string for the given channel
 */
u_char	*
get_channel_mode(channel, server)
	u_char	*channel;
	int	server;
{
   Channel *tmp;
   
   if ((tmp = lookup_channel(channel, server, CHAN_NOUNLINK)) && (tmp->status & CHAN_MODE))
     return recreate_mode(tmp);
   return empty_string;
}

/*
 * update_channel_mode: This will modify the mode for the given channel
 * according the the new mode given.  
 */
void
update_channel_mode(channel, server, mode, from, chan)
	u_char	*channel;
	int	server;
	u_char	*mode;
	u_char	*from;
	Channel *chan;
{
   Channel *tmp;
   int	limit;

   if (!chan && !channel)
     return;
   if (chan && chan->server == server)
     tmp = chan;
   else
     tmp = lookup_channel(channel, server, CHAN_NOUNLINK);
   if (!tmp)
     return;
   limit = decifer_mode(mode, tmp, from);
   if (limit != -1)
     tmp->limit = limit;
   update_all_status();
   
   /* check friend protection... */
   check_friend_protection(from, tmp, mode);
}

/*
 * is_channel_mode: returns the logical AND of the given mode with the
 * channels mode.  Useful for testing a channels mode
 * 
 */
int
is_channel_mode(channel, mode, server_index)
	u_char	*channel;
	int	mode;
	int	server_index;
{
   Channel *tmp;
   
   if ((tmp = lookup_channel(channel, server_index, CHAN_NOUNLINK)))
     return (tmp->mode & mode);
   return 0;
}

static	void
free_channel(channel)
	Channel **channel;
{
   clear_channel(*channel);
   new_free(&(*channel)->channel);
   new_free(&(*channel)->key);
   if ((*channel)->s_mode != empty_string)
     new_free(&(*channel)->s_mode);
   new_free(&(*channel));
}

/*
 * remove_channel: removes the named channel from the
 * server_list[server].chan_list.  If the channel is not on the
 * server_list[server].chan_list, nothing happens.  If the channel was
 * the current channel, this will select the top of the
 * server_list[server].chan_list to be the current_channel, or 0 if the
 * list is empty. 
 */
void
remove_channel(channel, server)
	u_char	*channel;
	int	server;
{
   Channel *tmp;
   
   if (channel)
     {
	int refnum = -1;
	
	if ((tmp = lookup_channel(channel, server, CHAN_UNLINK)))
	  {
	     if (tmp->window)
	       refnum = tmp->window->refnum;
	     free_channel(&tmp);
	  }
	
	(void)is_current_channel(channel, server, refnum);
     }
   else
     {
	Channel *next;
	
	for (tmp = server_list[server].chan_list; tmp; tmp = next)
	  {
	     next = tmp->next;
	     free_channel(&tmp);
	  }
	server_list[server].chan_list = (Channel *) 0;
     }
   update_all_windows();
}

/*
 * remove_from_channel: removes the given nickname from the given channel. If
 * the nickname is not on the channel or the channel doesn't exist, nothing
 * happens. 
 */
void
remove_from_channel(channel, nick, server)
	u_char	*channel;
	u_char	*nick;
	int	server;
{
   Channel *chan;
   Nick *tmp;
   
   if (channel)
     {
	if ((chan = lookup_channel(channel, server, CHAN_NOUNLINK)))
	  {
	     if ((tmp = (Nick *) list_lookup((List **) &(chan->nicks), nick, !USE_WILDCARDS, REMOVE_FROM_LIST)))
	       free_nick(&tmp);
	  }
     }
   else
     {
	for (chan = server_list[server].chan_list; chan; chan = chan->next)
	  {
	     if ((tmp = (Nick *) list_lookup((List **) &(chan->nicks), nick, !USE_WILDCARDS, REMOVE_FROM_LIST)))
	       free_nick(&tmp);
	  }
     }
}

/*
 * rename_nick: in response to a changed nickname, this looks up the given
 * nickname on all you channels and changes it the new_nick 
 */
void
rename_nick(old_nick, new_nick, server)
	u_char	*old_nick,
		*new_nick;
	int	server;
{
   Channel *chan;
   Nick *tmp;
   
   for (chan = server_list[server].chan_list; chan; chan = chan->next)
     {
	if ((chan->server == server))
	  {
	     if ((tmp = (Nick *) list_lookup((List **) &chan->nicks, old_nick, !USE_WILDCARDS, !REMOVE_FROM_LIST)))
	       {
		  /* malloc_strcpy calls free if the pointer is non-null
		  new_free(&tmp->nick);
		   */
		  malloc_strcpy(&tmp->nick, new_nick);
	       }
	  }
     }
}

/*
 * is_on_channel: returns true if the given nickname is in the given channel,
 * false otherwise.  Also returns false if the given channel is not on the
 * channel list. 
 */
int
is_on_channel(channel, server, nick)
	u_char	*channel;
	int	server;
	u_char	*nick;
{
   Channel *chan;
   
   chan = lookup_channel(channel, server, CHAN_NOUNLINK);
   if (chan && (chan->connected == CHAN_JOINED)
       /* channel may be "surviving" from a server disconnect/reconnect,
	make sure it's connected -Sol */
       && list_lookup((List **) &(chan->nicks), nick, !USE_WILDCARDS, !REMOVE_FROM_LIST))
     return 1;
   return 0;
}

int
is_chanop(channel, nick)
	u_char	*channel;
	u_char	*nick;
{
   Channel *chan;
   Nick *tnick;
   
   if ((chan = lookup_channel(channel, from_server, CHAN_NOUNLINK)) &&
       (chan->connected == CHAN_JOINED) &&
       /* channel may be "surviving" from a disconnect/connect
	check here too -Sol */
       (tnick = (Nick *) list_lookup((List **) &(chan->nicks), nick, !USE_WILDCARDS, !REMOVE_FROM_LIST)) && 
       (tnick->status & NICK_CHOP))
     return 1;
   return 0;
}

int
has_voice(channel, nick)
	u_char	*channel;
	u_char	*nick;
{
   Channel *chan;
   Nick *tnick;
   
   if ((chan = lookup_channel(channel, from_server, CHAN_NOUNLINK)) &&
       (chan->connected == CHAN_JOINED) &&
       /* channel may be "surviving" from a disconnect/connect
	check here too -Sol */
       (tnick = (Nick *) list_lookup((List **) &(chan->nicks), nick, !USE_WILDCARDS, !REMOVE_FROM_LIST)) && 
       (/* come on, this is for voice, not chanops
	 * (tnick->status & NICK_CHOP) || 
	 */
	(tnick->status & NICK_VOICE)))
     return 1;
   return 0;
}

static	void
show_channel(chan)
	Channel *chan;
{
   /*
   Nick *tmp;
   int	buffer_len, len;
   u_char	*nicks = (u_char *) 0;
   u_char	*s;
   u_char	buffer[BIG_BUFFER_SIZE];
   
   s = recreate_mode(chan);
   *buffer = (u_char) 0;
   buffer_len = 0;
   for (tmp = chan->nicks; tmp; tmp = tmp->next)
     {
	len = my_strlen(tmp->nick);
	if (buffer_len + len >= (BIG_BUFFER_SIZE / 2))
	  {
	     malloc_strcpy(&nicks, buffer);
	     say("\t%s +%s (%s): %s", chan->channel, s, get_server_name(chan->server), nicks);
	     *buffer = (u_char) 0;
	     buffer_len = 0;
	  }
	my_strmcat(buffer, tmp->nick, BIG_BUFFER_SIZE);
	my_strmcat(buffer, " ", BIG_BUFFER_SIZE);
	buffer_len += len + 1;
     }
   malloc_strcpy(&nicks, buffer);
   say("\t%s +%s (%s): %s", chan->channel, s, get_server_name(chan->server), nicks);
   new_free(&nicks);
    */
   ninja_scan("SCAN", chan->channel, NULL);
}

/* list_channels: displays your current channel and your channel list */
void
list_channels()
{
   Channel *tmp;
   int	server, no = 1;
   int	first;
   
   if (connected_to_server < 1)
     {
	say("You are not connected to a server, use /SERVER to connect.");
	return;
     }
   if (get_channel_by_refnum(0))
     say("Current channel %s", get_channel_by_refnum(0));
   else
     say("No current channel for this window");
   first = 1;
   for (tmp = server_list[get_window_server(0)].chan_list; tmp; tmp = tmp->next)
     {
	if (tmp->connected != CHAN_JOINED)
	  continue;
	if (first)
	  say("You are on the following channels:");
	show_channel(tmp);
	first = 0;
	no = 0;
     }
   
   if (connected_to_server > 1)
     {
	for (server = 0; server < number_of_servers; server++)
	  {
	     if (server == get_window_server(0))
	       continue;
	     first = 1;
	     for (tmp = server_list[server].chan_list; tmp; tmp = tmp->next)
	       {
		  if (tmp->connected != CHAN_JOINED)
		    continue;
		  if (first)
		    say("Other servers:");
		  show_channel(tmp);
		  first = 0;
		  no = 0;
	       }
	  }
     }
   if (no)
     say("You are not on any channels");
}

void
switch_channels(key, ptr)
	u_int	key;
	u_char	*ptr;
{
   Channel *	tmp;
   u_char *	s;
   
   if (server_list[from_server].chan_list)
     {
	if (get_channel_by_refnum(0))
	  {
	     if ((tmp = lookup_channel(get_channel_by_refnum(0), from_server, CHAN_NOUNLINK)))
	       {
		  for (tmp = tmp->next; tmp; tmp = tmp->next)
		    {
		       s = tmp->channel;
		       if (!is_current_channel(s, from_server, 0) && !(is_bound(s, from_server) && curr_scr_win != tmp->window))
			 {
			    set_channel_by_refnum(0, s);
			    update_all_windows();
			    return;
			 }
		    }
	       }
	  }
	for (tmp = server_list[from_server].chan_list; tmp; tmp = tmp->next)
	  {
	     s = tmp->channel;
	     if (!is_current_channel(s, from_server, 0) && !(is_bound(s, from_server) && curr_scr_win != tmp->window))
	       {
		  set_channel_by_refnum(0, s);
		  update_all_windows();
		  return;
	       }
	  }
     }
}

void
change_server_channels(old, new)
	int	old,
		new;
{
   Channel *tmp;
   
   if (new == old)
     return;
   if (old > -1)
     {
	for (tmp = server_list[old].chan_list; tmp ;tmp = tmp->next)
	  tmp->server = new;
	server_list[new].chan_list = server_list[old].chan_list;
     }
   else
     server_list[new].chan_list = (Channel *) 0;
}

void
clear_channel_list(server)
	int	server;
{
   Channel *tmp, *next;
   Window		*ptr;
   int		flag = 1;
   
   while ((ptr = traverse_all_windows(&flag)))
     if (ptr->server == server && ptr->current_channel)
       new_free(&ptr->current_channel);
   
   for (tmp = server_list[server].chan_list; tmp; tmp = next)
     {
	next = tmp->next;
	free_channel(&tmp);
     }
   server_list[server].chan_list = (Channel *) 0;
   return;
}

/*
 * reconnect_all_channels: used after you get disconnected from a server, 
 * clear each channel nickname list and re-JOINs each channel in the 
 * channel_list ..  
 */
void
reconnect_all_channels(server)
	int	server;
{
   Channel *tmp = (Channel *) 0;
   u_char	*mode, *chan, *key;
   
   for (tmp = server_list[server].chan_list; tmp; tmp = tmp->next)
     {
	chan = tmp->channel;
	if (tmp->key)
	  key = tmp->key;
	else
	  key = get_ckey(chan);

	if (get_server_version(server) >= Server2_8)
	  send_to_server("JOIN %s%s%s", tmp->channel, key ? " " : "", key ? key : (u_char *) "");
	else
	  {
	     mode = recreate_mode(tmp);
	     send_to_server("JOIN %s%s%s", tmp->channel, mode ? " " : "", mode ? mode : (u_char *) "");
	  }
	tmp->connected = CHAN_JOINING;
	/* mask off the some of the channel cache flags */
	tmp->status = 0;
     }
}

u_char	*
what_channel(nick, server)
	u_char	*nick;
	int	server;
{
   Channel *tmp;
   
   if (curr_scr_win->current_channel && is_on_channel(curr_scr_win->current_channel, curr_scr_win->server, nick))
     return curr_scr_win->current_channel;
   
   /* XXX: why are we passing "server" when we just use "from_server" */
   for (tmp = server_list[from_server].chan_list; tmp; tmp = tmp->next)
     if (list_lookup((List **) &(tmp->nicks), nick, !USE_WILDCARDS, !REMOVE_FROM_LIST))
       return tmp->channel;
   
   return (u_char *) 0;
}

u_char	*
walk_channels(nick, init, server)
	int	init;
	u_char	*nick;
	int	server;
{
   static	Channel *tmp = (Channel *) 0;
   
   if (init)
     tmp = server_list[server].chan_list;
   else if (tmp)
     tmp = tmp->next;
   /* once again?  why from_server ? */
   for (;tmp ; tmp = tmp->next)
     if ((tmp->server == from_server) && (list_lookup((List **) &(tmp->nicks), nick, !USE_WILDCARDS, !REMOVE_FROM_LIST)))
       return (tmp->channel);
   return (u_char *) 0;
}

int
get_channel_oper(channel, server)
	u_char	*channel;
	int	server;
{
   Channel *chan;
   
   if ((chan = lookup_channel(channel, server, CHAN_NOUNLINK)))
     return (chan->status & CHAN_CHOP);
   else
     return 1;
}

extern	void
set_channel_window(window, channel, server)
	Window	*window;
	u_char	*channel;
	int	server;
{
   Channel	*tmp;
   
   if (!channel)
     return;
   for (tmp = server_list[server].chan_list; tmp; tmp = tmp->next)
     if (!my_stricmp(channel, tmp->channel))
       {
	  tmp->window = window;
	  return;
       }
}

extern	u_char	*
create_channel_list(window)
	Window	*window;
{
   Channel	*tmp;
   u_char	*value = (u_char *) 0;
   u_char	buffer[BIG_BUFFER_SIZE];

   *buffer = '\0';
   for (tmp = server_list[window->server].chan_list; tmp; tmp = tmp->next)
     {
	if (*buffer != '\0')
	  my_strmcat(buffer, " ", sizeof(buffer)-1);
	my_strmcat(buffer, tmp->channel, sizeof(buffer)-1);
     }
   malloc_strcpy(&value, buffer);

   return value;
}

extern void
channel_server_delete(server)
	int     server;
{
   Channel     *tmp;
   int	i;
   
   for (i = server + 1; i < number_of_servers; i++)
     for (tmp = server_list[i].chan_list ; tmp; tmp = tmp->next)
       if (tmp->server >= server)
	 tmp->server--;
}

extern	int
chan_is_connected(channel, server)
	u_char *	channel;
	int	server;
{
   Channel *	cp = lookup_channel(channel, server, CHAN_NOUNLINK);
   
   if (!cp)
     return 0;
   
   return (cp->connected == CHAN_JOINED);
}

void
mark_not_connected(server)
	int	server;
{
   Channel	*tmp;
   
   for (tmp = server_list[server].chan_list; tmp; tmp = tmp->next)
     tmp->connected = CHAN_LIMBO;
}

/*
 * take the string specified in "chan" and make it into a channel name.
 *
 * basically, if there's a & or # at the beginning, get out.
 * otherwise we prepend a #
 *
 * uses a static buffer, so multiple calls on the same line won't work properly.
 * ie: printf("%s %s\n", make_chan("hi"), make_chan("yo"));
 * would print:
 * #yo #yo
 */
u_char *
make_chan(u_char *chan)
{
   static u_char channel[256];

   /* this should never happen. */
   if (!chan || !*chan)
     return channel;
   
   /* this code used to do more, but now it just does this: */
     else if (!is_channel(chan))
     {
	channel[0] = '\0';
	my_strmcat(channel, "#", sizeof(channel)-1);
	my_strmcat(channel, chan, sizeof(channel)-1);
	return channel;
     }
   return chan;
}

/*
 * is the channel cached?
 * 
 * this is only called in one place, then end of nincache.c
 * 
 * it is reponsible for printing the join sync time if needed
 */
int
chk_channel_cached(chan)
   u_char *chan;
{
   Channel *c;
   
   if (!chan)
     return 0;
   c = lookup_channel(chan, parsing_server_index, CHAN_NOUNLINK);
   if (!c)
     return 0;
   /* if mode, who, or bans isn't cached then neither is the channel..
    * there's also CHAN_TIME but i'm not sure what all server support it..
    */
   if (!(c->status & CHAN_MODE)
       || !(c->status & CHAN_WHO)
       || !(c->status & CHAN_BANS))
     return 0;
   
   /* if the extra channel mode stuff isn't cached, still not!
    * 
    * these aren't implemented yet
    * 
   if ((server_list[chan->server].cmodes
	&& my_index(server_list[chan->server].cmodes, 'e') 
	&& !(chan->status & CHAN_EXCEPT))
       || (server_list[chan->server].cmodes 
	   && my_index(server_list[chan->server].cmodes, 'd') 
	   && !(chan->status & CHAN_DENY)))
     return 0;
    */
#ifdef SHOW_JOIN_TIME
   if (!(c->status & CHAN_SHOWED_TIME))
     {
	struct timeval tv, td;
	gettimeofday(&tv, NULL);
	td.tv_sec = tv.tv_sec - c->join_time.tv_sec;
	td.tv_usec = tv.tv_usec - c->join_time.tv_usec;
	save_message_from();
	message_from(c->channel, LOG_CRAP);
	put_info("Channel %s cached in %0.3f seconds.",
		 c->channel, 
		 (double)td.tv_sec + ((double)td.tv_usec / 1000000.0));
	restore_message_from();
	c->status |= CHAN_SHOWED_TIME;
     }
#endif   

   return 1;
}

/*
 * is_channel: determines if the argument is a channel.  If it's a number,
 * begins with MULTI_CHANNEL and has no '*', or STRING_CHANNEL, then its a
 * channel 
 */
int
is_channel(to)
   u_char *to;
{
   int version;

   if (to == 0)
     return 0;
   version = get_server_version(from_server);
   return ((version < Server2_7 && (isdigit(*to) || (*to == STRING_CHANNEL)
				    || *to == '-'))
	   || (version > Server2_5 && *to == MULTI_CHANNEL)
	   || (version > Server2_7 && *to == LOCAL_CHANNEL)
	   || (version > Server2_8 && *to == STRING_CHANNEL)
	   || (version > Server2_9 && *to == SAFE_CHANNEL)
#ifdef SUPPORT_ICB
	   || (version == ServerICB && my_stricmp(to, get_server_icbgroup(from_server)) == 0)
#endif
	    );
}

