/*
 * dma.c
 *
 * This file contains implementation for dynamic memory allocation functions.
 * All prototypes are in the "dma.h" include file and all functions are prefixed with "dma_"
 *
 * written by Joshua J. Drake
 * copyright (c) 1998
 *
 */

#include "irc.h"
#include "dma.h"
#include "ircaux.h"

#include "output.h"

#ifdef DMA_DEBUG
# include <stdarg.h>
# define DMA_DEBUG_FILE "dma_debug.sql"
static unsigned long lcnt = 0;
void
dma_debug_log(void *addr, char *fmt, ...)
{
   va_list vl;
   char tbuf[1024];
   FILE *dfp;

   va_start(vl, fmt);
   vsnprintf(tbuf, sizeof(tbuf)-1, fmt, vl);
   va_end(vl);
   tbuf[sizeof(tbuf)-1] = '\0';

   dfp = fopen(DMA_DEBUG_FILE, "a");
   if (!dfp)
     return;
   lcnt++;
   if (strcmp(tbuf, "free()") == 0)
     fprintf(dfp, 
	     "update dbg set freed='Y' where addr='0x%lx' and freed='N' and pid='%u';\n",
	     (unsigned long)addr, getpid());
   else
     fprintf(dfp, 
	     "insert into dbg (addr, op, seq, time, stime, pid) values ('0x%lx', '%s', '%lu', '%lu', '%lu', '%u');\n",
	     (unsigned long)addr, tbuf, lcnt, time(NULL), start_time % 4, getpid());
   fclose(dfp);
}
#endif

#ifdef DMA_DEBUG_INT
# include "list.h"
# define DMA_DEBUG_FILE "dma_debug.sh"
# define MAX_RET_ADDR 20
typedef struct __dma_mem_list 
{
   struct __dma_mem_list *next;
   void *addr;
   void *ret_addr[MAX_RET_ADDR];
} dmaEnt;
static dmaEnt *dmaHead = NULL;
static dmaEnt *dmaTmp = NULL;
int
dma_add_addr_cmp(dmaEnt *a1, dmaEnt *a2)
{
   if (a1->addr > a2->addr)
     return 1;
   if (a1->addr == a2->addr)
     return 0;
   return -1;
}
int
dma_rem_addr_cmp(dmaEnt *a1, char *a2)
{
   return (a1->addr == a2);
}
dmaEnt *
new_dma_ent(void *addr)
{
   dmaEnt *tmp;
   void *vt;
   int i = 0;
   
   tmp = malloc(sizeof(dmaEnt));
   if (!tmp)
     return NULL;
   
   /* make sure its all 0 and save address of memory */
   memset(tmp, 0, sizeof(dmaEnt));
   tmp->addr = addr;

   /* get stack return addresses */
   vt = __builtin_frame_address(0);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(0);
   
   vt = __builtin_frame_address(1);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(1);
   
   vt = __builtin_frame_address(2);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(2);
   
   vt = __builtin_frame_address(3);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(3);
   
   vt = __builtin_frame_address(4);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(4);
   
   vt = __builtin_frame_address(5);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(5);
   
   vt = __builtin_frame_address(6);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(6);
   
   vt = __builtin_frame_address(7);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(7);
   
   vt = __builtin_frame_address(8);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(8);
   
   vt = __builtin_frame_address(9);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(9);
   
   vt = __builtin_frame_address(10);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(10);
   
   vt = __builtin_frame_address(11);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(11);
   
   vt = __builtin_frame_address(12);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(12);
   
   vt = __builtin_frame_address(13);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(13);
   
   vt = __builtin_frame_address(14);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(14);
   
   vt = __builtin_frame_address(15);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(15);
   
   vt = __builtin_frame_address(16);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(16);
   
   vt = __builtin_frame_address(17);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(17);
   
   vt = __builtin_frame_address(18);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(18);
   
   vt = __builtin_frame_address(19);
   if (!vt)
     return tmp;
   tmp->ret_addr[i++] = __builtin_return_address(19);
   
   return tmp;
}
void
save_dma_list(void)
{
   dmaEnt *tmp, *next;
   FILE *fp;
   int i;
   
   fp = fopen(DMA_DEBUG_FILE, "w");
   if (!fp)
     fp = stderr;
   for (tmp = dmaHead; tmp; tmp = next)
     {
	next = tmp->next;
	fprintf(fp, "echo \"un-free()'d memory: 0x%0lx\"\n", (unsigned long)tmp->addr);
	fprintf(fp, "echo \"`addr2line -fs -e source/ninja");
	for (i = 0; i < MAX_RET_ADDR && tmp->ret_addr[i]; i++)
	  fprintf(fp, " 0x%0lx", (unsigned long)tmp->ret_addr[i]);
	fprintf(fp, "`\"\necho \"\"\n\n");
	free(tmp);
     }
   if (fp != stderr)
     fclose(fp);
}
#endif

/*
 * dma_Free()
 *
 * this function is used when freeing any dynamically allocated memory.
 */
void
dma_Free(iptr)
   void *iptr;
{
   u_char **ptr = (u_char **)iptr;
   if (!ptr)
     return;
#ifndef IN_IRCIO
   if (*ptr == empty_string)
     {
	put_error("dma_Free() trying to free memory for \"empty_string\" global.");
	*ptr = NULL;
	return;
     }
#endif
   if (*ptr)
     {
#ifdef DMA_DEBUG
	dma_debug_log(*ptr, "free()");
#endif
#ifdef DMA_DEBUG_INT
	dmaTmp = (dmaEnt *)remove_from_list_ext(&dmaHead, *ptr, dma_rem_addr_cmp);
	if (dmaTmp)
	  {
	     free(dmaTmp);
	     dmaTmp = NULL;
	  }
#endif	
	free(*ptr);
	*ptr = NULL;
     }
}

#define WAIT_BUFFER 2048
static u_char *pointers[WAIT_BUFFER], **current = pointers;

/*
 * dma_WaitFree()
 *
 * same as dma_Free() except that free() is postponed.
 */
void
dma_WaitFree(ptr)
   u_char **ptr;
{
   if (*current)
      dma_Free(current);
   *current++ = *ptr;
   if (current >= pointers + WAIT_BUFFER)
      current = pointers;
   *ptr = NULL;
}

/*
 * dma_ReallyFree()
 * 
 * really free the data if level == 0.
 * only left here for ircii scripting compatability.
 *
 */
void
dma_ReallyFree(level)
    int level;
{
   if (level != 0)
     return;
   for (current = pointers; current < pointers + WAIT_BUFFER; current++)
     if (*current)
       dma_Free(current);
   current = pointers;
}

/*
 * dma_ReAlloc()
 *
 * change the amount of memory allocated to ptr to size.
 */
u_char *
dma_ReAlloc(ptr, size)
    u_char *ptr;
    int size;
{
   u_char *tmp;

   if (ptr)
     {
	if (size > 0)
	  tmp = (u_char *)realloc(ptr, size);
	else
	  {
	     tmp = ptr;
	     dma_Free(&tmp);
	     return tmp;
	  }
     }
   else
     tmp = (u_char *)dma_Malloc(size);

#ifndef IN_IRCIO
   if (tmp == NULL)
     put_error("Unable to reallocate memory to %d bytes: %s", size, strerror(errno));
#endif   
#ifdef DMA_DEBUG
   dma_debug_log(ptr, "realloc() to %u", size);
#endif
   return tmp;
}

/*
 * dma_Malloc()
 *
 * allocate size bytes of memory.
 */
u_char *
dma_Malloc(size)
   int size;
{
   u_char *tmp;

   if ((tmp = (u_char *)malloc(size)) == NULL)
#ifndef IN_IRCIO
     put_error("Unable to allocate %d bytes of memory: %s", size, strerror(errno));
#else
   	;
#endif
   else
     memset(tmp, 0, size);
   
#ifdef DMA_DEBUG
   if (tmp)
     dma_debug_log(tmp, "malloc() for %u", size);
#endif
#ifdef DMA_DEBUG_INT
   if (tmp)
     {
	dmaTmp = new_dma_ent(tmp);
	if (dmaTmp)
	  add_to_list_ext((List **)&dmaHead, (List *)dmaTmp, dma_add_addr_cmp);
	dmaTmp = NULL;
     }
#endif
   
   return tmp;
}

/*
 * dma_strcpy()
 *
 * allocates enough memory to hold str, points ptr to it, and copies it.
 * the resulting string will always be null terminated.
 * NOTE: remember to initialize any strings used with this to NULL.
 *
 * returns TRUE or FALSE (1 or 0) success status
 */
int
dma_strcpy(ptr, str)
   u_char **ptr;
   u_char *str;
{
   if (str == *ptr)
     return 0;
   dma_Free(ptr);
   if (str && ((*ptr = dma_Malloc(my_strlen(str) + 1)) != NULL))
     {
	my_strcpy(*ptr, str); /* safe, malloc'd */
	return 1;
     }
   *ptr = NULL;
   return 0;
}

/*
 * dma_strcat()
 *
 * allocates enough memory for *ptr and str to be stored, points *ptr to it and copies it all.
 */
void
dma_strcat(ptr, str)
   u_char **ptr;
   u_char *str;
{
   u_char *tmp;

   if (!str)
      return;
   if (*ptr && (tmp = (u_char *)dma_Malloc(my_strlen(*ptr) + my_strlen(str) + 1)) != NULL)
     {
	my_strcpy(tmp, *ptr); /* safe, malloc'd */
	my_strcat(tmp, str); /* safe, malloc'd */
	dma_Free(ptr);
	*ptr = tmp;
     }
   else
      dma_strcpy(ptr, str);
}
#ifndef IN_IRCIO


void
dma_strcat_ue(ptr, src)
   u_char **ptr;
   u_char *src;
{
   u_char *new;

   if (*ptr)
     {
	size_t len = my_strlen(*ptr) + my_strlen(src) + 1;

	new = (u_char *) dma_Malloc(len);
	my_strcpy(new, *ptr);
	strmcat_ue(new, src, len);
	dma_Free(ptr);
	*ptr = new;
     }
   else
      dma_strcpy(ptr, src);
}

#endif
