/*
 * friends.c: Ninja IRC friends list implementation
 * 
 * written by Joshua J. Drake and Kraig Amador
 */

#include "irc.h"
#include "friends.h"

#include "server.h"
#include "channels.h"
#include "dma.h"
#include "screen.h"
#include "vars.h"
#include "ircaux.h"
#include "output.h"
#include "ignore.h"
#include "whois.h"

#include "ninja.h"
#include "ckey.h"

/* arg... */
#include "enemies.h"

#ifdef HAVE_CRYPT
# ifdef HAVE_CRYPT_H
#  include <crypt.h>
# endif
#endif

#define MUST_BE_EXACT 1


/* local command translations.. */
static	void	fcmd_add _((u_char *));
static	void	fcmd_add_chan _((u_char *));
static	void	fcmd_chan_flags _((u_char *));
static	void	fcmd_flags _((u_char *));
static	void	fcmd_list _((u_char *));
static	void	fcmd_passwd _((u_char *));
static	void	fcmd_rehash _((u_char *));
static	void	fcmd_rm _((u_char *));
static	void	fcmd_rm_chan _((u_char *));
static	void	fcmd_rm_host _((u_char *));
static	void	fcmd_rename _((u_char *));
static	void	fcmd_whois _((u_char *));


/* struct/commands.. */
struct
{
   u_char *name;
   u_int uniq;
   void (*function) _((u_char *));
}
friend_commands[] =
{
     { UP("ADD"),		3, fcmd_add },
     { UP("ADDCHANNEL"),	4, fcmd_add_chan },
     { UP("ADDHOST"),		4, fcmd_add_host },
     { UP("CFLAGS"),		1, fcmd_chan_flags },
     { UP("FLAGS"),		1, fcmd_flags },
     { UP("LIST"),		1, fcmd_list },
     { UP("PASS"),		1, fcmd_passwd },
     { UP("REHASH"),		3, fcmd_rehash },
     { UP("REMOVE"),		3, fcmd_rm },
     { UP("REMCHANNEL"),       	4, fcmd_rm_chan },
     { UP("REMHOST"),		4, fcmd_rm_host },
     { UP("RENAME"),		3, fcmd_rename },
     { UP("WHOIS"),		1, fcmd_whois },
     { NULL, 0, (void (*)())NULL }
};

Friend *friend_list = NULL;

static const u_char usermode_str[] = FL_UMODE_STRING;
static const u_char chanmode_str[] = FL_CMODE_STRING;

/* for adding hosts from nicks that are on irc */
extern void addhost_queue _((WhoisStuff *, u_char *, u_char *));
extern void add_userhost_to_whois();

extern	void	timercmd _((u_char *, u_char *, u_char *));

/* static stuff */
static	int	decifer_usermode _((u_char *, unsigned long *));
static	int	decifer_chanmode _((u_char *, unsigned long *));
static	void	free_friend _((Friend *));
static	void	free_fchan _((FChan *));
static	void	free_fhost _((FHost *));
static	int 	friends_cmd_chk _((u_char *, u_char *, u_char *, u_char *, Friend **));
static	void	fc_sync_bans _((Channel *));


/*
 * this routine dispatches the friend commands based on
 * the first parameter to /friend
 */
void
friend_cmd(command, args, subargs)
   u_char *command, *args, *subargs;
{
   u_char *cmd;
   u_int i, len;
   
   if (!(cmd = next_arg(args, &args)))
     {
	usage("friend", "<command> [<args>]");
	return;
     }
   
   len = my_strlen(cmd);
   upper(cmd);
   for (i = 0; friend_commands[i].name != NULL; i++)
     {
	if (!my_strncmp(friend_commands[i].name, cmd, len))
	  {
	     if (len < friend_commands[i].uniq)
	       {
		  put_error("Friend: Ambiguous command: %s", cmd);
		  return;
	       }
	     friend_commands[i].function(args);
	     return;
	  }
       }
   put_error("Friend: Unknown command: %s", cmd);
}


/*
 * add users to the friends list
 */
void
fcmd_add(args)
   u_char *args;
{
   u_char *nick, *passwd, *flags, tb1[512];
   Friend *f;
   int count = 0;
   u_char *uastr = UP("<friend> [<flags> [<password>]]");
   u_char *cmdstr = UP("friend add");
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, uastr, UNULL, nick, &f))
     return;
   if (f)
     {
	put_info("%s is already on your friends list.", nick);
	return;
     }
   
   /* we already know we they aren't there! */
   flags = next_arg(args, &args);
   passwd = next_arg(args, &args);
   if ((f = (Friend *) dma_Malloc(sizeof(Friend))) != NULL)
     {
	dma_strcpy(&f->nick, nick);
	if (passwd)
	  dma_strcpy(&f->password, UP(crypt(passwd, make_salt(f->nick))));
	
	/* setup the flags.. */
	if (flags)
	  {
	     snprintf(tb1, sizeof(tb1)-1, "%s%s",
		      *flags == '+' ? empty_string : UP("+"),
		      flags);
	     tb1[sizeof(tb1)-1] = '\0';
	     decifer_usermode(tb1, &f->modes);
	  }
	add_to_list((List **) & friend_list, (List *) f);
	
	add_userhost_to_whois(nick, addhost_queue);
	put_info("%s is now your friend with flags: %s%s%s",
		 nick,
		 f->modes ? UP("+") : empty_string,
		 f->modes ? recreate_umode(f) : UP("<none>"),
		 passwd ? UP(" with a password.") : UP("."));
	
	/* tell them about it.. */
	my_strcpy(tb1, "You have been added to my friends list."); /* safe */
	if (f->modes)
	  {
	     strmcat(tb1, "  I will: ", sizeof(tb1)-1);
	     if (f->modes & FL_UMODE_AUTO_DCC)
	       {
		  count++;
		  strmcat(tb1, "auto-get DCC requests", sizeof(tb1)-1);
	       }
	     if (f->modes & FL_UMODE_NO_FLOOD)
	       {
		  if (count)
		    strmcat(tb1, ", ", sizeof(tb1)-1);
		  count++;
		  strmcat(tb1, "allow floods", sizeof(tb1)-1);
	       }
	     if (f->modes & FL_UMODE_SECURE_NDCC)
	       {
		  if (count)
		    strmcat(tb1, ", ", sizeof(tb1)-1);
		  count++;
		  strmcat(tb1, "allow secure NDCC access", sizeof(tb1)-1);
	       }
	     strmcat(tb1, " from you.", sizeof(tb1)-1);
	  }
	send_to_server("NOTICE %s :%s", nick, tb1);
	
	/* tell them about their password and how to get help */
	snprintf(tb1, sizeof(tb1)-1, "Use \"/ctcp %s help\" for more information.", get_server_nickname(from_server));
	tb1[sizeof(tb1)-1] = '\0';
	if (passwd)
	  send_to_server("NOTICE %s :Your password is \"%s\", don't forget it.  %s", nick, passwd, tb1);
	else
	  send_to_server("NOTICE %s :Please set a password by typing /ctcp %s pass <password>   %s",
			 nick, get_server_nickname(from_server), tb1);
     }
   (void) save_friends(1);
}


/*
 * add a channel to a friend's channel access list.
 * 
 * accessed via /friend addchan 
 */
void
fcmd_add_chan(args)
   u_char *args;
{
   FChan *c;
   Friend *f;
   u_char *nick = NULL, *channel = NULL, *flags = NULL, tb1[32];
   unsigned long tmpmodes = 0;
   u_char *uastr = UP("<friend> <channel> <flags>");
   u_char *cmdstr = UP("friend addchannel");
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, uastr, UNULL, nick, &f))
     return;
   
   if (!(channel = next_arg(args, &args)))
     {
	usage(cmdstr, uastr);
	return;
     }
   
   /* for this we remove it regardless! */
   if ((c = (FChan *) remove_from_list((List **) & (f->channels), channel)) != NULL)
     free_fchan(c);
   
   flags = next_arg(args, &args);
   if (flags == NULL)
     {
	put_info("Why add a channel without any flags?");
	return;
     }
   
   snprintf(tb1, sizeof(tb1)-1, "%s%s", *flags != '+' ? "+" : "", flags);
   tb1[sizeof(tb1)-1] = '\0';
   decifer_chanmode(tb1, &tmpmodes);
   
   /* alloc/add it.. */
   c = (FChan *) dma_Malloc(sizeof(FChan));
   c->modes = tmpmodes;
   dma_strcpy(&(c->channel), channel);
   add_to_list((List **) & (f->channels), (List *) c);
   put_info("%s now has access to \"%s\" with flags: +%s", f->nick, channel, recreate_cmode(c));
   (void) save_friends(1);
}


/*
 * add a hostmask to a friend's entry in the friends list
 * 
 * first argument: friend's nick
 * remaining arguments: hostmask or current nickname
 */
void
fcmd_add_host(args)
   u_char *args;
{
   FHost *h;
   Friend *tmp;
   u_char *nick = NULL, *newhost = NULL;
   u_char *uastr = UP("<friend> [<nick/host>] [<nick/host>] ...");
   u_char *cmdstr = UP("friend addhost");
   extern int in_ctcp_flag;
		      
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, uastr, UNULL, nick, &tmp))
     return;
   
   /* if there isn't at least one nick/host show usage.. */
   newhost = next_arg(args, &args);
   if (!newhost)
     {
	usage(cmdstr, uastr);
	return;
     }
   
   /* while we have one, add it.. */
   while (newhost)
     {
	/* is it a full hostmask?  if not use userhost information */
	if (!match("*!*@*", newhost))
	  add_to_whois_queue(newhost, addhost_queue, "%s", nick);
	else
	  {
	     h = (FHost *) remove_from_list((List **)&tmp->hosts, newhost);
	     if (h)
	       {
		  add_to_list((List **) & (tmp->hosts), (List *) h);
		  if (!in_ctcp_flag) /* quiet hack.. */
		    put_info("%s already has access from \"%s\".", nick, newhost);
	       }
	     else
	       {
		  h = (FHost *) dma_Malloc(sizeof(FHost));
		  dma_strcpy(&(h->host), newhost);
		  add_to_list((List **) & (tmp->hosts), (List *) h);
		  if (!in_ctcp_flag) /* quiet hack.. */
		    put_info("%s now has access from \"%s\".", nick, newhost);
	       }
	  }
	newhost = next_arg(args, &args);
     }
   (void) save_friends(1);
}


/*
 * change a friends flags on a giving channel..
 * accessable via /friend cflags
 */
void
fcmd_chan_flags(args)
   u_char *args;
{
   FChan *ptr;
   Friend *tmp;
   u_char *nick = NULL, *channel = NULL, *flags = NULL;

   nick = next_arg(args, &args);
   channel = next_arg(args, &args);
   flags = next_arg(args, &args);
   if (!(flags && *flags))
     {
	usage("friend cflags", "<friend> <channel> +/-<flags>");
	return;
     }
   if ((tmp = get_friend(nick)) == NULL)
     {
	put_info("%s is not on your friends list.", nick);
	return;
     }
   if ((ptr = get_fchan(tmp, channel, MUST_BE_EXACT)) == NULL)
     {
	put_info("%s does not have access to %s", nick, channel);
	return;
     }
   decifer_chanmode(flags, &ptr->modes);
   put_info("%s's flags on %s are now: %s%s",
	    nick, channel, 
	    ptr->modes ? "+" : "<empty>", recreate_cmode(ptr));
   (void) save_friends(1);
}


/*
 * change a friends user flags...
 * accessable via /friend flags
 */
void
fcmd_flags(args)
   u_char *args;
{
   Friend *tmp;
   u_char *nick, *attribute;

   nick = next_arg(args, &args);
   attribute = next_arg(args, &args);
   if (!nick || !*nick || !attribute || !*attribute)
     {
	usage("friend flags", "<friend's nick> +/-<flags>");
	return;
     }
   if ((tmp = get_friend(nick)) == NULL)
     {
	put_info("%s is not on your friends list.", nick);
	return;
     }
   decifer_usermode(attribute, &tmp->modes);
   put_info("%s's flags are now: %s%s", nick, 
	    tmp->modes ? "+" : "<empty>", recreate_umode(tmp));
   (void) save_friends(1);
}


/*
 * list all friends on the list...
 * accessed via /friend list
 */
static	void
fcmd_list(args)
   u_char *args;
{
   Friend *tmp;
   FChan *chan;
   int count = 0;
   u_char tbuf1[2048], fmt[128];

   if (!friend_list)
     {
	put_info("Your friends list is empty.");
	return;
     }
   
   my_strncpy(fmt, "%-9.9s %-8.8s %s%-5.5s %s", sizeof(fmt)-1);
   fmt[sizeof(fmt)-1] = '\0';
	    
   put_info(fmt, "Nick", "Pass", " ", "Flags", "Channels");

   for (tmp = friend_list; tmp; tmp = tmp->next)
     {
	tbuf1[0] = '\0';
	for (chan = tmp->channels; chan; chan = chan->next)
	  {
	     u_char tb2[256];
	     
	     snprintf(UP(tb2), sizeof(tb2)-1, "%s(%s%s)", 
		      chan->channel,
		      chan->modes ? "+" : "", 
		      recreate_cmode(chan));
	     tb2[sizeof(tb2)-1] = '\0';
	     
	     if (*tbuf1 != '\0')
	       my_strmcat(tbuf1, " ", sizeof(tbuf1)-1);
	     my_strmcat(tbuf1, tb2, sizeof(tbuf1)-1);
	  }
	put_info(fmt, tmp->nick, tmp->password ? "<hidden>" : "<none>",
		 tmp->modes ? "+" : " ", recreate_umode(tmp),
		 *tbuf1 == (u_char)'\0' ? empty_string : tbuf1);
	count++;
     }
   put_info("End of friends list, %d counted.", count);
}


/*
 * change a friends password..
 * accessed via /friend pass
 */
static	void
fcmd_passwd(args)
   u_char *args;
{
   Friend *tmp;
   u_char *nick, *password;
   int remove = 0;

   nick = next_arg(args, &args);
   if (nick && *nick && *nick == '-')
     {
	remove = 1;
	nick++;
     }
   password = next_arg(args, &args);
   if (!nick || !*nick
       || (!remove && (!password || !*password)))
     {
	usage("friend pass", "[-]<friend> [<password>]");
	return;
     }
   if ((tmp = get_friend(nick)) == NULL)
     {
	put_info("%s is not on your friends list.", nick);
	return;
     }
   if (!remove)
     {
	if (tmp->password)
	  dma_Free(&tmp->password);
	dma_strcpy(&tmp->password, UP(crypt(password, make_salt(tmp->nick))));
	put_info("%s's password has been changed.", nick);
     }
   else
     {
	if (tmp->password)
	  dma_Free(&tmp->password);
	tmp->password = UNULL;
	put_info("%s's password has been removed.", nick);
     }
   (void) save_friends(1);
}


/*
 * clear and reload the friends list...
 * for those manual edit people..
 */
static	void
fcmd_rehash(args)
   u_char *args;
{
   Friend *f, *n;
   
   /* clear the list.. */
   for (f = friend_list; f; f = n)
     {
	n = f->next;
	free_friend(f);
     }
   friend_list = (Friend *)NULL;
   
   /* reload it! */
   put_info("Friend list rehashed, loaded %d friends.", load_friends(1));
}


/*
 * remove a user from the friends list
 */
void
fcmd_rm(args)
   u_char *args;
{
   u_char *nick;
   Friend *f;
   u_char *urstr = UP("<friend>");
   u_char *cmdstr = UP("friend remove");
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, UNULL, urstr, nick, &f))
     return;
   
   /* we now have f if its ok! */
   f = (Friend *) remove_from_list((List **) & friend_list, f->nick);
   if (f)
     {
	free_friend(f);
	put_info("%s is no longer your friend.", nick);
	(void) save_friends(1);
     }
   else
     put_error("Unable to remove \"%s\" from your friends list.", nick);
}


/*
 * remove a channel from a friend's channel access list.
 * 
 * accessed via /friend remchan 
 */
void
fcmd_rm_chan(args)
   u_char *args;
{
   FChan *c;
   Friend *f;
   u_char *nick = NULL, *channel = NULL;
   u_char *urstr = UP("<friend> <channel>");
   u_char *cmdstr = UP("friend remchannel");
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, UNULL, urstr, nick, &f))
     return;
   
   if (!(channel = next_arg(args, &args)))
     {
	usage(cmdstr, urstr);
	return;
     }
   
   /* for this we remove it regardless! */
   if ((c = (FChan *) remove_from_list((List **) & (f->channels), channel)) != NULL)
     free_fchan(c);
   
   put_info("%s no longer has access to \"%s\".", f->nick, channel);
   (void) save_friends(1);
}


/*
 * remove a hostmask from a friend's entry in the friends list
 * 
 * first argument: friend's nick
 * remaining arguments: hostmask or #
 */
void
fcmd_rm_host(args)
   u_char *args;
{
   FHost *h;
   Friend *tmp;
   u_char *nick = NULL, *newhost = NULL;
   u_char *urstr = UP("<friend> [<mask #>/<mask>] [<mask #>/<mask>] ..");
   u_char *cmdstr = UP("friend remhost");
   int shift_mod = 1;
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, UNULL, urstr, nick, &tmp))
     return;
   
   /* if there isn't at least one nick/host show usage.. */
   newhost = next_arg(args, &args);
   if (!newhost)
     {
	usage(cmdstr, urstr);
	return;
     }
   
   /* while we have one, remove it.. */
   while (newhost)
     {
	int ti = atoi(newhost);
	
	if (ti > 0)
	  {
	     int oti = ti;
	     
	     ti -= shift_mod;
	     for (h = tmp->hosts; h && ti > 0;
		  h = h->next, ti--);
	     ti = oti;
	     if (h)
	       {
		  h = (FHost *) remove_from_list((List **)&tmp->hosts, h->host);
		  if (ti >= shift_mod)
		    shift_mod++;
	       }
	  }
	else
	  h = (FHost *) remove_from_list((List **)&tmp->hosts, newhost);
	
	/* if we have no host error.. else free it.. */
	if (!h)
	  {
	     if (ti > 0)
	       put_info("Unable to locate mask #%d for %s", ti, tmp->nick);
	     else
	       put_info("Unable to locate mask \"%s\" for %s.", newhost, nick);
	  }
	else
	  {
	     put_info("%s no longer has access from \"%s\".", nick, h->host);
	     free_fhost(h);
	  }
	newhost = next_arg(args, &args);
     }
   (void) save_friends(1);
}


/*
 * rename a user on the friends list
 */
void
fcmd_rename(args)
   u_char *args;
{
   u_char *nick, *nnick;
   Friend *f;
   u_char *urstr = UP("<friend> <new name>");
   u_char *cmdstr = UP("friend rename");
   
   nick = next_arg(args, &args);
   if (!friends_cmd_chk(cmdstr, UNULL, urstr, nick, &f))
     return;
   if (!(nnick = next_arg(args, &args)) || !*nnick)
     {
	usage(cmdstr, urstr);
	return;
     }
   
   /* check for new-nick in use */
   if (get_friend(nnick))
     {
	put_error("The new name you chose is already in use.");
	return;
     }
   
   /* we now have f and a new name if its ok! */
   dma_Free(&(f->nick));
   dma_strcpy(&(f->nick), nnick);
   put_info("%s is now known as %s.", nick, nnick);
   (void) save_friends(1);
}


/*
 * shows all information about a friend..
 * accessed via /friend whois
 */
static	void
fcmd_whois(args)
   u_char *args;
{
   Friend *tmp;
   FChan *chan;
   FHost *host;
   u_char *nick = NULL;
   u_char tobuf[BIG_BUFFER_SIZE], tb2[512];
   int count = 0;

   if (!friend_list)
     {
	put_info("Your friends list is empty.");
	return;
     }
   if ((nick = next_arg(args, &args)) == NULL)
     {
	usage("friend whois", "<friend's nick>");
	return;
     }
   if ((tmp = get_friend(nick)) == NULL)
     {
	put_info("The nickname \"%s\" is not on your friends list.", nick);
	return;
     }
   /* ok, we have a valid friend. */
   put_info("Your friend %s%s%s%s %s", tmp->nick, tmp->modes ? "(+" : "", recreate_umode(tmp), tmp->modes ? ")" : "",
   tmp->password ? "has a password set." : "does not have a password set.");
   
   /* show the channels they have access to, if any */
   *tobuf = '\0';
   for (chan = tmp->channels; chan; chan = chan->next)
     {
	snprintf(tb2, sizeof(tb2)-1, " %s%s%s%s", chan->channel, chan->modes ? "(+" : "",
		 recreate_cmode(chan), chan->modes ? ")" : "");
	strmcat(tobuf, tb2, sizeof(tobuf)-1);
	count++;
     }
   if (*tobuf)
     put_info("on channel%s:%s", PLURAL(count), tobuf);
   
   /* show what hostsmasks they use */
   *tobuf = '\0';
   count = 0;
   for (host = tmp->hosts; host; host = host->next)
     {
	strmcat(tobuf, " ", sizeof(tobuf)-1);
	strmcat(tobuf, host->host, sizeof(tobuf)-1);
	count++;
     }
   if (*tobuf)
     put_info("from hostmask%s:%s", PLURAL(count), tobuf);
}


/*
 * this does initial checking for a friend /command
 * .. if 0 is return, the supplied information is no good.
 * 
 * rm will be set to 1 or 0 depending on the first character of the cmd string.
 * f will be set to the friend entry for removes or not modified for !rm
 * 
 */
static	int
friends_cmd_chk(cmd, uastr, urstr, nick, f)
   u_char *cmd;
   u_char *uastr, *urstr;
   u_char *nick;
   Friend **f;
{
   Friend *tmp;
   
   if (!cmd || !*cmd) /* just in case */
     return 0;
   if (!nick || !*nick)
     {
	if (urstr)
	  usage(cmd, urstr);
	else
	  usage(cmd, uastr);
	return 0;
     }
   tmp = get_friend(nick);
   /* if no friend found and not adding a friend... */
   if (!tmp 
       && my_stricmp(cmd, "friend add") != 0)
     {
	put_error("Unable to locate \"%s\" in your friends list.", nick);
	return 0;
     }
   *f = tmp;
   return 1;
}


/*
 * decifer a user/channel mode string based on requested and allowed mode strings...
 */
int
decifer_mode(mstr, mode, amstr)
   u_char *mstr;
   unsigned long *mode;
   const u_char *amstr;
{
   unsigned long value = 0;
   int add = 0;
   const u_char *p;
   
   /* sanity */
   if (!mstr || !*mstr)
     return 0;
   
   /* go through the mode string and look for valid mode characters */
   for (; *mstr; mstr++)
     {
	for (p = amstr; *p; p++)
	  if (*p == *mstr)
	    break;
	if (*p)
	  {
	     unsigned long tv = (p - amstr);

	     if (tv == 0)
	       value = 1;
	     else
	       {
		  value = 2;
		  while (--tv)
		    value *= 2;
	       }
	  }
	else
	  {
	     switch (*mstr)
	       {
		case '+':	
		  add = 1;
		  value = 0;
		  break;
		case '-':
		  add = 0;
		  value = 0;
		  break;
		default:
		  put_error("decifer_chanmode: unknown mode character '%c'", *mstr);
		  break;
	       }
	  }
	if (value)
	  {
	     if (add)
	       *mode |= value;
	     else
	       *mode &= ~value;
	  }
     }
   return 1;
}

/*
 * decifer the usermode from mode_string
 * examples: +nfd +n +fd
 * modifies the char pointed to by mode to 
 * resemble the modes.
 */
int
decifer_usermode(mode_string, mode)
   u_char *mode_string;
   unsigned long *mode;
{
   return decifer_mode(mode_string, mode, usermode_str);
}

/* 
 * decifer the channel mode from mode_string
 * examples: +aiouk +acn etc.
 * modifies the long integer pointed to by mode to reflect the modes
 */
int
decifer_chanmode(mode_string, mode)
   u_char *mode_string;
   unsigned long *mode;
{
   return decifer_mode(mode_string, mode, chanmode_str);
}


/*
 * recreates a mode string from a bitwise mode and a string of correlating
 * characters...
 */
int
recreate_mode(bwm, mstr, dest, dlen)
   int bwm;
   const u_char *mstr;
   u_char *dest;
   int dlen;
{
   u_char *p = dest;
   int mi, mode;
   const u_char func_str[] = "recreate_mode";
   
   /* reset the buffer.. */
   memset(dest, 0, dlen);
   if (!bwm)
     return 1;

   /* go through the possible modes and check for
    * each being set.. if so add the character to the
    * return buffer
    */
   for (mode = bwm, mi = 0;
	mode && mstr[mi];
	mode /= 2, mi++)
     {
	if ((p - dest) >= dlen)
	  {
	     put_error("%s: truncating buffer", func_str);
	     break;
	  }
	if (mi >= strlen(mstr))
	  {
	     put_error("%s: non-zero mode with no more mode characters", func_str);
	     break;
	  }
	if (mode % 2)
	  *p++ = mstr[mi];
     }
   if (!mstr[mi] && mode)
     put_error("%s: ack! remaining mode value: 0x%08lx", func_str, mode);
   return 1;
}


/*
 * recreates the user mode..
 * 
 * takes a pointer to a friends list entry (or the address of one) and
 * returns a pointer to a string representing the ascii "readable" mode
 * which is created from the entries' mode element
 */
u_char *
recreate_umode(Friend * tmp)
{
   static u_char retbuf[32];

   (void) recreate_mode(tmp->modes, usermode_str, retbuf, sizeof(retbuf));
   return retbuf;
}

/*
 * returns a pointer to a string with the ascii representation of the modes
 * from the channel list entries' mode element
 * takes an argument of an address of a FChan entry
 */
u_char *
recreate_cmode(FChan * tmp)
{
   static u_char retbuf[32];

   (void) recreate_mode(tmp->modes, chanmode_str, retbuf, sizeof(retbuf));
   return retbuf;
}

/*
 * check to see if a protected friend is getting banned or deopped and
 * reverse the action...
 * 
 * future:
 * - enforce modes defined for the user... (+[ao]vu) of +p
 * - enforce bans on any friend hostmasks..
 */
void
check_friend_protection(u_char *from, Channel *chan, u_char *line)
{
   Friend *f;
   FChan *fc;
   FHost *fh;
   Nick *n;
   int add = 0, itsme = 0;
   u_char *user, *mode;
   u_char tb1[1024];
   u_char *free_copy = UNULL, *rest;

   if (!friend_list || !(chan->status & CHAN_CHOP))
     return;
   
   /* make a copy so we can mangle it.. */
   dma_strcpy(&free_copy, line);
   rest = free_copy;
   tb1[0] = '\0';
   
   /* me? */
   itsme = my_stricmp(from, get_server_nickname(chan->server)) == 0;
   
   /* while we have something... */
   mode = next_arg(rest, &rest);
   for (; *mode; mode++)
     {
	switch (*mode)
	  {
	   case '+':
	     add = 1;
	     break;
	   case '-':
	     add = 0;
	     break;
	   case 'o':
	     if (add)
	       break;
	     
	     /* who got deopped? */
	     user = next_arg(rest, &rest);
	     /* sanity... ignore: no-one, me-deopping, self deop.. (possibly friends deopping) */
	     if (!user || my_stricmp(user, from) == 0 || itsme)
	       break;
	     /* if we can't find the person getting deopped on the channel, give up... */
	     if ((n = find_nick(user, UNULL, chan->server, chan)) == NULL)
	       continue;
	     /* if the person getting deopped is a friend on this channel or * and is protected ... */
	     if ((f = get_friend_by_nuh(n->nick, n->user, n->host))
		 && (fc = get_fchan(f, chan->channel, !MUST_BE_EXACT))
		 && (fc->modes & FL_CMODE_PROTECTED))
	       {
		  strmcat(tb1, " +o ", sizeof(tb1)-1);
		  strmcat(tb1, user, sizeof(tb1)-1);
	       }
	     break;
	   case 'b':
	     if (!add)
	       break;

	     /* who got banned? */
	     user = next_arg(rest, &rest);
	     /* sanity... ignore: no-one, me-banning... (possibly friends banning) */
	     if (!user || my_stricmp(user, from) == 0 || itsme)
	       break;
	     
	     /* see if the mask getting banned matches any of our friends that are 
	      * protected on this channel.. 
	      */
	     for (f = friend_list; f; f = f->next)
	       {
		  if ((fc = get_fchan(f, chan->channel, !MUST_BE_EXACT))
		      && (fc->modes & FL_CMODE_PROTECTED))
		    {
		       for (fh = f->hosts; fh; fh = fh->next)
			 if (match(user, fh->host)
			     || match(fh->host, user))
			   break;
		       if (fh)
			 {
			    strmcat(tb1, " -b ", sizeof(tb1)-1);
			    strmcat(tb1, user, sizeof(tb1)-1);
			 }
		    }
	       }
	     break;
	  }
     }
   if (*tb1)
      send_to_server("MODE %s %s", chan->channel, tb1);
}

/*
 * checks to see if:
 * a) friends with ops aren't opped
 * b) friends with voice can't talk
 * c) protected friends are banned
 * and proceeds to:
 * a) op them
 * b) voice them
 * c) unban them
 */
void
friends_channel_sync(Channel *chan)
{
   Nick *n;
   Friend *f;
   FChan *fc;
   
   if (!friend_list
       || !(chan->status & CHAN_CHOP))
     return;
   
   /* see if anyone on the channel needs anything */
   for (n = chan->nicks; n; n = n->next)
     {
	/* they're not a friend? */
	if (!(f = get_friend_by_nuh(n->nick, n->user, n->host)))
	  continue;
	
	/* they've no access to the channel? */
	if (!(fc = get_fchan(f, chan->channel, !MUST_BE_EXACT)))
	  continue;
	
	/* if there's something they need, give it to them */
	if ((fc->modes & FL_CMODE_AUTOOP)
	    && !(n->status & NICK_CHOP))
	  adddelayop(chan->channel, n->nick, chan->server, 0, 1);
	if ((fc->modes & FL_CMODE_VOICE)
	    && !(n->status & NICK_VOICE))
/*	    && !(n->status & NICK_CHOP)) */
	  adddelayop(chan->channel, n->nick, chan->server, 1, 1);
     }
   
   /* see if any of my protected friend's hostmasks are banned.. if so remove the bans.. */
   fc_sync_bans(chan);
}

static void
fc_sync_bans(Channel *chan)
{
   Friend *f;
   FChan *fc;
   FHost *fh;
   Ban *b;
   int mch, rmch, count = 0;
   char tb[512];
   
   *tb = '\0';
   for (b = chan->bans; b; b = b->next)
     {
	/* see if this ban matches any one of a protected friends' hosts */
	for (f = friend_list; f; f = f->next)
	  {
	     /* no access to this channel? */
	     if (!(fc = get_fchan(f, chan->channel, !MUST_BE_EXACT)))
	       continue;
	     
	     /* not protected on this channel? */
	     if (!(fc->modes & FL_CMODE_PROTECTED))
	       continue;
	     
	     /* look for a matching host.. */
	     for (fh = f->hosts; fh; fh = fh->next)
	       {
		  mch = match(fh->host, b->ban);
		  rmch = match(b->ban, fh->host);
		  /* found one?  add it! */
		  if (mch || rmch)
		    {
		       if (*tb)
			 strmcat(tb, " ", sizeof(tb)-1);
		       strmcat(tb, b->ban, sizeof(tb)-1);
		       count++;
		       
		       /* max bans per line? */
		       if (count == 4)
			 {
			    send_to_server("MODE %s -bbbb %s", chan->channel, tb);
			    *tb = '\0';
			    count = 0;
			 }
		    }
	       }
	  }
     }
   
   /* if there are some left.. do the unbanning.. */
   if (count > 0)
     send_to_server("MODE %s -%s %s", chan->channel, strfill('b', count), tb);
}

void
floodprot(char *nick, char *user, char *host, char *type, int ctcp_type, int ignoretime, char *channel)
{
   int old_window_display;
   Friend *tmp;
   char tb1[1024];

   if (!my_stricmp(nick, get_server_nickname(from_server)))
      return;
   if (((tmp = get_friend_by_nuh(nick, user, host)) != NULL) && tmp->modes & FL_UMODE_NO_FLOOD)
      return;
   if (get_int_var(FLOOD_WARNING_VAR))
      put_info("%s flooding detected from %s(%s@%s)", type, nick, user, host);
   if (!get_int_var(FLOOD_PROTECTION_VAR) || !ignoretime)
      return;

   /* ignore user */
   snprintf(tb1, sizeof(tb1)-1, "%s@%s %s", user, host, type);
   tb1[sizeof(tb1)-1] = '\0';
   old_window_display = window_display;
   window_display = 0;
   ignore(NULL, tb1, NULL);
   window_display = old_window_display;
   
   /* auto-unignore user after a while */
   snprintf(tb1, sizeof(tb1)-1, "%d IGNORE %s@%s NONE", ignoretime, user, host);
   tb1[sizeof(tb1)-1] = '\0';
   timercmd("TIMER", tb1, NULL);
   put_info("Ignoring %s from %s@%s for %d seconds.", type, user, host, ignoretime);
}

/*
 * make a random salt.
 *
 * written by Joshua J. Drake
 * 1998-03-05 02:30:00
 *
 * doesn't use librand anymore, 11/20/98
 */
u_char *
make_salt(u_char *nick)
{
   static u_char salt[3];
   const u_char *saltchars = "./abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
   long numscs = strlen(saltchars);

   srand(time(NULL));
   salt[0] = saltchars[rand() % numscs];
   srand(time(NULL));
   salt[1] = saltchars[rand() % numscs];
   salt[2] = '\0';
   return salt;
}

/*
 * load the friends list...
 */
int
load_friends(quiet)
   int quiet;
{
   Friend *f = NULL;
   FHost *fh;
   FChan *fc;
   
   u_char file[] = NINJA_FRIEND_FILE;
   FILE *infile;
   
   u_char *field, *kraig, *tm, *ptr, *tc;
   u_char linebuf[1024];
   int position, count = 0, errocc = 0, linecnt = 0, added = 0;
   
   /* initialize some stuff.. */
   field = kraig = tm = tc = UNULL;
   
   /* open the friends list file */
   ptr = expand_twiddle(file);
   infile = fopen(ptr, "r");
   dma_Free(&ptr);
   if (infile == (FILE *) NULL)
     {
	if (!quiet)
	  put_info("Unable to load friends from \"%s\"...", file);
	return 0;
     }
   
   /* loop until we've read everything.. */
   memset(linebuf, 0, sizeof(linebuf));
   while (fgets(linebuf, sizeof(linebuf)-1, infile))
     {
	/* line number.. regardless of comments */
	linecnt++;

	/* comments.. */
	if (*linebuf == '#')
	  continue;
	
	/* strip cr/lf */
	if ((ptr = strchr(linebuf, '\r')))
	  *ptr = '\0';
	if ((ptr = strchr(linebuf, '\n')))
	  *ptr = '\0';
	
	/* empty line? */
	if (*linebuf == '\0')
	  continue;

	/* allocate a friend */
	f = (Friend *) dma_Malloc(sizeof(Friend));
	if (!f)
	  {
	     put_error("Unable to allocate memory for friend at %s:%d", file, linecnt);
	     break;
	  }

	/* reset position */
	position = added  = 0;

	/* while we have more fields..
	 * 
	 * nick:pass:mode:host1:..:hostN:#chan1+mode:..:#chanN+mode\n
	 */
	kraig = linebuf;
	while ((ptr = my_strsep(&kraig, ":")))
	  {
	     /* what position are we in? */
	     switch (position)
	       {
		  /* position 0: the nick */
		case 0:
		  if (!*ptr)
		    {
		       errocc = 1;
		       break;
		    }
		  dma_strcpy(&f->nick, ptr);
		  position++;
		  break;
		  
		  /* position 1: the password */
		case 1:
		  if (*ptr != '\0')
		    dma_strcpy(&f->password, ptr);
		  position++;
		  break;
		  
		  /* position 2: the user mode */
		case 2:
		  decifer_usermode(ptr, &f->modes);
		  position++;
		  add_to_list((List **) & friend_list, (List *) f);
		  added = 1;
		  
		  /* increase the count */
		  count++;
		  break;
		  
		  /* position 3+: a host or channel */
		default:
		  /* a host? */
		  if (!is_channel(ptr) 
		      && match("*!*@*", ptr))
		    {
		       /* free it we found it */
		       if ((fh = (FHost *) remove_from_list((List **)&f->hosts, ptr)) != NULL)
			 free_fhost(fh);
		       
		       /* add it (again?) */
		       fh = (FHost *) dma_Malloc(sizeof(FHost));
		       if (!fh)
			 {
			    put_error("Unable to allocate memory for friend host at %s:%d (%d)", file, linecnt, position);
			    errocc = quiet = 1;
			    break;
			 }
		       dma_strcpy(&fh->host, ptr);
		       add_to_list((List **) & (f->hosts), (List *) fh);
		    }

		  /* a channel? */
		  else if ((is_channel(ptr) || *ptr == '*'))
		    {
		       /* save the channel name because we can't overwrite the + with null.. */
		       dma_strcpy(&tc, ptr);
		       
		       /* does it have a mode?? */
		       tm = strrchr(ptr, '+');
		       if (tm)
			 {
			    /* truncate the copied string to the channel name.. */
			    *(strrchr(tc, '+')) = '\0';
			 }
		       
		       /* free it if we found it */
		       if ((fc = (FChan *) remove_from_list((List **)&f->channels, ptr)) != NULL)
			 free_fchan(fc);
		       
		       fc = (FChan *) dma_Malloc(sizeof(FChan));
		       if (!fc)
			 {
			    put_error("Unable to allocate memory for friend channel at %s:%d (%d)", file, linecnt, position);
			    errocc = quiet = 1;
			    break;
			 }
		       dma_strcpy(&fc->channel, tc);
		       dma_Free(&tc);
		       decifer_chanmode(tm, &fc->modes);
		       add_to_list((List **) & (f->channels), (List *) fc);
		    }
		  
		  /* regardless... */
		  position++;
		  break;
	       }
	     
	     /* error occurred?? */
	     if (errocc)
	       {
		  if (!quiet)
		    put_error("Skipping malformed saved friend at %s:%d (at position %d)", file, linecnt, position);
		  free_friend(f);
		  break;
	       }
	  }
	if (!added)
	  {
	     if (!quiet)
	       put_error("Not adding malformed saved friend at %s:%d (at position %d)", file, linecnt, position-1);
	     free_friend(f);
	  }
	memset(linebuf, 0, sizeof(linebuf));
     }
   fclose(infile);
   return count;
}

/*
 * Save's the friends list..
 * format:
 * <nick>:<enc pw>:<user modes>:<host1>:<hostn>:#channel1+chanmodes:*+chanmodes etc
 * saves to ~/.ninja/ninja.passwd  if (quiet) we don't show the output..  this is
 * when someone is ident'n (partially re-written by Senor Pato)
 * 
 * returns the # of friends saved
 */
int
save_friends(quiet)
    int quiet;
{
   Friend *friend;
   FChan *channels;
   FHost *hosts;
   u_char file[] = NINJA_FRIEND_FILE;
   FILE *outfile;
   u_char *ptr, tb1[1024], tb2[256];
   int count = 0;

   ptr = expand_twiddle(file);
   outfile = fopen(ptr, "w");
   dma_Free(&ptr);
   if (outfile == (FILE *) NULL)
     {
	put_info("Unable to write to \"%s\", aborting userlist save..", file);
	return 0;
     }
   for (friend = friend_list; friend; friend = friend->next)
     {
	snprintf(tb1, sizeof(tb1)-1, "%s:%s:+%s%s", 
		 friend->nick, 
		 friend->password ? friend->password : empty_string, 
		 recreate_umode(friend), 
		 friend->hosts || friend->channels ? UP(":") : empty_string);
	tb1[sizeof(tb1)-1] = '\0';
	
	for (hosts = friend->hosts; hosts; hosts = hosts->next)
	  {
	     snprintf(tb2, sizeof(tb2)-1, "%s%s", hosts->host, friend->channels ? ":" : "");
	     tb2[sizeof(tb2)-1] = '\0';
	     strmcat(tb1, tb2, sizeof(tb1)-1);
	  }
	for (channels = friend->channels; channels; channels = channels->next)
	  {
	     snprintf(tb2, sizeof(tb2)-1, "%s+%s%s", channels->channel, recreate_cmode(channels), channels->next ? ":" : "");
	     tb2[sizeof(tb2)-1] = '\0';
	     strmcat(tb1, tb2, sizeof(tb1)-1);
	  }
	count++;
	fprintf(outfile, "%s\n", tb1);
     }
   fclose(outfile);
   if (!quiet && count > 0)
     put_info("Saved %d friend%s...", count, PLURAL(count));
   return count;
}


/*
 * lookup a friend in the list
 */
Friend *
get_friend(u_char *nick)
{
   return (Friend *) find_in_list((List **)&friend_list, nick, !USE_WILDCARDS);
}


/*
 * lookup a channel for a friend
 */
FChan *
get_fchan(Friend *f, u_char *chan, int exact)
{
   FChan *fchan;

   if ((fchan = (FChan *) find_in_list((List **)&(f->channels), chan, !USE_WILDCARDS)) != NULL)
     return fchan;
   if (!exact)
     for (fchan = f->channels; fchan; fchan = fchan->next)
       {
	  if (!strcmp(fchan->channel, "*"))
	    return fchan;
       }
   return NULL;
}

/*
 * get a friends's host entry that the passed hostmask matches
 */
FHost *
get_fhost(Friend *f, u_char *hostmask)
{
   FHost *fh;
   int ip = 0;
   u_char *p, *oldp;

   for (fh = f->hosts; fh; fh = fh->next)
     {
	/* first see if this host is an IP */
	oldp = p = my_rindex(fh->host, '@');
	if (!p)
	  {
	     put_error("missing @ in fh->host");
	     return NULL;
	  }
	p++;
	while (*p)
	  {
	     if (isdigit(*p) || *p == '.' || *p == '*' || *p == '?' || *p == '%')
	       ip = 1;
	     else
	       {
		  ip = 0;
		  break;
	       }
	     p++;
	  }
	
	/* if its an IP treat it differently.. */
	if (ip)
	  {
	     u_char *q;
	     int mch = 0;
	     
	     /* split the hostmask */
	     q = my_rindex(hostmask, '@');
	     if (!q)
	       {
		  put_error("missing @ in hostmask");
		  return NULL;
	       }
	     *q = *oldp = '\0';
	     if (match(fh->host, hostmask) 
		 && ip_match(oldp+1, q+1))
	       mch = 1;
	     *q = *oldp = '@';
	     if (mch)
	       return fh;
	  }
	else if (match(fh->host, hostmask))
	  return fh;
     }
   return NULL;
}

/*
 * find a friend by trying to match the passed user@host
 */
Friend *
get_friend_by_mask(u_char *hostmask)
{
   Friend *f, *done = NULL;
   FHost *fhost;
   
   /* must we traverse the whole thing?!
    * 
    * well, its usually small and we need to check for dupes!
    * -jjd
    */
   for (f = friend_list; f; f = f->next)
     {
	fhost = get_fhost(f, hostmask);
	if (fhost)
	  {
	     if (done)
	       put_info("WARNING: %s matches more than one friend, using %s", hostmask, done->nick);
	     else
	       done = f;
	  }
     }
   return done;
}

/*
 * find a friend by trying to match the passed user@host
 */
Friend *
get_friend_by_nuh(u_char *nick, u_char *user, u_char *host)
{
   u_char tb1[1024], *p;
   
   if (!user || !host)
     return NULL;
   
   p = user;
   if (*p == '~') /* skip no-ident char */
     p++;
   snprintf(tb1, sizeof(tb1)-1, "%s!%s@%s",
	    nick ? nick : asterik,
	    p ? p : empty_string,
	    host);
   tb1[sizeof(tb1)-1] = '\0';
   return get_friend_by_mask(tb1);
}

/*
 * check to see if we should do something to this person that is joining
 * this channel...
 * 
 */
void
check_friend_join(chan, nick, user, host, delay)
   Channel *chan;
   u_char *nick, *user, *host;
   u_int delay;
{
   Friend *f;
   FChan *fc;
   Nick *n;
   
   /* there was not enough information specified? */
   if (!chan || !nick || !user || !host)
     return;
   
   /* they're not on the channel? */
   if (!(n = find_nick(nick, UNULL, chan->server, chan)))
     return;
   
   /* i'm not opped? */
   if (!(chan->status & CHAN_CHOP))
     return;
   
   /* they're not a friend? */
   if (!(f = get_friend_by_nuh(nick, user, host)))
     return;
   
   /* they've no access to the channel? */
   if (!(fc = get_fchan(f, chan->channel, !MUST_BE_EXACT)))
     return;
   
   /* if there's something they need, give it to them */
   if ((fc->modes & FL_CMODE_AUTOOP)
       && !(n->status & NICK_CHOP))
     adddelayop(chan->channel, nick, chan->server, 0, delay);
   if ((fc->modes & FL_CMODE_VOICE)
       && !(n->status & NICK_VOICE))
     adddelayop(chan->channel, nick, chan->server, 1, delay);
}

/*
 * check to see if the user on notify is a friend, if so,
 * see if they have +n and invite them to the channels they have +n on
 */
void
check_friend_notify(u_char *nick, u_char *user, u_char *hostname)
{
   Friend *f;
   FChan *fc;
   Channel *chan;
   Nick *n;

   if (!nick || !user || !hostname)
     return;
   
   if (!(f = get_friend_by_nuh(nick, user, hostname)))
     return;
   
   for (chan = server_list[from_server].chan_list; chan; chan = chan->next)
     {
	fc = get_fchan(f, chan->channel, !MUST_BE_EXACT);
	n = find_nick(nick, UNULL, from_server, chan);
	if (fc && !n
	    && (fc->modes & FL_CMODE_NOTIFY_INVITE))
	  send_to_server("INVITE %s %s", nick, chan->channel);
     }
}

/*
 * see if we are to auto-join invites from user, if so, do it
 */
void
check_friend_invite(channel, nick, user, host)
   u_char *channel, *nick, *user, *host;
{
   Friend *f;
   FChan *fc;

   if (!nick || !user || !host || !channel)
     return;
   if (!(f = get_friend_by_nuh(nick, user, host)))
     return;
   
   if (!(fc = get_fchan(f, channel, !MUST_BE_EXACT)))
     return;
   
   if (fc->modes & FL_CMODE_JOIN)
     {
	put_info("Auto-joining \"%s\" on invite from your friend \"%s\".", channel, f->nick);
	send_to_server("JOIN %s %s", channel, get_ckey(channel));
     }
}

/*
 * see if we should auto-get files from User@Host
 * if so, return a pointer to their friends list entry
 */
Friend *
check_friend_autoget(nick, user, host)
   u_char *nick, *user, *host;
{
   Friend *f;
   
   if (!nick || !user || !host)
     return NULL;
   if (!(f = get_friend_by_nuh(nick, user, host)))
     return NULL;
   
   if (f->modes & FL_UMODE_AUTO_DCC)
     return f;
   return NULL;
}

/*
 * see if we shouled enforce protection for someone
 * getting kicked.
 */
void
check_friend_kick(channel, nick, user, host, kickee)
   u_char *channel, *nick, *user, *host, *kickee;
{
   Friend *fe, *fk;
   FChan *fc;
   Nick *n;
   Channel *chan;
   
   chan = lookup_channel(channel, from_server, CHAN_NOUNLINK);
   if (!chan || !(chan->status & CHAN_CHOP))
     return;
   /* if its me, don't act.. */
   if (my_stricmp(nick, get_server_nickname(chan->server)) == 0)
     return;
   if (!(n = find_nick(kickee, UNULL, chan->server, chan)))
     return;
   if (!(fe = get_friend_by_nuh(n->nick, n->user, n->host)))
     return;
   /* don't enforce protection against friends */
   if ((fk = get_friend_by_nuh(nick, user, host))
       && (fc = get_fchan(fk, chan->channel, !MUST_BE_EXACT)))
     return;
   /* if the kickee is protected on this channel, enforce protection */
   if ((fc = get_fchan(fe, chan->channel, !MUST_BE_EXACT)) && (fc->modes & FL_CMODE_PROTECTED))
     {
	send_to_server("KICK %s %s :%s > you.", chan->channel, nick, kickee);
	send_to_server("INVITE %s %s", kickee, chan->channel);
     }
}



/* free routines... */
static	void
free_friend(f)
   Friend *f;
{
   FHost *host, *thost;
   FChan *chan, *tchan;
   
   dma_Free(&f->nick);
   dma_Free(&f->password);
   /* free hosts */
   for (host = f->hosts; host != NULL; host = thost)
     {
	thost = host->next;
	free_fhost(host);
     }
   /* free channels */
   for (chan = f->channels; chan != NULL; chan = tchan)
     {
	tchan = chan->next;
	free_fchan(chan);
     }
   dma_Free(&f);
}

static	void
free_fchan(fc)
   FChan *fc;
{
   dma_Free(&fc->channel);
   dma_Free(&fc);
}

static	void
free_fhost(fh)
   FHost *fh;
{
   dma_Free(&fh->host);
   dma_Free(&fh);
}

