(** Interface to the newt text window system. *)
(* Copyright (C) 2008 Richard W.M. Jones, Red Hat Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*)
(** The newt windowing system is a terminal-based window and widget
    library designed for writing applications with a simple, but
    user-friendly, interface. While newt is not intended to provide the
    rich feature set advanced applications may require, it has proven to
    be flexible enough for a wide range of applications (most notably, Red
    Hat's installation process).

    {2 Notes}

    There is not a large amount of documentation for newt.  The
    best documentation is probably Eric Troan's
    {{:http://www.whoow.org/people/Jean-Marc.Lienher/gnewt/tutorial.html}newt tutorial}.

    OCaml functions are derived using a predictable naming scheme.  So
    for example the C function [newtPopWindow] becomes
    {!Newt.pop_window}.

    There are plenty of OCaml examples in the [examples/] subdirectory.

    {2 Safety}

    The OCaml bindings attempt to be as type and memory safe as
    possible.  However newt itself makes this difficult.  Newt code
    has many arbitrary limits, no checks on memory
    allocation, and inherent confusion about types.

    Do not use functions from the unsafe {!Newt_int} (internal)
    module unless you know what you're doing.
*)

(** {2 Exceptions} *)

exception NewtFailure of string


(** {2 Types} *)

type color_set = Newt_int.newtColorSet
type colors = Newt_int.newtColors
type flags_sense = Newt_int.newtFlagsSense
type flag =
    RETURNEXIT
  | HIDDEN
  | SCROLL
  | DISABLED
  | BORDER
  | WRAP
  | NOF12
  | MULTIPLE
  | SELECTED
  | CHECKBOX
  | PASSWORD
  | SHOWCURSOR
type fd_flag = FD_READ | FD_WRITE | FD_EXCEPT

(** {2 Functions} *)

val init : unit -> unit
val finished : unit -> unit
val init_and_finish : (unit -> 'a) -> 'a
(** [init_and_finish f] calls {!init}, then runs the function [f ()],
    then finishes (cleans up) the terminal.

    It does the right thing even if [f] throws an exception.
*)
val cls : unit -> unit
val resize_screen : int -> unit
val wait_for_key : unit -> unit
val clear_key_buffer : unit -> unit
val delay : int -> unit
val open_window : int -> int -> int -> int -> string -> unit
val centered_window : int -> int -> string -> unit
val pop_window : unit -> unit
val pop_window_no_refresh : unit -> unit
val set_colors : Newt_int.newtColors -> unit
val set_color : Newt_int.newtColorSet -> string -> string -> unit
val refresh : unit -> unit
val suspend : unit -> unit
val resume : unit -> int
val push_help_line : string -> unit
val redraw_help_line : unit -> unit
val pop_help_line : unit -> unit
val draw_root_text : int -> int -> string -> unit
val bell : unit -> unit
val cursor_on : unit -> unit
val cursor_off : unit -> unit
val get_screen_size : unit -> int * int

(** {3 Components}

    We use phantom types here to add some type safety around
    components.  All components have a type of at least [[>
    `Component] component].  Specific types of components have an
    extra variant indicating the type and limiting what functions you
    can safely call.  Thus for example buttons have type
    [[`Button|`Component] component].

    A [component_ptr] is a special type of component that may
    only be compared against other components, but not used in
    any way.  It is returned by certain functions that get the
    "current" subcomponent from a form or component.
*)

type 'a component = _component
and _component

type component_ptr

val component_equals : component_ptr -> [>`Component] component -> bool

(** {4 Buttons} *)

val compact_button :
  int -> int -> string -> [ `Button | `Component ] component
val button : int -> int -> string -> [ `Button | `Component ] component

(** {4 Checkboxes, radio buttons} *)

val checkbox :
  int -> int ->
  string -> char -> string option -> [ `Checkbox | `Component ] component
val checkbox_get_value : [> `Checkbox ] component -> char
val checkbox_set_value : [> `Checkbox ] component -> char -> unit
val checkbox_set_flags :
  [> `Checkbox ] component -> flag list -> Newt_int.newtFlagsSense -> unit
val radio_button :
  int ->
  int ->
  string ->
  bool -> [> `Radio ] component option -> [ `Component | `Radio ] component
val radio_get_current : [> `Radio ] component -> component_ptr

(** {4 Labels} *)

val label : int -> int -> string -> [ `Component | `Label ] component
val label_set_text : [> `Label ] component -> string -> unit

(** {4 Scrollbars} *)

val vertical_scrollbar :
  int ->
  int ->
  int ->
  color_set -> color_set -> [ `Component | `VerticalScrollbar ] component
val scrollbar_set : [> `VerticalScrollbar ] component -> int -> int -> unit

(** {4 Listboxes}

    Newt lets you store arbitrary data for each row in a listbox.
    We can't let you do that, but instead we let you store
    values of a single type in the listbox (if you really
    want to store arbitrary values, use a variant).

    The newt documentation confuses the terms 'key' and 'data'
    as if they refer to something different.  In fact they are
    the same thing and we refer to it as 'data' everywhere.
*)

type 'a listbox

val listbox : int -> int -> int -> flag list -> 'a listbox
  (** Create a listbox. *)
val listbox_get_current : 'a listbox -> 'a option
  (** This returns the [data] field from the current item, for listboxes
      which only allow single selection.

      If nothing is selected, [None] is returned.
  *)
val listbox_set_current : 'a listbox -> int -> unit
  (** Set the current selection for listboxes which allow single selection. *)
val listbox_set_current_by_key : 'a listbox -> 'a -> unit
  (** Set the current selection to the row containing [data].
  
      @raise Not_found if data is not found. *)
val listbox_set_entry : 'a listbox -> int -> string -> unit
val listbox_set_width : 'a listbox -> int -> unit
val listbox_set_data : 'a listbox -> int -> 'a -> unit
val listbox_append_entry : 'a listbox -> string -> 'a -> unit
  (** [listbox_append_entry lb text data] appends a new listbox
      entry consisting of the text string [text] and data [data]
      to the end of the listbox.
  *)
val listbox_insert_entry :
  'a listbox -> string -> 'a -> 'a option -> unit
  (** [listbox_insert_entry lb text data after] inserts a new listbox
      entry consisting of the text string [text] and data [data]
      after the current listbox entry with data field [after].
      It returns the position of the new entry.
  
      @raise Not_found if [after] is not found.
  *)
val listbox_delete_entry : 'a listbox -> 'a -> unit
  (** [listbox_delete_entry lb data] deletes the listbox entry
      which contains [data].
  
      @raise Not_found if [data] is not found.
  *)
val listbox_clear : 'a listbox -> unit
  (** Remove all listbox entries. *)
val listbox_get_entry : 'a listbox -> int -> string * 'a
  (** Return a member of the listbox. *)
val listbox_get_selection : 'a listbox -> 'a array
  (** [listbox_get_selection lb] return an array of
      [data] from the items selected.
  *)
val listbox_clear_selection : 'a listbox -> unit
  (** Clear the current selection. *)
val listbox_select_item : 'a listbox -> 'a -> flags_sense -> unit
  (** Select the item containing [data].
  
      @raise Not_found if data is not found. *)
val listbox_item_count : 'a listbox -> int
  (** Return the total number of items in the listbox. *)

val component_of_listbox : 'a listbox -> [`Component|`Listbox] component
  (** Convert the specialised listbox type back to a normal component,
      eg. allowing it to be added to forms. *)

(** {4 Checkbox trees} *)

(** {4 Textboxes} *)

val textbox_reflowed :
  int ->
  int ->
  string ->
  int -> int -> int -> flag list -> [ `Component | `Textbox ] component
val textbox :
  int ->
  int -> int -> int -> flag list -> [ `Component | `Textbox ] component
val textbox_set_text : [> `Textbox ] component -> string -> unit
val textbox_set_height : [> `Textbox ] component -> int -> unit
val textbox_get_num_lines : [> `Textbox ] component -> int
val reflow_text : string -> int -> int -> int -> string * int * int

(** {4 Entries} *)

val entry :
  int ->
  int ->
  string option -> int -> flag list -> [ `Component | `Entry ] component
val entry_set : [> `Entry ] component -> string -> bool -> unit
val entry_get_value : [> `Entry ] component -> string
val entry_set_flags :
  [> `Entry ] component -> flag list -> flags_sense -> unit

(** {4 Scales} *)

val scale : int -> int -> int -> int64 -> [ `Component | `Scale ] component
val scale_set : [> `Scale ] component -> int64 -> unit

(** {3 Forms} *)

val form :
  [> `VerticalScrollbar ] component option ->
  string option -> flag list -> [ `Component | `Form ] component
val form_set_timer : [> `Form ] component -> int -> unit
val form_watch_fd :
  [> `Form ] component -> Unix.file_descr -> fd_flag list -> unit
val form_set_size : [> `Form ] component -> unit
val form_get_current : [> `Form ] component -> component_ptr
val form_set_background : [> `Form ] component -> int -> unit
val form_set_current :
  [> `Form ] component -> [> `Component ] component -> unit
val form_add_component :
  [> `Form ] component -> [> `Component ] component -> unit
val form_add_components :
  [> `Form ] component -> [> `Component ] component list -> unit
val form_set_height : [> `Form ] component -> int -> unit
val form_set_width : [> `Form ] component -> int -> unit
val run_form : [> `Form ] component -> component_ptr option
val form_run : [> `Form ] component -> Newt_int.newtExitStruct
val form_add_hot_key : [> `Form ] component -> int -> unit

(** {3 Generic component functions} *)

val component_takes_focus : [> `Component ] component -> bool -> unit
(** [component_takes_focus co bool] enables or disables
    a component. *)

val component_add_callback : [> `Component ] component -> (unit -> unit) -> unit
(** [component_add_callback co f] adds a general callback function [f]
    to a component.  This is only available for certain types
    of component -- see the Newt documentation for more information. *)

(** {2 Keys} *)
