/* Newt low-level bindings which camlidl cannot do automatically.
   Copyright (C) 2008 Richard W.M. Jones, Red Hat Inc.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/*#define DEBUG 1*/

/* Newt only lets us register one destroy-callback per
 * component.  This multiplexes them so we can register
 * as many as we like.
 */
struct cleanup {
  struct cleanup *next;
  newtCallback f;
  void *data;
};
struct cleanups {
  struct cleanups *next;
  newtComponent co;
  struct cleanup *head;
};
struct cleanups *cleanups = NULL;

static void
do_cleanup (newtComponent co, void *data)
{
  struct cleanups *this_cls = (struct cleanups *) data;
  struct cleanup *cl;
  struct cleanups *cls;

#if DEBUG
  fprintf (stderr, "in do_cleanup\n");
#endif

  /* Run the cleanup functions. */
  for (cl = this_cls->head; cl; cl = cl->next) {
#if DEBUG
    fprintf (stderr, "running cleanup on %ld: %ld (%ld)\n",
	     (long) co, (long) cl->f, (long) cl->data);
#endif
    cl->f (co, cl->data);
  }

  /* Tricky: we need to remove this_cls from the linked list of cleanups. */
  if (cleanups == this_cls)
    cleanups = cleanups->next;
  else {
    for (cls = cleanups; cls; cls = cls->next)
      if (cls->next == this_cls) {
	cls->next = this_cls->next;
	break;
      }
  }

  /* Now free the structure. */
  for (cl = this_cls->head; cl; ) {
    struct cleanup *next_cl = cl->next;
    free (cl);
    cl = next_cl;
  }

  free (this_cls);
}

static void
component_cleanup (newtComponent co, newtCallback f, void *data)
{
  struct cleanups *cls;
  struct cleanup *cl;

  /* Do we have a cleanup registered for this component already? */
  for (cls = cleanups; cls; cls = cls->next)
    if (cls->co == co)
      goto add_cleanup;

  cls = malloc (sizeof *cls);
  cls->co = co;
  cls->head = NULL;
#ifdef HAVE_NEWTCOMPONENTADDDESTROYCALLBACK
  newtComponentAddDestroyCallback (co, do_cleanup, cls);
#else
#if DEBUG
  fprintf (stderr,
	   "newtComponentAddDestroyCallback() not present in library,\n"
	   "so memory and some global roots leak.  Upgrade newt to fix\n"
	   "this issue.\n");
#endif
#endif
  cls->next = cleanups;
  cleanups = cls;

 add_cleanup:
  cl = malloc (sizeof *cl);
  cl->next = cls->head;
  cls->head = cl;
  cl->f = f;
  cl->data = data;

#if DEBUG
  fprintf (stderr, "registered cleanup on %ld: %ld (%ld)\n",
	   (long) co, (long) f, (long) data);
#endif
}

CAMLprim value
newt_stubs_newtCheckbox (value leftv, value topv, value strv,
			 value defv, value seqoptv)
{
  CAMLparam5 (leftv, topv, strv, defv, seqoptv);
  CAMLlocal1 (rv);
  char *seq = NULL;
  newtComponent r;

  if (seqoptv != Val_int (0))
    seq = String_val (Field (seqoptv, 0));

  r = newtCheckbox (Int_val (leftv), Int_val (topv),
		    String_val (strv), Int_val (defv), seq, NULL);
  rv = camlidl_c2ml_newt_int_newtComponent (&r, 0);

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtRadiobutton (value leftv, value topv, value strv,
			    value defaultv, value prevoptv)
{
  CAMLparam5 (leftv, topv, strv, defaultv, prevoptv);
  CAMLlocal1 (rv);
  newtComponent prev = NULL;
  newtComponent r;

  if (prevoptv != Val_int (0))
    camlidl_ml2c_newt_int_newtComponent (Field (prevoptv, 0), &prev, 0);
  r = newtRadiobutton (Int_val (leftv), Int_val (topv),
		       String_val (strv), Bool_val (defaultv),
		       prev);
  rv = camlidl_c2ml_newt_int_newtComponent (&r, 0);

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtListboxGetCurrent (value lbv)
{
  CAMLparam1 (lbv);
  CAMLlocal2 (rv, v);
  newtComponent lb;
  void *r;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  r = newtListboxGetCurrent (lb);

  if (r == NULL)
    rv = Val_int (0);
  else {
    v = Val_int ((long) r);
    rv = camlidl_alloc_small (1, 0);
    Field (rv, 0) = v;
  }

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtListboxSetCurrentByKey (value lbv, value datav)
{
  CAMLparam2 (lbv, datav);
  newtComponent lb;
  void *data = (void *) Long_val (datav);

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  newtListboxSetCurrentByKey (lb, data);

  CAMLreturn (Val_unit);
}

CAMLprim value
newt_stubs_newtListboxSetData (value lbv, value numv, value datav)
{
  CAMLparam3 (lbv, numv, datav);
  newtComponent lb;
  int num = Int_val (numv);
  void *data = (void *) Long_val (datav);

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  newtListboxSetData (lb, num, data);

  CAMLreturn (Val_unit);
}

CAMLprim value
newt_stubs_newtListboxAppendEntry (value lbv, value strv, value datav)
{
  CAMLparam3 (lbv, strv, datav);
  newtComponent lb;
  char *str = String_val (strv);
  void *data = (void *) Long_val (datav);
  int r;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  r = newtListboxAppendEntry (lb, str, data);
  CAMLreturn (Val_int (r));
}

CAMLprim value
newt_stubs_newtListboxInsertEntry (value lbv, value strv, value datav,
				  value afterv)
{
  CAMLparam4 (lbv, strv, datav, afterv);
  newtComponent lb;
  char *str = String_val (strv);
  void *data = (void *) Long_val (datav);
  void *after;
  int r;

  if (afterv == Val_int (0))
    after = NULL;
  else
    after = (void *) Long_val (Field (afterv, 0));

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  r = newtListboxInsertEntry (lb, str, data, after);
  CAMLreturn (Val_int (r));
}

CAMLprim value
newt_stubs_newtListboxDeleteEntry (value lbv, value datav)
{
  CAMLparam2 (lbv, datav);
  newtComponent lb;
  void *data = (void *) Long_val (datav);
  int r;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  r = newtListboxDeleteEntry (lb, data);
  CAMLreturn (Val_int (r));
}

CAMLprim value
newt_stubs_newtListboxGetEntry (value lbv, value numv)
{
  CAMLparam2 (lbv, numv);
  CAMLlocal3 (rv, strv, datav);
  newtComponent lb;
  int num = Int_val (numv);
  char *str = NULL;
  void *data = NULL;
  int r;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  newtListboxGetEntry (lb, num, &str, &data);

  if (!str || !data) caml_raise_not_found ();

  strv = caml_copy_string (str);
  free (str);
  datav = Val_int ((long) data);
  rv = camlidl_alloc_small (2, 0);
  Field (rv, 0) = strv;
  Field (rv, 1) = datav;

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtListboxGetSelection (value lbv)
{
  CAMLparam1 (lbv);
  CAMLlocal1 (rv);
  newtComponent lb;
  int numitems = 0, i;
  void **data;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);

  data = newtListboxGetSelection (lb, &numitems);
  assert (numitems == 0 || data != NULL);

  rv = camlidl_alloc_small (numitems, 0);
  for (i = 0; i < numitems; ++i)
    Field (rv, i) = Val_int ((long) data[i]);

  free (data);

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtListboxSelectItem (value lbv, value datav, value flagsv)
{
  CAMLparam3 (lbv, datav, flagsv);
  newtComponent lb;
  void *data = (void *) Long_val (datav);
  enum newtFlagsSense sense;

  camlidl_ml2c_newt_int_newtComponent (lbv, &lb, 0);
  sense = camlidl_ml2c_newt_int_enum_newtFlagsSense (flagsv);

  newtListboxSelectItem (lb, data, sense);

  CAMLreturn (Val_unit);
}

CAMLprim value
newt_stubs_newtForm (value vboptv, value stroptv, value flagsv)
{
  CAMLparam3 (vboptv, stroptv, flagsv);
  CAMLlocal1 (rv);
  newtComponent vb = NULL;
  char *str = NULL;
  newtComponent r;

  if (vboptv != Val_int (0))
    camlidl_ml2c_newt_int_newtComponent (Field (vboptv, 0), &vb, 0);
  if (stroptv != Val_int (0))
    str = String_val (Field (stroptv, 0));

  r = newtForm (vb, str, Int_val (flagsv));
  rv = camlidl_c2ml_newt_int_newtComponent (&r, 0);

  CAMLreturn (rv);
}

CAMLprim value
newt_stubs_newtRunForm (value formv)
{
  CAMLparam1 (formv);
  CAMLlocal2 (rv, v);
  newtComponent form;
  newtComponent r;

  camlidl_ml2c_newt_int_newtComponent (formv, &form, 0);

  r = newtRunForm (form);

  if (r == NULL)
    rv = Val_int (0);
  else {
    v = camlidl_c2ml_newt_int_newtComponent (&r, 0);
    rv = camlidl_alloc_small (1, 0);
    Field (rv, 0) = v;
  }

  CAMLreturn (rv);
}

static void
do_callback (newtComponent co, void *data)
{
  value *fvp = (value *) data;

#if DEBUG
  fprintf (stderr, "running callback for %ld\n", (long) co);
#endif

  caml_callback (*fvp, Val_unit);
}

static void
do_remove_callback (newtComponent co, void *data)
{
  caml_remove_global_root ((value *) data);
}

CAMLprim
newt_stubs_newtComponentAddCallback (value cov, value fv)
{
  CAMLparam2 (cov, fv);
  value *fvp;
  newtComponent co;
  struct callback *cb;

  camlidl_ml2c_newt_int_newtComponent (cov, &co, 0);

  fvp = malloc (sizeof *fvp);
  *fvp = fv;
  caml_register_global_root (fvp);
  component_cleanup (co, do_remove_callback, fvp);

  newtComponentAddCallback (co, do_callback, fvp);

  CAMLreturn (Val_unit);
}

CAMLprim
newt_stubs_newtComponentAddress (value cov)
{
  CAMLparam1 (cov);
  CAMLlocal1 (rv);
  newtComponent co;

  camlidl_ml2c_newt_int_newtComponent (cov, &co, 0);

  rv = caml_copy_int64 ((long) co);
  CAMLreturn (rv);
}
