/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2000, 2010 Oracle and/or its affiliates.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

#include <cppunit/simpleheader.hxx>
#include <odiapi/qname/QName.hxx>
#include <stdio.h>
#include <rtl/string.hxx>
#include <rtl/random.h>
#include <osl/time.h>
#include <hash_set>
#include <stdexcept>


using namespace writerfilter;


class TestQName : public CppUnit::TestFixture
{
public:
    void testAdd(const sal_Char* namespaceUri, const sal_Char *localName)
    {
        QName_t t=QName::tokenizer().insert(namespaceUri, localName);
        const sal_Char *ln=QName::serializer().getLocalName(t);
        const sal_Char *ns=QName::serializer().getNamespaceUri(t);
        CPPUNIT_ASSERT(rtl_str_compare(localName, ln)==0);
        CPPUNIT_ASSERT(rtl_str_compare(namespaceUri, ns)==0);
    }

    void test()
    { 
        rtlRandomPool rndPool=rtl_random_createPool();
        int numOfIterations=100000; //int numOfIterations=5000000;
        for(int i=0;i<numOfIterations;i++)
        {
            unsigned int id;
            rtl_random_getBytes(rndPool, &id, sizeof(id));
            id%=QName::tokenizer().getQNames();
            const sal_Char *ln=QName::serializer().getLocalName((QName_t)id);
            const sal_Char *ns=QName::serializer().getNamespaceUri((QName_t)id);
//		    printf("%i \"%s\" \"%s\"\n", i, ns, ln);
            testAdd(ns, ln);
        }
        rtl_random_destroyPool(rndPool);
    }

    void testperf()
    { 
        rtlRandomPool rndPool=rtl_random_createPool();
        int numOfIterations=5000000;
        TimeValue t1; osl_getSystemTime(&t1);
        for(int i=0;i<numOfIterations;i++)
        {
            unsigned int id;
            rtl_random_getBytes(rndPool, &id, sizeof(id));
            id%=QName::tokenizer().getQNames();
            const sal_Char *ln=QName::serializer().getLocalName((QName_t)id);
            const sal_Char *ns=QName::serializer().getNamespaceUri((QName_t)id);
            QName::tokenizer().insert(ns, ln);
        }
        TimeValue t2; osl_getSystemTime(&t2);
        printf("Events=%i time=%is time/event=%0.10fs\n", numOfIterations, t2.Seconds-t1.Seconds, (double)(t2.Seconds-t1.Seconds)/(double)numOfIterations);
        rtl_random_destroyPool(rndPool);
    }


    void testinsert()
    { 
        testAdd("myns", "myname");
    }

    void testdos()
    {
        bool flag=false;
        try {
            int i=0;
            for(;;)
            {
                char buf[100];
                sprintf(buf, "name%i", i++);
                testAdd("myns", buf);
            }
        } catch (std::logic_error &e)
        {
            // only way to exit this function
            flag=true;
        }
        CPPUNIT_ASSERT_MESSAGE("dos test failed", flag);
    }


    void test_anium(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:animation:1.0";
        flag&=QName::tokenizer().insert(ns, "animate") == NS_anium::LN_animate;
        flag&=QName::tokenizer().insert(ns, "set") == NS_anium::LN_set;
        flag&=QName::tokenizer().insert(ns, "animateMotion") == NS_anium::LN_animateMotion;
        flag&=QName::tokenizer().insert(ns, "animateColor") == NS_anium::LN_animateColor;
        flag&=QName::tokenizer().insert(ns, "color-interpolation") == NS_anium::LN_color_interpolation;
        flag&=QName::tokenizer().insert(ns, "color-interpolation-direction") == NS_anium::LN_color_interpolation_direction;
        flag&=QName::tokenizer().insert(ns, "animateTransform") == NS_anium::LN_animateTransform;
        flag&=QName::tokenizer().insert(ns, "transitionFilter") == NS_anium::LN_transitionFilter;
        flag&=QName::tokenizer().insert(ns, "id") == NS_anium::LN_id;
        flag&=QName::tokenizer().insert(ns, "sub-item") == NS_anium::LN_sub_item;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_anium::LN_formula;
        flag&=QName::tokenizer().insert(ns, "par") == NS_anium::LN_par;
        flag&=QName::tokenizer().insert(ns, "seq") == NS_anium::LN_seq;
        flag&=QName::tokenizer().insert(ns, "iterate") == NS_anium::LN_iterate;
        flag&=QName::tokenizer().insert(ns, "iterate-type") == NS_anium::LN_iterate_type;
        flag&=QName::tokenizer().insert(ns, "iterate-interval") == NS_anium::LN_iterate_interval;
        flag&=QName::tokenizer().insert(ns, "audio") == NS_anium::LN_audio;
        flag&=QName::tokenizer().insert(ns, "audio-level") == NS_anium::LN_audio_level;
        flag&=QName::tokenizer().insert(ns, "command") == NS_anium::LN_command;
        flag&=QName::tokenizer().insert(ns, "param") == NS_anium::LN_param;
        flag&=QName::tokenizer().insert(ns, "name") == NS_anium::LN_name;
        flag&=QName::tokenizer().insert(ns, "value") == NS_anium::LN_value;
        CPPUNIT_ASSERT_MESSAGE("test for namespace anium failed", flag);}
    void test_math(){
        bool flag=true;
        static const char ns[] = "http://www.w3.org/1998/Math/MathML";
        flag&=QName::tokenizer().insert(ns, "math") == NS_math::LN_math;
        CPPUNIT_ASSERT_MESSAGE("test for namespace math failed", flag);}
    void test_w10(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:word";
        flag&=QName::tokenizer().insert(ns, "bordertop") == NS_w10::LN_bordertop;
        flag&=QName::tokenizer().insert(ns, "borderleft") == NS_w10::LN_borderleft;
        flag&=QName::tokenizer().insert(ns, "borderright") == NS_w10::LN_borderright;
        flag&=QName::tokenizer().insert(ns, "borderbottom") == NS_w10::LN_borderbottom;
        flag&=QName::tokenizer().insert(ns, "wrap") == NS_w10::LN_wrap;
        flag&=QName::tokenizer().insert(ns, "anchorlock") == NS_w10::LN_anchorlock;
        CPPUNIT_ASSERT_MESSAGE("test for namespace w10 failed", flag);}
    void test_chart(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:chart:1.0";
        flag&=QName::tokenizer().insert(ns, "scale-text") == NS_chart::LN_scale_text;
        flag&=QName::tokenizer().insert(ns, "three-dimensional") == NS_chart::LN_three_dimensional;
        flag&=QName::tokenizer().insert(ns, "deep") == NS_chart::LN_deep;
        flag&=QName::tokenizer().insert(ns, "symbol-type") == NS_chart::LN_symbol_type;
        flag&=QName::tokenizer().insert(ns, "symbol-name") == NS_chart::LN_symbol_name;
        flag&=QName::tokenizer().insert(ns, "symbol-image") == NS_chart::LN_symbol_image;
        flag&=QName::tokenizer().insert(ns, "symbol-width") == NS_chart::LN_symbol_width;
        flag&=QName::tokenizer().insert(ns, "symbol-height") == NS_chart::LN_symbol_height;
        flag&=QName::tokenizer().insert(ns, "vertical") == NS_chart::LN_vertical;
        flag&=QName::tokenizer().insert(ns, "connect-bars") == NS_chart::LN_connect_bars;
        flag&=QName::tokenizer().insert(ns, "gap-width") == NS_chart::LN_gap_width;
        flag&=QName::tokenizer().insert(ns, "overlap") == NS_chart::LN_overlap;
        flag&=QName::tokenizer().insert(ns, "japanese-candle-stick") == NS_chart::LN_japanese_candle_stick;
        flag&=QName::tokenizer().insert(ns, "interpolation") == NS_chart::LN_interpolation;
        flag&=QName::tokenizer().insert(ns, "spline-order") == NS_chart::LN_spline_order;
        flag&=QName::tokenizer().insert(ns, "spline-resolution") == NS_chart::LN_spline_resolution;
        flag&=QName::tokenizer().insert(ns, "pie-offset") == NS_chart::LN_pie_offset;
        flag&=QName::tokenizer().insert(ns, "lines") == NS_chart::LN_lines;
        flag&=QName::tokenizer().insert(ns, "solid-type") == NS_chart::LN_solid_type;
        flag&=QName::tokenizer().insert(ns, "stacked") == NS_chart::LN_stacked;
        flag&=QName::tokenizer().insert(ns, "percentage") == NS_chart::LN_percentage;
        flag&=QName::tokenizer().insert(ns, "link-data-style-to-source") == NS_chart::LN_link_data_style_to_source;
        flag&=QName::tokenizer().insert(ns, "visible") == NS_chart::LN_visible;
        flag&=QName::tokenizer().insert(ns, "logarithmic") == NS_chart::LN_logarithmic;
        flag&=QName::tokenizer().insert(ns, "maximum") == NS_chart::LN_maximum;
        flag&=QName::tokenizer().insert(ns, "minimum") == NS_chart::LN_minimum;
        flag&=QName::tokenizer().insert(ns, "origin") == NS_chart::LN_origin;
        flag&=QName::tokenizer().insert(ns, "interval-major") == NS_chart::LN_interval_major;
        flag&=QName::tokenizer().insert(ns, "interval-minor") == NS_chart::LN_interval_minor;
        flag&=QName::tokenizer().insert(ns, "tick-marks-major-inner") == NS_chart::LN_tick_marks_major_inner;
        flag&=QName::tokenizer().insert(ns, "tick-marks-major-outer") == NS_chart::LN_tick_marks_major_outer;
        flag&=QName::tokenizer().insert(ns, "tick-marks-minor-inner") == NS_chart::LN_tick_marks_minor_inner;
        flag&=QName::tokenizer().insert(ns, "tick-marks-minor-outer") == NS_chart::LN_tick_marks_minor_outer;
        flag&=QName::tokenizer().insert(ns, "display-label") == NS_chart::LN_display_label;
        flag&=QName::tokenizer().insert(ns, "text-overlap") == NS_chart::LN_text_overlap;
        flag&=QName::tokenizer().insert(ns, "label-arrangement") == NS_chart::LN_label_arrangement;
        flag&=QName::tokenizer().insert(ns, "data-label-number") == NS_chart::LN_data_label_number;
        flag&=QName::tokenizer().insert(ns, "data-label-text") == NS_chart::LN_data_label_text;
        flag&=QName::tokenizer().insert(ns, "data-label-symbol") == NS_chart::LN_data_label_symbol;
        flag&=QName::tokenizer().insert(ns, "error-category") == NS_chart::LN_error_category;
        flag&=QName::tokenizer().insert(ns, "error-percentage") == NS_chart::LN_error_percentage;
        flag&=QName::tokenizer().insert(ns, "error-margin") == NS_chart::LN_error_margin;
        flag&=QName::tokenizer().insert(ns, "error-lower-limit") == NS_chart::LN_error_lower_limit;
        flag&=QName::tokenizer().insert(ns, "error-upper-limit") == NS_chart::LN_error_upper_limit;
        flag&=QName::tokenizer().insert(ns, "error-upper-indicator") == NS_chart::LN_error_upper_indicator;
        flag&=QName::tokenizer().insert(ns, "error-lower-indicator") == NS_chart::LN_error_lower_indicator;
        flag&=QName::tokenizer().insert(ns, "series-source") == NS_chart::LN_series_source;
        flag&=QName::tokenizer().insert(ns, "regression-type") == NS_chart::LN_regression_type;
        flag&=QName::tokenizer().insert(ns, "chart") == NS_chart::LN_chart;
        flag&=QName::tokenizer().insert(ns, "class") == NS_chart::LN_class;
        flag&=QName::tokenizer().insert(ns, "column-mapping") == NS_chart::LN_column_mapping;
        flag&=QName::tokenizer().insert(ns, "row-mapping") == NS_chart::LN_row_mapping;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_chart::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "title") == NS_chart::LN_title;
        flag&=QName::tokenizer().insert(ns, "subtitle") == NS_chart::LN_subtitle;
        flag&=QName::tokenizer().insert(ns, "footer") == NS_chart::LN_footer;
        flag&=QName::tokenizer().insert(ns, "legend") == NS_chart::LN_legend;
        flag&=QName::tokenizer().insert(ns, "legend-position") == NS_chart::LN_legend_position;
        flag&=QName::tokenizer().insert(ns, "legend-align") == NS_chart::LN_legend_align;
        flag&=QName::tokenizer().insert(ns, "plot-area") == NS_chart::LN_plot_area;
        flag&=QName::tokenizer().insert(ns, "data-source-has-labels") == NS_chart::LN_data_source_has_labels;
        flag&=QName::tokenizer().insert(ns, "wall") == NS_chart::LN_wall;
        flag&=QName::tokenizer().insert(ns, "floor") == NS_chart::LN_floor;
        flag&=QName::tokenizer().insert(ns, "axis") == NS_chart::LN_axis;
        flag&=QName::tokenizer().insert(ns, "dimension") == NS_chart::LN_dimension;
        flag&=QName::tokenizer().insert(ns, "name") == NS_chart::LN_name;
        flag&=QName::tokenizer().insert(ns, "grid") == NS_chart::LN_grid;
        flag&=QName::tokenizer().insert(ns, "series") == NS_chart::LN_series;
        flag&=QName::tokenizer().insert(ns, "values-cell-range-address") == NS_chart::LN_values_cell_range_address;
        flag&=QName::tokenizer().insert(ns, "label-cell-address") == NS_chart::LN_label_cell_address;
        flag&=QName::tokenizer().insert(ns, "attached-axis") == NS_chart::LN_attached_axis;
        flag&=QName::tokenizer().insert(ns, "domain") == NS_chart::LN_domain;
        flag&=QName::tokenizer().insert(ns, "categories") == NS_chart::LN_categories;
        flag&=QName::tokenizer().insert(ns, "data-point") == NS_chart::LN_data_point;
        flag&=QName::tokenizer().insert(ns, "repeated") == NS_chart::LN_repeated;
        flag&=QName::tokenizer().insert(ns, "mean-value") == NS_chart::LN_mean_value;
        flag&=QName::tokenizer().insert(ns, "error-indicator") == NS_chart::LN_error_indicator;
        flag&=QName::tokenizer().insert(ns, "regression-curve") == NS_chart::LN_regression_curve;
        flag&=QName::tokenizer().insert(ns, "stock-gain-marker") == NS_chart::LN_stock_gain_marker;
        flag&=QName::tokenizer().insert(ns, "stock-loss-marker") == NS_chart::LN_stock_loss_marker;
        flag&=QName::tokenizer().insert(ns, "stock-range-line") == NS_chart::LN_stock_range_line;
        CPPUNIT_ASSERT_MESSAGE("test for namespace chart failed", flag);}
    void test_sc12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/singlecells";
        flag&=QName::tokenizer().insert(ns, "singleCells") == NS_sc12::LN_singleCells;
        flag&=QName::tokenizer().insert(ns, "table") == NS_sc12::LN_table;
        CPPUNIT_ASSERT_MESSAGE("test for namespace sc12 failed", flag);}
    void test_odc12(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:odc";
        flag&=QName::tokenizer().insert(ns, "OfficeDataConnection") == NS_odc12::LN_OfficeDataConnection;
        flag&=QName::tokenizer().insert(ns, "Connection") == NS_odc12::LN_Connection;
        flag&=QName::tokenizer().insert(ns, "ConnectionString") == NS_odc12::LN_ConnectionString;
        flag&=QName::tokenizer().insert(ns, "CommandType") == NS_odc12::LN_CommandType;
        flag&=QName::tokenizer().insert(ns, "CommandText") == NS_odc12::LN_CommandText;
        flag&=QName::tokenizer().insert(ns, "Parameter") == NS_odc12::LN_Parameter;
        flag&=QName::tokenizer().insert(ns, "SourceFile") == NS_odc12::LN_SourceFile;
        flag&=QName::tokenizer().insert(ns, "HeaderFile") == NS_odc12::LN_HeaderFile;
        flag&=QName::tokenizer().insert(ns, "MailMerge") == NS_odc12::LN_MailMerge;
        flag&=QName::tokenizer().insert(ns, "Version") == NS_odc12::LN_Version;
        flag&=QName::tokenizer().insert(ns, "AlwaysUseConnectionFile") == NS_odc12::LN_AlwaysUseConnectionFile;
        flag&=QName::tokenizer().insert(ns, "CredentialsMethod") == NS_odc12::LN_CredentialsMethod;
        flag&=QName::tokenizer().insert(ns, "SSOApplicationID") == NS_odc12::LN_SSOApplicationID;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_odc12::LN_Name;
        flag&=QName::tokenizer().insert(ns, "DataType") == NS_odc12::LN_DataType;
        CPPUNIT_ASSERT_MESSAGE("test for namespace odc12 failed", flag);}
    void test_c11(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:component:spreadsheet";
        flag&=QName::tokenizer().insert(ns, "OWCVersion") == NS_c11::LN_OWCVersion;
        flag&=QName::tokenizer().insert(ns, "worksheetoptions") == NS_c11::LN_worksheetoptions;
        flag&=QName::tokenizer().insert(ns, "displaycustomheaders") == NS_c11::LN_displaycustomheaders;
        flag&=QName::tokenizer().insert(ns, "ComponentOptions") == NS_c11::LN_ComponentOptions;
        flag&=QName::tokenizer().insert(ns, "Toolbar") == NS_c11::LN_Toolbar;
        flag&=QName::tokenizer().insert(ns, "HideOfficeLogo") == NS_c11::LN_HideOfficeLogo;
        flag&=QName::tokenizer().insert(ns, "SpreadsheetAutoFit") == NS_c11::LN_SpreadsheetAutoFit;
        flag&=QName::tokenizer().insert(ns, "Label") == NS_c11::LN_Label;
        flag&=QName::tokenizer().insert(ns, "Caption") == NS_c11::LN_Caption;
        flag&=QName::tokenizer().insert(ns, "MaxHeight") == NS_c11::LN_MaxHeight;
        flag&=QName::tokenizer().insert(ns, "MaxWidth") == NS_c11::LN_MaxWidth;
        flag&=QName::tokenizer().insert(ns, "NextSheetNumber") == NS_c11::LN_NextSheetNumber;
        CPPUNIT_ASSERT_MESSAGE("test for namespace c11 failed", flag);}
    void test_rng(){
        bool flag=true;
        static const char ns[] = "http://relaxng.org/ns/structure/1.0";
        flag&=QName::tokenizer().insert(ns, "anyName") == NS_rng::LN_anyName;
        flag&=QName::tokenizer().insert(ns, "grammar") == NS_rng::LN_grammar;
        flag&=QName::tokenizer().insert(ns, "define") == NS_rng::LN_define;
        flag&=QName::tokenizer().insert(ns, "optional") == NS_rng::LN_optional;
        flag&=QName::tokenizer().insert(ns, "attribute") == NS_rng::LN_attribute;
        flag&=QName::tokenizer().insert(ns, "ref") == NS_rng::LN_ref;
        flag&=QName::tokenizer().insert(ns, "start") == NS_rng::LN_start;
        flag&=QName::tokenizer().insert(ns, "choice") == NS_rng::LN_choice;
        flag&=QName::tokenizer().insert(ns, "element") == NS_rng::LN_element;
        flag&=QName::tokenizer().insert(ns, "zeroOrMore") == NS_rng::LN_zeroOrMore;
        flag&=QName::tokenizer().insert(ns, "group") == NS_rng::LN_group;
        flag&=QName::tokenizer().insert(ns, "empty") == NS_rng::LN_empty;
        flag&=QName::tokenizer().insert(ns, "oneOrMore") == NS_rng::LN_oneOrMore;
        flag&=QName::tokenizer().insert(ns, "text") == NS_rng::LN_text;
        flag&=QName::tokenizer().insert(ns, "value") == NS_rng::LN_value;
        flag&=QName::tokenizer().insert(ns, "mixed") == NS_rng::LN_mixed;
        flag&=QName::tokenizer().insert(ns, "interleave") == NS_rng::LN_interleave;
        flag&=QName::tokenizer().insert(ns, "data") == NS_rng::LN_data;
        flag&=QName::tokenizer().insert(ns, "param") == NS_rng::LN_param;
        flag&=QName::tokenizer().insert(ns, "except") == NS_rng::LN_except;
        flag&=QName::tokenizer().insert(ns, "nsName") == NS_rng::LN_nsName;
        flag&=QName::tokenizer().insert(ns, "parentRef") == NS_rng::LN_parentRef;
        flag&=QName::tokenizer().insert(ns, "notAllowed") == NS_rng::LN_notAllowed;
        flag&=QName::tokenizer().insert(ns, "externalRef") == NS_rng::LN_externalRef;
        flag&=QName::tokenizer().insert(ns, "div") == NS_rng::LN_div;
        flag&=QName::tokenizer().insert(ns, "include") == NS_rng::LN_include;
        flag&=QName::tokenizer().insert(ns, "list") == NS_rng::LN_list;
        flag&=QName::tokenizer().insert(ns, "name") == NS_rng::LN_name;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rng failed", flag);}
    void test_meta(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:meta:1.0";
        flag&=QName::tokenizer().insert(ns, "date-string") == NS_meta::LN_date_string;
        flag&=QName::tokenizer().insert(ns, "generator") == NS_meta::LN_generator;
        flag&=QName::tokenizer().insert(ns, "keyword") == NS_meta::LN_keyword;
        flag&=QName::tokenizer().insert(ns, "initial-creator") == NS_meta::LN_initial_creator;
        flag&=QName::tokenizer().insert(ns, "printed-by") == NS_meta::LN_printed_by;
        flag&=QName::tokenizer().insert(ns, "creation-date") == NS_meta::LN_creation_date;
        flag&=QName::tokenizer().insert(ns, "print-date") == NS_meta::LN_print_date;
        flag&=QName::tokenizer().insert(ns, "template") == NS_meta::LN_template;
        flag&=QName::tokenizer().insert(ns, "date") == NS_meta::LN_date;
        flag&=QName::tokenizer().insert(ns, "auto-reload") == NS_meta::LN_auto_reload;
        flag&=QName::tokenizer().insert(ns, "delay") == NS_meta::LN_delay;
        flag&=QName::tokenizer().insert(ns, "hyperlink-behaviour") == NS_meta::LN_hyperlink_behaviour;
        flag&=QName::tokenizer().insert(ns, "editing-cycles") == NS_meta::LN_editing_cycles;
        flag&=QName::tokenizer().insert(ns, "editing-duration") == NS_meta::LN_editing_duration;
        flag&=QName::tokenizer().insert(ns, "document-statistic") == NS_meta::LN_document_statistic;
        flag&=QName::tokenizer().insert(ns, "page-count") == NS_meta::LN_page_count;
        flag&=QName::tokenizer().insert(ns, "table-count") == NS_meta::LN_table_count;
        flag&=QName::tokenizer().insert(ns, "draw-count") == NS_meta::LN_draw_count;
        flag&=QName::tokenizer().insert(ns, "image-count") == NS_meta::LN_image_count;
        flag&=QName::tokenizer().insert(ns, "ole-object-count") == NS_meta::LN_ole_object_count;
        flag&=QName::tokenizer().insert(ns, "paragraph-count") == NS_meta::LN_paragraph_count;
        flag&=QName::tokenizer().insert(ns, "word-count") == NS_meta::LN_word_count;
        flag&=QName::tokenizer().insert(ns, "character-count") == NS_meta::LN_character_count;
        flag&=QName::tokenizer().insert(ns, "frame-count") == NS_meta::LN_frame_count;
        flag&=QName::tokenizer().insert(ns, "sentence-count") == NS_meta::LN_sentence_count;
        flag&=QName::tokenizer().insert(ns, "syllable-count") == NS_meta::LN_syllable_count;
        flag&=QName::tokenizer().insert(ns, "non-whitespace-character-count") == NS_meta::LN_non_whitespace_character_count;
        flag&=QName::tokenizer().insert(ns, "row-count") == NS_meta::LN_row_count;
        flag&=QName::tokenizer().insert(ns, "cell-count") == NS_meta::LN_cell_count;
        flag&=QName::tokenizer().insert(ns, "object-count") == NS_meta::LN_object_count;
        flag&=QName::tokenizer().insert(ns, "user-defined") == NS_meta::LN_user_defined;
        flag&=QName::tokenizer().insert(ns, "name") == NS_meta::LN_name;
        flag&=QName::tokenizer().insert(ns, "value-type") == NS_meta::LN_value_type;
        CPPUNIT_ASSERT_MESSAGE("test for namespace meta failed", flag);}
    void test_udcs(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/data/udc/soap";
        flag&=QName::tokenizer().insert(ns, "Location") == NS_udcs::LN_Location;
        flag&=QName::tokenizer().insert(ns, "Proxy") == NS_udcs::LN_Proxy;
        flag&=QName::tokenizer().insert(ns, "Authentication") == NS_udcs::LN_Authentication;
        flag&=QName::tokenizer().insert(ns, "Windows") == NS_udcs::LN_Windows;
        flag&=QName::tokenizer().insert(ns, "Basic") == NS_udcs::LN_Basic;
        flag&=QName::tokenizer().insert(ns, "useExplicit") == NS_udcs::LN_useExplicit;
        flag&=QName::tokenizer().insert(ns, "userId") == NS_udcs::LN_userId;
        flag&=QName::tokenizer().insert(ns, "password") == NS_udcs::LN_password;
        flag&=QName::tokenizer().insert(ns, "SOAPAction") == NS_udcs::LN_SOAPAction;
        CPPUNIT_ASSERT_MESSAGE("test for namespace udcs failed", flag);}
    void test_xlink(){
        bool flag=true;
        static const char ns[] = "http://www.w3.org/1999/xlink";
        flag&=QName::tokenizer().insert(ns, "href") == NS_xlink::LN_href;
        flag&=QName::tokenizer().insert(ns, "type") == NS_xlink::LN_type;
        flag&=QName::tokenizer().insert(ns, "actuate") == NS_xlink::LN_actuate;
        flag&=QName::tokenizer().insert(ns, "title") == NS_xlink::LN_title;
        flag&=QName::tokenizer().insert(ns, "show") == NS_xlink::LN_show;
        CPPUNIT_ASSERT_MESSAGE("test for namespace xlink failed", flag);}
    void test_style(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:style:1.0";
        flag&=QName::tokenizer().insert(ns, "chart-properties") == NS_style::LN_chart_properties;
        flag&=QName::tokenizer().insert(ns, "num-prefix") == NS_style::LN_num_prefix;
        flag&=QName::tokenizer().insert(ns, "num-suffix") == NS_style::LN_num_suffix;
        flag&=QName::tokenizer().insert(ns, "num-format") == NS_style::LN_num_format;
        flag&=QName::tokenizer().insert(ns, "num-letter-sync") == NS_style::LN_num_letter_sync;
        flag&=QName::tokenizer().insert(ns, "style") == NS_style::LN_style;
        flag&=QName::tokenizer().insert(ns, "name") == NS_style::LN_name;
        flag&=QName::tokenizer().insert(ns, "display-name") == NS_style::LN_display_name;
        flag&=QName::tokenizer().insert(ns, "parent-style-name") == NS_style::LN_parent_style_name;
        flag&=QName::tokenizer().insert(ns, "next-style-name") == NS_style::LN_next_style_name;
        flag&=QName::tokenizer().insert(ns, "list-style-name") == NS_style::LN_list_style_name;
        flag&=QName::tokenizer().insert(ns, "master-page-name") == NS_style::LN_master_page_name;
        flag&=QName::tokenizer().insert(ns, "auto-update") == NS_style::LN_auto_update;
        flag&=QName::tokenizer().insert(ns, "class") == NS_style::LN_class;
        flag&=QName::tokenizer().insert(ns, "default-outline-level") == NS_style::LN_default_outline_level;
        flag&=QName::tokenizer().insert(ns, "map") == NS_style::LN_map;
        flag&=QName::tokenizer().insert(ns, "condition") == NS_style::LN_condition;
        flag&=QName::tokenizer().insert(ns, "apply-style-name") == NS_style::LN_apply_style_name;
        flag&=QName::tokenizer().insert(ns, "base-cell-address") == NS_style::LN_base_cell_address;
        flag&=QName::tokenizer().insert(ns, "default-style") == NS_style::LN_default_style;
        flag&=QName::tokenizer().insert(ns, "page-layout") == NS_style::LN_page_layout;
        flag&=QName::tokenizer().insert(ns, "page-usage") == NS_style::LN_page_usage;
        flag&=QName::tokenizer().insert(ns, "header-style") == NS_style::LN_header_style;
        flag&=QName::tokenizer().insert(ns, "footer-style") == NS_style::LN_footer_style;
        flag&=QName::tokenizer().insert(ns, "master-page") == NS_style::LN_master_page;
        flag&=QName::tokenizer().insert(ns, "header") == NS_style::LN_header;
        flag&=QName::tokenizer().insert(ns, "footer") == NS_style::LN_footer;
        flag&=QName::tokenizer().insert(ns, "header-left") == NS_style::LN_header_left;
        flag&=QName::tokenizer().insert(ns, "footer-left") == NS_style::LN_footer_left;
        flag&=QName::tokenizer().insert(ns, "display") == NS_style::LN_display;
        flag&=QName::tokenizer().insert(ns, "region-left") == NS_style::LN_region_left;
        flag&=QName::tokenizer().insert(ns, "region-center") == NS_style::LN_region_center;
        flag&=QName::tokenizer().insert(ns, "region-right") == NS_style::LN_region_right;
        flag&=QName::tokenizer().insert(ns, "font-face") == NS_style::LN_font_face;
        flag&=QName::tokenizer().insert(ns, "font-adornments") == NS_style::LN_font_adornments;
        flag&=QName::tokenizer().insert(ns, "font-family-generic") == NS_style::LN_font_family_generic;
        flag&=QName::tokenizer().insert(ns, "font-pitch") == NS_style::LN_font_pitch;
        flag&=QName::tokenizer().insert(ns, "font-charset") == NS_style::LN_font_charset;
        flag&=QName::tokenizer().insert(ns, "volatile") == NS_style::LN_volatile;
        flag&=QName::tokenizer().insert(ns, "leader-char") == NS_style::LN_leader_char;
        flag&=QName::tokenizer().insert(ns, "type") == NS_style::LN_type;
        flag&=QName::tokenizer().insert(ns, "position") == NS_style::LN_position;
        flag&=QName::tokenizer().insert(ns, "family") == NS_style::LN_family;
        flag&=QName::tokenizer().insert(ns, "font-size-asian") == NS_style::LN_font_size_asian;
        flag&=QName::tokenizer().insert(ns, "font-size-complex") == NS_style::LN_font_size_complex;
        flag&=QName::tokenizer().insert(ns, "font-size-rel") == NS_style::LN_font_size_rel;
        flag&=QName::tokenizer().insert(ns, "font-size-rel-asian") == NS_style::LN_font_size_rel_asian;
        flag&=QName::tokenizer().insert(ns, "font-size-rel-complex") == NS_style::LN_font_size_rel_complex;
        flag&=QName::tokenizer().insert(ns, "script-type") == NS_style::LN_script_type;
        flag&=QName::tokenizer().insert(ns, "language-asian") == NS_style::LN_language_asian;
        flag&=QName::tokenizer().insert(ns, "language-complex") == NS_style::LN_language_complex;
        flag&=QName::tokenizer().insert(ns, "graphic-properties") == NS_style::LN_graphic_properties;
        flag&=QName::tokenizer().insert(ns, "country-asian") == NS_style::LN_country_asian;
        flag&=QName::tokenizer().insert(ns, "drawing-page-properties") == NS_style::LN_drawing_page_properties;
        flag&=QName::tokenizer().insert(ns, "country-complex") == NS_style::LN_country_complex;
        flag&=QName::tokenizer().insert(ns, "font-style-asian") == NS_style::LN_font_style_asian;
        flag&=QName::tokenizer().insert(ns, "font-style-complex") == NS_style::LN_font_style_complex;
        flag&=QName::tokenizer().insert(ns, "font-relief") == NS_style::LN_font_relief;
        flag&=QName::tokenizer().insert(ns, "text-underline-type") == NS_style::LN_text_underline_type;
        flag&=QName::tokenizer().insert(ns, "text-underline-style") == NS_style::LN_text_underline_style;
        flag&=QName::tokenizer().insert(ns, "text-underline-width") == NS_style::LN_text_underline_width;
        flag&=QName::tokenizer().insert(ns, "text-underline-color") == NS_style::LN_text_underline_color;
        flag&=QName::tokenizer().insert(ns, "font-weight-asian") == NS_style::LN_font_weight_asian;
        flag&=QName::tokenizer().insert(ns, "font-weight-complex") == NS_style::LN_font_weight_complex;
        flag&=QName::tokenizer().insert(ns, "text-underline-mode") == NS_style::LN_text_underline_mode;
        flag&=QName::tokenizer().insert(ns, "text-line-through-mode") == NS_style::LN_text_line_through_mode;
        flag&=QName::tokenizer().insert(ns, "letter-kerning") == NS_style::LN_letter_kerning;
        flag&=QName::tokenizer().insert(ns, "text-blinking") == NS_style::LN_text_blinking;
        flag&=QName::tokenizer().insert(ns, "text-combine") == NS_style::LN_text_combine;
        flag&=QName::tokenizer().insert(ns, "text-combine-start-char") == NS_style::LN_text_combine_start_char;
        flag&=QName::tokenizer().insert(ns, "text-combine-end-char") == NS_style::LN_text_combine_end_char;
        flag&=QName::tokenizer().insert(ns, "text-emphasize") == NS_style::LN_text_emphasize;
        flag&=QName::tokenizer().insert(ns, "text-scale") == NS_style::LN_text_scale;
        flag&=QName::tokenizer().insert(ns, "text-rotation-angle") == NS_style::LN_text_rotation_angle;
        flag&=QName::tokenizer().insert(ns, "text-rotation-scale") == NS_style::LN_text_rotation_scale;
        flag&=QName::tokenizer().insert(ns, "presentation-page-layout") == NS_style::LN_presentation_page_layout;
        flag&=QName::tokenizer().insert(ns, "page-layout-properties") == NS_style::LN_page_layout_properties;
        flag&=QName::tokenizer().insert(ns, "paragraph-properties") == NS_style::LN_paragraph_properties;
        flag&=QName::tokenizer().insert(ns, "paper-tray-name") == NS_style::LN_paper_tray_name;
        flag&=QName::tokenizer().insert(ns, "print-orientation") == NS_style::LN_print_orientation;
        flag&=QName::tokenizer().insert(ns, "line-height-at-least") == NS_style::LN_line_height_at_least;
        flag&=QName::tokenizer().insert(ns, "register-truth-ref-style-name") == NS_style::LN_register_truth_ref_style_name;
        flag&=QName::tokenizer().insert(ns, "line-spacing") == NS_style::LN_line_spacing;
        flag&=QName::tokenizer().insert(ns, "print") == NS_style::LN_print;
        flag&=QName::tokenizer().insert(ns, "print-page-order") == NS_style::LN_print_page_order;
        flag&=QName::tokenizer().insert(ns, "font-independent-line-spacing") == NS_style::LN_font_independent_line_spacing;
        flag&=QName::tokenizer().insert(ns, "first-page-number") == NS_style::LN_first_page_number;
        flag&=QName::tokenizer().insert(ns, "scale-to") == NS_style::LN_scale_to;
        flag&=QName::tokenizer().insert(ns, "scale-to-pages") == NS_style::LN_scale_to_pages;
        flag&=QName::tokenizer().insert(ns, "justify-single-word") == NS_style::LN_justify_single_word;
        flag&=QName::tokenizer().insert(ns, "table-centering") == NS_style::LN_table_centering;
        flag&=QName::tokenizer().insert(ns, "footnote-max-height") == NS_style::LN_footnote_max_height;
        flag&=QName::tokenizer().insert(ns, "footnote-sep") == NS_style::LN_footnote_sep;
        flag&=QName::tokenizer().insert(ns, "width") == NS_style::LN_width;
        flag&=QName::tokenizer().insert(ns, "tab-stops") == NS_style::LN_tab_stops;
        flag&=QName::tokenizer().insert(ns, "color") == NS_style::LN_color;
        flag&=QName::tokenizer().insert(ns, "tab-stop") == NS_style::LN_tab_stop;
        flag&=QName::tokenizer().insert(ns, "char") == NS_style::LN_char;
        flag&=QName::tokenizer().insert(ns, "line-style") == NS_style::LN_line_style;
        flag&=QName::tokenizer().insert(ns, "leader-type") == NS_style::LN_leader_type;
        flag&=QName::tokenizer().insert(ns, "adjustment") == NS_style::LN_adjustment;
        flag&=QName::tokenizer().insert(ns, "leader-style") == NS_style::LN_leader_style;
        flag&=QName::tokenizer().insert(ns, "distance-before-sep") == NS_style::LN_distance_before_sep;
        flag&=QName::tokenizer().insert(ns, "leader-width") == NS_style::LN_leader_width;
        flag&=QName::tokenizer().insert(ns, "distance-after-sep") == NS_style::LN_distance_after_sep;
        flag&=QName::tokenizer().insert(ns, "leader-color") == NS_style::LN_leader_color;
        flag&=QName::tokenizer().insert(ns, "layout-grid-mode") == NS_style::LN_layout_grid_mode;
        flag&=QName::tokenizer().insert(ns, "leader-text") == NS_style::LN_leader_text;
        flag&=QName::tokenizer().insert(ns, "leader-text-style") == NS_style::LN_leader_text_style;
        flag&=QName::tokenizer().insert(ns, "layout-grid-base-height") == NS_style::LN_layout_grid_base_height;
        flag&=QName::tokenizer().insert(ns, "tab-stop-distance") == NS_style::LN_tab_stop_distance;
        flag&=QName::tokenizer().insert(ns, "layout-grid-ruby-height") == NS_style::LN_layout_grid_ruby_height;
        flag&=QName::tokenizer().insert(ns, "layout-grid-lines") == NS_style::LN_layout_grid_lines;
        flag&=QName::tokenizer().insert(ns, "layout-grid-color") == NS_style::LN_layout_grid_color;
        flag&=QName::tokenizer().insert(ns, "drop-cap") == NS_style::LN_drop_cap;
        flag&=QName::tokenizer().insert(ns, "length") == NS_style::LN_length;
        flag&=QName::tokenizer().insert(ns, "layout-grid-ruby-below") == NS_style::LN_layout_grid_ruby_below;
        flag&=QName::tokenizer().insert(ns, "lines") == NS_style::LN_lines;
        flag&=QName::tokenizer().insert(ns, "distance") == NS_style::LN_distance;
        flag&=QName::tokenizer().insert(ns, "layout-grid-print") == NS_style::LN_layout_grid_print;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_style::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "register-true") == NS_style::LN_register_true;
        flag&=QName::tokenizer().insert(ns, "layout-grid-display") == NS_style::LN_layout_grid_display;
        flag&=QName::tokenizer().insert(ns, "header-footer-properties") == NS_style::LN_header_footer_properties;
        flag&=QName::tokenizer().insert(ns, "dynamic-spacing") == NS_style::LN_dynamic_spacing;
        flag&=QName::tokenizer().insert(ns, "auto-text-indent") == NS_style::LN_auto_text_indent;
        flag&=QName::tokenizer().insert(ns, "text-properties") == NS_style::LN_text_properties;
        flag&=QName::tokenizer().insert(ns, "use-window-font-color") == NS_style::LN_use_window_font_color;
        flag&=QName::tokenizer().insert(ns, "text-outline") == NS_style::LN_text_outline;
        flag&=QName::tokenizer().insert(ns, "text-line-through-type") == NS_style::LN_text_line_through_type;
        flag&=QName::tokenizer().insert(ns, "background-image") == NS_style::LN_background_image;
        flag&=QName::tokenizer().insert(ns, "text-line-through-style") == NS_style::LN_text_line_through_style;
        flag&=QName::tokenizer().insert(ns, "repeat") == NS_style::LN_repeat;
        flag&=QName::tokenizer().insert(ns, "filter-name") == NS_style::LN_filter_name;
        flag&=QName::tokenizer().insert(ns, "text-line-through-width") == NS_style::LN_text_line_through_width;
        flag&=QName::tokenizer().insert(ns, "text-line-through-color") == NS_style::LN_text_line_through_color;
        flag&=QName::tokenizer().insert(ns, "text-line-through-text") == NS_style::LN_text_line_through_text;
        flag&=QName::tokenizer().insert(ns, "text-line-through-text-style") == NS_style::LN_text_line_through_text_style;
        flag&=QName::tokenizer().insert(ns, "border-line-width") == NS_style::LN_border_line_width;
        flag&=QName::tokenizer().insert(ns, "text-position") == NS_style::LN_text_position;
        flag&=QName::tokenizer().insert(ns, "border-line-width-top") == NS_style::LN_border_line_width_top;
        flag&=QName::tokenizer().insert(ns, "font-name") == NS_style::LN_font_name;
        flag&=QName::tokenizer().insert(ns, "border-line-width-bottom") == NS_style::LN_border_line_width_bottom;
        flag&=QName::tokenizer().insert(ns, "font-name-asian") == NS_style::LN_font_name_asian;
        flag&=QName::tokenizer().insert(ns, "font-name-complex") == NS_style::LN_font_name_complex;
        flag&=QName::tokenizer().insert(ns, "border-line-width-left") == NS_style::LN_border_line_width_left;
        flag&=QName::tokenizer().insert(ns, "font-family-asian") == NS_style::LN_font_family_asian;
        flag&=QName::tokenizer().insert(ns, "border-line-width-right") == NS_style::LN_border_line_width_right;
        flag&=QName::tokenizer().insert(ns, "font-family-complex") == NS_style::LN_font_family_complex;
        flag&=QName::tokenizer().insert(ns, "data-style-name") == NS_style::LN_data_style_name;
        flag&=QName::tokenizer().insert(ns, "font-family-generic-asian") == NS_style::LN_font_family_generic_asian;
        flag&=QName::tokenizer().insert(ns, "font-family-generic-complex") == NS_style::LN_font_family_generic_complex;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_style::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "font-style-name") == NS_style::LN_font_style_name;
        flag&=QName::tokenizer().insert(ns, "font-style-name-asian") == NS_style::LN_font_style_name_asian;
        flag&=QName::tokenizer().insert(ns, "font-style-name-complex") == NS_style::LN_font_style_name_complex;
        flag&=QName::tokenizer().insert(ns, "text-autospace") == NS_style::LN_text_autospace;
        flag&=QName::tokenizer().insert(ns, "font-pitch-asian") == NS_style::LN_font_pitch_asian;
        flag&=QName::tokenizer().insert(ns, "punctuation-wrap") == NS_style::LN_punctuation_wrap;
        flag&=QName::tokenizer().insert(ns, "line-break") == NS_style::LN_line_break;
        flag&=QName::tokenizer().insert(ns, "font-pitch-complex") == NS_style::LN_font_pitch_complex;
        flag&=QName::tokenizer().insert(ns, "vertical-align") == NS_style::LN_vertical_align;
        flag&=QName::tokenizer().insert(ns, "writing-mode") == NS_style::LN_writing_mode;
        flag&=QName::tokenizer().insert(ns, "writing-mode-automatic") == NS_style::LN_writing_mode_automatic;
        flag&=QName::tokenizer().insert(ns, "snap-to-layout-grid") == NS_style::LN_snap_to_layout_grid;
        flag&=QName::tokenizer().insert(ns, "page-number") == NS_style::LN_page_number;
        flag&=QName::tokenizer().insert(ns, "background-transparency") == NS_style::LN_background_transparency;
        flag&=QName::tokenizer().insert(ns, "ruby-properties") == NS_style::LN_ruby_properties;
        flag&=QName::tokenizer().insert(ns, "ruby-position") == NS_style::LN_ruby_position;
        flag&=QName::tokenizer().insert(ns, "ruby-align") == NS_style::LN_ruby_align;
        flag&=QName::tokenizer().insert(ns, "section-properties") == NS_style::LN_section_properties;
        flag&=QName::tokenizer().insert(ns, "columns") == NS_style::LN_columns;
        flag&=QName::tokenizer().insert(ns, "column") == NS_style::LN_column;
        flag&=QName::tokenizer().insert(ns, "column-sep") == NS_style::LN_column_sep;
        flag&=QName::tokenizer().insert(ns, "height") == NS_style::LN_height;
        flag&=QName::tokenizer().insert(ns, "protect") == NS_style::LN_protect;
        flag&=QName::tokenizer().insert(ns, "table-properties") == NS_style::LN_table_properties;
        flag&=QName::tokenizer().insert(ns, "may-break-between-rows") == NS_style::LN_may_break_between_rows;
        flag&=QName::tokenizer().insert(ns, "table-column-properties") == NS_style::LN_table_column_properties;
        flag&=QName::tokenizer().insert(ns, "column-width") == NS_style::LN_column_width;
        flag&=QName::tokenizer().insert(ns, "rel-column-width") == NS_style::LN_rel_column_width;
        flag&=QName::tokenizer().insert(ns, "use-optimal-column-width") == NS_style::LN_use_optimal_column_width;
        flag&=QName::tokenizer().insert(ns, "table-row-properties") == NS_style::LN_table_row_properties;
        flag&=QName::tokenizer().insert(ns, "row-height") == NS_style::LN_row_height;
        flag&=QName::tokenizer().insert(ns, "min-row-height") == NS_style::LN_min_row_height;
        flag&=QName::tokenizer().insert(ns, "use-optimal-row-height") == NS_style::LN_use_optimal_row_height;
        flag&=QName::tokenizer().insert(ns, "table-cell-properties") == NS_style::LN_table_cell_properties;
        flag&=QName::tokenizer().insert(ns, "text-align-source") == NS_style::LN_text_align_source;
        flag&=QName::tokenizer().insert(ns, "direction") == NS_style::LN_direction;
        flag&=QName::tokenizer().insert(ns, "glyph-orientation-vertical") == NS_style::LN_glyph_orientation_vertical;
        flag&=QName::tokenizer().insert(ns, "diagonal-tl-br") == NS_style::LN_diagonal_tl_br;
        flag&=QName::tokenizer().insert(ns, "diagonal-tl-br-widths") == NS_style::LN_diagonal_tl_br_widths;
        flag&=QName::tokenizer().insert(ns, "diagonal-bl-tr") == NS_style::LN_diagonal_bl_tr;
        flag&=QName::tokenizer().insert(ns, "diagonal-bl-tr-widths") == NS_style::LN_diagonal_bl_tr_widths;
        flag&=QName::tokenizer().insert(ns, "rotation-angle") == NS_style::LN_rotation_angle;
        flag&=QName::tokenizer().insert(ns, "rotation-align") == NS_style::LN_rotation_align;
        flag&=QName::tokenizer().insert(ns, "cell-protect") == NS_style::LN_cell_protect;
        flag&=QName::tokenizer().insert(ns, "print-content") == NS_style::LN_print_content;
        flag&=QName::tokenizer().insert(ns, "decimal-places") == NS_style::LN_decimal_places;
        flag&=QName::tokenizer().insert(ns, "repeat-content") == NS_style::LN_repeat_content;
        flag&=QName::tokenizer().insert(ns, "shrink-to-fit") == NS_style::LN_shrink_to_fit;
        flag&=QName::tokenizer().insert(ns, "list-level-properties") == NS_style::LN_list_level_properties;
        flag&=QName::tokenizer().insert(ns, "handout-master") == NS_style::LN_handout_master;
        flag&=QName::tokenizer().insert(ns, "page-layout-name") == NS_style::LN_page_layout_name;
        flag&=QName::tokenizer().insert(ns, "legend-expansion") == NS_style::LN_legend_expansion;
        flag&=QName::tokenizer().insert(ns, "legend-expansion-aspect-ratio") == NS_style::LN_legend_expansion_aspect_ratio;
        flag&=QName::tokenizer().insert(ns, "rel-width") == NS_style::LN_rel_width;
        flag&=QName::tokenizer().insert(ns, "rel-height") == NS_style::LN_rel_height;
        flag&=QName::tokenizer().insert(ns, "horizontal-pos") == NS_style::LN_horizontal_pos;
        flag&=QName::tokenizer().insert(ns, "horizontal-rel") == NS_style::LN_horizontal_rel;
        flag&=QName::tokenizer().insert(ns, "vertical-pos") == NS_style::LN_vertical_pos;
        flag&=QName::tokenizer().insert(ns, "vertical-rel") == NS_style::LN_vertical_rel;
        flag&=QName::tokenizer().insert(ns, "editable") == NS_style::LN_editable;
        flag&=QName::tokenizer().insert(ns, "wrap") == NS_style::LN_wrap;
        flag&=QName::tokenizer().insert(ns, "wrap-dynamic-treshold") == NS_style::LN_wrap_dynamic_treshold;
        flag&=QName::tokenizer().insert(ns, "number-wrapped-paragraphs") == NS_style::LN_number_wrapped_paragraphs;
        flag&=QName::tokenizer().insert(ns, "wrap-contour") == NS_style::LN_wrap_contour;
        flag&=QName::tokenizer().insert(ns, "wrap-contour-mode") == NS_style::LN_wrap_contour_mode;
        flag&=QName::tokenizer().insert(ns, "run-through") == NS_style::LN_run_through;
        flag&=QName::tokenizer().insert(ns, "flow-with-text") == NS_style::LN_flow_with_text;
        flag&=QName::tokenizer().insert(ns, "overflow-behavior") == NS_style::LN_overflow_behavior;
        flag&=QName::tokenizer().insert(ns, "mirror") == NS_style::LN_mirror;
        CPPUNIT_ASSERT_MESSAGE("test for namespace style failed", flag);}
    void test_fo(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0";
        flag&=QName::tokenizer().insert(ns, "clip") == NS_fo::LN_clip;
        flag&=QName::tokenizer().insert(ns, "language") == NS_fo::LN_language;
        flag&=QName::tokenizer().insert(ns, "country") == NS_fo::LN_country;
        flag&=QName::tokenizer().insert(ns, "font-size") == NS_fo::LN_font_size;
        flag&=QName::tokenizer().insert(ns, "letter-spacing") == NS_fo::LN_letter_spacing;
        flag&=QName::tokenizer().insert(ns, "font-style") == NS_fo::LN_font_style;
        flag&=QName::tokenizer().insert(ns, "text-shadow") == NS_fo::LN_text_shadow;
        flag&=QName::tokenizer().insert(ns, "font-weight") == NS_fo::LN_font_weight;
        flag&=QName::tokenizer().insert(ns, "hyphenate") == NS_fo::LN_hyphenate;
        flag&=QName::tokenizer().insert(ns, "hyphenation-remain-char-count") == NS_fo::LN_hyphenation_remain_char_count;
        flag&=QName::tokenizer().insert(ns, "page-width") == NS_fo::LN_page_width;
        flag&=QName::tokenizer().insert(ns, "hyphenation-push-char-count") == NS_fo::LN_hyphenation_push_char_count;
        flag&=QName::tokenizer().insert(ns, "page-height") == NS_fo::LN_page_height;
        flag&=QName::tokenizer().insert(ns, "line-height") == NS_fo::LN_line_height;
        flag&=QName::tokenizer().insert(ns, "text-align") == NS_fo::LN_text_align;
        flag&=QName::tokenizer().insert(ns, "text-align-last") == NS_fo::LN_text_align_last;
        flag&=QName::tokenizer().insert(ns, "keep-together") == NS_fo::LN_keep_together;
        flag&=QName::tokenizer().insert(ns, "widows") == NS_fo::LN_widows;
        flag&=QName::tokenizer().insert(ns, "orphans") == NS_fo::LN_orphans;
        flag&=QName::tokenizer().insert(ns, "hyphenation-keep") == NS_fo::LN_hyphenation_keep;
        flag&=QName::tokenizer().insert(ns, "hyphenation-ladder-count") == NS_fo::LN_hyphenation_ladder_count;
        flag&=QName::tokenizer().insert(ns, "margin-left") == NS_fo::LN_margin_left;
        flag&=QName::tokenizer().insert(ns, "margin-right") == NS_fo::LN_margin_right;
        flag&=QName::tokenizer().insert(ns, "text-indent") == NS_fo::LN_text_indent;
        flag&=QName::tokenizer().insert(ns, "font-variant") == NS_fo::LN_font_variant;
        flag&=QName::tokenizer().insert(ns, "margin-top") == NS_fo::LN_margin_top;
        flag&=QName::tokenizer().insert(ns, "text-transform") == NS_fo::LN_text_transform;
        flag&=QName::tokenizer().insert(ns, "color") == NS_fo::LN_color;
        flag&=QName::tokenizer().insert(ns, "margin-bottom") == NS_fo::LN_margin_bottom;
        flag&=QName::tokenizer().insert(ns, "margin") == NS_fo::LN_margin;
        flag&=QName::tokenizer().insert(ns, "break-before") == NS_fo::LN_break_before;
        flag&=QName::tokenizer().insert(ns, "break-after") == NS_fo::LN_break_after;
        flag&=QName::tokenizer().insert(ns, "background-color") == NS_fo::LN_background_color;
        flag&=QName::tokenizer().insert(ns, "border") == NS_fo::LN_border;
        flag&=QName::tokenizer().insert(ns, "border-top") == NS_fo::LN_border_top;
        flag&=QName::tokenizer().insert(ns, "border-bottom") == NS_fo::LN_border_bottom;
        flag&=QName::tokenizer().insert(ns, "border-left") == NS_fo::LN_border_left;
        flag&=QName::tokenizer().insert(ns, "border-right") == NS_fo::LN_border_right;
        flag&=QName::tokenizer().insert(ns, "font-family") == NS_fo::LN_font_family;
        flag&=QName::tokenizer().insert(ns, "padding") == NS_fo::LN_padding;
        flag&=QName::tokenizer().insert(ns, "padding-top") == NS_fo::LN_padding_top;
        flag&=QName::tokenizer().insert(ns, "padding-bottom") == NS_fo::LN_padding_bottom;
        flag&=QName::tokenizer().insert(ns, "padding-left") == NS_fo::LN_padding_left;
        flag&=QName::tokenizer().insert(ns, "padding-right") == NS_fo::LN_padding_right;
        flag&=QName::tokenizer().insert(ns, "keep-with-next") == NS_fo::LN_keep_with_next;
        flag&=QName::tokenizer().insert(ns, "column-count") == NS_fo::LN_column_count;
        flag&=QName::tokenizer().insert(ns, "column-gap") == NS_fo::LN_column_gap;
        flag&=QName::tokenizer().insert(ns, "start-indent") == NS_fo::LN_start_indent;
        flag&=QName::tokenizer().insert(ns, "end-indent") == NS_fo::LN_end_indent;
        flag&=QName::tokenizer().insert(ns, "space-before") == NS_fo::LN_space_before;
        flag&=QName::tokenizer().insert(ns, "space-after") == NS_fo::LN_space_after;
        flag&=QName::tokenizer().insert(ns, "wrap-option") == NS_fo::LN_wrap_option;
        flag&=QName::tokenizer().insert(ns, "width") == NS_fo::LN_width;
        flag&=QName::tokenizer().insert(ns, "height") == NS_fo::LN_height;
        flag&=QName::tokenizer().insert(ns, "min-height") == NS_fo::LN_min_height;
        flag&=QName::tokenizer().insert(ns, "min-width") == NS_fo::LN_min_width;
        flag&=QName::tokenizer().insert(ns, "max-height") == NS_fo::LN_max_height;
        flag&=QName::tokenizer().insert(ns, "max-width") == NS_fo::LN_max_width;
        CPPUNIT_ASSERT_MESSAGE("test for namespace fo failed", flag);}
    void test_xforms(){
        bool flag=true;
        static const char ns[] = "http://www.w3.org/2002/xforms";
        flag&=QName::tokenizer().insert(ns, "model") == NS_xforms::LN_model;
        flag&=QName::tokenizer().insert(ns, "bind") == NS_xforms::LN_bind;
        CPPUNIT_ASSERT_MESSAGE("test for namespace xforms failed", flag);}
    void test_draw(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:drawing:1.0";
        flag&=QName::tokenizer().insert(ns, "wrap-influence-on-position") == NS_draw::LN_wrap_influence_on_position;
        flag&=QName::tokenizer().insert(ns, "frame-display-scrollbar") == NS_draw::LN_frame_display_scrollbar;
        flag&=QName::tokenizer().insert(ns, "frame-display-border") == NS_draw::LN_frame_display_border;
        flag&=QName::tokenizer().insert(ns, "frame-margin-horizontal") == NS_draw::LN_frame_margin_horizontal;
        flag&=QName::tokenizer().insert(ns, "frame-margin-vertical") == NS_draw::LN_frame_margin_vertical;
        flag&=QName::tokenizer().insert(ns, "visible-area-left") == NS_draw::LN_visible_area_left;
        flag&=QName::tokenizer().insert(ns, "visible-area-top") == NS_draw::LN_visible_area_top;
        flag&=QName::tokenizer().insert(ns, "visible-area-width") == NS_draw::LN_visible_area_width;
        flag&=QName::tokenizer().insert(ns, "visible-area-height") == NS_draw::LN_visible_area_height;
        flag&=QName::tokenizer().insert(ns, "ole-draw-aspect") == NS_draw::LN_ole_draw_aspect;
        flag&=QName::tokenizer().insert(ns, "custom-shape") == NS_draw::LN_custom_shape;
        flag&=QName::tokenizer().insert(ns, "engine") == NS_draw::LN_engine;
        flag&=QName::tokenizer().insert(ns, "data") == NS_draw::LN_data;
        flag&=QName::tokenizer().insert(ns, "enhanced-geometry") == NS_draw::LN_enhanced_geometry;
        flag&=QName::tokenizer().insert(ns, "mirror-vertical") == NS_draw::LN_mirror_vertical;
        flag&=QName::tokenizer().insert(ns, "mirror-horizontal") == NS_draw::LN_mirror_horizontal;
        flag&=QName::tokenizer().insert(ns, "text-rotate-angle") == NS_draw::LN_text_rotate_angle;
        flag&=QName::tokenizer().insert(ns, "extrusion-allowed") == NS_draw::LN_extrusion_allowed;
        flag&=QName::tokenizer().insert(ns, "text-path-allowed") == NS_draw::LN_text_path_allowed;
        flag&=QName::tokenizer().insert(ns, "concentric-gradient-fill-allowed") == NS_draw::LN_concentric_gradient_fill_allowed;
        flag&=QName::tokenizer().insert(ns, "extrusion") == NS_draw::LN_extrusion;
        flag&=QName::tokenizer().insert(ns, "extrusion-brightness") == NS_draw::LN_extrusion_brightness;
        flag&=QName::tokenizer().insert(ns, "extrusion-depth") == NS_draw::LN_extrusion_depth;
        flag&=QName::tokenizer().insert(ns, "extrusion-diffusion") == NS_draw::LN_extrusion_diffusion;
        flag&=QName::tokenizer().insert(ns, "extrusion-number-of-line-segments") == NS_draw::LN_extrusion_number_of_line_segments;
        flag&=QName::tokenizer().insert(ns, "extrusion-light-face") == NS_draw::LN_extrusion_light_face;
        flag&=QName::tokenizer().insert(ns, "extrusion-first-light-harsh") == NS_draw::LN_extrusion_first_light_harsh;
        flag&=QName::tokenizer().insert(ns, "extrusion-second-light-harsh") == NS_draw::LN_extrusion_second_light_harsh;
        flag&=QName::tokenizer().insert(ns, "extrusion-first-light-level") == NS_draw::LN_extrusion_first_light_level;
        flag&=QName::tokenizer().insert(ns, "extrusion-second-light-level") == NS_draw::LN_extrusion_second_light_level;
        flag&=QName::tokenizer().insert(ns, "extrusion-first-light-direction") == NS_draw::LN_extrusion_first_light_direction;
        flag&=QName::tokenizer().insert(ns, "extrusion-second-light-direction") == NS_draw::LN_extrusion_second_light_direction;
        flag&=QName::tokenizer().insert(ns, "extrusion-metal") == NS_draw::LN_extrusion_metal;
        flag&=QName::tokenizer().insert(ns, "extrusion-rotation-angle") == NS_draw::LN_extrusion_rotation_angle;
        flag&=QName::tokenizer().insert(ns, "extrusion-rotation-center") == NS_draw::LN_extrusion_rotation_center;
        flag&=QName::tokenizer().insert(ns, "extrusion-shininess") == NS_draw::LN_extrusion_shininess;
        flag&=QName::tokenizer().insert(ns, "extrusion-skew") == NS_draw::LN_extrusion_skew;
        flag&=QName::tokenizer().insert(ns, "extrusion-specularity") == NS_draw::LN_extrusion_specularity;
        flag&=QName::tokenizer().insert(ns, "extrusion-viewpoint") == NS_draw::LN_extrusion_viewpoint;
        flag&=QName::tokenizer().insert(ns, "extrusion-origin") == NS_draw::LN_extrusion_origin;
        flag&=QName::tokenizer().insert(ns, "extrusion-color") == NS_draw::LN_extrusion_color;
        flag&=QName::tokenizer().insert(ns, "enhanced-path") == NS_draw::LN_enhanced_path;
        flag&=QName::tokenizer().insert(ns, "path-stretchpoint-x") == NS_draw::LN_path_stretchpoint_x;
        flag&=QName::tokenizer().insert(ns, "path-stretchpoint-y") == NS_draw::LN_path_stretchpoint_y;
        flag&=QName::tokenizer().insert(ns, "text-areas") == NS_draw::LN_text_areas;
        flag&=QName::tokenizer().insert(ns, "glue-points") == NS_draw::LN_glue_points;
        flag&=QName::tokenizer().insert(ns, "glue-point-type") == NS_draw::LN_glue_point_type;
        flag&=QName::tokenizer().insert(ns, "glue-point-leaving-directions") == NS_draw::LN_glue_point_leaving_directions;
        flag&=QName::tokenizer().insert(ns, "text-path") == NS_draw::LN_text_path;
        flag&=QName::tokenizer().insert(ns, "text-path-mode") == NS_draw::LN_text_path_mode;
        flag&=QName::tokenizer().insert(ns, "text-path-scale") == NS_draw::LN_text_path_scale;
        flag&=QName::tokenizer().insert(ns, "text-path-same-letter-heights") == NS_draw::LN_text_path_same_letter_heights;
        flag&=QName::tokenizer().insert(ns, "modifiers") == NS_draw::LN_modifiers;
        flag&=QName::tokenizer().insert(ns, "background-size") == NS_draw::LN_background_size;
        flag&=QName::tokenizer().insert(ns, "equation") == NS_draw::LN_equation;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_draw::LN_formula;
        flag&=QName::tokenizer().insert(ns, "handle") == NS_draw::LN_handle;
        flag&=QName::tokenizer().insert(ns, "handle-mirror-vertical") == NS_draw::LN_handle_mirror_vertical;
        flag&=QName::tokenizer().insert(ns, "handle-mirror-horizontal") == NS_draw::LN_handle_mirror_horizontal;
        flag&=QName::tokenizer().insert(ns, "handle-switched") == NS_draw::LN_handle_switched;
        flag&=QName::tokenizer().insert(ns, "handle-position") == NS_draw::LN_handle_position;
        flag&=QName::tokenizer().insert(ns, "handle-range-x-minimum") == NS_draw::LN_handle_range_x_minimum;
        flag&=QName::tokenizer().insert(ns, "handle-range-x-maximum") == NS_draw::LN_handle_range_x_maximum;
        flag&=QName::tokenizer().insert(ns, "handle-range-y-minimum") == NS_draw::LN_handle_range_y_minimum;
        flag&=QName::tokenizer().insert(ns, "handle-range-y-maximum") == NS_draw::LN_handle_range_y_maximum;
        flag&=QName::tokenizer().insert(ns, "handle-polar") == NS_draw::LN_handle_polar;
        flag&=QName::tokenizer().insert(ns, "handle-radius-range-minimum") == NS_draw::LN_handle_radius_range_minimum;
        flag&=QName::tokenizer().insert(ns, "handle-radius-range-maximum") == NS_draw::LN_handle_radius_range_maximum;
        flag&=QName::tokenizer().insert(ns, "gradient") == NS_draw::LN_gradient;
        flag&=QName::tokenizer().insert(ns, "display-name") == NS_draw::LN_display_name;
        flag&=QName::tokenizer().insert(ns, "style") == NS_draw::LN_style;
        flag&=QName::tokenizer().insert(ns, "cx") == NS_draw::LN_cx;
        flag&=QName::tokenizer().insert(ns, "cy") == NS_draw::LN_cy;
        flag&=QName::tokenizer().insert(ns, "start-color") == NS_draw::LN_start_color;
        flag&=QName::tokenizer().insert(ns, "end-color") == NS_draw::LN_end_color;
        flag&=QName::tokenizer().insert(ns, "start-intensity") == NS_draw::LN_start_intensity;
        flag&=QName::tokenizer().insert(ns, "end-intensity") == NS_draw::LN_end_intensity;
        flag&=QName::tokenizer().insert(ns, "angle") == NS_draw::LN_angle;
        flag&=QName::tokenizer().insert(ns, "border") == NS_draw::LN_border;
        flag&=QName::tokenizer().insert(ns, "hatch") == NS_draw::LN_hatch;
        flag&=QName::tokenizer().insert(ns, "distance") == NS_draw::LN_distance;
        flag&=QName::tokenizer().insert(ns, "rotation") == NS_draw::LN_rotation;
        flag&=QName::tokenizer().insert(ns, "fill-image") == NS_draw::LN_fill_image;
        flag&=QName::tokenizer().insert(ns, "opacity") == NS_draw::LN_opacity;
        flag&=QName::tokenizer().insert(ns, "start") == NS_draw::LN_start;
        flag&=QName::tokenizer().insert(ns, "end") == NS_draw::LN_end;
        flag&=QName::tokenizer().insert(ns, "marker") == NS_draw::LN_marker;
        flag&=QName::tokenizer().insert(ns, "stroke-dash") == NS_draw::LN_stroke_dash;
        flag&=QName::tokenizer().insert(ns, "dots1") == NS_draw::LN_dots1;
        flag&=QName::tokenizer().insert(ns, "dots1-length") == NS_draw::LN_dots1_length;
        flag&=QName::tokenizer().insert(ns, "dots2") == NS_draw::LN_dots2;
        flag&=QName::tokenizer().insert(ns, "dots2-length") == NS_draw::LN_dots2_length;
        flag&=QName::tokenizer().insert(ns, "stroke") == NS_draw::LN_stroke;
        flag&=QName::tokenizer().insert(ns, "stroke-dash-names") == NS_draw::LN_stroke_dash_names;
        flag&=QName::tokenizer().insert(ns, "marker-start") == NS_draw::LN_marker_start;
        flag&=QName::tokenizer().insert(ns, "marker-end") == NS_draw::LN_marker_end;
        flag&=QName::tokenizer().insert(ns, "marker-start-width") == NS_draw::LN_marker_start_width;
        flag&=QName::tokenizer().insert(ns, "marker-end-width") == NS_draw::LN_marker_end_width;
        flag&=QName::tokenizer().insert(ns, "marker-start-center") == NS_draw::LN_marker_start_center;
        flag&=QName::tokenizer().insert(ns, "marker-end-center") == NS_draw::LN_marker_end_center;
        flag&=QName::tokenizer().insert(ns, "stroke-linejoin") == NS_draw::LN_stroke_linejoin;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_draw::LN_fill;
        flag&=QName::tokenizer().insert(ns, "fill-color") == NS_draw::LN_fill_color;
        flag&=QName::tokenizer().insert(ns, "secondary-fill-color") == NS_draw::LN_secondary_fill_color;
        flag&=QName::tokenizer().insert(ns, "fill-gradient-name") == NS_draw::LN_fill_gradient_name;
        flag&=QName::tokenizer().insert(ns, "gradient-step-count") == NS_draw::LN_gradient_step_count;
        flag&=QName::tokenizer().insert(ns, "fill-hatch-name") == NS_draw::LN_fill_hatch_name;
        flag&=QName::tokenizer().insert(ns, "fill-hatch-solid") == NS_draw::LN_fill_hatch_solid;
        flag&=QName::tokenizer().insert(ns, "fill-image-name") == NS_draw::LN_fill_image_name;
        flag&=QName::tokenizer().insert(ns, "fill-image-width") == NS_draw::LN_fill_image_width;
        flag&=QName::tokenizer().insert(ns, "fill-image-height") == NS_draw::LN_fill_image_height;
        flag&=QName::tokenizer().insert(ns, "fill-image-ref-point-x") == NS_draw::LN_fill_image_ref_point_x;
        flag&=QName::tokenizer().insert(ns, "fill-image-ref-point-y") == NS_draw::LN_fill_image_ref_point_y;
        flag&=QName::tokenizer().insert(ns, "fill-image-ref-point") == NS_draw::LN_fill_image_ref_point;
        flag&=QName::tokenizer().insert(ns, "tile-repeat-offset") == NS_draw::LN_tile_repeat_offset;
        flag&=QName::tokenizer().insert(ns, "opacity-name") == NS_draw::LN_opacity_name;
        flag&=QName::tokenizer().insert(ns, "symbol-color") == NS_draw::LN_symbol_color;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_draw::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "layer-set") == NS_draw::LN_layer_set;
        flag&=QName::tokenizer().insert(ns, "layer") == NS_draw::LN_layer;
        flag&=QName::tokenizer().insert(ns, "name") == NS_draw::LN_name;
        flag&=QName::tokenizer().insert(ns, "protected") == NS_draw::LN_protected;
        flag&=QName::tokenizer().insert(ns, "display") == NS_draw::LN_display;
        flag&=QName::tokenizer().insert(ns, "page") == NS_draw::LN_page;
        flag&=QName::tokenizer().insert(ns, "master-page-name") == NS_draw::LN_master_page_name;
        flag&=QName::tokenizer().insert(ns, "shape-id") == NS_draw::LN_shape_id;
        flag&=QName::tokenizer().insert(ns, "auto-grow-width") == NS_draw::LN_auto_grow_width;
        flag&=QName::tokenizer().insert(ns, "auto-grow-height") == NS_draw::LN_auto_grow_height;
        flag&=QName::tokenizer().insert(ns, "id") == NS_draw::LN_id;
        flag&=QName::tokenizer().insert(ns, "fit-to-size") == NS_draw::LN_fit_to_size;
        flag&=QName::tokenizer().insert(ns, "rect") == NS_draw::LN_rect;
        flag&=QName::tokenizer().insert(ns, "fit-to-contour") == NS_draw::LN_fit_to_contour;
        flag&=QName::tokenizer().insert(ns, "corner-radius") == NS_draw::LN_corner_radius;
        flag&=QName::tokenizer().insert(ns, "line") == NS_draw::LN_line;
        flag&=QName::tokenizer().insert(ns, "textarea-vertical-align") == NS_draw::LN_textarea_vertical_align;
        flag&=QName::tokenizer().insert(ns, "color") == NS_draw::LN_color;
        flag&=QName::tokenizer().insert(ns, "textarea-horizontal-align") == NS_draw::LN_textarea_horizontal_align;
        flag&=QName::tokenizer().insert(ns, "color-mode") == NS_draw::LN_color_mode;
        flag&=QName::tokenizer().insert(ns, "polyline") == NS_draw::LN_polyline;
        flag&=QName::tokenizer().insert(ns, "points") == NS_draw::LN_points;
        flag&=QName::tokenizer().insert(ns, "color-inversion") == NS_draw::LN_color_inversion;
        flag&=QName::tokenizer().insert(ns, "polygon") == NS_draw::LN_polygon;
        flag&=QName::tokenizer().insert(ns, "luminance") == NS_draw::LN_luminance;
        flag&=QName::tokenizer().insert(ns, "regular-polygon") == NS_draw::LN_regular_polygon;
        flag&=QName::tokenizer().insert(ns, "contrast") == NS_draw::LN_contrast;
        flag&=QName::tokenizer().insert(ns, "concave") == NS_draw::LN_concave;
        flag&=QName::tokenizer().insert(ns, "gamma") == NS_draw::LN_gamma;
        flag&=QName::tokenizer().insert(ns, "red") == NS_draw::LN_red;
        flag&=QName::tokenizer().insert(ns, "corners") == NS_draw::LN_corners;
        flag&=QName::tokenizer().insert(ns, "green") == NS_draw::LN_green;
        flag&=QName::tokenizer().insert(ns, "sharpness") == NS_draw::LN_sharpness;
        flag&=QName::tokenizer().insert(ns, "blue") == NS_draw::LN_blue;
        flag&=QName::tokenizer().insert(ns, "path") == NS_draw::LN_path;
        flag&=QName::tokenizer().insert(ns, "image-opacity") == NS_draw::LN_image_opacity;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_draw::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "circle") == NS_draw::LN_circle;
        flag&=QName::tokenizer().insert(ns, "shadow-offset-x") == NS_draw::LN_shadow_offset_x;
        flag&=QName::tokenizer().insert(ns, "kind") == NS_draw::LN_kind;
        flag&=QName::tokenizer().insert(ns, "shadow-offset-y") == NS_draw::LN_shadow_offset_y;
        flag&=QName::tokenizer().insert(ns, "start-angle") == NS_draw::LN_start_angle;
        flag&=QName::tokenizer().insert(ns, "shadow-color") == NS_draw::LN_shadow_color;
        flag&=QName::tokenizer().insert(ns, "end-angle") == NS_draw::LN_end_angle;
        flag&=QName::tokenizer().insert(ns, "shadow-opacity") == NS_draw::LN_shadow_opacity;
        flag&=QName::tokenizer().insert(ns, "ellipse") == NS_draw::LN_ellipse;
        flag&=QName::tokenizer().insert(ns, "start-line-spacing-horizontal") == NS_draw::LN_start_line_spacing_horizontal;
        flag&=QName::tokenizer().insert(ns, "connector") == NS_draw::LN_connector;
        flag&=QName::tokenizer().insert(ns, "type") == NS_draw::LN_type;
        flag&=QName::tokenizer().insert(ns, "start-shape") == NS_draw::LN_start_shape;
        flag&=QName::tokenizer().insert(ns, "start-line-spacing-vertical") == NS_draw::LN_start_line_spacing_vertical;
        flag&=QName::tokenizer().insert(ns, "start-glue-point") == NS_draw::LN_start_glue_point;
        flag&=QName::tokenizer().insert(ns, "end-line-spacing-horizontal") == NS_draw::LN_end_line_spacing_horizontal;
        flag&=QName::tokenizer().insert(ns, "end-shape") == NS_draw::LN_end_shape;
        flag&=QName::tokenizer().insert(ns, "end-glue-point") == NS_draw::LN_end_glue_point;
        flag&=QName::tokenizer().insert(ns, "end-line-spacing-vertical") == NS_draw::LN_end_line_spacing_vertical;
        flag&=QName::tokenizer().insert(ns, "line-skew") == NS_draw::LN_line_skew;
        flag&=QName::tokenizer().insert(ns, "line-distance") == NS_draw::LN_line_distance;
        flag&=QName::tokenizer().insert(ns, "caption") == NS_draw::LN_caption;
        flag&=QName::tokenizer().insert(ns, "guide-overhang") == NS_draw::LN_guide_overhang;
        flag&=QName::tokenizer().insert(ns, "caption-point-x") == NS_draw::LN_caption_point_x;
        flag&=QName::tokenizer().insert(ns, "guide-distance") == NS_draw::LN_guide_distance;
        flag&=QName::tokenizer().insert(ns, "caption-point-y") == NS_draw::LN_caption_point_y;
        flag&=QName::tokenizer().insert(ns, "measure") == NS_draw::LN_measure;
        flag&=QName::tokenizer().insert(ns, "start-guide") == NS_draw::LN_start_guide;
        flag&=QName::tokenizer().insert(ns, "control") == NS_draw::LN_control;
        flag&=QName::tokenizer().insert(ns, "end-guide") == NS_draw::LN_end_guide;
        flag&=QName::tokenizer().insert(ns, "placing") == NS_draw::LN_placing;
        flag&=QName::tokenizer().insert(ns, "page-thumbnail") == NS_draw::LN_page_thumbnail;
        flag&=QName::tokenizer().insert(ns, "page-number") == NS_draw::LN_page_number;
        flag&=QName::tokenizer().insert(ns, "parallel") == NS_draw::LN_parallel;
        flag&=QName::tokenizer().insert(ns, "g") == NS_draw::LN_g;
        flag&=QName::tokenizer().insert(ns, "measure-align") == NS_draw::LN_measure_align;
        flag&=QName::tokenizer().insert(ns, "measure-vertical-align") == NS_draw::LN_measure_vertical_align;
        flag&=QName::tokenizer().insert(ns, "unit") == NS_draw::LN_unit;
        flag&=QName::tokenizer().insert(ns, "transform") == NS_draw::LN_transform;
        flag&=QName::tokenizer().insert(ns, "show-unit") == NS_draw::LN_show_unit;
        flag&=QName::tokenizer().insert(ns, "decimal-places") == NS_draw::LN_decimal_places;
        flag&=QName::tokenizer().insert(ns, "class-names") == NS_draw::LN_class_names;
        flag&=QName::tokenizer().insert(ns, "caption-type") == NS_draw::LN_caption_type;
        flag&=QName::tokenizer().insert(ns, "text-style-name") == NS_draw::LN_text_style_name;
        flag&=QName::tokenizer().insert(ns, "caption-angle-type") == NS_draw::LN_caption_angle_type;
        flag&=QName::tokenizer().insert(ns, "z-index") == NS_draw::LN_z_index;
        flag&=QName::tokenizer().insert(ns, "caption-angle") == NS_draw::LN_caption_angle;
        flag&=QName::tokenizer().insert(ns, "caption-gap") == NS_draw::LN_caption_gap;
        flag&=QName::tokenizer().insert(ns, "caption-escape-direction") == NS_draw::LN_caption_escape_direction;
        flag&=QName::tokenizer().insert(ns, "caption-escape") == NS_draw::LN_caption_escape;
        flag&=QName::tokenizer().insert(ns, "caption-line-length") == NS_draw::LN_caption_line_length;
        flag&=QName::tokenizer().insert(ns, "glue-point") == NS_draw::LN_glue_point;
        flag&=QName::tokenizer().insert(ns, "caption-fit-line-length") == NS_draw::LN_caption_fit_line_length;
        flag&=QName::tokenizer().insert(ns, "align") == NS_draw::LN_align;
        flag&=QName::tokenizer().insert(ns, "escape-direction") == NS_draw::LN_escape_direction;
        flag&=QName::tokenizer().insert(ns, "frame") == NS_draw::LN_frame;
        flag&=QName::tokenizer().insert(ns, "copy-of") == NS_draw::LN_copy_of;
        flag&=QName::tokenizer().insert(ns, "text-box") == NS_draw::LN_text_box;
        flag&=QName::tokenizer().insert(ns, "chain-next-name") == NS_draw::LN_chain_next_name;
        flag&=QName::tokenizer().insert(ns, "image") == NS_draw::LN_image;
        flag&=QName::tokenizer().insert(ns, "filter-name") == NS_draw::LN_filter_name;
        flag&=QName::tokenizer().insert(ns, "object") == NS_draw::LN_object;
        flag&=QName::tokenizer().insert(ns, "object-ole") == NS_draw::LN_object_ole;
        flag&=QName::tokenizer().insert(ns, "notify-on-update-of-ranges") == NS_draw::LN_notify_on_update_of_ranges;
        flag&=QName::tokenizer().insert(ns, "class-id") == NS_draw::LN_class_id;
        flag&=QName::tokenizer().insert(ns, "applet") == NS_draw::LN_applet;
        flag&=QName::tokenizer().insert(ns, "code") == NS_draw::LN_code;
        flag&=QName::tokenizer().insert(ns, "archive") == NS_draw::LN_archive;
        flag&=QName::tokenizer().insert(ns, "may-script") == NS_draw::LN_may_script;
        flag&=QName::tokenizer().insert(ns, "plugin") == NS_draw::LN_plugin;
        flag&=QName::tokenizer().insert(ns, "mime-type") == NS_draw::LN_mime_type;
        flag&=QName::tokenizer().insert(ns, "param") == NS_draw::LN_param;
        flag&=QName::tokenizer().insert(ns, "value") == NS_draw::LN_value;
        flag&=QName::tokenizer().insert(ns, "floating-frame") == NS_draw::LN_floating_frame;
        flag&=QName::tokenizer().insert(ns, "frame-name") == NS_draw::LN_frame_name;
        flag&=QName::tokenizer().insert(ns, "contour-polygon") == NS_draw::LN_contour_polygon;
        flag&=QName::tokenizer().insert(ns, "contour-path") == NS_draw::LN_contour_path;
        flag&=QName::tokenizer().insert(ns, "recreate-on-edit") == NS_draw::LN_recreate_on_edit;
        flag&=QName::tokenizer().insert(ns, "a") == NS_draw::LN_a;
        flag&=QName::tokenizer().insert(ns, "image-map") == NS_draw::LN_image_map;
        flag&=QName::tokenizer().insert(ns, "area-rectangle") == NS_draw::LN_area_rectangle;
        flag&=QName::tokenizer().insert(ns, "area-circle") == NS_draw::LN_area_circle;
        flag&=QName::tokenizer().insert(ns, "area-polygon") == NS_draw::LN_area_polygon;
        flag&=QName::tokenizer().insert(ns, "nohref") == NS_draw::LN_nohref;
        CPPUNIT_ASSERT_MESSAGE("test for namespace draw failed", flag);}
    void test_prj12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/project";
        flag&=QName::tokenizer().insert(ns, "Type") == NS_prj12::LN_Type;
        flag&=QName::tokenizer().insert(ns, "UID") == NS_prj12::LN_UID;
        flag&=QName::tokenizer().insert(ns, "Start") == NS_prj12::LN_Start;
        flag&=QName::tokenizer().insert(ns, "Finish") == NS_prj12::LN_Finish;
        flag&=QName::tokenizer().insert(ns, "Unit") == NS_prj12::LN_Unit;
        flag&=QName::tokenizer().insert(ns, "Value") == NS_prj12::LN_Value;
        flag&=QName::tokenizer().insert(ns, "Project") == NS_prj12::LN_Project;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_prj12::LN_Name;
        flag&=QName::tokenizer().insert(ns, "Title") == NS_prj12::LN_Title;
        flag&=QName::tokenizer().insert(ns, "Subject") == NS_prj12::LN_Subject;
        flag&=QName::tokenizer().insert(ns, "Category") == NS_prj12::LN_Category;
        flag&=QName::tokenizer().insert(ns, "Company") == NS_prj12::LN_Company;
        flag&=QName::tokenizer().insert(ns, "Manager") == NS_prj12::LN_Manager;
        flag&=QName::tokenizer().insert(ns, "Author") == NS_prj12::LN_Author;
        flag&=QName::tokenizer().insert(ns, "CreationDate") == NS_prj12::LN_CreationDate;
        flag&=QName::tokenizer().insert(ns, "Revision") == NS_prj12::LN_Revision;
        flag&=QName::tokenizer().insert(ns, "LastSaved") == NS_prj12::LN_LastSaved;
        flag&=QName::tokenizer().insert(ns, "ScheduleFromStart") == NS_prj12::LN_ScheduleFromStart;
        flag&=QName::tokenizer().insert(ns, "StartDate") == NS_prj12::LN_StartDate;
        flag&=QName::tokenizer().insert(ns, "FinishDate") == NS_prj12::LN_FinishDate;
        flag&=QName::tokenizer().insert(ns, "FYStartDate") == NS_prj12::LN_FYStartDate;
        flag&=QName::tokenizer().insert(ns, "CriticalSlackLimit") == NS_prj12::LN_CriticalSlackLimit;
        flag&=QName::tokenizer().insert(ns, "CurrencyDigits") == NS_prj12::LN_CurrencyDigits;
        flag&=QName::tokenizer().insert(ns, "CurrencySymbol") == NS_prj12::LN_CurrencySymbol;
        flag&=QName::tokenizer().insert(ns, "CurrencySymbolPosition") == NS_prj12::LN_CurrencySymbolPosition;
        flag&=QName::tokenizer().insert(ns, "CalendarUID") == NS_prj12::LN_CalendarUID;
        flag&=QName::tokenizer().insert(ns, "DefaultStartTime") == NS_prj12::LN_DefaultStartTime;
        flag&=QName::tokenizer().insert(ns, "DefaultFinishTime") == NS_prj12::LN_DefaultFinishTime;
        flag&=QName::tokenizer().insert(ns, "MinutesPerDay") == NS_prj12::LN_MinutesPerDay;
        flag&=QName::tokenizer().insert(ns, "MinutesPerWeek") == NS_prj12::LN_MinutesPerWeek;
        flag&=QName::tokenizer().insert(ns, "DaysPerMonth") == NS_prj12::LN_DaysPerMonth;
        flag&=QName::tokenizer().insert(ns, "DefaultTaskType") == NS_prj12::LN_DefaultTaskType;
        flag&=QName::tokenizer().insert(ns, "DefaultFixedCostAccrual") == NS_prj12::LN_DefaultFixedCostAccrual;
        flag&=QName::tokenizer().insert(ns, "DefaultStandardRate") == NS_prj12::LN_DefaultStandardRate;
        flag&=QName::tokenizer().insert(ns, "DefaultOvertimeRate") == NS_prj12::LN_DefaultOvertimeRate;
        flag&=QName::tokenizer().insert(ns, "DurationFormat") == NS_prj12::LN_DurationFormat;
        flag&=QName::tokenizer().insert(ns, "WorkFormat") == NS_prj12::LN_WorkFormat;
        flag&=QName::tokenizer().insert(ns, "EditableActualCosts") == NS_prj12::LN_EditableActualCosts;
        flag&=QName::tokenizer().insert(ns, "HonorConstraints") == NS_prj12::LN_HonorConstraints;
        flag&=QName::tokenizer().insert(ns, "EarnedValueMethod") == NS_prj12::LN_EarnedValueMethod;
        flag&=QName::tokenizer().insert(ns, "InsertedProjectsLikeSummary") == NS_prj12::LN_InsertedProjectsLikeSummary;
        flag&=QName::tokenizer().insert(ns, "MultipleCriticalPaths") == NS_prj12::LN_MultipleCriticalPaths;
        flag&=QName::tokenizer().insert(ns, "NewTasksEffortDriven") == NS_prj12::LN_NewTasksEffortDriven;
        flag&=QName::tokenizer().insert(ns, "NewTasksEstimated") == NS_prj12::LN_NewTasksEstimated;
        flag&=QName::tokenizer().insert(ns, "SplitsInProgressTasks") == NS_prj12::LN_SplitsInProgressTasks;
        flag&=QName::tokenizer().insert(ns, "SpreadActualCost") == NS_prj12::LN_SpreadActualCost;
        flag&=QName::tokenizer().insert(ns, "SpreadPercentComplete") == NS_prj12::LN_SpreadPercentComplete;
        flag&=QName::tokenizer().insert(ns, "TaskUpdatesResource") == NS_prj12::LN_TaskUpdatesResource;
        flag&=QName::tokenizer().insert(ns, "FiscalYearStart") == NS_prj12::LN_FiscalYearStart;
        flag&=QName::tokenizer().insert(ns, "WeekStartDay") == NS_prj12::LN_WeekStartDay;
        flag&=QName::tokenizer().insert(ns, "MoveCompletedEndsBack") == NS_prj12::LN_MoveCompletedEndsBack;
        flag&=QName::tokenizer().insert(ns, "MoveRemainingStartsBack") == NS_prj12::LN_MoveRemainingStartsBack;
        flag&=QName::tokenizer().insert(ns, "MoveRemainingStartsForward") == NS_prj12::LN_MoveRemainingStartsForward;
        flag&=QName::tokenizer().insert(ns, "MoveCompletedEndsForward") == NS_prj12::LN_MoveCompletedEndsForward;
        flag&=QName::tokenizer().insert(ns, "BaselineForEarnedValue") == NS_prj12::LN_BaselineForEarnedValue;
        flag&=QName::tokenizer().insert(ns, "AutoAddNewResourcesAndTasks") == NS_prj12::LN_AutoAddNewResourcesAndTasks;
        flag&=QName::tokenizer().insert(ns, "StatusDate") == NS_prj12::LN_StatusDate;
        flag&=QName::tokenizer().insert(ns, "CurrentDate") == NS_prj12::LN_CurrentDate;
        flag&=QName::tokenizer().insert(ns, "MicrosoftProjectServerURL") == NS_prj12::LN_MicrosoftProjectServerURL;
        flag&=QName::tokenizer().insert(ns, "Autolink") == NS_prj12::LN_Autolink;
        flag&=QName::tokenizer().insert(ns, "NewTaskStartDate") == NS_prj12::LN_NewTaskStartDate;
        flag&=QName::tokenizer().insert(ns, "DefaultTaskEVMethod") == NS_prj12::LN_DefaultTaskEVMethod;
        flag&=QName::tokenizer().insert(ns, "ProjectExternallyEdited") == NS_prj12::LN_ProjectExternallyEdited;
        flag&=QName::tokenizer().insert(ns, "ExtendedCreationDate") == NS_prj12::LN_ExtendedCreationDate;
        flag&=QName::tokenizer().insert(ns, "ActualsInSync") == NS_prj12::LN_ActualsInSync;
        flag&=QName::tokenizer().insert(ns, "RemoveFileProperties") == NS_prj12::LN_RemoveFileProperties;
        flag&=QName::tokenizer().insert(ns, "AdminProject") == NS_prj12::LN_AdminProject;
        flag&=QName::tokenizer().insert(ns, "OutlineCodes") == NS_prj12::LN_OutlineCodes;
        flag&=QName::tokenizer().insert(ns, "OutlineCode") == NS_prj12::LN_OutlineCode;
        flag&=QName::tokenizer().insert(ns, "FieldID") == NS_prj12::LN_FieldID;
        flag&=QName::tokenizer().insert(ns, "FieldName") == NS_prj12::LN_FieldName;
        flag&=QName::tokenizer().insert(ns, "Alias") == NS_prj12::LN_Alias;
        flag&=QName::tokenizer().insert(ns, "PhoneticAlias") == NS_prj12::LN_PhoneticAlias;
        flag&=QName::tokenizer().insert(ns, "Values") == NS_prj12::LN_Values;
        flag&=QName::tokenizer().insert(ns, "ValueID") == NS_prj12::LN_ValueID;
        flag&=QName::tokenizer().insert(ns, "ParentValueID") == NS_prj12::LN_ParentValueID;
        flag&=QName::tokenizer().insert(ns, "Description") == NS_prj12::LN_Description;
        flag&=QName::tokenizer().insert(ns, "Enterprise") == NS_prj12::LN_Enterprise;
        flag&=QName::tokenizer().insert(ns, "EnterpriseOutlineCodeAlias") == NS_prj12::LN_EnterpriseOutlineCodeAlias;
        flag&=QName::tokenizer().insert(ns, "ResourceSubstitutionEnabled") == NS_prj12::LN_ResourceSubstitutionEnabled;
        flag&=QName::tokenizer().insert(ns, "LeafOnly") == NS_prj12::LN_LeafOnly;
        flag&=QName::tokenizer().insert(ns, "AllLevelsRequired") == NS_prj12::LN_AllLevelsRequired;
        flag&=QName::tokenizer().insert(ns, "OnlyTableValuesAllowed") == NS_prj12::LN_OnlyTableValuesAllowed;
        flag&=QName::tokenizer().insert(ns, "Masks") == NS_prj12::LN_Masks;
        flag&=QName::tokenizer().insert(ns, "Mask") == NS_prj12::LN_Mask;
        flag&=QName::tokenizer().insert(ns, "Level") == NS_prj12::LN_Level;
        flag&=QName::tokenizer().insert(ns, "Length") == NS_prj12::LN_Length;
        flag&=QName::tokenizer().insert(ns, "Separator") == NS_prj12::LN_Separator;
        flag&=QName::tokenizer().insert(ns, "WBSMasks") == NS_prj12::LN_WBSMasks;
        flag&=QName::tokenizer().insert(ns, "VerifyUniqueCodes") == NS_prj12::LN_VerifyUniqueCodes;
        flag&=QName::tokenizer().insert(ns, "GenerateCodes") == NS_prj12::LN_GenerateCodes;
        flag&=QName::tokenizer().insert(ns, "Prefix") == NS_prj12::LN_Prefix;
        flag&=QName::tokenizer().insert(ns, "WBSMask") == NS_prj12::LN_WBSMask;
        flag&=QName::tokenizer().insert(ns, "ExtendedAttributes") == NS_prj12::LN_ExtendedAttributes;
        flag&=QName::tokenizer().insert(ns, "ExtendedAttribute") == NS_prj12::LN_ExtendedAttribute;
        flag&=QName::tokenizer().insert(ns, "RollupType") == NS_prj12::LN_RollupType;
        flag&=QName::tokenizer().insert(ns, "CalculationType") == NS_prj12::LN_CalculationType;
        flag&=QName::tokenizer().insert(ns, "Formula") == NS_prj12::LN_Formula;
        flag&=QName::tokenizer().insert(ns, "RestrictValues") == NS_prj12::LN_RestrictValues;
        flag&=QName::tokenizer().insert(ns, "ValuelistSortOrder") == NS_prj12::LN_ValuelistSortOrder;
        flag&=QName::tokenizer().insert(ns, "AppendNewValues") == NS_prj12::LN_AppendNewValues;
        flag&=QName::tokenizer().insert(ns, "Default") == NS_prj12::LN_Default;
        flag&=QName::tokenizer().insert(ns, "ValueList") == NS_prj12::LN_ValueList;
        flag&=QName::tokenizer().insert(ns, "ID") == NS_prj12::LN_ID;
        flag&=QName::tokenizer().insert(ns, "Calendars") == NS_prj12::LN_Calendars;
        flag&=QName::tokenizer().insert(ns, "Calendar") == NS_prj12::LN_Calendar;
        flag&=QName::tokenizer().insert(ns, "IsBaseCalendar") == NS_prj12::LN_IsBaseCalendar;
        flag&=QName::tokenizer().insert(ns, "BaseCalendarUID") == NS_prj12::LN_BaseCalendarUID;
        flag&=QName::tokenizer().insert(ns, "WeekDays") == NS_prj12::LN_WeekDays;
        flag&=QName::tokenizer().insert(ns, "WeekDay") == NS_prj12::LN_WeekDay;
        flag&=QName::tokenizer().insert(ns, "DayType") == NS_prj12::LN_DayType;
        flag&=QName::tokenizer().insert(ns, "DayWorking") == NS_prj12::LN_DayWorking;
        flag&=QName::tokenizer().insert(ns, "TimePeriod") == NS_prj12::LN_TimePeriod;
        flag&=QName::tokenizer().insert(ns, "FromDate") == NS_prj12::LN_FromDate;
        flag&=QName::tokenizer().insert(ns, "ToDate") == NS_prj12::LN_ToDate;
        flag&=QName::tokenizer().insert(ns, "WorkingTimes") == NS_prj12::LN_WorkingTimes;
        flag&=QName::tokenizer().insert(ns, "WorkingTime") == NS_prj12::LN_WorkingTime;
        flag&=QName::tokenizer().insert(ns, "FromTime") == NS_prj12::LN_FromTime;
        flag&=QName::tokenizer().insert(ns, "ToTime") == NS_prj12::LN_ToTime;
        flag&=QName::tokenizer().insert(ns, "Tasks") == NS_prj12::LN_Tasks;
        flag&=QName::tokenizer().insert(ns, "Task") == NS_prj12::LN_Task;
        flag&=QName::tokenizer().insert(ns, "IsNull") == NS_prj12::LN_IsNull;
        flag&=QName::tokenizer().insert(ns, "CreateDate") == NS_prj12::LN_CreateDate;
        flag&=QName::tokenizer().insert(ns, "Contact") == NS_prj12::LN_Contact;
        flag&=QName::tokenizer().insert(ns, "WBS") == NS_prj12::LN_WBS;
        flag&=QName::tokenizer().insert(ns, "WBSLevel") == NS_prj12::LN_WBSLevel;
        flag&=QName::tokenizer().insert(ns, "OutlineNumber") == NS_prj12::LN_OutlineNumber;
        flag&=QName::tokenizer().insert(ns, "OutlineLevel") == NS_prj12::LN_OutlineLevel;
        flag&=QName::tokenizer().insert(ns, "Priority") == NS_prj12::LN_Priority;
        flag&=QName::tokenizer().insert(ns, "Duration") == NS_prj12::LN_Duration;
        flag&=QName::tokenizer().insert(ns, "Work") == NS_prj12::LN_Work;
        flag&=QName::tokenizer().insert(ns, "Stop") == NS_prj12::LN_Stop;
        flag&=QName::tokenizer().insert(ns, "Resume") == NS_prj12::LN_Resume;
        flag&=QName::tokenizer().insert(ns, "ResumeValid") == NS_prj12::LN_ResumeValid;
        flag&=QName::tokenizer().insert(ns, "EffortDriven") == NS_prj12::LN_EffortDriven;
        flag&=QName::tokenizer().insert(ns, "Recurring") == NS_prj12::LN_Recurring;
        flag&=QName::tokenizer().insert(ns, "OverAllocated") == NS_prj12::LN_OverAllocated;
        flag&=QName::tokenizer().insert(ns, "Estimated") == NS_prj12::LN_Estimated;
        flag&=QName::tokenizer().insert(ns, "Milestone") == NS_prj12::LN_Milestone;
        flag&=QName::tokenizer().insert(ns, "Summary") == NS_prj12::LN_Summary;
        flag&=QName::tokenizer().insert(ns, "Critical") == NS_prj12::LN_Critical;
        flag&=QName::tokenizer().insert(ns, "IsSubproject") == NS_prj12::LN_IsSubproject;
        flag&=QName::tokenizer().insert(ns, "IsSubprojectReadOnly") == NS_prj12::LN_IsSubprojectReadOnly;
        flag&=QName::tokenizer().insert(ns, "SubprojectName") == NS_prj12::LN_SubprojectName;
        flag&=QName::tokenizer().insert(ns, "ExternalTask") == NS_prj12::LN_ExternalTask;
        flag&=QName::tokenizer().insert(ns, "ExternalTaskProject") == NS_prj12::LN_ExternalTaskProject;
        flag&=QName::tokenizer().insert(ns, "EarlyStart") == NS_prj12::LN_EarlyStart;
        flag&=QName::tokenizer().insert(ns, "EarlyFinish") == NS_prj12::LN_EarlyFinish;
        flag&=QName::tokenizer().insert(ns, "LateStart") == NS_prj12::LN_LateStart;
        flag&=QName::tokenizer().insert(ns, "LateFinish") == NS_prj12::LN_LateFinish;
        flag&=QName::tokenizer().insert(ns, "StartVariance") == NS_prj12::LN_StartVariance;
        flag&=QName::tokenizer().insert(ns, "FinishVariance") == NS_prj12::LN_FinishVariance;
        flag&=QName::tokenizer().insert(ns, "WorkVariance") == NS_prj12::LN_WorkVariance;
        flag&=QName::tokenizer().insert(ns, "FreeSlack") == NS_prj12::LN_FreeSlack;
        flag&=QName::tokenizer().insert(ns, "TotalSlack") == NS_prj12::LN_TotalSlack;
        flag&=QName::tokenizer().insert(ns, "FixedCost") == NS_prj12::LN_FixedCost;
        flag&=QName::tokenizer().insert(ns, "FixedCostAccrual") == NS_prj12::LN_FixedCostAccrual;
        flag&=QName::tokenizer().insert(ns, "PercentComplete") == NS_prj12::LN_PercentComplete;
        flag&=QName::tokenizer().insert(ns, "PercentWorkComplete") == NS_prj12::LN_PercentWorkComplete;
        flag&=QName::tokenizer().insert(ns, "Cost") == NS_prj12::LN_Cost;
        flag&=QName::tokenizer().insert(ns, "OvertimeCost") == NS_prj12::LN_OvertimeCost;
        flag&=QName::tokenizer().insert(ns, "OvertimeWork") == NS_prj12::LN_OvertimeWork;
        flag&=QName::tokenizer().insert(ns, "ActualStart") == NS_prj12::LN_ActualStart;
        flag&=QName::tokenizer().insert(ns, "ActualFinish") == NS_prj12::LN_ActualFinish;
        flag&=QName::tokenizer().insert(ns, "ActualDuration") == NS_prj12::LN_ActualDuration;
        flag&=QName::tokenizer().insert(ns, "ActualCost") == NS_prj12::LN_ActualCost;
        flag&=QName::tokenizer().insert(ns, "ActualOvertimeCost") == NS_prj12::LN_ActualOvertimeCost;
        flag&=QName::tokenizer().insert(ns, "ActualWork") == NS_prj12::LN_ActualWork;
        flag&=QName::tokenizer().insert(ns, "ActualOvertimeWork") == NS_prj12::LN_ActualOvertimeWork;
        flag&=QName::tokenizer().insert(ns, "RegularWork") == NS_prj12::LN_RegularWork;
        flag&=QName::tokenizer().insert(ns, "RemainingDuration") == NS_prj12::LN_RemainingDuration;
        flag&=QName::tokenizer().insert(ns, "RemainingCost") == NS_prj12::LN_RemainingCost;
        flag&=QName::tokenizer().insert(ns, "RemainingWork") == NS_prj12::LN_RemainingWork;
        flag&=QName::tokenizer().insert(ns, "RemainingOvertimeCost") == NS_prj12::LN_RemainingOvertimeCost;
        flag&=QName::tokenizer().insert(ns, "RemainingOvertimeWork") == NS_prj12::LN_RemainingOvertimeWork;
        flag&=QName::tokenizer().insert(ns, "ACWP") == NS_prj12::LN_ACWP;
        flag&=QName::tokenizer().insert(ns, "CV") == NS_prj12::LN_CV;
        flag&=QName::tokenizer().insert(ns, "ConstraintType") == NS_prj12::LN_ConstraintType;
        flag&=QName::tokenizer().insert(ns, "ConstraintDate") == NS_prj12::LN_ConstraintDate;
        flag&=QName::tokenizer().insert(ns, "Deadline") == NS_prj12::LN_Deadline;
        flag&=QName::tokenizer().insert(ns, "LevelAssignments") == NS_prj12::LN_LevelAssignments;
        flag&=QName::tokenizer().insert(ns, "LevelingCanSplit") == NS_prj12::LN_LevelingCanSplit;
        flag&=QName::tokenizer().insert(ns, "LevelingDelay") == NS_prj12::LN_LevelingDelay;
        flag&=QName::tokenizer().insert(ns, "LevelingDelayFormat") == NS_prj12::LN_LevelingDelayFormat;
        flag&=QName::tokenizer().insert(ns, "PreLeveledStart") == NS_prj12::LN_PreLeveledStart;
        flag&=QName::tokenizer().insert(ns, "PreLeveledFinish") == NS_prj12::LN_PreLeveledFinish;
        flag&=QName::tokenizer().insert(ns, "Hyperlink") == NS_prj12::LN_Hyperlink;
        flag&=QName::tokenizer().insert(ns, "HyperlinkAddress") == NS_prj12::LN_HyperlinkAddress;
        flag&=QName::tokenizer().insert(ns, "HyperlinkSubAddress") == NS_prj12::LN_HyperlinkSubAddress;
        flag&=QName::tokenizer().insert(ns, "IgnoreResourceCalendar") == NS_prj12::LN_IgnoreResourceCalendar;
        flag&=QName::tokenizer().insert(ns, "Notes") == NS_prj12::LN_Notes;
        flag&=QName::tokenizer().insert(ns, "HideBar") == NS_prj12::LN_HideBar;
        flag&=QName::tokenizer().insert(ns, "Rollup") == NS_prj12::LN_Rollup;
        flag&=QName::tokenizer().insert(ns, "BCWS") == NS_prj12::LN_BCWS;
        flag&=QName::tokenizer().insert(ns, "BCWP") == NS_prj12::LN_BCWP;
        flag&=QName::tokenizer().insert(ns, "PhysicalPercentComplete") == NS_prj12::LN_PhysicalPercentComplete;
        flag&=QName::tokenizer().insert(ns, "PredecessorLink") == NS_prj12::LN_PredecessorLink;
        flag&=QName::tokenizer().insert(ns, "PredecessorUID") == NS_prj12::LN_PredecessorUID;
        flag&=QName::tokenizer().insert(ns, "CrossProject") == NS_prj12::LN_CrossProject;
        flag&=QName::tokenizer().insert(ns, "CrossProjectName") == NS_prj12::LN_CrossProjectName;
        flag&=QName::tokenizer().insert(ns, "LinkLag") == NS_prj12::LN_LinkLag;
        flag&=QName::tokenizer().insert(ns, "LagFormat") == NS_prj12::LN_LagFormat;
        flag&=QName::tokenizer().insert(ns, "ActualWorkProtected") == NS_prj12::LN_ActualWorkProtected;
        flag&=QName::tokenizer().insert(ns, "ActualOvertimeWorkProtected") == NS_prj12::LN_ActualOvertimeWorkProtected;
        flag&=QName::tokenizer().insert(ns, "Baseline") == NS_prj12::LN_Baseline;
        flag&=QName::tokenizer().insert(ns, "TimephasedData") == NS_prj12::LN_TimephasedData;
        flag&=QName::tokenizer().insert(ns, "Number") == NS_prj12::LN_Number;
        flag&=QName::tokenizer().insert(ns, "Interim") == NS_prj12::LN_Interim;
        flag&=QName::tokenizer().insert(ns, "EstimatedDuration") == NS_prj12::LN_EstimatedDuration;
        flag&=QName::tokenizer().insert(ns, "Resources") == NS_prj12::LN_Resources;
        flag&=QName::tokenizer().insert(ns, "Resource") == NS_prj12::LN_Resource;
        flag&=QName::tokenizer().insert(ns, "Initials") == NS_prj12::LN_Initials;
        flag&=QName::tokenizer().insert(ns, "Phonetics") == NS_prj12::LN_Phonetics;
        flag&=QName::tokenizer().insert(ns, "NTAccount") == NS_prj12::LN_NTAccount;
        flag&=QName::tokenizer().insert(ns, "MaterialLabel") == NS_prj12::LN_MaterialLabel;
        flag&=QName::tokenizer().insert(ns, "Code") == NS_prj12::LN_Code;
        flag&=QName::tokenizer().insert(ns, "Group") == NS_prj12::LN_Group;
        flag&=QName::tokenizer().insert(ns, "WorkGroup") == NS_prj12::LN_WorkGroup;
        flag&=QName::tokenizer().insert(ns, "EmailAddress") == NS_prj12::LN_EmailAddress;
        flag&=QName::tokenizer().insert(ns, "MaxUnits") == NS_prj12::LN_MaxUnits;
        flag&=QName::tokenizer().insert(ns, "PeakUnits") == NS_prj12::LN_PeakUnits;
        flag&=QName::tokenizer().insert(ns, "AvailableFrom") == NS_prj12::LN_AvailableFrom;
        flag&=QName::tokenizer().insert(ns, "AvailableTo") == NS_prj12::LN_AvailableTo;
        flag&=QName::tokenizer().insert(ns, "CanLevel") == NS_prj12::LN_CanLevel;
        flag&=QName::tokenizer().insert(ns, "AccrueAt") == NS_prj12::LN_AccrueAt;
        flag&=QName::tokenizer().insert(ns, "StandardRate") == NS_prj12::LN_StandardRate;
        flag&=QName::tokenizer().insert(ns, "StandardRateFormat") == NS_prj12::LN_StandardRateFormat;
        flag&=QName::tokenizer().insert(ns, "OvertimeRate") == NS_prj12::LN_OvertimeRate;
        flag&=QName::tokenizer().insert(ns, "OvertimeRateFormat") == NS_prj12::LN_OvertimeRateFormat;
        flag&=QName::tokenizer().insert(ns, "CostPerUse") == NS_prj12::LN_CostPerUse;
        flag&=QName::tokenizer().insert(ns, "CostVariance") == NS_prj12::LN_CostVariance;
        flag&=QName::tokenizer().insert(ns, "SV") == NS_prj12::LN_SV;
        flag&=QName::tokenizer().insert(ns, "IsGeneric") == NS_prj12::LN_IsGeneric;
        flag&=QName::tokenizer().insert(ns, "IsInactive") == NS_prj12::LN_IsInactive;
        flag&=QName::tokenizer().insert(ns, "IsEnterprise") == NS_prj12::LN_IsEnterprise;
        flag&=QName::tokenizer().insert(ns, "BookingType") == NS_prj12::LN_BookingType;
        flag&=QName::tokenizer().insert(ns, "ActiveDirectoryGUID") == NS_prj12::LN_ActiveDirectoryGUID;
        flag&=QName::tokenizer().insert(ns, "AvailabilityPeriods") == NS_prj12::LN_AvailabilityPeriods;
        flag&=QName::tokenizer().insert(ns, "AvailabilityPeriod") == NS_prj12::LN_AvailabilityPeriod;
        flag&=QName::tokenizer().insert(ns, "AvailableUnits") == NS_prj12::LN_AvailableUnits;
        flag&=QName::tokenizer().insert(ns, "Rates") == NS_prj12::LN_Rates;
        flag&=QName::tokenizer().insert(ns, "Rate") == NS_prj12::LN_Rate;
        flag&=QName::tokenizer().insert(ns, "RatesFrom") == NS_prj12::LN_RatesFrom;
        flag&=QName::tokenizer().insert(ns, "RatesTo") == NS_prj12::LN_RatesTo;
        flag&=QName::tokenizer().insert(ns, "RateTable") == NS_prj12::LN_RateTable;
        flag&=QName::tokenizer().insert(ns, "Assignments") == NS_prj12::LN_Assignments;
        flag&=QName::tokenizer().insert(ns, "Assignment") == NS_prj12::LN_Assignment;
        flag&=QName::tokenizer().insert(ns, "TaskUID") == NS_prj12::LN_TaskUID;
        flag&=QName::tokenizer().insert(ns, "ResourceUID") == NS_prj12::LN_ResourceUID;
        flag&=QName::tokenizer().insert(ns, "Confirmed") == NS_prj12::LN_Confirmed;
        flag&=QName::tokenizer().insert(ns, "CostRateTable") == NS_prj12::LN_CostRateTable;
        flag&=QName::tokenizer().insert(ns, "Delay") == NS_prj12::LN_Delay;
        flag&=QName::tokenizer().insert(ns, "HasFixedRateUnits") == NS_prj12::LN_HasFixedRateUnits;
        flag&=QName::tokenizer().insert(ns, "FixedMaterial") == NS_prj12::LN_FixedMaterial;
        flag&=QName::tokenizer().insert(ns, "LinkedFields") == NS_prj12::LN_LinkedFields;
        flag&=QName::tokenizer().insert(ns, "Overallocated") == NS_prj12::LN_Overallocated;
        flag&=QName::tokenizer().insert(ns, "ResponsePending") == NS_prj12::LN_ResponsePending;
        flag&=QName::tokenizer().insert(ns, "Units") == NS_prj12::LN_Units;
        flag&=QName::tokenizer().insert(ns, "UpdateNeeded") == NS_prj12::LN_UpdateNeeded;
        flag&=QName::tokenizer().insert(ns, "VAC") == NS_prj12::LN_VAC;
        flag&=QName::tokenizer().insert(ns, "WorkContour") == NS_prj12::LN_WorkContour;
        CPPUNIT_ASSERT_MESSAGE("test for namespace prj12 failed", flag);}
    void test_mdc12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/package/2005/06/metadata/core-properties";
        flag&=QName::tokenizer().insert(ns, "CoreProperties") == NS_mdc12::LN_CoreProperties;
        flag&=QName::tokenizer().insert(ns, "Creator") == NS_mdc12::LN_Creator;
        flag&=QName::tokenizer().insert(ns, "Title") == NS_mdc12::LN_Title;
        flag&=QName::tokenizer().insert(ns, "Subject") == NS_mdc12::LN_Subject;
        flag&=QName::tokenizer().insert(ns, "Category") == NS_mdc12::LN_Category;
        flag&=QName::tokenizer().insert(ns, "Keywords") == NS_mdc12::LN_Keywords;
        flag&=QName::tokenizer().insert(ns, "Description") == NS_mdc12::LN_Description;
        flag&=QName::tokenizer().insert(ns, "ContentType") == NS_mdc12::LN_ContentType;
        flag&=QName::tokenizer().insert(ns, "ContentStatus") == NS_mdc12::LN_ContentStatus;
        flag&=QName::tokenizer().insert(ns, "Version") == NS_mdc12::LN_Version;
        flag&=QName::tokenizer().insert(ns, "Revision") == NS_mdc12::LN_Revision;
        flag&=QName::tokenizer().insert(ns, "DateCreated") == NS_mdc12::LN_DateCreated;
        flag&=QName::tokenizer().insert(ns, "DateModified") == NS_mdc12::LN_DateModified;
        flag&=QName::tokenizer().insert(ns, "LastModifiedBy") == NS_mdc12::LN_LastModifiedBy;
        flag&=QName::tokenizer().insert(ns, "LastPrinted") == NS_mdc12::LN_LastPrinted;
        flag&=QName::tokenizer().insert(ns, "Language") == NS_mdc12::LN_Language;
        flag&=QName::tokenizer().insert(ns, "Identifier") == NS_mdc12::LN_Identifier;
        CPPUNIT_ASSERT_MESSAGE("test for namespace mdc12 failed", flag);}
    void test_table(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:table:1.0";
        flag&=QName::tokenizer().insert(ns, "dde-link") == NS_table::LN_dde_link;
        flag&=QName::tokenizer().insert(ns, "table-template") == NS_table::LN_table_template;
        flag&=QName::tokenizer().insert(ns, "first-row") == NS_table::LN_first_row;
        flag&=QName::tokenizer().insert(ns, "last-row") == NS_table::LN_last_row;
        flag&=QName::tokenizer().insert(ns, "first-column") == NS_table::LN_first_column;
        flag&=QName::tokenizer().insert(ns, "last-column") == NS_table::LN_last_column;
        flag&=QName::tokenizer().insert(ns, "body") == NS_table::LN_body;
        flag&=QName::tokenizer().insert(ns, "even-rows") == NS_table::LN_even_rows;
        flag&=QName::tokenizer().insert(ns, "odd-rows") == NS_table::LN_odd_rows;
        flag&=QName::tokenizer().insert(ns, "even-columns") == NS_table::LN_even_columns;
        flag&=QName::tokenizer().insert(ns, "odd-columns") == NS_table::LN_odd_columns;
        flag&=QName::tokenizer().insert(ns, "table") == NS_table::LN_table;
        flag&=QName::tokenizer().insert(ns, "name") == NS_table::LN_name;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_table::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "protected") == NS_table::LN_protected;
        flag&=QName::tokenizer().insert(ns, "protection-key") == NS_table::LN_protection_key;
        flag&=QName::tokenizer().insert(ns, "print") == NS_table::LN_print;
        flag&=QName::tokenizer().insert(ns, "print-ranges") == NS_table::LN_print_ranges;
        flag&=QName::tokenizer().insert(ns, "table-row") == NS_table::LN_table_row;
        flag&=QName::tokenizer().insert(ns, "number-rows-repeated") == NS_table::LN_number_rows_repeated;
        flag&=QName::tokenizer().insert(ns, "default-cell-style-name") == NS_table::LN_default_cell_style_name;
        flag&=QName::tokenizer().insert(ns, "visibility") == NS_table::LN_visibility;
        flag&=QName::tokenizer().insert(ns, "table-cell") == NS_table::LN_table_cell;
        flag&=QName::tokenizer().insert(ns, "covered-table-cell") == NS_table::LN_covered_table_cell;
        flag&=QName::tokenizer().insert(ns, "number-columns-repeated") == NS_table::LN_number_columns_repeated;
        flag&=QName::tokenizer().insert(ns, "number-columns-spanned") == NS_table::LN_number_columns_spanned;
        flag&=QName::tokenizer().insert(ns, "number-rows-spanned") == NS_table::LN_number_rows_spanned;
        flag&=QName::tokenizer().insert(ns, "content-validation-name") == NS_table::LN_content_validation_name;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_table::LN_formula;
        flag&=QName::tokenizer().insert(ns, "number-matrix-columns-spanned") == NS_table::LN_number_matrix_columns_spanned;
        flag&=QName::tokenizer().insert(ns, "number-matrix-rows-spanned") == NS_table::LN_number_matrix_rows_spanned;
        flag&=QName::tokenizer().insert(ns, "protect") == NS_table::LN_protect;
        flag&=QName::tokenizer().insert(ns, "table-column") == NS_table::LN_table_column;
        flag&=QName::tokenizer().insert(ns, "table-header-columns") == NS_table::LN_table_header_columns;
        flag&=QName::tokenizer().insert(ns, "table-columns") == NS_table::LN_table_columns;
        flag&=QName::tokenizer().insert(ns, "table-column-group") == NS_table::LN_table_column_group;
        flag&=QName::tokenizer().insert(ns, "display") == NS_table::LN_display;
        flag&=QName::tokenizer().insert(ns, "table-header-rows") == NS_table::LN_table_header_rows;
        flag&=QName::tokenizer().insert(ns, "table-rows") == NS_table::LN_table_rows;
        flag&=QName::tokenizer().insert(ns, "table-row-group") == NS_table::LN_table_row_group;
        flag&=QName::tokenizer().insert(ns, "is-sub-table") == NS_table::LN_is_sub_table;
        flag&=QName::tokenizer().insert(ns, "table-source") == NS_table::LN_table_source;
        flag&=QName::tokenizer().insert(ns, "mode") == NS_table::LN_mode;
        flag&=QName::tokenizer().insert(ns, "table-name") == NS_table::LN_table_name;
        flag&=QName::tokenizer().insert(ns, "filter-name") == NS_table::LN_filter_name;
        flag&=QName::tokenizer().insert(ns, "filter-options") == NS_table::LN_filter_options;
        flag&=QName::tokenizer().insert(ns, "refresh-delay") == NS_table::LN_refresh_delay;
        flag&=QName::tokenizer().insert(ns, "scenario") == NS_table::LN_scenario;
        flag&=QName::tokenizer().insert(ns, "scenario-ranges") == NS_table::LN_scenario_ranges;
        flag&=QName::tokenizer().insert(ns, "is-active") == NS_table::LN_is_active;
        flag&=QName::tokenizer().insert(ns, "display-border") == NS_table::LN_display_border;
        flag&=QName::tokenizer().insert(ns, "border-color") == NS_table::LN_border_color;
        flag&=QName::tokenizer().insert(ns, "copy-back") == NS_table::LN_copy_back;
        flag&=QName::tokenizer().insert(ns, "copy-styles") == NS_table::LN_copy_styles;
        flag&=QName::tokenizer().insert(ns, "copy-formulas") == NS_table::LN_copy_formulas;
        flag&=QName::tokenizer().insert(ns, "comment") == NS_table::LN_comment;
        flag&=QName::tokenizer().insert(ns, "shapes") == NS_table::LN_shapes;
        flag&=QName::tokenizer().insert(ns, "cell-range-source") == NS_table::LN_cell_range_source;
        flag&=QName::tokenizer().insert(ns, "last-column-spanned") == NS_table::LN_last_column_spanned;
        flag&=QName::tokenizer().insert(ns, "last-row-spanned") == NS_table::LN_last_row_spanned;
        flag&=QName::tokenizer().insert(ns, "detective") == NS_table::LN_detective;
        flag&=QName::tokenizer().insert(ns, "operation") == NS_table::LN_operation;
        flag&=QName::tokenizer().insert(ns, "index") == NS_table::LN_index;
        flag&=QName::tokenizer().insert(ns, "highlighted-range") == NS_table::LN_highlighted_range;
        flag&=QName::tokenizer().insert(ns, "cell-range-address") == NS_table::LN_cell_range_address;
        flag&=QName::tokenizer().insert(ns, "direction") == NS_table::LN_direction;
        flag&=QName::tokenizer().insert(ns, "contains-error") == NS_table::LN_contains_error;
        flag&=QName::tokenizer().insert(ns, "marked-invalid") == NS_table::LN_marked_invalid;
        flag&=QName::tokenizer().insert(ns, "structure-protected") == NS_table::LN_structure_protected;
        flag&=QName::tokenizer().insert(ns, "calculation-settings") == NS_table::LN_calculation_settings;
        flag&=QName::tokenizer().insert(ns, "case-sensitive") == NS_table::LN_case_sensitive;
        flag&=QName::tokenizer().insert(ns, "precision-as-shown") == NS_table::LN_precision_as_shown;
        flag&=QName::tokenizer().insert(ns, "search-criteria-must-apply-to-whole-cell") == NS_table::LN_search_criteria_must_apply_to_whole_cell;
        flag&=QName::tokenizer().insert(ns, "automatic-find-labels") == NS_table::LN_automatic_find_labels;
        flag&=QName::tokenizer().insert(ns, "use-regular-expressions") == NS_table::LN_use_regular_expressions;
        flag&=QName::tokenizer().insert(ns, "null-year") == NS_table::LN_null_year;
        flag&=QName::tokenizer().insert(ns, "null-date") == NS_table::LN_null_date;
        flag&=QName::tokenizer().insert(ns, "value-type") == NS_table::LN_value_type;
        flag&=QName::tokenizer().insert(ns, "date-value-type") == NS_table::LN_date_value_type;
        flag&=QName::tokenizer().insert(ns, "iteration") == NS_table::LN_iteration;
        flag&=QName::tokenizer().insert(ns, "status") == NS_table::LN_status;
        flag&=QName::tokenizer().insert(ns, "steps") == NS_table::LN_steps;
        flag&=QName::tokenizer().insert(ns, "maximum-difference") == NS_table::LN_maximum_difference;
        flag&=QName::tokenizer().insert(ns, "content-validations") == NS_table::LN_content_validations;
        flag&=QName::tokenizer().insert(ns, "content-validation") == NS_table::LN_content_validation;
        flag&=QName::tokenizer().insert(ns, "condition") == NS_table::LN_condition;
        flag&=QName::tokenizer().insert(ns, "base-cell-address") == NS_table::LN_base_cell_address;
        flag&=QName::tokenizer().insert(ns, "allow-empty-cell") == NS_table::LN_allow_empty_cell;
        flag&=QName::tokenizer().insert(ns, "display-list") == NS_table::LN_display_list;
        flag&=QName::tokenizer().insert(ns, "help-message") == NS_table::LN_help_message;
        flag&=QName::tokenizer().insert(ns, "title") == NS_table::LN_title;
        flag&=QName::tokenizer().insert(ns, "error-message") == NS_table::LN_error_message;
        flag&=QName::tokenizer().insert(ns, "message-type") == NS_table::LN_message_type;
        flag&=QName::tokenizer().insert(ns, "error-macro") == NS_table::LN_error_macro;
        flag&=QName::tokenizer().insert(ns, "execute") == NS_table::LN_execute;
        flag&=QName::tokenizer().insert(ns, "label-ranges") == NS_table::LN_label_ranges;
        flag&=QName::tokenizer().insert(ns, "label-range") == NS_table::LN_label_range;
        flag&=QName::tokenizer().insert(ns, "label-cell-range-address") == NS_table::LN_label_cell_range_address;
        flag&=QName::tokenizer().insert(ns, "data-cell-range-address") == NS_table::LN_data_cell_range_address;
        flag&=QName::tokenizer().insert(ns, "orientation") == NS_table::LN_orientation;
        flag&=QName::tokenizer().insert(ns, "named-expressions") == NS_table::LN_named_expressions;
        flag&=QName::tokenizer().insert(ns, "named-range") == NS_table::LN_named_range;
        flag&=QName::tokenizer().insert(ns, "range-usable-as") == NS_table::LN_range_usable_as;
        flag&=QName::tokenizer().insert(ns, "named-expression") == NS_table::LN_named_expression;
        flag&=QName::tokenizer().insert(ns, "expression") == NS_table::LN_expression;
        flag&=QName::tokenizer().insert(ns, "database-ranges") == NS_table::LN_database_ranges;
        flag&=QName::tokenizer().insert(ns, "database-range") == NS_table::LN_database_range;
        flag&=QName::tokenizer().insert(ns, "is-selection") == NS_table::LN_is_selection;
        flag&=QName::tokenizer().insert(ns, "on-update-keep-styles") == NS_table::LN_on_update_keep_styles;
        flag&=QName::tokenizer().insert(ns, "on-update-keep-size") == NS_table::LN_on_update_keep_size;
        flag&=QName::tokenizer().insert(ns, "has-persistent-data") == NS_table::LN_has_persistent_data;
        flag&=QName::tokenizer().insert(ns, "contains-header") == NS_table::LN_contains_header;
        flag&=QName::tokenizer().insert(ns, "display-filter-buttons") == NS_table::LN_display_filter_buttons;
        flag&=QName::tokenizer().insert(ns, "target-range-address") == NS_table::LN_target_range_address;
        flag&=QName::tokenizer().insert(ns, "database-source-sql") == NS_table::LN_database_source_sql;
        flag&=QName::tokenizer().insert(ns, "database-name") == NS_table::LN_database_name;
        flag&=QName::tokenizer().insert(ns, "sql-statement") == NS_table::LN_sql_statement;
        flag&=QName::tokenizer().insert(ns, "parse-sql-statement") == NS_table::LN_parse_sql_statement;
        flag&=QName::tokenizer().insert(ns, "database-source-table") == NS_table::LN_database_source_table;
        flag&=QName::tokenizer().insert(ns, "database-table-name") == NS_table::LN_database_table_name;
        flag&=QName::tokenizer().insert(ns, "database-source-query") == NS_table::LN_database_source_query;
        flag&=QName::tokenizer().insert(ns, "query-name") == NS_table::LN_query_name;
        flag&=QName::tokenizer().insert(ns, "sort") == NS_table::LN_sort;
        flag&=QName::tokenizer().insert(ns, "bind-styles-to-content") == NS_table::LN_bind_styles_to_content;
        flag&=QName::tokenizer().insert(ns, "language") == NS_table::LN_language;
        flag&=QName::tokenizer().insert(ns, "country") == NS_table::LN_country;
        flag&=QName::tokenizer().insert(ns, "algorithm") == NS_table::LN_algorithm;
        flag&=QName::tokenizer().insert(ns, "sort-by") == NS_table::LN_sort_by;
        flag&=QName::tokenizer().insert(ns, "field-number") == NS_table::LN_field_number;
        flag&=QName::tokenizer().insert(ns, "data-type") == NS_table::LN_data_type;
        flag&=QName::tokenizer().insert(ns, "order") == NS_table::LN_order;
        flag&=QName::tokenizer().insert(ns, "subtotal-rules") == NS_table::LN_subtotal_rules;
        flag&=QName::tokenizer().insert(ns, "page-breaks-on-group-change") == NS_table::LN_page_breaks_on_group_change;
        flag&=QName::tokenizer().insert(ns, "sort-groups") == NS_table::LN_sort_groups;
        flag&=QName::tokenizer().insert(ns, "subtotal-rule") == NS_table::LN_subtotal_rule;
        flag&=QName::tokenizer().insert(ns, "group-by-field-number") == NS_table::LN_group_by_field_number;
        flag&=QName::tokenizer().insert(ns, "subtotal-field") == NS_table::LN_subtotal_field;
        flag&=QName::tokenizer().insert(ns, "function") == NS_table::LN_function;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_table::LN_filter;
        flag&=QName::tokenizer().insert(ns, "condition-source") == NS_table::LN_condition_source;
        flag&=QName::tokenizer().insert(ns, "condition-source-range-address") == NS_table::LN_condition_source_range_address;
        flag&=QName::tokenizer().insert(ns, "display-duplicates") == NS_table::LN_display_duplicates;
        flag&=QName::tokenizer().insert(ns, "filter-and") == NS_table::LN_filter_and;
        flag&=QName::tokenizer().insert(ns, "filter-or") == NS_table::LN_filter_or;
        flag&=QName::tokenizer().insert(ns, "filter-condition") == NS_table::LN_filter_condition;
        flag&=QName::tokenizer().insert(ns, "value") == NS_table::LN_value;
        flag&=QName::tokenizer().insert(ns, "operator") == NS_table::LN_operator;
        flag&=QName::tokenizer().insert(ns, "data-pilot-tables") == NS_table::LN_data_pilot_tables;
        flag&=QName::tokenizer().insert(ns, "data-pilot-table") == NS_table::LN_data_pilot_table;
        flag&=QName::tokenizer().insert(ns, "application-data") == NS_table::LN_application_data;
        flag&=QName::tokenizer().insert(ns, "grand-total") == NS_table::LN_grand_total;
        flag&=QName::tokenizer().insert(ns, "ignore-empty-rows") == NS_table::LN_ignore_empty_rows;
        flag&=QName::tokenizer().insert(ns, "identify-categories") == NS_table::LN_identify_categories;
        flag&=QName::tokenizer().insert(ns, "buttons") == NS_table::LN_buttons;
        flag&=QName::tokenizer().insert(ns, "show-filter-button") == NS_table::LN_show_filter_button;
        flag&=QName::tokenizer().insert(ns, "drill-down-on-double-click") == NS_table::LN_drill_down_on_double_click;
        flag&=QName::tokenizer().insert(ns, "source-cell-range") == NS_table::LN_source_cell_range;
        flag&=QName::tokenizer().insert(ns, "source-service") == NS_table::LN_source_service;
        flag&=QName::tokenizer().insert(ns, "source-name") == NS_table::LN_source_name;
        flag&=QName::tokenizer().insert(ns, "object-name") == NS_table::LN_object_name;
        flag&=QName::tokenizer().insert(ns, "user-name") == NS_table::LN_user_name;
        flag&=QName::tokenizer().insert(ns, "password") == NS_table::LN_password;
        flag&=QName::tokenizer().insert(ns, "data-pilot-field") == NS_table::LN_data_pilot_field;
        flag&=QName::tokenizer().insert(ns, "source-field-name") == NS_table::LN_source_field_name;
        flag&=QName::tokenizer().insert(ns, "selected-page") == NS_table::LN_selected_page;
        flag&=QName::tokenizer().insert(ns, "is-data-layout-field") == NS_table::LN_is_data_layout_field;
        flag&=QName::tokenizer().insert(ns, "used-hierarchy") == NS_table::LN_used_hierarchy;
        flag&=QName::tokenizer().insert(ns, "data-pilot-level") == NS_table::LN_data_pilot_level;
        flag&=QName::tokenizer().insert(ns, "show-empty") == NS_table::LN_show_empty;
        flag&=QName::tokenizer().insert(ns, "data-pilot-subtotals") == NS_table::LN_data_pilot_subtotals;
        flag&=QName::tokenizer().insert(ns, "align") == NS_table::LN_align;
        flag&=QName::tokenizer().insert(ns, "data-pilot-subtotal") == NS_table::LN_data_pilot_subtotal;
        flag&=QName::tokenizer().insert(ns, "data-pilot-members") == NS_table::LN_data_pilot_members;
        flag&=QName::tokenizer().insert(ns, "border-model") == NS_table::LN_border_model;
        flag&=QName::tokenizer().insert(ns, "data-pilot-member") == NS_table::LN_data_pilot_member;
        flag&=QName::tokenizer().insert(ns, "show-details") == NS_table::LN_show_details;
        flag&=QName::tokenizer().insert(ns, "data-pilot-display-info") == NS_table::LN_data_pilot_display_info;
        flag&=QName::tokenizer().insert(ns, "enabled") == NS_table::LN_enabled;
        flag&=QName::tokenizer().insert(ns, "data-field") == NS_table::LN_data_field;
        flag&=QName::tokenizer().insert(ns, "member-count") == NS_table::LN_member_count;
        flag&=QName::tokenizer().insert(ns, "display-member-mode") == NS_table::LN_display_member_mode;
        flag&=QName::tokenizer().insert(ns, "data-pilot-sort-info") == NS_table::LN_data_pilot_sort_info;
        flag&=QName::tokenizer().insert(ns, "sort-mode") == NS_table::LN_sort_mode;
        flag&=QName::tokenizer().insert(ns, "data-pilot-layout-info") == NS_table::LN_data_pilot_layout_info;
        flag&=QName::tokenizer().insert(ns, "layout-mode") == NS_table::LN_layout_mode;
        flag&=QName::tokenizer().insert(ns, "add-empty-lines") == NS_table::LN_add_empty_lines;
        flag&=QName::tokenizer().insert(ns, "data-pilot-field-reference") == NS_table::LN_data_pilot_field_reference;
        flag&=QName::tokenizer().insert(ns, "field-name") == NS_table::LN_field_name;
        flag&=QName::tokenizer().insert(ns, "member-type") == NS_table::LN_member_type;
        flag&=QName::tokenizer().insert(ns, "member-name") == NS_table::LN_member_name;
        flag&=QName::tokenizer().insert(ns, "type") == NS_table::LN_type;
        flag&=QName::tokenizer().insert(ns, "data-pilot-groups") == NS_table::LN_data_pilot_groups;
        flag&=QName::tokenizer().insert(ns, "date-start") == NS_table::LN_date_start;
        flag&=QName::tokenizer().insert(ns, "start") == NS_table::LN_start;
        flag&=QName::tokenizer().insert(ns, "date-end") == NS_table::LN_date_end;
        flag&=QName::tokenizer().insert(ns, "end") == NS_table::LN_end;
        flag&=QName::tokenizer().insert(ns, "step") == NS_table::LN_step;
        flag&=QName::tokenizer().insert(ns, "grouped-by") == NS_table::LN_grouped_by;
        flag&=QName::tokenizer().insert(ns, "data-pilot-group") == NS_table::LN_data_pilot_group;
        flag&=QName::tokenizer().insert(ns, "data-pilot-group-member") == NS_table::LN_data_pilot_group_member;
        flag&=QName::tokenizer().insert(ns, "consolidation") == NS_table::LN_consolidation;
        flag&=QName::tokenizer().insert(ns, "source-cell-range-addresses") == NS_table::LN_source_cell_range_addresses;
        flag&=QName::tokenizer().insert(ns, "target-cell-address") == NS_table::LN_target_cell_address;
        flag&=QName::tokenizer().insert(ns, "use-labels") == NS_table::LN_use_labels;
        flag&=QName::tokenizer().insert(ns, "link-to-source-data") == NS_table::LN_link_to_source_data;
        flag&=QName::tokenizer().insert(ns, "dde-links") == NS_table::LN_dde_links;
        flag&=QName::tokenizer().insert(ns, "tracked-changes") == NS_table::LN_tracked_changes;
        flag&=QName::tokenizer().insert(ns, "track-changes") == NS_table::LN_track_changes;
        flag&=QName::tokenizer().insert(ns, "insertion") == NS_table::LN_insertion;
        flag&=QName::tokenizer().insert(ns, "position") == NS_table::LN_position;
        flag&=QName::tokenizer().insert(ns, "count") == NS_table::LN_count;
        flag&=QName::tokenizer().insert(ns, "dependencies") == NS_table::LN_dependencies;
        flag&=QName::tokenizer().insert(ns, "dependency") == NS_table::LN_dependency;
        flag&=QName::tokenizer().insert(ns, "id") == NS_table::LN_id;
        flag&=QName::tokenizer().insert(ns, "deletions") == NS_table::LN_deletions;
        flag&=QName::tokenizer().insert(ns, "cell-content-deletion") == NS_table::LN_cell_content_deletion;
        flag&=QName::tokenizer().insert(ns, "change-deletion") == NS_table::LN_change_deletion;
        flag&=QName::tokenizer().insert(ns, "deletion") == NS_table::LN_deletion;
        flag&=QName::tokenizer().insert(ns, "multi-deletion-spanned") == NS_table::LN_multi_deletion_spanned;
        flag&=QName::tokenizer().insert(ns, "cut-offs") == NS_table::LN_cut_offs;
        flag&=QName::tokenizer().insert(ns, "insertion-cut-off") == NS_table::LN_insertion_cut_off;
        flag&=QName::tokenizer().insert(ns, "movement-cut-off") == NS_table::LN_movement_cut_off;
        flag&=QName::tokenizer().insert(ns, "start-position") == NS_table::LN_start_position;
        flag&=QName::tokenizer().insert(ns, "end-position") == NS_table::LN_end_position;
        flag&=QName::tokenizer().insert(ns, "movement") == NS_table::LN_movement;
        flag&=QName::tokenizer().insert(ns, "source-range-address") == NS_table::LN_source_range_address;
        flag&=QName::tokenizer().insert(ns, "column") == NS_table::LN_column;
        flag&=QName::tokenizer().insert(ns, "row") == NS_table::LN_row;
        flag&=QName::tokenizer().insert(ns, "start-column") == NS_table::LN_start_column;
        flag&=QName::tokenizer().insert(ns, "start-row") == NS_table::LN_start_row;
        flag&=QName::tokenizer().insert(ns, "start-table") == NS_table::LN_start_table;
        flag&=QName::tokenizer().insert(ns, "end-column") == NS_table::LN_end_column;
        flag&=QName::tokenizer().insert(ns, "end-row") == NS_table::LN_end_row;
        flag&=QName::tokenizer().insert(ns, "end-table") == NS_table::LN_end_table;
        flag&=QName::tokenizer().insert(ns, "change-track-table-cell") == NS_table::LN_change_track_table_cell;
        flag&=QName::tokenizer().insert(ns, "cell-address") == NS_table::LN_cell_address;
        flag&=QName::tokenizer().insert(ns, "matrix-covered") == NS_table::LN_matrix_covered;
        flag&=QName::tokenizer().insert(ns, "cell-content-change") == NS_table::LN_cell_content_change;
        flag&=QName::tokenizer().insert(ns, "previous") == NS_table::LN_previous;
        flag&=QName::tokenizer().insert(ns, "acceptance-state") == NS_table::LN_acceptance_state;
        flag&=QName::tokenizer().insert(ns, "rejecting-change-id") == NS_table::LN_rejecting_change_id;
        flag&=QName::tokenizer().insert(ns, "end-cell-address") == NS_table::LN_end_cell_address;
        flag&=QName::tokenizer().insert(ns, "cell-range") == NS_table::LN_cell_range;
        flag&=QName::tokenizer().insert(ns, "end-x") == NS_table::LN_end_x;
        flag&=QName::tokenizer().insert(ns, "end-y") == NS_table::LN_end_y;
        flag&=QName::tokenizer().insert(ns, "table-background") == NS_table::LN_table_background;
        CPPUNIT_ASSERT_MESSAGE("test for namespace table failed", flag);}
    void test_rs11(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:rowset";
        flag&=QName::tokenizer().insert(ns, "data") == NS_rs11::LN_data;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rs11 failed", flag);}
    void test_sxr12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/sxref";
        flag&=QName::tokenizer().insert(ns, "pivotRefs") == NS_sxr12::LN_pivotRefs;
        flag&=QName::tokenizer().insert(ns, "pivotRef") == NS_sxr12::LN_pivotRef;
        flag&=QName::tokenizer().insert(ns, "rule") == NS_sxr12::LN_rule;
        flag&=QName::tokenizer().insert(ns, "pivotRefInternal") == NS_sxr12::LN_pivotRefInternal;
        flag&=QName::tokenizer().insert(ns, "flags") == NS_sxr12::LN_flags;
        flag&=QName::tokenizer().insert(ns, "slicers") == NS_sxr12::LN_slicers;
        flag&=QName::tokenizer().insert(ns, "slicer") == NS_sxr12::LN_slicer;
        flag&=QName::tokenizer().insert(ns, "elements") == NS_sxr12::LN_elements;
        flag&=QName::tokenizer().insert(ns, "element") == NS_sxr12::LN_element;
        flag&=QName::tokenizer().insert(ns, "n") == NS_sxr12::LN_n;
        flag&=QName::tokenizer().insert(ns, "d") == NS_sxr12::LN_d;
        flag&=QName::tokenizer().insert(ns, "x") == NS_sxr12::LN_x;
        flag&=QName::tokenizer().insert(ns, "s") == NS_sxr12::LN_s;
        flag&=QName::tokenizer().insert(ns, "b") == NS_sxr12::LN_b;
        flag&=QName::tokenizer().insert(ns, "e") == NS_sxr12::LN_e;
        CPPUNIT_ASSERT_MESSAGE("test for namespace sxr12 failed", flag);}
    void test_xldr12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/officeart/2005/8/xldr";
        flag&=QName::tokenizer().insert(ns, "cdrElemPr") == NS_xldr12::LN_cdrElemPr;
        flag&=QName::tokenizer().insert(ns, "spPr") == NS_xldr12::LN_spPr;
        flag&=QName::tokenizer().insert(ns, "style") == NS_xldr12::LN_style;
        flag&=QName::tokenizer().insert(ns, "txBody") == NS_xldr12::LN_txBody;
        flag&=QName::tokenizer().insert(ns, "xfrm") == NS_xldr12::LN_xfrm;
        flag&=QName::tokenizer().insert(ns, "stCxn") == NS_xldr12::LN_stCxn;
        flag&=QName::tokenizer().insert(ns, "endCxn") == NS_xldr12::LN_endCxn;
        flag&=QName::tokenizer().insert(ns, "graphic") == NS_xldr12::LN_graphic;
        flag&=QName::tokenizer().insert(ns, "gspPr") == NS_xldr12::LN_gspPr;
        flag&=QName::tokenizer().insert(ns, "sp") == NS_xldr12::LN_sp;
        flag&=QName::tokenizer().insert(ns, "gsp") == NS_xldr12::LN_gsp;
        flag&=QName::tokenizer().insert(ns, "graphicFrame") == NS_xldr12::LN_graphicFrame;
        flag&=QName::tokenizer().insert(ns, "cxnsp") == NS_xldr12::LN_cxnsp;
        flag&=QName::tokenizer().insert(ns, "col") == NS_xldr12::LN_col;
        flag&=QName::tokenizer().insert(ns, "colOff") == NS_xldr12::LN_colOff;
        flag&=QName::tokenizer().insert(ns, "row") == NS_xldr12::LN_row;
        flag&=QName::tokenizer().insert(ns, "rowOff") == NS_xldr12::LN_rowOff;
        flag&=QName::tokenizer().insert(ns, "from") == NS_xldr12::LN_from;
        flag&=QName::tokenizer().insert(ns, "to") == NS_xldr12::LN_to;
        flag&=QName::tokenizer().insert(ns, "clientData") == NS_xldr12::LN_clientData;
        flag&=QName::tokenizer().insert(ns, "extents") == NS_xldr12::LN_extents;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_xldr12::LN_pos;
        flag&=QName::tokenizer().insert(ns, "twoCellAnchor") == NS_xldr12::LN_twoCellAnchor;
        flag&=QName::tokenizer().insert(ns, "oneCellAnchor") == NS_xldr12::LN_oneCellAnchor;
        flag&=QName::tokenizer().insert(ns, "absoluteAnchor") == NS_xldr12::LN_absoluteAnchor;
        CPPUNIT_ASSERT_MESSAGE("test for namespace xldr12 failed", flag);}
    void test_m12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/omml/2004/12/core";
        flag&=QName::tokenizer().insert(ns, "nor") == NS_m12::LN_nor;
        flag&=QName::tokenizer().insert(ns, "lit") == NS_m12::LN_lit;
        flag&=QName::tokenizer().insert(ns, "scr") == NS_m12::LN_scr;
        flag&=QName::tokenizer().insert(ns, "sty") == NS_m12::LN_sty;
        flag&=QName::tokenizer().insert(ns, "brk") == NS_m12::LN_brk;
        flag&=QName::tokenizer().insert(ns, "aln") == NS_m12::LN_aln;
        flag&=QName::tokenizer().insert(ns, "rPr") == NS_m12::LN_rPr;
        flag&=QName::tokenizer().insert(ns, "t") == NS_m12::LN_t;
        flag&=QName::tokenizer().insert(ns, "chr") == NS_m12::LN_chr;
        flag&=QName::tokenizer().insert(ns, "accPr") == NS_m12::LN_accPr;
        flag&=QName::tokenizer().insert(ns, "e") == NS_m12::LN_e;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_m12::LN_pos;
        flag&=QName::tokenizer().insert(ns, "barPr") == NS_m12::LN_barPr;
        flag&=QName::tokenizer().insert(ns, "opEmu") == NS_m12::LN_opEmu;
        flag&=QName::tokenizer().insert(ns, "noBreak") == NS_m12::LN_noBreak;
        flag&=QName::tokenizer().insert(ns, "boxPr") == NS_m12::LN_boxPr;
        flag&=QName::tokenizer().insert(ns, "hideTop") == NS_m12::LN_hideTop;
        flag&=QName::tokenizer().insert(ns, "hideBot") == NS_m12::LN_hideBot;
        flag&=QName::tokenizer().insert(ns, "hideLeft") == NS_m12::LN_hideLeft;
        flag&=QName::tokenizer().insert(ns, "hideRight") == NS_m12::LN_hideRight;
        flag&=QName::tokenizer().insert(ns, "strikeH") == NS_m12::LN_strikeH;
        flag&=QName::tokenizer().insert(ns, "strikeV") == NS_m12::LN_strikeV;
        flag&=QName::tokenizer().insert(ns, "strikeBLTR") == NS_m12::LN_strikeBLTR;
        flag&=QName::tokenizer().insert(ns, "strikeTLBR") == NS_m12::LN_strikeTLBR;
        flag&=QName::tokenizer().insert(ns, "boxFormPr") == NS_m12::LN_boxFormPr;
        flag&=QName::tokenizer().insert(ns, "begChr") == NS_m12::LN_begChr;
        flag&=QName::tokenizer().insert(ns, "sepChr") == NS_m12::LN_sepChr;
        flag&=QName::tokenizer().insert(ns, "endChr") == NS_m12::LN_endChr;
        flag&=QName::tokenizer().insert(ns, "grow") == NS_m12::LN_grow;
        flag&=QName::tokenizer().insert(ns, "shp") == NS_m12::LN_shp;
        flag&=QName::tokenizer().insert(ns, "dPr") == NS_m12::LN_dPr;
        flag&=QName::tokenizer().insert(ns, "baseJc") == NS_m12::LN_baseJc;
        flag&=QName::tokenizer().insert(ns, "maxDist") == NS_m12::LN_maxDist;
        flag&=QName::tokenizer().insert(ns, "objDist") == NS_m12::LN_objDist;
        flag&=QName::tokenizer().insert(ns, "rSpRule") == NS_m12::LN_rSpRule;
        flag&=QName::tokenizer().insert(ns, "rSp") == NS_m12::LN_rSp;
        flag&=QName::tokenizer().insert(ns, "eqArrPr") == NS_m12::LN_eqArrPr;
        flag&=QName::tokenizer().insert(ns, "type") == NS_m12::LN_type;
        flag&=QName::tokenizer().insert(ns, "numJc") == NS_m12::LN_numJc;
        flag&=QName::tokenizer().insert(ns, "denJc") == NS_m12::LN_denJc;
        flag&=QName::tokenizer().insert(ns, "fPr") == NS_m12::LN_fPr;
        flag&=QName::tokenizer().insert(ns, "num") == NS_m12::LN_num;
        flag&=QName::tokenizer().insert(ns, "den") == NS_m12::LN_den;
        flag&=QName::tokenizer().insert(ns, "fName") == NS_m12::LN_fName;
        flag&=QName::tokenizer().insert(ns, "groupChrPr") == NS_m12::LN_groupChrPr;
        flag&=QName::tokenizer().insert(ns, "lim") == NS_m12::LN_lim;
        flag&=QName::tokenizer().insert(ns, "count") == NS_m12::LN_count;
        flag&=QName::tokenizer().insert(ns, "mcJc") == NS_m12::LN_mcJc;
        flag&=QName::tokenizer().insert(ns, "mcPr") == NS_m12::LN_mcPr;
        flag&=QName::tokenizer().insert(ns, "mc") == NS_m12::LN_mc;
        flag&=QName::tokenizer().insert(ns, "plcHide") == NS_m12::LN_plcHide;
        flag&=QName::tokenizer().insert(ns, "cGpRule") == NS_m12::LN_cGpRule;
        flag&=QName::tokenizer().insert(ns, "cSp") == NS_m12::LN_cSp;
        flag&=QName::tokenizer().insert(ns, "cGp") == NS_m12::LN_cGp;
        flag&=QName::tokenizer().insert(ns, "mcs") == NS_m12::LN_mcs;
        flag&=QName::tokenizer().insert(ns, "mPr") == NS_m12::LN_mPr;
        flag&=QName::tokenizer().insert(ns, "mr") == NS_m12::LN_mr;
        flag&=QName::tokenizer().insert(ns, "limLoc") == NS_m12::LN_limLoc;
        flag&=QName::tokenizer().insert(ns, "subHide") == NS_m12::LN_subHide;
        flag&=QName::tokenizer().insert(ns, "supHide") == NS_m12::LN_supHide;
        flag&=QName::tokenizer().insert(ns, "naryPr") == NS_m12::LN_naryPr;
        flag&=QName::tokenizer().insert(ns, "sub") == NS_m12::LN_sub;
        flag&=QName::tokenizer().insert(ns, "sup") == NS_m12::LN_sup;
        flag&=QName::tokenizer().insert(ns, "show") == NS_m12::LN_show;
        flag&=QName::tokenizer().insert(ns, "zeroWid") == NS_m12::LN_zeroWid;
        flag&=QName::tokenizer().insert(ns, "zeroAsc") == NS_m12::LN_zeroAsc;
        flag&=QName::tokenizer().insert(ns, "zeroDesc") == NS_m12::LN_zeroDesc;
        flag&=QName::tokenizer().insert(ns, "transp") == NS_m12::LN_transp;
        flag&=QName::tokenizer().insert(ns, "phantPr") == NS_m12::LN_phantPr;
        flag&=QName::tokenizer().insert(ns, "degHide") == NS_m12::LN_degHide;
        flag&=QName::tokenizer().insert(ns, "radPr") == NS_m12::LN_radPr;
        flag&=QName::tokenizer().insert(ns, "deg") == NS_m12::LN_deg;
        flag&=QName::tokenizer().insert(ns, "acc") == NS_m12::LN_acc;
        flag&=QName::tokenizer().insert(ns, "bar") == NS_m12::LN_bar;
        flag&=QName::tokenizer().insert(ns, "box") == NS_m12::LN_box;
        flag&=QName::tokenizer().insert(ns, "boxForm") == NS_m12::LN_boxForm;
        flag&=QName::tokenizer().insert(ns, "d") == NS_m12::LN_d;
        flag&=QName::tokenizer().insert(ns, "eqArr") == NS_m12::LN_eqArr;
        flag&=QName::tokenizer().insert(ns, "f") == NS_m12::LN_f;
        flag&=QName::tokenizer().insert(ns, "func") == NS_m12::LN_func;
        flag&=QName::tokenizer().insert(ns, "groupChr") == NS_m12::LN_groupChr;
        flag&=QName::tokenizer().insert(ns, "limLow") == NS_m12::LN_limLow;
        flag&=QName::tokenizer().insert(ns, "limUpp") == NS_m12::LN_limUpp;
        flag&=QName::tokenizer().insert(ns, "m") == NS_m12::LN_m;
        flag&=QName::tokenizer().insert(ns, "nary") == NS_m12::LN_nary;
        flag&=QName::tokenizer().insert(ns, "phant") == NS_m12::LN_phant;
        flag&=QName::tokenizer().insert(ns, "rad") == NS_m12::LN_rad;
        flag&=QName::tokenizer().insert(ns, "sPre") == NS_m12::LN_sPre;
        flag&=QName::tokenizer().insert(ns, "sSub") == NS_m12::LN_sSub;
        flag&=QName::tokenizer().insert(ns, "sSubSup") == NS_m12::LN_sSubSup;
        flag&=QName::tokenizer().insert(ns, "sSup") == NS_m12::LN_sSup;
        flag&=QName::tokenizer().insert(ns, "r") == NS_m12::LN_r;
        flag&=QName::tokenizer().insert(ns, "argSz") == NS_m12::LN_argSz;
        flag&=QName::tokenizer().insert(ns, "argPr") == NS_m12::LN_argPr;
        flag&=QName::tokenizer().insert(ns, "jc") == NS_m12::LN_jc;
        flag&=QName::tokenizer().insert(ns, "mathFont") == NS_m12::LN_mathFont;
        flag&=QName::tokenizer().insert(ns, "defSz") == NS_m12::LN_defSz;
        flag&=QName::tokenizer().insert(ns, "brkBin") == NS_m12::LN_brkBin;
        flag&=QName::tokenizer().insert(ns, "brkBinSub") == NS_m12::LN_brkBinSub;
        flag&=QName::tokenizer().insert(ns, "smallFrac") == NS_m12::LN_smallFrac;
        flag&=QName::tokenizer().insert(ns, "dispDef") == NS_m12::LN_dispDef;
        flag&=QName::tokenizer().insert(ns, "lMargin") == NS_m12::LN_lMargin;
        flag&=QName::tokenizer().insert(ns, "rMargin") == NS_m12::LN_rMargin;
        flag&=QName::tokenizer().insert(ns, "defJc") == NS_m12::LN_defJc;
        flag&=QName::tokenizer().insert(ns, "preSp") == NS_m12::LN_preSp;
        flag&=QName::tokenizer().insert(ns, "postSp") == NS_m12::LN_postSp;
        flag&=QName::tokenizer().insert(ns, "intraSp") == NS_m12::LN_intraSp;
        flag&=QName::tokenizer().insert(ns, "interSp") == NS_m12::LN_interSp;
        flag&=QName::tokenizer().insert(ns, "wrapRight") == NS_m12::LN_wrapRight;
        flag&=QName::tokenizer().insert(ns, "wrapIndent") == NS_m12::LN_wrapIndent;
        flag&=QName::tokenizer().insert(ns, "naryGrow") == NS_m12::LN_naryGrow;
        flag&=QName::tokenizer().insert(ns, "intLim") == NS_m12::LN_intLim;
        flag&=QName::tokenizer().insert(ns, "naryLim") == NS_m12::LN_naryLim;
        flag&=QName::tokenizer().insert(ns, "mathPr") == NS_m12::LN_mathPr;
        flag&=QName::tokenizer().insert(ns, "oMathParaPr") == NS_m12::LN_oMathParaPr;
        flag&=QName::tokenizer().insert(ns, "oMath") == NS_m12::LN_oMath;
        flag&=QName::tokenizer().insert(ns, "oMathPara") == NS_m12::LN_oMathPara;
        CPPUNIT_ASSERT_MESSAGE("test for namespace m12 failed", flag);}
    void test_number(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0";
        flag&=QName::tokenizer().insert(ns, "number-style") == NS_number::LN_number_style;
        flag&=QName::tokenizer().insert(ns, "number") == NS_number::LN_number;
        flag&=QName::tokenizer().insert(ns, "decimal-replacement") == NS_number::LN_decimal_replacement;
        flag&=QName::tokenizer().insert(ns, "display-factor") == NS_number::LN_display_factor;
        flag&=QName::tokenizer().insert(ns, "embedded-text") == NS_number::LN_embedded_text;
        flag&=QName::tokenizer().insert(ns, "position") == NS_number::LN_position;
        flag&=QName::tokenizer().insert(ns, "scientific-number") == NS_number::LN_scientific_number;
        flag&=QName::tokenizer().insert(ns, "min-exponent-digits") == NS_number::LN_min_exponent_digits;
        flag&=QName::tokenizer().insert(ns, "fraction") == NS_number::LN_fraction;
        flag&=QName::tokenizer().insert(ns, "min-numerator-digits") == NS_number::LN_min_numerator_digits;
        flag&=QName::tokenizer().insert(ns, "min-denominator-digits") == NS_number::LN_min_denominator_digits;
        flag&=QName::tokenizer().insert(ns, "denominator-value") == NS_number::LN_denominator_value;
        flag&=QName::tokenizer().insert(ns, "currency-style") == NS_number::LN_currency_style;
        flag&=QName::tokenizer().insert(ns, "currency-symbol") == NS_number::LN_currency_symbol;
        flag&=QName::tokenizer().insert(ns, "language") == NS_number::LN_language;
        flag&=QName::tokenizer().insert(ns, "country") == NS_number::LN_country;
        flag&=QName::tokenizer().insert(ns, "percentage-style") == NS_number::LN_percentage_style;
        flag&=QName::tokenizer().insert(ns, "date-style") == NS_number::LN_date_style;
        flag&=QName::tokenizer().insert(ns, "day") == NS_number::LN_day;
        flag&=QName::tokenizer().insert(ns, "style") == NS_number::LN_style;
        flag&=QName::tokenizer().insert(ns, "month") == NS_number::LN_month;
        flag&=QName::tokenizer().insert(ns, "textual") == NS_number::LN_textual;
        flag&=QName::tokenizer().insert(ns, "possessive-form") == NS_number::LN_possessive_form;
        flag&=QName::tokenizer().insert(ns, "year") == NS_number::LN_year;
        flag&=QName::tokenizer().insert(ns, "era") == NS_number::LN_era;
        flag&=QName::tokenizer().insert(ns, "day-of-week") == NS_number::LN_day_of_week;
        flag&=QName::tokenizer().insert(ns, "week-of-year") == NS_number::LN_week_of_year;
        flag&=QName::tokenizer().insert(ns, "quarter") == NS_number::LN_quarter;
        flag&=QName::tokenizer().insert(ns, "time-style") == NS_number::LN_time_style;
        flag&=QName::tokenizer().insert(ns, "truncate-on-overflow") == NS_number::LN_truncate_on_overflow;
        flag&=QName::tokenizer().insert(ns, "hours") == NS_number::LN_hours;
        flag&=QName::tokenizer().insert(ns, "minutes") == NS_number::LN_minutes;
        flag&=QName::tokenizer().insert(ns, "seconds") == NS_number::LN_seconds;
        flag&=QName::tokenizer().insert(ns, "decimal-places") == NS_number::LN_decimal_places;
        flag&=QName::tokenizer().insert(ns, "am-pm") == NS_number::LN_am_pm;
        flag&=QName::tokenizer().insert(ns, "boolean-style") == NS_number::LN_boolean_style;
        flag&=QName::tokenizer().insert(ns, "boolean") == NS_number::LN_boolean;
        flag&=QName::tokenizer().insert(ns, "text-style") == NS_number::LN_text_style;
        flag&=QName::tokenizer().insert(ns, "text") == NS_number::LN_text;
        flag&=QName::tokenizer().insert(ns, "text-content") == NS_number::LN_text_content;
        flag&=QName::tokenizer().insert(ns, "title") == NS_number::LN_title;
        flag&=QName::tokenizer().insert(ns, "automatic-order") == NS_number::LN_automatic_order;
        flag&=QName::tokenizer().insert(ns, "format-source") == NS_number::LN_format_source;
        flag&=QName::tokenizer().insert(ns, "transliteration-format") == NS_number::LN_transliteration_format;
        flag&=QName::tokenizer().insert(ns, "transliteration-language") == NS_number::LN_transliteration_language;
        flag&=QName::tokenizer().insert(ns, "transliteration-country") == NS_number::LN_transliteration_country;
        flag&=QName::tokenizer().insert(ns, "transliteration-style") == NS_number::LN_transliteration_style;
        flag&=QName::tokenizer().insert(ns, "min-integer-digits") == NS_number::LN_min_integer_digits;
        flag&=QName::tokenizer().insert(ns, "grouping") == NS_number::LN_grouping;
        flag&=QName::tokenizer().insert(ns, "calendar") == NS_number::LN_calendar;
        CPPUNIT_ASSERT_MESSAGE("test for namespace number failed", flag);}
    void test_p12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/package/2005/06/content-types";
        flag&=QName::tokenizer().insert(ns, "Types") == NS_p12::LN_Types;
        flag&=QName::tokenizer().insert(ns, "Default") == NS_p12::LN_Default;
        flag&=QName::tokenizer().insert(ns, "Override") == NS_p12::LN_Override;
        CPPUNIT_ASSERT_MESSAGE("test for namespace p12 failed", flag);}
    void test_voldeps12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/voldeps";
        flag&=QName::tokenizer().insert(ns, "volTypes") == NS_voldeps12::LN_volTypes;
        flag&=QName::tokenizer().insert(ns, "volType") == NS_voldeps12::LN_volType;
        flag&=QName::tokenizer().insert(ns, "main") == NS_voldeps12::LN_main;
        flag&=QName::tokenizer().insert(ns, "topic") == NS_voldeps12::LN_topic;
        flag&=QName::tokenizer().insert(ns, "value") == NS_voldeps12::LN_value;
        flag&=QName::tokenizer().insert(ns, "subtopic") == NS_voldeps12::LN_subtopic;
        flag&=QName::tokenizer().insert(ns, "ref") == NS_voldeps12::LN_ref;
        CPPUNIT_ASSERT_MESSAGE("test for namespace voldeps12 failed", flag);}
    void test_pt12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/pivot";
        flag&=QName::tokenizer().insert(ns, "pivotCacheDefinition") == NS_pt12::LN_pivotCacheDefinition;
        flag&=QName::tokenizer().insert(ns, "pivotCacheRecords") == NS_pt12::LN_pivotCacheRecords;
        flag&=QName::tokenizer().insert(ns, "pivotTableDefinition") == NS_pt12::LN_pivotTableDefinition;
        flag&=QName::tokenizer().insert(ns, "cacheSource") == NS_pt12::LN_cacheSource;
        flag&=QName::tokenizer().insert(ns, "fields") == NS_pt12::LN_fields;
        flag&=QName::tokenizer().insert(ns, "hierarchies") == NS_pt12::LN_hierarchies;
        flag&=QName::tokenizer().insert(ns, "kpis") == NS_pt12::LN_kpis;
        flag&=QName::tokenizer().insert(ns, "sdTupleCache") == NS_pt12::LN_sdTupleCache;
        flag&=QName::tokenizer().insert(ns, "ciFmlas") == NS_pt12::LN_ciFmlas;
        flag&=QName::tokenizer().insert(ns, "calcMbrs") == NS_pt12::LN_calcMbrs;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_pt12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "field") == NS_pt12::LN_field;
        flag&=QName::tokenizer().insert(ns, "shItems") == NS_pt12::LN_shItems;
        flag&=QName::tokenizer().insert(ns, "group") == NS_pt12::LN_group;
        flag&=QName::tokenizer().insert(ns, "wksht") == NS_pt12::LN_wksht;
        flag&=QName::tokenizer().insert(ns, "consol") == NS_pt12::LN_consol;
        flag&=QName::tokenizer().insert(ns, "pages") == NS_pt12::LN_pages;
        flag&=QName::tokenizer().insert(ns, "rngSets") == NS_pt12::LN_rngSets;
        flag&=QName::tokenizer().insert(ns, "page") == NS_pt12::LN_page;
        flag&=QName::tokenizer().insert(ns, "item") == NS_pt12::LN_item;
        flag&=QName::tokenizer().insert(ns, "rngSet") == NS_pt12::LN_rngSet;
        flag&=QName::tokenizer().insert(ns, "m") == NS_pt12::LN_m;
        flag&=QName::tokenizer().insert(ns, "n") == NS_pt12::LN_n;
        flag&=QName::tokenizer().insert(ns, "b") == NS_pt12::LN_b;
        flag&=QName::tokenizer().insert(ns, "e") == NS_pt12::LN_e;
        flag&=QName::tokenizer().insert(ns, "s") == NS_pt12::LN_s;
        flag&=QName::tokenizer().insert(ns, "d") == NS_pt12::LN_d;
        flag&=QName::tokenizer().insert(ns, "mbrs") == NS_pt12::LN_mbrs;
        flag&=QName::tokenizer().insert(ns, "mpm") == NS_pt12::LN_mpm;
        flag&=QName::tokenizer().insert(ns, "mpn") == NS_pt12::LN_mpn;
        flag&=QName::tokenizer().insert(ns, "mpb") == NS_pt12::LN_mpb;
        flag&=QName::tokenizer().insert(ns, "mps") == NS_pt12::LN_mps;
        flag&=QName::tokenizer().insert(ns, "mpd") == NS_pt12::LN_mpd;
        flag&=QName::tokenizer().insert(ns, "rangeDesc") == NS_pt12::LN_rangeDesc;
        flag&=QName::tokenizer().insert(ns, "discMapping") == NS_pt12::LN_discMapping;
        flag&=QName::tokenizer().insert(ns, "grItems") == NS_pt12::LN_grItems;
        flag&=QName::tokenizer().insert(ns, "x") == NS_pt12::LN_x;
        flag&=QName::tokenizer().insert(ns, "r") == NS_pt12::LN_r;
        flag&=QName::tokenizer().insert(ns, "kpi") == NS_pt12::LN_kpi;
        flag&=QName::tokenizer().insert(ns, "hierarchy") == NS_pt12::LN_hierarchy;
        flag&=QName::tokenizer().insert(ns, "flds") == NS_pt12::LN_flds;
        flag&=QName::tokenizer().insert(ns, "grLvls") == NS_pt12::LN_grLvls;
        flag&=QName::tokenizer().insert(ns, "fld") == NS_pt12::LN_fld;
        flag&=QName::tokenizer().insert(ns, "grLvl") == NS_pt12::LN_grLvl;
        flag&=QName::tokenizer().insert(ns, "grps") == NS_pt12::LN_grps;
        flag&=QName::tokenizer().insert(ns, "grp") == NS_pt12::LN_grp;
        flag&=QName::tokenizer().insert(ns, "grMbrs") == NS_pt12::LN_grMbrs;
        flag&=QName::tokenizer().insert(ns, "grMbr") == NS_pt12::LN_grMbr;
        flag&=QName::tokenizer().insert(ns, "entries") == NS_pt12::LN_entries;
        flag&=QName::tokenizer().insert(ns, "sets") == NS_pt12::LN_sets;
        flag&=QName::tokenizer().insert(ns, "qryCache") == NS_pt12::LN_qryCache;
        flag&=QName::tokenizer().insert(ns, "mbr") == NS_pt12::LN_mbr;
        flag&=QName::tokenizer().insert(ns, "set") == NS_pt12::LN_set;
        flag&=QName::tokenizer().insert(ns, "tuples") == NS_pt12::LN_tuples;
        flag&=QName::tokenizer().insert(ns, "sortByTuple") == NS_pt12::LN_sortByTuple;
        flag&=QName::tokenizer().insert(ns, "qry") == NS_pt12::LN_qry;
        flag&=QName::tokenizer().insert(ns, "ciFmla") == NS_pt12::LN_ciFmla;
        flag&=QName::tokenizer().insert(ns, "rule") == NS_pt12::LN_rule;
        flag&=QName::tokenizer().insert(ns, "calcMbr") == NS_pt12::LN_calcMbr;
        flag&=QName::tokenizer().insert(ns, "filters") == NS_pt12::LN_filters;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_pt12::LN_filter;
        flag&=QName::tokenizer().insert(ns, "location") == NS_pt12::LN_location;
        flag&=QName::tokenizer().insert(ns, "rwFlds") == NS_pt12::LN_rwFlds;
        flag&=QName::tokenizer().insert(ns, "rwItms") == NS_pt12::LN_rwItms;
        flag&=QName::tokenizer().insert(ns, "rwDrills") == NS_pt12::LN_rwDrills;
        flag&=QName::tokenizer().insert(ns, "colFlds") == NS_pt12::LN_colFlds;
        flag&=QName::tokenizer().insert(ns, "colItms") == NS_pt12::LN_colItms;
        flag&=QName::tokenizer().insert(ns, "colDrills") == NS_pt12::LN_colDrills;
        flag&=QName::tokenizer().insert(ns, "pgFields") == NS_pt12::LN_pgFields;
        flag&=QName::tokenizer().insert(ns, "datFields") == NS_pt12::LN_datFields;
        flag&=QName::tokenizer().insert(ns, "fmts") == NS_pt12::LN_fmts;
        flag&=QName::tokenizer().insert(ns, "condfmts") == NS_pt12::LN_condfmts;
        flag&=QName::tokenizer().insert(ns, "crtfmts") == NS_pt12::LN_crtfmts;
        flag&=QName::tokenizer().insert(ns, "drillSaved") == NS_pt12::LN_drillSaved;
        flag&=QName::tokenizer().insert(ns, "strSelects") == NS_pt12::LN_strSelects;
        flag&=QName::tokenizer().insert(ns, "dims") == NS_pt12::LN_dims;
        flag&=QName::tokenizer().insert(ns, "mgs") == NS_pt12::LN_mgs;
        flag&=QName::tokenizer().insert(ns, "tableStyle") == NS_pt12::LN_tableStyle;
        flag&=QName::tokenizer().insert(ns, "sxfilters") == NS_pt12::LN_sxfilters;
        flag&=QName::tokenizer().insert(ns, "items") == NS_pt12::LN_items;
        flag&=QName::tokenizer().insert(ns, "cdls") == NS_pt12::LN_cdls;
        flag&=QName::tokenizer().insert(ns, "autoSortScope") == NS_pt12::LN_autoSortScope;
        flag&=QName::tokenizer().insert(ns, "cdl") == NS_pt12::LN_cdl;
        flag&=QName::tokenizer().insert(ns, "pgField") == NS_pt12::LN_pgField;
        flag&=QName::tokenizer().insert(ns, "datField") == NS_pt12::LN_datField;
        flag&=QName::tokenizer().insert(ns, "i") == NS_pt12::LN_i;
        flag&=QName::tokenizer().insert(ns, "fmt") == NS_pt12::LN_fmt;
        flag&=QName::tokenizer().insert(ns, "cfmt") == NS_pt12::LN_cfmt;
        flag&=QName::tokenizer().insert(ns, "crtfmt") == NS_pt12::LN_crtfmt;
        flag&=QName::tokenizer().insert(ns, "drill") == NS_pt12::LN_drill;
        flag&=QName::tokenizer().insert(ns, "strSelect") == NS_pt12::LN_strSelect;
        flag&=QName::tokenizer().insert(ns, "mbrProps") == NS_pt12::LN_mbrProps;
        flag&=QName::tokenizer().insert(ns, "itms") == NS_pt12::LN_itms;
        flag&=QName::tokenizer().insert(ns, "selctPg") == NS_pt12::LN_selctPg;
        flag&=QName::tokenizer().insert(ns, "mp") == NS_pt12::LN_mp;
        flag&=QName::tokenizer().insert(ns, "dim") == NS_pt12::LN_dim;
        flag&=QName::tokenizer().insert(ns, "mg") == NS_pt12::LN_mg;
        flag&=QName::tokenizer().insert(ns, "map") == NS_pt12::LN_map;
        flag&=QName::tokenizer().insert(ns, "sxfilter") == NS_pt12::LN_sxfilter;
        flag&=QName::tokenizer().insert(ns, "autofilter") == NS_pt12::LN_autofilter;
        CPPUNIT_ASSERT_MESSAGE("test for namespace pt12 failed", flag);}
    void test_xml(){
        bool flag=true;
        static const char ns[] = "http://www.w3.org/XML/1998/namespace";
        flag&=QName::tokenizer().insert(ns, "lang") == NS_xml::LN_lang;
        flag&=QName::tokenizer().insert(ns, "base") == NS_xml::LN_base;
        CPPUNIT_ASSERT_MESSAGE("test for namespace xml failed", flag);}
    void test_w11(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/word/2003/wordml";
        flag&=QName::tokenizer().insert(ns, "ilvl") == NS_w11::LN_ilvl;
        flag&=QName::tokenizer().insert(ns, "ilfo") == NS_w11::LN_ilfo;
        flag&=QName::tokenizer().insert(ns, "top") == NS_w11::LN_top;
        flag&=QName::tokenizer().insert(ns, "left") == NS_w11::LN_left;
        flag&=QName::tokenizer().insert(ns, "bottom") == NS_w11::LN_bottom;
        flag&=QName::tokenizer().insert(ns, "right") == NS_w11::LN_right;
        flag&=QName::tokenizer().insert(ns, "between") == NS_w11::LN_between;
        flag&=QName::tokenizer().insert(ns, "bar") == NS_w11::LN_bar;
        flag&=QName::tokenizer().insert(ns, "tab") == NS_w11::LN_tab;
        flag&=QName::tokenizer().insert(ns, "pStyle") == NS_w11::LN_pStyle;
        flag&=QName::tokenizer().insert(ns, "keepNext") == NS_w11::LN_keepNext;
        flag&=QName::tokenizer().insert(ns, "keepLines") == NS_w11::LN_keepLines;
        flag&=QName::tokenizer().insert(ns, "pageBreakBefore") == NS_w11::LN_pageBreakBefore;
        flag&=QName::tokenizer().insert(ns, "framePr") == NS_w11::LN_framePr;
        flag&=QName::tokenizer().insert(ns, "widowControl") == NS_w11::LN_widowControl;
        flag&=QName::tokenizer().insert(ns, "listPr") == NS_w11::LN_listPr;
        flag&=QName::tokenizer().insert(ns, "supressLineNumbers") == NS_w11::LN_supressLineNumbers;
        flag&=QName::tokenizer().insert(ns, "pBdr") == NS_w11::LN_pBdr;
        flag&=QName::tokenizer().insert(ns, "shd") == NS_w11::LN_shd;
        flag&=QName::tokenizer().insert(ns, "tabs") == NS_w11::LN_tabs;
        flag&=QName::tokenizer().insert(ns, "suppressAutoHyphens") == NS_w11::LN_suppressAutoHyphens;
        flag&=QName::tokenizer().insert(ns, "kinsoku") == NS_w11::LN_kinsoku;
        flag&=QName::tokenizer().insert(ns, "wordWrap") == NS_w11::LN_wordWrap;
        flag&=QName::tokenizer().insert(ns, "overflowPunct") == NS_w11::LN_overflowPunct;
        flag&=QName::tokenizer().insert(ns, "topLinePunct") == NS_w11::LN_topLinePunct;
        flag&=QName::tokenizer().insert(ns, "autoSpaceDE") == NS_w11::LN_autoSpaceDE;
        flag&=QName::tokenizer().insert(ns, "autoSpaceDN") == NS_w11::LN_autoSpaceDN;
        flag&=QName::tokenizer().insert(ns, "bidi") == NS_w11::LN_bidi;
        flag&=QName::tokenizer().insert(ns, "adjustRightInd") == NS_w11::LN_adjustRightInd;
        flag&=QName::tokenizer().insert(ns, "snapToGrid") == NS_w11::LN_snapToGrid;
        flag&=QName::tokenizer().insert(ns, "spacing") == NS_w11::LN_spacing;
        flag&=QName::tokenizer().insert(ns, "ind") == NS_w11::LN_ind;
        flag&=QName::tokenizer().insert(ns, "contextualSpacing") == NS_w11::LN_contextualSpacing;
        flag&=QName::tokenizer().insert(ns, "suppressOverlap") == NS_w11::LN_suppressOverlap;
        flag&=QName::tokenizer().insert(ns, "jc") == NS_w11::LN_jc;
        flag&=QName::tokenizer().insert(ns, "textDirection") == NS_w11::LN_textDirection;
        flag&=QName::tokenizer().insert(ns, "textAlignment") == NS_w11::LN_textAlignment;
        flag&=QName::tokenizer().insert(ns, "outlineLvl") == NS_w11::LN_outlineLvl;
        flag&=QName::tokenizer().insert(ns, "divId") == NS_w11::LN_divId;
        flag&=QName::tokenizer().insert(ns, "cnfStyle") == NS_w11::LN_cnfStyle;
        flag&=QName::tokenizer().insert(ns, "rPr") == NS_w11::LN_rPr;
        flag&=QName::tokenizer().insert(ns, "sectPr") == NS_w11::LN_sectPr;
        flag&=QName::tokenizer().insert(ns, "binData") == NS_w11::LN_binData;
        flag&=QName::tokenizer().insert(ns, "movie") == NS_w11::LN_movie;
        flag&=QName::tokenizer().insert(ns, "background") == NS_w11::LN_background;
        flag&=QName::tokenizer().insert(ns, "applet") == NS_w11::LN_applet;
        flag&=QName::tokenizer().insert(ns, "msAccessHTML") == NS_w11::LN_msAccessHTML;
        flag&=QName::tokenizer().insert(ns, "scriptAnchor") == NS_w11::LN_scriptAnchor;
        flag&=QName::tokenizer().insert(ns, "ocx") == NS_w11::LN_ocx;
        flag&=QName::tokenizer().insert(ns, "args") == NS_w11::LN_args;
        flag&=QName::tokenizer().insert(ns, "appletText") == NS_w11::LN_appletText;
        flag&=QName::tokenizer().insert(ns, "scriptId") == NS_w11::LN_scriptId;
        flag&=QName::tokenizer().insert(ns, "language") == NS_w11::LN_language;
        flag&=QName::tokenizer().insert(ns, "scriptText") == NS_w11::LN_scriptText;
        flag&=QName::tokenizer().insert(ns, "fldData") == NS_w11::LN_fldData;
        flag&=QName::tokenizer().insert(ns, "r") == NS_w11::LN_r;
        flag&=QName::tokenizer().insert(ns, "fldSimple") == NS_w11::LN_fldSimple;
        flag&=QName::tokenizer().insert(ns, "hlink") == NS_w11::LN_hlink;
        flag&=QName::tokenizer().insert(ns, "col") == NS_w11::LN_col;
        flag&=QName::tokenizer().insert(ns, "hdr") == NS_w11::LN_hdr;
        flag&=QName::tokenizer().insert(ns, "ftr") == NS_w11::LN_ftr;
        flag&=QName::tokenizer().insert(ns, "footnotePr") == NS_w11::LN_footnotePr;
        flag&=QName::tokenizer().insert(ns, "endnotePr") == NS_w11::LN_endnotePr;
        flag&=QName::tokenizer().insert(ns, "type") == NS_w11::LN_type;
        flag&=QName::tokenizer().insert(ns, "pgSz") == NS_w11::LN_pgSz;
        flag&=QName::tokenizer().insert(ns, "pgMar") == NS_w11::LN_pgMar;
        flag&=QName::tokenizer().insert(ns, "paperSrc") == NS_w11::LN_paperSrc;
        flag&=QName::tokenizer().insert(ns, "pgBorders") == NS_w11::LN_pgBorders;
        flag&=QName::tokenizer().insert(ns, "lnNumType") == NS_w11::LN_lnNumType;
        flag&=QName::tokenizer().insert(ns, "pgNumType") == NS_w11::LN_pgNumType;
        flag&=QName::tokenizer().insert(ns, "cols") == NS_w11::LN_cols;
        flag&=QName::tokenizer().insert(ns, "formProt") == NS_w11::LN_formProt;
        flag&=QName::tokenizer().insert(ns, "vAlign") == NS_w11::LN_vAlign;
        flag&=QName::tokenizer().insert(ns, "noEndnote") == NS_w11::LN_noEndnote;
        flag&=QName::tokenizer().insert(ns, "titlePg") == NS_w11::LN_titlePg;
        flag&=QName::tokenizer().insert(ns, "textFlow") == NS_w11::LN_textFlow;
        flag&=QName::tokenizer().insert(ns, "rtlGutter") == NS_w11::LN_rtlGutter;
        flag&=QName::tokenizer().insert(ns, "docGrid") == NS_w11::LN_docGrid;
        flag&=QName::tokenizer().insert(ns, "br") == NS_w11::LN_br;
        flag&=QName::tokenizer().insert(ns, "t") == NS_w11::LN_t;
        flag&=QName::tokenizer().insert(ns, "delText") == NS_w11::LN_delText;
        flag&=QName::tokenizer().insert(ns, "instrText") == NS_w11::LN_instrText;
        flag&=QName::tokenizer().insert(ns, "delInstrText") == NS_w11::LN_delInstrText;
        flag&=QName::tokenizer().insert(ns, "noBreakHyphen") == NS_w11::LN_noBreakHyphen;
        flag&=QName::tokenizer().insert(ns, "softHyphen") == NS_w11::LN_softHyphen;
        flag&=QName::tokenizer().insert(ns, "annotationRef") == NS_w11::LN_annotationRef;
        flag&=QName::tokenizer().insert(ns, "footnoteRef") == NS_w11::LN_footnoteRef;
        flag&=QName::tokenizer().insert(ns, "endnoteRef") == NS_w11::LN_endnoteRef;
        flag&=QName::tokenizer().insert(ns, "separator") == NS_w11::LN_separator;
        flag&=QName::tokenizer().insert(ns, "continuationSeparator") == NS_w11::LN_continuationSeparator;
        flag&=QName::tokenizer().insert(ns, "footnote") == NS_w11::LN_footnote;
        flag&=QName::tokenizer().insert(ns, "endnote") == NS_w11::LN_endnote;
        flag&=QName::tokenizer().insert(ns, "sym") == NS_w11::LN_sym;
        flag&=QName::tokenizer().insert(ns, "pgNum") == NS_w11::LN_pgNum;
        flag&=QName::tokenizer().insert(ns, "cr") == NS_w11::LN_cr;
        flag&=QName::tokenizer().insert(ns, "pict") == NS_w11::LN_pict;
        flag&=QName::tokenizer().insert(ns, "fldChar") == NS_w11::LN_fldChar;
        flag&=QName::tokenizer().insert(ns, "ruby") == NS_w11::LN_ruby;
        flag&=QName::tokenizer().insert(ns, "rStyle") == NS_w11::LN_rStyle;
        flag&=QName::tokenizer().insert(ns, "rFonts") == NS_w11::LN_rFonts;
        flag&=QName::tokenizer().insert(ns, "b") == NS_w11::LN_b;
        flag&=QName::tokenizer().insert(ns, "b-cs") == NS_w11::LN_b_cs;
        flag&=QName::tokenizer().insert(ns, "i") == NS_w11::LN_i;
        flag&=QName::tokenizer().insert(ns, "i-cs") == NS_w11::LN_i_cs;
        flag&=QName::tokenizer().insert(ns, "caps") == NS_w11::LN_caps;
        flag&=QName::tokenizer().insert(ns, "smallCaps") == NS_w11::LN_smallCaps;
        flag&=QName::tokenizer().insert(ns, "strike") == NS_w11::LN_strike;
        flag&=QName::tokenizer().insert(ns, "dstrike") == NS_w11::LN_dstrike;
        flag&=QName::tokenizer().insert(ns, "outline") == NS_w11::LN_outline;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_w11::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "emboss") == NS_w11::LN_emboss;
        flag&=QName::tokenizer().insert(ns, "imprint") == NS_w11::LN_imprint;
        flag&=QName::tokenizer().insert(ns, "noProof") == NS_w11::LN_noProof;
        flag&=QName::tokenizer().insert(ns, "vanish") == NS_w11::LN_vanish;
        flag&=QName::tokenizer().insert(ns, "webHidden") == NS_w11::LN_webHidden;
        flag&=QName::tokenizer().insert(ns, "color") == NS_w11::LN_color;
        flag&=QName::tokenizer().insert(ns, "w") == NS_w11::LN_w;
        flag&=QName::tokenizer().insert(ns, "kern") == NS_w11::LN_kern;
        flag&=QName::tokenizer().insert(ns, "position") == NS_w11::LN_position;
        flag&=QName::tokenizer().insert(ns, "sz") == NS_w11::LN_sz;
        flag&=QName::tokenizer().insert(ns, "sz-cs") == NS_w11::LN_sz_cs;
        flag&=QName::tokenizer().insert(ns, "highlight") == NS_w11::LN_highlight;
        flag&=QName::tokenizer().insert(ns, "u") == NS_w11::LN_u;
        flag&=QName::tokenizer().insert(ns, "effect") == NS_w11::LN_effect;
        flag&=QName::tokenizer().insert(ns, "bdr") == NS_w11::LN_bdr;
        flag&=QName::tokenizer().insert(ns, "fitText") == NS_w11::LN_fitText;
        flag&=QName::tokenizer().insert(ns, "vertAlign") == NS_w11::LN_vertAlign;
        flag&=QName::tokenizer().insert(ns, "rtl") == NS_w11::LN_rtl;
        flag&=QName::tokenizer().insert(ns, "cs") == NS_w11::LN_cs;
        flag&=QName::tokenizer().insert(ns, "em") == NS_w11::LN_em;
        flag&=QName::tokenizer().insert(ns, "hyphen") == NS_w11::LN_hyphen;
        flag&=QName::tokenizer().insert(ns, "lang") == NS_w11::LN_lang;
        flag&=QName::tokenizer().insert(ns, "asianLayout") == NS_w11::LN_asianLayout;
        flag&=QName::tokenizer().insert(ns, "specVanish") == NS_w11::LN_specVanish;
        flag&=QName::tokenizer().insert(ns, "lists") == NS_w11::LN_lists;
        flag&=QName::tokenizer().insert(ns, "styles") == NS_w11::LN_styles;
        flag&=QName::tokenizer().insert(ns, "fonts") == NS_w11::LN_fonts;
        flag&=QName::tokenizer().insert(ns, "rubyAlign") == NS_w11::LN_rubyAlign;
        flag&=QName::tokenizer().insert(ns, "hps") == NS_w11::LN_hps;
        flag&=QName::tokenizer().insert(ns, "hpsRaise") == NS_w11::LN_hpsRaise;
        flag&=QName::tokenizer().insert(ns, "hpsBaseText") == NS_w11::LN_hpsBaseText;
        flag&=QName::tokenizer().insert(ns, "lid") == NS_w11::LN_lid;
        flag&=QName::tokenizer().insert(ns, "rubyPr") == NS_w11::LN_rubyPr;
        flag&=QName::tokenizer().insert(ns, "rt") == NS_w11::LN_rt;
        flag&=QName::tokenizer().insert(ns, "rubyBase") == NS_w11::LN_rubyBase;
        flag&=QName::tokenizer().insert(ns, "pPr") == NS_w11::LN_pPr;
        flag&=QName::tokenizer().insert(ns, "subDoc") == NS_w11::LN_subDoc;
        flag&=QName::tokenizer().insert(ns, "gridCol") == NS_w11::LN_gridCol;
        flag&=QName::tokenizer().insert(ns, "insideH") == NS_w11::LN_insideH;
        flag&=QName::tokenizer().insert(ns, "insideV") == NS_w11::LN_insideV;
        flag&=QName::tokenizer().insert(ns, "tl2br") == NS_w11::LN_tl2br;
        flag&=QName::tokenizer().insert(ns, "tr2bl") == NS_w11::LN_tr2bl;
        flag&=QName::tokenizer().insert(ns, "tcW") == NS_w11::LN_tcW;
        flag&=QName::tokenizer().insert(ns, "gridSpan") == NS_w11::LN_gridSpan;
        flag&=QName::tokenizer().insert(ns, "hmerge") == NS_w11::LN_hmerge;
        flag&=QName::tokenizer().insert(ns, "vmerge") == NS_w11::LN_vmerge;
        flag&=QName::tokenizer().insert(ns, "tcBorders") == NS_w11::LN_tcBorders;
        flag&=QName::tokenizer().insert(ns, "noWrap") == NS_w11::LN_noWrap;
        flag&=QName::tokenizer().insert(ns, "tcMar") == NS_w11::LN_tcMar;
        flag&=QName::tokenizer().insert(ns, "tcFitText") == NS_w11::LN_tcFitText;
        flag&=QName::tokenizer().insert(ns, "tcPr") == NS_w11::LN_tcPr;
        flag&=QName::tokenizer().insert(ns, "gridBefore") == NS_w11::LN_gridBefore;
        flag&=QName::tokenizer().insert(ns, "gridAfter") == NS_w11::LN_gridAfter;
        flag&=QName::tokenizer().insert(ns, "wBefore") == NS_w11::LN_wBefore;
        flag&=QName::tokenizer().insert(ns, "wAfter") == NS_w11::LN_wAfter;
        flag&=QName::tokenizer().insert(ns, "cantSplit") == NS_w11::LN_cantSplit;
        flag&=QName::tokenizer().insert(ns, "trHeight") == NS_w11::LN_trHeight;
        flag&=QName::tokenizer().insert(ns, "tblHeader") == NS_w11::LN_tblHeader;
        flag&=QName::tokenizer().insert(ns, "tblCellSpacing") == NS_w11::LN_tblCellSpacing;
        flag&=QName::tokenizer().insert(ns, "tblPrEx") == NS_w11::LN_tblPrEx;
        flag&=QName::tokenizer().insert(ns, "trPr") == NS_w11::LN_trPr;
        flag&=QName::tokenizer().insert(ns, "tc") == NS_w11::LN_tc;
        flag&=QName::tokenizer().insert(ns, "tblStyle") == NS_w11::LN_tblStyle;
        flag&=QName::tokenizer().insert(ns, "tblpPr") == NS_w11::LN_tblpPr;
        flag&=QName::tokenizer().insert(ns, "tblOverlap") == NS_w11::LN_tblOverlap;
        flag&=QName::tokenizer().insert(ns, "tblRtl") == NS_w11::LN_tblRtl;
        flag&=QName::tokenizer().insert(ns, "bidiVisual") == NS_w11::LN_bidiVisual;
        flag&=QName::tokenizer().insert(ns, "tblStyleRowBandSize") == NS_w11::LN_tblStyleRowBandSize;
        flag&=QName::tokenizer().insert(ns, "tblStyleColBandSize") == NS_w11::LN_tblStyleColBandSize;
        flag&=QName::tokenizer().insert(ns, "tblW") == NS_w11::LN_tblW;
        flag&=QName::tokenizer().insert(ns, "tblInd") == NS_w11::LN_tblInd;
        flag&=QName::tokenizer().insert(ns, "tblBorders") == NS_w11::LN_tblBorders;
        flag&=QName::tokenizer().insert(ns, "tblLayout") == NS_w11::LN_tblLayout;
        flag&=QName::tokenizer().insert(ns, "tblCellMar") == NS_w11::LN_tblCellMar;
        flag&=QName::tokenizer().insert(ns, "tblLook") == NS_w11::LN_tblLook;
        flag&=QName::tokenizer().insert(ns, "tblPr") == NS_w11::LN_tblPr;
        flag&=QName::tokenizer().insert(ns, "tblGrid") == NS_w11::LN_tblGrid;
        flag&=QName::tokenizer().insert(ns, "tr") == NS_w11::LN_tr;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_w11::LN_pos;
        flag&=QName::tokenizer().insert(ns, "numFmt") == NS_w11::LN_numFmt;
        flag&=QName::tokenizer().insert(ns, "numStart") == NS_w11::LN_numStart;
        flag&=QName::tokenizer().insert(ns, "numRestart") == NS_w11::LN_numRestart;
        flag&=QName::tokenizer().insert(ns, "active") == NS_w11::LN_active;
        flag&=QName::tokenizer().insert(ns, "hash") == NS_w11::LN_hash;
        flag&=QName::tokenizer().insert(ns, "column") == NS_w11::LN_column;
        flag&=QName::tokenizer().insert(ns, "uniqueTag") == NS_w11::LN_uniqueTag;
        flag&=QName::tokenizer().insert(ns, "name") == NS_w11::LN_name;
        flag&=QName::tokenizer().insert(ns, "mappedName") == NS_w11::LN_mappedName;
        flag&=QName::tokenizer().insert(ns, "dynamicAddress") == NS_w11::LN_dynamicAddress;
        flag&=QName::tokenizer().insert(ns, "udl") == NS_w11::LN_udl;
        flag&=QName::tokenizer().insert(ns, "table") == NS_w11::LN_table;
        flag&=QName::tokenizer().insert(ns, "src") == NS_w11::LN_src;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_w11::LN_filter;
        flag&=QName::tokenizer().insert(ns, "sort") == NS_w11::LN_sort;
        flag&=QName::tokenizer().insert(ns, "fieldMapData") == NS_w11::LN_fieldMapData;
        flag&=QName::tokenizer().insert(ns, "colDelim") == NS_w11::LN_colDelim;
        flag&=QName::tokenizer().insert(ns, "jdsoType") == NS_w11::LN_jdsoType;
        flag&=QName::tokenizer().insert(ns, "fHdr") == NS_w11::LN_fHdr;
        flag&=QName::tokenizer().insert(ns, "recipientData") == NS_w11::LN_recipientData;
        flag&=QName::tokenizer().insert(ns, "mainDocumentType") == NS_w11::LN_mainDocumentType;
        flag&=QName::tokenizer().insert(ns, "linkToQuery") == NS_w11::LN_linkToQuery;
        flag&=QName::tokenizer().insert(ns, "dataType") == NS_w11::LN_dataType;
        flag&=QName::tokenizer().insert(ns, "defaultSQL") == NS_w11::LN_defaultSQL;
        flag&=QName::tokenizer().insert(ns, "connectString") == NS_w11::LN_connectString;
        flag&=QName::tokenizer().insert(ns, "query") == NS_w11::LN_query;
        flag&=QName::tokenizer().insert(ns, "dataSource") == NS_w11::LN_dataSource;
        flag&=QName::tokenizer().insert(ns, "headerSource") == NS_w11::LN_headerSource;
        flag&=QName::tokenizer().insert(ns, "doNotSuppressBlankLines") == NS_w11::LN_doNotSuppressBlankLines;
        flag&=QName::tokenizer().insert(ns, "destination") == NS_w11::LN_destination;
        flag&=QName::tokenizer().insert(ns, "addressFieldName") == NS_w11::LN_addressFieldName;
        flag&=QName::tokenizer().insert(ns, "mailSubject") == NS_w11::LN_mailSubject;
        flag&=QName::tokenizer().insert(ns, "mailAsAttachment") == NS_w11::LN_mailAsAttachment;
        flag&=QName::tokenizer().insert(ns, "viewMergedData") == NS_w11::LN_viewMergedData;
        flag&=QName::tokenizer().insert(ns, "activeRecord") == NS_w11::LN_activeRecord;
        flag&=QName::tokenizer().insert(ns, "checkErrors") == NS_w11::LN_checkErrors;
        flag&=QName::tokenizer().insert(ns, "odso") == NS_w11::LN_odso;
        flag&=QName::tokenizer().insert(ns, "origWordTableRules") == NS_w11::LN_origWordTableRules;
        flag&=QName::tokenizer().insert(ns, "wpJustification") == NS_w11::LN_wpJustification;
        flag&=QName::tokenizer().insert(ns, "noTabHangInd") == NS_w11::LN_noTabHangInd;
        flag&=QName::tokenizer().insert(ns, "noLeading") == NS_w11::LN_noLeading;
        flag&=QName::tokenizer().insert(ns, "spaceForUL") == NS_w11::LN_spaceForUL;
        flag&=QName::tokenizer().insert(ns, "noColumnBalance") == NS_w11::LN_noColumnBalance;
        flag&=QName::tokenizer().insert(ns, "balanceSingleByteDoubleByteWidth") == NS_w11::LN_balanceSingleByteDoubleByteWidth;
        flag&=QName::tokenizer().insert(ns, "transparentMetafiles") == NS_w11::LN_transparentMetafiles;
        flag&=QName::tokenizer().insert(ns, "noExtraLineSpacing") == NS_w11::LN_noExtraLineSpacing;
        flag&=QName::tokenizer().insert(ns, "doNotLeaveBackslashAlone") == NS_w11::LN_doNotLeaveBackslashAlone;
        flag&=QName::tokenizer().insert(ns, "ulTrailSpace") == NS_w11::LN_ulTrailSpace;
        flag&=QName::tokenizer().insert(ns, "doNotExpandShiftReturn") == NS_w11::LN_doNotExpandShiftReturn;
        flag&=QName::tokenizer().insert(ns, "spacingInWholePoints") == NS_w11::LN_spacingInWholePoints;
        flag&=QName::tokenizer().insert(ns, "lineWrapLikeWord6") == NS_w11::LN_lineWrapLikeWord6;
        flag&=QName::tokenizer().insert(ns, "printBodyTextBeforeHeader") == NS_w11::LN_printBodyTextBeforeHeader;
        flag&=QName::tokenizer().insert(ns, "printColBlack") == NS_w11::LN_printColBlack;
        flag&=QName::tokenizer().insert(ns, "wpSpaceWidth") == NS_w11::LN_wpSpaceWidth;
        flag&=QName::tokenizer().insert(ns, "showBreaksInFrames") == NS_w11::LN_showBreaksInFrames;
        flag&=QName::tokenizer().insert(ns, "subFontBySize") == NS_w11::LN_subFontBySize;
        flag&=QName::tokenizer().insert(ns, "suppressBottomSpacing") == NS_w11::LN_suppressBottomSpacing;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacing") == NS_w11::LN_suppressTopSpacing;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacingMac5") == NS_w11::LN_suppressTopSpacingMac5;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacingWP") == NS_w11::LN_suppressTopSpacingWP;
        flag&=QName::tokenizer().insert(ns, "suppressSpBfAfterPgBrk") == NS_w11::LN_suppressSpBfAfterPgBrk;
        flag&=QName::tokenizer().insert(ns, "swapBordersFacingPages") == NS_w11::LN_swapBordersFacingPages;
        flag&=QName::tokenizer().insert(ns, "convMailMergeEsc") == NS_w11::LN_convMailMergeEsc;
        flag&=QName::tokenizer().insert(ns, "truncateFontHeight") == NS_w11::LN_truncateFontHeight;
        flag&=QName::tokenizer().insert(ns, "mwSmallCaps") == NS_w11::LN_mwSmallCaps;
        flag&=QName::tokenizer().insert(ns, "usePrinterMetrics") == NS_w11::LN_usePrinterMetrics;
        flag&=QName::tokenizer().insert(ns, "ww6BorderRules") == NS_w11::LN_ww6BorderRules;
        flag&=QName::tokenizer().insert(ns, "wrapTrailSpaces") == NS_w11::LN_wrapTrailSpaces;
        flag&=QName::tokenizer().insert(ns, "footnoteLayoutLikeWW8") == NS_w11::LN_footnoteLayoutLikeWW8;
        flag&=QName::tokenizer().insert(ns, "shapeLayoutLikeWW8") == NS_w11::LN_shapeLayoutLikeWW8;
        flag&=QName::tokenizer().insert(ns, "alignTablesRowByRow") == NS_w11::LN_alignTablesRowByRow;
        flag&=QName::tokenizer().insert(ns, "forgetLastTabAlignment") == NS_w11::LN_forgetLastTabAlignment;
        flag&=QName::tokenizer().insert(ns, "adjustLineHeightInTable") == NS_w11::LN_adjustLineHeightInTable;
        flag&=QName::tokenizer().insert(ns, "autoSpaceLikeWord95") == NS_w11::LN_autoSpaceLikeWord95;
        flag&=QName::tokenizer().insert(ns, "noSpaceRaiseLower") == NS_w11::LN_noSpaceRaiseLower;
        flag&=QName::tokenizer().insert(ns, "doNotUseHTMLParagraphAutoSpacing") == NS_w11::LN_doNotUseHTMLParagraphAutoSpacing;
        flag&=QName::tokenizer().insert(ns, "layoutRawTableWidth") == NS_w11::LN_layoutRawTableWidth;
        flag&=QName::tokenizer().insert(ns, "layoutTableRowsApart") == NS_w11::LN_layoutTableRowsApart;
        flag&=QName::tokenizer().insert(ns, "useWord97LineBreakingRules") == NS_w11::LN_useWord97LineBreakingRules;
        flag&=QName::tokenizer().insert(ns, "breakWrappedTables") == NS_w11::LN_breakWrappedTables;
        flag&=QName::tokenizer().insert(ns, "snapToGridInCell") == NS_w11::LN_snapToGridInCell;
        flag&=QName::tokenizer().insert(ns, "dontAllowFieldEndSelect") == NS_w11::LN_dontAllowFieldEndSelect;
        flag&=QName::tokenizer().insert(ns, "applyBreakingRules") == NS_w11::LN_applyBreakingRules;
        flag&=QName::tokenizer().insert(ns, "wrapTextWithPunct") == NS_w11::LN_wrapTextWithPunct;
        flag&=QName::tokenizer().insert(ns, "useAsianBreakRules") == NS_w11::LN_useAsianBreakRules;
        flag&=QName::tokenizer().insert(ns, "useWord2002TableStyleRules") == NS_w11::LN_useWord2002TableStyleRules;
        flag&=QName::tokenizer().insert(ns, "dontGrowAutofit") == NS_w11::LN_dontGrowAutofit;
        flag&=QName::tokenizer().insert(ns, "useFELayout") == NS_w11::LN_useFELayout;
        flag&=QName::tokenizer().insert(ns, "docVar") == NS_w11::LN_docVar;
        flag&=QName::tokenizer().insert(ns, "view") == NS_w11::LN_view;
        flag&=QName::tokenizer().insert(ns, "zoom") == NS_w11::LN_zoom;
        flag&=QName::tokenizer().insert(ns, "removePersonalInformation") == NS_w11::LN_removePersonalInformation;
        flag&=QName::tokenizer().insert(ns, "dontDisplayPageBoundaries") == NS_w11::LN_dontDisplayPageBoundaries;
        flag&=QName::tokenizer().insert(ns, "displayBackgroundShape") == NS_w11::LN_displayBackgroundShape;
        flag&=QName::tokenizer().insert(ns, "printPostScriptOverText") == NS_w11::LN_printPostScriptOverText;
        flag&=QName::tokenizer().insert(ns, "printFractionalCharacterWidth") == NS_w11::LN_printFractionalCharacterWidth;
        flag&=QName::tokenizer().insert(ns, "printFormsData") == NS_w11::LN_printFormsData;
        flag&=QName::tokenizer().insert(ns, "embedTrueTypeFonts") == NS_w11::LN_embedTrueTypeFonts;
        flag&=QName::tokenizer().insert(ns, "doNotEmbedSystemFonts") == NS_w11::LN_doNotEmbedSystemFonts;
        flag&=QName::tokenizer().insert(ns, "saveSubsetFonts") == NS_w11::LN_saveSubsetFonts;
        flag&=QName::tokenizer().insert(ns, "saveFormsData") == NS_w11::LN_saveFormsData;
        flag&=QName::tokenizer().insert(ns, "mirrorMargins") == NS_w11::LN_mirrorMargins;
        flag&=QName::tokenizer().insert(ns, "alignBordersAndEdges") == NS_w11::LN_alignBordersAndEdges;
        flag&=QName::tokenizer().insert(ns, "bordersDontSurroundHeader") == NS_w11::LN_bordersDontSurroundHeader;
        flag&=QName::tokenizer().insert(ns, "bordersDontSurroundFooter") == NS_w11::LN_bordersDontSurroundFooter;
        flag&=QName::tokenizer().insert(ns, "gutterAtTop") == NS_w11::LN_gutterAtTop;
        flag&=QName::tokenizer().insert(ns, "hideSpellingErrors") == NS_w11::LN_hideSpellingErrors;
        flag&=QName::tokenizer().insert(ns, "hideGrammaticalErrors") == NS_w11::LN_hideGrammaticalErrors;
        flag&=QName::tokenizer().insert(ns, "activeWritingStyle") == NS_w11::LN_activeWritingStyle;
        flag&=QName::tokenizer().insert(ns, "proofState") == NS_w11::LN_proofState;
        flag&=QName::tokenizer().insert(ns, "formsDesign") == NS_w11::LN_formsDesign;
        flag&=QName::tokenizer().insert(ns, "attachedTemplate") == NS_w11::LN_attachedTemplate;
        flag&=QName::tokenizer().insert(ns, "linkStyles") == NS_w11::LN_linkStyles;
        flag&=QName::tokenizer().insert(ns, "stylePaneFormatFilter") == NS_w11::LN_stylePaneFormatFilter;
        flag&=QName::tokenizer().insert(ns, "documentType") == NS_w11::LN_documentType;
        flag&=QName::tokenizer().insert(ns, "mailMerge") == NS_w11::LN_mailMerge;
        flag&=QName::tokenizer().insert(ns, "revisionView") == NS_w11::LN_revisionView;
        flag&=QName::tokenizer().insert(ns, "trackRevisions") == NS_w11::LN_trackRevisions;
        flag&=QName::tokenizer().insert(ns, "documentProtection") == NS_w11::LN_documentProtection;
        flag&=QName::tokenizer().insert(ns, "autoFormatOverride") == NS_w11::LN_autoFormatOverride;
        flag&=QName::tokenizer().insert(ns, "defaultTabStop") == NS_w11::LN_defaultTabStop;
        flag&=QName::tokenizer().insert(ns, "autoHyphenation") == NS_w11::LN_autoHyphenation;
        flag&=QName::tokenizer().insert(ns, "consecutiveHyphenLimit") == NS_w11::LN_consecutiveHyphenLimit;
        flag&=QName::tokenizer().insert(ns, "hyphenationZone") == NS_w11::LN_hyphenationZone;
        flag&=QName::tokenizer().insert(ns, "doNotHyphenateCaps") == NS_w11::LN_doNotHyphenateCaps;
        flag&=QName::tokenizer().insert(ns, "showEnvelope") == NS_w11::LN_showEnvelope;
        flag&=QName::tokenizer().insert(ns, "summaryLength") == NS_w11::LN_summaryLength;
        flag&=QName::tokenizer().insert(ns, "clickAndTypeStyle") == NS_w11::LN_clickAndTypeStyle;
        flag&=QName::tokenizer().insert(ns, "defaultTableStyle") == NS_w11::LN_defaultTableStyle;
        flag&=QName::tokenizer().insert(ns, "evenAndOddHeaders") == NS_w11::LN_evenAndOddHeaders;
        flag&=QName::tokenizer().insert(ns, "bookFoldRevPrinting") == NS_w11::LN_bookFoldRevPrinting;
        flag&=QName::tokenizer().insert(ns, "bookFoldPrinting") == NS_w11::LN_bookFoldPrinting;
        flag&=QName::tokenizer().insert(ns, "bookFoldPrintingSheets") == NS_w11::LN_bookFoldPrintingSheets;
        flag&=QName::tokenizer().insert(ns, "drawingGridHorizontalSpacing") == NS_w11::LN_drawingGridHorizontalSpacing;
        flag&=QName::tokenizer().insert(ns, "drawingGridVerticalSpacing") == NS_w11::LN_drawingGridVerticalSpacing;
        flag&=QName::tokenizer().insert(ns, "displayHorizontalDrawingGridEvery") == NS_w11::LN_displayHorizontalDrawingGridEvery;
        flag&=QName::tokenizer().insert(ns, "displayVerticalDrawingGridEvery") == NS_w11::LN_displayVerticalDrawingGridEvery;
        flag&=QName::tokenizer().insert(ns, "useMarginsForDrawingGridOrigin") == NS_w11::LN_useMarginsForDrawingGridOrigin;
        flag&=QName::tokenizer().insert(ns, "drawingGridHorizontalOrigin") == NS_w11::LN_drawingGridHorizontalOrigin;
        flag&=QName::tokenizer().insert(ns, "drawingGridVerticalOrigin") == NS_w11::LN_drawingGridVerticalOrigin;
        flag&=QName::tokenizer().insert(ns, "doNotShadeFormData") == NS_w11::LN_doNotShadeFormData;
        flag&=QName::tokenizer().insert(ns, "punctuationKerning") == NS_w11::LN_punctuationKerning;
        flag&=QName::tokenizer().insert(ns, "characterSpacingControl") == NS_w11::LN_characterSpacingControl;
        flag&=QName::tokenizer().insert(ns, "printTwoOnOne") == NS_w11::LN_printTwoOnOne;
        flag&=QName::tokenizer().insert(ns, "strictFirstAndLastChars") == NS_w11::LN_strictFirstAndLastChars;
        flag&=QName::tokenizer().insert(ns, "noLineBreaksAfter") == NS_w11::LN_noLineBreaksAfter;
        flag&=QName::tokenizer().insert(ns, "noLineBreaksBefore") == NS_w11::LN_noLineBreaksBefore;
        flag&=QName::tokenizer().insert(ns, "webPageEncoding") == NS_w11::LN_webPageEncoding;
        flag&=QName::tokenizer().insert(ns, "optimizeForBrowser") == NS_w11::LN_optimizeForBrowser;
        flag&=QName::tokenizer().insert(ns, "relyOnVML") == NS_w11::LN_relyOnVML;
        flag&=QName::tokenizer().insert(ns, "allowPNG") == NS_w11::LN_allowPNG;
        flag&=QName::tokenizer().insert(ns, "doNotRelyOnCSS") == NS_w11::LN_doNotRelyOnCSS;
        flag&=QName::tokenizer().insert(ns, "doNotSaveWebPagesAsSingleFile") == NS_w11::LN_doNotSaveWebPagesAsSingleFile;
        flag&=QName::tokenizer().insert(ns, "doNotOrganizeInFolder") == NS_w11::LN_doNotOrganizeInFolder;
        flag&=QName::tokenizer().insert(ns, "doNotUseLongFileNames") == NS_w11::LN_doNotUseLongFileNames;
        flag&=QName::tokenizer().insert(ns, "pixelsPerInch") == NS_w11::LN_pixelsPerInch;
        flag&=QName::tokenizer().insert(ns, "targetScreenSz") == NS_w11::LN_targetScreenSz;
        flag&=QName::tokenizer().insert(ns, "savePreviewPicture") == NS_w11::LN_savePreviewPicture;
        flag&=QName::tokenizer().insert(ns, "validateAgainstSchema") == NS_w11::LN_validateAgainstSchema;
        flag&=QName::tokenizer().insert(ns, "saveInvalidXML") == NS_w11::LN_saveInvalidXML;
        flag&=QName::tokenizer().insert(ns, "ignoreMixedContent") == NS_w11::LN_ignoreMixedContent;
        flag&=QName::tokenizer().insert(ns, "alwaysShowPlaceholderText") == NS_w11::LN_alwaysShowPlaceholderText;
        flag&=QName::tokenizer().insert(ns, "doNotUnderlineInvalidXML") == NS_w11::LN_doNotUnderlineInvalidXML;
        flag&=QName::tokenizer().insert(ns, "removeWordSchemaOnSave") == NS_w11::LN_removeWordSchemaOnSave;
        flag&=QName::tokenizer().insert(ns, "useXSLTWhenSaving") == NS_w11::LN_useXSLTWhenSaving;
        flag&=QName::tokenizer().insert(ns, "saveThroughXSLT") == NS_w11::LN_saveThroughXSLT;
        flag&=QName::tokenizer().insert(ns, "showXMLTags") == NS_w11::LN_showXMLTags;
        flag&=QName::tokenizer().insert(ns, "alwaysMergeEmptyNamespace") == NS_w11::LN_alwaysMergeEmptyNamespace;
        flag&=QName::tokenizer().insert(ns, "hdrShapeDefaults") == NS_w11::LN_hdrShapeDefaults;
        flag&=QName::tokenizer().insert(ns, "compat") == NS_w11::LN_compat;
        flag&=QName::tokenizer().insert(ns, "docVars") == NS_w11::LN_docVars;
        flag&=QName::tokenizer().insert(ns, "sourceFileName") == NS_w11::LN_sourceFileName;
        flag&=QName::tokenizer().insert(ns, "marW") == NS_w11::LN_marW;
        flag&=QName::tokenizer().insert(ns, "marH") == NS_w11::LN_marH;
        flag&=QName::tokenizer().insert(ns, "scrollbar") == NS_w11::LN_scrollbar;
        flag&=QName::tokenizer().insert(ns, "noResizeAllowed") == NS_w11::LN_noResizeAllowed;
        flag&=QName::tokenizer().insert(ns, "linkedToFile") == NS_w11::LN_linkedToFile;
        flag&=QName::tokenizer().insert(ns, "noBorder") == NS_w11::LN_noBorder;
        flag&=QName::tokenizer().insert(ns, "flatBorders") == NS_w11::LN_flatBorders;
        flag&=QName::tokenizer().insert(ns, "framesetSplitbar") == NS_w11::LN_framesetSplitbar;
        flag&=QName::tokenizer().insert(ns, "frameLayout") == NS_w11::LN_frameLayout;
        flag&=QName::tokenizer().insert(ns, "frameset") == NS_w11::LN_frameset;
        flag&=QName::tokenizer().insert(ns, "frame") == NS_w11::LN_frame;
        flag&=QName::tokenizer().insert(ns, "start") == NS_w11::LN_start;
        flag&=QName::tokenizer().insert(ns, "nfc") == NS_w11::LN_nfc;
        flag&=QName::tokenizer().insert(ns, "lvlRestart") == NS_w11::LN_lvlRestart;
        flag&=QName::tokenizer().insert(ns, "isLgl") == NS_w11::LN_isLgl;
        flag&=QName::tokenizer().insert(ns, "suff") == NS_w11::LN_suff;
        flag&=QName::tokenizer().insert(ns, "lvlText") == NS_w11::LN_lvlText;
        flag&=QName::tokenizer().insert(ns, "lvlPicBulletId") == NS_w11::LN_lvlPicBulletId;
        flag&=QName::tokenizer().insert(ns, "legacy") == NS_w11::LN_legacy;
        flag&=QName::tokenizer().insert(ns, "lvlJc") == NS_w11::LN_lvlJc;
        flag&=QName::tokenizer().insert(ns, "lsid") == NS_w11::LN_lsid;
        flag&=QName::tokenizer().insert(ns, "plt") == NS_w11::LN_plt;
        flag&=QName::tokenizer().insert(ns, "tmpl") == NS_w11::LN_tmpl;
        flag&=QName::tokenizer().insert(ns, "styleLink") == NS_w11::LN_styleLink;
        flag&=QName::tokenizer().insert(ns, "listStyleLink") == NS_w11::LN_listStyleLink;
        flag&=QName::tokenizer().insert(ns, "lvl") == NS_w11::LN_lvl;
        flag&=QName::tokenizer().insert(ns, "startOverride") == NS_w11::LN_startOverride;
        flag&=QName::tokenizer().insert(ns, "ilst") == NS_w11::LN_ilst;
        flag&=QName::tokenizer().insert(ns, "lvlOverride") == NS_w11::LN_lvlOverride;
        flag&=QName::tokenizer().insert(ns, "listPicBullet") == NS_w11::LN_listPicBullet;
        flag&=QName::tokenizer().insert(ns, "listDef") == NS_w11::LN_listDef;
        flag&=QName::tokenizer().insert(ns, "list") == NS_w11::LN_list;
        flag&=QName::tokenizer().insert(ns, "ilfoMacAtCleanup") == NS_w11::LN_ilfoMacAtCleanup;
        flag&=QName::tokenizer().insert(ns, "aliases") == NS_w11::LN_aliases;
        flag&=QName::tokenizer().insert(ns, "sti") == NS_w11::LN_sti;
        flag&=QName::tokenizer().insert(ns, "basedOn") == NS_w11::LN_basedOn;
        flag&=QName::tokenizer().insert(ns, "next") == NS_w11::LN_next;
        flag&=QName::tokenizer().insert(ns, "link") == NS_w11::LN_link;
        flag&=QName::tokenizer().insert(ns, "autoRedefine") == NS_w11::LN_autoRedefine;
        flag&=QName::tokenizer().insert(ns, "hidden") == NS_w11::LN_hidden;
        flag&=QName::tokenizer().insert(ns, "semiHidden") == NS_w11::LN_semiHidden;
        flag&=QName::tokenizer().insert(ns, "locked") == NS_w11::LN_locked;
        flag&=QName::tokenizer().insert(ns, "personal") == NS_w11::LN_personal;
        flag&=QName::tokenizer().insert(ns, "personalCompose") == NS_w11::LN_personalCompose;
        flag&=QName::tokenizer().insert(ns, "personalReply") == NS_w11::LN_personalReply;
        flag&=QName::tokenizer().insert(ns, "rsid") == NS_w11::LN_rsid;
        flag&=QName::tokenizer().insert(ns, "tblStylePr") == NS_w11::LN_tblStylePr;
        flag&=QName::tokenizer().insert(ns, "lsdException") == NS_w11::LN_lsdException;
        flag&=QName::tokenizer().insert(ns, "versionOfBuiltInStylenames") == NS_w11::LN_versionOfBuiltInStylenames;
        flag&=QName::tokenizer().insert(ns, "latentStyles") == NS_w11::LN_latentStyles;
        flag&=QName::tokenizer().insert(ns, "style") == NS_w11::LN_style;
        flag&=QName::tokenizer().insert(ns, "altName") == NS_w11::LN_altName;
        flag&=QName::tokenizer().insert(ns, "panose-1") == NS_w11::LN_panose_1;
        flag&=QName::tokenizer().insert(ns, "charset") == NS_w11::LN_charset;
        flag&=QName::tokenizer().insert(ns, "family") == NS_w11::LN_family;
        flag&=QName::tokenizer().insert(ns, "notTrueType") == NS_w11::LN_notTrueType;
        flag&=QName::tokenizer().insert(ns, "pitch") == NS_w11::LN_pitch;
        flag&=QName::tokenizer().insert(ns, "sig") == NS_w11::LN_sig;
        flag&=QName::tokenizer().insert(ns, "defaultFonts") == NS_w11::LN_defaultFonts;
        flag&=QName::tokenizer().insert(ns, "font") == NS_w11::LN_font;
        flag&=QName::tokenizer().insert(ns, "blockQuote") == NS_w11::LN_blockQuote;
        flag&=QName::tokenizer().insert(ns, "bodyDiv") == NS_w11::LN_bodyDiv;
        flag&=QName::tokenizer().insert(ns, "marLeft") == NS_w11::LN_marLeft;
        flag&=QName::tokenizer().insert(ns, "marRight") == NS_w11::LN_marRight;
        flag&=QName::tokenizer().insert(ns, "marTop") == NS_w11::LN_marTop;
        flag&=QName::tokenizer().insert(ns, "marBottom") == NS_w11::LN_marBottom;
        flag&=QName::tokenizer().insert(ns, "divBdr") == NS_w11::LN_divBdr;
        flag&=QName::tokenizer().insert(ns, "divsChild") == NS_w11::LN_divsChild;
        flag&=QName::tokenizer().insert(ns, "div") == NS_w11::LN_div;
        flag&=QName::tokenizer().insert(ns, "txbxContent") == NS_w11::LN_txbxContent;
        flag&=QName::tokenizer().insert(ns, "p") == NS_w11::LN_p;
        flag&=QName::tokenizer().insert(ns, "tbl") == NS_w11::LN_tbl;
        flag&=QName::tokenizer().insert(ns, "cfChunk") == NS_w11::LN_cfChunk;
        flag&=QName::tokenizer().insert(ns, "proofErr") == NS_w11::LN_proofErr;
        flag&=QName::tokenizer().insert(ns, "permStart") == NS_w11::LN_permStart;
        flag&=QName::tokenizer().insert(ns, "permEnd") == NS_w11::LN_permEnd;
        flag&=QName::tokenizer().insert(ns, "ignoreSubtree") == NS_w11::LN_ignoreSubtree;
        flag&=QName::tokenizer().insert(ns, "ignoreElements") == NS_w11::LN_ignoreElements;
        flag&=QName::tokenizer().insert(ns, "divs") == NS_w11::LN_divs;
        flag&=QName::tokenizer().insert(ns, "docOleData") == NS_w11::LN_docOleData;
        flag&=QName::tokenizer().insert(ns, "docSuppData") == NS_w11::LN_docSuppData;
        flag&=QName::tokenizer().insert(ns, "shapeDefaults") == NS_w11::LN_shapeDefaults;
        flag&=QName::tokenizer().insert(ns, "bgPict") == NS_w11::LN_bgPict;
        flag&=QName::tokenizer().insert(ns, "docPr") == NS_w11::LN_docPr;
        flag&=QName::tokenizer().insert(ns, "body") == NS_w11::LN_body;
        flag&=QName::tokenizer().insert(ns, "wordDocument") == NS_w11::LN_wordDocument;
        CPPUNIT_ASSERT_MESSAGE("test for namespace w11 failed", flag);}
    void test_sl12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/schemaLibrary/2003/core";
        flag&=QName::tokenizer().insert(ns, "schemaLibrary") == NS_sl12::LN_schemaLibrary;
        flag&=QName::tokenizer().insert(ns, "schema") == NS_sl12::LN_schema;
        CPPUNIT_ASSERT_MESSAGE("test for namespace sl12 failed", flag);}
    void test_s12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/styles";
        flag&=QName::tokenizer().insert(ns, "Styles") == NS_s12::LN_Styles;
        flag&=QName::tokenizer().insert(ns, "fmts") == NS_s12::LN_fmts;
        flag&=QName::tokenizer().insert(ns, "fonts") == NS_s12::LN_fonts;
        flag&=QName::tokenizer().insert(ns, "fills") == NS_s12::LN_fills;
        flag&=QName::tokenizer().insert(ns, "borders") == NS_s12::LN_borders;
        flag&=QName::tokenizer().insert(ns, "xfs") == NS_s12::LN_xfs;
        flag&=QName::tokenizer().insert(ns, "styles") == NS_s12::LN_styles;
        flag&=QName::tokenizer().insert(ns, "dxfs") == NS_s12::LN_dxfs;
        flag&=QName::tokenizer().insert(ns, "tableStyles") == NS_s12::LN_tableStyles;
        flag&=QName::tokenizer().insert(ns, "colorpalette") == NS_s12::LN_colorpalette;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_s12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "color") == NS_s12::LN_color;
        flag&=QName::tokenizer().insert(ns, "left") == NS_s12::LN_left;
        flag&=QName::tokenizer().insert(ns, "right") == NS_s12::LN_right;
        flag&=QName::tokenizer().insert(ns, "top") == NS_s12::LN_top;
        flag&=QName::tokenizer().insert(ns, "bottom") == NS_s12::LN_bottom;
        flag&=QName::tokenizer().insert(ns, "diag") == NS_s12::LN_diag;
        flag&=QName::tokenizer().insert(ns, "vertical") == NS_s12::LN_vertical;
        flag&=QName::tokenizer().insert(ns, "horizontal") == NS_s12::LN_horizontal;
        flag&=QName::tokenizer().insert(ns, "fontPr") == NS_s12::LN_fontPr;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_s12::LN_fill;
        flag&=QName::tokenizer().insert(ns, "pattern") == NS_s12::LN_pattern;
        flag&=QName::tokenizer().insert(ns, "gradient") == NS_s12::LN_gradient;
        flag&=QName::tokenizer().insert(ns, "image") == NS_s12::LN_image;
        flag&=QName::tokenizer().insert(ns, "fgColor") == NS_s12::LN_fgColor;
        flag&=QName::tokenizer().insert(ns, "bgColor") == NS_s12::LN_bgColor;
        flag&=QName::tokenizer().insert(ns, "stop") == NS_s12::LN_stop;
        flag&=QName::tokenizer().insert(ns, "fmt") == NS_s12::LN_fmt;
        flag&=QName::tokenizer().insert(ns, "xf") == NS_s12::LN_xf;
        flag&=QName::tokenizer().insert(ns, "alignment") == NS_s12::LN_alignment;
        flag&=QName::tokenizer().insert(ns, "protection") == NS_s12::LN_protection;
        flag&=QName::tokenizer().insert(ns, "styleInfo") == NS_s12::LN_styleInfo;
        flag&=QName::tokenizer().insert(ns, "style") == NS_s12::LN_style;
        flag&=QName::tokenizer().insert(ns, "dxf") == NS_s12::LN_dxf;
        flag&=QName::tokenizer().insert(ns, "font") == NS_s12::LN_font;
        flag&=QName::tokenizer().insert(ns, "indexedColors") == NS_s12::LN_indexedColors;
        flag&=QName::tokenizer().insert(ns, "themeColors") == NS_s12::LN_themeColors;
        flag&=QName::tokenizer().insert(ns, "mruColors") == NS_s12::LN_mruColors;
        flag&=QName::tokenizer().insert(ns, "rgbcolor") == NS_s12::LN_rgbcolor;
        flag&=QName::tokenizer().insert(ns, "tableStyle") == NS_s12::LN_tableStyle;
        flag&=QName::tokenizer().insert(ns, "tableStyleElement") == NS_s12::LN_tableStyleElement;
        flag&=QName::tokenizer().insert(ns, "charset") == NS_s12::LN_charset;
        flag&=QName::tokenizer().insert(ns, "family") == NS_s12::LN_family;
        flag&=QName::tokenizer().insert(ns, "b") == NS_s12::LN_b;
        flag&=QName::tokenizer().insert(ns, "i") == NS_s12::LN_i;
        flag&=QName::tokenizer().insert(ns, "strike") == NS_s12::LN_strike;
        flag&=QName::tokenizer().insert(ns, "outline") == NS_s12::LN_outline;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_s12::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "condense") == NS_s12::LN_condense;
        flag&=QName::tokenizer().insert(ns, "extend") == NS_s12::LN_extend;
        flag&=QName::tokenizer().insert(ns, "sz") == NS_s12::LN_sz;
        flag&=QName::tokenizer().insert(ns, "u") == NS_s12::LN_u;
        flag&=QName::tokenizer().insert(ns, "vertAlign") == NS_s12::LN_vertAlign;
        flag&=QName::tokenizer().insert(ns, "fontScheme") == NS_s12::LN_fontScheme;
        CPPUNIT_ASSERT_MESSAGE("test for namespace s12 failed", flag);}
    void test_udc(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/data/udc";
        flag&=QName::tokenizer().insert(ns, "DataSource") == NS_udc::LN_DataSource;
        flag&=QName::tokenizer().insert(ns, "Type") == NS_udc::LN_Type;
        flag&=QName::tokenizer().insert(ns, "SubType") == NS_udc::LN_SubType;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_udc::LN_Name;
        flag&=QName::tokenizer().insert(ns, "ConnectionInfo") == NS_udc::LN_ConnectionInfo;
        flag&=QName::tokenizer().insert(ns, "Description") == NS_udc::LN_Description;
        flag&=QName::tokenizer().insert(ns, "Keywords") == NS_udc::LN_Keywords;
        flag&=QName::tokenizer().insert(ns, "TooltipInfo") == NS_udc::LN_TooltipInfo;
        flag&=QName::tokenizer().insert(ns, "ClientParameterBindings") == NS_udc::LN_ClientParameterBindings;
        flag&=QName::tokenizer().insert(ns, "ClientParameterBinding") == NS_udc::LN_ClientParameterBinding;
        flag&=QName::tokenizer().insert(ns, "CredentialBinding") == NS_udc::LN_CredentialBinding;
        flag&=QName::tokenizer().insert(ns, "Credential") == NS_udc::LN_Credential;
        flag&=QName::tokenizer().insert(ns, "Attribute") == NS_udc::LN_Attribute;
        flag&=QName::tokenizer().insert(ns, "If") == NS_udc::LN_If;
        flag&=QName::tokenizer().insert(ns, "ClientParameterValue") == NS_udc::LN_ClientParameterValue;
        flag&=QName::tokenizer().insert(ns, "CredentialValue") == NS_udc::LN_CredentialValue;
        CPPUNIT_ASSERT_MESSAGE("test for namespace udc failed", flag);}
    void test_dr3d(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0";
        flag&=QName::tokenizer().insert(ns, "distance") == NS_dr3d::LN_distance;
        flag&=QName::tokenizer().insert(ns, "focal-length") == NS_dr3d::LN_focal_length;
        flag&=QName::tokenizer().insert(ns, "shadow-slant") == NS_dr3d::LN_shadow_slant;
        flag&=QName::tokenizer().insert(ns, "shade-mode") == NS_dr3d::LN_shade_mode;
        flag&=QName::tokenizer().insert(ns, "ambient-color") == NS_dr3d::LN_ambient_color;
        flag&=QName::tokenizer().insert(ns, "lighting-mode") == NS_dr3d::LN_lighting_mode;
        flag&=QName::tokenizer().insert(ns, "transform") == NS_dr3d::LN_transform;
        flag&=QName::tokenizer().insert(ns, "light") == NS_dr3d::LN_light;
        flag&=QName::tokenizer().insert(ns, "diffuse-color") == NS_dr3d::LN_diffuse_color;
        flag&=QName::tokenizer().insert(ns, "direction") == NS_dr3d::LN_direction;
        flag&=QName::tokenizer().insert(ns, "enabled") == NS_dr3d::LN_enabled;
        flag&=QName::tokenizer().insert(ns, "specular") == NS_dr3d::LN_specular;
        flag&=QName::tokenizer().insert(ns, "cube") == NS_dr3d::LN_cube;
        flag&=QName::tokenizer().insert(ns, "min-edge") == NS_dr3d::LN_min_edge;
        flag&=QName::tokenizer().insert(ns, "max-edge") == NS_dr3d::LN_max_edge;
        flag&=QName::tokenizer().insert(ns, "sphere") == NS_dr3d::LN_sphere;
        flag&=QName::tokenizer().insert(ns, "center") == NS_dr3d::LN_center;
        flag&=QName::tokenizer().insert(ns, "size") == NS_dr3d::LN_size;
        flag&=QName::tokenizer().insert(ns, "extrude") == NS_dr3d::LN_extrude;
        flag&=QName::tokenizer().insert(ns, "rotate") == NS_dr3d::LN_rotate;
        flag&=QName::tokenizer().insert(ns, "horizontal-segments") == NS_dr3d::LN_horizontal_segments;
        flag&=QName::tokenizer().insert(ns, "vertical-segments") == NS_dr3d::LN_vertical_segments;
        flag&=QName::tokenizer().insert(ns, "edge-rounding") == NS_dr3d::LN_edge_rounding;
        flag&=QName::tokenizer().insert(ns, "edge-rounding-mode") == NS_dr3d::LN_edge_rounding_mode;
        flag&=QName::tokenizer().insert(ns, "back-scale") == NS_dr3d::LN_back_scale;
        flag&=QName::tokenizer().insert(ns, "depth") == NS_dr3d::LN_depth;
        flag&=QName::tokenizer().insert(ns, "backface-culling") == NS_dr3d::LN_backface_culling;
        flag&=QName::tokenizer().insert(ns, "end-angle") == NS_dr3d::LN_end_angle;
        flag&=QName::tokenizer().insert(ns, "close-front") == NS_dr3d::LN_close_front;
        flag&=QName::tokenizer().insert(ns, "close-back") == NS_dr3d::LN_close_back;
        flag&=QName::tokenizer().insert(ns, "normals-kind") == NS_dr3d::LN_normals_kind;
        flag&=QName::tokenizer().insert(ns, "normals-direction") == NS_dr3d::LN_normals_direction;
        flag&=QName::tokenizer().insert(ns, "texture-generation-mode-x") == NS_dr3d::LN_texture_generation_mode_x;
        flag&=QName::tokenizer().insert(ns, "texture-generation-mode-y") == NS_dr3d::LN_texture_generation_mode_y;
        flag&=QName::tokenizer().insert(ns, "texture-kind") == NS_dr3d::LN_texture_kind;
        flag&=QName::tokenizer().insert(ns, "texture-filter") == NS_dr3d::LN_texture_filter;
        flag&=QName::tokenizer().insert(ns, "texture-mode") == NS_dr3d::LN_texture_mode;
        flag&=QName::tokenizer().insert(ns, "emissive-color") == NS_dr3d::LN_emissive_color;
        flag&=QName::tokenizer().insert(ns, "specular-color") == NS_dr3d::LN_specular_color;
        flag&=QName::tokenizer().insert(ns, "shininess") == NS_dr3d::LN_shininess;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_dr3d::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "scene") == NS_dr3d::LN_scene;
        flag&=QName::tokenizer().insert(ns, "vrp") == NS_dr3d::LN_vrp;
        flag&=QName::tokenizer().insert(ns, "vpn") == NS_dr3d::LN_vpn;
        flag&=QName::tokenizer().insert(ns, "vup") == NS_dr3d::LN_vup;
        flag&=QName::tokenizer().insert(ns, "projection") == NS_dr3d::LN_projection;
        CPPUNIT_ASSERT_MESSAGE("test for namespace dr3d failed", flag);}
    void test_rdf(){
        bool flag=true;
        static const char ns[] = "http://www.w3.org/1999/02/22-rdf-syntax-ns#";
        flag&=QName::tokenizer().insert(ns, "RDF") == NS_rdf::LN_RDF;
        flag&=QName::tokenizer().insert(ns, "ID") == NS_rdf::LN_ID;
        flag&=QName::tokenizer().insert(ns, "nodeID") == NS_rdf::LN_nodeID;
        flag&=QName::tokenizer().insert(ns, "about") == NS_rdf::LN_about;
        flag&=QName::tokenizer().insert(ns, "resource") == NS_rdf::LN_resource;
        flag&=QName::tokenizer().insert(ns, "datatype") == NS_rdf::LN_datatype;
        flag&=QName::tokenizer().insert(ns, "parseType") == NS_rdf::LN_parseType;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rdf failed", flag);}
    void test_sb12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/supbook";
        flag&=QName::tokenizer().insert(ns, "supBook") == NS_sb12::LN_supBook;
        flag&=QName::tokenizer().insert(ns, "sheetNames") == NS_sb12::LN_sheetNames;
        flag&=QName::tokenizer().insert(ns, "namedLinks") == NS_sb12::LN_namedLinks;
        flag&=QName::tokenizer().insert(ns, "supSheet") == NS_sb12::LN_supSheet;
        flag&=QName::tokenizer().insert(ns, "sheetName") == NS_sb12::LN_sheetName;
        flag&=QName::tokenizer().insert(ns, "xrow") == NS_sb12::LN_xrow;
        flag&=QName::tokenizer().insert(ns, "xc") == NS_sb12::LN_xc;
        flag&=QName::tokenizer().insert(ns, "xv") == NS_sb12::LN_xv;
        flag&=QName::tokenizer().insert(ns, "namedRange") == NS_sb12::LN_namedRange;
        flag&=QName::tokenizer().insert(ns, "DDELink") == NS_sb12::LN_DDELink;
        flag&=QName::tokenizer().insert(ns, "OLELink") == NS_sb12::LN_OLELink;
        flag&=QName::tokenizer().insert(ns, "values") == NS_sb12::LN_values;
        flag&=QName::tokenizer().insert(ns, "value") == NS_sb12::LN_value;
        flag&=QName::tokenizer().insert(ns, "val") == NS_sb12::LN_val;
        CPPUNIT_ASSERT_MESSAGE("test for namespace sb12 failed", flag);}
    void test_wb12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/workbook";
        flag&=QName::tokenizer().insert(ns, "supBookSource") == NS_wb12::LN_supBookSource;
        flag&=QName::tokenizer().insert(ns, "imageRelation") == NS_wb12::LN_imageRelation;
        flag&=QName::tokenizer().insert(ns, "pivotCache") == NS_wb12::LN_pivotCache;
        flag&=QName::tokenizer().insert(ns, "docRouteInfo") == NS_wb12::LN_docRouteInfo;
        flag&=QName::tokenizer().insert(ns, "docRouteOriginator") == NS_wb12::LN_docRouteOriginator;
        flag&=QName::tokenizer().insert(ns, "docRouteRecipient") == NS_wb12::LN_docRouteRecipient;
        flag&=QName::tokenizer().insert(ns, "book") == NS_wb12::LN_book;
        flag&=QName::tokenizer().insert(ns, "fileVersion") == NS_wb12::LN_fileVersion;
        flag&=QName::tokenizer().insert(ns, "fileSharing") == NS_wb12::LN_fileSharing;
        flag&=QName::tokenizer().insert(ns, "bookViews") == NS_wb12::LN_bookViews;
        flag&=QName::tokenizer().insert(ns, "userBookViews") == NS_wb12::LN_userBookViews;
        flag&=QName::tokenizer().insert(ns, "sheets") == NS_wb12::LN_sheets;
        flag&=QName::tokenizer().insert(ns, "wbPr") == NS_wb12::LN_wbPr;
        flag&=QName::tokenizer().insert(ns, "webPub") == NS_wb12::LN_webPub;
        flag&=QName::tokenizer().insert(ns, "webPubItems") == NS_wb12::LN_webPubItems;
        flag&=QName::tokenizer().insert(ns, "smartTag") == NS_wb12::LN_smartTag;
        flag&=QName::tokenizer().insert(ns, "smartTagTypes") == NS_wb12::LN_smartTagTypes;
        flag&=QName::tokenizer().insert(ns, "fileRecovery") == NS_wb12::LN_fileRecovery;
        flag&=QName::tokenizer().insert(ns, "calcPr") == NS_wb12::LN_calcPr;
        flag&=QName::tokenizer().insert(ns, "extRefs") == NS_wb12::LN_extRefs;
        flag&=QName::tokenizer().insert(ns, "namedRanges") == NS_wb12::LN_namedRanges;
        flag&=QName::tokenizer().insert(ns, "pivotCaches") == NS_wb12::LN_pivotCaches;
        flag&=QName::tokenizer().insert(ns, "docRoute") == NS_wb12::LN_docRoute;
        flag&=QName::tokenizer().insert(ns, "oleSize") == NS_wb12::LN_oleSize;
        flag&=QName::tokenizer().insert(ns, "bookProtection") == NS_wb12::LN_bookProtection;
        flag&=QName::tokenizer().insert(ns, "image") == NS_wb12::LN_image;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_wb12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "bookView") == NS_wb12::LN_bookView;
        flag&=QName::tokenizer().insert(ns, "userBookView") == NS_wb12::LN_userBookView;
        flag&=QName::tokenizer().insert(ns, "sheet") == NS_wb12::LN_sheet;
        flag&=QName::tokenizer().insert(ns, "smartTagType") == NS_wb12::LN_smartTagType;
        flag&=QName::tokenizer().insert(ns, "namedRange") == NS_wb12::LN_namedRange;
        flag&=QName::tokenizer().insert(ns, "name") == NS_wb12::LN_name;
        flag&=QName::tokenizer().insert(ns, "value") == NS_wb12::LN_value;
        flag&=QName::tokenizer().insert(ns, "comment") == NS_wb12::LN_comment;
        flag&=QName::tokenizer().insert(ns, "customMenu") == NS_wb12::LN_customMenu;
        flag&=QName::tokenizer().insert(ns, "description") == NS_wb12::LN_description;
        flag&=QName::tokenizer().insert(ns, "help") == NS_wb12::LN_help;
        flag&=QName::tokenizer().insert(ns, "statusBar") == NS_wb12::LN_statusBar;
        CPPUNIT_ASSERT_MESSAGE("test for namespace wb12 failed", flag);}
    void test_RowsetSchema(){
        bool flag=true;
        static const char ns[] = "#RowsetSchema";
        flag&=QName::tokenizer().insert(ns, "row") == NS_RowsetSchema::LN_row;
        CPPUNIT_ASSERT_MESSAGE("test for namespace RowsetSchema failed", flag);}
    void test_ec12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/externalconnection";
        flag&=QName::tokenizer().insert(ns, "webqProps") == NS_ec12::LN_webqProps;
        flag&=QName::tokenizer().insert(ns, "txtWiz") == NS_ec12::LN_txtWiz;
        flag&=QName::tokenizer().insert(ns, "qParams") == NS_ec12::LN_qParams;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_ec12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "tables") == NS_ec12::LN_tables;
        flag&=QName::tokenizer().insert(ns, "qParam") == NS_ec12::LN_qParam;
        flag&=QName::tokenizer().insert(ns, "m") == NS_ec12::LN_m;
        flag&=QName::tokenizer().insert(ns, "s") == NS_ec12::LN_s;
        flag&=QName::tokenizer().insert(ns, "x") == NS_ec12::LN_x;
        flag&=QName::tokenizer().insert(ns, "fldInfos") == NS_ec12::LN_fldInfos;
        flag&=QName::tokenizer().insert(ns, "fldInfo") == NS_ec12::LN_fldInfo;
        flag&=QName::tokenizer().insert(ns, "extConns") == NS_ec12::LN_extConns;
        flag&=QName::tokenizer().insert(ns, "extConn") == NS_ec12::LN_extConn;
        flag&=QName::tokenizer().insert(ns, "dbProps") == NS_ec12::LN_dbProps;
        flag&=QName::tokenizer().insert(ns, "olapProps") == NS_ec12::LN_olapProps;
        CPPUNIT_ASSERT_MESSAGE("test for namespace ec12 failed", flag);}
    void test_qt12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/querytable";
        flag&=QName::tokenizer().insert(ns, "querytable") == NS_qt12::LN_querytable;
        flag&=QName::tokenizer().insert(ns, "qtRefr") == NS_qt12::LN_qtRefr;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_qt12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "delNms") == NS_qt12::LN_delNms;
        flag&=QName::tokenizer().insert(ns, "qtflds") == NS_qt12::LN_qtflds;
        flag&=QName::tokenizer().insert(ns, "delNm") == NS_qt12::LN_delNm;
        flag&=QName::tokenizer().insert(ns, "qtfld") == NS_qt12::LN_qtfld;
        CPPUNIT_ASSERT_MESSAGE("test for namespace qt12 failed", flag);}
    void test_o12(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:office";
        flag&=QName::tokenizer().insert(ns, "DocumentProperties") == NS_o12::LN_DocumentProperties;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_o12::LN_Name;
        flag&=QName::tokenizer().insert(ns, "Description") == NS_o12::LN_Description;
        flag&=QName::tokenizer().insert(ns, "Keywords") == NS_o12::LN_Keywords;
        flag&=QName::tokenizer().insert(ns, "File") == NS_o12::LN_File;
        flag&=QName::tokenizer().insert(ns, "MainFile") == NS_o12::LN_MainFile;
        flag&=QName::tokenizer().insert(ns, "Title") == NS_o12::LN_Title;
        flag&=QName::tokenizer().insert(ns, "Subject") == NS_o12::LN_Subject;
        flag&=QName::tokenizer().insert(ns, "Author") == NS_o12::LN_Author;
        flag&=QName::tokenizer().insert(ns, "LastAuthor") == NS_o12::LN_LastAuthor;
        flag&=QName::tokenizer().insert(ns, "Revision") == NS_o12::LN_Revision;
        flag&=QName::tokenizer().insert(ns, "AppName") == NS_o12::LN_AppName;
        flag&=QName::tokenizer().insert(ns, "TotalTime") == NS_o12::LN_TotalTime;
        flag&=QName::tokenizer().insert(ns, "LastPrinted") == NS_o12::LN_LastPrinted;
        flag&=QName::tokenizer().insert(ns, "Created") == NS_o12::LN_Created;
        flag&=QName::tokenizer().insert(ns, "LastSaved") == NS_o12::LN_LastSaved;
        flag&=QName::tokenizer().insert(ns, "Pages") == NS_o12::LN_Pages;
        flag&=QName::tokenizer().insert(ns, "Words") == NS_o12::LN_Words;
        flag&=QName::tokenizer().insert(ns, "Characters") == NS_o12::LN_Characters;
        flag&=QName::tokenizer().insert(ns, "Category") == NS_o12::LN_Category;
        flag&=QName::tokenizer().insert(ns, "PresentationFormat") == NS_o12::LN_PresentationFormat;
        flag&=QName::tokenizer().insert(ns, "Manager") == NS_o12::LN_Manager;
        flag&=QName::tokenizer().insert(ns, "Company") == NS_o12::LN_Company;
        flag&=QName::tokenizer().insert(ns, "Guid") == NS_o12::LN_Guid;
        flag&=QName::tokenizer().insert(ns, "HyperlinkBase") == NS_o12::LN_HyperlinkBase;
        flag&=QName::tokenizer().insert(ns, "Bytes") == NS_o12::LN_Bytes;
        flag&=QName::tokenizer().insert(ns, "Lines") == NS_o12::LN_Lines;
        flag&=QName::tokenizer().insert(ns, "Paragraphs") == NS_o12::LN_Paragraphs;
        flag&=QName::tokenizer().insert(ns, "CharactersWithSpaces") == NS_o12::LN_CharactersWithSpaces;
        flag&=QName::tokenizer().insert(ns, "Version") == NS_o12::LN_Version;
        flag&=QName::tokenizer().insert(ns, "SmartTagType") == NS_o12::LN_SmartTagType;
        flag&=QName::tokenizer().insert(ns, "SmartTags") == NS_o12::LN_SmartTags;
        flag&=QName::tokenizer().insert(ns, "CustomDocumentProperties") == NS_o12::LN_CustomDocumentProperties;
        flag&=QName::tokenizer().insert(ns, "OfficeDocumentSettings") == NS_o12::LN_OfficeDocumentSettings;
        flag&=QName::tokenizer().insert(ns, "DownloadComponents") == NS_o12::LN_DownloadComponents;
        flag&=QName::tokenizer().insert(ns, "LocationOfComponents") == NS_o12::LN_LocationOfComponents;
        flag&=QName::tokenizer().insert(ns, "shapedefaults") == NS_o12::LN_shapedefaults;
        flag&=QName::tokenizer().insert(ns, "colormru") == NS_o12::LN_colormru;
        flag&=QName::tokenizer().insert(ns, "colormenu") == NS_o12::LN_colormenu;
        flag&=QName::tokenizer().insert(ns, "colors") == NS_o12::LN_colors;
        flag&=QName::tokenizer().insert(ns, "shapelayout") == NS_o12::LN_shapelayout;
        flag&=QName::tokenizer().insert(ns, "idmap") == NS_o12::LN_idmap;
        flag&=QName::tokenizer().insert(ns, "regrouptable") == NS_o12::LN_regrouptable;
        flag&=QName::tokenizer().insert(ns, "rules") == NS_o12::LN_rules;
        flag&=QName::tokenizer().insert(ns, "entry") == NS_o12::LN_entry;
        flag&=QName::tokenizer().insert(ns, "r") == NS_o12::LN_r;
        flag&=QName::tokenizer().insert(ns, "proxy") == NS_o12::LN_proxy;
        flag&=QName::tokenizer().insert(ns, "skew") == NS_o12::LN_skew;
        flag&=QName::tokenizer().insert(ns, "extrusion") == NS_o12::LN_extrusion;
        flag&=QName::tokenizer().insert(ns, "callout") == NS_o12::LN_callout;
        flag&=QName::tokenizer().insert(ns, "lock") == NS_o12::LN_lock;
        flag&=QName::tokenizer().insert(ns, "OLEObject") == NS_o12::LN_OLEObject;
        flag&=QName::tokenizer().insert(ns, "complex") == NS_o12::LN_complex;
        CPPUNIT_ASSERT_MESSAGE("test for namespace o12 failed", flag);}
    void test_s11(){
        bool flag=true;
        static const char ns[] = "uuid:BDC6E3F0-6DA3-11d1-A2A3-00AA00C14882";
        flag&=QName::tokenizer().insert(ns, "Schema") == NS_s11::LN_Schema;
        flag&=QName::tokenizer().insert(ns, "ElementType") == NS_s11::LN_ElementType;
        flag&=QName::tokenizer().insert(ns, "attribute") == NS_s11::LN_attribute;
        flag&=QName::tokenizer().insert(ns, "extends") == NS_s11::LN_extends;
        flag&=QName::tokenizer().insert(ns, "AttributeType") == NS_s11::LN_AttributeType;
        flag&=QName::tokenizer().insert(ns, "datatype") == NS_s11::LN_datatype;
        CPPUNIT_ASSERT_MESSAGE("test for namespace s11 failed", flag);}
    void test_x11(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:excel";
        flag&=QName::tokenizer().insert(ns, "CacheDetails") == NS_x11::LN_CacheDetails;
        flag&=QName::tokenizer().insert(ns, "FieldListTop") == NS_x11::LN_FieldListTop;
        flag&=QName::tokenizer().insert(ns, "FieldListBottom") == NS_x11::LN_FieldListBottom;
        flag&=QName::tokenizer().insert(ns, "FieldListRight") == NS_x11::LN_FieldListRight;
        flag&=QName::tokenizer().insert(ns, "FieldListLeft") == NS_x11::LN_FieldListLeft;
        flag&=QName::tokenizer().insert(ns, "PivotAxis") == NS_x11::LN_PivotAxis;
        flag&=QName::tokenizer().insert(ns, "VersionUpdateableMin") == NS_x11::LN_VersionUpdateableMin;
        flag&=QName::tokenizer().insert(ns, "VersionLastUpdate") == NS_x11::LN_VersionLastUpdate;
        flag&=QName::tokenizer().insert(ns, "VersionRefreshableMin") == NS_x11::LN_VersionRefreshableMin;
        flag&=QName::tokenizer().insert(ns, "DefaultVersion") == NS_x11::LN_DefaultVersion;
        flag&=QName::tokenizer().insert(ns, "PivotField") == NS_x11::LN_PivotField;
        flag&=QName::tokenizer().insert(ns, "CubeField") == NS_x11::LN_CubeField;
        flag&=QName::tokenizer().insert(ns, "CalculatedMember") == NS_x11::LN_CalculatedMember;
        flag&=QName::tokenizer().insert(ns, "MemberName") == NS_x11::LN_MemberName;
        flag&=QName::tokenizer().insert(ns, "ParentUniqueName") == NS_x11::LN_ParentUniqueName;
        flag&=QName::tokenizer().insert(ns, "SolveOrder") == NS_x11::LN_SolveOrder;
        flag&=QName::tokenizer().insert(ns, "Invalid") == NS_x11::LN_Invalid;
        flag&=QName::tokenizer().insert(ns, "Set") == NS_x11::LN_Set;
        flag&=QName::tokenizer().insert(ns, "PTLineItems") == NS_x11::LN_PTLineItems;
        flag&=QName::tokenizer().insert(ns, "PTSource") == NS_x11::LN_PTSource;
        flag&=QName::tokenizer().insert(ns, "PTFormula") == NS_x11::LN_PTFormula;
        flag&=QName::tokenizer().insert(ns, "PTFormat") == NS_x11::LN_PTFormat;
        flag&=QName::tokenizer().insert(ns, "CacheFile") == NS_x11::LN_CacheFile;
        flag&=QName::tokenizer().insert(ns, "QuerySource") == NS_x11::LN_QuerySource;
        flag&=QName::tokenizer().insert(ns, "DefaultItem") == NS_x11::LN_DefaultItem;
        flag&=QName::tokenizer().insert(ns, "AllItemName") == NS_x11::LN_AllItemName;
        flag&=QName::tokenizer().insert(ns, "Dimension") == NS_x11::LN_Dimension;
        flag&=QName::tokenizer().insert(ns, "DontShowInFieldList") == NS_x11::LN_DontShowInFieldList;
        flag&=QName::tokenizer().insert(ns, "Measure") == NS_x11::LN_Measure;
        flag&=QName::tokenizer().insert(ns, "EnableMultiplePageItems") == NS_x11::LN_EnableMultiplePageItems;
        flag&=QName::tokenizer().insert(ns, "GroupLevel") == NS_x11::LN_GroupLevel;
        flag&=QName::tokenizer().insert(ns, "IsGroupLevel") == NS_x11::LN_IsGroupLevel;
        flag&=QName::tokenizer().insert(ns, "GroupDefinition") == NS_x11::LN_GroupDefinition;
        flag&=QName::tokenizer().insert(ns, "LengthLevelUniqueName") == NS_x11::LN_LengthLevelUniqueName;
        flag&=QName::tokenizer().insert(ns, "DisplayIn") == NS_x11::LN_DisplayIn;
        flag&=QName::tokenizer().insert(ns, "GroupNumber") == NS_x11::LN_GroupNumber;
        flag&=QName::tokenizer().insert(ns, "ParentName") == NS_x11::LN_ParentName;
        flag&=QName::tokenizer().insert(ns, "ParentIsOther") == NS_x11::LN_ParentIsOther;
        flag&=QName::tokenizer().insert(ns, "GroupMember") == NS_x11::LN_GroupMember;
        flag&=QName::tokenizer().insert(ns, "IsNotFiltered") == NS_x11::LN_IsNotFiltered;
        flag&=QName::tokenizer().insert(ns, "TotalAllMembers") == NS_x11::LN_TotalAllMembers;
        flag&=QName::tokenizer().insert(ns, "DetailMaxHeight") == NS_x11::LN_DetailMaxHeight;
        flag&=QName::tokenizer().insert(ns, "DetailMaxWidth") == NS_x11::LN_DetailMaxWidth;
        flag&=QName::tokenizer().insert(ns, "NoDetailAutoFit") == NS_x11::LN_NoDetailAutoFit;
        flag&=QName::tokenizer().insert(ns, "DetailRowHeight") == NS_x11::LN_DetailRowHeight;
        flag&=QName::tokenizer().insert(ns, "TotalFormat") == NS_x11::LN_TotalFormat;
        flag&=QName::tokenizer().insert(ns, "MemberFormat") == NS_x11::LN_MemberFormat;
        flag&=QName::tokenizer().insert(ns, "FieldLabelFormat") == NS_x11::LN_FieldLabelFormat;
        flag&=QName::tokenizer().insert(ns, "DetailSortOrder") == NS_x11::LN_DetailSortOrder;
        flag&=QName::tokenizer().insert(ns, "CellsExpanded") == NS_x11::LN_CellsExpanded;
        flag&=QName::tokenizer().insert(ns, "CellsNotExpanded") == NS_x11::LN_CellsNotExpanded;
        flag&=QName::tokenizer().insert(ns, "CellsExpandedSeqNum") == NS_x11::LN_CellsExpandedSeqNum;
        flag&=QName::tokenizer().insert(ns, "Operator") == NS_x11::LN_Operator;
        flag&=QName::tokenizer().insert(ns, "Alignment") == NS_x11::LN_Alignment;
        flag&=QName::tokenizer().insert(ns, "CaptionAlignment") == NS_x11::LN_CaptionAlignment;
        flag&=QName::tokenizer().insert(ns, "FitWidth") == NS_x11::LN_FitWidth;
        flag&=QName::tokenizer().insert(ns, "FitHeight") == NS_x11::LN_FitHeight;
        flag&=QName::tokenizer().insert(ns, "LeftToRight") == NS_x11::LN_LeftToRight;
        flag&=QName::tokenizer().insert(ns, "BlackAndWhite") == NS_x11::LN_BlackAndWhite;
        flag&=QName::tokenizer().insert(ns, "NoPrinterInfo") == NS_x11::LN_NoPrinterInfo;
        flag&=QName::tokenizer().insert(ns, "DraftQuality") == NS_x11::LN_DraftQuality;
        flag&=QName::tokenizer().insert(ns, "CommentsLayout") == NS_x11::LN_CommentsLayout;
        flag&=QName::tokenizer().insert(ns, "Scale") == NS_x11::LN_Scale;
        flag&=QName::tokenizer().insert(ns, "PrintErrors") == NS_x11::LN_PrintErrors;
        flag&=QName::tokenizer().insert(ns, "ValidPrinterInfo") == NS_x11::LN_ValidPrinterInfo;
        flag&=QName::tokenizer().insert(ns, "PaperSizeIndex") == NS_x11::LN_PaperSizeIndex;
        flag&=QName::tokenizer().insert(ns, "HorizontalResolution") == NS_x11::LN_HorizontalResolution;
        flag&=QName::tokenizer().insert(ns, "VerticalResolution") == NS_x11::LN_VerticalResolution;
        flag&=QName::tokenizer().insert(ns, "Gridlines") == NS_x11::LN_Gridlines;
        flag&=QName::tokenizer().insert(ns, "NumberOfCopies") == NS_x11::LN_NumberOfCopies;
        flag&=QName::tokenizer().insert(ns, "RowColHeadings") == NS_x11::LN_RowColHeadings;
        flag&=QName::tokenizer().insert(ns, "PTRule") == NS_x11::LN_PTRule;
        flag&=QName::tokenizer().insert(ns, "FormatType") == NS_x11::LN_FormatType;
        flag&=QName::tokenizer().insert(ns, "IndividualCellBorders") == NS_x11::LN_IndividualCellBorders;
        flag&=QName::tokenizer().insert(ns, "ItemType") == NS_x11::LN_ItemType;
        flag&=QName::tokenizer().insert(ns, "BlockTotal") == NS_x11::LN_BlockTotal;
        flag&=QName::tokenizer().insert(ns, "CountOfSameItems") == NS_x11::LN_CountOfSameItems;
        flag&=QName::tokenizer().insert(ns, "PTLineItem") == NS_x11::LN_PTLineItem;
        flag&=QName::tokenizer().insert(ns, "RuleType") == NS_x11::LN_RuleType;
        flag&=QName::tokenizer().insert(ns, "Rule") == NS_x11::LN_Rule;
        flag&=QName::tokenizer().insert(ns, "RuleV10") == NS_x11::LN_RuleV10;
        flag&=QName::tokenizer().insert(ns, "ParseRuleAsV10") == NS_x11::LN_ParseRuleAsV10;
        flag&=QName::tokenizer().insert(ns, "Area") == NS_x11::LN_Area;
        flag&=QName::tokenizer().insert(ns, "Outline") == NS_x11::LN_Outline;
        flag&=QName::tokenizer().insert(ns, "MissingItemsLimit") == NS_x11::LN_MissingItemsLimit;
        flag&=QName::tokenizer().insert(ns, "VersionLastRefresh") == NS_x11::LN_VersionLastRefresh;
        flag&=QName::tokenizer().insert(ns, "RefreshName") == NS_x11::LN_RefreshName;
        flag&=QName::tokenizer().insert(ns, "RefreshDate") == NS_x11::LN_RefreshDate;
        flag&=QName::tokenizer().insert(ns, "RefreshDateCopy") == NS_x11::LN_RefreshDateCopy;
        flag&=QName::tokenizer().insert(ns, "ConsolidationReference") == NS_x11::LN_ConsolidationReference;
        flag&=QName::tokenizer().insert(ns, "NoSaveData") == NS_x11::LN_NoSaveData;
        flag&=QName::tokenizer().insert(ns, "SourceConsolidation") == NS_x11::LN_SourceConsolidation;
        flag&=QName::tokenizer().insert(ns, "BackgroundQuery") == NS_x11::LN_BackgroundQuery;
        flag&=QName::tokenizer().insert(ns, "CubeSource") == NS_x11::LN_CubeSource;
        flag&=QName::tokenizer().insert(ns, "DisableRefresh") == NS_x11::LN_DisableRefresh;
        flag&=QName::tokenizer().insert(ns, "HasNoRecords") == NS_x11::LN_HasNoRecords;
        flag&=QName::tokenizer().insert(ns, "OptimizeCache") == NS_x11::LN_OptimizeCache;
        flag&=QName::tokenizer().insert(ns, "RefreshOnFileOpen") == NS_x11::LN_RefreshOnFileOpen;
        flag&=QName::tokenizer().insert(ns, "PublishObject") == NS_x11::LN_PublishObject;
        flag&=QName::tokenizer().insert(ns, "DivID") == NS_x11::LN_DivID;
        flag&=QName::tokenizer().insert(ns, "SourceType") == NS_x11::LN_SourceType;
        flag&=QName::tokenizer().insert(ns, "HtmlType") == NS_x11::LN_HtmlType;
        flag&=QName::tokenizer().insert(ns, "AutoRepublish") == NS_x11::LN_AutoRepublish;
        flag&=QName::tokenizer().insert(ns, "QueryType") == NS_x11::LN_QueryType;
        flag&=QName::tokenizer().insert(ns, "CommandText") == NS_x11::LN_CommandText;
        flag&=QName::tokenizer().insert(ns, "CommandType") == NS_x11::LN_CommandType;
        flag&=QName::tokenizer().insert(ns, "Maintain") == NS_x11::LN_Maintain;
        flag&=QName::tokenizer().insert(ns, "Connection") == NS_x11::LN_Connection;
        flag&=QName::tokenizer().insert(ns, "Parameter") == NS_x11::LN_Parameter;
        flag&=QName::tokenizer().insert(ns, "CommandTextOrignal") == NS_x11::LN_CommandTextOrignal;
        flag&=QName::tokenizer().insert(ns, "SourceConnectionFile") == NS_x11::LN_SourceConnectionFile;
        flag&=QName::tokenizer().insert(ns, "SourceDataFile") == NS_x11::LN_SourceDataFile;
        flag&=QName::tokenizer().insert(ns, "RobustConnect") == NS_x11::LN_RobustConnect;
        flag&=QName::tokenizer().insert(ns, "RefreshTimeSpan") == NS_x11::LN_RefreshTimeSpan;
        flag&=QName::tokenizer().insert(ns, "LocalConnection") == NS_x11::LN_LocalConnection;
        flag&=QName::tokenizer().insert(ns, "NoRefreshCache") == NS_x11::LN_NoRefreshCache;
        flag&=QName::tokenizer().insert(ns, "UseLocalConnection") == NS_x11::LN_UseLocalConnection;
        flag&=QName::tokenizer().insert(ns, "VersionLastEdit") == NS_x11::LN_VersionLastEdit;
        flag&=QName::tokenizer().insert(ns, "DoNotJoinDelimiters") == NS_x11::LN_DoNotJoinDelimiters;
        flag&=QName::tokenizer().insert(ns, "NoTextToColumns") == NS_x11::LN_NoTextToColumns;
        flag&=QName::tokenizer().insert(ns, "DisableDateRecognition") == NS_x11::LN_DisableDateRecognition;
        flag&=QName::tokenizer().insert(ns, "EnableRedirections") == NS_x11::LN_EnableRedirections;
        flag&=QName::tokenizer().insert(ns, "EditWebPage") == NS_x11::LN_EditWebPage;
        flag&=QName::tokenizer().insert(ns, "RefreshedInXl9") == NS_x11::LN_RefreshedInXl9;
        flag&=QName::tokenizer().insert(ns, "EntirePage") == NS_x11::LN_EntirePage;
        flag&=QName::tokenizer().insert(ns, "HTMLFormat") == NS_x11::LN_HTMLFormat;
        flag&=QName::tokenizer().insert(ns, "URLString") == NS_x11::LN_URLString;
        flag&=QName::tokenizer().insert(ns, "WebPostString") == NS_x11::LN_WebPostString;
        flag&=QName::tokenizer().insert(ns, "RefreshInfo") == NS_x11::LN_RefreshInfo;
        flag&=QName::tokenizer().insert(ns, "TextWizardSettings") == NS_x11::LN_TextWizardSettings;
        flag&=QName::tokenizer().insert(ns, "HTMLTables") == NS_x11::LN_HTMLTables;
        flag&=QName::tokenizer().insert(ns, "UseSameSettings") == NS_x11::LN_UseSameSettings;
        flag&=QName::tokenizer().insert(ns, "Query97") == NS_x11::LN_Query97;
        flag&=QName::tokenizer().insert(ns, "DoNotPromptForFile") == NS_x11::LN_DoNotPromptForFile;
        flag&=QName::tokenizer().insert(ns, "AutoFormatWidth") == NS_x11::LN_AutoFormatWidth;
        flag&=QName::tokenizer().insert(ns, "DisableEdit") == NS_x11::LN_DisableEdit;
        flag&=QName::tokenizer().insert(ns, "Filled") == NS_x11::LN_Filled;
        flag&=QName::tokenizer().insert(ns, "InsertEntireRows") == NS_x11::LN_InsertEntireRows;
        flag&=QName::tokenizer().insert(ns, "NewAsync") == NS_x11::LN_NewAsync;
        flag&=QName::tokenizer().insert(ns, "NoAutofit") == NS_x11::LN_NoAutofit;
        flag&=QName::tokenizer().insert(ns, "NoTitles") == NS_x11::LN_NoTitles;
        flag&=QName::tokenizer().insert(ns, "OverwriteCells") == NS_x11::LN_OverwriteCells;
        flag&=QName::tokenizer().insert(ns, "Synchronous") == NS_x11::LN_Synchronous;
        flag&=QName::tokenizer().insert(ns, "NextId") == NS_x11::LN_NextId;
        flag&=QName::tokenizer().insert(ns, "ColumnInfo") == NS_x11::LN_ColumnInfo;
        flag&=QName::tokenizer().insert(ns, "DoNotPersist") == NS_x11::LN_DoNotPersist;
        flag&=QName::tokenizer().insert(ns, "DoNotPersistSort") == NS_x11::LN_DoNotPersistSort;
        flag&=QName::tokenizer().insert(ns, "DoNotPersstAF") == NS_x11::LN_DoNotPersstAF;
        flag&=QName::tokenizer().insert(ns, "Sort") == NS_x11::LN_Sort;
        flag&=QName::tokenizer().insert(ns, "CaseSensitive") == NS_x11::LN_CaseSensitive;
        flag&=QName::tokenizer().insert(ns, "FuturePersist") == NS_x11::LN_FuturePersist;
        flag&=QName::tokenizer().insert(ns, "HeaderRow") == NS_x11::LN_HeaderRow;
        flag&=QName::tokenizer().insert(ns, "IdWrapped") == NS_x11::LN_IdWrapped;
        flag&=QName::tokenizer().insert(ns, "AlertVersion") == NS_x11::LN_AlertVersion;
        flag&=QName::tokenizer().insert(ns, "SortOrder") == NS_x11::LN_SortOrder;
        flag&=QName::tokenizer().insert(ns, "ExtraLeftColumns") == NS_x11::LN_ExtraLeftColumns;
        flag&=QName::tokenizer().insert(ns, "ExtraRightColumns") == NS_x11::LN_ExtraRightColumns;
        flag&=QName::tokenizer().insert(ns, "DeletedTitle") == NS_x11::LN_DeletedTitle;
        flag&=QName::tokenizer().insert(ns, "NoToggleDataHeader") == NS_x11::LN_NoToggleDataHeader;
        flag&=QName::tokenizer().insert(ns, "AlternateMethod") == NS_x11::LN_AlternateMethod;
        flag&=QName::tokenizer().insert(ns, "NoAutoPage") == NS_x11::LN_NoAutoPage;
        flag&=QName::tokenizer().insert(ns, "SheetName") == NS_x11::LN_SheetName;
        flag&=QName::tokenizer().insert(ns, "ExternName") == NS_x11::LN_ExternName;
        flag&=QName::tokenizer().insert(ns, "Xct") == NS_x11::LN_Xct;
        flag&=QName::tokenizer().insert(ns, "FormatSettings") == NS_x11::LN_FormatSettings;
        flag&=QName::tokenizer().insert(ns, "Delimiters") == NS_x11::LN_Delimiters;
        flag&=QName::tokenizer().insert(ns, "Comma") == NS_x11::LN_Comma;
        flag&=QName::tokenizer().insert(ns, "Consecutive") == NS_x11::LN_Consecutive;
        flag&=QName::tokenizer().insert(ns, "Custom") == NS_x11::LN_Custom;
        flag&=QName::tokenizer().insert(ns, "SemiColon") == NS_x11::LN_SemiColon;
        flag&=QName::tokenizer().insert(ns, "Space") == NS_x11::LN_Space;
        flag&=QName::tokenizer().insert(ns, "Tab") == NS_x11::LN_Tab;
        flag&=QName::tokenizer().insert(ns, "TextQualifier") == NS_x11::LN_TextQualifier;
        flag&=QName::tokenizer().insert(ns, "StartRow") == NS_x11::LN_StartRow;
        flag&=QName::tokenizer().insert(ns, "Decimal") == NS_x11::LN_Decimal;
        flag&=QName::tokenizer().insert(ns, "ThousandSeparator") == NS_x11::LN_ThousandSeparator;
        flag&=QName::tokenizer().insert(ns, "FitToPage") == NS_x11::LN_FitToPage;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayColHeaders") == NS_x11::LN_DoNotDisplayColHeaders;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayRowHeaders") == NS_x11::LN_DoNotDisplayRowHeaders;
        flag&=QName::tokenizer().insert(ns, "ViewableRange") == NS_x11::LN_ViewableRange;
        flag&=QName::tokenizer().insert(ns, "GridlineColor") == NS_x11::LN_GridlineColor;
        flag&=QName::tokenizer().insert(ns, "ExcelWorksheetType") == NS_x11::LN_ExcelWorksheetType;
        flag&=QName::tokenizer().insert(ns, "IntlMacro") == NS_x11::LN_IntlMacro;
        flag&=QName::tokenizer().insert(ns, "Unsynced") == NS_x11::LN_Unsynced;
        flag&=QName::tokenizer().insert(ns, "Selected") == NS_x11::LN_Selected;
        flag&=QName::tokenizer().insert(ns, "CodeName") == NS_x11::LN_CodeName;
        flag&=QName::tokenizer().insert(ns, "PageSetup") == NS_x11::LN_PageSetup;
        flag&=QName::tokenizer().insert(ns, "DisplayPageBreak") == NS_x11::LN_DisplayPageBreak;
        flag&=QName::tokenizer().insert(ns, "TransitionExpressionEvaluation") == NS_x11::LN_TransitionExpressionEvaluation;
        flag&=QName::tokenizer().insert(ns, "TransitionFormulaEntry") == NS_x11::LN_TransitionFormulaEntry;
        flag&=QName::tokenizer().insert(ns, "Print") == NS_x11::LN_Print;
        flag&=QName::tokenizer().insert(ns, "Zoom") == NS_x11::LN_Zoom;
        flag&=QName::tokenizer().insert(ns, "PageBreakZoom") == NS_x11::LN_PageBreakZoom;
        flag&=QName::tokenizer().insert(ns, "ShowPageBreakZoom") == NS_x11::LN_ShowPageBreakZoom;
        flag&=QName::tokenizer().insert(ns, "DefaultRowHeight") == NS_x11::LN_DefaultRowHeight;
        flag&=QName::tokenizer().insert(ns, "DefaultColumnWidth") == NS_x11::LN_DefaultColumnWidth;
        flag&=QName::tokenizer().insert(ns, "StandardWidth") == NS_x11::LN_StandardWidth;
        flag&=QName::tokenizer().insert(ns, "Visible") == NS_x11::LN_Visible;
        flag&=QName::tokenizer().insert(ns, "LeftColumnVisible") == NS_x11::LN_LeftColumnVisible;
        flag&=QName::tokenizer().insert(ns, "DisplayRightToLeft") == NS_x11::LN_DisplayRightToLeft;
        flag&=QName::tokenizer().insert(ns, "GridlineColorIndex") == NS_x11::LN_GridlineColorIndex;
        flag&=QName::tokenizer().insert(ns, "DisplayFormulas") == NS_x11::LN_DisplayFormulas;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayGridlines") == NS_x11::LN_DoNotDisplayGridlines;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayHeadings") == NS_x11::LN_DoNotDisplayHeadings;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayOutline") == NS_x11::LN_DoNotDisplayOutline;
        flag&=QName::tokenizer().insert(ns, "ApplyAutomaticOutlineStyles") == NS_x11::LN_ApplyAutomaticOutlineStyles;
        flag&=QName::tokenizer().insert(ns, "NoSummaryRowsBelowDetail") == NS_x11::LN_NoSummaryRowsBelowDetail;
        flag&=QName::tokenizer().insert(ns, "NoSummaryColumnsRightDetail") == NS_x11::LN_NoSummaryColumnsRightDetail;
        flag&=QName::tokenizer().insert(ns, "DoNotDisplayZeros") == NS_x11::LN_DoNotDisplayZeros;
        flag&=QName::tokenizer().insert(ns, "ActiveColumn") == NS_x11::LN_ActiveColumn;
        flag&=QName::tokenizer().insert(ns, "FilterOn") == NS_x11::LN_FilterOn;
        flag&=QName::tokenizer().insert(ns, "TopRowVisible") == NS_x11::LN_TopRowVisible;
        flag&=QName::tokenizer().insert(ns, "TopRowBottomPane") == NS_x11::LN_TopRowBottomPane;
        flag&=QName::tokenizer().insert(ns, "LeftColumnRightPane") == NS_x11::LN_LeftColumnRightPane;
        flag&=QName::tokenizer().insert(ns, "ActivePane") == NS_x11::LN_ActivePane;
        flag&=QName::tokenizer().insert(ns, "SplitHorizontal") == NS_x11::LN_SplitHorizontal;
        flag&=QName::tokenizer().insert(ns, "SplitVertical") == NS_x11::LN_SplitVertical;
        flag&=QName::tokenizer().insert(ns, "FreezePanes") == NS_x11::LN_FreezePanes;
        flag&=QName::tokenizer().insert(ns, "FrozenNoSplit") == NS_x11::LN_FrozenNoSplit;
        flag&=QName::tokenizer().insert(ns, "TabColorIndex") == NS_x11::LN_TabColorIndex;
        flag&=QName::tokenizer().insert(ns, "Panes") == NS_x11::LN_Panes;
        flag&=QName::tokenizer().insert(ns, "ProtectContents") == NS_x11::LN_ProtectContents;
        flag&=QName::tokenizer().insert(ns, "ProtectObjects") == NS_x11::LN_ProtectObjects;
        flag&=QName::tokenizer().insert(ns, "ProtectScenarios") == NS_x11::LN_ProtectScenarios;
        flag&=QName::tokenizer().insert(ns, "EnableSelection") == NS_x11::LN_EnableSelection;
        flag&=QName::tokenizer().insert(ns, "AllowFormatCells") == NS_x11::LN_AllowFormatCells;
        flag&=QName::tokenizer().insert(ns, "AllowSizeCols") == NS_x11::LN_AllowSizeCols;
        flag&=QName::tokenizer().insert(ns, "AllowSizeRows") == NS_x11::LN_AllowSizeRows;
        flag&=QName::tokenizer().insert(ns, "AllowInsertCols") == NS_x11::LN_AllowInsertCols;
        flag&=QName::tokenizer().insert(ns, "AllowInsertRows") == NS_x11::LN_AllowInsertRows;
        flag&=QName::tokenizer().insert(ns, "AllowInsertHyperlinks") == NS_x11::LN_AllowInsertHyperlinks;
        flag&=QName::tokenizer().insert(ns, "AllowDeleteCols") == NS_x11::LN_AllowDeleteCols;
        flag&=QName::tokenizer().insert(ns, "AllowDeleteRows") == NS_x11::LN_AllowDeleteRows;
        flag&=QName::tokenizer().insert(ns, "AllowSort") == NS_x11::LN_AllowSort;
        flag&=QName::tokenizer().insert(ns, "AllowFilter") == NS_x11::LN_AllowFilter;
        flag&=QName::tokenizer().insert(ns, "AllowUsePivotTables") == NS_x11::LN_AllowUsePivotTables;
        flag&=QName::tokenizer().insert(ns, "SpaceAbove") == NS_x11::LN_SpaceAbove;
        flag&=QName::tokenizer().insert(ns, "SpaceBelow") == NS_x11::LN_SpaceBelow;
        flag&=QName::tokenizer().insert(ns, "ZeroHeight") == NS_x11::LN_ZeroHeight;
        flag&=QName::tokenizer().insert(ns, "Layout") == NS_x11::LN_Layout;
        flag&=QName::tokenizer().insert(ns, "Header") == NS_x11::LN_Header;
        flag&=QName::tokenizer().insert(ns, "Footer") == NS_x11::LN_Footer;
        flag&=QName::tokenizer().insert(ns, "PageMargins") == NS_x11::LN_PageMargins;
        flag&=QName::tokenizer().insert(ns, "Count") == NS_x11::LN_Count;
        flag&=QName::tokenizer().insert(ns, "SheetIndex") == NS_x11::LN_SheetIndex;
        flag&=QName::tokenizer().insert(ns, "Crn") == NS_x11::LN_Crn;
        flag&=QName::tokenizer().insert(ns, "Moper") == NS_x11::LN_Moper;
        flag&=QName::tokenizer().insert(ns, "ExcelName") == NS_x11::LN_ExcelName;
        flag&=QName::tokenizer().insert(ns, "FunctionGroup") == NS_x11::LN_FunctionGroup;
        flag&=QName::tokenizer().insert(ns, "FunctionGroupIndex") == NS_x11::LN_FunctionGroupIndex;
        flag&=QName::tokenizer().insert(ns, "KeyboardShortcut") == NS_x11::LN_KeyboardShortcut;
        flag&=QName::tokenizer().insert(ns, "Macro") == NS_x11::LN_Macro;
        flag&=QName::tokenizer().insert(ns, "phonetictext") == NS_x11::LN_phonetictext;
        flag&=QName::tokenizer().insert(ns, "AutoFilterColumn") == NS_x11::LN_AutoFilterColumn;
        flag&=QName::tokenizer().insert(ns, "WorkbookOptions") == NS_x11::LN_WorkbookOptions;
        flag&=QName::tokenizer().insert(ns, "MoveAfterReturn") == NS_x11::LN_MoveAfterReturn;
        flag&=QName::tokenizer().insert(ns, "ParameterValue") == NS_x11::LN_ParameterValue;
        flag&=QName::tokenizer().insert(ns, "PromptString") == NS_x11::LN_PromptString;
        flag&=QName::tokenizer().insert(ns, "NonDefaultName") == NS_x11::LN_NonDefaultName;
        flag&=QName::tokenizer().insert(ns, "SQLType") == NS_x11::LN_SQLType;
        flag&=QName::tokenizer().insert(ns, "RefreshOnChange") == NS_x11::LN_RefreshOnChange;
        flag&=QName::tokenizer().insert(ns, "DisplayEmptyMembers") == NS_x11::LN_DisplayEmptyMembers;
        flag&=QName::tokenizer().insert(ns, "Label") == NS_x11::LN_Label;
        flag&=QName::tokenizer().insert(ns, "Caption") == NS_x11::LN_Caption;
        flag&=QName::tokenizer().insert(ns, "NotVisible") == NS_x11::LN_NotVisible;
        flag&=QName::tokenizer().insert(ns, "Orientation") == NS_x11::LN_Orientation;
        flag&=QName::tokenizer().insert(ns, "PivotCache") == NS_x11::LN_PivotCache;
        flag&=QName::tokenizer().insert(ns, "CacheIndex") == NS_x11::LN_CacheIndex;
        flag&=QName::tokenizer().insert(ns, "PTPivotData") == NS_x11::LN_PTPivotData;
        flag&=QName::tokenizer().insert(ns, "InvertedRowMember") == NS_x11::LN_InvertedRowMember;
        flag&=QName::tokenizer().insert(ns, "InvertedColumnMember") == NS_x11::LN_InvertedColumnMember;
        flag&=QName::tokenizer().insert(ns, "LeafRowMember") == NS_x11::LN_LeafRowMember;
        flag&=QName::tokenizer().insert(ns, "LeafColumnMember") == NS_x11::LN_LeafColumnMember;
        flag&=QName::tokenizer().insert(ns, "Item") == NS_x11::LN_Item;
        flag&=QName::tokenizer().insert(ns, "SourceName") == NS_x11::LN_SourceName;
        flag&=QName::tokenizer().insert(ns, "BoundField") == NS_x11::LN_BoundField;
        flag&=QName::tokenizer().insert(ns, "DataType") == NS_x11::LN_DataType;
        flag&=QName::tokenizer().insert(ns, "FilterCaption") == NS_x11::LN_FilterCaption;
        flag&=QName::tokenizer().insert(ns, "PLDataOrientation") == NS_x11::LN_PLDataOrientation;
        flag&=QName::tokenizer().insert(ns, "PLPosition") == NS_x11::LN_PLPosition;
        flag&=QName::tokenizer().insert(ns, "Position") == NS_x11::LN_Position;
        flag&=QName::tokenizer().insert(ns, "ShowAllItems") == NS_x11::LN_ShowAllItems;
        flag&=QName::tokenizer().insert(ns, "NumberFormat") == NS_x11::LN_NumberFormat;
        flag&=QName::tokenizer().insert(ns, "CurrentPage") == NS_x11::LN_CurrentPage;
        flag&=QName::tokenizer().insert(ns, "NewItemsHidden") == NS_x11::LN_NewItemsHidden;
        flag&=QName::tokenizer().insert(ns, "DataTable") == NS_x11::LN_DataTable;
        flag&=QName::tokenizer().insert(ns, "Subtotal") == NS_x11::LN_Subtotal;
        flag&=QName::tokenizer().insert(ns, "Range") == NS_x11::LN_Range;
        flag&=QName::tokenizer().insert(ns, "PLSubtotal") == NS_x11::LN_PLSubtotal;
        flag&=QName::tokenizer().insert(ns, "RowInputCell") == NS_x11::LN_RowInputCell;
        flag&=QName::tokenizer().insert(ns, "DetailFormat") == NS_x11::LN_DetailFormat;
        flag&=QName::tokenizer().insert(ns, "ColumnInputCell") == NS_x11::LN_ColumnInputCell;
        flag&=QName::tokenizer().insert(ns, "SubtotalFormat") == NS_x11::LN_SubtotalFormat;
        flag&=QName::tokenizer().insert(ns, "ExcelWorkbook") == NS_x11::LN_ExcelWorkbook;
        flag&=QName::tokenizer().insert(ns, "DetailWidth") == NS_x11::LN_DetailWidth;
        flag&=QName::tokenizer().insert(ns, "PublishObjects") == NS_x11::LN_PublishObjects;
        flag&=QName::tokenizer().insert(ns, "GroupedWidth") == NS_x11::LN_GroupedWidth;
        flag&=QName::tokenizer().insert(ns, "SupBook") == NS_x11::LN_SupBook;
        flag&=QName::tokenizer().insert(ns, "TotalWidth") == NS_x11::LN_TotalWidth;
        flag&=QName::tokenizer().insert(ns, "SelectedSheets") == NS_x11::LN_SelectedSheets;
        flag&=QName::tokenizer().insert(ns, "PivotItem") == NS_x11::LN_PivotItem;
        flag&=QName::tokenizer().insert(ns, "DataField") == NS_x11::LN_DataField;
        flag&=QName::tokenizer().insert(ns, "WindowHidden") == NS_x11::LN_WindowHidden;
        flag&=QName::tokenizer().insert(ns, "MapChildItems") == NS_x11::LN_MapChildItems;
        flag&=QName::tokenizer().insert(ns, "HideHorizontalScrollBar") == NS_x11::LN_HideHorizontalScrollBar;
        flag&=QName::tokenizer().insert(ns, "AutoShowCount") == NS_x11::LN_AutoShowCount;
        flag&=QName::tokenizer().insert(ns, "HideVerticalScrollBar") == NS_x11::LN_HideVerticalScrollBar;
        flag&=QName::tokenizer().insert(ns, "AutoShowField") == NS_x11::LN_AutoShowField;
        flag&=QName::tokenizer().insert(ns, "AutoShowRange") == NS_x11::LN_AutoShowRange;
        flag&=QName::tokenizer().insert(ns, "HideWorkbookTabs") == NS_x11::LN_HideWorkbookTabs;
        flag&=QName::tokenizer().insert(ns, "AutoShowType") == NS_x11::LN_AutoShowType;
        flag&=QName::tokenizer().insert(ns, "WindowHeight") == NS_x11::LN_WindowHeight;
        flag&=QName::tokenizer().insert(ns, "WindowWidth") == NS_x11::LN_WindowWidth;
        flag&=QName::tokenizer().insert(ns, "AutoSortOrder") == NS_x11::LN_AutoSortOrder;
        flag&=QName::tokenizer().insert(ns, "WindowTopX") == NS_x11::LN_WindowTopX;
        flag&=QName::tokenizer().insert(ns, "AutoSortField") == NS_x11::LN_AutoSortField;
        flag&=QName::tokenizer().insert(ns, "WindowTopY") == NS_x11::LN_WindowTopY;
        flag&=QName::tokenizer().insert(ns, "BaseField") == NS_x11::LN_BaseField;
        flag&=QName::tokenizer().insert(ns, "BaseItem") == NS_x11::LN_BaseItem;
        flag&=QName::tokenizer().insert(ns, "ActiveSheet") == NS_x11::LN_ActiveSheet;
        flag&=QName::tokenizer().insert(ns, "ActiveChart") == NS_x11::LN_ActiveChart;
        flag&=QName::tokenizer().insert(ns, "BlankLineAfterItems") == NS_x11::LN_BlankLineAfterItems;
        flag&=QName::tokenizer().insert(ns, "FirstVisibleSheet") == NS_x11::LN_FirstVisibleSheet;
        flag&=QName::tokenizer().insert(ns, "CantGetUniqueItems") == NS_x11::LN_CantGetUniqueItems;
        flag&=QName::tokenizer().insert(ns, "GroupBy") == NS_x11::LN_GroupBy;
        flag&=QName::tokenizer().insert(ns, "HidePivotTableFieldList") == NS_x11::LN_HidePivotTableFieldList;
        flag&=QName::tokenizer().insert(ns, "GroupStart") == NS_x11::LN_GroupStart;
        flag&=QName::tokenizer().insert(ns, "ProtectStructure") == NS_x11::LN_ProtectStructure;
        flag&=QName::tokenizer().insert(ns, "GroupEnd") == NS_x11::LN_GroupEnd;
        flag&=QName::tokenizer().insert(ns, "ProtectWindows") == NS_x11::LN_ProtectWindows;
        flag&=QName::tokenizer().insert(ns, "GroupStartAuto") == NS_x11::LN_GroupStartAuto;
        flag&=QName::tokenizer().insert(ns, "DisplayInkNotes") == NS_x11::LN_DisplayInkNotes;
        flag&=QName::tokenizer().insert(ns, "GroupEndAuto") == NS_x11::LN_GroupEndAuto;
        flag&=QName::tokenizer().insert(ns, "GroupType") == NS_x11::LN_GroupType;
        flag&=QName::tokenizer().insert(ns, "EmbedSaveSmartTags") == NS_x11::LN_EmbedSaveSmartTags;
        flag&=QName::tokenizer().insert(ns, "LayoutPageBreak") == NS_x11::LN_LayoutPageBreak;
        flag&=QName::tokenizer().insert(ns, "FutureVer") == NS_x11::LN_FutureVer;
        flag&=QName::tokenizer().insert(ns, "TabRatio") == NS_x11::LN_TabRatio;
        flag&=QName::tokenizer().insert(ns, "ParentField") == NS_x11::LN_ParentField;
        flag&=QName::tokenizer().insert(ns, "Function") == NS_x11::LN_Function;
        flag&=QName::tokenizer().insert(ns, "WindowIconic") == NS_x11::LN_WindowIconic;
        flag&=QName::tokenizer().insert(ns, "LayoutForm") == NS_x11::LN_LayoutForm;
        flag&=QName::tokenizer().insert(ns, "DisplayDrawingObjects") == NS_x11::LN_DisplayDrawingObjects;
        flag&=QName::tokenizer().insert(ns, "LayoutSubtotalLocation") == NS_x11::LN_LayoutSubtotalLocation;
        flag&=QName::tokenizer().insert(ns, "CreateBackup") == NS_x11::LN_CreateBackup;
        flag&=QName::tokenizer().insert(ns, "NoDragToRow") == NS_x11::LN_NoDragToRow;
        flag&=QName::tokenizer().insert(ns, "Calculation") == NS_x11::LN_Calculation;
        flag&=QName::tokenizer().insert(ns, "NoDragToColumn") == NS_x11::LN_NoDragToColumn;
        flag&=QName::tokenizer().insert(ns, "DoNotCalculateBeforeSave") == NS_x11::LN_DoNotCalculateBeforeSave;
        flag&=QName::tokenizer().insert(ns, "NoDragToData") == NS_x11::LN_NoDragToData;
        flag&=QName::tokenizer().insert(ns, "Date1904") == NS_x11::LN_Date1904;
        flag&=QName::tokenizer().insert(ns, "NoDragToHide") == NS_x11::LN_NoDragToHide;
        flag&=QName::tokenizer().insert(ns, "RefModeR1C1") == NS_x11::LN_RefModeR1C1;
        flag&=QName::tokenizer().insert(ns, "NoDragToPage") == NS_x11::LN_NoDragToPage;
        flag&=QName::tokenizer().insert(ns, "Iteration") == NS_x11::LN_Iteration;
        flag&=QName::tokenizer().insert(ns, "MemberProperty") == NS_x11::LN_MemberProperty;
        flag&=QName::tokenizer().insert(ns, "MaxIterations") == NS_x11::LN_MaxIterations;
        flag&=QName::tokenizer().insert(ns, "MaxChange") == NS_x11::LN_MaxChange;
        flag&=QName::tokenizer().insert(ns, "MemberPropertiesOrder") == NS_x11::LN_MemberPropertiesOrder;
        flag&=QName::tokenizer().insert(ns, "HideDropDowns") == NS_x11::LN_HideDropDowns;
        flag&=QName::tokenizer().insert(ns, "PrecisionAsDisplayed") == NS_x11::LN_PrecisionAsDisplayed;
        flag&=QName::tokenizer().insert(ns, "DrilledLevel") == NS_x11::LN_DrilledLevel;
        flag&=QName::tokenizer().insert(ns, "DoNotSaveLinkValues") == NS_x11::LN_DoNotSaveLinkValues;
        flag&=QName::tokenizer().insert(ns, "ServerSortOrder") == NS_x11::LN_ServerSortOrder;
        flag&=QName::tokenizer().insert(ns, "NoAutoRecover") == NS_x11::LN_NoAutoRecover;
        flag&=QName::tokenizer().insert(ns, "IsMemberProperty") == NS_x11::LN_IsMemberProperty;
        flag&=QName::tokenizer().insert(ns, "AcceptLabelsInFormulas") == NS_x11::LN_AcceptLabelsInFormulas;
        flag&=QName::tokenizer().insert(ns, "FilterMember") == NS_x11::LN_FilterMember;
        flag&=QName::tokenizer().insert(ns, "Uncalced") == NS_x11::LN_Uncalced;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_x11::LN_Name;
        flag&=QName::tokenizer().insert(ns, "UniqueName") == NS_x11::LN_UniqueName;
        flag&=QName::tokenizer().insert(ns, "Level") == NS_x11::LN_Level;
        flag&=QName::tokenizer().insert(ns, "DataValidation") == NS_x11::LN_DataValidation;
        flag&=QName::tokenizer().insert(ns, "ServerBased") == NS_x11::LN_ServerBased;
        flag&=QName::tokenizer().insert(ns, "AutoFilter") == NS_x11::LN_AutoFilter;
        flag&=QName::tokenizer().insert(ns, "SubtotalName") == NS_x11::LN_SubtotalName;
        flag&=QName::tokenizer().insert(ns, "PLTPivotItem") == NS_x11::LN_PLTPivotItem;
        flag&=QName::tokenizer().insert(ns, "AutoFilterCondition") == NS_x11::LN_AutoFilterCondition;
        flag&=QName::tokenizer().insert(ns, "FormulaV10") == NS_x11::LN_FormulaV10;
        flag&=QName::tokenizer().insert(ns, "AutoFilterOr") == NS_x11::LN_AutoFilterOr;
        flag&=QName::tokenizer().insert(ns, "ParseFormulaAsV10") == NS_x11::LN_ParseFormulaAsV10;
        flag&=QName::tokenizer().insert(ns, "AutoFilterAnd") == NS_x11::LN_AutoFilterAnd;
        flag&=QName::tokenizer().insert(ns, "PLCaption") == NS_x11::LN_PLCaption;
        flag&=QName::tokenizer().insert(ns, "Sorting") == NS_x11::LN_Sorting;
        flag&=QName::tokenizer().insert(ns, "PLGroupType") == NS_x11::LN_PLGroupType;
        flag&=QName::tokenizer().insert(ns, "ConditionalFormatting") == NS_x11::LN_ConditionalFormatting;
        flag&=QName::tokenizer().insert(ns, "PLName") == NS_x11::LN_PLName;
        flag&=QName::tokenizer().insert(ns, "QueryTable") == NS_x11::LN_QueryTable;
        flag&=QName::tokenizer().insert(ns, "SourceHierarchy") == NS_x11::LN_SourceHierarchy;
        flag&=QName::tokenizer().insert(ns, "PivotTable") == NS_x11::LN_PivotTable;
        flag&=QName::tokenizer().insert(ns, "MemberPropertyParent") == NS_x11::LN_MemberPropertyParent;
        flag&=QName::tokenizer().insert(ns, "WorksheetSource") == NS_x11::LN_WorksheetSource;
        flag&=QName::tokenizer().insert(ns, "SourceHierarchyLevel") == NS_x11::LN_SourceHierarchyLevel;
        flag&=QName::tokenizer().insert(ns, "WorksheetOptions") == NS_x11::LN_WorksheetOptions;
        flag&=QName::tokenizer().insert(ns, "Watches") == NS_x11::LN_Watches;
        flag&=QName::tokenizer().insert(ns, "TotalAlignment") == NS_x11::LN_TotalAlignment;
        flag&=QName::tokenizer().insert(ns, "Watch") == NS_x11::LN_Watch;
        flag&=QName::tokenizer().insert(ns, "Source") == NS_x11::LN_Source;
        flag&=QName::tokenizer().insert(ns, "TotalCaptionAlignment") == NS_x11::LN_TotalCaptionAlignment;
        flag&=QName::tokenizer().insert(ns, "PageBreaks") == NS_x11::LN_PageBreaks;
        flag&=QName::tokenizer().insert(ns, "FormulaIndex") == NS_x11::LN_FormulaIndex;
        flag&=QName::tokenizer().insert(ns, "ColBreaks") == NS_x11::LN_ColBreaks;
        flag&=QName::tokenizer().insert(ns, "Hidden") == NS_x11::LN_Hidden;
        flag&=QName::tokenizer().insert(ns, "ColBreak") == NS_x11::LN_ColBreak;
        flag&=QName::tokenizer().insert(ns, "HideDetail") == NS_x11::LN_HideDetail;
        flag&=QName::tokenizer().insert(ns, "Column") == NS_x11::LN_Column;
        flag&=QName::tokenizer().insert(ns, "CachePosition") == NS_x11::LN_CachePosition;
        flag&=QName::tokenizer().insert(ns, "RowBreaks") == NS_x11::LN_RowBreaks;
        flag&=QName::tokenizer().insert(ns, "Member") == NS_x11::LN_Member;
        flag&=QName::tokenizer().insert(ns, "RowBreak") == NS_x11::LN_RowBreak;
        flag&=QName::tokenizer().insert(ns, "Row") == NS_x11::LN_Row;
        flag&=QName::tokenizer().insert(ns, "DrilledMember") == NS_x11::LN_DrilledMember;
        flag&=QName::tokenizer().insert(ns, "ColFirst") == NS_x11::LN_ColFirst;
        flag&=QName::tokenizer().insert(ns, "DataValueEditing") == NS_x11::LN_DataValueEditing;
        flag&=QName::tokenizer().insert(ns, "ColLast") == NS_x11::LN_ColLast;
        flag&=QName::tokenizer().insert(ns, "Number") == NS_x11::LN_Number;
        flag&=QName::tokenizer().insert(ns, "QTSource") == NS_x11::LN_QTSource;
        flag&=QName::tokenizer().insert(ns, "Text") == NS_x11::LN_Text;
        flag&=QName::tokenizer().insert(ns, "Boolean") == NS_x11::LN_Boolean;
        flag&=QName::tokenizer().insert(ns, "ErrorString") == NS_x11::LN_ErrorString;
        flag&=QName::tokenizer().insert(ns, "Error") == NS_x11::LN_Error;
        flag&=QName::tokenizer().insert(ns, "NullString") == NS_x11::LN_NullString;
        flag&=QName::tokenizer().insert(ns, "Horizontal") == NS_x11::LN_Horizontal;
        flag&=QName::tokenizer().insert(ns, "PageFieldOrder") == NS_x11::LN_PageFieldOrder;
        flag&=QName::tokenizer().insert(ns, "Rotation") == NS_x11::LN_Rotation;
        flag&=QName::tokenizer().insert(ns, "Vertical") == NS_x11::LN_Vertical;
        flag&=QName::tokenizer().insert(ns, "PageFieldWrapCount") == NS_x11::LN_PageFieldWrapCount;
        flag&=QName::tokenizer().insert(ns, "ReadingOrder") == NS_x11::LN_ReadingOrder;
        flag&=QName::tokenizer().insert(ns, "DisableDrillDown") == NS_x11::LN_DisableDrillDown;
        flag&=QName::tokenizer().insert(ns, "Path") == NS_x11::LN_Path;
        flag&=QName::tokenizer().insert(ns, "Left") == NS_x11::LN_Left;
        flag&=QName::tokenizer().insert(ns, "DisableFieldDialog") == NS_x11::LN_DisableFieldDialog;
        flag&=QName::tokenizer().insert(ns, "Cell") == NS_x11::LN_Cell;
        flag&=QName::tokenizer().insert(ns, "Top") == NS_x11::LN_Top;
        flag&=QName::tokenizer().insert(ns, "DisableWizard") == NS_x11::LN_DisableWizard;
        flag&=QName::tokenizer().insert(ns, "NotInverted") == NS_x11::LN_NotInverted;
        flag&=QName::tokenizer().insert(ns, "Id") == NS_x11::LN_Id;
        flag&=QName::tokenizer().insert(ns, "DisplayErrorString") == NS_x11::LN_DisplayErrorString;
        flag&=QName::tokenizer().insert(ns, "Clipped") == NS_x11::LN_Clipped;
        flag&=QName::tokenizer().insert(ns, "NoPreserveFormatting") == NS_x11::LN_NoPreserveFormatting;
        flag&=QName::tokenizer().insert(ns, "Descending") == NS_x11::LN_Descending;
        flag&=QName::tokenizer().insert(ns, "FillDown") == NS_x11::LN_FillDown;
        flag&=QName::tokenizer().insert(ns, "HasNoAutoFormat") == NS_x11::LN_HasNoAutoFormat;
        flag&=QName::tokenizer().insert(ns, "RowNumbers") == NS_x11::LN_RowNumbers;
        flag&=QName::tokenizer().insert(ns, "SortKey") == NS_x11::LN_SortKey;
        flag&=QName::tokenizer().insert(ns, "User") == NS_x11::LN_User;
        flag&=QName::tokenizer().insert(ns, "SubtotalHiddenPageItems") == NS_x11::LN_SubtotalHiddenPageItems;
        flag&=QName::tokenizer().insert(ns, "Condition") == NS_x11::LN_Condition;
        flag&=QName::tokenizer().insert(ns, "NoAutoFormatWidth") == NS_x11::LN_NoAutoFormatWidth;
        flag&=QName::tokenizer().insert(ns, "Qualifier") == NS_x11::LN_Qualifier;
        flag&=QName::tokenizer().insert(ns, "NoRowGrand") == NS_x11::LN_NoRowGrand;
        flag&=QName::tokenizer().insert(ns, "Value1") == NS_x11::LN_Value1;
        flag&=QName::tokenizer().insert(ns, "Value2") == NS_x11::LN_Value2;
        flag&=QName::tokenizer().insert(ns, "NoColumnGrand") == NS_x11::LN_NoColumnGrand;
        flag&=QName::tokenizer().insert(ns, "Format") == NS_x11::LN_Format;
        flag&=QName::tokenizer().insert(ns, "FileName") == NS_x11::LN_FileName;
        flag&=QName::tokenizer().insert(ns, "NoDisplayNullString") == NS_x11::LN_NoDisplayNullString;
        flag&=QName::tokenizer().insert(ns, "Reference") == NS_x11::LN_Reference;
        flag&=QName::tokenizer().insert(ns, "Type") == NS_x11::LN_Type;
        flag&=QName::tokenizer().insert(ns, "NoPrintRepeatItems") == NS_x11::LN_NoPrintRepeatItems;
        flag&=QName::tokenizer().insert(ns, "CellRangeList") == NS_x11::LN_CellRangeList;
        flag&=QName::tokenizer().insert(ns, "PivotView") == NS_x11::LN_PivotView;
        flag&=QName::tokenizer().insert(ns, "UseBlank") == NS_x11::LN_UseBlank;
        flag&=QName::tokenizer().insert(ns, "Min") == NS_x11::LN_Min;
        flag&=QName::tokenizer().insert(ns, "AutoFormatAlignment") == NS_x11::LN_AutoFormatAlignment;
        flag&=QName::tokenizer().insert(ns, "Max") == NS_x11::LN_Max;
        flag&=QName::tokenizer().insert(ns, "Value") == NS_x11::LN_Value;
        flag&=QName::tokenizer().insert(ns, "AutoFormatBorder") == NS_x11::LN_AutoFormatBorder;
        flag&=QName::tokenizer().insert(ns, "ComboHide") == NS_x11::LN_ComboHide;
        flag&=QName::tokenizer().insert(ns, "AutoFormatFont") == NS_x11::LN_AutoFormatFont;
        flag&=QName::tokenizer().insert(ns, "IMEMode") == NS_x11::LN_IMEMode;
        flag&=QName::tokenizer().insert(ns, "AutoFormatName") == NS_x11::LN_AutoFormatName;
        flag&=QName::tokenizer().insert(ns, "InputHide") == NS_x11::LN_InputHide;
        flag&=QName::tokenizer().insert(ns, "InputTitle") == NS_x11::LN_InputTitle;
        flag&=QName::tokenizer().insert(ns, "AutoFormatNumber") == NS_x11::LN_AutoFormatNumber;
        flag&=QName::tokenizer().insert(ns, "InputMessage") == NS_x11::LN_InputMessage;
        flag&=QName::tokenizer().insert(ns, "AutoFormatPattern") == NS_x11::LN_AutoFormatPattern;
        flag&=QName::tokenizer().insert(ns, "ErrorHide") == NS_x11::LN_ErrorHide;
        flag&=QName::tokenizer().insert(ns, "GrandTotalString") == NS_x11::LN_GrandTotalString;
        flag&=QName::tokenizer().insert(ns, "ErrorStyle") == NS_x11::LN_ErrorStyle;
        flag&=QName::tokenizer().insert(ns, "ErrorMessage") == NS_x11::LN_ErrorMessage;
        flag&=QName::tokenizer().insert(ns, "PrintSetTitles") == NS_x11::LN_PrintSetTitles;
        flag&=QName::tokenizer().insert(ns, "ErrorTitle") == NS_x11::LN_ErrorTitle;
        flag&=QName::tokenizer().insert(ns, "MergeLabels") == NS_x11::LN_MergeLabels;
        flag&=QName::tokenizer().insert(ns, "Formula") == NS_x11::LN_Formula;
        flag&=QName::tokenizer().insert(ns, "OleLink") == NS_x11::LN_OleLink;
        flag&=QName::tokenizer().insert(ns, "PageFieldStyle") == NS_x11::LN_PageFieldStyle;
        flag&=QName::tokenizer().insert(ns, "PLExport") == NS_x11::LN_PLExport;
        flag&=QName::tokenizer().insert(ns, "WantAdvise") == NS_x11::LN_WantAdvise;
        flag&=QName::tokenizer().insert(ns, "WantPict") == NS_x11::LN_WantPict;
        flag&=QName::tokenizer().insert(ns, "ImmediateItemsOnDrop") == NS_x11::LN_ImmediateItemsOnDrop;
        flag&=QName::tokenizer().insert(ns, "Cf") == NS_x11::LN_Cf;
        flag&=QName::tokenizer().insert(ns, "FieldType") == NS_x11::LN_FieldType;
        flag&=QName::tokenizer().insert(ns, "ShowPageMultipleItemLabel") == NS_x11::LN_ShowPageMultipleItemLabel;
        flag&=QName::tokenizer().insert(ns, "FieldStart") == NS_x11::LN_FieldStart;
        flag&=QName::tokenizer().insert(ns, "SmallGrid") == NS_x11::LN_SmallGrid;
        flag&=QName::tokenizer().insert(ns, "Missing") == NS_x11::LN_Missing;
        flag&=QName::tokenizer().insert(ns, "SeqNum") == NS_x11::LN_SeqNum;
        flag&=QName::tokenizer().insert(ns, "TableStyle") == NS_x11::LN_TableStyle;
        flag&=QName::tokenizer().insert(ns, "Tag") == NS_x11::LN_Tag;
        flag&=QName::tokenizer().insert(ns, "Expanded") == NS_x11::LN_Expanded;
        flag&=QName::tokenizer().insert(ns, "RowLast") == NS_x11::LN_RowLast;
        flag&=QName::tokenizer().insert(ns, "VacatedStyle") == NS_x11::LN_VacatedStyle;
        flag&=QName::tokenizer().insert(ns, "Pane") == NS_x11::LN_Pane;
        flag&=QName::tokenizer().insert(ns, "ActiveRow") == NS_x11::LN_ActiveRow;
        flag&=QName::tokenizer().insert(ns, "HideTotalsAnnotation") == NS_x11::LN_HideTotalsAnnotation;
        flag&=QName::tokenizer().insert(ns, "ActiveCol") == NS_x11::LN_ActiveCol;
        flag&=QName::tokenizer().insert(ns, "ShowCellBackgroundFromOLAP") == NS_x11::LN_ShowCellBackgroundFromOLAP;
        flag&=QName::tokenizer().insert(ns, "RangeSelection") == NS_x11::LN_RangeSelection;
        flag&=QName::tokenizer().insert(ns, "Selection") == NS_x11::LN_Selection;
        flag&=QName::tokenizer().insert(ns, "NoViewCalculatedMembers") == NS_x11::LN_NoViewCalculatedMembers;
        flag&=QName::tokenizer().insert(ns, "ExcelType") == NS_x11::LN_ExcelType;
        flag&=QName::tokenizer().insert(ns, "VisualTotals") == NS_x11::LN_VisualTotals;
        flag&=QName::tokenizer().insert(ns, "ParameterType") == NS_x11::LN_ParameterType;
        flag&=QName::tokenizer().insert(ns, "Location") == NS_x11::LN_Location;
        flag&=QName::tokenizer().insert(ns, "PLPivotField") == NS_x11::LN_PLPivotField;
        flag&=QName::tokenizer().insert(ns, "PLTotal") == NS_x11::LN_PLTotal;
        flag&=QName::tokenizer().insert(ns, "DataMember") == NS_x11::LN_DataMember;
        flag&=QName::tokenizer().insert(ns, "DataAxisEmpty") == NS_x11::LN_DataAxisEmpty;
        flag&=QName::tokenizer().insert(ns, "DisplayFieldList") == NS_x11::LN_DisplayFieldList;
        flag&=QName::tokenizer().insert(ns, "NoAutoFit") == NS_x11::LN_NoAutoFit;
        flag&=QName::tokenizer().insert(ns, "Height") == NS_x11::LN_Height;
        flag&=QName::tokenizer().insert(ns, "Width") == NS_x11::LN_Width;
        flag&=QName::tokenizer().insert(ns, "MaxHeight") == NS_x11::LN_MaxHeight;
        flag&=QName::tokenizer().insert(ns, "MaxWidth") == NS_x11::LN_MaxWidth;
        flag&=QName::tokenizer().insert(ns, "MemberExpand") == NS_x11::LN_MemberExpand;
        flag&=QName::tokenizer().insert(ns, "ConnectionString") == NS_x11::LN_ConnectionString;
        CPPUNIT_ASSERT_MESSAGE("test for namespace x11 failed", flag);}
    void test_ss11(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:office:spreadsheet";
        flag&=QName::tokenizer().insert(ns, "Workbook") == NS_ss11::LN_Workbook;
        flag&=QName::tokenizer().insert(ns, "Styles") == NS_ss11::LN_Styles;
        flag&=QName::tokenizer().insert(ns, "Style") == NS_ss11::LN_Style;
        flag&=QName::tokenizer().insert(ns, "Names") == NS_ss11::LN_Names;
        flag&=QName::tokenizer().insert(ns, "Worksheet") == NS_ss11::LN_Worksheet;
        flag&=QName::tokenizer().insert(ns, "NamedRange") == NS_ss11::LN_NamedRange;
        flag&=QName::tokenizer().insert(ns, "Alignment") == NS_ss11::LN_Alignment;
        flag&=QName::tokenizer().insert(ns, "Borders") == NS_ss11::LN_Borders;
        flag&=QName::tokenizer().insert(ns, "Font") == NS_ss11::LN_Font;
        flag&=QName::tokenizer().insert(ns, "Interior") == NS_ss11::LN_Interior;
        flag&=QName::tokenizer().insert(ns, "NumberFormat") == NS_ss11::LN_NumberFormat;
        flag&=QName::tokenizer().insert(ns, "Protection") == NS_ss11::LN_Protection;
        flag&=QName::tokenizer().insert(ns, "Border") == NS_ss11::LN_Border;
        flag&=QName::tokenizer().insert(ns, "Column") == NS_ss11::LN_Column;
        flag&=QName::tokenizer().insert(ns, "Row") == NS_ss11::LN_Row;
        flag&=QName::tokenizer().insert(ns, "Cell") == NS_ss11::LN_Cell;
        flag&=QName::tokenizer().insert(ns, "Data") == NS_ss11::LN_Data;
        flag&=QName::tokenizer().insert(ns, "Comment") == NS_ss11::LN_Comment;
        flag&=QName::tokenizer().insert(ns, "NamedCell") == NS_ss11::LN_NamedCell;
        flag&=QName::tokenizer().insert(ns, "Table") == NS_ss11::LN_Table;
        CPPUNIT_ASSERT_MESSAGE("test for namespace ss11 failed", flag);}
    void test_rtf(){
        bool flag=true;
        static const char ns[] = "urn:rtf:1.7";
        flag&=QName::tokenizer().insert(ns, "tsvertalb") == NS_rtf::LN_tsvertalb;
        flag&=QName::tokenizer().insert(ns, "tsvertalc") == NS_rtf::LN_tsvertalc;
        flag&=QName::tokenizer().insert(ns, "tsvertalt") == NS_rtf::LN_tsvertalt;
        flag&=QName::tokenizer().insert(ns, "twoonone") == NS_rtf::LN_twoonone;
        flag&=QName::tokenizer().insert(ns, "tx") == NS_rtf::LN_tx;
        flag&=QName::tokenizer().insert(ns, "txe") == NS_rtf::LN_txe;
        flag&=QName::tokenizer().insert(ns, "uc") == NS_rtf::LN_uc;
        flag&=QName::tokenizer().insert(ns, "ud") == NS_rtf::LN_ud;
        flag&=QName::tokenizer().insert(ns, "ul") == NS_rtf::LN_ul;
        flag&=QName::tokenizer().insert(ns, "ulc") == NS_rtf::LN_ulc;
        flag&=QName::tokenizer().insert(ns, "uld") == NS_rtf::LN_uld;
        flag&=QName::tokenizer().insert(ns, "uldash") == NS_rtf::LN_uldash;
        flag&=QName::tokenizer().insert(ns, "uldashd") == NS_rtf::LN_uldashd;
        flag&=QName::tokenizer().insert(ns, "uldashdd") == NS_rtf::LN_uldashdd;
        flag&=QName::tokenizer().insert(ns, "uldb") == NS_rtf::LN_uldb;
        flag&=QName::tokenizer().insert(ns, "ulhair") == NS_rtf::LN_ulhair;
        flag&=QName::tokenizer().insert(ns, "ulhwave") == NS_rtf::LN_ulhwave;
        flag&=QName::tokenizer().insert(ns, "ulldash") == NS_rtf::LN_ulldash;
        flag&=QName::tokenizer().insert(ns, "ulnone") == NS_rtf::LN_ulnone;
        flag&=QName::tokenizer().insert(ns, "ulth") == NS_rtf::LN_ulth;
        flag&=QName::tokenizer().insert(ns, "ulthd") == NS_rtf::LN_ulthd;
        flag&=QName::tokenizer().insert(ns, "ulthdash") == NS_rtf::LN_ulthdash;
        flag&=QName::tokenizer().insert(ns, "ulthdashd") == NS_rtf::LN_ulthdashd;
        flag&=QName::tokenizer().insert(ns, "ulthdashdd") == NS_rtf::LN_ulthdashdd;
        flag&=QName::tokenizer().insert(ns, "ulthldash") == NS_rtf::LN_ulthldash;
        flag&=QName::tokenizer().insert(ns, "ululdbwave") == NS_rtf::LN_ululdbwave;
        flag&=QName::tokenizer().insert(ns, "ulw") == NS_rtf::LN_ulw;
        flag&=QName::tokenizer().insert(ns, "ulwave") == NS_rtf::LN_ulwave;
        flag&=QName::tokenizer().insert(ns, "u") == NS_rtf::LN_u;
        flag&=QName::tokenizer().insert(ns, "up") == NS_rtf::LN_up;
        flag&=QName::tokenizer().insert(ns, "upr") == NS_rtf::LN_upr;
        flag&=QName::tokenizer().insert(ns, "urtf") == NS_rtf::LN_urtf;
        flag&=QName::tokenizer().insert(ns, "useltbaln") == NS_rtf::LN_useltbaln;
        flag&=QName::tokenizer().insert(ns, "userprops") == NS_rtf::LN_userprops;
        flag&=QName::tokenizer().insert(ns, "v") == NS_rtf::LN_v;
        flag&=QName::tokenizer().insert(ns, "vern") == NS_rtf::LN_vern;
        flag&=QName::tokenizer().insert(ns, "version") == NS_rtf::LN_version;
        flag&=QName::tokenizer().insert(ns, "vertalb") == NS_rtf::LN_vertalb;
        flag&=QName::tokenizer().insert(ns, "vertalc") == NS_rtf::LN_vertalc;
        flag&=QName::tokenizer().insert(ns, "vertalj") == NS_rtf::LN_vertalj;
        flag&=QName::tokenizer().insert(ns, "vertalt") == NS_rtf::LN_vertalt;
        flag&=QName::tokenizer().insert(ns, "vertdoc") == NS_rtf::LN_vertdoc;
        flag&=QName::tokenizer().insert(ns, "vertsect") == NS_rtf::LN_vertsect;
        flag&=QName::tokenizer().insert(ns, "viewkind") == NS_rtf::LN_viewkind;
        flag&=QName::tokenizer().insert(ns, "viewnobound") == NS_rtf::LN_viewnobound;
        flag&=QName::tokenizer().insert(ns, "viewscale") == NS_rtf::LN_viewscale;
        flag&=QName::tokenizer().insert(ns, "viewzk") == NS_rtf::LN_viewzk;
        flag&=QName::tokenizer().insert(ns, "wbitmap") == NS_rtf::LN_wbitmap;
        flag&=QName::tokenizer().insert(ns, "wbmbitspixel") == NS_rtf::LN_wbmbitspixel;
        flag&=QName::tokenizer().insert(ns, "wbmplanes") == NS_rtf::LN_wbmplanes;
        flag&=QName::tokenizer().insert(ns, "wbmwidthbytes") == NS_rtf::LN_wbmwidthbytes;
        flag&=QName::tokenizer().insert(ns, "webhidden") == NS_rtf::LN_webhidden;
        flag&=QName::tokenizer().insert(ns, "widctlpar") == NS_rtf::LN_widctlpar;
        flag&=QName::tokenizer().insert(ns, "widowctrl") == NS_rtf::LN_widowctrl;
        flag&=QName::tokenizer().insert(ns, "windowcaption") == NS_rtf::LN_windowcaption;
        flag&=QName::tokenizer().insert(ns, "wmetafile") == NS_rtf::LN_wmetafile;
        flag&=QName::tokenizer().insert(ns, "wpeqn") == NS_rtf::LN_wpeqn;
        flag&=QName::tokenizer().insert(ns, "wpjst") == NS_rtf::LN_wpjst;
        flag&=QName::tokenizer().insert(ns, "wpsp") == NS_rtf::LN_wpsp;
        flag&=QName::tokenizer().insert(ns, "wraptrsp") == NS_rtf::LN_wraptrsp;
        flag&=QName::tokenizer().insert(ns, "wrppunct") == NS_rtf::LN_wrppunct;
        flag&=QName::tokenizer().insert(ns, "xe") == NS_rtf::LN_xe;
        flag&=QName::tokenizer().insert(ns, "xef") == NS_rtf::LN_xef;
        flag&=QName::tokenizer().insert(ns, "yr") == NS_rtf::LN_yr;
        flag&=QName::tokenizer().insert(ns, "yts") == NS_rtf::LN_yts;
        flag&=QName::tokenizer().insert(ns, "yxe") == NS_rtf::LN_yxe;
        flag&=QName::tokenizer().insert(ns, "zwbo") == NS_rtf::LN_zwbo;
        flag&=QName::tokenizer().insert(ns, "zwj") == NS_rtf::LN_zwj;
        flag&=QName::tokenizer().insert(ns, "zwnbo") == NS_rtf::LN_zwnbo;
        flag&=QName::tokenizer().insert(ns, "zwnj") == NS_rtf::LN_zwnj;
        flag&=QName::tokenizer().insert(ns, "dobypara") == NS_rtf::LN_dobypara;
        flag&=QName::tokenizer().insert(ns, "doccomm") == NS_rtf::LN_doccomm;
        flag&=QName::tokenizer().insert(ns, "doctemp") == NS_rtf::LN_doctemp;
        flag&=QName::tokenizer().insert(ns, "doctype") == NS_rtf::LN_doctype;
        flag&=QName::tokenizer().insert(ns, "docvar") == NS_rtf::LN_docvar;
        flag&=QName::tokenizer().insert(ns, "dodhgt") == NS_rtf::LN_dodhgt;
        flag&=QName::tokenizer().insert(ns, "dolock") == NS_rtf::LN_dolock;
        flag&=QName::tokenizer().insert(ns, "donotshowcomments") == NS_rtf::LN_donotshowcomments;
        flag&=QName::tokenizer().insert(ns, "donotshowinsdel") == NS_rtf::LN_donotshowinsdel;
        flag&=QName::tokenizer().insert(ns, "donotshowmarkup") == NS_rtf::LN_donotshowmarkup;
        flag&=QName::tokenizer().insert(ns, "donotshowprops") == NS_rtf::LN_donotshowprops;
        flag&=QName::tokenizer().insert(ns, "dpaendhol") == NS_rtf::LN_dpaendhol;
        flag&=QName::tokenizer().insert(ns, "dpaendl") == NS_rtf::LN_dpaendl;
        flag&=QName::tokenizer().insert(ns, "dpaendsol") == NS_rtf::LN_dpaendsol;
        flag&=QName::tokenizer().insert(ns, "dpaendw") == NS_rtf::LN_dpaendw;
        flag&=QName::tokenizer().insert(ns, "dparc") == NS_rtf::LN_dparc;
        flag&=QName::tokenizer().insert(ns, "dparcflipx") == NS_rtf::LN_dparcflipx;
        flag&=QName::tokenizer().insert(ns, "dparcflipy") == NS_rtf::LN_dparcflipy;
        flag&=QName::tokenizer().insert(ns, "dpastarthol") == NS_rtf::LN_dpastarthol;
        flag&=QName::tokenizer().insert(ns, "dpastartl") == NS_rtf::LN_dpastartl;
        flag&=QName::tokenizer().insert(ns, "dpastartsol") == NS_rtf::LN_dpastartsol;
        flag&=QName::tokenizer().insert(ns, "dpastartw") == NS_rtf::LN_dpastartw;
        flag&=QName::tokenizer().insert(ns, "dpcallout") == NS_rtf::LN_dpcallout;
        flag&=QName::tokenizer().insert(ns, "dpcoa") == NS_rtf::LN_dpcoa;
        flag&=QName::tokenizer().insert(ns, "dpcoaccent") == NS_rtf::LN_dpcoaccent;
        flag&=QName::tokenizer().insert(ns, "dpcobestfit") == NS_rtf::LN_dpcobestfit;
        flag&=QName::tokenizer().insert(ns, "dpcoborder") == NS_rtf::LN_dpcoborder;
        flag&=QName::tokenizer().insert(ns, "dpcodabs") == NS_rtf::LN_dpcodabs;
        flag&=QName::tokenizer().insert(ns, "dpcodbottom") == NS_rtf::LN_dpcodbottom;
        flag&=QName::tokenizer().insert(ns, "dpcodcenter") == NS_rtf::LN_dpcodcenter;
        flag&=QName::tokenizer().insert(ns, "dpcodescent") == NS_rtf::LN_dpcodescent;
        flag&=QName::tokenizer().insert(ns, "dpcodtop") == NS_rtf::LN_dpcodtop;
        flag&=QName::tokenizer().insert(ns, "dpcolength") == NS_rtf::LN_dpcolength;
        flag&=QName::tokenizer().insert(ns, "dpcominusx") == NS_rtf::LN_dpcominusx;
        flag&=QName::tokenizer().insert(ns, "dpcominusy") == NS_rtf::LN_dpcominusy;
        flag&=QName::tokenizer().insert(ns, "dpcooffset") == NS_rtf::LN_dpcooffset;
        flag&=QName::tokenizer().insert(ns, "dpcosmarta") == NS_rtf::LN_dpcosmarta;
        flag&=QName::tokenizer().insert(ns, "dpcotdouble") == NS_rtf::LN_dpcotdouble;
        flag&=QName::tokenizer().insert(ns, "dpcotright") == NS_rtf::LN_dpcotright;
        flag&=QName::tokenizer().insert(ns, "dpcotsingle") == NS_rtf::LN_dpcotsingle;
        flag&=QName::tokenizer().insert(ns, "dpcottriple") == NS_rtf::LN_dpcottriple;
        flag&=QName::tokenizer().insert(ns, "dpcount") == NS_rtf::LN_dpcount;
        flag&=QName::tokenizer().insert(ns, "dpellipse") == NS_rtf::LN_dpellipse;
        flag&=QName::tokenizer().insert(ns, "dpendgroup") == NS_rtf::LN_dpendgroup;
        flag&=QName::tokenizer().insert(ns, "dpfillbgcb") == NS_rtf::LN_dpfillbgcb;
        flag&=QName::tokenizer().insert(ns, "dpfillbgcg") == NS_rtf::LN_dpfillbgcg;
        flag&=QName::tokenizer().insert(ns, "dpfillbgcr") == NS_rtf::LN_dpfillbgcr;
        flag&=QName::tokenizer().insert(ns, "dpfillbggray") == NS_rtf::LN_dpfillbggray;
        flag&=QName::tokenizer().insert(ns, "dpfillbgpal") == NS_rtf::LN_dpfillbgpal;
        flag&=QName::tokenizer().insert(ns, "dpfillfgcb") == NS_rtf::LN_dpfillfgcb;
        flag&=QName::tokenizer().insert(ns, "dpfillfgcg") == NS_rtf::LN_dpfillfgcg;
        flag&=QName::tokenizer().insert(ns, "dpfillfgcr") == NS_rtf::LN_dpfillfgcr;
        flag&=QName::tokenizer().insert(ns, "dpfillfggray") == NS_rtf::LN_dpfillfggray;
        flag&=QName::tokenizer().insert(ns, "dpfillfgpal") == NS_rtf::LN_dpfillfgpal;
        flag&=QName::tokenizer().insert(ns, "dpfillpat") == NS_rtf::LN_dpfillpat;
        flag&=QName::tokenizer().insert(ns, "dpgroup") == NS_rtf::LN_dpgroup;
        flag&=QName::tokenizer().insert(ns, "dpline") == NS_rtf::LN_dpline;
        flag&=QName::tokenizer().insert(ns, "dplinecob") == NS_rtf::LN_dplinecob;
        flag&=QName::tokenizer().insert(ns, "dplinecog") == NS_rtf::LN_dplinecog;
        flag&=QName::tokenizer().insert(ns, "dplinecor") == NS_rtf::LN_dplinecor;
        flag&=QName::tokenizer().insert(ns, "dplinedado") == NS_rtf::LN_dplinedado;
        flag&=QName::tokenizer().insert(ns, "dplinedadodo") == NS_rtf::LN_dplinedadodo;
        flag&=QName::tokenizer().insert(ns, "dplinedash") == NS_rtf::LN_dplinedash;
        flag&=QName::tokenizer().insert(ns, "dplinedot") == NS_rtf::LN_dplinedot;
        flag&=QName::tokenizer().insert(ns, "dplinegray") == NS_rtf::LN_dplinegray;
        flag&=QName::tokenizer().insert(ns, "dplinehollow") == NS_rtf::LN_dplinehollow;
        flag&=QName::tokenizer().insert(ns, "dplinepal") == NS_rtf::LN_dplinepal;
        flag&=QName::tokenizer().insert(ns, "dplinesolid") == NS_rtf::LN_dplinesolid;
        flag&=QName::tokenizer().insert(ns, "dplinew") == NS_rtf::LN_dplinew;
        flag&=QName::tokenizer().insert(ns, "dppolycount") == NS_rtf::LN_dppolycount;
        flag&=QName::tokenizer().insert(ns, "dppolygon") == NS_rtf::LN_dppolygon;
        flag&=QName::tokenizer().insert(ns, "dppolyline") == NS_rtf::LN_dppolyline;
        flag&=QName::tokenizer().insert(ns, "dpptx") == NS_rtf::LN_dpptx;
        flag&=QName::tokenizer().insert(ns, "dppty") == NS_rtf::LN_dppty;
        flag&=QName::tokenizer().insert(ns, "dprect") == NS_rtf::LN_dprect;
        flag&=QName::tokenizer().insert(ns, "dproundr") == NS_rtf::LN_dproundr;
        flag&=QName::tokenizer().insert(ns, "dpshadow") == NS_rtf::LN_dpshadow;
        flag&=QName::tokenizer().insert(ns, "dpshadx") == NS_rtf::LN_dpshadx;
        flag&=QName::tokenizer().insert(ns, "dpshady") == NS_rtf::LN_dpshady;
        flag&=QName::tokenizer().insert(ns, "dptxbtlr") == NS_rtf::LN_dptxbtlr;
        flag&=QName::tokenizer().insert(ns, "dptxbx") == NS_rtf::LN_dptxbx;
        flag&=QName::tokenizer().insert(ns, "dptxbxmar") == NS_rtf::LN_dptxbxmar;
        flag&=QName::tokenizer().insert(ns, "dptxbxtext") == NS_rtf::LN_dptxbxtext;
        flag&=QName::tokenizer().insert(ns, "dptxlrtb") == NS_rtf::LN_dptxlrtb;
        flag&=QName::tokenizer().insert(ns, "dptxlrtbv") == NS_rtf::LN_dptxlrtbv;
        flag&=QName::tokenizer().insert(ns, "dptxtbrl") == NS_rtf::LN_dptxtbrl;
        flag&=QName::tokenizer().insert(ns, "dptxtbrlv") == NS_rtf::LN_dptxtbrlv;
        flag&=QName::tokenizer().insert(ns, "dpx") == NS_rtf::LN_dpx;
        flag&=QName::tokenizer().insert(ns, "dpxsize") == NS_rtf::LN_dpxsize;
        flag&=QName::tokenizer().insert(ns, "dpy") == NS_rtf::LN_dpy;
        flag&=QName::tokenizer().insert(ns, "dpysize") == NS_rtf::LN_dpysize;
        flag&=QName::tokenizer().insert(ns, "dropcapli") == NS_rtf::LN_dropcapli;
        flag&=QName::tokenizer().insert(ns, "dropcapt") == NS_rtf::LN_dropcapt;
        flag&=QName::tokenizer().insert(ns, "ds") == NS_rtf::LN_ds;
        flag&=QName::tokenizer().insert(ns, "dxfrtext") == NS_rtf::LN_dxfrtext;
        flag&=QName::tokenizer().insert(ns, "dy") == NS_rtf::LN_dy;
        flag&=QName::tokenizer().insert(ns, "edmins") == NS_rtf::LN_edmins;
        flag&=QName::tokenizer().insert(ns, "embo") == NS_rtf::LN_embo;
        flag&=QName::tokenizer().insert(ns, "emdash") == NS_rtf::LN_emdash;
        flag&=QName::tokenizer().insert(ns, "emfblip") == NS_rtf::LN_emfblip;
        flag&=QName::tokenizer().insert(ns, "emspace") == NS_rtf::LN_emspace;
        flag&=QName::tokenizer().insert(ns, "endash") == NS_rtf::LN_endash;
        flag&=QName::tokenizer().insert(ns, "enddoc") == NS_rtf::LN_enddoc;
        flag&=QName::tokenizer().insert(ns, "endnhere") == NS_rtf::LN_endnhere;
        flag&=QName::tokenizer().insert(ns, "endnotes") == NS_rtf::LN_endnotes;
        flag&=QName::tokenizer().insert(ns, "enspace") == NS_rtf::LN_enspace;
        flag&=QName::tokenizer().insert(ns, "expnd") == NS_rtf::LN_expnd;
        flag&=QName::tokenizer().insert(ns, "expndtw") == NS_rtf::LN_expndtw;
        flag&=QName::tokenizer().insert(ns, "expshrtn") == NS_rtf::LN_expshrtn;
        flag&=QName::tokenizer().insert(ns, "f") == NS_rtf::LN_f;
        flag&=QName::tokenizer().insert(ns, "faauto") == NS_rtf::LN_faauto;
        flag&=QName::tokenizer().insert(ns, "facenter7.0") == NS_rtf::LN_facenter7_0;
        flag&=QName::tokenizer().insert(ns, "facingp") == NS_rtf::LN_facingp;
        flag&=QName::tokenizer().insert(ns, "fahang") == NS_rtf::LN_fahang;
        flag&=QName::tokenizer().insert(ns, "falt") == NS_rtf::LN_falt;
        flag&=QName::tokenizer().insert(ns, "faroman") == NS_rtf::LN_faroman;
        flag&=QName::tokenizer().insert(ns, "favar") == NS_rtf::LN_favar;
        flag&=QName::tokenizer().insert(ns, "fbias") == NS_rtf::LN_fbias;
        flag&=QName::tokenizer().insert(ns, "fbidi") == NS_rtf::LN_fbidi;
        flag&=QName::tokenizer().insert(ns, "fchars") == NS_rtf::LN_fchars;
        flag&=QName::tokenizer().insert(ns, "fcharset") == NS_rtf::LN_fcharset;
        flag&=QName::tokenizer().insert(ns, "fdecor") == NS_rtf::LN_fdecor;
        flag&=QName::tokenizer().insert(ns, "fet") == NS_rtf::LN_fet;
        flag&=QName::tokenizer().insert(ns, "fetch") == NS_rtf::LN_fetch;
        flag&=QName::tokenizer().insert(ns, "ffdefres") == NS_rtf::LN_ffdefres;
        flag&=QName::tokenizer().insert(ns, "ffdeftext") == NS_rtf::LN_ffdeftext;
        flag&=QName::tokenizer().insert(ns, "ffentrymcr") == NS_rtf::LN_ffentrymcr;
        flag&=QName::tokenizer().insert(ns, "ffexitmcr") == NS_rtf::LN_ffexitmcr;
        flag&=QName::tokenizer().insert(ns, "ffformat") == NS_rtf::LN_ffformat;
        flag&=QName::tokenizer().insert(ns, "ffhaslistbox") == NS_rtf::LN_ffhaslistbox;
        flag&=QName::tokenizer().insert(ns, "ffhelptext") == NS_rtf::LN_ffhelptext;
        flag&=QName::tokenizer().insert(ns, "ffhps") == NS_rtf::LN_ffhps;
        flag&=QName::tokenizer().insert(ns, "ffl") == NS_rtf::LN_ffl;
        flag&=QName::tokenizer().insert(ns, "ffmaxlen") == NS_rtf::LN_ffmaxlen;
        flag&=QName::tokenizer().insert(ns, "ffname") == NS_rtf::LN_ffname;
        flag&=QName::tokenizer().insert(ns, "ffownhelp") == NS_rtf::LN_ffownhelp;
        flag&=QName::tokenizer().insert(ns, "ffownstat") == NS_rtf::LN_ffownstat;
        flag&=QName::tokenizer().insert(ns, "ffprot") == NS_rtf::LN_ffprot;
        flag&=QName::tokenizer().insert(ns, "ffrecalc") == NS_rtf::LN_ffrecalc;
        flag&=QName::tokenizer().insert(ns, "ffres") == NS_rtf::LN_ffres;
        flag&=QName::tokenizer().insert(ns, "ffsize") == NS_rtf::LN_ffsize;
        flag&=QName::tokenizer().insert(ns, "ffstattext") == NS_rtf::LN_ffstattext;
        flag&=QName::tokenizer().insert(ns, "fftype") == NS_rtf::LN_fftype;
        flag&=QName::tokenizer().insert(ns, "fftypetxt") == NS_rtf::LN_fftypetxt;
        flag&=QName::tokenizer().insert(ns, "fi") == NS_rtf::LN_fi;
        flag&=QName::tokenizer().insert(ns, "fid") == NS_rtf::LN_fid;
        flag&=QName::tokenizer().insert(ns, "field") == NS_rtf::LN_field;
        flag&=QName::tokenizer().insert(ns, "file") == NS_rtf::LN_file;
        flag&=QName::tokenizer().insert(ns, "filetbl") == NS_rtf::LN_filetbl;
        flag&=QName::tokenizer().insert(ns, "fittext") == NS_rtf::LN_fittext;
        flag&=QName::tokenizer().insert(ns, "fldalt") == NS_rtf::LN_fldalt;
        flag&=QName::tokenizer().insert(ns, "flddirty") == NS_rtf::LN_flddirty;
        flag&=QName::tokenizer().insert(ns, "fldedit") == NS_rtf::LN_fldedit;
        flag&=QName::tokenizer().insert(ns, "fldinst") == NS_rtf::LN_fldinst;
        flag&=QName::tokenizer().insert(ns, "fldlock") == NS_rtf::LN_fldlock;
        flag&=QName::tokenizer().insert(ns, "fldpriv") == NS_rtf::LN_fldpriv;
        flag&=QName::tokenizer().insert(ns, "fldrslt") == NS_rtf::LN_fldrslt;
        flag&=QName::tokenizer().insert(ns, "fldtype") == NS_rtf::LN_fldtype;
        flag&=QName::tokenizer().insert(ns, "fmodern") == NS_rtf::LN_fmodern;
        flag&=QName::tokenizer().insert(ns, "fn") == NS_rtf::LN_fn;
        flag&=QName::tokenizer().insert(ns, "fname") == NS_rtf::LN_fname;
        flag&=QName::tokenizer().insert(ns, "fnetwork") == NS_rtf::LN_fnetwork;
        flag&=QName::tokenizer().insert(ns, "fnil") == NS_rtf::LN_fnil;
        flag&=QName::tokenizer().insert(ns, "fnonfilesys") == NS_rtf::LN_fnonfilesys;
        flag&=QName::tokenizer().insert(ns, "fontemb") == NS_rtf::LN_fontemb;
        flag&=QName::tokenizer().insert(ns, "fontfile") == NS_rtf::LN_fontfile;
        flag&=QName::tokenizer().insert(ns, "fonttbl") == NS_rtf::LN_fonttbl;
        flag&=QName::tokenizer().insert(ns, "footer") == NS_rtf::LN_footer;
        flag&=QName::tokenizer().insert(ns, "footerf") == NS_rtf::LN_footerf;
        flag&=QName::tokenizer().insert(ns, "footerl") == NS_rtf::LN_footerl;
        flag&=QName::tokenizer().insert(ns, "footery") == NS_rtf::LN_footery;
        flag&=QName::tokenizer().insert(ns, "footnote") == NS_rtf::LN_footnote;
        flag&=QName::tokenizer().insert(ns, "formdisp") == NS_rtf::LN_formdisp;
        flag&=QName::tokenizer().insert(ns, "formfield") == NS_rtf::LN_formfield;
        flag&=QName::tokenizer().insert(ns, "formprot") == NS_rtf::LN_formprot;
        flag&=QName::tokenizer().insert(ns, "formshade") == NS_rtf::LN_formshade;
        flag&=QName::tokenizer().insert(ns, "fosnum") == NS_rtf::LN_fosnum;
        flag&=QName::tokenizer().insert(ns, "fprq") == NS_rtf::LN_fprq;
        flag&=QName::tokenizer().insert(ns, "fracwidth") == NS_rtf::LN_fracwidth;
        flag&=QName::tokenizer().insert(ns, "frelative") == NS_rtf::LN_frelative;
        flag&=QName::tokenizer().insert(ns, "frmtxbtlr") == NS_rtf::LN_frmtxbtlr;
        flag&=QName::tokenizer().insert(ns, "frmtxlrtb") == NS_rtf::LN_frmtxlrtb;
        flag&=QName::tokenizer().insert(ns, "frmtxlrtbv") == NS_rtf::LN_frmtxlrtbv;
        flag&=QName::tokenizer().insert(ns, "frmtxtbrl") == NS_rtf::LN_frmtxtbrl;
        flag&=QName::tokenizer().insert(ns, "frmtxtbrlv") == NS_rtf::LN_frmtxtbrlv;
        flag&=QName::tokenizer().insert(ns, "froman") == NS_rtf::LN_froman;
        flag&=QName::tokenizer().insert(ns, "fromhtml") == NS_rtf::LN_fromhtml;
        flag&=QName::tokenizer().insert(ns, "fromtext") == NS_rtf::LN_fromtext;
        flag&=QName::tokenizer().insert(ns, "fs") == NS_rtf::LN_fs;
        flag&=QName::tokenizer().insert(ns, "fscript") == NS_rtf::LN_fscript;
        flag&=QName::tokenizer().insert(ns, "fswiss") == NS_rtf::LN_fswiss;
        flag&=QName::tokenizer().insert(ns, "ftnalt") == NS_rtf::LN_ftnalt;
        flag&=QName::tokenizer().insert(ns, "ftnbj") == NS_rtf::LN_ftnbj;
        flag&=QName::tokenizer().insert(ns, "ftncn") == NS_rtf::LN_ftncn;
        flag&=QName::tokenizer().insert(ns, "ftnil") == NS_rtf::LN_ftnil;
        flag&=QName::tokenizer().insert(ns, "ftnlytwnine") == NS_rtf::LN_ftnlytwnine;
        flag&=QName::tokenizer().insert(ns, "ftnnalc") == NS_rtf::LN_ftnnalc;
        flag&=QName::tokenizer().insert(ns, "ftnnar") == NS_rtf::LN_ftnnar;
        flag&=QName::tokenizer().insert(ns, "ftnnauc") == NS_rtf::LN_ftnnauc;
        flag&=QName::tokenizer().insert(ns, "ftnnchi") == NS_rtf::LN_ftnnchi;
        flag&=QName::tokenizer().insert(ns, "ftnnchosung") == NS_rtf::LN_ftnnchosung;
        flag&=QName::tokenizer().insert(ns, "ftnncnum") == NS_rtf::LN_ftnncnum;
        flag&=QName::tokenizer().insert(ns, "ftnndbar") == NS_rtf::LN_ftnndbar;
        flag&=QName::tokenizer().insert(ns, "ftnndbnum") == NS_rtf::LN_ftnndbnum;
        flag&=QName::tokenizer().insert(ns, "ftnndbnumd") == NS_rtf::LN_ftnndbnumd;
        flag&=QName::tokenizer().insert(ns, "ftnndbnumk") == NS_rtf::LN_ftnndbnumk;
        flag&=QName::tokenizer().insert(ns, "ftnndbnumt") == NS_rtf::LN_ftnndbnumt;
        flag&=QName::tokenizer().insert(ns, "ftnnganada") == NS_rtf::LN_ftnnganada;
        flag&=QName::tokenizer().insert(ns, "ftnngbnum") == NS_rtf::LN_ftnngbnum;
        flag&=QName::tokenizer().insert(ns, "ftnngbnumd") == NS_rtf::LN_ftnngbnumd;
        flag&=QName::tokenizer().insert(ns, "ftnngbnumk") == NS_rtf::LN_ftnngbnumk;
        flag&=QName::tokenizer().insert(ns, "ftnngbnuml") == NS_rtf::LN_ftnngbnuml;
        flag&=QName::tokenizer().insert(ns, "ftnnrlc") == NS_rtf::LN_ftnnrlc;
        flag&=QName::tokenizer().insert(ns, "ftnnruc") == NS_rtf::LN_ftnnruc;
        flag&=QName::tokenizer().insert(ns, "ftnnzodiac") == NS_rtf::LN_ftnnzodiac;
        flag&=QName::tokenizer().insert(ns, "ftnnzodiacd") == NS_rtf::LN_ftnnzodiacd;
        flag&=QName::tokenizer().insert(ns, "ftnnzodiacl") == NS_rtf::LN_ftnnzodiacl;
        flag&=QName::tokenizer().insert(ns, "ftnrestart") == NS_rtf::LN_ftnrestart;
        flag&=QName::tokenizer().insert(ns, "ftnrstcont") == NS_rtf::LN_ftnrstcont;
        flag&=QName::tokenizer().insert(ns, "ftnrstpg") == NS_rtf::LN_ftnrstpg;
        flag&=QName::tokenizer().insert(ns, "ftnsep") == NS_rtf::LN_ftnsep;
        flag&=QName::tokenizer().insert(ns, "ftnsepc") == NS_rtf::LN_ftnsepc;
        flag&=QName::tokenizer().insert(ns, "ftnstart") == NS_rtf::LN_ftnstart;
        flag&=QName::tokenizer().insert(ns, "ftntj") == NS_rtf::LN_ftntj;
        flag&=QName::tokenizer().insert(ns, "fttruetype") == NS_rtf::LN_fttruetype;
        flag&=QName::tokenizer().insert(ns, "fvaliddos") == NS_rtf::LN_fvaliddos;
        flag&=QName::tokenizer().insert(ns, "fvalidhpfs") == NS_rtf::LN_fvalidhpfs;
        flag&=QName::tokenizer().insert(ns, "fvalidmac") == NS_rtf::LN_fvalidmac;
        flag&=QName::tokenizer().insert(ns, "fvalidntfs") == NS_rtf::LN_fvalidntfs;
        flag&=QName::tokenizer().insert(ns, "g") == NS_rtf::LN_g;
        flag&=QName::tokenizer().insert(ns, "gcw") == NS_rtf::LN_gcw;
        flag&=QName::tokenizer().insert(ns, "generator") == NS_rtf::LN_generator;
        flag&=QName::tokenizer().insert(ns, "green") == NS_rtf::LN_green;
        flag&=QName::tokenizer().insert(ns, "gridtbl") == NS_rtf::LN_gridtbl;
        flag&=QName::tokenizer().insert(ns, "gutter") == NS_rtf::LN_gutter;
        flag&=QName::tokenizer().insert(ns, "gutterprl") == NS_rtf::LN_gutterprl;
        flag&=QName::tokenizer().insert(ns, "guttersxn") == NS_rtf::LN_guttersxn;
        flag&=QName::tokenizer().insert(ns, "header") == NS_rtf::LN_header;
        flag&=QName::tokenizer().insert(ns, "headerf") == NS_rtf::LN_headerf;
        flag&=QName::tokenizer().insert(ns, "headerl") == NS_rtf::LN_headerl;
        flag&=QName::tokenizer().insert(ns, "headery") == NS_rtf::LN_headery;
        flag&=QName::tokenizer().insert(ns, "hich") == NS_rtf::LN_hich;
        flag&=QName::tokenizer().insert(ns, "highlight") == NS_rtf::LN_highlight;
        flag&=QName::tokenizer().insert(ns, "hlfr") == NS_rtf::LN_hlfr;
        flag&=QName::tokenizer().insert(ns, "hlinkbase") == NS_rtf::LN_hlinkbase;
        flag&=QName::tokenizer().insert(ns, "hlloc") == NS_rtf::LN_hlloc;
        flag&=QName::tokenizer().insert(ns, "hlsrc") == NS_rtf::LN_hlsrc;
        flag&=QName::tokenizer().insert(ns, "horzdoc") == NS_rtf::LN_horzdoc;
        flag&=QName::tokenizer().insert(ns, "horzsect") == NS_rtf::LN_horzsect;
        flag&=QName::tokenizer().insert(ns, "hr") == NS_rtf::LN_hr;
        flag&=QName::tokenizer().insert(ns, "htmautsp") == NS_rtf::LN_htmautsp;
        flag&=QName::tokenizer().insert(ns, "htmlbase") == NS_rtf::LN_htmlbase;
        flag&=QName::tokenizer().insert(ns, "htmlrtf") == NS_rtf::LN_htmlrtf;
        flag&=QName::tokenizer().insert(ns, "htmltag") == NS_rtf::LN_htmltag;
        flag&=QName::tokenizer().insert(ns, "hyphauto") == NS_rtf::LN_hyphauto;
        flag&=QName::tokenizer().insert(ns, "hyphcaps") == NS_rtf::LN_hyphcaps;
        flag&=QName::tokenizer().insert(ns, "hyphconsec") == NS_rtf::LN_hyphconsec;
        flag&=QName::tokenizer().insert(ns, "hyphhotz") == NS_rtf::LN_hyphhotz;
        flag&=QName::tokenizer().insert(ns, "hyphpar") == NS_rtf::LN_hyphpar;
        flag&=QName::tokenizer().insert(ns, "i") == NS_rtf::LN_i;
        flag&=QName::tokenizer().insert(ns, "id") == NS_rtf::LN_id;
        flag&=QName::tokenizer().insert(ns, "ilvl") == NS_rtf::LN_ilvl;
        flag&=QName::tokenizer().insert(ns, "impr") == NS_rtf::LN_impr;
        flag&=QName::tokenizer().insert(ns, "info") == NS_rtf::LN_info;
        flag&=QName::tokenizer().insert(ns, "insrsid") == NS_rtf::LN_insrsid;
        flag&=QName::tokenizer().insert(ns, "intbl") == NS_rtf::LN_intbl;
        flag&=QName::tokenizer().insert(ns, "ipgp") == NS_rtf::LN_ipgp;
        flag&=QName::tokenizer().insert(ns, "irow") == NS_rtf::LN_irow;
        flag&=QName::tokenizer().insert(ns, "irowband") == NS_rtf::LN_irowband;
        flag&=QName::tokenizer().insert(ns, "itap") == NS_rtf::LN_itap;
        flag&=QName::tokenizer().insert(ns, "ixe") == NS_rtf::LN_ixe;
        flag&=QName::tokenizer().insert(ns, "jcompress") == NS_rtf::LN_jcompress;
        flag&=QName::tokenizer().insert(ns, "jexpand") == NS_rtf::LN_jexpand;
        flag&=QName::tokenizer().insert(ns, "jpegblip") == NS_rtf::LN_jpegblip;
        flag&=QName::tokenizer().insert(ns, "jsksu") == NS_rtf::LN_jsksu;
        flag&=QName::tokenizer().insert(ns, "keep") == NS_rtf::LN_keep;
        flag&=QName::tokenizer().insert(ns, "keepn") == NS_rtf::LN_keepn;
        flag&=QName::tokenizer().insert(ns, "kerning") == NS_rtf::LN_kerning;
        flag&=QName::tokenizer().insert(ns, "keycode") == NS_rtf::LN_keycode;
        flag&=QName::tokenizer().insert(ns, "keywords") == NS_rtf::LN_keywords;
        flag&=QName::tokenizer().insert(ns, "ksulang") == NS_rtf::LN_ksulang;
        flag&=QName::tokenizer().insert(ns, "landscape") == NS_rtf::LN_landscape;
        flag&=QName::tokenizer().insert(ns, "lang") == NS_rtf::LN_lang;
        flag&=QName::tokenizer().insert(ns, "langfe") == NS_rtf::LN_langfe;
        flag&=QName::tokenizer().insert(ns, "langfenp") == NS_rtf::LN_langfenp;
        flag&=QName::tokenizer().insert(ns, "langnp") == NS_rtf::LN_langnp;
        flag&=QName::tokenizer().insert(ns, "lastrow") == NS_rtf::LN_lastrow;
        flag&=QName::tokenizer().insert(ns, "lbr") == NS_rtf::LN_lbr;
        flag&=QName::tokenizer().insert(ns, "lchars") == NS_rtf::LN_lchars;
        flag&=QName::tokenizer().insert(ns, "ldblquote") == NS_rtf::LN_ldblquote;
        flag&=QName::tokenizer().insert(ns, "level") == NS_rtf::LN_level;
        flag&=QName::tokenizer().insert(ns, "levelfollow") == NS_rtf::LN_levelfollow;
        flag&=QName::tokenizer().insert(ns, "levelindent") == NS_rtf::LN_levelindent;
        flag&=QName::tokenizer().insert(ns, "leveljc") == NS_rtf::LN_leveljc;
        flag&=QName::tokenizer().insert(ns, "leveljcn") == NS_rtf::LN_leveljcn;
        flag&=QName::tokenizer().insert(ns, "levellegal") == NS_rtf::LN_levellegal;
        flag&=QName::tokenizer().insert(ns, "levelnfc") == NS_rtf::LN_levelnfc;
        flag&=QName::tokenizer().insert(ns, "levelnfcn") == NS_rtf::LN_levelnfcn;
        flag&=QName::tokenizer().insert(ns, "levelnorestart") == NS_rtf::LN_levelnorestart;
        flag&=QName::tokenizer().insert(ns, "levelnumbers") == NS_rtf::LN_levelnumbers;
        flag&=QName::tokenizer().insert(ns, "levelold") == NS_rtf::LN_levelold;
        flag&=QName::tokenizer().insert(ns, "levelpicture") == NS_rtf::LN_levelpicture;
        flag&=QName::tokenizer().insert(ns, "levelprev") == NS_rtf::LN_levelprev;
        flag&=QName::tokenizer().insert(ns, "levelprevspace") == NS_rtf::LN_levelprevspace;
        flag&=QName::tokenizer().insert(ns, "psover") == NS_rtf::LN_psover;
        flag&=QName::tokenizer().insert(ns, "psz") == NS_rtf::LN_psz;
        flag&=QName::tokenizer().insert(ns, "levelspace") == NS_rtf::LN_levelspace;
        flag&=QName::tokenizer().insert(ns, "pubauto") == NS_rtf::LN_pubauto;
        flag&=QName::tokenizer().insert(ns, "levelstartat") == NS_rtf::LN_levelstartat;
        flag&=QName::tokenizer().insert(ns, "pvmrg") == NS_rtf::LN_pvmrg;
        flag&=QName::tokenizer().insert(ns, "pvpara") == NS_rtf::LN_pvpara;
        flag&=QName::tokenizer().insert(ns, "leveltemplateid") == NS_rtf::LN_leveltemplateid;
        flag&=QName::tokenizer().insert(ns, "pvpg") == NS_rtf::LN_pvpg;
        flag&=QName::tokenizer().insert(ns, "leveltext") == NS_rtf::LN_leveltext;
        flag&=QName::tokenizer().insert(ns, "pwd") == NS_rtf::LN_pwd;
        flag&=QName::tokenizer().insert(ns, "li") == NS_rtf::LN_li;
        flag&=QName::tokenizer().insert(ns, "pxe") == NS_rtf::LN_pxe;
        flag&=QName::tokenizer().insert(ns, "line") == NS_rtf::LN_line;
        flag&=QName::tokenizer().insert(ns, "qc") == NS_rtf::LN_qc;
        flag&=QName::tokenizer().insert(ns, "qd") == NS_rtf::LN_qd;
        flag&=QName::tokenizer().insert(ns, "linebetcol") == NS_rtf::LN_linebetcol;
        flag&=QName::tokenizer().insert(ns, "qj") == NS_rtf::LN_qj;
        flag&=QName::tokenizer().insert(ns, "linecont") == NS_rtf::LN_linecont;
        flag&=QName::tokenizer().insert(ns, "qk") == NS_rtf::LN_qk;
        flag&=QName::tokenizer().insert(ns, "ql") == NS_rtf::LN_ql;
        flag&=QName::tokenizer().insert(ns, "linemod") == NS_rtf::LN_linemod;
        flag&=QName::tokenizer().insert(ns, "qmspace") == NS_rtf::LN_qmspace;
        flag&=QName::tokenizer().insert(ns, "lineppage") == NS_rtf::LN_lineppage;
        flag&=QName::tokenizer().insert(ns, "qr") == NS_rtf::LN_qr;
        flag&=QName::tokenizer().insert(ns, "qt") == NS_rtf::LN_qt;
        flag&=QName::tokenizer().insert(ns, "linerestart") == NS_rtf::LN_linerestart;
        flag&=QName::tokenizer().insert(ns, "linestart") == NS_rtf::LN_linestart;
        flag&=QName::tokenizer().insert(ns, "rawclbgbdiag") == NS_rtf::LN_rawclbgbdiag;
        flag&=QName::tokenizer().insert(ns, "linestarts") == NS_rtf::LN_linestarts;
        flag&=QName::tokenizer().insert(ns, "rawclbgcross") == NS_rtf::LN_rawclbgcross;
        flag&=QName::tokenizer().insert(ns, "linex") == NS_rtf::LN_linex;
        flag&=QName::tokenizer().insert(ns, "rawclbgdcross") == NS_rtf::LN_rawclbgdcross;
        flag&=QName::tokenizer().insert(ns, "linkself") == NS_rtf::LN_linkself;
        flag&=QName::tokenizer().insert(ns, "rawbgdkbdiag") == NS_rtf::LN_rawbgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "linkstyles") == NS_rtf::LN_linkstyles;
        flag&=QName::tokenizer().insert(ns, "linkval") == NS_rtf::LN_linkval;
        flag&=QName::tokenizer().insert(ns, "rawclbgdkcross") == NS_rtf::LN_rawclbgdkcross;
        flag&=QName::tokenizer().insert(ns, "lin") == NS_rtf::LN_lin;
        flag&=QName::tokenizer().insert(ns, "lisa") == NS_rtf::LN_lisa;
        flag&=QName::tokenizer().insert(ns, "rawclbgdkdcross") == NS_rtf::LN_rawclbgdkdcross;
        flag&=QName::tokenizer().insert(ns, "lisb") == NS_rtf::LN_lisb;
        flag&=QName::tokenizer().insert(ns, "rawclbgdkfdiag") == NS_rtf::LN_rawclbgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "listhybrid") == NS_rtf::LN_listhybrid;
        flag&=QName::tokenizer().insert(ns, "listid") == NS_rtf::LN_listid;
        flag&=QName::tokenizer().insert(ns, "rawclbgdkhor") == NS_rtf::LN_rawclbgdkhor;
        flag&=QName::tokenizer().insert(ns, "listname") == NS_rtf::LN_listname;
        flag&=QName::tokenizer().insert(ns, "rawclbgdkvert") == NS_rtf::LN_rawclbgdkvert;
        flag&=QName::tokenizer().insert(ns, "ab") == NS_rtf::LN_ab;
        flag&=QName::tokenizer().insert(ns, "listoverridecount") == NS_rtf::LN_listoverridecount;
        flag&=QName::tokenizer().insert(ns, "rawclbgfdiag") == NS_rtf::LN_rawclbgfdiag;
        flag&=QName::tokenizer().insert(ns, "absh") == NS_rtf::LN_absh;
        flag&=QName::tokenizer().insert(ns, "rawclbghoriz") == NS_rtf::LN_rawclbghoriz;
        flag&=QName::tokenizer().insert(ns, "listoverrideformat") == NS_rtf::LN_listoverrideformat;
        flag&=QName::tokenizer().insert(ns, "abslock") == NS_rtf::LN_abslock;
        flag&=QName::tokenizer().insert(ns, "rawclbgvert") == NS_rtf::LN_rawclbgvert;
        flag&=QName::tokenizer().insert(ns, "listoverridestart") == NS_rtf::LN_listoverridestart;
        flag&=QName::tokenizer().insert(ns, "rdblquote") == NS_rtf::LN_rdblquote;
        flag&=QName::tokenizer().insert(ns, "absnoovrlp") == NS_rtf::LN_absnoovrlp;
        flag&=QName::tokenizer().insert(ns, "red") == NS_rtf::LN_red;
        flag&=QName::tokenizer().insert(ns, "listpicture") == NS_rtf::LN_listpicture;
        flag&=QName::tokenizer().insert(ns, "absw") == NS_rtf::LN_absw;
        flag&=QName::tokenizer().insert(ns, "rempersonalinfo") == NS_rtf::LN_rempersonalinfo;
        flag&=QName::tokenizer().insert(ns, "acaps") == NS_rtf::LN_acaps;
        flag&=QName::tokenizer().insert(ns, "listrestarthdn") == NS_rtf::LN_listrestarthdn;
        flag&=QName::tokenizer().insert(ns, "result") == NS_rtf::LN_result;
        flag&=QName::tokenizer().insert(ns, "acccomma") == NS_rtf::LN_acccomma;
        flag&=QName::tokenizer().insert(ns, "listsimple") == NS_rtf::LN_listsimple;
        flag&=QName::tokenizer().insert(ns, "revauth") == NS_rtf::LN_revauth;
        flag&=QName::tokenizer().insert(ns, "accdot") == NS_rtf::LN_accdot;
        flag&=QName::tokenizer().insert(ns, "liststyleid") == NS_rtf::LN_liststyleid;
        flag&=QName::tokenizer().insert(ns, "accnone") == NS_rtf::LN_accnone;
        flag&=QName::tokenizer().insert(ns, "revauthdel") == NS_rtf::LN_revauthdel;
        flag&=QName::tokenizer().insert(ns, "acf") == NS_rtf::LN_acf;
        flag&=QName::tokenizer().insert(ns, "liststylename") == NS_rtf::LN_liststylename;
        flag&=QName::tokenizer().insert(ns, "revbar") == NS_rtf::LN_revbar;
        flag&=QName::tokenizer().insert(ns, "additive") == NS_rtf::LN_additive;
        flag&=QName::tokenizer().insert(ns, "revdttm") == NS_rtf::LN_revdttm;
        flag&=QName::tokenizer().insert(ns, "listtemplateid") == NS_rtf::LN_listtemplateid;
        flag&=QName::tokenizer().insert(ns, "adjustright") == NS_rtf::LN_adjustright;
        flag&=QName::tokenizer().insert(ns, "revdttmdel") == NS_rtf::LN_revdttmdel;
        flag&=QName::tokenizer().insert(ns, "adn") == NS_rtf::LN_adn;
        flag&=QName::tokenizer().insert(ns, "listtext") == NS_rtf::LN_listtext;
        flag&=QName::tokenizer().insert(ns, "revised") == NS_rtf::LN_revised;
        flag&=QName::tokenizer().insert(ns, "aenddoc") == NS_rtf::LN_aenddoc;
        flag&=QName::tokenizer().insert(ns, "lnbrkrule") == NS_rtf::LN_lnbrkrule;
        flag&=QName::tokenizer().insert(ns, "revisions") == NS_rtf::LN_revisions;
        flag&=QName::tokenizer().insert(ns, "aendnotes") == NS_rtf::LN_aendnotes;
        flag&=QName::tokenizer().insert(ns, "lndscpsxn") == NS_rtf::LN_lndscpsxn;
        flag&=QName::tokenizer().insert(ns, "revprop") == NS_rtf::LN_revprop;
        flag&=QName::tokenizer().insert(ns, "aexpnd") == NS_rtf::LN_aexpnd;
        flag&=QName::tokenizer().insert(ns, "lnongrid") == NS_rtf::LN_lnongrid;
        flag&=QName::tokenizer().insert(ns, "revprot") == NS_rtf::LN_revprot;
        flag&=QName::tokenizer().insert(ns, "af") == NS_rtf::LN_af;
        flag&=QName::tokenizer().insert(ns, "loch") == NS_rtf::LN_loch;
        flag&=QName::tokenizer().insert(ns, "revtbl") == NS_rtf::LN_revtbl;
        flag&=QName::tokenizer().insert(ns, "affixed") == NS_rtf::LN_affixed;
        flag&=QName::tokenizer().insert(ns, "lquote") == NS_rtf::LN_lquote;
        flag&=QName::tokenizer().insert(ns, "revtim") == NS_rtf::LN_revtim;
        flag&=QName::tokenizer().insert(ns, "afs") == NS_rtf::LN_afs;
        flag&=QName::tokenizer().insert(ns, "ls") == NS_rtf::LN_ls;
        flag&=QName::tokenizer().insert(ns, "ri") == NS_rtf::LN_ri;
        flag&=QName::tokenizer().insert(ns, "aftnbj") == NS_rtf::LN_aftnbj;
        flag&=QName::tokenizer().insert(ns, "ltrch") == NS_rtf::LN_ltrch;
        flag&=QName::tokenizer().insert(ns, "rin") == NS_rtf::LN_rin;
        flag&=QName::tokenizer().insert(ns, "row") == NS_rtf::LN_row;
        flag&=QName::tokenizer().insert(ns, "aftncn") == NS_rtf::LN_aftncn;
        flag&=QName::tokenizer().insert(ns, "ltrdoc") == NS_rtf::LN_ltrdoc;
        flag&=QName::tokenizer().insert(ns, "rquote") == NS_rtf::LN_rquote;
        flag&=QName::tokenizer().insert(ns, "ltrmark") == NS_rtf::LN_ltrmark;
        flag&=QName::tokenizer().insert(ns, "rsid") == NS_rtf::LN_rsid;
        flag&=QName::tokenizer().insert(ns, "ltrpar") == NS_rtf::LN_ltrpar;
        flag&=QName::tokenizer().insert(ns, "aftnnalc") == NS_rtf::LN_aftnnalc;
        flag&=QName::tokenizer().insert(ns, "rsidroot") == NS_rtf::LN_rsidroot;
        flag&=QName::tokenizer().insert(ns, "ltrrow") == NS_rtf::LN_ltrrow;
        flag&=QName::tokenizer().insert(ns, "aftnnar") == NS_rtf::LN_aftnnar;
        flag&=QName::tokenizer().insert(ns, "rsidtbl") == NS_rtf::LN_rsidtbl;
        flag&=QName::tokenizer().insert(ns, "ltrsect") == NS_rtf::LN_ltrsect;
        flag&=QName::tokenizer().insert(ns, "aftnnauc") == NS_rtf::LN_aftnnauc;
        flag&=QName::tokenizer().insert(ns, "rsltbmp") == NS_rtf::LN_rsltbmp;
        flag&=QName::tokenizer().insert(ns, "aftnnchi") == NS_rtf::LN_aftnnchi;
        flag&=QName::tokenizer().insert(ns, "rslthtml") == NS_rtf::LN_rslthtml;
        flag&=QName::tokenizer().insert(ns, "lytcalctblwd") == NS_rtf::LN_lytcalctblwd;
        flag&=QName::tokenizer().insert(ns, "rsltmerge") == NS_rtf::LN_rsltmerge;
        flag&=QName::tokenizer().insert(ns, "lytexcttp") == NS_rtf::LN_lytexcttp;
        flag&=QName::tokenizer().insert(ns, "aftnnchosung") == NS_rtf::LN_aftnnchosung;
        flag&=QName::tokenizer().insert(ns, "rsltpict") == NS_rtf::LN_rsltpict;
        flag&=QName::tokenizer().insert(ns, "lytprtmet") == NS_rtf::LN_lytprtmet;
        flag&=QName::tokenizer().insert(ns, "aftnncnum") == NS_rtf::LN_aftnncnum;
        flag&=QName::tokenizer().insert(ns, "rsltrtf") == NS_rtf::LN_rsltrtf;
        flag&=QName::tokenizer().insert(ns, "lyttblrtgr") == NS_rtf::LN_lyttblrtgr;
        flag&=QName::tokenizer().insert(ns, "aftnndbar") == NS_rtf::LN_aftnndbar;
        flag&=QName::tokenizer().insert(ns, "rslttxt") == NS_rtf::LN_rslttxt;
        flag&=QName::tokenizer().insert(ns, "mac") == NS_rtf::LN_mac;
        flag&=QName::tokenizer().insert(ns, "aftnndbnum") == NS_rtf::LN_aftnndbnum;
        flag&=QName::tokenizer().insert(ns, "rtf") == NS_rtf::LN_rtf;
        flag&=QName::tokenizer().insert(ns, "macpict") == NS_rtf::LN_macpict;
        flag&=QName::tokenizer().insert(ns, "rtlch") == NS_rtf::LN_rtlch;
        flag&=QName::tokenizer().insert(ns, "aftnndbnumd") == NS_rtf::LN_aftnndbnumd;
        flag&=QName::tokenizer().insert(ns, "makebackup") == NS_rtf::LN_makebackup;
        flag&=QName::tokenizer().insert(ns, "rtldoc") == NS_rtf::LN_rtldoc;
        flag&=QName::tokenizer().insert(ns, "aftnndbnumk") == NS_rtf::LN_aftnndbnumk;
        flag&=QName::tokenizer().insert(ns, "manager") == NS_rtf::LN_manager;
        flag&=QName::tokenizer().insert(ns, "rtlgutter") == NS_rtf::LN_rtlgutter;
        flag&=QName::tokenizer().insert(ns, "margb") == NS_rtf::LN_margb;
        flag&=QName::tokenizer().insert(ns, "aftnndbnumt") == NS_rtf::LN_aftnndbnumt;
        flag&=QName::tokenizer().insert(ns, "rtlmark") == NS_rtf::LN_rtlmark;
        flag&=QName::tokenizer().insert(ns, "margbsxn") == NS_rtf::LN_margbsxn;
        flag&=QName::tokenizer().insert(ns, "aftnnganada") == NS_rtf::LN_aftnnganada;
        flag&=QName::tokenizer().insert(ns, "rtlpar") == NS_rtf::LN_rtlpar;
        flag&=QName::tokenizer().insert(ns, "margl") == NS_rtf::LN_margl;
        flag&=QName::tokenizer().insert(ns, "rtlrow") == NS_rtf::LN_rtlrow;
        flag&=QName::tokenizer().insert(ns, "aftnngbnum") == NS_rtf::LN_aftnngbnum;
        flag&=QName::tokenizer().insert(ns, "marglsxn") == NS_rtf::LN_marglsxn;
        flag&=QName::tokenizer().insert(ns, "rtlsect") == NS_rtf::LN_rtlsect;
        flag&=QName::tokenizer().insert(ns, "aftnngbnumd") == NS_rtf::LN_aftnngbnumd;
        flag&=QName::tokenizer().insert(ns, "margmirror") == NS_rtf::LN_margmirror;
        flag&=QName::tokenizer().insert(ns, "rxe") == NS_rtf::LN_rxe;
        flag&=QName::tokenizer().insert(ns, "s") == NS_rtf::LN_s;
        flag&=QName::tokenizer().insert(ns, "margr") == NS_rtf::LN_margr;
        flag&=QName::tokenizer().insert(ns, "aftnngbnumk") == NS_rtf::LN_aftnngbnumk;
        flag&=QName::tokenizer().insert(ns, "sa") == NS_rtf::LN_sa;
        flag&=QName::tokenizer().insert(ns, "margrsxn") == NS_rtf::LN_margrsxn;
        flag&=QName::tokenizer().insert(ns, "saauto") == NS_rtf::LN_saauto;
        flag&=QName::tokenizer().insert(ns, "aftnngbnuml") == NS_rtf::LN_aftnngbnuml;
        flag&=QName::tokenizer().insert(ns, "margt") == NS_rtf::LN_margt;
        flag&=QName::tokenizer().insert(ns, "aftnnrlc") == NS_rtf::LN_aftnnrlc;
        flag&=QName::tokenizer().insert(ns, "saftnnalc") == NS_rtf::LN_saftnnalc;
        flag&=QName::tokenizer().insert(ns, "margtsxn") == NS_rtf::LN_margtsxn;
        flag&=QName::tokenizer().insert(ns, "aftnnruc") == NS_rtf::LN_aftnnruc;
        flag&=QName::tokenizer().insert(ns, "saftnnar") == NS_rtf::LN_saftnnar;
        flag&=QName::tokenizer().insert(ns, "mhtmltag") == NS_rtf::LN_mhtmltag;
        flag&=QName::tokenizer().insert(ns, "saftnnauc") == NS_rtf::LN_saftnnauc;
        flag&=QName::tokenizer().insert(ns, "aftnnzodiac") == NS_rtf::LN_aftnnzodiac;
        flag&=QName::tokenizer().insert(ns, "min") == NS_rtf::LN_min;
        flag&=QName::tokenizer().insert(ns, "mo") == NS_rtf::LN_mo;
        flag&=QName::tokenizer().insert(ns, "saftnnchi") == NS_rtf::LN_saftnnchi;
        flag&=QName::tokenizer().insert(ns, "aftnnzodiacd") == NS_rtf::LN_aftnnzodiacd;
        flag&=QName::tokenizer().insert(ns, "msmcap") == NS_rtf::LN_msmcap;
        flag&=QName::tokenizer().insert(ns, "saftnnchosung") == NS_rtf::LN_saftnnchosung;
        flag&=QName::tokenizer().insert(ns, "aftnnzodiacl") == NS_rtf::LN_aftnnzodiacl;
        flag&=QName::tokenizer().insert(ns, "nestcell") == NS_rtf::LN_nestcell;
        flag&=QName::tokenizer().insert(ns, "saftnncnum") == NS_rtf::LN_saftnncnum;
        flag&=QName::tokenizer().insert(ns, "nestrow") == NS_rtf::LN_nestrow;
        flag&=QName::tokenizer().insert(ns, "aftnrestart") == NS_rtf::LN_aftnrestart;
        flag&=QName::tokenizer().insert(ns, "saftnndbar") == NS_rtf::LN_saftnndbar;
        flag&=QName::tokenizer().insert(ns, "aftnrstcont") == NS_rtf::LN_aftnrstcont;
        flag&=QName::tokenizer().insert(ns, "nesttableprops") == NS_rtf::LN_nesttableprops;
        flag&=QName::tokenizer().insert(ns, "saftnndbnum") == NS_rtf::LN_saftnndbnum;
        flag&=QName::tokenizer().insert(ns, "aftnsep") == NS_rtf::LN_aftnsep;
        flag&=QName::tokenizer().insert(ns, "nextfile") == NS_rtf::LN_nextfile;
        flag&=QName::tokenizer().insert(ns, "aftnsepc") == NS_rtf::LN_aftnsepc;
        flag&=QName::tokenizer().insert(ns, "saftnndbnumd") == NS_rtf::LN_saftnndbnumd;
        flag&=QName::tokenizer().insert(ns, "nobrkwrptbl") == NS_rtf::LN_nobrkwrptbl;
        flag&=QName::tokenizer().insert(ns, "aftnstart") == NS_rtf::LN_aftnstart;
        flag&=QName::tokenizer().insert(ns, "nocolbal") == NS_rtf::LN_nocolbal;
        flag&=QName::tokenizer().insert(ns, "saftnndbnumk") == NS_rtf::LN_saftnndbnumk;
        flag&=QName::tokenizer().insert(ns, "aftntj") == NS_rtf::LN_aftntj;
        flag&=QName::tokenizer().insert(ns, "ai") == NS_rtf::LN_ai;
        flag&=QName::tokenizer().insert(ns, "saftnndbnumt") == NS_rtf::LN_saftnndbnumt;
        flag&=QName::tokenizer().insert(ns, "nocompatoptions") == NS_rtf::LN_nocompatoptions;
        flag&=QName::tokenizer().insert(ns, "alang") == NS_rtf::LN_alang;
        flag&=QName::tokenizer().insert(ns, "nocwrap") == NS_rtf::LN_nocwrap;
        flag&=QName::tokenizer().insert(ns, "saftnnganada") == NS_rtf::LN_saftnnganada;
        flag&=QName::tokenizer().insert(ns, "noextrasprl") == NS_rtf::LN_noextrasprl;
        flag&=QName::tokenizer().insert(ns, "allowfieldendsel") == NS_rtf::LN_allowfieldendsel;
        flag&=QName::tokenizer().insert(ns, "saftnngbnum") == NS_rtf::LN_saftnngbnum;
        flag&=QName::tokenizer().insert(ns, "allprot") == NS_rtf::LN_allprot;
        flag&=QName::tokenizer().insert(ns, "nofchars") == NS_rtf::LN_nofchars;
        flag&=QName::tokenizer().insert(ns, "saftnngbnumd") == NS_rtf::LN_saftnngbnumd;
        flag&=QName::tokenizer().insert(ns, "alntblind") == NS_rtf::LN_alntblind;
        flag&=QName::tokenizer().insert(ns, "nofcharsws") == NS_rtf::LN_nofcharsws;
        flag&=QName::tokenizer().insert(ns, "saftnngbnumk") == NS_rtf::LN_saftnngbnumk;
        flag&=QName::tokenizer().insert(ns, "alt") == NS_rtf::LN_alt;
        flag&=QName::tokenizer().insert(ns, "nofpages") == NS_rtf::LN_nofpages;
        flag&=QName::tokenizer().insert(ns, "animtext") == NS_rtf::LN_animtext;
        flag&=QName::tokenizer().insert(ns, "saftnngbnuml") == NS_rtf::LN_saftnngbnuml;
        flag&=QName::tokenizer().insert(ns, "nofwords") == NS_rtf::LN_nofwords;
        flag&=QName::tokenizer().insert(ns, "annotation") == NS_rtf::LN_annotation;
        flag&=QName::tokenizer().insert(ns, "saftnnrlc") == NS_rtf::LN_saftnnrlc;
        flag&=QName::tokenizer().insert(ns, "nolead") == NS_rtf::LN_nolead;
        flag&=QName::tokenizer().insert(ns, "annotprot") == NS_rtf::LN_annotprot;
        flag&=QName::tokenizer().insert(ns, "noline") == NS_rtf::LN_noline;
        flag&=QName::tokenizer().insert(ns, "saftnnruc") == NS_rtf::LN_saftnnruc;
        flag&=QName::tokenizer().insert(ns, "ansi") == NS_rtf::LN_ansi;
        flag&=QName::tokenizer().insert(ns, "ansicpg") == NS_rtf::LN_ansicpg;
        flag&=QName::tokenizer().insert(ns, "nolnhtadjtbl") == NS_rtf::LN_nolnhtadjtbl;
        flag&=QName::tokenizer().insert(ns, "saftnnzodiac") == NS_rtf::LN_saftnnzodiac;
        flag&=QName::tokenizer().insert(ns, "aoutl") == NS_rtf::LN_aoutl;
        flag&=QName::tokenizer().insert(ns, "nonesttables") == NS_rtf::LN_nonesttables;
        flag&=QName::tokenizer().insert(ns, "saftnnzodiacd") == NS_rtf::LN_saftnnzodiacd;
        flag&=QName::tokenizer().insert(ns, "ApplyBrkRules") == NS_rtf::LN_ApplyBrkRules;
        flag&=QName::tokenizer().insert(ns, "nonshppict") == NS_rtf::LN_nonshppict;
        flag&=QName::tokenizer().insert(ns, "saftnnzodiacl") == NS_rtf::LN_saftnnzodiacl;
        flag&=QName::tokenizer().insert(ns, "ascaps") == NS_rtf::LN_ascaps;
        flag&=QName::tokenizer().insert(ns, "nooverflow") == NS_rtf::LN_nooverflow;
        flag&=QName::tokenizer().insert(ns, "ashad") == NS_rtf::LN_ashad;
        flag&=QName::tokenizer().insert(ns, "saftnrestart") == NS_rtf::LN_saftnrestart;
        flag&=QName::tokenizer().insert(ns, "noproof") == NS_rtf::LN_noproof;
        flag&=QName::tokenizer().insert(ns, "asianbrkrule") == NS_rtf::LN_asianbrkrule;
        flag&=QName::tokenizer().insert(ns, "saftnrstcont") == NS_rtf::LN_saftnrstcont;
        flag&=QName::tokenizer().insert(ns, "nosectexpand") == NS_rtf::LN_nosectexpand;
        flag&=QName::tokenizer().insert(ns, "aspalpha") == NS_rtf::LN_aspalpha;
        flag&=QName::tokenizer().insert(ns, "saftnstart") == NS_rtf::LN_saftnstart;
        flag&=QName::tokenizer().insert(ns, "aspnum") == NS_rtf::LN_aspnum;
        flag&=QName::tokenizer().insert(ns, "nosnaplinegrid") == NS_rtf::LN_nosnaplinegrid;
        flag&=QName::tokenizer().insert(ns, "sautoupd") == NS_rtf::LN_sautoupd;
        flag&=QName::tokenizer().insert(ns, "astrike") == NS_rtf::LN_astrike;
        flag&=QName::tokenizer().insert(ns, "sb") == NS_rtf::LN_sb;
        flag&=QName::tokenizer().insert(ns, "nospaceforul") == NS_rtf::LN_nospaceforul;
        flag&=QName::tokenizer().insert(ns, "atnauthor") == NS_rtf::LN_atnauthor;
        flag&=QName::tokenizer().insert(ns, "sbasedon") == NS_rtf::LN_sbasedon;
        flag&=QName::tokenizer().insert(ns, "nosupersub") == NS_rtf::LN_nosupersub;
        flag&=QName::tokenizer().insert(ns, "atndate") == NS_rtf::LN_atndate;
        flag&=QName::tokenizer().insert(ns, "sbauto") == NS_rtf::LN_sbauto;
        flag&=QName::tokenizer().insert(ns, "notabind") == NS_rtf::LN_notabind;
        flag&=QName::tokenizer().insert(ns, "atnicn") == NS_rtf::LN_atnicn;
        flag&=QName::tokenizer().insert(ns, "sbkcol") == NS_rtf::LN_sbkcol;
        flag&=QName::tokenizer().insert(ns, "atnid") == NS_rtf::LN_atnid;
        flag&=QName::tokenizer().insert(ns, "noultrlspc") == NS_rtf::LN_noultrlspc;
        flag&=QName::tokenizer().insert(ns, "sbkeven") == NS_rtf::LN_sbkeven;
        flag&=QName::tokenizer().insert(ns, "atnparent") == NS_rtf::LN_atnparent;
        flag&=QName::tokenizer().insert(ns, "sbknone") == NS_rtf::LN_sbknone;
        flag&=QName::tokenizer().insert(ns, "nowidctlpar") == NS_rtf::LN_nowidctlpar;
        flag&=QName::tokenizer().insert(ns, "atnref") == NS_rtf::LN_atnref;
        flag&=QName::tokenizer().insert(ns, "sbkodd") == NS_rtf::LN_sbkodd;
        flag&=QName::tokenizer().insert(ns, "nowrap") == NS_rtf::LN_nowrap;
        flag&=QName::tokenizer().insert(ns, "atntime") == NS_rtf::LN_atntime;
        flag&=QName::tokenizer().insert(ns, "sbkpage") == NS_rtf::LN_sbkpage;
        flag&=QName::tokenizer().insert(ns, "nowwrap") == NS_rtf::LN_nowwrap;
        flag&=QName::tokenizer().insert(ns, "sbys") == NS_rtf::LN_sbys;
        flag&=QName::tokenizer().insert(ns, "atrfend") == NS_rtf::LN_atrfend;
        flag&=QName::tokenizer().insert(ns, "noxlattoyen") == NS_rtf::LN_noxlattoyen;
        flag&=QName::tokenizer().insert(ns, "scaps") == NS_rtf::LN_scaps;
        flag&=QName::tokenizer().insert(ns, "atrfstart") == NS_rtf::LN_atrfstart;
        flag&=QName::tokenizer().insert(ns, "objalias") == NS_rtf::LN_objalias;
        flag&=QName::tokenizer().insert(ns, "scompose") == NS_rtf::LN_scompose;
        flag&=QName::tokenizer().insert(ns, "aul") == NS_rtf::LN_aul;
        flag&=QName::tokenizer().insert(ns, "objalign") == NS_rtf::LN_objalign;
        flag&=QName::tokenizer().insert(ns, "sec") == NS_rtf::LN_sec;
        flag&=QName::tokenizer().insert(ns, "auld") == NS_rtf::LN_auld;
        flag&=QName::tokenizer().insert(ns, "sect") == NS_rtf::LN_sect;
        flag&=QName::tokenizer().insert(ns, "auldb") == NS_rtf::LN_auldb;
        flag&=QName::tokenizer().insert(ns, "objattph") == NS_rtf::LN_objattph;
        flag&=QName::tokenizer().insert(ns, "sectd") == NS_rtf::LN_sectd;
        flag&=QName::tokenizer().insert(ns, "aulnone") == NS_rtf::LN_aulnone;
        flag&=QName::tokenizer().insert(ns, "objautlink") == NS_rtf::LN_objautlink;
        flag&=QName::tokenizer().insert(ns, "sectdefaultcl") == NS_rtf::LN_sectdefaultcl;
        flag&=QName::tokenizer().insert(ns, "aulw") == NS_rtf::LN_aulw;
        flag&=QName::tokenizer().insert(ns, "objclass") == NS_rtf::LN_objclass;
        flag&=QName::tokenizer().insert(ns, "aup") == NS_rtf::LN_aup;
        flag&=QName::tokenizer().insert(ns, "sectexpand") == NS_rtf::LN_sectexpand;
        flag&=QName::tokenizer().insert(ns, "objcropb") == NS_rtf::LN_objcropb;
        flag&=QName::tokenizer().insert(ns, "author") == NS_rtf::LN_author;
        flag&=QName::tokenizer().insert(ns, "b") == NS_rtf::LN_b;
        flag&=QName::tokenizer().insert(ns, "sectlinegrid") == NS_rtf::LN_sectlinegrid;
        flag&=QName::tokenizer().insert(ns, "objcropl") == NS_rtf::LN_objcropl;
        flag&=QName::tokenizer().insert(ns, "background") == NS_rtf::LN_background;
        flag&=QName::tokenizer().insert(ns, "sectnum") == NS_rtf::LN_sectnum;
        flag&=QName::tokenizer().insert(ns, "objcropr") == NS_rtf::LN_objcropr;
        flag&=QName::tokenizer().insert(ns, "bdbfhdr") == NS_rtf::LN_bdbfhdr;
        flag&=QName::tokenizer().insert(ns, "sectrsid") == NS_rtf::LN_sectrsid;
        flag&=QName::tokenizer().insert(ns, "objcropt") == NS_rtf::LN_objcropt;
        flag&=QName::tokenizer().insert(ns, "bdrrlswsix") == NS_rtf::LN_bdrrlswsix;
        flag&=QName::tokenizer().insert(ns, "objdata") == NS_rtf::LN_objdata;
        flag&=QName::tokenizer().insert(ns, "bgbdiag") == NS_rtf::LN_bgbdiag;
        flag&=QName::tokenizer().insert(ns, "sectspecifycl") == NS_rtf::LN_sectspecifycl;
        flag&=QName::tokenizer().insert(ns, "object") == NS_rtf::LN_object;
        flag&=QName::tokenizer().insert(ns, "bgcross") == NS_rtf::LN_bgcross;
        flag&=QName::tokenizer().insert(ns, "objemb") == NS_rtf::LN_objemb;
        flag&=QName::tokenizer().insert(ns, "sectspecifygen") == NS_rtf::LN_sectspecifygen;
        flag&=QName::tokenizer().insert(ns, "bgdcross") == NS_rtf::LN_bgdcross;
        flag&=QName::tokenizer().insert(ns, "objh") == NS_rtf::LN_objh;
        flag&=QName::tokenizer().insert(ns, "sectspecifyl") == NS_rtf::LN_sectspecifyl;
        flag&=QName::tokenizer().insert(ns, "objhtml") == NS_rtf::LN_objhtml;
        flag&=QName::tokenizer().insert(ns, "bgdkbdiag") == NS_rtf::LN_bgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "sectunlocked") == NS_rtf::LN_sectunlocked;
        flag&=QName::tokenizer().insert(ns, "objicemb") == NS_rtf::LN_objicemb;
        flag&=QName::tokenizer().insert(ns, "bgdkcross") == NS_rtf::LN_bgdkcross;
        flag&=QName::tokenizer().insert(ns, "sftnbj") == NS_rtf::LN_sftnbj;
        flag&=QName::tokenizer().insert(ns, "objlink") == NS_rtf::LN_objlink;
        flag&=QName::tokenizer().insert(ns, "bgdkdcross") == NS_rtf::LN_bgdkdcross;
        flag&=QName::tokenizer().insert(ns, "sftnnalc") == NS_rtf::LN_sftnnalc;
        flag&=QName::tokenizer().insert(ns, "objlock") == NS_rtf::LN_objlock;
        flag&=QName::tokenizer().insert(ns, "bgdkfdiag") == NS_rtf::LN_bgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "sftnnar") == NS_rtf::LN_sftnnar;
        flag&=QName::tokenizer().insert(ns, "objname") == NS_rtf::LN_objname;
        flag&=QName::tokenizer().insert(ns, "bgdkhoriz") == NS_rtf::LN_bgdkhoriz;
        flag&=QName::tokenizer().insert(ns, "objocx") == NS_rtf::LN_objocx;
        flag&=QName::tokenizer().insert(ns, "sftnnauc") == NS_rtf::LN_sftnnauc;
        flag&=QName::tokenizer().insert(ns, "bgdkvert") == NS_rtf::LN_bgdkvert;
        flag&=QName::tokenizer().insert(ns, "objpub") == NS_rtf::LN_objpub;
        flag&=QName::tokenizer().insert(ns, "sftnnchi") == NS_rtf::LN_sftnnchi;
        flag&=QName::tokenizer().insert(ns, "bgfdiag") == NS_rtf::LN_bgfdiag;
        flag&=QName::tokenizer().insert(ns, "objscalex") == NS_rtf::LN_objscalex;
        flag&=QName::tokenizer().insert(ns, "sftnnchosung") == NS_rtf::LN_sftnnchosung;
        flag&=QName::tokenizer().insert(ns, "bghoriz") == NS_rtf::LN_bghoriz;
        flag&=QName::tokenizer().insert(ns, "objscaley") == NS_rtf::LN_objscaley;
        flag&=QName::tokenizer().insert(ns, "sftnncnum") == NS_rtf::LN_sftnncnum;
        flag&=QName::tokenizer().insert(ns, "bgvert") == NS_rtf::LN_bgvert;
        flag&=QName::tokenizer().insert(ns, "bin") == NS_rtf::LN_bin;
        flag&=QName::tokenizer().insert(ns, "objsect") == NS_rtf::LN_objsect;
        flag&=QName::tokenizer().insert(ns, "sftnndbar") == NS_rtf::LN_sftnndbar;
        flag&=QName::tokenizer().insert(ns, "binfsxn") == NS_rtf::LN_binfsxn;
        flag&=QName::tokenizer().insert(ns, "objsetsize") == NS_rtf::LN_objsetsize;
        flag&=QName::tokenizer().insert(ns, "sftnndbnum") == NS_rtf::LN_sftnndbnum;
        flag&=QName::tokenizer().insert(ns, "binsxn") == NS_rtf::LN_binsxn;
        flag&=QName::tokenizer().insert(ns, "objsub") == NS_rtf::LN_objsub;
        flag&=QName::tokenizer().insert(ns, "sftnndbnumd") == NS_rtf::LN_sftnndbnumd;
        flag&=QName::tokenizer().insert(ns, "bkmkcolf") == NS_rtf::LN_bkmkcolf;
        flag&=QName::tokenizer().insert(ns, "objtime") == NS_rtf::LN_objtime;
        flag&=QName::tokenizer().insert(ns, "bkmkcoll") == NS_rtf::LN_bkmkcoll;
        flag&=QName::tokenizer().insert(ns, "sftnndbnumk") == NS_rtf::LN_sftnndbnumk;
        flag&=QName::tokenizer().insert(ns, "objtransy") == NS_rtf::LN_objtransy;
        flag&=QName::tokenizer().insert(ns, "bkmkend") == NS_rtf::LN_bkmkend;
        flag&=QName::tokenizer().insert(ns, "objupdate") == NS_rtf::LN_objupdate;
        flag&=QName::tokenizer().insert(ns, "sftnndbnumt") == NS_rtf::LN_sftnndbnumt;
        flag&=QName::tokenizer().insert(ns, "bkmkpub") == NS_rtf::LN_bkmkpub;
        flag&=QName::tokenizer().insert(ns, "objw") == NS_rtf::LN_objw;
        flag&=QName::tokenizer().insert(ns, "sftnnganada") == NS_rtf::LN_sftnnganada;
        flag&=QName::tokenizer().insert(ns, "oldas") == NS_rtf::LN_oldas;
        flag&=QName::tokenizer().insert(ns, "bkmkstart") == NS_rtf::LN_bkmkstart;
        flag&=QName::tokenizer().insert(ns, "sftnngbnum") == NS_rtf::LN_sftnngbnum;
        flag&=QName::tokenizer().insert(ns, "bliptag") == NS_rtf::LN_bliptag;
        flag&=QName::tokenizer().insert(ns, "oldcprops") == NS_rtf::LN_oldcprops;
        flag&=QName::tokenizer().insert(ns, "sftnngbnumd") == NS_rtf::LN_sftnngbnumd;
        flag&=QName::tokenizer().insert(ns, "blipuid") == NS_rtf::LN_blipuid;
        flag&=QName::tokenizer().insert(ns, "oldpprops") == NS_rtf::LN_oldpprops;
        flag&=QName::tokenizer().insert(ns, "blipupi") == NS_rtf::LN_blipupi;
        flag&=QName::tokenizer().insert(ns, "sftnngbnumk") == NS_rtf::LN_sftnngbnumk;
        flag&=QName::tokenizer().insert(ns, "oldsprops") == NS_rtf::LN_oldsprops;
        flag&=QName::tokenizer().insert(ns, "blue") == NS_rtf::LN_blue;
        flag&=QName::tokenizer().insert(ns, "oldtprops") == NS_rtf::LN_oldtprops;
        flag&=QName::tokenizer().insert(ns, "sftnngbnuml") == NS_rtf::LN_sftnngbnuml;
        flag&=QName::tokenizer().insert(ns, "bookfold") == NS_rtf::LN_bookfold;
        flag&=QName::tokenizer().insert(ns, "sftnnrlc") == NS_rtf::LN_sftnnrlc;
        flag&=QName::tokenizer().insert(ns, "oldlinewrap") == NS_rtf::LN_oldlinewrap;
        flag&=QName::tokenizer().insert(ns, "bookfoldrev") == NS_rtf::LN_bookfoldrev;
        flag&=QName::tokenizer().insert(ns, "operator") == NS_rtf::LN_operator;
        flag&=QName::tokenizer().insert(ns, "sftnnruc") == NS_rtf::LN_sftnnruc;
        flag&=QName::tokenizer().insert(ns, "otblrul") == NS_rtf::LN_otblrul;
        flag&=QName::tokenizer().insert(ns, "bookfoldsheets") == NS_rtf::LN_bookfoldsheets;
        flag&=QName::tokenizer().insert(ns, "box") == NS_rtf::LN_box;
        flag&=QName::tokenizer().insert(ns, "sftnnzodiac") == NS_rtf::LN_sftnnzodiac;
        flag&=QName::tokenizer().insert(ns, "outl") == NS_rtf::LN_outl;
        flag&=QName::tokenizer().insert(ns, "brdrart") == NS_rtf::LN_brdrart;
        flag&=QName::tokenizer().insert(ns, "sftnnzodiacd") == NS_rtf::LN_sftnnzodiacd;
        flag&=QName::tokenizer().insert(ns, "outlinelevel") == NS_rtf::LN_outlinelevel;
        flag&=QName::tokenizer().insert(ns, "brdrb") == NS_rtf::LN_brdrb;
        flag&=QName::tokenizer().insert(ns, "overlay") == NS_rtf::LN_overlay;
        flag&=QName::tokenizer().insert(ns, "sftnnzodiacl") == NS_rtf::LN_sftnnzodiacl;
        flag&=QName::tokenizer().insert(ns, "brdrbar") == NS_rtf::LN_brdrbar;
        flag&=QName::tokenizer().insert(ns, "page") == NS_rtf::LN_page;
        flag&=QName::tokenizer().insert(ns, "brdrbtw") == NS_rtf::LN_brdrbtw;
        flag&=QName::tokenizer().insert(ns, "sftnrestart") == NS_rtf::LN_sftnrestart;
        flag&=QName::tokenizer().insert(ns, "pagebb") == NS_rtf::LN_pagebb;
        flag&=QName::tokenizer().insert(ns, "brdrcf") == NS_rtf::LN_brdrcf;
        flag&=QName::tokenizer().insert(ns, "panose") == NS_rtf::LN_panose;
        flag&=QName::tokenizer().insert(ns, "sftnrstcont") == NS_rtf::LN_sftnrstcont;
        flag&=QName::tokenizer().insert(ns, "paperh") == NS_rtf::LN_paperh;
        flag&=QName::tokenizer().insert(ns, "brdrdash") == NS_rtf::LN_brdrdash;
        flag&=QName::tokenizer().insert(ns, "sftnrstpg") == NS_rtf::LN_sftnrstpg;
        flag&=QName::tokenizer().insert(ns, "paperw") == NS_rtf::LN_paperw;
        flag&=QName::tokenizer().insert(ns, "brdrdashd") == NS_rtf::LN_brdrdashd;
        flag&=QName::tokenizer().insert(ns, "par") == NS_rtf::LN_par;
        flag&=QName::tokenizer().insert(ns, "sftnstart") == NS_rtf::LN_sftnstart;
        flag&=QName::tokenizer().insert(ns, "brdrdashdd") == NS_rtf::LN_brdrdashdd;
        flag&=QName::tokenizer().insert(ns, "pararsid") == NS_rtf::LN_pararsid;
        flag&=QName::tokenizer().insert(ns, "sftntj") == NS_rtf::LN_sftntj;
        flag&=QName::tokenizer().insert(ns, "pard") == NS_rtf::LN_pard;
        flag&=QName::tokenizer().insert(ns, "shad") == NS_rtf::LN_shad;
        flag&=QName::tokenizer().insert(ns, "brdrdashdotstr") == NS_rtf::LN_brdrdashdotstr;
        flag&=QName::tokenizer().insert(ns, "pc") == NS_rtf::LN_pc;
        flag&=QName::tokenizer().insert(ns, "shading") == NS_rtf::LN_shading;
        flag&=QName::tokenizer().insert(ns, "pca") == NS_rtf::LN_pca;
        flag&=QName::tokenizer().insert(ns, "brdrdashsm") == NS_rtf::LN_brdrdashsm;
        flag&=QName::tokenizer().insert(ns, "shidden") == NS_rtf::LN_shidden;
        flag&=QName::tokenizer().insert(ns, "pgbrdrb") == NS_rtf::LN_pgbrdrb;
        flag&=QName::tokenizer().insert(ns, "brdrdb") == NS_rtf::LN_brdrdb;
        flag&=QName::tokenizer().insert(ns, "shift") == NS_rtf::LN_shift;
        flag&=QName::tokenizer().insert(ns, "brdrdot") == NS_rtf::LN_brdrdot;
        flag&=QName::tokenizer().insert(ns, "pgbrdrfoot") == NS_rtf::LN_pgbrdrfoot;
        flag&=QName::tokenizer().insert(ns, "shpbottom") == NS_rtf::LN_shpbottom;
        flag&=QName::tokenizer().insert(ns, "brdremboss") == NS_rtf::LN_brdremboss;
        flag&=QName::tokenizer().insert(ns, "pgbrdrhead") == NS_rtf::LN_pgbrdrhead;
        flag&=QName::tokenizer().insert(ns, "shpbxcolumn") == NS_rtf::LN_shpbxcolumn;
        flag&=QName::tokenizer().insert(ns, "pgbrdrl") == NS_rtf::LN_pgbrdrl;
        flag&=QName::tokenizer().insert(ns, "brdrengrave") == NS_rtf::LN_brdrengrave;
        flag&=QName::tokenizer().insert(ns, "shpbxignore") == NS_rtf::LN_shpbxignore;
        flag&=QName::tokenizer().insert(ns, "pgbrdropt") == NS_rtf::LN_pgbrdropt;
        flag&=QName::tokenizer().insert(ns, "brdrframe") == NS_rtf::LN_brdrframe;
        flag&=QName::tokenizer().insert(ns, "shpbxmargin") == NS_rtf::LN_shpbxmargin;
        flag&=QName::tokenizer().insert(ns, "pgbrdrr") == NS_rtf::LN_pgbrdrr;
        flag&=QName::tokenizer().insert(ns, "brdrhair") == NS_rtf::LN_brdrhair;
        flag&=QName::tokenizer().insert(ns, "shpbxpage") == NS_rtf::LN_shpbxpage;
        flag&=QName::tokenizer().insert(ns, "pgbrdrsnap") == NS_rtf::LN_pgbrdrsnap;
        flag&=QName::tokenizer().insert(ns, "brdrinset") == NS_rtf::LN_brdrinset;
        flag&=QName::tokenizer().insert(ns, "pgbrdrt") == NS_rtf::LN_pgbrdrt;
        flag&=QName::tokenizer().insert(ns, "shpbyignore") == NS_rtf::LN_shpbyignore;
        flag&=QName::tokenizer().insert(ns, "brdrl") == NS_rtf::LN_brdrl;
        flag&=QName::tokenizer().insert(ns, "pghsxn") == NS_rtf::LN_pghsxn;
        flag&=QName::tokenizer().insert(ns, "brdrnil") == NS_rtf::LN_brdrnil;
        flag&=QName::tokenizer().insert(ns, "shpbymargin") == NS_rtf::LN_shpbymargin;
        flag&=QName::tokenizer().insert(ns, "pgnbidia") == NS_rtf::LN_pgnbidia;
        flag&=QName::tokenizer().insert(ns, "brdroutset") == NS_rtf::LN_brdroutset;
        flag&=QName::tokenizer().insert(ns, "shpbypage") == NS_rtf::LN_shpbypage;
        flag&=QName::tokenizer().insert(ns, "pgnbidib") == NS_rtf::LN_pgnbidib;
        flag&=QName::tokenizer().insert(ns, "brdrr") == NS_rtf::LN_brdrr;
        flag&=QName::tokenizer().insert(ns, "shpbypara") == NS_rtf::LN_shpbypara;
        flag&=QName::tokenizer().insert(ns, "brdrs") == NS_rtf::LN_brdrs;
        flag&=QName::tokenizer().insert(ns, "pgnchosung") == NS_rtf::LN_pgnchosung;
        flag&=QName::tokenizer().insert(ns, "shpfblwtxt") == NS_rtf::LN_shpfblwtxt;
        flag&=QName::tokenizer().insert(ns, "brdrsh") == NS_rtf::LN_brdrsh;
        flag&=QName::tokenizer().insert(ns, "pgncnum") == NS_rtf::LN_pgncnum;
        flag&=QName::tokenizer().insert(ns, "shpfhdr") == NS_rtf::LN_shpfhdr;
        flag&=QName::tokenizer().insert(ns, "brdrt") == NS_rtf::LN_brdrt;
        flag&=QName::tokenizer().insert(ns, "pgncont") == NS_rtf::LN_pgncont;
        flag&=QName::tokenizer().insert(ns, "shpgrp") == NS_rtf::LN_shpgrp;
        flag&=QName::tokenizer().insert(ns, "brdrtbl") == NS_rtf::LN_brdrtbl;
        flag&=QName::tokenizer().insert(ns, "pgndbnum") == NS_rtf::LN_pgndbnum;
        flag&=QName::tokenizer().insert(ns, "shpleft") == NS_rtf::LN_shpleft;
        flag&=QName::tokenizer().insert(ns, "brdrth") == NS_rtf::LN_brdrth;
        flag&=QName::tokenizer().insert(ns, "pgndbnumd") == NS_rtf::LN_pgndbnumd;
        flag&=QName::tokenizer().insert(ns, "shplid") == NS_rtf::LN_shplid;
        flag&=QName::tokenizer().insert(ns, "brdrthtnlg") == NS_rtf::LN_brdrthtnlg;
        flag&=QName::tokenizer().insert(ns, "pgndbnumk") == NS_rtf::LN_pgndbnumk;
        flag&=QName::tokenizer().insert(ns, "shplockanchor") == NS_rtf::LN_shplockanchor;
        flag&=QName::tokenizer().insert(ns, "brdrthtnmg") == NS_rtf::LN_brdrthtnmg;
        flag&=QName::tokenizer().insert(ns, "pgndbnumt") == NS_rtf::LN_pgndbnumt;
        flag&=QName::tokenizer().insert(ns, "shppict") == NS_rtf::LN_shppict;
        flag&=QName::tokenizer().insert(ns, "brdrthtnsg") == NS_rtf::LN_brdrthtnsg;
        flag&=QName::tokenizer().insert(ns, "pgndec") == NS_rtf::LN_pgndec;
        flag&=QName::tokenizer().insert(ns, "shpright") == NS_rtf::LN_shpright;
        flag&=QName::tokenizer().insert(ns, "brdrtnthlg") == NS_rtf::LN_brdrtnthlg;
        flag&=QName::tokenizer().insert(ns, "pgndecd") == NS_rtf::LN_pgndecd;
        flag&=QName::tokenizer().insert(ns, "shprslt") == NS_rtf::LN_shprslt;
        flag&=QName::tokenizer().insert(ns, "brdrtnthmg") == NS_rtf::LN_brdrtnthmg;
        flag&=QName::tokenizer().insert(ns, "pgnganada") == NS_rtf::LN_pgnganada;
        flag&=QName::tokenizer().insert(ns, "shptop") == NS_rtf::LN_shptop;
        flag&=QName::tokenizer().insert(ns, "pgngbnum") == NS_rtf::LN_pgngbnum;
        flag&=QName::tokenizer().insert(ns, "brdrtnthsg") == NS_rtf::LN_brdrtnthsg;
        flag&=QName::tokenizer().insert(ns, "shptxt") == NS_rtf::LN_shptxt;
        flag&=QName::tokenizer().insert(ns, "shpwrk") == NS_rtf::LN_shpwrk;
        flag&=QName::tokenizer().insert(ns, "pgngbnumd") == NS_rtf::LN_pgngbnumd;
        flag&=QName::tokenizer().insert(ns, "brdrtnthtnlg") == NS_rtf::LN_brdrtnthtnlg;
        flag&=QName::tokenizer().insert(ns, "shpwr") == NS_rtf::LN_shpwr;
        flag&=QName::tokenizer().insert(ns, "pgngbnumk") == NS_rtf::LN_pgngbnumk;
        flag&=QName::tokenizer().insert(ns, "brdrtnthtnmg") == NS_rtf::LN_brdrtnthtnmg;
        flag&=QName::tokenizer().insert(ns, "shpz") == NS_rtf::LN_shpz;
        flag&=QName::tokenizer().insert(ns, "pgngbnuml") == NS_rtf::LN_pgngbnuml;
        flag&=QName::tokenizer().insert(ns, "sl") == NS_rtf::LN_sl;
        flag&=QName::tokenizer().insert(ns, "brdrtnthtnsg") == NS_rtf::LN_brdrtnthtnsg;
        flag&=QName::tokenizer().insert(ns, "slmult") == NS_rtf::LN_slmult;
        flag&=QName::tokenizer().insert(ns, "pgnhindia") == NS_rtf::LN_pgnhindia;
        flag&=QName::tokenizer().insert(ns, "brdrtriple") == NS_rtf::LN_brdrtriple;
        flag&=QName::tokenizer().insert(ns, "pgnhindib") == NS_rtf::LN_pgnhindib;
        flag&=QName::tokenizer().insert(ns, "snaptogridincell") == NS_rtf::LN_snaptogridincell;
        flag&=QName::tokenizer().insert(ns, "brdrw") == NS_rtf::LN_brdrw;
        flag&=QName::tokenizer().insert(ns, "pgnhindic") == NS_rtf::LN_pgnhindic;
        flag&=QName::tokenizer().insert(ns, "snext") == NS_rtf::LN_snext;
        flag&=QName::tokenizer().insert(ns, "brdrwavy") == NS_rtf::LN_brdrwavy;
        flag&=QName::tokenizer().insert(ns, "softcol") == NS_rtf::LN_softcol;
        flag&=QName::tokenizer().insert(ns, "pgnhindid") == NS_rtf::LN_pgnhindid;
        flag&=QName::tokenizer().insert(ns, "brdrwavydb") == NS_rtf::LN_brdrwavydb;
        flag&=QName::tokenizer().insert(ns, "pgnhn") == NS_rtf::LN_pgnhn;
        flag&=QName::tokenizer().insert(ns, "softlheight") == NS_rtf::LN_softlheight;
        flag&=QName::tokenizer().insert(ns, "brkfrm") == NS_rtf::LN_brkfrm;
        flag&=QName::tokenizer().insert(ns, "pgnhnsc") == NS_rtf::LN_pgnhnsc;
        flag&=QName::tokenizer().insert(ns, "brsp") == NS_rtf::LN_brsp;
        flag&=QName::tokenizer().insert(ns, "softline") == NS_rtf::LN_softline;
        flag&=QName::tokenizer().insert(ns, "pgnhnsh") == NS_rtf::LN_pgnhnsh;
        flag&=QName::tokenizer().insert(ns, "bullet") == NS_rtf::LN_bullet;
        flag&=QName::tokenizer().insert(ns, "softpage") == NS_rtf::LN_softpage;
        flag&=QName::tokenizer().insert(ns, "pgnhnsm") == NS_rtf::LN_pgnhnsm;
        flag&=QName::tokenizer().insert(ns, "spersonal") == NS_rtf::LN_spersonal;
        flag&=QName::tokenizer().insert(ns, "pgnhnsn") == NS_rtf::LN_pgnhnsn;
        flag&=QName::tokenizer().insert(ns, "buptim") == NS_rtf::LN_buptim;
        flag&=QName::tokenizer().insert(ns, "splytwnine") == NS_rtf::LN_splytwnine;
        flag&=QName::tokenizer().insert(ns, "bxe") == NS_rtf::LN_bxe;
        flag&=QName::tokenizer().insert(ns, "pgnhnsp") == NS_rtf::LN_pgnhnsp;
        flag&=QName::tokenizer().insert(ns, "caps") == NS_rtf::LN_caps;
        flag&=QName::tokenizer().insert(ns, "sprsbsp") == NS_rtf::LN_sprsbsp;
        flag&=QName::tokenizer().insert(ns, "pgnid") == NS_rtf::LN_pgnid;
        flag&=QName::tokenizer().insert(ns, "category") == NS_rtf::LN_category;
        flag&=QName::tokenizer().insert(ns, "cb") == NS_rtf::LN_cb;
        flag&=QName::tokenizer().insert(ns, "sprslnsp") == NS_rtf::LN_sprslnsp;
        flag&=QName::tokenizer().insert(ns, "pgnlcltr") == NS_rtf::LN_pgnlcltr;
        flag&=QName::tokenizer().insert(ns, "cbpat") == NS_rtf::LN_cbpat;
        flag&=QName::tokenizer().insert(ns, "sprsspbf") == NS_rtf::LN_sprsspbf;
        flag&=QName::tokenizer().insert(ns, "pgnlcrm") == NS_rtf::LN_pgnlcrm;
        flag&=QName::tokenizer().insert(ns, "cchs") == NS_rtf::LN_cchs;
        flag&=QName::tokenizer().insert(ns, "cell") == NS_rtf::LN_cell;
        flag&=QName::tokenizer().insert(ns, "sprstsm") == NS_rtf::LN_sprstsm;
        flag&=QName::tokenizer().insert(ns, "pgnrestart") == NS_rtf::LN_pgnrestart;
        flag&=QName::tokenizer().insert(ns, "cellx") == NS_rtf::LN_cellx;
        flag&=QName::tokenizer().insert(ns, "sprstsp") == NS_rtf::LN_sprstsp;
        flag&=QName::tokenizer().insert(ns, "pgnstart") == NS_rtf::LN_pgnstart;
        flag&=QName::tokenizer().insert(ns, "cf") == NS_rtf::LN_cf;
        flag&=QName::tokenizer().insert(ns, "spv") == NS_rtf::LN_spv;
        flag&=QName::tokenizer().insert(ns, "cfpat") == NS_rtf::LN_cfpat;
        flag&=QName::tokenizer().insert(ns, "pgnstarts") == NS_rtf::LN_pgnstarts;
        flag&=QName::tokenizer().insert(ns, "sreply") == NS_rtf::LN_sreply;
        flag&=QName::tokenizer().insert(ns, "cgrid") == NS_rtf::LN_cgrid;
        flag&=QName::tokenizer().insert(ns, "pgnthaia") == NS_rtf::LN_pgnthaia;
        flag&=QName::tokenizer().insert(ns, "ssemihidden") == NS_rtf::LN_ssemihidden;
        flag&=QName::tokenizer().insert(ns, "charrsid") == NS_rtf::LN_charrsid;
        flag&=QName::tokenizer().insert(ns, "pgnthaib") == NS_rtf::LN_pgnthaib;
        flag&=QName::tokenizer().insert(ns, "staticval") == NS_rtf::LN_staticval;
        flag&=QName::tokenizer().insert(ns, "charscalex") == NS_rtf::LN_charscalex;
        flag&=QName::tokenizer().insert(ns, "pgnthaic") == NS_rtf::LN_pgnthaic;
        flag&=QName::tokenizer().insert(ns, "stextflow") == NS_rtf::LN_stextflow;
        flag&=QName::tokenizer().insert(ns, "chatn") == NS_rtf::LN_chatn;
        flag&=QName::tokenizer().insert(ns, "strike") == NS_rtf::LN_strike;
        flag&=QName::tokenizer().insert(ns, "pgnucltr") == NS_rtf::LN_pgnucltr;
        flag&=QName::tokenizer().insert(ns, "chbgbdiag") == NS_rtf::LN_chbgbdiag;
        flag&=QName::tokenizer().insert(ns, "pgnucrm") == NS_rtf::LN_pgnucrm;
        flag&=QName::tokenizer().insert(ns, "striked1") == NS_rtf::LN_striked1;
        flag&=QName::tokenizer().insert(ns, "chbgcross") == NS_rtf::LN_chbgcross;
        flag&=QName::tokenizer().insert(ns, "stshfbi") == NS_rtf::LN_stshfbi;
        flag&=QName::tokenizer().insert(ns, "pgnvieta") == NS_rtf::LN_pgnvieta;
        flag&=QName::tokenizer().insert(ns, "pgnx") == NS_rtf::LN_pgnx;
        flag&=QName::tokenizer().insert(ns, "chbgdcross") == NS_rtf::LN_chbgdcross;
        flag&=QName::tokenizer().insert(ns, "stshfdbch") == NS_rtf::LN_stshfdbch;
        flag&=QName::tokenizer().insert(ns, "pgny") == NS_rtf::LN_pgny;
        flag&=QName::tokenizer().insert(ns, "stshfhich") == NS_rtf::LN_stshfhich;
        flag&=QName::tokenizer().insert(ns, "chbgdkbdiag") == NS_rtf::LN_chbgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "pgnzodiac") == NS_rtf::LN_pgnzodiac;
        flag&=QName::tokenizer().insert(ns, "stshfloch") == NS_rtf::LN_stshfloch;
        flag&=QName::tokenizer().insert(ns, "chbgdkcross") == NS_rtf::LN_chbgdkcross;
        flag&=QName::tokenizer().insert(ns, "pgnzodiacd") == NS_rtf::LN_pgnzodiacd;
        flag&=QName::tokenizer().insert(ns, "stylesheet") == NS_rtf::LN_stylesheet;
        flag&=QName::tokenizer().insert(ns, "chbgdkdcross") == NS_rtf::LN_chbgdkdcross;
        flag&=QName::tokenizer().insert(ns, "pgnzodiacl") == NS_rtf::LN_pgnzodiacl;
        flag&=QName::tokenizer().insert(ns, "styrsid") == NS_rtf::LN_styrsid;
        flag&=QName::tokenizer().insert(ns, "pgp") == NS_rtf::LN_pgp;
        flag&=QName::tokenizer().insert(ns, "sub") == NS_rtf::LN_sub;
        flag&=QName::tokenizer().insert(ns, "chbgdkfdiag") == NS_rtf::LN_chbgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "pgptbl") == NS_rtf::LN_pgptbl;
        flag&=QName::tokenizer().insert(ns, "subdocument") == NS_rtf::LN_subdocument;
        flag&=QName::tokenizer().insert(ns, "chbgdkhoriz") == NS_rtf::LN_chbgdkhoriz;
        flag&=QName::tokenizer().insert(ns, "pgwsxn") == NS_rtf::LN_pgwsxn;
        flag&=QName::tokenizer().insert(ns, "chbgdkvert") == NS_rtf::LN_chbgdkvert;
        flag&=QName::tokenizer().insert(ns, "phcol") == NS_rtf::LN_phcol;
        flag&=QName::tokenizer().insert(ns, "subfontbysize") == NS_rtf::LN_subfontbysize;
        flag&=QName::tokenizer().insert(ns, "subject") == NS_rtf::LN_subject;
        flag&=QName::tokenizer().insert(ns, "phmrg") == NS_rtf::LN_phmrg;
        flag&=QName::tokenizer().insert(ns, "chbgfdiag") == NS_rtf::LN_chbgfdiag;
        flag&=QName::tokenizer().insert(ns, "super") == NS_rtf::LN_super;
        flag&=QName::tokenizer().insert(ns, "phpg") == NS_rtf::LN_phpg;
        flag&=QName::tokenizer().insert(ns, "chbghoriz") == NS_rtf::LN_chbghoriz;
        flag&=QName::tokenizer().insert(ns, "swpbdr") == NS_rtf::LN_swpbdr;
        flag&=QName::tokenizer().insert(ns, "picbmp") == NS_rtf::LN_picbmp;
        flag&=QName::tokenizer().insert(ns, "tab") == NS_rtf::LN_tab;
        flag&=QName::tokenizer().insert(ns, "chbgvert") == NS_rtf::LN_chbgvert;
        flag&=QName::tokenizer().insert(ns, "picbpp") == NS_rtf::LN_picbpp;
        flag&=QName::tokenizer().insert(ns, "tabsnoovrlp") == NS_rtf::LN_tabsnoovrlp;
        flag&=QName::tokenizer().insert(ns, "chbrdr") == NS_rtf::LN_chbrdr;
        flag&=QName::tokenizer().insert(ns, "piccropb") == NS_rtf::LN_piccropb;
        flag&=QName::tokenizer().insert(ns, "taprtl") == NS_rtf::LN_taprtl;
        flag&=QName::tokenizer().insert(ns, "chcbpat") == NS_rtf::LN_chcbpat;
        flag&=QName::tokenizer().insert(ns, "piccropl") == NS_rtf::LN_piccropl;
        flag&=QName::tokenizer().insert(ns, "tb") == NS_rtf::LN_tb;
        flag&=QName::tokenizer().insert(ns, "chcfpat") == NS_rtf::LN_chcfpat;
        flag&=QName::tokenizer().insert(ns, "piccropr") == NS_rtf::LN_piccropr;
        flag&=QName::tokenizer().insert(ns, "tbllkbestfit") == NS_rtf::LN_tbllkbestfit;
        flag&=QName::tokenizer().insert(ns, "chdate") == NS_rtf::LN_chdate;
        flag&=QName::tokenizer().insert(ns, "piccropt") == NS_rtf::LN_piccropt;
        flag&=QName::tokenizer().insert(ns, "chdpa") == NS_rtf::LN_chdpa;
        flag&=QName::tokenizer().insert(ns, "tbllkborder") == NS_rtf::LN_tbllkborder;
        flag&=QName::tokenizer().insert(ns, "pich") == NS_rtf::LN_pich;
        flag&=QName::tokenizer().insert(ns, "chdpl") == NS_rtf::LN_chdpl;
        flag&=QName::tokenizer().insert(ns, "tbllkcolor") == NS_rtf::LN_tbllkcolor;
        flag&=QName::tokenizer().insert(ns, "pichgoal") == NS_rtf::LN_pichgoal;
        flag&=QName::tokenizer().insert(ns, "chftn") == NS_rtf::LN_chftn;
        flag&=QName::tokenizer().insert(ns, "tbllkfont") == NS_rtf::LN_tbllkfont;
        flag&=QName::tokenizer().insert(ns, "picprop") == NS_rtf::LN_picprop;
        flag&=QName::tokenizer().insert(ns, "chftnsep") == NS_rtf::LN_chftnsep;
        flag&=QName::tokenizer().insert(ns, "picscaled") == NS_rtf::LN_picscaled;
        flag&=QName::tokenizer().insert(ns, "chftnsepc") == NS_rtf::LN_chftnsepc;
        flag&=QName::tokenizer().insert(ns, "tbllkhdrcols") == NS_rtf::LN_tbllkhdrcols;
        flag&=QName::tokenizer().insert(ns, "chpgn") == NS_rtf::LN_chpgn;
        flag&=QName::tokenizer().insert(ns, "picscalex") == NS_rtf::LN_picscalex;
        flag&=QName::tokenizer().insert(ns, "tbllkhdrrows") == NS_rtf::LN_tbllkhdrrows;
        flag&=QName::tokenizer().insert(ns, "chshdng") == NS_rtf::LN_chshdng;
        flag&=QName::tokenizer().insert(ns, "picscaley") == NS_rtf::LN_picscaley;
        flag&=QName::tokenizer().insert(ns, "tbllklastcol") == NS_rtf::LN_tbllklastcol;
        flag&=QName::tokenizer().insert(ns, "pict") == NS_rtf::LN_pict;
        flag&=QName::tokenizer().insert(ns, "chtime") == NS_rtf::LN_chtime;
        flag&=QName::tokenizer().insert(ns, "picw") == NS_rtf::LN_picw;
        flag&=QName::tokenizer().insert(ns, "clbgbdiag") == NS_rtf::LN_clbgbdiag;
        flag&=QName::tokenizer().insert(ns, "tbllklastrow") == NS_rtf::LN_tbllklastrow;
        flag&=QName::tokenizer().insert(ns, "picwgoal") == NS_rtf::LN_picwgoal;
        flag&=QName::tokenizer().insert(ns, "clbgcross") == NS_rtf::LN_clbgcross;
        flag&=QName::tokenizer().insert(ns, "tbllkshading") == NS_rtf::LN_tbllkshading;
        flag&=QName::tokenizer().insert(ns, "plain") == NS_rtf::LN_plain;
        flag&=QName::tokenizer().insert(ns, "clbgdcross") == NS_rtf::LN_clbgdcross;
        flag&=QName::tokenizer().insert(ns, "tblrsid") == NS_rtf::LN_tblrsid;
        flag&=QName::tokenizer().insert(ns, "pmmetafile") == NS_rtf::LN_pmmetafile;
        flag&=QName::tokenizer().insert(ns, "tc") == NS_rtf::LN_tc;
        flag&=QName::tokenizer().insert(ns, "pn") == NS_rtf::LN_pn;
        flag&=QName::tokenizer().insert(ns, "clbgdkbdiag") == NS_rtf::LN_clbgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "tcelld") == NS_rtf::LN_tcelld;
        flag&=QName::tokenizer().insert(ns, "pnacross") == NS_rtf::LN_pnacross;
        flag&=QName::tokenizer().insert(ns, "tcf") == NS_rtf::LN_tcf;
        flag&=QName::tokenizer().insert(ns, "clbgdkcross") == NS_rtf::LN_clbgdkcross;
        flag&=QName::tokenizer().insert(ns, "tcl") == NS_rtf::LN_tcl;
        flag&=QName::tokenizer().insert(ns, "pnaiu") == NS_rtf::LN_pnaiu;
        flag&=QName::tokenizer().insert(ns, "tcn") == NS_rtf::LN_tcn;
        flag&=QName::tokenizer().insert(ns, "clbgdkdcross") == NS_rtf::LN_clbgdkdcross;
        flag&=QName::tokenizer().insert(ns, "pnaiud") == NS_rtf::LN_pnaiud;
        flag&=QName::tokenizer().insert(ns, "pnaiueo") == NS_rtf::LN_pnaiueo;
        flag&=QName::tokenizer().insert(ns, "clbgdkfdiag") == NS_rtf::LN_clbgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "tdfrmtxtBottom") == NS_rtf::LN_tdfrmtxtBottom;
        flag&=QName::tokenizer().insert(ns, "pnaiueod") == NS_rtf::LN_pnaiueod;
        flag&=QName::tokenizer().insert(ns, "clbgdkhor") == NS_rtf::LN_clbgdkhor;
        flag&=QName::tokenizer().insert(ns, "tdfrmtxtLeft") == NS_rtf::LN_tdfrmtxtLeft;
        flag&=QName::tokenizer().insert(ns, "pnb") == NS_rtf::LN_pnb;
        flag&=QName::tokenizer().insert(ns, "clbgdkvert") == NS_rtf::LN_clbgdkvert;
        flag&=QName::tokenizer().insert(ns, "pnbidia") == NS_rtf::LN_pnbidia;
        flag&=QName::tokenizer().insert(ns, "tdfrmtxtRight") == NS_rtf::LN_tdfrmtxtRight;
        flag&=QName::tokenizer().insert(ns, "clbgfdiag") == NS_rtf::LN_clbgfdiag;
        flag&=QName::tokenizer().insert(ns, "pnbidib") == NS_rtf::LN_pnbidib;
        flag&=QName::tokenizer().insert(ns, "tdfrmtxtTop") == NS_rtf::LN_tdfrmtxtTop;
        flag&=QName::tokenizer().insert(ns, "template") == NS_rtf::LN_template;
        flag&=QName::tokenizer().insert(ns, "clbghoriz") == NS_rtf::LN_clbghoriz;
        flag&=QName::tokenizer().insert(ns, "pncaps") == NS_rtf::LN_pncaps;
        flag&=QName::tokenizer().insert(ns, "time") == NS_rtf::LN_time;
        flag&=QName::tokenizer().insert(ns, "title") == NS_rtf::LN_title;
        flag&=QName::tokenizer().insert(ns, "pncard") == NS_rtf::LN_pncard;
        flag&=QName::tokenizer().insert(ns, "clbgvert") == NS_rtf::LN_clbgvert;
        flag&=QName::tokenizer().insert(ns, "pncf") == NS_rtf::LN_pncf;
        flag&=QName::tokenizer().insert(ns, "titlepg") == NS_rtf::LN_titlepg;
        flag&=QName::tokenizer().insert(ns, "clbrdrb") == NS_rtf::LN_clbrdrb;
        flag&=QName::tokenizer().insert(ns, "tldot") == NS_rtf::LN_tldot;
        flag&=QName::tokenizer().insert(ns, "pnchosung") == NS_rtf::LN_pnchosung;
        flag&=QName::tokenizer().insert(ns, "clbrdrl") == NS_rtf::LN_clbrdrl;
        flag&=QName::tokenizer().insert(ns, "tleq") == NS_rtf::LN_tleq;
        flag&=QName::tokenizer().insert(ns, "pncnum") == NS_rtf::LN_pncnum;
        flag&=QName::tokenizer().insert(ns, "clbrdrr") == NS_rtf::LN_clbrdrr;
        flag&=QName::tokenizer().insert(ns, "tlhyph") == NS_rtf::LN_tlhyph;
        flag&=QName::tokenizer().insert(ns, "pndbnum") == NS_rtf::LN_pndbnum;
        flag&=QName::tokenizer().insert(ns, "clbrdrt") == NS_rtf::LN_clbrdrt;
        flag&=QName::tokenizer().insert(ns, "tlmdot") == NS_rtf::LN_tlmdot;
        flag&=QName::tokenizer().insert(ns, "pndbnumd") == NS_rtf::LN_pndbnumd;
        flag&=QName::tokenizer().insert(ns, "clcbpat") == NS_rtf::LN_clcbpat;
        flag&=QName::tokenizer().insert(ns, "tlth") == NS_rtf::LN_tlth;
        flag&=QName::tokenizer().insert(ns, "pndbnumk") == NS_rtf::LN_pndbnumk;
        flag&=QName::tokenizer().insert(ns, "tlul") == NS_rtf::LN_tlul;
        flag&=QName::tokenizer().insert(ns, "clcbpatraw") == NS_rtf::LN_clcbpatraw;
        flag&=QName::tokenizer().insert(ns, "pndbnuml") == NS_rtf::LN_pndbnuml;
        flag&=QName::tokenizer().insert(ns, "clcfpat") == NS_rtf::LN_clcfpat;
        flag&=QName::tokenizer().insert(ns, "toplinepunct") == NS_rtf::LN_toplinepunct;
        flag&=QName::tokenizer().insert(ns, "pndbnumt") == NS_rtf::LN_pndbnumt;
        flag&=QName::tokenizer().insert(ns, "tphcol") == NS_rtf::LN_tphcol;
        flag&=QName::tokenizer().insert(ns, "clcfpatraw") == NS_rtf::LN_clcfpatraw;
        flag&=QName::tokenizer().insert(ns, "pndec") == NS_rtf::LN_pndec;
        flag&=QName::tokenizer().insert(ns, "tphmrg") == NS_rtf::LN_tphmrg;
        flag&=QName::tokenizer().insert(ns, "cldgll") == NS_rtf::LN_cldgll;
        flag&=QName::tokenizer().insert(ns, "pndecd") == NS_rtf::LN_pndecd;
        flag&=QName::tokenizer().insert(ns, "tphpg") == NS_rtf::LN_tphpg;
        flag&=QName::tokenizer().insert(ns, "cldglu") == NS_rtf::LN_cldglu;
        flag&=QName::tokenizer().insert(ns, "pnf") == NS_rtf::LN_pnf;
        flag&=QName::tokenizer().insert(ns, "tposnegx") == NS_rtf::LN_tposnegx;
        flag&=QName::tokenizer().insert(ns, "clFitText") == NS_rtf::LN_clFitText;
        flag&=QName::tokenizer().insert(ns, "pnfs") == NS_rtf::LN_pnfs;
        flag&=QName::tokenizer().insert(ns, "tposnegy") == NS_rtf::LN_tposnegy;
        flag&=QName::tokenizer().insert(ns, "pnganada") == NS_rtf::LN_pnganada;
        flag&=QName::tokenizer().insert(ns, "clftsWidth") == NS_rtf::LN_clftsWidth;
        flag&=QName::tokenizer().insert(ns, "tposxc") == NS_rtf::LN_tposxc;
        flag&=QName::tokenizer().insert(ns, "clmgf") == NS_rtf::LN_clmgf;
        flag&=QName::tokenizer().insert(ns, "pngblip") == NS_rtf::LN_pngblip;
        flag&=QName::tokenizer().insert(ns, "tposxi") == NS_rtf::LN_tposxi;
        flag&=QName::tokenizer().insert(ns, "clmrg") == NS_rtf::LN_clmrg;
        flag&=QName::tokenizer().insert(ns, "pngbnum") == NS_rtf::LN_pngbnum;
        flag&=QName::tokenizer().insert(ns, "tposxl") == NS_rtf::LN_tposxl;
        flag&=QName::tokenizer().insert(ns, "cloWrap") == NS_rtf::LN_cloWrap;
        flag&=QName::tokenizer().insert(ns, "pngbnumd") == NS_rtf::LN_pngbnumd;
        flag&=QName::tokenizer().insert(ns, "tposx") == NS_rtf::LN_tposx;
        flag&=QName::tokenizer().insert(ns, "clpadb") == NS_rtf::LN_clpadb;
        flag&=QName::tokenizer().insert(ns, "pngbnumk") == NS_rtf::LN_pngbnumk;
        flag&=QName::tokenizer().insert(ns, "tposxo") == NS_rtf::LN_tposxo;
        flag&=QName::tokenizer().insert(ns, "clpadfb") == NS_rtf::LN_clpadfb;
        flag&=QName::tokenizer().insert(ns, "pngbnuml") == NS_rtf::LN_pngbnuml;
        flag&=QName::tokenizer().insert(ns, "tposxr") == NS_rtf::LN_tposxr;
        flag&=QName::tokenizer().insert(ns, "clpadfl") == NS_rtf::LN_clpadfl;
        flag&=QName::tokenizer().insert(ns, "pnhang") == NS_rtf::LN_pnhang;
        flag&=QName::tokenizer().insert(ns, "tposy") == NS_rtf::LN_tposy;
        flag&=QName::tokenizer().insert(ns, "clpadfr") == NS_rtf::LN_clpadfr;
        flag&=QName::tokenizer().insert(ns, "pni") == NS_rtf::LN_pni;
        flag&=QName::tokenizer().insert(ns, "tposyb") == NS_rtf::LN_tposyb;
        flag&=QName::tokenizer().insert(ns, "clpadft") == NS_rtf::LN_clpadft;
        flag&=QName::tokenizer().insert(ns, "pnindent") == NS_rtf::LN_pnindent;
        flag&=QName::tokenizer().insert(ns, "tposyc") == NS_rtf::LN_tposyc;
        flag&=QName::tokenizer().insert(ns, "clpadl") == NS_rtf::LN_clpadl;
        flag&=QName::tokenizer().insert(ns, "pniroha") == NS_rtf::LN_pniroha;
        flag&=QName::tokenizer().insert(ns, "tposyil") == NS_rtf::LN_tposyil;
        flag&=QName::tokenizer().insert(ns, "clpadr") == NS_rtf::LN_clpadr;
        flag&=QName::tokenizer().insert(ns, "pnirohad") == NS_rtf::LN_pnirohad;
        flag&=QName::tokenizer().insert(ns, "tposyin") == NS_rtf::LN_tposyin;
        flag&=QName::tokenizer().insert(ns, "clpadt") == NS_rtf::LN_clpadt;
        flag&=QName::tokenizer().insert(ns, "pnlcltr") == NS_rtf::LN_pnlcltr;
        flag&=QName::tokenizer().insert(ns, "tposyoutv") == NS_rtf::LN_tposyoutv;
        flag&=QName::tokenizer().insert(ns, "clshdng") == NS_rtf::LN_clshdng;
        flag&=QName::tokenizer().insert(ns, "pnlcrm") == NS_rtf::LN_pnlcrm;
        flag&=QName::tokenizer().insert(ns, "tposyt") == NS_rtf::LN_tposyt;
        flag&=QName::tokenizer().insert(ns, "clshdngraw") == NS_rtf::LN_clshdngraw;
        flag&=QName::tokenizer().insert(ns, "pnlvl") == NS_rtf::LN_pnlvl;
        flag&=QName::tokenizer().insert(ns, "tpvmrg") == NS_rtf::LN_tpvmrg;
        flag&=QName::tokenizer().insert(ns, "pnlvlblt") == NS_rtf::LN_pnlvlblt;
        flag&=QName::tokenizer().insert(ns, "clshdrawnil") == NS_rtf::LN_clshdrawnil;
        flag&=QName::tokenizer().insert(ns, "tpvpara") == NS_rtf::LN_tpvpara;
        flag&=QName::tokenizer().insert(ns, "tpvpg") == NS_rtf::LN_tpvpg;
        flag&=QName::tokenizer().insert(ns, "pnlvlbody") == NS_rtf::LN_pnlvlbody;
        flag&=QName::tokenizer().insert(ns, "cltxbtlr") == NS_rtf::LN_cltxbtlr;
        flag&=QName::tokenizer().insert(ns, "tqc") == NS_rtf::LN_tqc;
        flag&=QName::tokenizer().insert(ns, "pnlvlcont") == NS_rtf::LN_pnlvlcont;
        flag&=QName::tokenizer().insert(ns, "cltxlrtb") == NS_rtf::LN_cltxlrtb;
        flag&=QName::tokenizer().insert(ns, "tqdec") == NS_rtf::LN_tqdec;
        flag&=QName::tokenizer().insert(ns, "pnnumonce") == NS_rtf::LN_pnnumonce;
        flag&=QName::tokenizer().insert(ns, "cltxlrtbv") == NS_rtf::LN_cltxlrtbv;
        flag&=QName::tokenizer().insert(ns, "tqr") == NS_rtf::LN_tqr;
        flag&=QName::tokenizer().insert(ns, "pnord") == NS_rtf::LN_pnord;
        flag&=QName::tokenizer().insert(ns, "cltxtbrl") == NS_rtf::LN_cltxtbrl;
        flag&=QName::tokenizer().insert(ns, "transmf") == NS_rtf::LN_transmf;
        flag&=QName::tokenizer().insert(ns, "pnordt") == NS_rtf::LN_pnordt;
        flag&=QName::tokenizer().insert(ns, "trauth") == NS_rtf::LN_trauth;
        flag&=QName::tokenizer().insert(ns, "cltxtbrlv") == NS_rtf::LN_cltxtbrlv;
        flag&=QName::tokenizer().insert(ns, "pnprev") == NS_rtf::LN_pnprev;
        flag&=QName::tokenizer().insert(ns, "trautofit") == NS_rtf::LN_trautofit;
        flag&=QName::tokenizer().insert(ns, "clvertalb") == NS_rtf::LN_clvertalb;
        flag&=QName::tokenizer().insert(ns, "pnqc") == NS_rtf::LN_pnqc;
        flag&=QName::tokenizer().insert(ns, "trbgbdiag") == NS_rtf::LN_trbgbdiag;
        flag&=QName::tokenizer().insert(ns, "pnql") == NS_rtf::LN_pnql;
        flag&=QName::tokenizer().insert(ns, "clvertalc") == NS_rtf::LN_clvertalc;
        flag&=QName::tokenizer().insert(ns, "pnqr") == NS_rtf::LN_pnqr;
        flag&=QName::tokenizer().insert(ns, "trbgcross") == NS_rtf::LN_trbgcross;
        flag&=QName::tokenizer().insert(ns, "clvertalt") == NS_rtf::LN_clvertalt;
        flag&=QName::tokenizer().insert(ns, "pnrauth") == NS_rtf::LN_pnrauth;
        flag&=QName::tokenizer().insert(ns, "clvmgf") == NS_rtf::LN_clvmgf;
        flag&=QName::tokenizer().insert(ns, "trbgdcross") == NS_rtf::LN_trbgdcross;
        flag&=QName::tokenizer().insert(ns, "pnrdate") == NS_rtf::LN_pnrdate;
        flag&=QName::tokenizer().insert(ns, "clvmrg") == NS_rtf::LN_clvmrg;
        flag&=QName::tokenizer().insert(ns, "trbgdkbdiag") == NS_rtf::LN_trbgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "pnrestart") == NS_rtf::LN_pnrestart;
        flag&=QName::tokenizer().insert(ns, "clwWidth") == NS_rtf::LN_clwWidth;
        flag&=QName::tokenizer().insert(ns, "trbgdkcross") == NS_rtf::LN_trbgdkcross;
        flag&=QName::tokenizer().insert(ns, "pnrnfc") == NS_rtf::LN_pnrnfc;
        flag&=QName::tokenizer().insert(ns, "collapsed") == NS_rtf::LN_collapsed;
        flag&=QName::tokenizer().insert(ns, "pnrnot") == NS_rtf::LN_pnrnot;
        flag&=QName::tokenizer().insert(ns, "trbgdkdcross") == NS_rtf::LN_trbgdkdcross;
        flag&=QName::tokenizer().insert(ns, "colno") == NS_rtf::LN_colno;
        flag&=QName::tokenizer().insert(ns, "pnrpnbr") == NS_rtf::LN_pnrpnbr;
        flag&=QName::tokenizer().insert(ns, "colortbl") == NS_rtf::LN_colortbl;
        flag&=QName::tokenizer().insert(ns, "trbgdkfdiag") == NS_rtf::LN_trbgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "cols") == NS_rtf::LN_cols;
        flag&=QName::tokenizer().insert(ns, "pnrrgb") == NS_rtf::LN_pnrrgb;
        flag&=QName::tokenizer().insert(ns, "trbgdkhor") == NS_rtf::LN_trbgdkhor;
        flag&=QName::tokenizer().insert(ns, "colsr") == NS_rtf::LN_colsr;
        flag&=QName::tokenizer().insert(ns, "pnrstart") == NS_rtf::LN_pnrstart;
        flag&=QName::tokenizer().insert(ns, "colsx") == NS_rtf::LN_colsx;
        flag&=QName::tokenizer().insert(ns, "trbgdkvert") == NS_rtf::LN_trbgdkvert;
        flag&=QName::tokenizer().insert(ns, "pnrstop") == NS_rtf::LN_pnrstop;
        flag&=QName::tokenizer().insert(ns, "column") == NS_rtf::LN_column;
        flag&=QName::tokenizer().insert(ns, "trbgfdiag") == NS_rtf::LN_trbgfdiag;
        flag&=QName::tokenizer().insert(ns, "pnrxst") == NS_rtf::LN_pnrxst;
        flag&=QName::tokenizer().insert(ns, "colw") == NS_rtf::LN_colw;
        flag&=QName::tokenizer().insert(ns, "comment") == NS_rtf::LN_comment;
        flag&=QName::tokenizer().insert(ns, "pnscaps") == NS_rtf::LN_pnscaps;
        flag&=QName::tokenizer().insert(ns, "trbghoriz") == NS_rtf::LN_trbghoriz;
        flag&=QName::tokenizer().insert(ns, "company") == NS_rtf::LN_company;
        flag&=QName::tokenizer().insert(ns, "pnseclvl") == NS_rtf::LN_pnseclvl;
        flag&=QName::tokenizer().insert(ns, "trbgvert") == NS_rtf::LN_trbgvert;
        flag&=QName::tokenizer().insert(ns, "cpg") == NS_rtf::LN_cpg;
        flag&=QName::tokenizer().insert(ns, "pnsp") == NS_rtf::LN_pnsp;
        flag&=QName::tokenizer().insert(ns, "trbrdrb") == NS_rtf::LN_trbrdrb;
        flag&=QName::tokenizer().insert(ns, "crauth") == NS_rtf::LN_crauth;
        flag&=QName::tokenizer().insert(ns, "pnstart") == NS_rtf::LN_pnstart;
        flag&=QName::tokenizer().insert(ns, "crdate") == NS_rtf::LN_crdate;
        flag&=QName::tokenizer().insert(ns, "trbrdrh") == NS_rtf::LN_trbrdrh;
        flag&=QName::tokenizer().insert(ns, "pnstrike") == NS_rtf::LN_pnstrike;
        flag&=QName::tokenizer().insert(ns, "creatim") == NS_rtf::LN_creatim;
        flag&=QName::tokenizer().insert(ns, "trbrdrl") == NS_rtf::LN_trbrdrl;
        flag&=QName::tokenizer().insert(ns, "cs") == NS_rtf::LN_cs;
        flag&=QName::tokenizer().insert(ns, "pntext") == NS_rtf::LN_pntext;
        flag&=QName::tokenizer().insert(ns, "trbrdrr") == NS_rtf::LN_trbrdrr;
        flag&=QName::tokenizer().insert(ns, "ctrl") == NS_rtf::LN_ctrl;
        flag&=QName::tokenizer().insert(ns, "pntxta") == NS_rtf::LN_pntxta;
        flag&=QName::tokenizer().insert(ns, "trbrdrt") == NS_rtf::LN_trbrdrt;
        flag&=QName::tokenizer().insert(ns, "cts") == NS_rtf::LN_cts;
        flag&=QName::tokenizer().insert(ns, "pntxtb") == NS_rtf::LN_pntxtb;
        flag&=QName::tokenizer().insert(ns, "cufi") == NS_rtf::LN_cufi;
        flag&=QName::tokenizer().insert(ns, "trbrdrv") == NS_rtf::LN_trbrdrv;
        flag&=QName::tokenizer().insert(ns, "pnucltr") == NS_rtf::LN_pnucltr;
        flag&=QName::tokenizer().insert(ns, "culi") == NS_rtf::LN_culi;
        flag&=QName::tokenizer().insert(ns, "trcbpat") == NS_rtf::LN_trcbpat;
        flag&=QName::tokenizer().insert(ns, "curi") == NS_rtf::LN_curi;
        flag&=QName::tokenizer().insert(ns, "pnucrm") == NS_rtf::LN_pnucrm;
        flag&=QName::tokenizer().insert(ns, "trcfpat") == NS_rtf::LN_trcfpat;
        flag&=QName::tokenizer().insert(ns, "pnul") == NS_rtf::LN_pnul;
        flag&=QName::tokenizer().insert(ns, "cvmme") == NS_rtf::LN_cvmme;
        flag&=QName::tokenizer().insert(ns, "trdate") == NS_rtf::LN_trdate;
        flag&=QName::tokenizer().insert(ns, "pnuld") == NS_rtf::LN_pnuld;
        flag&=QName::tokenizer().insert(ns, "datafield") == NS_rtf::LN_datafield;
        flag&=QName::tokenizer().insert(ns, "date") == NS_rtf::LN_date;
        flag&=QName::tokenizer().insert(ns, "trftsWidthA") == NS_rtf::LN_trftsWidthA;
        flag&=QName::tokenizer().insert(ns, "pnuldash") == NS_rtf::LN_pnuldash;
        flag&=QName::tokenizer().insert(ns, "dbch") == NS_rtf::LN_dbch;
        flag&=QName::tokenizer().insert(ns, "pnuldashd") == NS_rtf::LN_pnuldashd;
        flag&=QName::tokenizer().insert(ns, "trftsWidthB") == NS_rtf::LN_trftsWidthB;
        flag&=QName::tokenizer().insert(ns, "deff") == NS_rtf::LN_deff;
        flag&=QName::tokenizer().insert(ns, "pnuldashdd") == NS_rtf::LN_pnuldashdd;
        flag&=QName::tokenizer().insert(ns, "trftsWidth") == NS_rtf::LN_trftsWidth;
        flag&=QName::tokenizer().insert(ns, "defformat") == NS_rtf::LN_defformat;
        flag&=QName::tokenizer().insert(ns, "pnuldb") == NS_rtf::LN_pnuldb;
        flag&=QName::tokenizer().insert(ns, "trgaph") == NS_rtf::LN_trgaph;
        flag&=QName::tokenizer().insert(ns, "deflang") == NS_rtf::LN_deflang;
        flag&=QName::tokenizer().insert(ns, "trhdr") == NS_rtf::LN_trhdr;
        flag&=QName::tokenizer().insert(ns, "pnulhair") == NS_rtf::LN_pnulhair;
        flag&=QName::tokenizer().insert(ns, "deflangfe") == NS_rtf::LN_deflangfe;
        flag&=QName::tokenizer().insert(ns, "trkeep") == NS_rtf::LN_trkeep;
        flag&=QName::tokenizer().insert(ns, "pnulnone") == NS_rtf::LN_pnulnone;
        flag&=QName::tokenizer().insert(ns, "defshp") == NS_rtf::LN_defshp;
        flag&=QName::tokenizer().insert(ns, "trleft") == NS_rtf::LN_trleft;
        flag&=QName::tokenizer().insert(ns, "pnulth") == NS_rtf::LN_pnulth;
        flag&=QName::tokenizer().insert(ns, "deftab") == NS_rtf::LN_deftab;
        flag&=QName::tokenizer().insert(ns, "trowd") == NS_rtf::LN_trowd;
        flag&=QName::tokenizer().insert(ns, "pnulw") == NS_rtf::LN_pnulw;
        flag&=QName::tokenizer().insert(ns, "deleted") == NS_rtf::LN_deleted;
        flag&=QName::tokenizer().insert(ns, "trpaddb") == NS_rtf::LN_trpaddb;
        flag&=QName::tokenizer().insert(ns, "pnulwave") == NS_rtf::LN_pnulwave;
        flag&=QName::tokenizer().insert(ns, "delrsid") == NS_rtf::LN_delrsid;
        flag&=QName::tokenizer().insert(ns, "trpaddfb") == NS_rtf::LN_trpaddfb;
        flag&=QName::tokenizer().insert(ns, "pnzodiac") == NS_rtf::LN_pnzodiac;
        flag&=QName::tokenizer().insert(ns, "dfrauth") == NS_rtf::LN_dfrauth;
        flag&=QName::tokenizer().insert(ns, "trpaddfl") == NS_rtf::LN_trpaddfl;
        flag&=QName::tokenizer().insert(ns, "pnzodiacd") == NS_rtf::LN_pnzodiacd;
        flag&=QName::tokenizer().insert(ns, "dfrdate") == NS_rtf::LN_dfrdate;
        flag&=QName::tokenizer().insert(ns, "trpaddfr") == NS_rtf::LN_trpaddfr;
        flag&=QName::tokenizer().insert(ns, "dfrmtxtx") == NS_rtf::LN_dfrmtxtx;
        flag&=QName::tokenizer().insert(ns, "pnzodiacl") == NS_rtf::LN_pnzodiacl;
        flag&=QName::tokenizer().insert(ns, "trpaddft") == NS_rtf::LN_trpaddft;
        flag&=QName::tokenizer().insert(ns, "posnegx") == NS_rtf::LN_posnegx;
        flag&=QName::tokenizer().insert(ns, "dfrmtxty") == NS_rtf::LN_dfrmtxty;
        flag&=QName::tokenizer().insert(ns, "trpaddl") == NS_rtf::LN_trpaddl;
        flag&=QName::tokenizer().insert(ns, "posnegy") == NS_rtf::LN_posnegy;
        flag&=QName::tokenizer().insert(ns, "dfrstart") == NS_rtf::LN_dfrstart;
        flag&=QName::tokenizer().insert(ns, "trpaddr") == NS_rtf::LN_trpaddr;
        flag&=QName::tokenizer().insert(ns, "posx") == NS_rtf::LN_posx;
        flag&=QName::tokenizer().insert(ns, "dfrstop") == NS_rtf::LN_dfrstop;
        flag&=QName::tokenizer().insert(ns, "trpaddt") == NS_rtf::LN_trpaddt;
        flag&=QName::tokenizer().insert(ns, "posxc") == NS_rtf::LN_posxc;
        flag&=QName::tokenizer().insert(ns, "dfrxst") == NS_rtf::LN_dfrxst;
        flag&=QName::tokenizer().insert(ns, "trpat") == NS_rtf::LN_trpat;
        flag&=QName::tokenizer().insert(ns, "posxi") == NS_rtf::LN_posxi;
        flag&=QName::tokenizer().insert(ns, "dghorigin") == NS_rtf::LN_dghorigin;
        flag&=QName::tokenizer().insert(ns, "trqc") == NS_rtf::LN_trqc;
        flag&=QName::tokenizer().insert(ns, "posxl") == NS_rtf::LN_posxl;
        flag&=QName::tokenizer().insert(ns, "trql") == NS_rtf::LN_trql;
        flag&=QName::tokenizer().insert(ns, "dghshow") == NS_rtf::LN_dghshow;
        flag&=QName::tokenizer().insert(ns, "posxo") == NS_rtf::LN_posxo;
        flag&=QName::tokenizer().insert(ns, "trqr") == NS_rtf::LN_trqr;
        flag&=QName::tokenizer().insert(ns, "posxr") == NS_rtf::LN_posxr;
        flag&=QName::tokenizer().insert(ns, "dghspace") == NS_rtf::LN_dghspace;
        flag&=QName::tokenizer().insert(ns, "trrh") == NS_rtf::LN_trrh;
        flag&=QName::tokenizer().insert(ns, "posy") == NS_rtf::LN_posy;
        flag&=QName::tokenizer().insert(ns, "dgmargin") == NS_rtf::LN_dgmargin;
        flag&=QName::tokenizer().insert(ns, "trshdng") == NS_rtf::LN_trshdng;
        flag&=QName::tokenizer().insert(ns, "posyb") == NS_rtf::LN_posyb;
        flag&=QName::tokenizer().insert(ns, "dgsnap") == NS_rtf::LN_dgsnap;
        flag&=QName::tokenizer().insert(ns, "trspdb") == NS_rtf::LN_trspdb;
        flag&=QName::tokenizer().insert(ns, "posyc") == NS_rtf::LN_posyc;
        flag&=QName::tokenizer().insert(ns, "dgvorigin") == NS_rtf::LN_dgvorigin;
        flag&=QName::tokenizer().insert(ns, "trspdfb") == NS_rtf::LN_trspdfb;
        flag&=QName::tokenizer().insert(ns, "posyil") == NS_rtf::LN_posyil;
        flag&=QName::tokenizer().insert(ns, "dgvshow") == NS_rtf::LN_dgvshow;
        flag&=QName::tokenizer().insert(ns, "posyin") == NS_rtf::LN_posyin;
        flag&=QName::tokenizer().insert(ns, "trspdfl") == NS_rtf::LN_trspdfl;
        flag&=QName::tokenizer().insert(ns, "dgvspace") == NS_rtf::LN_dgvspace;
        flag&=QName::tokenizer().insert(ns, "posyout") == NS_rtf::LN_posyout;
        flag&=QName::tokenizer().insert(ns, "trspdfr") == NS_rtf::LN_trspdfr;
        flag&=QName::tokenizer().insert(ns, "dibitmap") == NS_rtf::LN_dibitmap;
        flag&=QName::tokenizer().insert(ns, "posyt") == NS_rtf::LN_posyt;
        flag&=QName::tokenizer().insert(ns, "trspdft") == NS_rtf::LN_trspdft;
        flag&=QName::tokenizer().insert(ns, "dn") == NS_rtf::LN_dn;
        flag&=QName::tokenizer().insert(ns, "prcolbl") == NS_rtf::LN_prcolbl;
        flag&=QName::tokenizer().insert(ns, "trspdl") == NS_rtf::LN_trspdl;
        flag&=QName::tokenizer().insert(ns, "dntblnsbdb") == NS_rtf::LN_dntblnsbdb;
        flag&=QName::tokenizer().insert(ns, "trspdr") == NS_rtf::LN_trspdr;
        flag&=QName::tokenizer().insert(ns, "printdata") == NS_rtf::LN_printdata;
        flag&=QName::tokenizer().insert(ns, "do") == NS_rtf::LN_do;
        flag&=QName::tokenizer().insert(ns, "trspdt") == NS_rtf::LN_trspdt;
        flag&=QName::tokenizer().insert(ns, "printim") == NS_rtf::LN_printim;
        flag&=QName::tokenizer().insert(ns, "dobxcolumn") == NS_rtf::LN_dobxcolumn;
        flag&=QName::tokenizer().insert(ns, "private") == NS_rtf::LN_private;
        flag&=QName::tokenizer().insert(ns, "truncatefontheight") == NS_rtf::LN_truncatefontheight;
        flag&=QName::tokenizer().insert(ns, "dobxmargin") == NS_rtf::LN_dobxmargin;
        flag&=QName::tokenizer().insert(ns, "propname") == NS_rtf::LN_propname;
        flag&=QName::tokenizer().insert(ns, "trwWidthA") == NS_rtf::LN_trwWidthA;
        flag&=QName::tokenizer().insert(ns, "dobxpage") == NS_rtf::LN_dobxpage;
        flag&=QName::tokenizer().insert(ns, "proptype") == NS_rtf::LN_proptype;
        flag&=QName::tokenizer().insert(ns, "trwWidthB") == NS_rtf::LN_trwWidthB;
        flag&=QName::tokenizer().insert(ns, "dobymargin") == NS_rtf::LN_dobymargin;
        flag&=QName::tokenizer().insert(ns, "trwWidth") == NS_rtf::LN_trwWidth;
        flag&=QName::tokenizer().insert(ns, "dobypage") == NS_rtf::LN_dobypage;
        flag&=QName::tokenizer().insert(ns, "ts") == NS_rtf::LN_ts;
        flag&=QName::tokenizer().insert(ns, "tsbgbdiag") == NS_rtf::LN_tsbgbdiag;
        flag&=QName::tokenizer().insert(ns, "tsbgcross") == NS_rtf::LN_tsbgcross;
        flag&=QName::tokenizer().insert(ns, "tsbgdcross") == NS_rtf::LN_tsbgdcross;
        flag&=QName::tokenizer().insert(ns, "tsbgdkbdiag") == NS_rtf::LN_tsbgdkbdiag;
        flag&=QName::tokenizer().insert(ns, "tsbgdkcross") == NS_rtf::LN_tsbgdkcross;
        flag&=QName::tokenizer().insert(ns, "tsbgdkdcross") == NS_rtf::LN_tsbgdkdcross;
        flag&=QName::tokenizer().insert(ns, "tsbgdkfdiag") == NS_rtf::LN_tsbgdkfdiag;
        flag&=QName::tokenizer().insert(ns, "tsbgdkhor") == NS_rtf::LN_tsbgdkhor;
        flag&=QName::tokenizer().insert(ns, "tsbgdkvert") == NS_rtf::LN_tsbgdkvert;
        flag&=QName::tokenizer().insert(ns, "tsbgfdiag") == NS_rtf::LN_tsbgfdiag;
        flag&=QName::tokenizer().insert(ns, "tsbghoriz") == NS_rtf::LN_tsbghoriz;
        flag&=QName::tokenizer().insert(ns, "tsbgvert") == NS_rtf::LN_tsbgvert;
        flag&=QName::tokenizer().insert(ns, "tsbrdrb") == NS_rtf::LN_tsbrdrb;
        flag&=QName::tokenizer().insert(ns, "tsbrdrdgl") == NS_rtf::LN_tsbrdrdgl;
        flag&=QName::tokenizer().insert(ns, "tsbrdrdgr") == NS_rtf::LN_tsbrdrdgr;
        flag&=QName::tokenizer().insert(ns, "tsbrdrh") == NS_rtf::LN_tsbrdrh;
        flag&=QName::tokenizer().insert(ns, "tsbrdrl") == NS_rtf::LN_tsbrdrl;
        flag&=QName::tokenizer().insert(ns, "tsbrdrr") == NS_rtf::LN_tsbrdrr;
        flag&=QName::tokenizer().insert(ns, "tsbrdrt") == NS_rtf::LN_tsbrdrt;
        flag&=QName::tokenizer().insert(ns, "tsbrdrv") == NS_rtf::LN_tsbrdrv;
        flag&=QName::tokenizer().insert(ns, "tscbandhorzeven") == NS_rtf::LN_tscbandhorzeven;
        flag&=QName::tokenizer().insert(ns, "tscbandhorzodd") == NS_rtf::LN_tscbandhorzodd;
        flag&=QName::tokenizer().insert(ns, "tscbandsh") == NS_rtf::LN_tscbandsh;
        flag&=QName::tokenizer().insert(ns, "tscbandsv") == NS_rtf::LN_tscbandsv;
        flag&=QName::tokenizer().insert(ns, "tscbandverteven") == NS_rtf::LN_tscbandverteven;
        flag&=QName::tokenizer().insert(ns, "tscbandvertodd") == NS_rtf::LN_tscbandvertodd;
        flag&=QName::tokenizer().insert(ns, "tscellcbpat") == NS_rtf::LN_tscellcbpat;
        flag&=QName::tokenizer().insert(ns, "tscellcfpat") == NS_rtf::LN_tscellcfpat;
        flag&=QName::tokenizer().insert(ns, "tscellpaddb") == NS_rtf::LN_tscellpaddb;
        flag&=QName::tokenizer().insert(ns, "tscellpaddfb") == NS_rtf::LN_tscellpaddfb;
        flag&=QName::tokenizer().insert(ns, "tscellpaddfl") == NS_rtf::LN_tscellpaddfl;
        flag&=QName::tokenizer().insert(ns, "tscellpaddfr") == NS_rtf::LN_tscellpaddfr;
        flag&=QName::tokenizer().insert(ns, "tscellpaddft") == NS_rtf::LN_tscellpaddft;
        flag&=QName::tokenizer().insert(ns, "tscellpaddl") == NS_rtf::LN_tscellpaddl;
        flag&=QName::tokenizer().insert(ns, "tscellpaddr") == NS_rtf::LN_tscellpaddr;
        flag&=QName::tokenizer().insert(ns, "tscellpaddt") == NS_rtf::LN_tscellpaddt;
        flag&=QName::tokenizer().insert(ns, "tscellpct") == NS_rtf::LN_tscellpct;
        flag&=QName::tokenizer().insert(ns, "tscellwidth") == NS_rtf::LN_tscellwidth;
        flag&=QName::tokenizer().insert(ns, "tscellwidthfts") == NS_rtf::LN_tscellwidthfts;
        flag&=QName::tokenizer().insert(ns, "tscfirstcol") == NS_rtf::LN_tscfirstcol;
        flag&=QName::tokenizer().insert(ns, "tscfirstrow") == NS_rtf::LN_tscfirstrow;
        flag&=QName::tokenizer().insert(ns, "tsclastcol") == NS_rtf::LN_tsclastcol;
        flag&=QName::tokenizer().insert(ns, "tsclastrow") == NS_rtf::LN_tsclastrow;
        flag&=QName::tokenizer().insert(ns, "tscnecell") == NS_rtf::LN_tscnecell;
        flag&=QName::tokenizer().insert(ns, "tscnwcell") == NS_rtf::LN_tscnwcell;
        flag&=QName::tokenizer().insert(ns, "tscsecell") == NS_rtf::LN_tscsecell;
        flag&=QName::tokenizer().insert(ns, "tscswcell") == NS_rtf::LN_tscswcell;
        flag&=QName::tokenizer().insert(ns, "tsd") == NS_rtf::LN_tsd;
        flag&=QName::tokenizer().insert(ns, "tsnowrap") == NS_rtf::LN_tsnowrap;
        flag&=QName::tokenizer().insert(ns, "tsrowd") == NS_rtf::LN_tsrowd;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rtf failed", flag);}
    void test_smil(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:smil-compatible:1.0";
        flag&=QName::tokenizer().insert(ns, "calcMode") == NS_smil::LN_calcMode;
        flag&=QName::tokenizer().insert(ns, "type") == NS_smil::LN_type;
        flag&=QName::tokenizer().insert(ns, "subtype") == NS_smil::LN_subtype;
        flag&=QName::tokenizer().insert(ns, "direction") == NS_smil::LN_direction;
        flag&=QName::tokenizer().insert(ns, "fadeColor") == NS_smil::LN_fadeColor;
        flag&=QName::tokenizer().insert(ns, "mode") == NS_smil::LN_mode;
        flag&=QName::tokenizer().insert(ns, "targetElement") == NS_smil::LN_targetElement;
        flag&=QName::tokenizer().insert(ns, "attributeName") == NS_smil::LN_attributeName;
        flag&=QName::tokenizer().insert(ns, "values") == NS_smil::LN_values;
        flag&=QName::tokenizer().insert(ns, "keyTimes") == NS_smil::LN_keyTimes;
        flag&=QName::tokenizer().insert(ns, "keySplines") == NS_smil::LN_keySplines;
        flag&=QName::tokenizer().insert(ns, "accumulate") == NS_smil::LN_accumulate;
        flag&=QName::tokenizer().insert(ns, "additive") == NS_smil::LN_additive;
        flag&=QName::tokenizer().insert(ns, "to") == NS_smil::LN_to;
        flag&=QName::tokenizer().insert(ns, "from") == NS_smil::LN_from;
        flag&=QName::tokenizer().insert(ns, "by") == NS_smil::LN_by;
        flag&=QName::tokenizer().insert(ns, "begin") == NS_smil::LN_begin;
        flag&=QName::tokenizer().insert(ns, "end") == NS_smil::LN_end;
        flag&=QName::tokenizer().insert(ns, "dur") == NS_smil::LN_dur;
        flag&=QName::tokenizer().insert(ns, "endsync") == NS_smil::LN_endsync;
        flag&=QName::tokenizer().insert(ns, "repeatDur") == NS_smil::LN_repeatDur;
        flag&=QName::tokenizer().insert(ns, "repeatCount") == NS_smil::LN_repeatCount;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_smil::LN_fill;
        flag&=QName::tokenizer().insert(ns, "fillDefault") == NS_smil::LN_fillDefault;
        flag&=QName::tokenizer().insert(ns, "restart") == NS_smil::LN_restart;
        flag&=QName::tokenizer().insert(ns, "restartDefault") == NS_smil::LN_restartDefault;
        flag&=QName::tokenizer().insert(ns, "accelerate") == NS_smil::LN_accelerate;
        flag&=QName::tokenizer().insert(ns, "decelerate") == NS_smil::LN_decelerate;
        flag&=QName::tokenizer().insert(ns, "autoReverse") == NS_smil::LN_autoReverse;
        CPPUNIT_ASSERT_MESSAGE("test for namespace smil failed", flag);}
    void test_w12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/word/2005/8/25/wordml";
        flag&=QName::tokenizer().insert(ns, "truncateFontHeight") == NS_w12::LN_truncateFontHeight;
        flag&=QName::tokenizer().insert(ns, "mwSmallCaps") == NS_w12::LN_mwSmallCaps;
        flag&=QName::tokenizer().insert(ns, "usePrinterMetrics") == NS_w12::LN_usePrinterMetrics;
        flag&=QName::tokenizer().insert(ns, "ww6BorderRules") == NS_w12::LN_ww6BorderRules;
        flag&=QName::tokenizer().insert(ns, "wrapTrailSpaces") == NS_w12::LN_wrapTrailSpaces;
        flag&=QName::tokenizer().insert(ns, "footnoteLayoutLikeWW8") == NS_w12::LN_footnoteLayoutLikeWW8;
        flag&=QName::tokenizer().insert(ns, "shapeLayoutLikeWW8") == NS_w12::LN_shapeLayoutLikeWW8;
        flag&=QName::tokenizer().insert(ns, "alignTablesRowByRow") == NS_w12::LN_alignTablesRowByRow;
        flag&=QName::tokenizer().insert(ns, "forgetLastTabAlignment") == NS_w12::LN_forgetLastTabAlignment;
        flag&=QName::tokenizer().insert(ns, "adjustLineHeightInTable") == NS_w12::LN_adjustLineHeightInTable;
        flag&=QName::tokenizer().insert(ns, "autoSpaceLikeWord95") == NS_w12::LN_autoSpaceLikeWord95;
        flag&=QName::tokenizer().insert(ns, "noSpaceRaiseLower") == NS_w12::LN_noSpaceRaiseLower;
        flag&=QName::tokenizer().insert(ns, "doNotUseHTMLParagraphAutoSpacing") == NS_w12::LN_doNotUseHTMLParagraphAutoSpacing;
        flag&=QName::tokenizer().insert(ns, "layoutRawTableWidth") == NS_w12::LN_layoutRawTableWidth;
        flag&=QName::tokenizer().insert(ns, "layoutTableRowsApart") == NS_w12::LN_layoutTableRowsApart;
        flag&=QName::tokenizer().insert(ns, "useWord97LineBreakingRules") == NS_w12::LN_useWord97LineBreakingRules;
        flag&=QName::tokenizer().insert(ns, "breakWrappedTables") == NS_w12::LN_breakWrappedTables;
        flag&=QName::tokenizer().insert(ns, "snapToGridInCell") == NS_w12::LN_snapToGridInCell;
        flag&=QName::tokenizer().insert(ns, "dontAllowFieldEndSelect") == NS_w12::LN_dontAllowFieldEndSelect;
        flag&=QName::tokenizer().insert(ns, "applyBreakingRules") == NS_w12::LN_applyBreakingRules;
        flag&=QName::tokenizer().insert(ns, "wrapTextWithPunct") == NS_w12::LN_wrapTextWithPunct;
        flag&=QName::tokenizer().insert(ns, "useAsianBreakRules") == NS_w12::LN_useAsianBreakRules;
        flag&=QName::tokenizer().insert(ns, "useWord2002TableStyleRules") == NS_w12::LN_useWord2002TableStyleRules;
        flag&=QName::tokenizer().insert(ns, "dontGrowAutofit") == NS_w12::LN_dontGrowAutofit;
        flag&=QName::tokenizer().insert(ns, "useFELayout") == NS_w12::LN_useFELayout;
        flag&=QName::tokenizer().insert(ns, "useNormalStyleForList") == NS_w12::LN_useNormalStyleForList;
        flag&=QName::tokenizer().insert(ns, "dontUseIndentAsNumberingTabStop") == NS_w12::LN_dontUseIndentAsNumberingTabStop;
        flag&=QName::tokenizer().insert(ns, "dontUseWord11FELineBreak") == NS_w12::LN_dontUseWord11FELineBreak;
        flag&=QName::tokenizer().insert(ns, "allowSpaceOfSameStyleInTable") == NS_w12::LN_allowSpaceOfSameStyleInTable;
        flag&=QName::tokenizer().insert(ns, "ww11IndentRules") == NS_w12::LN_ww11IndentRules;
        flag&=QName::tokenizer().insert(ns, "dontAutofitConstrainedTables") == NS_w12::LN_dontAutofitConstrainedTables;
        flag&=QName::tokenizer().insert(ns, "autofitLikeWW11") == NS_w12::LN_autofitLikeWW11;
        flag&=QName::tokenizer().insert(ns, "underlineTabInNumList") == NS_w12::LN_underlineTabInNumList;
        flag&=QName::tokenizer().insert(ns, "docVar") == NS_w12::LN_docVar;
        flag&=QName::tokenizer().insert(ns, "rsidRoot") == NS_w12::LN_rsidRoot;
        flag&=QName::tokenizer().insert(ns, "rsid") == NS_w12::LN_rsid;
        flag&=QName::tokenizer().insert(ns, "rPrDefault") == NS_w12::LN_rPrDefault;
        flag&=QName::tokenizer().insert(ns, "pPrDefault") == NS_w12::LN_pPrDefault;
        flag&=QName::tokenizer().insert(ns, "view") == NS_w12::LN_view;
        flag&=QName::tokenizer().insert(ns, "zoom") == NS_w12::LN_zoom;
        flag&=QName::tokenizer().insert(ns, "removePersonalInformation") == NS_w12::LN_removePersonalInformation;
        flag&=QName::tokenizer().insert(ns, "removeDateAndTime") == NS_w12::LN_removeDateAndTime;
        flag&=QName::tokenizer().insert(ns, "dontDisplayPageBoundaries") == NS_w12::LN_dontDisplayPageBoundaries;
        flag&=QName::tokenizer().insert(ns, "displayBackgroundShape") == NS_w12::LN_displayBackgroundShape;
        flag&=QName::tokenizer().insert(ns, "printPostScriptOverText") == NS_w12::LN_printPostScriptOverText;
        flag&=QName::tokenizer().insert(ns, "printFractionalCharacterWidth") == NS_w12::LN_printFractionalCharacterWidth;
        flag&=QName::tokenizer().insert(ns, "printFormsData") == NS_w12::LN_printFormsData;
        flag&=QName::tokenizer().insert(ns, "embedTrueTypeFonts") == NS_w12::LN_embedTrueTypeFonts;
        flag&=QName::tokenizer().insert(ns, "doNotEmbedSystemFonts") == NS_w12::LN_doNotEmbedSystemFonts;
        flag&=QName::tokenizer().insert(ns, "saveSubsetFonts") == NS_w12::LN_saveSubsetFonts;
        flag&=QName::tokenizer().insert(ns, "saveFormsData") == NS_w12::LN_saveFormsData;
        flag&=QName::tokenizer().insert(ns, "mirrorMargins") == NS_w12::LN_mirrorMargins;
        flag&=QName::tokenizer().insert(ns, "alignBordersAndEdges") == NS_w12::LN_alignBordersAndEdges;
        flag&=QName::tokenizer().insert(ns, "bordersDontSurroundHeader") == NS_w12::LN_bordersDontSurroundHeader;
        flag&=QName::tokenizer().insert(ns, "bordersDontSurroundFooter") == NS_w12::LN_bordersDontSurroundFooter;
        flag&=QName::tokenizer().insert(ns, "gutterAtTop") == NS_w12::LN_gutterAtTop;
        flag&=QName::tokenizer().insert(ns, "hideSpellingErrors") == NS_w12::LN_hideSpellingErrors;
        flag&=QName::tokenizer().insert(ns, "hideGrammaticalErrors") == NS_w12::LN_hideGrammaticalErrors;
        flag&=QName::tokenizer().insert(ns, "activeWritingStyle") == NS_w12::LN_activeWritingStyle;
        flag&=QName::tokenizer().insert(ns, "proofState") == NS_w12::LN_proofState;
        flag&=QName::tokenizer().insert(ns, "formsDesign") == NS_w12::LN_formsDesign;
        flag&=QName::tokenizer().insert(ns, "attachedTemplate") == NS_w12::LN_attachedTemplate;
        flag&=QName::tokenizer().insert(ns, "linkStyles") == NS_w12::LN_linkStyles;
        flag&=QName::tokenizer().insert(ns, "stylePaneFormatFilter") == NS_w12::LN_stylePaneFormatFilter;
        flag&=QName::tokenizer().insert(ns, "stylePaneSortMethod") == NS_w12::LN_stylePaneSortMethod;
        flag&=QName::tokenizer().insert(ns, "documentType") == NS_w12::LN_documentType;
        flag&=QName::tokenizer().insert(ns, "mailMerge") == NS_w12::LN_mailMerge;
        flag&=QName::tokenizer().insert(ns, "revisionView") == NS_w12::LN_revisionView;
        flag&=QName::tokenizer().insert(ns, "ilvl") == NS_w12::LN_ilvl;
        flag&=QName::tokenizer().insert(ns, "ilfo") == NS_w12::LN_ilfo;
        flag&=QName::tokenizer().insert(ns, "top") == NS_w12::LN_top;
        flag&=QName::tokenizer().insert(ns, "left") == NS_w12::LN_left;
        flag&=QName::tokenizer().insert(ns, "bottom") == NS_w12::LN_bottom;
        flag&=QName::tokenizer().insert(ns, "right") == NS_w12::LN_right;
        flag&=QName::tokenizer().insert(ns, "between") == NS_w12::LN_between;
        flag&=QName::tokenizer().insert(ns, "bar") == NS_w12::LN_bar;
        flag&=QName::tokenizer().insert(ns, "tab") == NS_w12::LN_tab;
        flag&=QName::tokenizer().insert(ns, "pStyle") == NS_w12::LN_pStyle;
        flag&=QName::tokenizer().insert(ns, "keepNext") == NS_w12::LN_keepNext;
        flag&=QName::tokenizer().insert(ns, "keepLines") == NS_w12::LN_keepLines;
        flag&=QName::tokenizer().insert(ns, "pageBreakBefore") == NS_w12::LN_pageBreakBefore;
        flag&=QName::tokenizer().insert(ns, "framePr") == NS_w12::LN_framePr;
        flag&=QName::tokenizer().insert(ns, "widowControl") == NS_w12::LN_widowControl;
        flag&=QName::tokenizer().insert(ns, "listPr") == NS_w12::LN_listPr;
        flag&=QName::tokenizer().insert(ns, "supressLineNumbers") == NS_w12::LN_supressLineNumbers;
        flag&=QName::tokenizer().insert(ns, "pBdr") == NS_w12::LN_pBdr;
        flag&=QName::tokenizer().insert(ns, "shd") == NS_w12::LN_shd;
        flag&=QName::tokenizer().insert(ns, "tabs") == NS_w12::LN_tabs;
        flag&=QName::tokenizer().insert(ns, "suppressAutoHyphens") == NS_w12::LN_suppressAutoHyphens;
        flag&=QName::tokenizer().insert(ns, "kinsoku") == NS_w12::LN_kinsoku;
        flag&=QName::tokenizer().insert(ns, "wordWrap") == NS_w12::LN_wordWrap;
        flag&=QName::tokenizer().insert(ns, "overflowPunct") == NS_w12::LN_overflowPunct;
        flag&=QName::tokenizer().insert(ns, "topLinePunct") == NS_w12::LN_topLinePunct;
        flag&=QName::tokenizer().insert(ns, "autoSpaceDE") == NS_w12::LN_autoSpaceDE;
        flag&=QName::tokenizer().insert(ns, "autoSpaceDN") == NS_w12::LN_autoSpaceDN;
        flag&=QName::tokenizer().insert(ns, "bidi") == NS_w12::LN_bidi;
        flag&=QName::tokenizer().insert(ns, "adjustRightInd") == NS_w12::LN_adjustRightInd;
        flag&=QName::tokenizer().insert(ns, "snapToGrid") == NS_w12::LN_snapToGrid;
        flag&=QName::tokenizer().insert(ns, "spacing") == NS_w12::LN_spacing;
        flag&=QName::tokenizer().insert(ns, "ind") == NS_w12::LN_ind;
        flag&=QName::tokenizer().insert(ns, "contextualSpacing") == NS_w12::LN_contextualSpacing;
        flag&=QName::tokenizer().insert(ns, "mirrorIndents") == NS_w12::LN_mirrorIndents;
        flag&=QName::tokenizer().insert(ns, "textboxTightWrap") == NS_w12::LN_textboxTightWrap;
        flag&=QName::tokenizer().insert(ns, "suppressOverlap") == NS_w12::LN_suppressOverlap;
        flag&=QName::tokenizer().insert(ns, "jc") == NS_w12::LN_jc;
        flag&=QName::tokenizer().insert(ns, "textDirection") == NS_w12::LN_textDirection;
        flag&=QName::tokenizer().insert(ns, "textAlignment") == NS_w12::LN_textAlignment;
        flag&=QName::tokenizer().insert(ns, "outlineLvl") == NS_w12::LN_outlineLvl;
        flag&=QName::tokenizer().insert(ns, "divId") == NS_w12::LN_divId;
        flag&=QName::tokenizer().insert(ns, "cnfStyle") == NS_w12::LN_cnfStyle;
        flag&=QName::tokenizer().insert(ns, "rPr") == NS_w12::LN_rPr;
        flag&=QName::tokenizer().insert(ns, "sectPr") == NS_w12::LN_sectPr;
        flag&=QName::tokenizer().insert(ns, "movie") == NS_w12::LN_movie;
        flag&=QName::tokenizer().insert(ns, "background") == NS_w12::LN_background;
        flag&=QName::tokenizer().insert(ns, "applet") == NS_w12::LN_applet;
        flag&=QName::tokenizer().insert(ns, "msAccessHTML") == NS_w12::LN_msAccessHTML;
        flag&=QName::tokenizer().insert(ns, "anchor") == NS_w12::LN_anchor;
        flag&=QName::tokenizer().insert(ns, "inline") == NS_w12::LN_inline;
        flag&=QName::tokenizer().insert(ns, "args") == NS_w12::LN_args;
        flag&=QName::tokenizer().insert(ns, "appletText") == NS_w12::LN_appletText;
        flag&=QName::tokenizer().insert(ns, "fldData") == NS_w12::LN_fldData;
        flag&=QName::tokenizer().insert(ns, "r") == NS_w12::LN_r;
        flag&=QName::tokenizer().insert(ns, "fldSimple") == NS_w12::LN_fldSimple;
        flag&=QName::tokenizer().insert(ns, "hyperlink") == NS_w12::LN_hyperlink;
        flag&=QName::tokenizer().insert(ns, "ffData") == NS_w12::LN_ffData;
        flag&=QName::tokenizer().insert(ns, "name") == NS_w12::LN_name;
        flag&=QName::tokenizer().insert(ns, "trackRevisions") == NS_w12::LN_trackRevisions;
        flag&=QName::tokenizer().insert(ns, "enabled") == NS_w12::LN_enabled;
        flag&=QName::tokenizer().insert(ns, "trackMoves") == NS_w12::LN_trackMoves;
        flag&=QName::tokenizer().insert(ns, "calcOnExit") == NS_w12::LN_calcOnExit;
        flag&=QName::tokenizer().insert(ns, "entryMacro") == NS_w12::LN_entryMacro;
        flag&=QName::tokenizer().insert(ns, "trackFormatting") == NS_w12::LN_trackFormatting;
        flag&=QName::tokenizer().insert(ns, "exitMacro") == NS_w12::LN_exitMacro;
        flag&=QName::tokenizer().insert(ns, "documentProtection") == NS_w12::LN_documentProtection;
        flag&=QName::tokenizer().insert(ns, "helpText") == NS_w12::LN_helpText;
        flag&=QName::tokenizer().insert(ns, "statusText") == NS_w12::LN_statusText;
        flag&=QName::tokenizer().insert(ns, "autoFormatOverride") == NS_w12::LN_autoFormatOverride;
        flag&=QName::tokenizer().insert(ns, "checkBox") == NS_w12::LN_checkBox;
        flag&=QName::tokenizer().insert(ns, "styleLockTheme") == NS_w12::LN_styleLockTheme;
        flag&=QName::tokenizer().insert(ns, "ddList") == NS_w12::LN_ddList;
        flag&=QName::tokenizer().insert(ns, "textInput") == NS_w12::LN_textInput;
        flag&=QName::tokenizer().insert(ns, "styleLockQFSet") == NS_w12::LN_styleLockQFSet;
        flag&=QName::tokenizer().insert(ns, "size") == NS_w12::LN_size;
        flag&=QName::tokenizer().insert(ns, "defaultTabStop") == NS_w12::LN_defaultTabStop;
        flag&=QName::tokenizer().insert(ns, "sizeAuto") == NS_w12::LN_sizeAuto;
        flag&=QName::tokenizer().insert(ns, "default") == NS_w12::LN_default;
        flag&=QName::tokenizer().insert(ns, "autoHyphenation") == NS_w12::LN_autoHyphenation;
        flag&=QName::tokenizer().insert(ns, "listEntry") == NS_w12::LN_listEntry;
        flag&=QName::tokenizer().insert(ns, "type") == NS_w12::LN_type;
        flag&=QName::tokenizer().insert(ns, "maxLengh") == NS_w12::LN_maxLengh;
        flag&=QName::tokenizer().insert(ns, "consecutiveHyphenLimit") == NS_w12::LN_consecutiveHyphenLimit;
        flag&=QName::tokenizer().insert(ns, "format") == NS_w12::LN_format;
        flag&=QName::tokenizer().insert(ns, "hyphenationZone") == NS_w12::LN_hyphenationZone;
        flag&=QName::tokenizer().insert(ns, "col") == NS_w12::LN_col;
        flag&=QName::tokenizer().insert(ns, "hdr") == NS_w12::LN_hdr;
        flag&=QName::tokenizer().insert(ns, "ftr") == NS_w12::LN_ftr;
        flag&=QName::tokenizer().insert(ns, "doNotHyphenateCaps") == NS_w12::LN_doNotHyphenateCaps;
        flag&=QName::tokenizer().insert(ns, "showEnvelope") == NS_w12::LN_showEnvelope;
        flag&=QName::tokenizer().insert(ns, "headerReference") == NS_w12::LN_headerReference;
        flag&=QName::tokenizer().insert(ns, "summaryLength") == NS_w12::LN_summaryLength;
        flag&=QName::tokenizer().insert(ns, "footerReference") == NS_w12::LN_footerReference;
        flag&=QName::tokenizer().insert(ns, "footnotePr") == NS_w12::LN_footnotePr;
        flag&=QName::tokenizer().insert(ns, "clickAndTypeStyle") == NS_w12::LN_clickAndTypeStyle;
        flag&=QName::tokenizer().insert(ns, "endnotePr") == NS_w12::LN_endnotePr;
        flag&=QName::tokenizer().insert(ns, "defaultTableStyle") == NS_w12::LN_defaultTableStyle;
        flag&=QName::tokenizer().insert(ns, "pgSz") == NS_w12::LN_pgSz;
        flag&=QName::tokenizer().insert(ns, "pgMar") == NS_w12::LN_pgMar;
        flag&=QName::tokenizer().insert(ns, "evenAndOddHeaders") == NS_w12::LN_evenAndOddHeaders;
        flag&=QName::tokenizer().insert(ns, "paperSrc") == NS_w12::LN_paperSrc;
        flag&=QName::tokenizer().insert(ns, "pgBorders") == NS_w12::LN_pgBorders;
        flag&=QName::tokenizer().insert(ns, "bookFoldRevPrinting") == NS_w12::LN_bookFoldRevPrinting;
        flag&=QName::tokenizer().insert(ns, "lnNumType") == NS_w12::LN_lnNumType;
        flag&=QName::tokenizer().insert(ns, "bookFoldPrinting") == NS_w12::LN_bookFoldPrinting;
        flag&=QName::tokenizer().insert(ns, "pgNumType") == NS_w12::LN_pgNumType;
        flag&=QName::tokenizer().insert(ns, "cols") == NS_w12::LN_cols;
        flag&=QName::tokenizer().insert(ns, "bookFoldPrintingSheets") == NS_w12::LN_bookFoldPrintingSheets;
        flag&=QName::tokenizer().insert(ns, "formProt") == NS_w12::LN_formProt;
        flag&=QName::tokenizer().insert(ns, "vAlign") == NS_w12::LN_vAlign;
        flag&=QName::tokenizer().insert(ns, "drawingGridHorizontalSpacing") == NS_w12::LN_drawingGridHorizontalSpacing;
        flag&=QName::tokenizer().insert(ns, "noEndnote") == NS_w12::LN_noEndnote;
        flag&=QName::tokenizer().insert(ns, "titlePg") == NS_w12::LN_titlePg;
        flag&=QName::tokenizer().insert(ns, "textFlow") == NS_w12::LN_textFlow;
        flag&=QName::tokenizer().insert(ns, "drawingGridVerticalSpacing") == NS_w12::LN_drawingGridVerticalSpacing;
        flag&=QName::tokenizer().insert(ns, "rtlGutter") == NS_w12::LN_rtlGutter;
        flag&=QName::tokenizer().insert(ns, "docGrid") == NS_w12::LN_docGrid;
        flag&=QName::tokenizer().insert(ns, "br") == NS_w12::LN_br;
        flag&=QName::tokenizer().insert(ns, "displayHorizontalDrawingGridEvery") == NS_w12::LN_displayHorizontalDrawingGridEvery;
        flag&=QName::tokenizer().insert(ns, "t") == NS_w12::LN_t;
        flag&=QName::tokenizer().insert(ns, "delText") == NS_w12::LN_delText;
        flag&=QName::tokenizer().insert(ns, "displayVerticalDrawingGridEvery") == NS_w12::LN_displayVerticalDrawingGridEvery;
        flag&=QName::tokenizer().insert(ns, "instrText") == NS_w12::LN_instrText;
        flag&=QName::tokenizer().insert(ns, "delInstrText") == NS_w12::LN_delInstrText;
        flag&=QName::tokenizer().insert(ns, "useMarginsForDrawingGridOrigin") == NS_w12::LN_useMarginsForDrawingGridOrigin;
        flag&=QName::tokenizer().insert(ns, "noBreakHyphen") == NS_w12::LN_noBreakHyphen;
        flag&=QName::tokenizer().insert(ns, "softHyphen") == NS_w12::LN_softHyphen;
        flag&=QName::tokenizer().insert(ns, "drawingGridHorizontalOrigin") == NS_w12::LN_drawingGridHorizontalOrigin;
        flag&=QName::tokenizer().insert(ns, "dayShort") == NS_w12::LN_dayShort;
        flag&=QName::tokenizer().insert(ns, "monthShort") == NS_w12::LN_monthShort;
        flag&=QName::tokenizer().insert(ns, "drawingGridVerticalOrigin") == NS_w12::LN_drawingGridVerticalOrigin;
        flag&=QName::tokenizer().insert(ns, "yearShort") == NS_w12::LN_yearShort;
        flag&=QName::tokenizer().insert(ns, "dayLong") == NS_w12::LN_dayLong;
        flag&=QName::tokenizer().insert(ns, "doNotShadeFormData") == NS_w12::LN_doNotShadeFormData;
        flag&=QName::tokenizer().insert(ns, "monthLong") == NS_w12::LN_monthLong;
        flag&=QName::tokenizer().insert(ns, "punctuationKerning") == NS_w12::LN_punctuationKerning;
        flag&=QName::tokenizer().insert(ns, "yearLong") == NS_w12::LN_yearLong;
        flag&=QName::tokenizer().insert(ns, "annotationRef") == NS_w12::LN_annotationRef;
        flag&=QName::tokenizer().insert(ns, "characterSpacingControl") == NS_w12::LN_characterSpacingControl;
        flag&=QName::tokenizer().insert(ns, "footnoteRef") == NS_w12::LN_footnoteRef;
        flag&=QName::tokenizer().insert(ns, "printTwoOnOne") == NS_w12::LN_printTwoOnOne;
        flag&=QName::tokenizer().insert(ns, "endnoteRef") == NS_w12::LN_endnoteRef;
        flag&=QName::tokenizer().insert(ns, "separator") == NS_w12::LN_separator;
        flag&=QName::tokenizer().insert(ns, "strictFirstAndLastChars") == NS_w12::LN_strictFirstAndLastChars;
        flag&=QName::tokenizer().insert(ns, "footnote") == NS_w12::LN_footnote;
        flag&=QName::tokenizer().insert(ns, "noLineBreaksAfter") == NS_w12::LN_noLineBreaksAfter;
        flag&=QName::tokenizer().insert(ns, "endnote") == NS_w12::LN_endnote;
        flag&=QName::tokenizer().insert(ns, "noLineBreaksBefore") == NS_w12::LN_noLineBreaksBefore;
        flag&=QName::tokenizer().insert(ns, "continuationSeparator") == NS_w12::LN_continuationSeparator;
        flag&=QName::tokenizer().insert(ns, "sym") == NS_w12::LN_sym;
        flag&=QName::tokenizer().insert(ns, "webPageEncoding") == NS_w12::LN_webPageEncoding;
        flag&=QName::tokenizer().insert(ns, "pgNum") == NS_w12::LN_pgNum;
        flag&=QName::tokenizer().insert(ns, "cr") == NS_w12::LN_cr;
        flag&=QName::tokenizer().insert(ns, "object") == NS_w12::LN_object;
        flag&=QName::tokenizer().insert(ns, "optimizeForBrowser") == NS_w12::LN_optimizeForBrowser;
        flag&=QName::tokenizer().insert(ns, "pict") == NS_w12::LN_pict;
        flag&=QName::tokenizer().insert(ns, "relyOnVML") == NS_w12::LN_relyOnVML;
        flag&=QName::tokenizer().insert(ns, "fldChar") == NS_w12::LN_fldChar;
        flag&=QName::tokenizer().insert(ns, "allowPNG") == NS_w12::LN_allowPNG;
        flag&=QName::tokenizer().insert(ns, "ruby") == NS_w12::LN_ruby;
        flag&=QName::tokenizer().insert(ns, "doNotRelyOnCSS") == NS_w12::LN_doNotRelyOnCSS;
        flag&=QName::tokenizer().insert(ns, "footnoteReference") == NS_w12::LN_footnoteReference;
        flag&=QName::tokenizer().insert(ns, "endnoteReference") == NS_w12::LN_endnoteReference;
        flag&=QName::tokenizer().insert(ns, "doNotSaveWebPagesAsSingleFile") == NS_w12::LN_doNotSaveWebPagesAsSingleFile;
        flag&=QName::tokenizer().insert(ns, "oArt") == NS_w12::LN_oArt;
        flag&=QName::tokenizer().insert(ns, "ptab") == NS_w12::LN_ptab;
        flag&=QName::tokenizer().insert(ns, "doNotOrganizeInFolder") == NS_w12::LN_doNotOrganizeInFolder;
        flag&=QName::tokenizer().insert(ns, "rStyle") == NS_w12::LN_rStyle;
        flag&=QName::tokenizer().insert(ns, "rFonts") == NS_w12::LN_rFonts;
        flag&=QName::tokenizer().insert(ns, "b") == NS_w12::LN_b;
        flag&=QName::tokenizer().insert(ns, "doNotUseLongFileNames") == NS_w12::LN_doNotUseLongFileNames;
        flag&=QName::tokenizer().insert(ns, "bCs") == NS_w12::LN_bCs;
        flag&=QName::tokenizer().insert(ns, "i") == NS_w12::LN_i;
        flag&=QName::tokenizer().insert(ns, "pixelsPerInch") == NS_w12::LN_pixelsPerInch;
        flag&=QName::tokenizer().insert(ns, "iCs") == NS_w12::LN_iCs;
        flag&=QName::tokenizer().insert(ns, "caps") == NS_w12::LN_caps;
        flag&=QName::tokenizer().insert(ns, "targetScreenSz") == NS_w12::LN_targetScreenSz;
        flag&=QName::tokenizer().insert(ns, "smallCaps") == NS_w12::LN_smallCaps;
        flag&=QName::tokenizer().insert(ns, "savePreviewPicture") == NS_w12::LN_savePreviewPicture;
        flag&=QName::tokenizer().insert(ns, "strike") == NS_w12::LN_strike;
        flag&=QName::tokenizer().insert(ns, "dstrike") == NS_w12::LN_dstrike;
        flag&=QName::tokenizer().insert(ns, "outline") == NS_w12::LN_outline;
        flag&=QName::tokenizer().insert(ns, "validateAgainstSchema") == NS_w12::LN_validateAgainstSchema;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_w12::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "emboss") == NS_w12::LN_emboss;
        flag&=QName::tokenizer().insert(ns, "saveInvalidXML") == NS_w12::LN_saveInvalidXML;
        flag&=QName::tokenizer().insert(ns, "imprint") == NS_w12::LN_imprint;
        flag&=QName::tokenizer().insert(ns, "noProof") == NS_w12::LN_noProof;
        flag&=QName::tokenizer().insert(ns, "ignoreMixedContent") == NS_w12::LN_ignoreMixedContent;
        flag&=QName::tokenizer().insert(ns, "vanish") == NS_w12::LN_vanish;
        flag&=QName::tokenizer().insert(ns, "webHidden") == NS_w12::LN_webHidden;
        flag&=QName::tokenizer().insert(ns, "color") == NS_w12::LN_color;
        flag&=QName::tokenizer().insert(ns, "alwaysShowPlaceholderText") == NS_w12::LN_alwaysShowPlaceholderText;
        flag&=QName::tokenizer().insert(ns, "w") == NS_w12::LN_w;
        flag&=QName::tokenizer().insert(ns, "kern") == NS_w12::LN_kern;
        flag&=QName::tokenizer().insert(ns, "position") == NS_w12::LN_position;
        flag&=QName::tokenizer().insert(ns, "doNotUnderlineInvalidXML") == NS_w12::LN_doNotUnderlineInvalidXML;
        flag&=QName::tokenizer().insert(ns, "sz") == NS_w12::LN_sz;
        flag&=QName::tokenizer().insert(ns, "szCs") == NS_w12::LN_szCs;
        flag&=QName::tokenizer().insert(ns, "highlight") == NS_w12::LN_highlight;
        flag&=QName::tokenizer().insert(ns, "removeWordSchemaOnSave") == NS_w12::LN_removeWordSchemaOnSave;
        flag&=QName::tokenizer().insert(ns, "u") == NS_w12::LN_u;
        flag&=QName::tokenizer().insert(ns, "effect") == NS_w12::LN_effect;
        flag&=QName::tokenizer().insert(ns, "useXSLTWhenSaving") == NS_w12::LN_useXSLTWhenSaving;
        flag&=QName::tokenizer().insert(ns, "bdr") == NS_w12::LN_bdr;
        flag&=QName::tokenizer().insert(ns, "fitText") == NS_w12::LN_fitText;
        flag&=QName::tokenizer().insert(ns, "saveThroughXSLT") == NS_w12::LN_saveThroughXSLT;
        flag&=QName::tokenizer().insert(ns, "vertAlign") == NS_w12::LN_vertAlign;
        flag&=QName::tokenizer().insert(ns, "showXMLTags") == NS_w12::LN_showXMLTags;
        flag&=QName::tokenizer().insert(ns, "rtl") == NS_w12::LN_rtl;
        flag&=QName::tokenizer().insert(ns, "cs") == NS_w12::LN_cs;
        flag&=QName::tokenizer().insert(ns, "em") == NS_w12::LN_em;
        flag&=QName::tokenizer().insert(ns, "alwaysMergeEmptyNamespace") == NS_w12::LN_alwaysMergeEmptyNamespace;
        flag&=QName::tokenizer().insert(ns, "hyphen") == NS_w12::LN_hyphen;
        flag&=QName::tokenizer().insert(ns, "lang") == NS_w12::LN_lang;
        flag&=QName::tokenizer().insert(ns, "doNotEmbedLinguisticData") == NS_w12::LN_doNotEmbedLinguisticData;
        flag&=QName::tokenizer().insert(ns, "asianLayout") == NS_w12::LN_asianLayout;
        flag&=QName::tokenizer().insert(ns, "specVanish") == NS_w12::LN_specVanish;
        flag&=QName::tokenizer().insert(ns, "hdrShapeDefaults") == NS_w12::LN_hdrShapeDefaults;
        flag&=QName::tokenizer().insert(ns, "oMath") == NS_w12::LN_oMath;
        flag&=QName::tokenizer().insert(ns, "compat") == NS_w12::LN_compat;
        flag&=QName::tokenizer().insert(ns, "lists") == NS_w12::LN_lists;
        flag&=QName::tokenizer().insert(ns, "styles") == NS_w12::LN_styles;
        flag&=QName::tokenizer().insert(ns, "docVars") == NS_w12::LN_docVars;
        flag&=QName::tokenizer().insert(ns, "fonts") == NS_w12::LN_fonts;
        flag&=QName::tokenizer().insert(ns, "rsids") == NS_w12::LN_rsids;
        flag&=QName::tokenizer().insert(ns, "rubyAlign") == NS_w12::LN_rubyAlign;
        flag&=QName::tokenizer().insert(ns, "hps") == NS_w12::LN_hps;
        flag&=QName::tokenizer().insert(ns, "uiCompat97To2003") == NS_w12::LN_uiCompat97To2003;
        flag&=QName::tokenizer().insert(ns, "hpsRaise") == NS_w12::LN_hpsRaise;
        flag&=QName::tokenizer().insert(ns, "binData") == NS_w12::LN_binData;
        flag&=QName::tokenizer().insert(ns, "hpsBaseText") == NS_w12::LN_hpsBaseText;
        flag&=QName::tokenizer().insert(ns, "sourceFileName") == NS_w12::LN_sourceFileName;
        flag&=QName::tokenizer().insert(ns, "lid") == NS_w12::LN_lid;
        flag&=QName::tokenizer().insert(ns, "marW") == NS_w12::LN_marW;
        flag&=QName::tokenizer().insert(ns, "rubyPr") == NS_w12::LN_rubyPr;
        flag&=QName::tokenizer().insert(ns, "marH") == NS_w12::LN_marH;
        flag&=QName::tokenizer().insert(ns, "rt") == NS_w12::LN_rt;
        flag&=QName::tokenizer().insert(ns, "scrollbar") == NS_w12::LN_scrollbar;
        flag&=QName::tokenizer().insert(ns, "rubyBase") == NS_w12::LN_rubyBase;
        flag&=QName::tokenizer().insert(ns, "dateFormat") == NS_w12::LN_dateFormat;
        flag&=QName::tokenizer().insert(ns, "noResizeAllowed") == NS_w12::LN_noResizeAllowed;
        flag&=QName::tokenizer().insert(ns, "listItem") == NS_w12::LN_listItem;
        flag&=QName::tokenizer().insert(ns, "linkedToFile") == NS_w12::LN_linkedToFile;
        flag&=QName::tokenizer().insert(ns, "docPartType") == NS_w12::LN_docPartType;
        flag&=QName::tokenizer().insert(ns, "noBorder") == NS_w12::LN_noBorder;
        flag&=QName::tokenizer().insert(ns, "flatBorders") == NS_w12::LN_flatBorders;
        flag&=QName::tokenizer().insert(ns, "docPartCategory") == NS_w12::LN_docPartCategory;
        flag&=QName::tokenizer().insert(ns, "docPartUnique") == NS_w12::LN_docPartUnique;
        flag&=QName::tokenizer().insert(ns, "framesetSplitbar") == NS_w12::LN_framesetSplitbar;
        flag&=QName::tokenizer().insert(ns, "frameLayout") == NS_w12::LN_frameLayout;
        flag&=QName::tokenizer().insert(ns, "pictureStorageType") == NS_w12::LN_pictureStorageType;
        flag&=QName::tokenizer().insert(ns, "frameset") == NS_w12::LN_frameset;
        flag&=QName::tokenizer().insert(ns, "docPart") == NS_w12::LN_docPart;
        flag&=QName::tokenizer().insert(ns, "frame") == NS_w12::LN_frame;
        flag&=QName::tokenizer().insert(ns, "start") == NS_w12::LN_start;
        flag&=QName::tokenizer().insert(ns, "alias") == NS_w12::LN_alias;
        flag&=QName::tokenizer().insert(ns, "nfc") == NS_w12::LN_nfc;
        flag&=QName::tokenizer().insert(ns, "invalidIfBlank") == NS_w12::LN_invalidIfBlank;
        flag&=QName::tokenizer().insert(ns, "lvlRestart") == NS_w12::LN_lvlRestart;
        flag&=QName::tokenizer().insert(ns, "lock") == NS_w12::LN_lock;
        flag&=QName::tokenizer().insert(ns, "isLgl") == NS_w12::LN_isLgl;
        flag&=QName::tokenizer().insert(ns, "placeholder") == NS_w12::LN_placeholder;
        flag&=QName::tokenizer().insert(ns, "suff") == NS_w12::LN_suff;
        flag&=QName::tokenizer().insert(ns, "showingPlcHdr") == NS_w12::LN_showingPlcHdr;
        flag&=QName::tokenizer().insert(ns, "lvlText") == NS_w12::LN_lvlText;
        flag&=QName::tokenizer().insert(ns, "edittingPlcHdr") == NS_w12::LN_edittingPlcHdr;
        flag&=QName::tokenizer().insert(ns, "lvlPicBulletId") == NS_w12::LN_lvlPicBulletId;
        flag&=QName::tokenizer().insert(ns, "dataBinding") == NS_w12::LN_dataBinding;
        flag&=QName::tokenizer().insert(ns, "legacy") == NS_w12::LN_legacy;
        flag&=QName::tokenizer().insert(ns, "lvlJc") == NS_w12::LN_lvlJc;
        flag&=QName::tokenizer().insert(ns, "comboBox") == NS_w12::LN_comboBox;
        flag&=QName::tokenizer().insert(ns, "date") == NS_w12::LN_date;
        flag&=QName::tokenizer().insert(ns, "lsid") == NS_w12::LN_lsid;
        flag&=QName::tokenizer().insert(ns, "plt") == NS_w12::LN_plt;
        flag&=QName::tokenizer().insert(ns, "docPartObj") == NS_w12::LN_docPartObj;
        flag&=QName::tokenizer().insert(ns, "tmpl") == NS_w12::LN_tmpl;
        flag&=QName::tokenizer().insert(ns, "docPartList") == NS_w12::LN_docPartList;
        flag&=QName::tokenizer().insert(ns, "styleLink") == NS_w12::LN_styleLink;
        flag&=QName::tokenizer().insert(ns, "dropDownList") == NS_w12::LN_dropDownList;
        flag&=QName::tokenizer().insert(ns, "listStyleLink") == NS_w12::LN_listStyleLink;
        flag&=QName::tokenizer().insert(ns, "lvl") == NS_w12::LN_lvl;
        flag&=QName::tokenizer().insert(ns, "picture") == NS_w12::LN_picture;
        flag&=QName::tokenizer().insert(ns, "richText") == NS_w12::LN_richText;
        flag&=QName::tokenizer().insert(ns, "startOverride") == NS_w12::LN_startOverride;
        flag&=QName::tokenizer().insert(ns, "text") == NS_w12::LN_text;
        flag&=QName::tokenizer().insert(ns, "p") == NS_w12::LN_p;
        flag&=QName::tokenizer().insert(ns, "ilst") == NS_w12::LN_ilst;
        flag&=QName::tokenizer().insert(ns, "tbl") == NS_w12::LN_tbl;
        flag&=QName::tokenizer().insert(ns, "tr") == NS_w12::LN_tr;
        flag&=QName::tokenizer().insert(ns, "lvlOverride") == NS_w12::LN_lvlOverride;
        flag&=QName::tokenizer().insert(ns, "tc") == NS_w12::LN_tc;
        flag&=QName::tokenizer().insert(ns, "sdtPr") == NS_w12::LN_sdtPr;
        flag&=QName::tokenizer().insert(ns, "listPicBullet") == NS_w12::LN_listPicBullet;
        flag&=QName::tokenizer().insert(ns, "listDef") == NS_w12::LN_listDef;
        flag&=QName::tokenizer().insert(ns, "sdtContent") == NS_w12::LN_sdtContent;
        flag&=QName::tokenizer().insert(ns, "list") == NS_w12::LN_list;
        flag&=QName::tokenizer().insert(ns, "pPr") == NS_w12::LN_pPr;
        flag&=QName::tokenizer().insert(ns, "sdt") == NS_w12::LN_sdt;
        flag&=QName::tokenizer().insert(ns, "ilfoMacAtCleanup") == NS_w12::LN_ilfoMacAtCleanup;
        flag&=QName::tokenizer().insert(ns, "subDoc") == NS_w12::LN_subDoc;
        flag&=QName::tokenizer().insert(ns, "gridCol") == NS_w12::LN_gridCol;
        flag&=QName::tokenizer().insert(ns, "aliases") == NS_w12::LN_aliases;
        flag&=QName::tokenizer().insert(ns, "insideH") == NS_w12::LN_insideH;
        flag&=QName::tokenizer().insert(ns, "sti") == NS_w12::LN_sti;
        flag&=QName::tokenizer().insert(ns, "insideV") == NS_w12::LN_insideV;
        flag&=QName::tokenizer().insert(ns, "tl2br") == NS_w12::LN_tl2br;
        flag&=QName::tokenizer().insert(ns, "basedOn") == NS_w12::LN_basedOn;
        flag&=QName::tokenizer().insert(ns, "tr2bl") == NS_w12::LN_tr2bl;
        flag&=QName::tokenizer().insert(ns, "tcW") == NS_w12::LN_tcW;
        flag&=QName::tokenizer().insert(ns, "next") == NS_w12::LN_next;
        flag&=QName::tokenizer().insert(ns, "link") == NS_w12::LN_link;
        flag&=QName::tokenizer().insert(ns, "gridSpan") == NS_w12::LN_gridSpan;
        flag&=QName::tokenizer().insert(ns, "priority") == NS_w12::LN_priority;
        flag&=QName::tokenizer().insert(ns, "hmerge") == NS_w12::LN_hmerge;
        flag&=QName::tokenizer().insert(ns, "autoRedefine") == NS_w12::LN_autoRedefine;
        flag&=QName::tokenizer().insert(ns, "vmerge") == NS_w12::LN_vmerge;
        flag&=QName::tokenizer().insert(ns, "hidden") == NS_w12::LN_hidden;
        flag&=QName::tokenizer().insert(ns, "tcBorders") == NS_w12::LN_tcBorders;
        flag&=QName::tokenizer().insert(ns, "noWrap") == NS_w12::LN_noWrap;
        flag&=QName::tokenizer().insert(ns, "semiHidden") == NS_w12::LN_semiHidden;
        flag&=QName::tokenizer().insert(ns, "tcMar") == NS_w12::LN_tcMar;
        flag&=QName::tokenizer().insert(ns, "locked") == NS_w12::LN_locked;
        flag&=QName::tokenizer().insert(ns, "tcFitText") == NS_w12::LN_tcFitText;
        flag&=QName::tokenizer().insert(ns, "unhideWhenUsed") == NS_w12::LN_unhideWhenUsed;
        flag&=QName::tokenizer().insert(ns, "hideMark") == NS_w12::LN_hideMark;
        flag&=QName::tokenizer().insert(ns, "tcPr") == NS_w12::LN_tcPr;
        flag&=QName::tokenizer().insert(ns, "qformat") == NS_w12::LN_qformat;
        flag&=QName::tokenizer().insert(ns, "personal") == NS_w12::LN_personal;
        flag&=QName::tokenizer().insert(ns, "gridBefore") == NS_w12::LN_gridBefore;
        flag&=QName::tokenizer().insert(ns, "gridAfter") == NS_w12::LN_gridAfter;
        flag&=QName::tokenizer().insert(ns, "personalCompose") == NS_w12::LN_personalCompose;
        flag&=QName::tokenizer().insert(ns, "wBefore") == NS_w12::LN_wBefore;
        flag&=QName::tokenizer().insert(ns, "personalReply") == NS_w12::LN_personalReply;
        flag&=QName::tokenizer().insert(ns, "wAfter") == NS_w12::LN_wAfter;
        flag&=QName::tokenizer().insert(ns, "tblStylePr") == NS_w12::LN_tblStylePr;
        flag&=QName::tokenizer().insert(ns, "cantSplit") == NS_w12::LN_cantSplit;
        flag&=QName::tokenizer().insert(ns, "trHeight") == NS_w12::LN_trHeight;
        flag&=QName::tokenizer().insert(ns, "lsdException") == NS_w12::LN_lsdException;
        flag&=QName::tokenizer().insert(ns, "tblHeader") == NS_w12::LN_tblHeader;
        flag&=QName::tokenizer().insert(ns, "versionOfBuiltInStylenames") == NS_w12::LN_versionOfBuiltInStylenames;
        flag&=QName::tokenizer().insert(ns, "tblCellSpacing") == NS_w12::LN_tblCellSpacing;
        flag&=QName::tokenizer().insert(ns, "latentStyles") == NS_w12::LN_latentStyles;
        flag&=QName::tokenizer().insert(ns, "tblPrEx") == NS_w12::LN_tblPrEx;
        flag&=QName::tokenizer().insert(ns, "style") == NS_w12::LN_style;
        flag&=QName::tokenizer().insert(ns, "trPr") == NS_w12::LN_trPr;
        flag&=QName::tokenizer().insert(ns, "altName") == NS_w12::LN_altName;
        flag&=QName::tokenizer().insert(ns, "tblStyle") == NS_w12::LN_tblStyle;
        flag&=QName::tokenizer().insert(ns, "panose1") == NS_w12::LN_panose1;
        flag&=QName::tokenizer().insert(ns, "tblpPr") == NS_w12::LN_tblpPr;
        flag&=QName::tokenizer().insert(ns, "charset") == NS_w12::LN_charset;
        flag&=QName::tokenizer().insert(ns, "family") == NS_w12::LN_family;
        flag&=QName::tokenizer().insert(ns, "tblOverlap") == NS_w12::LN_tblOverlap;
        flag&=QName::tokenizer().insert(ns, "tblRtl") == NS_w12::LN_tblRtl;
        flag&=QName::tokenizer().insert(ns, "notTrueType") == NS_w12::LN_notTrueType;
        flag&=QName::tokenizer().insert(ns, "pitch") == NS_w12::LN_pitch;
        flag&=QName::tokenizer().insert(ns, "bidiVisual") == NS_w12::LN_bidiVisual;
        flag&=QName::tokenizer().insert(ns, "sig") == NS_w12::LN_sig;
        flag&=QName::tokenizer().insert(ns, "tblStyleRowBandSize") == NS_w12::LN_tblStyleRowBandSize;
        flag&=QName::tokenizer().insert(ns, "defaultFonts") == NS_w12::LN_defaultFonts;
        flag&=QName::tokenizer().insert(ns, "font") == NS_w12::LN_font;
        flag&=QName::tokenizer().insert(ns, "tblStyleColBandSize") == NS_w12::LN_tblStyleColBandSize;
        flag&=QName::tokenizer().insert(ns, "blockQuote") == NS_w12::LN_blockQuote;
        flag&=QName::tokenizer().insert(ns, "tblW") == NS_w12::LN_tblW;
        flag&=QName::tokenizer().insert(ns, "bodyDiv") == NS_w12::LN_bodyDiv;
        flag&=QName::tokenizer().insert(ns, "tblInd") == NS_w12::LN_tblInd;
        flag&=QName::tokenizer().insert(ns, "marLeft") == NS_w12::LN_marLeft;
        flag&=QName::tokenizer().insert(ns, "tblBorders") == NS_w12::LN_tblBorders;
        flag&=QName::tokenizer().insert(ns, "marRight") == NS_w12::LN_marRight;
        flag&=QName::tokenizer().insert(ns, "tblLayout") == NS_w12::LN_tblLayout;
        flag&=QName::tokenizer().insert(ns, "marTop") == NS_w12::LN_marTop;
        flag&=QName::tokenizer().insert(ns, "tblCellMar") == NS_w12::LN_tblCellMar;
        flag&=QName::tokenizer().insert(ns, "marBottom") == NS_w12::LN_marBottom;
        flag&=QName::tokenizer().insert(ns, "tblLook") == NS_w12::LN_tblLook;
        flag&=QName::tokenizer().insert(ns, "divBdr") == NS_w12::LN_divBdr;
        flag&=QName::tokenizer().insert(ns, "tblPr") == NS_w12::LN_tblPr;
        flag&=QName::tokenizer().insert(ns, "tblGrid") == NS_w12::LN_tblGrid;
        flag&=QName::tokenizer().insert(ns, "divsChild") == NS_w12::LN_divsChild;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_w12::LN_pos;
        flag&=QName::tokenizer().insert(ns, "div") == NS_w12::LN_div;
        flag&=QName::tokenizer().insert(ns, "numFmt") == NS_w12::LN_numFmt;
        flag&=QName::tokenizer().insert(ns, "txbxContent") == NS_w12::LN_txbxContent;
        flag&=QName::tokenizer().insert(ns, "numStart") == NS_w12::LN_numStart;
        flag&=QName::tokenizer().insert(ns, "cfChunk") == NS_w12::LN_cfChunk;
        flag&=QName::tokenizer().insert(ns, "numRestart") == NS_w12::LN_numRestart;
        flag&=QName::tokenizer().insert(ns, "active") == NS_w12::LN_active;
        flag&=QName::tokenizer().insert(ns, "proofErr") == NS_w12::LN_proofErr;
        flag&=QName::tokenizer().insert(ns, "hash") == NS_w12::LN_hash;
        flag&=QName::tokenizer().insert(ns, "permStart") == NS_w12::LN_permStart;
        flag&=QName::tokenizer().insert(ns, "column") == NS_w12::LN_column;
        flag&=QName::tokenizer().insert(ns, "permEnd") == NS_w12::LN_permEnd;
        flag&=QName::tokenizer().insert(ns, "uniqueTag") == NS_w12::LN_uniqueTag;
        flag&=QName::tokenizer().insert(ns, "comments") == NS_w12::LN_comments;
        flag&=QName::tokenizer().insert(ns, "mappedName") == NS_w12::LN_mappedName;
        flag&=QName::tokenizer().insert(ns, "footnotes") == NS_w12::LN_footnotes;
        flag&=QName::tokenizer().insert(ns, "dynamicAddress") == NS_w12::LN_dynamicAddress;
        flag&=QName::tokenizer().insert(ns, "endnotes") == NS_w12::LN_endnotes;
        flag&=QName::tokenizer().insert(ns, "udl") == NS_w12::LN_udl;
        flag&=QName::tokenizer().insert(ns, "table") == NS_w12::LN_table;
        flag&=QName::tokenizer().insert(ns, "category") == NS_w12::LN_category;
        flag&=QName::tokenizer().insert(ns, "src") == NS_w12::LN_src;
        flag&=QName::tokenizer().insert(ns, "types") == NS_w12::LN_types;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_w12::LN_filter;
        flag&=QName::tokenizer().insert(ns, "description") == NS_w12::LN_description;
        flag&=QName::tokenizer().insert(ns, "sort") == NS_w12::LN_sort;
        flag&=QName::tokenizer().insert(ns, "guid") == NS_w12::LN_guid;
        flag&=QName::tokenizer().insert(ns, "fieldMapData") == NS_w12::LN_fieldMapData;
        flag&=QName::tokenizer().insert(ns, "docPartPr") == NS_w12::LN_docPartPr;
        flag&=QName::tokenizer().insert(ns, "colDelim") == NS_w12::LN_colDelim;
        flag&=QName::tokenizer().insert(ns, "docPartBody") == NS_w12::LN_docPartBody;
        flag&=QName::tokenizer().insert(ns, "jdsoType") == NS_w12::LN_jdsoType;
        flag&=QName::tokenizer().insert(ns, "docPr") == NS_w12::LN_docPr;
        flag&=QName::tokenizer().insert(ns, "fHdr") == NS_w12::LN_fHdr;
        flag&=QName::tokenizer().insert(ns, "docParts") == NS_w12::LN_docParts;
        flag&=QName::tokenizer().insert(ns, "recipientData") == NS_w12::LN_recipientData;
        flag&=QName::tokenizer().insert(ns, "ignoreSubtree") == NS_w12::LN_ignoreSubtree;
        flag&=QName::tokenizer().insert(ns, "mainDocumentType") == NS_w12::LN_mainDocumentType;
        flag&=QName::tokenizer().insert(ns, "ignoreElements") == NS_w12::LN_ignoreElements;
        flag&=QName::tokenizer().insert(ns, "body") == NS_w12::LN_body;
        flag&=QName::tokenizer().insert(ns, "linkToQuery") == NS_w12::LN_linkToQuery;
        flag&=QName::tokenizer().insert(ns, "wordDocument") == NS_w12::LN_wordDocument;
        flag&=QName::tokenizer().insert(ns, "dataType") == NS_w12::LN_dataType;
        flag&=QName::tokenizer().insert(ns, "wordTemplate") == NS_w12::LN_wordTemplate;
        flag&=QName::tokenizer().insert(ns, "defaultSQL") == NS_w12::LN_defaultSQL;
        flag&=QName::tokenizer().insert(ns, "glossaryDocument") == NS_w12::LN_glossaryDocument;
        flag&=QName::tokenizer().insert(ns, "connectString") == NS_w12::LN_connectString;
        flag&=QName::tokenizer().insert(ns, "query") == NS_w12::LN_query;
        flag&=QName::tokenizer().insert(ns, "dataSource") == NS_w12::LN_dataSource;
        flag&=QName::tokenizer().insert(ns, "headerSource") == NS_w12::LN_headerSource;
        flag&=QName::tokenizer().insert(ns, "doNotSuppressBlankLines") == NS_w12::LN_doNotSuppressBlankLines;
        flag&=QName::tokenizer().insert(ns, "destination") == NS_w12::LN_destination;
        flag&=QName::tokenizer().insert(ns, "addressFieldName") == NS_w12::LN_addressFieldName;
        flag&=QName::tokenizer().insert(ns, "mailSubject") == NS_w12::LN_mailSubject;
        flag&=QName::tokenizer().insert(ns, "mailAsAttachment") == NS_w12::LN_mailAsAttachment;
        flag&=QName::tokenizer().insert(ns, "viewMergedData") == NS_w12::LN_viewMergedData;
        flag&=QName::tokenizer().insert(ns, "activeRecord") == NS_w12::LN_activeRecord;
        flag&=QName::tokenizer().insert(ns, "checkErrors") == NS_w12::LN_checkErrors;
        flag&=QName::tokenizer().insert(ns, "odso") == NS_w12::LN_odso;
        flag&=QName::tokenizer().insert(ns, "origWordTableRules") == NS_w12::LN_origWordTableRules;
        flag&=QName::tokenizer().insert(ns, "wpJustification") == NS_w12::LN_wpJustification;
        flag&=QName::tokenizer().insert(ns, "noTabHangInd") == NS_w12::LN_noTabHangInd;
        flag&=QName::tokenizer().insert(ns, "noLeading") == NS_w12::LN_noLeading;
        flag&=QName::tokenizer().insert(ns, "spaceForUL") == NS_w12::LN_spaceForUL;
        flag&=QName::tokenizer().insert(ns, "noColumnBalance") == NS_w12::LN_noColumnBalance;
        flag&=QName::tokenizer().insert(ns, "balanceSingleByteDoubleByteWidth") == NS_w12::LN_balanceSingleByteDoubleByteWidth;
        flag&=QName::tokenizer().insert(ns, "transparentMetafiles") == NS_w12::LN_transparentMetafiles;
        flag&=QName::tokenizer().insert(ns, "noExtraLineSpacing") == NS_w12::LN_noExtraLineSpacing;
        flag&=QName::tokenizer().insert(ns, "doNotLeaveBackslashAlone") == NS_w12::LN_doNotLeaveBackslashAlone;
        flag&=QName::tokenizer().insert(ns, "ulTrailSpace") == NS_w12::LN_ulTrailSpace;
        flag&=QName::tokenizer().insert(ns, "doNotExpandShiftReturn") == NS_w12::LN_doNotExpandShiftReturn;
        flag&=QName::tokenizer().insert(ns, "spacingInWholePoints") == NS_w12::LN_spacingInWholePoints;
        flag&=QName::tokenizer().insert(ns, "lineWrapLikeWord6") == NS_w12::LN_lineWrapLikeWord6;
        flag&=QName::tokenizer().insert(ns, "printBodyTextBeforeHeader") == NS_w12::LN_printBodyTextBeforeHeader;
        flag&=QName::tokenizer().insert(ns, "printColBlack") == NS_w12::LN_printColBlack;
        flag&=QName::tokenizer().insert(ns, "wpSpaceWidth") == NS_w12::LN_wpSpaceWidth;
        flag&=QName::tokenizer().insert(ns, "showBreaksInFrames") == NS_w12::LN_showBreaksInFrames;
        flag&=QName::tokenizer().insert(ns, "subFontBySize") == NS_w12::LN_subFontBySize;
        flag&=QName::tokenizer().insert(ns, "suppressBottomSpacing") == NS_w12::LN_suppressBottomSpacing;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacing") == NS_w12::LN_suppressTopSpacing;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacingMac5") == NS_w12::LN_suppressTopSpacingMac5;
        flag&=QName::tokenizer().insert(ns, "suppressTopSpacingWP") == NS_w12::LN_suppressTopSpacingWP;
        flag&=QName::tokenizer().insert(ns, "suppressSpBfAfterPgBrk") == NS_w12::LN_suppressSpBfAfterPgBrk;
        flag&=QName::tokenizer().insert(ns, "swapBordersFacingPages") == NS_w12::LN_swapBordersFacingPages;
        flag&=QName::tokenizer().insert(ns, "convMailMergeEsc") == NS_w12::LN_convMailMergeEsc;
        CPPUNIT_ASSERT_MESSAGE("test for namespace w12 failed", flag);}
    void test_rtfs(){
        bool flag=true;
        static const char ns[] = "urn:rtf:schema";
        flag&=QName::tokenizer().insert(ns, "Toggle") == NS_rtfs::LN_Toggle;
        flag&=QName::tokenizer().insert(ns, "Value") == NS_rtfs::LN_Value;
        flag&=QName::tokenizer().insert(ns, "Flag") == NS_rtfs::LN_Flag;
        flag&=QName::tokenizer().insert(ns, "Destination") == NS_rtfs::LN_Destination;
        flag&=QName::tokenizer().insert(ns, "Symbol") == NS_rtfs::LN_Symbol;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rtfs failed", flag);}
    void test_x2(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2003/xml";
        flag&=QName::tokenizer().insert(ns, "MapInfo") == NS_x2::LN_MapInfo;
        flag&=QName::tokenizer().insert(ns, "Binding") == NS_x2::LN_Binding;
        flag&=QName::tokenizer().insert(ns, "MapID") == NS_x2::LN_MapID;
        flag&=QName::tokenizer().insert(ns, "Schema") == NS_x2::LN_Schema;
        flag&=QName::tokenizer().insert(ns, "Map") == NS_x2::LN_Map;
        flag&=QName::tokenizer().insert(ns, "ShowImportExportValidationErrors") == NS_x2::LN_ShowImportExportValidationErrors;
        flag&=QName::tokenizer().insert(ns, "Append") == NS_x2::LN_Append;
        flag&=QName::tokenizer().insert(ns, "Entry") == NS_x2::LN_Entry;
        flag&=QName::tokenizer().insert(ns, "Mapdata") == NS_x2::LN_Mapdata;
        flag&=QName::tokenizer().insert(ns, "Range") == NS_x2::LN_Range;
        flag&=QName::tokenizer().insert(ns, "HeaderRange") == NS_x2::LN_HeaderRange;
        flag&=QName::tokenizer().insert(ns, "XPath") == NS_x2::LN_XPath;
        flag&=QName::tokenizer().insert(ns, "Field") == NS_x2::LN_Field;
        flag&=QName::tokenizer().insert(ns, "ActiveRows") == NS_x2::LN_ActiveRows;
        flag&=QName::tokenizer().insert(ns, "NoInserts") == NS_x2::LN_NoInserts;
        flag&=QName::tokenizer().insert(ns, "ReadOnly") == NS_x2::LN_ReadOnly;
        flag&=QName::tokenizer().insert(ns, "InstanceShape") == NS_x2::LN_InstanceShape;
        flag&=QName::tokenizer().insert(ns, "ActiveRow") == NS_x2::LN_ActiveRow;
        flag&=QName::tokenizer().insert(ns, "XSDType") == NS_x2::LN_XSDType;
        flag&=QName::tokenizer().insert(ns, "Aggregate") == NS_x2::LN_Aggregate;
        CPPUNIT_ASSERT_MESSAGE("test for namespace x2 failed", flag);}
    void test_ip12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/infopath/2003/solutionDefinition";
        flag&=QName::tokenizer().insert(ns, "partFragment") == NS_ip12::LN_partFragment;
        flag&=QName::tokenizer().insert(ns, "xmlFileAdapter") == NS_ip12::LN_xmlFileAdapter;
        flag&=QName::tokenizer().insert(ns, "sharepointListAdapter") == NS_ip12::LN_sharepointListAdapter;
        flag&=QName::tokenizer().insert(ns, "field") == NS_ip12::LN_field;
        flag&=QName::tokenizer().insert(ns, "davAdapter") == NS_ip12::LN_davAdapter;
        flag&=QName::tokenizer().insert(ns, "folderURL") == NS_ip12::LN_folderURL;
        flag&=QName::tokenizer().insert(ns, "fileName") == NS_ip12::LN_fileName;
        flag&=QName::tokenizer().insert(ns, "emailAdapter") == NS_ip12::LN_emailAdapter;
        flag&=QName::tokenizer().insert(ns, "to") == NS_ip12::LN_to;
        flag&=QName::tokenizer().insert(ns, "cc") == NS_ip12::LN_cc;
        flag&=QName::tokenizer().insert(ns, "bcc") == NS_ip12::LN_bcc;
        flag&=QName::tokenizer().insert(ns, "subject") == NS_ip12::LN_subject;
        flag&=QName::tokenizer().insert(ns, "intro") == NS_ip12::LN_intro;
        flag&=QName::tokenizer().insert(ns, "attachmentFileName") == NS_ip12::LN_attachmentFileName;
        flag&=QName::tokenizer().insert(ns, "dataAdapters") == NS_ip12::LN_dataAdapters;
        flag&=QName::tokenizer().insert(ns, "documentSchemas") == NS_ip12::LN_documentSchemas;
        flag&=QName::tokenizer().insert(ns, "documentSchema") == NS_ip12::LN_documentSchema;
        flag&=QName::tokenizer().insert(ns, "customValidation") == NS_ip12::LN_customValidation;
        flag&=QName::tokenizer().insert(ns, "errorCondition") == NS_ip12::LN_errorCondition;
        flag&=QName::tokenizer().insert(ns, "errorMessage") == NS_ip12::LN_errorMessage;
        flag&=QName::tokenizer().insert(ns, "domEventHandlers") == NS_ip12::LN_domEventHandlers;
        flag&=QName::tokenizer().insert(ns, "domEventHandler") == NS_ip12::LN_domEventHandler;
        flag&=QName::tokenizer().insert(ns, "importParameters") == NS_ip12::LN_importParameters;
        flag&=QName::tokenizer().insert(ns, "importSource") == NS_ip12::LN_importSource;
        flag&=QName::tokenizer().insert(ns, "listProperties") == NS_ip12::LN_listProperties;
        flag&=QName::tokenizer().insert(ns, "fields") == NS_ip12::LN_fields;
        flag&=QName::tokenizer().insert(ns, "submit") == NS_ip12::LN_submit;
        flag&=QName::tokenizer().insert(ns, "submitAction") == NS_ip12::LN_submitAction;
        flag&=QName::tokenizer().insert(ns, "successMessage") == NS_ip12::LN_successMessage;
        flag&=QName::tokenizer().insert(ns, "useHttpHandler") == NS_ip12::LN_useHttpHandler;
        flag&=QName::tokenizer().insert(ns, "useScriptHandler") == NS_ip12::LN_useScriptHandler;
        flag&=QName::tokenizer().insert(ns, "useQueryAdapter") == NS_ip12::LN_useQueryAdapter;
        flag&=QName::tokenizer().insert(ns, "onLoad") == NS_ip12::LN_onLoad;
        flag&=QName::tokenizer().insert(ns, "roles") == NS_ip12::LN_roles;
        flag&=QName::tokenizer().insert(ns, "role") == NS_ip12::LN_role;
        flag&=QName::tokenizer().insert(ns, "membership") == NS_ip12::LN_membership;
        flag&=QName::tokenizer().insert(ns, "getUserNameFromData") == NS_ip12::LN_getUserNameFromData;
        flag&=QName::tokenizer().insert(ns, "userName") == NS_ip12::LN_userName;
        flag&=QName::tokenizer().insert(ns, "group") == NS_ip12::LN_group;
        flag&=QName::tokenizer().insert(ns, "hwsWorkflow") == NS_ip12::LN_hwsWorkflow;
        flag&=QName::tokenizer().insert(ns, "location") == NS_ip12::LN_location;
        flag&=QName::tokenizer().insert(ns, "allowedActions") == NS_ip12::LN_allowedActions;
        flag&=QName::tokenizer().insert(ns, "action") == NS_ip12::LN_action;
        flag&=QName::tokenizer().insert(ns, "allowedTasks") == NS_ip12::LN_allowedTasks;
        flag&=QName::tokenizer().insert(ns, "task") == NS_ip12::LN_task;
        flag&=QName::tokenizer().insert(ns, "fileNew") == NS_ip12::LN_fileNew;
        flag&=QName::tokenizer().insert(ns, "initialXmlDocument") == NS_ip12::LN_initialXmlDocument;
        flag&=QName::tokenizer().insert(ns, "customCategory") == NS_ip12::LN_customCategory;
        flag&=QName::tokenizer().insert(ns, "package") == NS_ip12::LN_package;
        flag&=QName::tokenizer().insert(ns, "files") == NS_ip12::LN_files;
        flag&=QName::tokenizer().insert(ns, "file") == NS_ip12::LN_file;
        flag&=QName::tokenizer().insert(ns, "fileProperties") == NS_ip12::LN_fileProperties;
        flag&=QName::tokenizer().insert(ns, "property") == NS_ip12::LN_property;
        flag&=QName::tokenizer().insert(ns, "permissions") == NS_ip12::LN_permissions;
        flag&=QName::tokenizer().insert(ns, "allowedControl") == NS_ip12::LN_allowedControl;
        flag&=QName::tokenizer().insert(ns, "externalViews") == NS_ip12::LN_externalViews;
        flag&=QName::tokenizer().insert(ns, "externalView") == NS_ip12::LN_externalView;
        flag&=QName::tokenizer().insert(ns, "attributeData") == NS_ip12::LN_attributeData;
        flag&=QName::tokenizer().insert(ns, "button") == NS_ip12::LN_button;
        flag&=QName::tokenizer().insert(ns, "chooseFragment") == NS_ip12::LN_chooseFragment;
        flag&=QName::tokenizer().insert(ns, "editWith") == NS_ip12::LN_editWith;
        flag&=QName::tokenizer().insert(ns, "unboundControls") == NS_ip12::LN_unboundControls;
        flag&=QName::tokenizer().insert(ns, "editing") == NS_ip12::LN_editing;
        flag&=QName::tokenizer().insert(ns, "masterDetail") == NS_ip12::LN_masterDetail;
        flag&=QName::tokenizer().insert(ns, "fragmentToInsert") == NS_ip12::LN_fragmentToInsert;
        flag&=QName::tokenizer().insert(ns, "mainpane") == NS_ip12::LN_mainpane;
        flag&=QName::tokenizer().insert(ns, "printSettings") == NS_ip12::LN_printSettings;
        flag&=QName::tokenizer().insert(ns, "header") == NS_ip12::LN_header;
        flag&=QName::tokenizer().insert(ns, "footer") == NS_ip12::LN_footer;
        flag&=QName::tokenizer().insert(ns, "toolbar") == NS_ip12::LN_toolbar;
        flag&=QName::tokenizer().insert(ns, "menu") == NS_ip12::LN_menu;
        flag&=QName::tokenizer().insert(ns, "menuArea") == NS_ip12::LN_menuArea;
        flag&=QName::tokenizer().insert(ns, "taskpane") == NS_ip12::LN_taskpane;
        flag&=QName::tokenizer().insert(ns, "views") == NS_ip12::LN_views;
        flag&=QName::tokenizer().insert(ns, "view") == NS_ip12::LN_view;
        flag&=QName::tokenizer().insert(ns, "xmlToEdit") == NS_ip12::LN_xmlToEdit;
        flag&=QName::tokenizer().insert(ns, "documentSignatures") == NS_ip12::LN_documentSignatures;
        flag&=QName::tokenizer().insert(ns, "signedDataBlock") == NS_ip12::LN_signedDataBlock;
        flag&=QName::tokenizer().insert(ns, "message") == NS_ip12::LN_message;
        flag&=QName::tokenizer().insert(ns, "documentVersionUpgrade") == NS_ip12::LN_documentVersionUpgrade;
        flag&=QName::tokenizer().insert(ns, "useTransform") == NS_ip12::LN_useTransform;
        flag&=QName::tokenizer().insert(ns, "extensions") == NS_ip12::LN_extensions;
        flag&=QName::tokenizer().insert(ns, "extension") == NS_ip12::LN_extension;
        flag&=QName::tokenizer().insert(ns, "ruleSetAction") == NS_ip12::LN_ruleSetAction;
        flag&=QName::tokenizer().insert(ns, "rule") == NS_ip12::LN_rule;
        flag&=QName::tokenizer().insert(ns, "exitRuleSet") == NS_ip12::LN_exitRuleSet;
        flag&=QName::tokenizer().insert(ns, "dialogBoxMessageAction") == NS_ip12::LN_dialogBoxMessageAction;
        flag&=QName::tokenizer().insert(ns, "dialogBoxExpressionAction") == NS_ip12::LN_dialogBoxExpressionAction;
        flag&=QName::tokenizer().insert(ns, "switchViewAction") == NS_ip12::LN_switchViewAction;
        flag&=QName::tokenizer().insert(ns, "assignmentAction") == NS_ip12::LN_assignmentAction;
        flag&=QName::tokenizer().insert(ns, "queryAction") == NS_ip12::LN_queryAction;
        flag&=QName::tokenizer().insert(ns, "openNewDocumentAction") == NS_ip12::LN_openNewDocumentAction;
        flag&=QName::tokenizer().insert(ns, "closeDocumentAction") == NS_ip12::LN_closeDocumentAction;
        flag&=QName::tokenizer().insert(ns, "ruleSet") == NS_ip12::LN_ruleSet;
        flag&=QName::tokenizer().insert(ns, "ruleSets") == NS_ip12::LN_ruleSets;
        flag&=QName::tokenizer().insert(ns, "calculations") == NS_ip12::LN_calculations;
        flag&=QName::tokenizer().insert(ns, "calculatedField") == NS_ip12::LN_calculatedField;
        flag&=QName::tokenizer().insert(ns, "xDocumentClass") == NS_ip12::LN_xDocumentClass;
        flag&=QName::tokenizer().insert(ns, "schemaErrorMessages") == NS_ip12::LN_schemaErrorMessages;
        flag&=QName::tokenizer().insert(ns, "override") == NS_ip12::LN_override;
        flag&=QName::tokenizer().insert(ns, "applicationParameters") == NS_ip12::LN_applicationParameters;
        flag&=QName::tokenizer().insert(ns, "solutionProperties") == NS_ip12::LN_solutionProperties;
        flag&=QName::tokenizer().insert(ns, "featureRestrictions") == NS_ip12::LN_featureRestrictions;
        flag&=QName::tokenizer().insert(ns, "save") == NS_ip12::LN_save;
        flag&=QName::tokenizer().insert(ns, "exportToWeb") == NS_ip12::LN_exportToWeb;
        flag&=QName::tokenizer().insert(ns, "exportToExcel") == NS_ip12::LN_exportToExcel;
        flag&=QName::tokenizer().insert(ns, "print") == NS_ip12::LN_print;
        flag&=QName::tokenizer().insert(ns, "sendMail") == NS_ip12::LN_sendMail;
        flag&=QName::tokenizer().insert(ns, "autoRecovery") == NS_ip12::LN_autoRecovery;
        flag&=QName::tokenizer().insert(ns, "query") == NS_ip12::LN_query;
        flag&=QName::tokenizer().insert(ns, "scripts") == NS_ip12::LN_scripts;
        flag&=QName::tokenizer().insert(ns, "script") == NS_ip12::LN_script;
        flag&=QName::tokenizer().insert(ns, "dataObjects") == NS_ip12::LN_dataObjects;
        flag&=QName::tokenizer().insert(ns, "dataObject") == NS_ip12::LN_dataObject;
        flag&=QName::tokenizer().insert(ns, "adoAdapter") == NS_ip12::LN_adoAdapter;
        flag&=QName::tokenizer().insert(ns, "webServiceAdapter") == NS_ip12::LN_webServiceAdapter;
        flag&=QName::tokenizer().insert(ns, "hwsAdapter") == NS_ip12::LN_hwsAdapter;
        flag&=QName::tokenizer().insert(ns, "operation") == NS_ip12::LN_operation;
        flag&=QName::tokenizer().insert(ns, "hwsOperation") == NS_ip12::LN_hwsOperation;
        flag&=QName::tokenizer().insert(ns, "input") == NS_ip12::LN_input;
        CPPUNIT_ASSERT_MESSAGE("test for namespace ip12 failed", flag);}
    void test_rel12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/package/2005/06/relationships";
        flag&=QName::tokenizer().insert(ns, "Relationships") == NS_rel12::LN_Relationships;
        flag&=QName::tokenizer().insert(ns, "Relationship") == NS_rel12::LN_Relationship;
        CPPUNIT_ASSERT_MESSAGE("test for namespace rel12 failed", flag);}
    void test_NONE(){
        bool flag=true;
        static const char ns[] = "";
        flag&=QName::tokenizer().insert(ns, "datatypeLibrary") == NS_NONE::LN_datatypeLibrary;
        flag&=QName::tokenizer().insert(ns, "name") == NS_NONE::LN_name;
        flag&=QName::tokenizer().insert(ns, "combine") == NS_NONE::LN_combine;
        flag&=QName::tokenizer().insert(ns, "type") == NS_NONE::LN_type;
        flag&=QName::tokenizer().insert(ns, "ns") == NS_NONE::LN_ns;
        flag&=QName::tokenizer().insert(ns, "href") == NS_NONE::LN_href;
        CPPUNIT_ASSERT_MESSAGE("test for namespace NONE failed", flag);}
    void test_config(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:config:1.0";
        flag&=QName::tokenizer().insert(ns, "config-item-set") == NS_config::LN_config_item_set;
        flag&=QName::tokenizer().insert(ns, "name") == NS_config::LN_name;
        flag&=QName::tokenizer().insert(ns, "config-item") == NS_config::LN_config_item;
        flag&=QName::tokenizer().insert(ns, "type") == NS_config::LN_type;
        flag&=QName::tokenizer().insert(ns, "config-item-map-indexed") == NS_config::LN_config_item_map_indexed;
        flag&=QName::tokenizer().insert(ns, "config-item-map-entry") == NS_config::LN_config_item_map_entry;
        flag&=QName::tokenizer().insert(ns, "config-item-map-named") == NS_config::LN_config_item_map_named;
        CPPUNIT_ASSERT_MESSAGE("test for namespace config failed", flag);}
    void test_text(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:text:1.0";
        flag&=QName::tokenizer().insert(ns, "dde-connection-decl") == NS_text::LN_dde_connection_decl;
        flag&=QName::tokenizer().insert(ns, "first-row-start-column") == NS_text::LN_first_row_start_column;
        flag&=QName::tokenizer().insert(ns, "first-row-end-column") == NS_text::LN_first_row_end_column;
        flag&=QName::tokenizer().insert(ns, "last-row-start-column") == NS_text::LN_last_row_start_column;
        flag&=QName::tokenizer().insert(ns, "last-row-end-column") == NS_text::LN_last_row_end_column;
        flag&=QName::tokenizer().insert(ns, "global") == NS_text::LN_global;
        flag&=QName::tokenizer().insert(ns, "object-index-source") == NS_text::LN_object_index_source;
        flag&=QName::tokenizer().insert(ns, "use-spreadsheet-objects") == NS_text::LN_use_spreadsheet_objects;
        flag&=QName::tokenizer().insert(ns, "use-math-objects") == NS_text::LN_use_math_objects;
        flag&=QName::tokenizer().insert(ns, "use-draw-objects") == NS_text::LN_use_draw_objects;
        flag&=QName::tokenizer().insert(ns, "use-chart-objects") == NS_text::LN_use_chart_objects;
        flag&=QName::tokenizer().insert(ns, "use-other-objects") == NS_text::LN_use_other_objects;
        flag&=QName::tokenizer().insert(ns, "object-index-entry-template") == NS_text::LN_object_index_entry_template;
        flag&=QName::tokenizer().insert(ns, "user-index") == NS_text::LN_user_index;
        flag&=QName::tokenizer().insert(ns, "user-index-source") == NS_text::LN_user_index_source;
        flag&=QName::tokenizer().insert(ns, "use-graphics") == NS_text::LN_use_graphics;
        flag&=QName::tokenizer().insert(ns, "use-tables") == NS_text::LN_use_tables;
        flag&=QName::tokenizer().insert(ns, "use-floating-frames") == NS_text::LN_use_floating_frames;
        flag&=QName::tokenizer().insert(ns, "use-objects") == NS_text::LN_use_objects;
        flag&=QName::tokenizer().insert(ns, "copy-outline-levels") == NS_text::LN_copy_outline_levels;
        flag&=QName::tokenizer().insert(ns, "user-index-entry-template") == NS_text::LN_user_index_entry_template;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index") == NS_text::LN_alphabetical_index;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-source") == NS_text::LN_alphabetical_index_source;
        flag&=QName::tokenizer().insert(ns, "ignore-case") == NS_text::LN_ignore_case;
        flag&=QName::tokenizer().insert(ns, "main-entry-style-name") == NS_text::LN_main_entry_style_name;
        flag&=QName::tokenizer().insert(ns, "alphabetical-separators") == NS_text::LN_alphabetical_separators;
        flag&=QName::tokenizer().insert(ns, "combine-entries") == NS_text::LN_combine_entries;
        flag&=QName::tokenizer().insert(ns, "combine-entries-with-dash") == NS_text::LN_combine_entries_with_dash;
        flag&=QName::tokenizer().insert(ns, "combine-entries-with-pp") == NS_text::LN_combine_entries_with_pp;
        flag&=QName::tokenizer().insert(ns, "use-keys-as-entries") == NS_text::LN_use_keys_as_entries;
        flag&=QName::tokenizer().insert(ns, "capitalize-entries") == NS_text::LN_capitalize_entries;
        flag&=QName::tokenizer().insert(ns, "comma-separated") == NS_text::LN_comma_separated;
        flag&=QName::tokenizer().insert(ns, "sort-algorithm") == NS_text::LN_sort_algorithm;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-auto-mark-file") == NS_text::LN_alphabetical_index_auto_mark_file;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-entry-template") == NS_text::LN_alphabetical_index_entry_template;
        flag&=QName::tokenizer().insert(ns, "bibliography") == NS_text::LN_bibliography;
        flag&=QName::tokenizer().insert(ns, "bibliography-source") == NS_text::LN_bibliography_source;
        flag&=QName::tokenizer().insert(ns, "bibliography-entry-template") == NS_text::LN_bibliography_entry_template;
        flag&=QName::tokenizer().insert(ns, "index-source-styles") == NS_text::LN_index_source_styles;
        flag&=QName::tokenizer().insert(ns, "h") == NS_text::LN_h;
        flag&=QName::tokenizer().insert(ns, "index-source-style") == NS_text::LN_index_source_style;
        flag&=QName::tokenizer().insert(ns, "outline-level") == NS_text::LN_outline_level;
        flag&=QName::tokenizer().insert(ns, "index-title-template") == NS_text::LN_index_title_template;
        flag&=QName::tokenizer().insert(ns, "restart-numbering") == NS_text::LN_restart_numbering;
        flag&=QName::tokenizer().insert(ns, "start-value") == NS_text::LN_start_value;
        flag&=QName::tokenizer().insert(ns, "index-entry-chapter") == NS_text::LN_index_entry_chapter;
        flag&=QName::tokenizer().insert(ns, "is-list-header") == NS_text::LN_is_list_header;
        flag&=QName::tokenizer().insert(ns, "index-entry-text") == NS_text::LN_index_entry_text;
        flag&=QName::tokenizer().insert(ns, "number") == NS_text::LN_number;
        flag&=QName::tokenizer().insert(ns, "p") == NS_text::LN_p;
        flag&=QName::tokenizer().insert(ns, "index-entry-page-number") == NS_text::LN_index_entry_page_number;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_text::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "class-names") == NS_text::LN_class_names;
        flag&=QName::tokenizer().insert(ns, "index-entry-span") == NS_text::LN_index_entry_span;
        flag&=QName::tokenizer().insert(ns, "cond-style-name") == NS_text::LN_cond_style_name;
        flag&=QName::tokenizer().insert(ns, "index-entry-bibliography") == NS_text::LN_index_entry_bibliography;
        flag&=QName::tokenizer().insert(ns, "page-sequence") == NS_text::LN_page_sequence;
        flag&=QName::tokenizer().insert(ns, "page") == NS_text::LN_page;
        flag&=QName::tokenizer().insert(ns, "bibliography-data-field") == NS_text::LN_bibliography_data_field;
        flag&=QName::tokenizer().insert(ns, "master-page-name") == NS_text::LN_master_page_name;
        flag&=QName::tokenizer().insert(ns, "index-entry-tab-stop") == NS_text::LN_index_entry_tab_stop;
        flag&=QName::tokenizer().insert(ns, "list") == NS_text::LN_list;
        flag&=QName::tokenizer().insert(ns, "continue-numbering") == NS_text::LN_continue_numbering;
        flag&=QName::tokenizer().insert(ns, "list-item") == NS_text::LN_list_item;
        flag&=QName::tokenizer().insert(ns, "list-header") == NS_text::LN_list_header;
        flag&=QName::tokenizer().insert(ns, "index-entry-link-start") == NS_text::LN_index_entry_link_start;
        flag&=QName::tokenizer().insert(ns, "numbered-paragraph") == NS_text::LN_numbered_paragraph;
        flag&=QName::tokenizer().insert(ns, "index-entry-link-end") == NS_text::LN_index_entry_link_end;
        flag&=QName::tokenizer().insert(ns, "level") == NS_text::LN_level;
        flag&=QName::tokenizer().insert(ns, "section") == NS_text::LN_section;
        flag&=QName::tokenizer().insert(ns, "name") == NS_text::LN_name;
        flag&=QName::tokenizer().insert(ns, "protected") == NS_text::LN_protected;
        flag&=QName::tokenizer().insert(ns, "protection-key") == NS_text::LN_protection_key;
        flag&=QName::tokenizer().insert(ns, "display") == NS_text::LN_display;
        flag&=QName::tokenizer().insert(ns, "condition") == NS_text::LN_condition;
        flag&=QName::tokenizer().insert(ns, "section-source") == NS_text::LN_section_source;
        flag&=QName::tokenizer().insert(ns, "section-name") == NS_text::LN_section_name;
        flag&=QName::tokenizer().insert(ns, "filter-name") == NS_text::LN_filter_name;
        flag&=QName::tokenizer().insert(ns, "linenumbering-configuration") == NS_text::LN_linenumbering_configuration;
        flag&=QName::tokenizer().insert(ns, "tracked-changes") == NS_text::LN_tracked_changes;
        flag&=QName::tokenizer().insert(ns, "number-lines") == NS_text::LN_number_lines;
        flag&=QName::tokenizer().insert(ns, "track-changes") == NS_text::LN_track_changes;
        flag&=QName::tokenizer().insert(ns, "increment") == NS_text::LN_increment;
        flag&=QName::tokenizer().insert(ns, "changed-region") == NS_text::LN_changed_region;
        flag&=QName::tokenizer().insert(ns, "number-position") == NS_text::LN_number_position;
        flag&=QName::tokenizer().insert(ns, "id") == NS_text::LN_id;
        flag&=QName::tokenizer().insert(ns, "offset") == NS_text::LN_offset;
        flag&=QName::tokenizer().insert(ns, "insertion") == NS_text::LN_insertion;
        flag&=QName::tokenizer().insert(ns, "count-empty-lines") == NS_text::LN_count_empty_lines;
        flag&=QName::tokenizer().insert(ns, "deletion") == NS_text::LN_deletion;
        flag&=QName::tokenizer().insert(ns, "format-change") == NS_text::LN_format_change;
        flag&=QName::tokenizer().insert(ns, "count-in-text-boxes") == NS_text::LN_count_in_text_boxes;
        flag&=QName::tokenizer().insert(ns, "change") == NS_text::LN_change;
        flag&=QName::tokenizer().insert(ns, "restart-on-page") == NS_text::LN_restart_on_page;
        flag&=QName::tokenizer().insert(ns, "change-start") == NS_text::LN_change_start;
        flag&=QName::tokenizer().insert(ns, "change-end") == NS_text::LN_change_end;
        flag&=QName::tokenizer().insert(ns, "linenumbering-separator") == NS_text::LN_linenumbering_separator;
        flag&=QName::tokenizer().insert(ns, "change-id") == NS_text::LN_change_id;
        flag&=QName::tokenizer().insert(ns, "notes-configuration") == NS_text::LN_notes_configuration;
        flag&=QName::tokenizer().insert(ns, "variable-decls") == NS_text::LN_variable_decls;
        flag&=QName::tokenizer().insert(ns, "citation-style-name") == NS_text::LN_citation_style_name;
        flag&=QName::tokenizer().insert(ns, "sequence-decls") == NS_text::LN_sequence_decls;
        flag&=QName::tokenizer().insert(ns, "user-field-decls") == NS_text::LN_user_field_decls;
        flag&=QName::tokenizer().insert(ns, "citation-body-style-name") == NS_text::LN_citation_body_style_name;
        flag&=QName::tokenizer().insert(ns, "dde-connection-decls") == NS_text::LN_dde_connection_decls;
        flag&=QName::tokenizer().insert(ns, "default-style-name") == NS_text::LN_default_style_name;
        flag&=QName::tokenizer().insert(ns, "s") == NS_text::LN_s;
        flag&=QName::tokenizer().insert(ns, "c") == NS_text::LN_c;
        flag&=QName::tokenizer().insert(ns, "start-numbering-at") == NS_text::LN_start_numbering_at;
        flag&=QName::tokenizer().insert(ns, "tab") == NS_text::LN_tab;
        flag&=QName::tokenizer().insert(ns, "tab-ref") == NS_text::LN_tab_ref;
        flag&=QName::tokenizer().insert(ns, "footnotes-position") == NS_text::LN_footnotes_position;
        flag&=QName::tokenizer().insert(ns, "line-break") == NS_text::LN_line_break;
        flag&=QName::tokenizer().insert(ns, "span") == NS_text::LN_span;
        flag&=QName::tokenizer().insert(ns, "a") == NS_text::LN_a;
        flag&=QName::tokenizer().insert(ns, "note-continuation-notice-forward") == NS_text::LN_note_continuation_notice_forward;
        flag&=QName::tokenizer().insert(ns, "visited-style-name") == NS_text::LN_visited_style_name;
        flag&=QName::tokenizer().insert(ns, "note-continuation-notice-backward") == NS_text::LN_note_continuation_notice_backward;
        flag&=QName::tokenizer().insert(ns, "bookmark") == NS_text::LN_bookmark;
        flag&=QName::tokenizer().insert(ns, "bookmark-start") == NS_text::LN_bookmark_start;
        flag&=QName::tokenizer().insert(ns, "bibliography-configuration") == NS_text::LN_bibliography_configuration;
        flag&=QName::tokenizer().insert(ns, "bookmark-end") == NS_text::LN_bookmark_end;
        flag&=QName::tokenizer().insert(ns, "prefix") == NS_text::LN_prefix;
        flag&=QName::tokenizer().insert(ns, "suffix") == NS_text::LN_suffix;
        flag&=QName::tokenizer().insert(ns, "reference-mark") == NS_text::LN_reference_mark;
        flag&=QName::tokenizer().insert(ns, "numbered-entries") == NS_text::LN_numbered_entries;
        flag&=QName::tokenizer().insert(ns, "reference-mark-start") == NS_text::LN_reference_mark_start;
        flag&=QName::tokenizer().insert(ns, "sort-by-position") == NS_text::LN_sort_by_position;
        flag&=QName::tokenizer().insert(ns, "reference-mark-end") == NS_text::LN_reference_mark_end;
        flag&=QName::tokenizer().insert(ns, "sort-key") == NS_text::LN_sort_key;
        flag&=QName::tokenizer().insert(ns, "note") == NS_text::LN_note;
        flag&=QName::tokenizer().insert(ns, "key") == NS_text::LN_key;
        flag&=QName::tokenizer().insert(ns, "note-citation") == NS_text::LN_note_citation;
        flag&=QName::tokenizer().insert(ns, "sort-ascending") == NS_text::LN_sort_ascending;
        flag&=QName::tokenizer().insert(ns, "label") == NS_text::LN_label;
        flag&=QName::tokenizer().insert(ns, "list-style") == NS_text::LN_list_style;
        flag&=QName::tokenizer().insert(ns, "note-body") == NS_text::LN_note_body;
        flag&=QName::tokenizer().insert(ns, "note-class") == NS_text::LN_note_class;
        flag&=QName::tokenizer().insert(ns, "consecutive-numbering") == NS_text::LN_consecutive_numbering;
        flag&=QName::tokenizer().insert(ns, "ruby") == NS_text::LN_ruby;
        flag&=QName::tokenizer().insert(ns, "ruby-base") == NS_text::LN_ruby_base;
        flag&=QName::tokenizer().insert(ns, "list-level-style-number") == NS_text::LN_list_level_style_number;
        flag&=QName::tokenizer().insert(ns, "ruby-text") == NS_text::LN_ruby_text;
        flag&=QName::tokenizer().insert(ns, "date") == NS_text::LN_date;
        flag&=QName::tokenizer().insert(ns, "display-levels") == NS_text::LN_display_levels;
        flag&=QName::tokenizer().insert(ns, "date-value") == NS_text::LN_date_value;
        flag&=QName::tokenizer().insert(ns, "list-level-style-bullet") == NS_text::LN_list_level_style_bullet;
        flag&=QName::tokenizer().insert(ns, "date-adjust") == NS_text::LN_date_adjust;
        flag&=QName::tokenizer().insert(ns, "time") == NS_text::LN_time;
        flag&=QName::tokenizer().insert(ns, "bullet-char") == NS_text::LN_bullet_char;
        flag&=QName::tokenizer().insert(ns, "time-value") == NS_text::LN_time_value;
        flag&=QName::tokenizer().insert(ns, "bullet-relative-size") == NS_text::LN_bullet_relative_size;
        flag&=QName::tokenizer().insert(ns, "time-adjust") == NS_text::LN_time_adjust;
        flag&=QName::tokenizer().insert(ns, "page-number") == NS_text::LN_page_number;
        flag&=QName::tokenizer().insert(ns, "list-level-style-image") == NS_text::LN_list_level_style_image;
        flag&=QName::tokenizer().insert(ns, "page-adjust") == NS_text::LN_page_adjust;
        flag&=QName::tokenizer().insert(ns, "outline-style") == NS_text::LN_outline_style;
        flag&=QName::tokenizer().insert(ns, "select-page") == NS_text::LN_select_page;
        flag&=QName::tokenizer().insert(ns, "outline-level-style") == NS_text::LN_outline_level_style;
        flag&=QName::tokenizer().insert(ns, "page-continuation") == NS_text::LN_page_continuation;
        flag&=QName::tokenizer().insert(ns, "string-value") == NS_text::LN_string_value;
        flag&=QName::tokenizer().insert(ns, "sender-firstname") == NS_text::LN_sender_firstname;
        flag&=QName::tokenizer().insert(ns, "sender-lastname") == NS_text::LN_sender_lastname;
        flag&=QName::tokenizer().insert(ns, "sender-initials") == NS_text::LN_sender_initials;
        flag&=QName::tokenizer().insert(ns, "sender-title") == NS_text::LN_sender_title;
        flag&=QName::tokenizer().insert(ns, "sender-position") == NS_text::LN_sender_position;
        flag&=QName::tokenizer().insert(ns, "sender-email") == NS_text::LN_sender_email;
        flag&=QName::tokenizer().insert(ns, "sender-phone-private") == NS_text::LN_sender_phone_private;
        flag&=QName::tokenizer().insert(ns, "sender-fax") == NS_text::LN_sender_fax;
        flag&=QName::tokenizer().insert(ns, "sender-company") == NS_text::LN_sender_company;
        flag&=QName::tokenizer().insert(ns, "sender-phone-work") == NS_text::LN_sender_phone_work;
        flag&=QName::tokenizer().insert(ns, "sender-street") == NS_text::LN_sender_street;
        flag&=QName::tokenizer().insert(ns, "sender-city") == NS_text::LN_sender_city;
        flag&=QName::tokenizer().insert(ns, "sender-postal-code") == NS_text::LN_sender_postal_code;
        flag&=QName::tokenizer().insert(ns, "sender-country") == NS_text::LN_sender_country;
        flag&=QName::tokenizer().insert(ns, "sender-state-or-province") == NS_text::LN_sender_state_or_province;
        flag&=QName::tokenizer().insert(ns, "author-name") == NS_text::LN_author_name;
        flag&=QName::tokenizer().insert(ns, "author-initials") == NS_text::LN_author_initials;
        flag&=QName::tokenizer().insert(ns, "chapter") == NS_text::LN_chapter;
        flag&=QName::tokenizer().insert(ns, "file-name") == NS_text::LN_file_name;
        flag&=QName::tokenizer().insert(ns, "template-name") == NS_text::LN_template_name;
        flag&=QName::tokenizer().insert(ns, "sheet-name") == NS_text::LN_sheet_name;
        flag&=QName::tokenizer().insert(ns, "variable-decl") == NS_text::LN_variable_decl;
        flag&=QName::tokenizer().insert(ns, "variable-set") == NS_text::LN_variable_set;
        flag&=QName::tokenizer().insert(ns, "variable-get") == NS_text::LN_variable_get;
        flag&=QName::tokenizer().insert(ns, "variable-input") == NS_text::LN_variable_input;
        flag&=QName::tokenizer().insert(ns, "user-field-decl") == NS_text::LN_user_field_decl;
        flag&=QName::tokenizer().insert(ns, "user-field-get") == NS_text::LN_user_field_get;
        flag&=QName::tokenizer().insert(ns, "user-field-input") == NS_text::LN_user_field_input;
        flag&=QName::tokenizer().insert(ns, "sequence-decl") == NS_text::LN_sequence_decl;
        flag&=QName::tokenizer().insert(ns, "display-outline-level") == NS_text::LN_display_outline_level;
        flag&=QName::tokenizer().insert(ns, "separation-character") == NS_text::LN_separation_character;
        flag&=QName::tokenizer().insert(ns, "sequence") == NS_text::LN_sequence;
        flag&=QName::tokenizer().insert(ns, "ref-name") == NS_text::LN_ref_name;
        flag&=QName::tokenizer().insert(ns, "expression") == NS_text::LN_expression;
        flag&=QName::tokenizer().insert(ns, "text-input") == NS_text::LN_text_input;
        flag&=QName::tokenizer().insert(ns, "initial-creator") == NS_text::LN_initial_creator;
        flag&=QName::tokenizer().insert(ns, "creation-date") == NS_text::LN_creation_date;
        flag&=QName::tokenizer().insert(ns, "creation-time") == NS_text::LN_creation_time;
        flag&=QName::tokenizer().insert(ns, "description") == NS_text::LN_description;
        flag&=QName::tokenizer().insert(ns, "user-defined") == NS_text::LN_user_defined;
        flag&=QName::tokenizer().insert(ns, "print-time") == NS_text::LN_print_time;
        flag&=QName::tokenizer().insert(ns, "print-date") == NS_text::LN_print_date;
        flag&=QName::tokenizer().insert(ns, "printed-by") == NS_text::LN_printed_by;
        flag&=QName::tokenizer().insert(ns, "title") == NS_text::LN_title;
        flag&=QName::tokenizer().insert(ns, "subject") == NS_text::LN_subject;
        flag&=QName::tokenizer().insert(ns, "keywords") == NS_text::LN_keywords;
        flag&=QName::tokenizer().insert(ns, "editing-cycles") == NS_text::LN_editing_cycles;
        flag&=QName::tokenizer().insert(ns, "editing-duration") == NS_text::LN_editing_duration;
        flag&=QName::tokenizer().insert(ns, "duration") == NS_text::LN_duration;
        flag&=QName::tokenizer().insert(ns, "modification-time") == NS_text::LN_modification_time;
        flag&=QName::tokenizer().insert(ns, "modification-date") == NS_text::LN_modification_date;
        flag&=QName::tokenizer().insert(ns, "creator") == NS_text::LN_creator;
        flag&=QName::tokenizer().insert(ns, "database-name") == NS_text::LN_database_name;
        flag&=QName::tokenizer().insert(ns, "table-name") == NS_text::LN_table_name;
        flag&=QName::tokenizer().insert(ns, "table-type") == NS_text::LN_table_type;
        flag&=QName::tokenizer().insert(ns, "database-display") == NS_text::LN_database_display;
        flag&=QName::tokenizer().insert(ns, "column-name") == NS_text::LN_column_name;
        flag&=QName::tokenizer().insert(ns, "database-next") == NS_text::LN_database_next;
        flag&=QName::tokenizer().insert(ns, "database-row-select") == NS_text::LN_database_row_select;
        flag&=QName::tokenizer().insert(ns, "row-number") == NS_text::LN_row_number;
        flag&=QName::tokenizer().insert(ns, "database-row-number") == NS_text::LN_database_row_number;
        flag&=QName::tokenizer().insert(ns, "value") == NS_text::LN_value;
        flag&=QName::tokenizer().insert(ns, "page-variable-set") == NS_text::LN_page_variable_set;
        flag&=QName::tokenizer().insert(ns, "active") == NS_text::LN_active;
        flag&=QName::tokenizer().insert(ns, "page-variable-get") == NS_text::LN_page_variable_get;
        flag&=QName::tokenizer().insert(ns, "placeholder") == NS_text::LN_placeholder;
        flag&=QName::tokenizer().insert(ns, "placeholder-type") == NS_text::LN_placeholder_type;
        flag&=QName::tokenizer().insert(ns, "conditional-text") == NS_text::LN_conditional_text;
        flag&=QName::tokenizer().insert(ns, "string-value-if-true") == NS_text::LN_string_value_if_true;
        flag&=QName::tokenizer().insert(ns, "string-value-if-false") == NS_text::LN_string_value_if_false;
        flag&=QName::tokenizer().insert(ns, "current-value") == NS_text::LN_current_value;
        flag&=QName::tokenizer().insert(ns, "hidden-text") == NS_text::LN_hidden_text;
        flag&=QName::tokenizer().insert(ns, "is-hidden") == NS_text::LN_is_hidden;
        flag&=QName::tokenizer().insert(ns, "note-ref") == NS_text::LN_note_ref;
        flag&=QName::tokenizer().insert(ns, "sequence-ref") == NS_text::LN_sequence_ref;
        flag&=QName::tokenizer().insert(ns, "reference-format") == NS_text::LN_reference_format;
        flag&=QName::tokenizer().insert(ns, "script") == NS_text::LN_script;
        flag&=QName::tokenizer().insert(ns, "execute-macro") == NS_text::LN_execute_macro;
        flag&=QName::tokenizer().insert(ns, "hidden-paragraph") == NS_text::LN_hidden_paragraph;
        flag&=QName::tokenizer().insert(ns, "dde-connection") == NS_text::LN_dde_connection;
        flag&=QName::tokenizer().insert(ns, "connection-name") == NS_text::LN_connection_name;
        flag&=QName::tokenizer().insert(ns, "measure") == NS_text::LN_measure;
        flag&=QName::tokenizer().insert(ns, "kind") == NS_text::LN_kind;
        flag&=QName::tokenizer().insert(ns, "table-formula") == NS_text::LN_table_formula;
        flag&=QName::tokenizer().insert(ns, "fixed") == NS_text::LN_fixed;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_text::LN_formula;
        flag&=QName::tokenizer().insert(ns, "toc-mark-start") == NS_text::LN_toc_mark_start;
        flag&=QName::tokenizer().insert(ns, "toc-mark-end") == NS_text::LN_toc_mark_end;
        flag&=QName::tokenizer().insert(ns, "toc-mark") == NS_text::LN_toc_mark;
        flag&=QName::tokenizer().insert(ns, "user-index-mark-start") == NS_text::LN_user_index_mark_start;
        flag&=QName::tokenizer().insert(ns, "line-number") == NS_text::LN_line_number;
        flag&=QName::tokenizer().insert(ns, "user-index-mark-end") == NS_text::LN_user_index_mark_end;
        flag&=QName::tokenizer().insert(ns, "user-index-mark") == NS_text::LN_user_index_mark;
        flag&=QName::tokenizer().insert(ns, "index-name") == NS_text::LN_index_name;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-mark-start") == NS_text::LN_alphabetical_index_mark_start;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-mark-end") == NS_text::LN_alphabetical_index_mark_end;
        flag&=QName::tokenizer().insert(ns, "alphabetical-index-mark") == NS_text::LN_alphabetical_index_mark;
        flag&=QName::tokenizer().insert(ns, "key1") == NS_text::LN_key1;
        flag&=QName::tokenizer().insert(ns, "key2") == NS_text::LN_key2;
        flag&=QName::tokenizer().insert(ns, "string-value-phonetic") == NS_text::LN_string_value_phonetic;
        flag&=QName::tokenizer().insert(ns, "key1-phonetic") == NS_text::LN_key1_phonetic;
        flag&=QName::tokenizer().insert(ns, "key2-phonetic") == NS_text::LN_key2_phonetic;
        flag&=QName::tokenizer().insert(ns, "main-entry") == NS_text::LN_main_entry;
        flag&=QName::tokenizer().insert(ns, "bibliography-mark") == NS_text::LN_bibliography_mark;
        flag&=QName::tokenizer().insert(ns, "bibliography-type") == NS_text::LN_bibliography_type;
        flag&=QName::tokenizer().insert(ns, "index-body") == NS_text::LN_index_body;
        flag&=QName::tokenizer().insert(ns, "index-title") == NS_text::LN_index_title;
        flag&=QName::tokenizer().insert(ns, "table-of-content") == NS_text::LN_table_of_content;
        flag&=QName::tokenizer().insert(ns, "table-of-content-source") == NS_text::LN_table_of_content_source;
        flag&=QName::tokenizer().insert(ns, "use-outline-level") == NS_text::LN_use_outline_level;
        flag&=QName::tokenizer().insert(ns, "use-index-marks") == NS_text::LN_use_index_marks;
        flag&=QName::tokenizer().insert(ns, "dont-balance-text-columns") == NS_text::LN_dont_balance_text_columns;
        flag&=QName::tokenizer().insert(ns, "use-index-source-styles") == NS_text::LN_use_index_source_styles;
        flag&=QName::tokenizer().insert(ns, "index-scope") == NS_text::LN_index_scope;
        flag&=QName::tokenizer().insert(ns, "relative-tab-stop-position") == NS_text::LN_relative_tab_stop_position;
        flag&=QName::tokenizer().insert(ns, "table-of-content-entry-template") == NS_text::LN_table_of_content_entry_template;
        flag&=QName::tokenizer().insert(ns, "illustration-index") == NS_text::LN_illustration_index;
        flag&=QName::tokenizer().insert(ns, "illustration-index-source") == NS_text::LN_illustration_index_source;
        flag&=QName::tokenizer().insert(ns, "use-caption") == NS_text::LN_use_caption;
        flag&=QName::tokenizer().insert(ns, "caption-sequence-name") == NS_text::LN_caption_sequence_name;
        flag&=QName::tokenizer().insert(ns, "caption-sequence-format") == NS_text::LN_caption_sequence_format;
        flag&=QName::tokenizer().insert(ns, "illustration-index-entry-template") == NS_text::LN_illustration_index_entry_template;
        flag&=QName::tokenizer().insert(ns, "table-index") == NS_text::LN_table_index;
        flag&=QName::tokenizer().insert(ns, "table-index-source") == NS_text::LN_table_index_source;
        flag&=QName::tokenizer().insert(ns, "table-index-entry-template") == NS_text::LN_table_index_entry_template;
        flag&=QName::tokenizer().insert(ns, "object-index") == NS_text::LN_object_index;
        flag&=QName::tokenizer().insert(ns, "space-before") == NS_text::LN_space_before;
        flag&=QName::tokenizer().insert(ns, "min-label-width") == NS_text::LN_min_label_width;
        flag&=QName::tokenizer().insert(ns, "min-label-distance") == NS_text::LN_min_label_distance;
        flag&=QName::tokenizer().insert(ns, "animation") == NS_text::LN_animation;
        flag&=QName::tokenizer().insert(ns, "animation-direction") == NS_text::LN_animation_direction;
        flag&=QName::tokenizer().insert(ns, "animation-start-inside") == NS_text::LN_animation_start_inside;
        flag&=QName::tokenizer().insert(ns, "animation-stop-inside") == NS_text::LN_animation_stop_inside;
        flag&=QName::tokenizer().insert(ns, "animation-repeat") == NS_text::LN_animation_repeat;
        flag&=QName::tokenizer().insert(ns, "animation-delay") == NS_text::LN_animation_delay;
        flag&=QName::tokenizer().insert(ns, "animation-steps") == NS_text::LN_animation_steps;
        flag&=QName::tokenizer().insert(ns, "anchor-type") == NS_text::LN_anchor_type;
        flag&=QName::tokenizer().insert(ns, "anchor-page-number") == NS_text::LN_anchor_page_number;
        CPPUNIT_ASSERT_MESSAGE("test for namespace text failed", flag);}
    void test_tbl12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/table";
        flag&=QName::tokenizer().insert(ns, "table") == NS_tbl12::LN_table;
        flag&=QName::tokenizer().insert(ns, "autofilter") == NS_tbl12::LN_autofilter;
        flag&=QName::tokenizer().insert(ns, "sortState") == NS_tbl12::LN_sortState;
        flag&=QName::tokenizer().insert(ns, "cols") == NS_tbl12::LN_cols;
        flag&=QName::tokenizer().insert(ns, "tableStyle") == NS_tbl12::LN_tableStyle;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_tbl12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "col") == NS_tbl12::LN_col;
        flag&=QName::tokenizer().insert(ns, "ccFmla") == NS_tbl12::LN_ccFmla;
        flag&=QName::tokenizer().insert(ns, "trFmla") == NS_tbl12::LN_trFmla;
        flag&=QName::tokenizer().insert(ns, "xmlCPr") == NS_tbl12::LN_xmlCPr;
        CPPUNIT_ASSERT_MESSAGE("test for namespace tbl12 failed", flag);}
    void test_pr(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:presentation:1.0";
        flag&=QName::tokenizer().insert(ns, "transition-type") == NS_pr::LN_transition_type;
        flag&=QName::tokenizer().insert(ns, "transition-style") == NS_pr::LN_transition_style;
        flag&=QName::tokenizer().insert(ns, "transition-speed") == NS_pr::LN_transition_speed;
        flag&=QName::tokenizer().insert(ns, "duration") == NS_pr::LN_duration;
        flag&=QName::tokenizer().insert(ns, "visibility") == NS_pr::LN_visibility;
        flag&=QName::tokenizer().insert(ns, "background-objects-visible") == NS_pr::LN_background_objects_visible;
        flag&=QName::tokenizer().insert(ns, "background-visible") == NS_pr::LN_background_visible;
        flag&=QName::tokenizer().insert(ns, "display-header") == NS_pr::LN_display_header;
        flag&=QName::tokenizer().insert(ns, "display-footer") == NS_pr::LN_display_footer;
        flag&=QName::tokenizer().insert(ns, "display-page-number") == NS_pr::LN_display_page_number;
        flag&=QName::tokenizer().insert(ns, "display-date-time") == NS_pr::LN_display_date_time;
        flag&=QName::tokenizer().insert(ns, "class") == NS_pr::LN_class;
        flag&=QName::tokenizer().insert(ns, "placeholder") == NS_pr::LN_placeholder;
        flag&=QName::tokenizer().insert(ns, "user-transformed") == NS_pr::LN_user_transformed;
        flag&=QName::tokenizer().insert(ns, "animations") == NS_pr::LN_animations;
        flag&=QName::tokenizer().insert(ns, "sound") == NS_pr::LN_sound;
        flag&=QName::tokenizer().insert(ns, "play-full") == NS_pr::LN_play_full;
        flag&=QName::tokenizer().insert(ns, "show-shape") == NS_pr::LN_show_shape;
        flag&=QName::tokenizer().insert(ns, "notes") == NS_pr::LN_notes;
        flag&=QName::tokenizer().insert(ns, "object") == NS_pr::LN_object;
        flag&=QName::tokenizer().insert(ns, "presentation-page-layout-name") == NS_pr::LN_presentation_page_layout_name;
        flag&=QName::tokenizer().insert(ns, "use-header-name") == NS_pr::LN_use_header_name;
        flag&=QName::tokenizer().insert(ns, "effect") == NS_pr::LN_effect;
        flag&=QName::tokenizer().insert(ns, "direction") == NS_pr::LN_direction;
        flag&=QName::tokenizer().insert(ns, "speed") == NS_pr::LN_speed;
        flag&=QName::tokenizer().insert(ns, "use-footer-name") == NS_pr::LN_use_footer_name;
        flag&=QName::tokenizer().insert(ns, "delay") == NS_pr::LN_delay;
        flag&=QName::tokenizer().insert(ns, "start-scale") == NS_pr::LN_start_scale;
        flag&=QName::tokenizer().insert(ns, "use-date-time-name") == NS_pr::LN_use_date_time_name;
        flag&=QName::tokenizer().insert(ns, "path-id") == NS_pr::LN_path_id;
        flag&=QName::tokenizer().insert(ns, "show-text") == NS_pr::LN_show_text;
        flag&=QName::tokenizer().insert(ns, "hide-shape") == NS_pr::LN_hide_shape;
        flag&=QName::tokenizer().insert(ns, "hide-text") == NS_pr::LN_hide_text;
        flag&=QName::tokenizer().insert(ns, "dim") == NS_pr::LN_dim;
        flag&=QName::tokenizer().insert(ns, "play") == NS_pr::LN_play;
        flag&=QName::tokenizer().insert(ns, "animation-group") == NS_pr::LN_animation_group;
        flag&=QName::tokenizer().insert(ns, "node-type") == NS_pr::LN_node_type;
        flag&=QName::tokenizer().insert(ns, "preset-id") == NS_pr::LN_preset_id;
        flag&=QName::tokenizer().insert(ns, "preset-sub-type") == NS_pr::LN_preset_sub_type;
        flag&=QName::tokenizer().insert(ns, "preset-class") == NS_pr::LN_preset_class;
        flag&=QName::tokenizer().insert(ns, "master-element") == NS_pr::LN_master_element;
        flag&=QName::tokenizer().insert(ns, "group-id") == NS_pr::LN_group_id;
        flag&=QName::tokenizer().insert(ns, "event-listener") == NS_pr::LN_event_listener;
        flag&=QName::tokenizer().insert(ns, "action") == NS_pr::LN_action;
        flag&=QName::tokenizer().insert(ns, "verb") == NS_pr::LN_verb;
        flag&=QName::tokenizer().insert(ns, "header") == NS_pr::LN_header;
        flag&=QName::tokenizer().insert(ns, "footer") == NS_pr::LN_footer;
        flag&=QName::tokenizer().insert(ns, "date-time") == NS_pr::LN_date_time;
        flag&=QName::tokenizer().insert(ns, "header-decl") == NS_pr::LN_header_decl;
        flag&=QName::tokenizer().insert(ns, "name") == NS_pr::LN_name;
        flag&=QName::tokenizer().insert(ns, "footer-decl") == NS_pr::LN_footer_decl;
        flag&=QName::tokenizer().insert(ns, "date-time-decl") == NS_pr::LN_date_time_decl;
        flag&=QName::tokenizer().insert(ns, "source") == NS_pr::LN_source;
        flag&=QName::tokenizer().insert(ns, "settings") == NS_pr::LN_settings;
        flag&=QName::tokenizer().insert(ns, "start-page") == NS_pr::LN_start_page;
        flag&=QName::tokenizer().insert(ns, "show") == NS_pr::LN_show;
        flag&=QName::tokenizer().insert(ns, "full-screen") == NS_pr::LN_full_screen;
        flag&=QName::tokenizer().insert(ns, "endless") == NS_pr::LN_endless;
        flag&=QName::tokenizer().insert(ns, "pause") == NS_pr::LN_pause;
        flag&=QName::tokenizer().insert(ns, "show-logo") == NS_pr::LN_show_logo;
        flag&=QName::tokenizer().insert(ns, "force-manual") == NS_pr::LN_force_manual;
        flag&=QName::tokenizer().insert(ns, "mouse-visible") == NS_pr::LN_mouse_visible;
        flag&=QName::tokenizer().insert(ns, "mouse-as-pen") == NS_pr::LN_mouse_as_pen;
        flag&=QName::tokenizer().insert(ns, "start-with-navigator") == NS_pr::LN_start_with_navigator;
        flag&=QName::tokenizer().insert(ns, "transition-on-click") == NS_pr::LN_transition_on_click;
        flag&=QName::tokenizer().insert(ns, "stay-on-top") == NS_pr::LN_stay_on_top;
        flag&=QName::tokenizer().insert(ns, "pages") == NS_pr::LN_pages;
        flag&=QName::tokenizer().insert(ns, "style-name") == NS_pr::LN_style_name;
        flag&=QName::tokenizer().insert(ns, "class-names") == NS_pr::LN_class_names;
        CPPUNIT_ASSERT_MESSAGE("test for namespace pr failed", flag);}
    void test_udcxf11(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/data/udc/xmlfile";
        flag&=QName::tokenizer().insert(ns, "File") == NS_udcxf11::LN_File;
        CPPUNIT_ASSERT_MESSAGE("test for namespace udcxf11 failed", flag);}
    void test_office(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:office:1.0";
        flag&=QName::tokenizer().insert(ns, "annotation") == NS_office::LN_annotation;
        flag&=QName::tokenizer().insert(ns, "display") == NS_office::LN_display;
        flag&=QName::tokenizer().insert(ns, "change-info") == NS_office::LN_change_info;
        flag&=QName::tokenizer().insert(ns, "event-listeners") == NS_office::LN_event_listeners;
        flag&=QName::tokenizer().insert(ns, "dde-application") == NS_office::LN_dde_application;
        flag&=QName::tokenizer().insert(ns, "dde-topic") == NS_office::LN_dde_topic;
        flag&=QName::tokenizer().insert(ns, "dde-item") == NS_office::LN_dde_item;
        flag&=QName::tokenizer().insert(ns, "automatic-update") == NS_office::LN_automatic_update;
        flag&=QName::tokenizer().insert(ns, "dde-source") == NS_office::LN_dde_source;
        flag&=QName::tokenizer().insert(ns, "conversion-mode") == NS_office::LN_conversion_mode;
        flag&=QName::tokenizer().insert(ns, "process-content") == NS_office::LN_process_content;
        flag&=QName::tokenizer().insert(ns, "document") == NS_office::LN_document;
        flag&=QName::tokenizer().insert(ns, "document-content") == NS_office::LN_document_content;
        flag&=QName::tokenizer().insert(ns, "document-styles") == NS_office::LN_document_styles;
        flag&=QName::tokenizer().insert(ns, "document-meta") == NS_office::LN_document_meta;
        flag&=QName::tokenizer().insert(ns, "document-settings") == NS_office::LN_document_settings;
        flag&=QName::tokenizer().insert(ns, "version") == NS_office::LN_version;
        flag&=QName::tokenizer().insert(ns, "mimetype") == NS_office::LN_mimetype;
        flag&=QName::tokenizer().insert(ns, "meta") == NS_office::LN_meta;
        flag&=QName::tokenizer().insert(ns, "body") == NS_office::LN_body;
        flag&=QName::tokenizer().insert(ns, "text") == NS_office::LN_text;
        flag&=QName::tokenizer().insert(ns, "drawing") == NS_office::LN_drawing;
        flag&=QName::tokenizer().insert(ns, "presentation") == NS_office::LN_presentation;
        flag&=QName::tokenizer().insert(ns, "spreadsheet") == NS_office::LN_spreadsheet;
        flag&=QName::tokenizer().insert(ns, "chart") == NS_office::LN_chart;
        flag&=QName::tokenizer().insert(ns, "image") == NS_office::LN_image;
        flag&=QName::tokenizer().insert(ns, "settings") == NS_office::LN_settings;
        flag&=QName::tokenizer().insert(ns, "scripts") == NS_office::LN_scripts;
        flag&=QName::tokenizer().insert(ns, "script") == NS_office::LN_script;
        flag&=QName::tokenizer().insert(ns, "font-face-decls") == NS_office::LN_font_face_decls;
        flag&=QName::tokenizer().insert(ns, "styles") == NS_office::LN_styles;
        flag&=QName::tokenizer().insert(ns, "automatic-styles") == NS_office::LN_automatic_styles;
        flag&=QName::tokenizer().insert(ns, "master-styles") == NS_office::LN_master_styles;
        flag&=QName::tokenizer().insert(ns, "target-frame-name") == NS_office::LN_target_frame_name;
        flag&=QName::tokenizer().insert(ns, "name") == NS_office::LN_name;
        flag&=QName::tokenizer().insert(ns, "value") == NS_office::LN_value;
        flag&=QName::tokenizer().insert(ns, "date-value") == NS_office::LN_date_value;
        flag&=QName::tokenizer().insert(ns, "time-value") == NS_office::LN_time_value;
        flag&=QName::tokenizer().insert(ns, "boolean-value") == NS_office::LN_boolean_value;
        flag&=QName::tokenizer().insert(ns, "string-value") == NS_office::LN_string_value;
        flag&=QName::tokenizer().insert(ns, "value-type") == NS_office::LN_value_type;
        flag&=QName::tokenizer().insert(ns, "currency") == NS_office::LN_currency;
        flag&=QName::tokenizer().insert(ns, "binary-data") == NS_office::LN_binary_data;
        flag&=QName::tokenizer().insert(ns, "forms") == NS_office::LN_forms;
        flag&=QName::tokenizer().insert(ns, "target-frame") == NS_office::LN_target_frame;
        flag&=QName::tokenizer().insert(ns, "server-map") == NS_office::LN_server_map;
        CPPUNIT_ASSERT_MESSAGE("test for namespace office failed", flag);}
    void test_md12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/metadata";
        flag&=QName::tokenizer().insert(ns, "Metadata") == NS_md12::LN_Metadata;
        flag&=QName::tokenizer().insert(ns, "metadataTypes") == NS_md12::LN_metadataTypes;
        flag&=QName::tokenizer().insert(ns, "cellMetadata") == NS_md12::LN_cellMetadata;
        flag&=QName::tokenizer().insert(ns, "valueMetadata") == NS_md12::LN_valueMetadata;
        flag&=QName::tokenizer().insert(ns, "mdxMetadata") == NS_md12::LN_mdxMetadata;
        flag&=QName::tokenizer().insert(ns, "strMetadata") == NS_md12::LN_strMetadata;
        flag&=QName::tokenizer().insert(ns, "futureMetadata") == NS_md12::LN_futureMetadata;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_md12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "metadataType") == NS_md12::LN_metadataType;
        flag&=QName::tokenizer().insert(ns, "b") == NS_md12::LN_b;
        flag&=QName::tokenizer().insert(ns, "r") == NS_md12::LN_r;
        flag&=QName::tokenizer().insert(ns, "t") == NS_md12::LN_t;
        flag&=QName::tokenizer().insert(ns, "s") == NS_md12::LN_s;
        flag&=QName::tokenizer().insert(ns, "p") == NS_md12::LN_p;
        flag&=QName::tokenizer().insert(ns, "k") == NS_md12::LN_k;
        flag&=QName::tokenizer().insert(ns, "i") == NS_md12::LN_i;
        flag&=QName::tokenizer().insert(ns, "ib") == NS_md12::LN_ib;
        CPPUNIT_ASSERT_MESSAGE("test for namespace md12 failed", flag);}
    void test_pp12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/powerpoint/2005/8/pml";
        flag&=QName::tokenizer().insert(ns, "snd") == NS_pp12::LN_snd;
        flag&=QName::tokenizer().insert(ns, "stSnd") == NS_pp12::LN_stSnd;
        flag&=QName::tokenizer().insert(ns, "endSnd") == NS_pp12::LN_endSnd;
        flag&=QName::tokenizer().insert(ns, "blinds") == NS_pp12::LN_blinds;
        flag&=QName::tokenizer().insert(ns, "checker") == NS_pp12::LN_checker;
        flag&=QName::tokenizer().insert(ns, "circle") == NS_pp12::LN_circle;
        flag&=QName::tokenizer().insert(ns, "dissolve") == NS_pp12::LN_dissolve;
        flag&=QName::tokenizer().insert(ns, "comb") == NS_pp12::LN_comb;
        flag&=QName::tokenizer().insert(ns, "cover") == NS_pp12::LN_cover;
        flag&=QName::tokenizer().insert(ns, "cut") == NS_pp12::LN_cut;
        flag&=QName::tokenizer().insert(ns, "diamond") == NS_pp12::LN_diamond;
        flag&=QName::tokenizer().insert(ns, "fade") == NS_pp12::LN_fade;
        flag&=QName::tokenizer().insert(ns, "newsflash") == NS_pp12::LN_newsflash;
        flag&=QName::tokenizer().insert(ns, "plus") == NS_pp12::LN_plus;
        flag&=QName::tokenizer().insert(ns, "pull") == NS_pp12::LN_pull;
        flag&=QName::tokenizer().insert(ns, "push") == NS_pp12::LN_push;
        flag&=QName::tokenizer().insert(ns, "random") == NS_pp12::LN_random;
        flag&=QName::tokenizer().insert(ns, "randomBar") == NS_pp12::LN_randomBar;
        flag&=QName::tokenizer().insert(ns, "split") == NS_pp12::LN_split;
        flag&=QName::tokenizer().insert(ns, "strips") == NS_pp12::LN_strips;
        flag&=QName::tokenizer().insert(ns, "wedge") == NS_pp12::LN_wedge;
        flag&=QName::tokenizer().insert(ns, "wheel") == NS_pp12::LN_wheel;
        flag&=QName::tokenizer().insert(ns, "wipe") == NS_pp12::LN_wipe;
        flag&=QName::tokenizer().insert(ns, "zoom") == NS_pp12::LN_zoom;
        flag&=QName::tokenizer().insert(ns, "sndAc") == NS_pp12::LN_sndAc;
        flag&=QName::tokenizer().insert(ns, "sldRef") == NS_pp12::LN_sldRef;
        flag&=QName::tokenizer().insert(ns, "ssJumpAc") == NS_pp12::LN_ssJumpAc;
        flag&=QName::tokenizer().insert(ns, "sldJumpAc") == NS_pp12::LN_sldJumpAc;
        flag&=QName::tokenizer().insert(ns, "urlAc") == NS_pp12::LN_urlAc;
        flag&=QName::tokenizer().insert(ns, "fileAc") == NS_pp12::LN_fileAc;
        flag&=QName::tokenizer().insert(ns, "presAc") == NS_pp12::LN_presAc;
        flag&=QName::tokenizer().insert(ns, "tooltip") == NS_pp12::LN_tooltip;
        flag&=QName::tokenizer().insert(ns, "custShow") == NS_pp12::LN_custShow;
        flag&=QName::tokenizer().insert(ns, "programAc") == NS_pp12::LN_programAc;
        flag&=QName::tokenizer().insert(ns, "macroAc") == NS_pp12::LN_macroAc;
        flag&=QName::tokenizer().insert(ns, "hlinkAc") == NS_pp12::LN_hlinkAc;
        flag&=QName::tokenizer().insert(ns, "oleVerbAc") == NS_pp12::LN_oleVerbAc;
        flag&=QName::tokenizer().insert(ns, "mediaAc") == NS_pp12::LN_mediaAc;
        flag&=QName::tokenizer().insert(ns, "custShowAc") == NS_pp12::LN_custShowAc;
        flag&=QName::tokenizer().insert(ns, "onClick") == NS_pp12::LN_onClick;
        flag&=QName::tokenizer().insert(ns, "onMouseOver") == NS_pp12::LN_onMouseOver;
        flag&=QName::tokenizer().insert(ns, "seconds") == NS_pp12::LN_seconds;
        flag&=QName::tokenizer().insert(ns, "percent") == NS_pp12::LN_percent;
        flag&=QName::tokenizer().insert(ns, "chRg") == NS_pp12::LN_chRg;
        flag&=QName::tokenizer().insert(ns, "pRg") == NS_pp12::LN_pRg;
        flag&=QName::tokenizer().insert(ns, "bg") == NS_pp12::LN_bg;
        flag&=QName::tokenizer().insert(ns, "subSp") == NS_pp12::LN_subSp;
        flag&=QName::tokenizer().insert(ns, "oleChartEl") == NS_pp12::LN_oleChartEl;
        flag&=QName::tokenizer().insert(ns, "tx") == NS_pp12::LN_tx;
        flag&=QName::tokenizer().insert(ns, "e2o") == NS_pp12::LN_e2o;
        flag&=QName::tokenizer().insert(ns, "sldTgt") == NS_pp12::LN_sldTgt;
        flag&=QName::tokenizer().insert(ns, "sndTgt") == NS_pp12::LN_sndTgt;
        flag&=QName::tokenizer().insert(ns, "spTgt") == NS_pp12::LN_spTgt;
        flag&=QName::tokenizer().insert(ns, "tgtEl") == NS_pp12::LN_tgtEl;
        flag&=QName::tokenizer().insert(ns, "tn") == NS_pp12::LN_tn;
        flag&=QName::tokenizer().insert(ns, "rtn") == NS_pp12::LN_rtn;
        flag&=QName::tokenizer().insert(ns, "cond") == NS_pp12::LN_cond;
        flag&=QName::tokenizer().insert(ns, "par") == NS_pp12::LN_par;
        flag&=QName::tokenizer().insert(ns, "seq") == NS_pp12::LN_seq;
        flag&=QName::tokenizer().insert(ns, "excl") == NS_pp12::LN_excl;
        flag&=QName::tokenizer().insert(ns, "animate") == NS_pp12::LN_animate;
        flag&=QName::tokenizer().insert(ns, "animateColor") == NS_pp12::LN_animateColor;
        flag&=QName::tokenizer().insert(ns, "animateEffect") == NS_pp12::LN_animateEffect;
        flag&=QName::tokenizer().insert(ns, "animateMotion") == NS_pp12::LN_animateMotion;
        flag&=QName::tokenizer().insert(ns, "animateRotation") == NS_pp12::LN_animateRotation;
        flag&=QName::tokenizer().insert(ns, "animateScale") == NS_pp12::LN_animateScale;
        flag&=QName::tokenizer().insert(ns, "command") == NS_pp12::LN_command;
        flag&=QName::tokenizer().insert(ns, "set") == NS_pp12::LN_set;
        flag&=QName::tokenizer().insert(ns, "audio") == NS_pp12::LN_audio;
        flag&=QName::tokenizer().insert(ns, "video") == NS_pp12::LN_video;
        flag&=QName::tokenizer().insert(ns, "beginCondLst") == NS_pp12::LN_beginCondLst;
        flag&=QName::tokenizer().insert(ns, "endCondLst") == NS_pp12::LN_endCondLst;
        flag&=QName::tokenizer().insert(ns, "endSync") == NS_pp12::LN_endSync;
        flag&=QName::tokenizer().insert(ns, "iterate") == NS_pp12::LN_iterate;
        flag&=QName::tokenizer().insert(ns, "childLst") == NS_pp12::LN_childLst;
        flag&=QName::tokenizer().insert(ns, "subLst") == NS_pp12::LN_subLst;
        flag&=QName::tokenizer().insert(ns, "timeNodeBase") == NS_pp12::LN_timeNodeBase;
        flag&=QName::tokenizer().insert(ns, "prevCondLst") == NS_pp12::LN_prevCondLst;
        flag&=QName::tokenizer().insert(ns, "nextCondLst") == NS_pp12::LN_nextCondLst;
        flag&=QName::tokenizer().insert(ns, "attrName") == NS_pp12::LN_attrName;
        flag&=QName::tokenizer().insert(ns, "attrNameLst") == NS_pp12::LN_attrNameLst;
        flag&=QName::tokenizer().insert(ns, "boolVal") == NS_pp12::LN_boolVal;
        flag&=QName::tokenizer().insert(ns, "intVal") == NS_pp12::LN_intVal;
        flag&=QName::tokenizer().insert(ns, "fltVal") == NS_pp12::LN_fltVal;
        flag&=QName::tokenizer().insert(ns, "strVal") == NS_pp12::LN_strVal;
        flag&=QName::tokenizer().insert(ns, "val") == NS_pp12::LN_val;
        flag&=QName::tokenizer().insert(ns, "tav") == NS_pp12::LN_tav;
        flag&=QName::tokenizer().insert(ns, "behaviorBase") == NS_pp12::LN_behaviorBase;
        flag&=QName::tokenizer().insert(ns, "tavLst") == NS_pp12::LN_tavLst;
        flag&=QName::tokenizer().insert(ns, "rgb") == NS_pp12::LN_rgb;
        flag&=QName::tokenizer().insert(ns, "hsl") == NS_pp12::LN_hsl;
        flag&=QName::tokenizer().insert(ns, "by") == NS_pp12::LN_by;
        flag&=QName::tokenizer().insert(ns, "from") == NS_pp12::LN_from;
        flag&=QName::tokenizer().insert(ns, "to") == NS_pp12::LN_to;
        flag&=QName::tokenizer().insert(ns, "progress") == NS_pp12::LN_progress;
        flag&=QName::tokenizer().insert(ns, "rCenter") == NS_pp12::LN_rCenter;
        flag&=QName::tokenizer().insert(ns, "mediaNodeBase") == NS_pp12::LN_mediaNodeBase;
        flag&=QName::tokenizer().insert(ns, "timeNodeLst") == NS_pp12::LN_timeNodeLst;
        flag&=QName::tokenizer().insert(ns, "tmpl") == NS_pp12::LN_tmpl;
        flag&=QName::tokenizer().insert(ns, "tmplLst") == NS_pp12::LN_tmplLst;
        flag&=QName::tokenizer().insert(ns, "bldAsOne") == NS_pp12::LN_bldAsOne;
        flag&=QName::tokenizer().insert(ns, "bldSub") == NS_pp12::LN_bldSub;
        flag&=QName::tokenizer().insert(ns, "bldP") == NS_pp12::LN_bldP;
        flag&=QName::tokenizer().insert(ns, "bldDgm") == NS_pp12::LN_bldDgm;
        flag&=QName::tokenizer().insert(ns, "bldOleChart") == NS_pp12::LN_bldOleChart;
        flag&=QName::tokenizer().insert(ns, "bldE2o") == NS_pp12::LN_bldE2o;
        flag&=QName::tokenizer().insert(ns, "buildLst") == NS_pp12::LN_buildLst;
        flag&=QName::tokenizer().insert(ns, "sld") == NS_pp12::LN_sld;
        flag&=QName::tokenizer().insert(ns, "all") == NS_pp12::LN_all;
        flag&=QName::tokenizer().insert(ns, "range") == NS_pp12::LN_range;
        flag&=QName::tokenizer().insert(ns, "custData") == NS_pp12::LN_custData;
        flag&=QName::tokenizer().insert(ns, "commentAuthor") == NS_pp12::LN_commentAuthor;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_pp12::LN_pos;
        flag&=QName::tokenizer().insert(ns, "text") == NS_pp12::LN_text;
        flag&=QName::tokenizer().insert(ns, "cm") == NS_pp12::LN_cm;
        flag&=QName::tokenizer().insert(ns, "embed") == NS_pp12::LN_embed;
        flag&=QName::tokenizer().insert(ns, "link") == NS_pp12::LN_link;
        flag&=QName::tokenizer().insert(ns, "sldId") == NS_pp12::LN_sldId;
        flag&=QName::tokenizer().insert(ns, "sldMasterId") == NS_pp12::LN_sldMasterId;
        flag&=QName::tokenizer().insert(ns, "notesMasterId") == NS_pp12::LN_notesMasterId;
        flag&=QName::tokenizer().insert(ns, "handoutMasterId") == NS_pp12::LN_handoutMasterId;
        flag&=QName::tokenizer().insert(ns, "sldLst") == NS_pp12::LN_sldLst;
        flag&=QName::tokenizer().insert(ns, "sldMasterIdLst") == NS_pp12::LN_sldMasterIdLst;
        flag&=QName::tokenizer().insert(ns, "sldIdLst") == NS_pp12::LN_sldIdLst;
        flag&=QName::tokenizer().insert(ns, "notesMasterIdLst") == NS_pp12::LN_notesMasterIdLst;
        flag&=QName::tokenizer().insert(ns, "handoutMasterIdLst") == NS_pp12::LN_handoutMasterIdLst;
        flag&=QName::tokenizer().insert(ns, "slideSize") == NS_pp12::LN_slideSize;
        flag&=QName::tokenizer().insert(ns, "notesSize") == NS_pp12::LN_notesSize;
        flag&=QName::tokenizer().insert(ns, "custShowLst") == NS_pp12::LN_custShowLst;
        flag&=QName::tokenizer().insert(ns, "photoAlbum") == NS_pp12::LN_photoAlbum;
        flag&=QName::tokenizer().insert(ns, "commentAuthorLst") == NS_pp12::LN_commentAuthorLst;
        flag&=QName::tokenizer().insert(ns, "custDataLst") == NS_pp12::LN_custDataLst;
        flag&=QName::tokenizer().insert(ns, "creatingAppVersion") == NS_pp12::LN_creatingAppVersion;
        flag&=QName::tokenizer().insert(ns, "editingAppVersion") == NS_pp12::LN_editingAppVersion;
        flag&=QName::tokenizer().insert(ns, "presentation") == NS_pp12::LN_presentation;
        flag&=QName::tokenizer().insert(ns, "pubWhat") == NS_pp12::LN_pubWhat;
        flag&=QName::tokenizer().insert(ns, "pubAs") == NS_pp12::LN_pubAs;
        flag&=QName::tokenizer().insert(ns, "present") == NS_pp12::LN_present;
        flag&=QName::tokenizer().insert(ns, "browse") == NS_pp12::LN_browse;
        flag&=QName::tokenizer().insert(ns, "kiosk") == NS_pp12::LN_kiosk;
        flag&=QName::tokenizer().insert(ns, "showType") == NS_pp12::LN_showType;
        flag&=QName::tokenizer().insert(ns, "showWhat") == NS_pp12::LN_showWhat;
        flag&=QName::tokenizer().insert(ns, "penClr") == NS_pp12::LN_penClr;
        flag&=QName::tokenizer().insert(ns, "htmlPubPr") == NS_pp12::LN_htmlPubPr;
        flag&=QName::tokenizer().insert(ns, "webPr") == NS_pp12::LN_webPr;
        flag&=QName::tokenizer().insert(ns, "prnPr") == NS_pp12::LN_prnPr;
        flag&=QName::tokenizer().insert(ns, "showPr") == NS_pp12::LN_showPr;
        flag&=QName::tokenizer().insert(ns, "presentationPr") == NS_pp12::LN_presentationPr;
        flag&=QName::tokenizer().insert(ns, "str") == NS_pp12::LN_str;
        flag&=QName::tokenizer().insert(ns, "bin") == NS_pp12::LN_bin;
        flag&=QName::tokenizer().insert(ns, "tag") == NS_pp12::LN_tag;
        flag&=QName::tokenizer().insert(ns, "ph") == NS_pp12::LN_ph;
        flag&=QName::tokenizer().insert(ns, "media") == NS_pp12::LN_media;
        flag&=QName::tokenizer().insert(ns, "actions") == NS_pp12::LN_actions;
        flag&=QName::tokenizer().insert(ns, "anvPr") == NS_pp12::LN_anvPr;
        flag&=QName::tokenizer().insert(ns, "anvSpPr") == NS_pp12::LN_anvSpPr;
        flag&=QName::tokenizer().insert(ns, "pnvPr") == NS_pp12::LN_pnvPr;
        flag&=QName::tokenizer().insert(ns, "nvSpPr") == NS_pp12::LN_nvSpPr;
        flag&=QName::tokenizer().insert(ns, "spPr") == NS_pp12::LN_spPr;
        flag&=QName::tokenizer().insert(ns, "style") == NS_pp12::LN_style;
        flag&=QName::tokenizer().insert(ns, "txBody") == NS_pp12::LN_txBody;
        flag&=QName::tokenizer().insert(ns, "anvCxnSpPr") == NS_pp12::LN_anvCxnSpPr;
        flag&=QName::tokenizer().insert(ns, "nvCxnSpPr") == NS_pp12::LN_nvCxnSpPr;
        flag&=QName::tokenizer().insert(ns, "anvGraphicFramePr") == NS_pp12::LN_anvGraphicFramePr;
        flag&=QName::tokenizer().insert(ns, "nvGraphicFramePr") == NS_pp12::LN_nvGraphicFramePr;
        flag&=QName::tokenizer().insert(ns, "xfrm") == NS_pp12::LN_xfrm;
        flag&=QName::tokenizer().insert(ns, "graphic") == NS_pp12::LN_graphic;
        flag&=QName::tokenizer().insert(ns, "anvGspPr") == NS_pp12::LN_anvGspPr;
        flag&=QName::tokenizer().insert(ns, "nvGspPr") == NS_pp12::LN_nvGspPr;
        flag&=QName::tokenizer().insert(ns, "gspPr") == NS_pp12::LN_gspPr;
        flag&=QName::tokenizer().insert(ns, "sp") == NS_pp12::LN_sp;
        flag&=QName::tokenizer().insert(ns, "gsp") == NS_pp12::LN_gsp;
        flag&=QName::tokenizer().insert(ns, "graphicFrame") == NS_pp12::LN_graphicFrame;
        flag&=QName::tokenizer().insert(ns, "cxnsp") == NS_pp12::LN_cxnsp;
        flag&=QName::tokenizer().insert(ns, "clrMap") == NS_pp12::LN_clrMap;
        flag&=QName::tokenizer().insert(ns, "clrMapOvr") == NS_pp12::LN_clrMapOvr;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_pp12::LN_fill;
        flag&=QName::tokenizer().insert(ns, "bgPr") == NS_pp12::LN_bgPr;
        flag&=QName::tokenizer().insert(ns, "bgLink") == NS_pp12::LN_bgLink;
        flag&=QName::tokenizer().insert(ns, "bgMaster") == NS_pp12::LN_bgMaster;
        flag&=QName::tokenizer().insert(ns, "spTree") == NS_pp12::LN_spTree;
        flag&=QName::tokenizer().insert(ns, "tagLst") == NS_pp12::LN_tagLst;
        flag&=QName::tokenizer().insert(ns, "sldBase") == NS_pp12::LN_sldBase;
        flag&=QName::tokenizer().insert(ns, "cmLst") == NS_pp12::LN_cmLst;
        flag&=QName::tokenizer().insert(ns, "transition") == NS_pp12::LN_transition;
        flag&=QName::tokenizer().insert(ns, "timingInfo") == NS_pp12::LN_timingInfo;
        flag&=QName::tokenizer().insert(ns, "hf") == NS_pp12::LN_hf;
        flag&=QName::tokenizer().insert(ns, "sldLayout") == NS_pp12::LN_sldLayout;
        flag&=QName::tokenizer().insert(ns, "defLink") == NS_pp12::LN_defLink;
        flag&=QName::tokenizer().insert(ns, "outline1Link") == NS_pp12::LN_outline1Link;
        flag&=QName::tokenizer().insert(ns, "outline2Link") == NS_pp12::LN_outline2Link;
        flag&=QName::tokenizer().insert(ns, "lvl1Link") == NS_pp12::LN_lvl1Link;
        flag&=QName::tokenizer().insert(ns, "lvl2Link") == NS_pp12::LN_lvl2Link;
        flag&=QName::tokenizer().insert(ns, "lvl3Link") == NS_pp12::LN_lvl3Link;
        flag&=QName::tokenizer().insert(ns, "lvl4Link") == NS_pp12::LN_lvl4Link;
        flag&=QName::tokenizer().insert(ns, "lvl5Link") == NS_pp12::LN_lvl5Link;
        flag&=QName::tokenizer().insert(ns, "lvl6Link") == NS_pp12::LN_lvl6Link;
        flag&=QName::tokenizer().insert(ns, "lvl7Link") == NS_pp12::LN_lvl7Link;
        flag&=QName::tokenizer().insert(ns, "lvl8Link") == NS_pp12::LN_lvl8Link;
        flag&=QName::tokenizer().insert(ns, "lvl9Link") == NS_pp12::LN_lvl9Link;
        flag&=QName::tokenizer().insert(ns, "txLstStyle") == NS_pp12::LN_txLstStyle;
        flag&=QName::tokenizer().insert(ns, "themeLinkLst") == NS_pp12::LN_themeLinkLst;
        flag&=QName::tokenizer().insert(ns, "titleStyle") == NS_pp12::LN_titleStyle;
        flag&=QName::tokenizer().insert(ns, "bodyStyle") == NS_pp12::LN_bodyStyle;
        flag&=QName::tokenizer().insert(ns, "notesStyle") == NS_pp12::LN_notesStyle;
        flag&=QName::tokenizer().insert(ns, "otherStyle") == NS_pp12::LN_otherStyle;
        flag&=QName::tokenizer().insert(ns, "sldLayoutId") == NS_pp12::LN_sldLayoutId;
        flag&=QName::tokenizer().insert(ns, "sldLayoutIdLst") == NS_pp12::LN_sldLayoutIdLst;
        flag&=QName::tokenizer().insert(ns, "txStyles") == NS_pp12::LN_txStyles;
        flag&=QName::tokenizer().insert(ns, "sldMaster") == NS_pp12::LN_sldMaster;
        flag&=QName::tokenizer().insert(ns, "handoutMaster") == NS_pp12::LN_handoutMaster;
        flag&=QName::tokenizer().insert(ns, "notesMaster") == NS_pp12::LN_notesMaster;
        flag&=QName::tokenizer().insert(ns, "notes") == NS_pp12::LN_notes;
        flag&=QName::tokenizer().insert(ns, "restoredLeft") == NS_pp12::LN_restoredLeft;
        flag&=QName::tokenizer().insert(ns, "restoredTop") == NS_pp12::LN_restoredTop;
        flag&=QName::tokenizer().insert(ns, "scale") == NS_pp12::LN_scale;
        flag&=QName::tokenizer().insert(ns, "origin") == NS_pp12::LN_origin;
        flag&=QName::tokenizer().insert(ns, "viewPrBase") == NS_pp12::LN_viewPrBase;
        flag&=QName::tokenizer().insert(ns, "guide") == NS_pp12::LN_guide;
        flag&=QName::tokenizer().insert(ns, "guideLst") == NS_pp12::LN_guideLst;
        flag&=QName::tokenizer().insert(ns, "sldViewPrBase") == NS_pp12::LN_sldViewPrBase;
        flag&=QName::tokenizer().insert(ns, "normalViewPr") == NS_pp12::LN_normalViewPr;
        flag&=QName::tokenizer().insert(ns, "slideViewPr") == NS_pp12::LN_slideViewPr;
        flag&=QName::tokenizer().insert(ns, "outlineViewPr") == NS_pp12::LN_outlineViewPr;
        flag&=QName::tokenizer().insert(ns, "notesTextViewPr") == NS_pp12::LN_notesTextViewPr;
        flag&=QName::tokenizer().insert(ns, "sorterViewPr") == NS_pp12::LN_sorterViewPr;
        flag&=QName::tokenizer().insert(ns, "notesViewPr") == NS_pp12::LN_notesViewPr;
        flag&=QName::tokenizer().insert(ns, "gridSpacing") == NS_pp12::LN_gridSpacing;
        flag&=QName::tokenizer().insert(ns, "viewPr") == NS_pp12::LN_viewPr;
        CPPUNIT_ASSERT_MESSAGE("test for namespace pp12 failed", flag);}
    void test_v11(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/visio/2003/core";
        flag&=QName::tokenizer().insert(ns, "VisioDocument") == NS_v11::LN_VisioDocument;
        flag&=QName::tokenizer().insert(ns, "SolutionXML") == NS_v11::LN_SolutionXML;
        flag&=QName::tokenizer().insert(ns, "cp") == NS_v11::LN_cp;
        flag&=QName::tokenizer().insert(ns, "pp") == NS_v11::LN_pp;
        flag&=QName::tokenizer().insert(ns, "tp") == NS_v11::LN_tp;
        flag&=QName::tokenizer().insert(ns, "fld") == NS_v11::LN_fld;
        flag&=QName::tokenizer().insert(ns, "XProp") == NS_v11::LN_XProp;
        flag&=QName::tokenizer().insert(ns, "Text") == NS_v11::LN_Text;
        flag&=QName::tokenizer().insert(ns, "XForm") == NS_v11::LN_XForm;
        flag&=QName::tokenizer().insert(ns, "Line") == NS_v11::LN_Line;
        flag&=QName::tokenizer().insert(ns, "Fill") == NS_v11::LN_Fill;
        flag&=QName::tokenizer().insert(ns, "XForm1D") == NS_v11::LN_XForm1D;
        flag&=QName::tokenizer().insert(ns, "Event") == NS_v11::LN_Event;
        flag&=QName::tokenizer().insert(ns, "LayerMem") == NS_v11::LN_LayerMem;
        flag&=QName::tokenizer().insert(ns, "StyleProp") == NS_v11::LN_StyleProp;
        flag&=QName::tokenizer().insert(ns, "Foreign") == NS_v11::LN_Foreign;
        flag&=QName::tokenizer().insert(ns, "PageProps") == NS_v11::LN_PageProps;
        flag&=QName::tokenizer().insert(ns, "TextBlock") == NS_v11::LN_TextBlock;
        flag&=QName::tokenizer().insert(ns, "TextXForm") == NS_v11::LN_TextXForm;
        flag&=QName::tokenizer().insert(ns, "Align") == NS_v11::LN_Align;
        flag&=QName::tokenizer().insert(ns, "Protection") == NS_v11::LN_Protection;
        flag&=QName::tokenizer().insert(ns, "Help") == NS_v11::LN_Help;
        flag&=QName::tokenizer().insert(ns, "Misc") == NS_v11::LN_Misc;
        flag&=QName::tokenizer().insert(ns, "RulerGrid") == NS_v11::LN_RulerGrid;
        flag&=QName::tokenizer().insert(ns, "DocProps") == NS_v11::LN_DocProps;
        flag&=QName::tokenizer().insert(ns, "Image") == NS_v11::LN_Image;
        flag&=QName::tokenizer().insert(ns, "Group") == NS_v11::LN_Group;
        flag&=QName::tokenizer().insert(ns, "Layout") == NS_v11::LN_Layout;
        flag&=QName::tokenizer().insert(ns, "PageLayout") == NS_v11::LN_PageLayout;
        flag&=QName::tokenizer().insert(ns, "PrintProps") == NS_v11::LN_PrintProps;
        flag&=QName::tokenizer().insert(ns, "Char") == NS_v11::LN_Char;
        flag&=QName::tokenizer().insert(ns, "Para") == NS_v11::LN_Para;
        flag&=QName::tokenizer().insert(ns, "Tabs") == NS_v11::LN_Tabs;
        flag&=QName::tokenizer().insert(ns, "Scratch") == NS_v11::LN_Scratch;
        flag&=QName::tokenizer().insert(ns, "Connection") == NS_v11::LN_Connection;
        flag&=QName::tokenizer().insert(ns, "ConnectionABCD") == NS_v11::LN_ConnectionABCD;
        flag&=QName::tokenizer().insert(ns, "Field") == NS_v11::LN_Field;
        flag&=QName::tokenizer().insert(ns, "Control") == NS_v11::LN_Control;
        flag&=QName::tokenizer().insert(ns, "Geom") == NS_v11::LN_Geom;
        flag&=QName::tokenizer().insert(ns, "Act") == NS_v11::LN_Act;
        flag&=QName::tokenizer().insert(ns, "Layer") == NS_v11::LN_Layer;
        flag&=QName::tokenizer().insert(ns, "User") == NS_v11::LN_User;
        flag&=QName::tokenizer().insert(ns, "Prop") == NS_v11::LN_Prop;
        flag&=QName::tokenizer().insert(ns, "Hyperlink") == NS_v11::LN_Hyperlink;
        flag&=QName::tokenizer().insert(ns, "Reviewer") == NS_v11::LN_Reviewer;
        flag&=QName::tokenizer().insert(ns, "Annotation") == NS_v11::LN_Annotation;
        flag&=QName::tokenizer().insert(ns, "SmartTagDef") == NS_v11::LN_SmartTagDef;
        flag&=QName::tokenizer().insert(ns, "Data1") == NS_v11::LN_Data1;
        flag&=QName::tokenizer().insert(ns, "Data2") == NS_v11::LN_Data2;
        flag&=QName::tokenizer().insert(ns, "Data3") == NS_v11::LN_Data3;
        flag&=QName::tokenizer().insert(ns, "ForeignData") == NS_v11::LN_ForeignData;
        flag&=QName::tokenizer().insert(ns, "PinX") == NS_v11::LN_PinX;
        flag&=QName::tokenizer().insert(ns, "PinY") == NS_v11::LN_PinY;
        flag&=QName::tokenizer().insert(ns, "Width") == NS_v11::LN_Width;
        flag&=QName::tokenizer().insert(ns, "Height") == NS_v11::LN_Height;
        flag&=QName::tokenizer().insert(ns, "LocPinX") == NS_v11::LN_LocPinX;
        flag&=QName::tokenizer().insert(ns, "LocPinY") == NS_v11::LN_LocPinY;
        flag&=QName::tokenizer().insert(ns, "Angle") == NS_v11::LN_Angle;
        flag&=QName::tokenizer().insert(ns, "FlipX") == NS_v11::LN_FlipX;
        flag&=QName::tokenizer().insert(ns, "FlipY") == NS_v11::LN_FlipY;
        flag&=QName::tokenizer().insert(ns, "ResizeMode") == NS_v11::LN_ResizeMode;
        flag&=QName::tokenizer().insert(ns, "LineWeight") == NS_v11::LN_LineWeight;
        flag&=QName::tokenizer().insert(ns, "LineColor") == NS_v11::LN_LineColor;
        flag&=QName::tokenizer().insert(ns, "LinePattern") == NS_v11::LN_LinePattern;
        flag&=QName::tokenizer().insert(ns, "Rounding") == NS_v11::LN_Rounding;
        flag&=QName::tokenizer().insert(ns, "EndArrowSize") == NS_v11::LN_EndArrowSize;
        flag&=QName::tokenizer().insert(ns, "BeginArrow") == NS_v11::LN_BeginArrow;
        flag&=QName::tokenizer().insert(ns, "EndArrow") == NS_v11::LN_EndArrow;
        flag&=QName::tokenizer().insert(ns, "LineCap") == NS_v11::LN_LineCap;
        flag&=QName::tokenizer().insert(ns, "BeginArrowSize") == NS_v11::LN_BeginArrowSize;
        flag&=QName::tokenizer().insert(ns, "LineColorTrans") == NS_v11::LN_LineColorTrans;
        flag&=QName::tokenizer().insert(ns, "FillForegnd") == NS_v11::LN_FillForegnd;
        flag&=QName::tokenizer().insert(ns, "FillBkgnd") == NS_v11::LN_FillBkgnd;
        flag&=QName::tokenizer().insert(ns, "FillPattern") == NS_v11::LN_FillPattern;
        flag&=QName::tokenizer().insert(ns, "ShdwForegnd") == NS_v11::LN_ShdwForegnd;
        flag&=QName::tokenizer().insert(ns, "ShdwBkgnd") == NS_v11::LN_ShdwBkgnd;
        flag&=QName::tokenizer().insert(ns, "ShdwPattern") == NS_v11::LN_ShdwPattern;
        flag&=QName::tokenizer().insert(ns, "FillForegndTrans") == NS_v11::LN_FillForegndTrans;
        flag&=QName::tokenizer().insert(ns, "FillBkgndTrans") == NS_v11::LN_FillBkgndTrans;
        flag&=QName::tokenizer().insert(ns, "ShdwForegndTrans") == NS_v11::LN_ShdwForegndTrans;
        flag&=QName::tokenizer().insert(ns, "ShdwBkgndTrans") == NS_v11::LN_ShdwBkgndTrans;
        flag&=QName::tokenizer().insert(ns, "ShapeShdwType") == NS_v11::LN_ShapeShdwType;
        flag&=QName::tokenizer().insert(ns, "ShapeShdwOffsetX") == NS_v11::LN_ShapeShdwOffsetX;
        flag&=QName::tokenizer().insert(ns, "ShapeShdwOffsetY") == NS_v11::LN_ShapeShdwOffsetY;
        flag&=QName::tokenizer().insert(ns, "ShapeShdwObliqueAngle") == NS_v11::LN_ShapeShdwObliqueAngle;
        flag&=QName::tokenizer().insert(ns, "ShapeShdwScaleFactor") == NS_v11::LN_ShapeShdwScaleFactor;
        flag&=QName::tokenizer().insert(ns, "BeginX") == NS_v11::LN_BeginX;
        flag&=QName::tokenizer().insert(ns, "BeginY") == NS_v11::LN_BeginY;
        flag&=QName::tokenizer().insert(ns, "EndX") == NS_v11::LN_EndX;
        flag&=QName::tokenizer().insert(ns, "EndY") == NS_v11::LN_EndY;
        flag&=QName::tokenizer().insert(ns, "TheData") == NS_v11::LN_TheData;
        flag&=QName::tokenizer().insert(ns, "TheText") == NS_v11::LN_TheText;
        flag&=QName::tokenizer().insert(ns, "EventDblClick") == NS_v11::LN_EventDblClick;
        flag&=QName::tokenizer().insert(ns, "EventXFMod") == NS_v11::LN_EventXFMod;
        flag&=QName::tokenizer().insert(ns, "EventDrop") == NS_v11::LN_EventDrop;
        flag&=QName::tokenizer().insert(ns, "LayerMember") == NS_v11::LN_LayerMember;
        flag&=QName::tokenizer().insert(ns, "EnableLineProps") == NS_v11::LN_EnableLineProps;
        flag&=QName::tokenizer().insert(ns, "EnableFillProps") == NS_v11::LN_EnableFillProps;
        flag&=QName::tokenizer().insert(ns, "EnableTextProps") == NS_v11::LN_EnableTextProps;
        flag&=QName::tokenizer().insert(ns, "HideForApply") == NS_v11::LN_HideForApply;
        flag&=QName::tokenizer().insert(ns, "ImgOffsetX") == NS_v11::LN_ImgOffsetX;
        flag&=QName::tokenizer().insert(ns, "ImgOffsetY") == NS_v11::LN_ImgOffsetY;
        flag&=QName::tokenizer().insert(ns, "ImgWidth") == NS_v11::LN_ImgWidth;
        flag&=QName::tokenizer().insert(ns, "ImgHeight") == NS_v11::LN_ImgHeight;
        flag&=QName::tokenizer().insert(ns, "PageWidth") == NS_v11::LN_PageWidth;
        flag&=QName::tokenizer().insert(ns, "PageHeight") == NS_v11::LN_PageHeight;
        flag&=QName::tokenizer().insert(ns, "ShdwOffsetX") == NS_v11::LN_ShdwOffsetX;
        flag&=QName::tokenizer().insert(ns, "ShdwOffsetY") == NS_v11::LN_ShdwOffsetY;
        flag&=QName::tokenizer().insert(ns, "PageScale") == NS_v11::LN_PageScale;
        flag&=QName::tokenizer().insert(ns, "DrawingScale") == NS_v11::LN_DrawingScale;
        flag&=QName::tokenizer().insert(ns, "DrawingSizeType") == NS_v11::LN_DrawingSizeType;
        flag&=QName::tokenizer().insert(ns, "DrawingScaleType") == NS_v11::LN_DrawingScaleType;
        flag&=QName::tokenizer().insert(ns, "InhibitSnap") == NS_v11::LN_InhibitSnap;
        flag&=QName::tokenizer().insert(ns, "UIVisibility") == NS_v11::LN_UIVisibility;
        flag&=QName::tokenizer().insert(ns, "ShdwType") == NS_v11::LN_ShdwType;
        flag&=QName::tokenizer().insert(ns, "ShdwObliqueAngle") == NS_v11::LN_ShdwObliqueAngle;
        flag&=QName::tokenizer().insert(ns, "ShdwScaleFactor") == NS_v11::LN_ShdwScaleFactor;
        flag&=QName::tokenizer().insert(ns, "LeftMargin") == NS_v11::LN_LeftMargin;
        flag&=QName::tokenizer().insert(ns, "RightMargin") == NS_v11::LN_RightMargin;
        flag&=QName::tokenizer().insert(ns, "TopMargin") == NS_v11::LN_TopMargin;
        flag&=QName::tokenizer().insert(ns, "BottomMargin") == NS_v11::LN_BottomMargin;
        flag&=QName::tokenizer().insert(ns, "VerticalAlign") == NS_v11::LN_VerticalAlign;
        flag&=QName::tokenizer().insert(ns, "TextBkgnd") == NS_v11::LN_TextBkgnd;
        flag&=QName::tokenizer().insert(ns, "DefaultTabStop") == NS_v11::LN_DefaultTabStop;
        flag&=QName::tokenizer().insert(ns, "TextDirection") == NS_v11::LN_TextDirection;
        flag&=QName::tokenizer().insert(ns, "TextBkgndTrans") == NS_v11::LN_TextBkgndTrans;
        flag&=QName::tokenizer().insert(ns, "TxtPinX") == NS_v11::LN_TxtPinX;
        flag&=QName::tokenizer().insert(ns, "TxtPinY") == NS_v11::LN_TxtPinY;
        flag&=QName::tokenizer().insert(ns, "TxtWidth") == NS_v11::LN_TxtWidth;
        flag&=QName::tokenizer().insert(ns, "TxtHeight") == NS_v11::LN_TxtHeight;
        flag&=QName::tokenizer().insert(ns, "TxtLocPinX") == NS_v11::LN_TxtLocPinX;
        flag&=QName::tokenizer().insert(ns, "TxtLocPinY") == NS_v11::LN_TxtLocPinY;
        flag&=QName::tokenizer().insert(ns, "TxtAngle") == NS_v11::LN_TxtAngle;
        flag&=QName::tokenizer().insert(ns, "AlignLeft") == NS_v11::LN_AlignLeft;
        flag&=QName::tokenizer().insert(ns, "AlignCenter") == NS_v11::LN_AlignCenter;
        flag&=QName::tokenizer().insert(ns, "AlignRight") == NS_v11::LN_AlignRight;
        flag&=QName::tokenizer().insert(ns, "AlignTop") == NS_v11::LN_AlignTop;
        flag&=QName::tokenizer().insert(ns, "AlignMiddle") == NS_v11::LN_AlignMiddle;
        flag&=QName::tokenizer().insert(ns, "AlignBottom") == NS_v11::LN_AlignBottom;
        flag&=QName::tokenizer().insert(ns, "LockWidth") == NS_v11::LN_LockWidth;
        flag&=QName::tokenizer().insert(ns, "LockHeight") == NS_v11::LN_LockHeight;
        flag&=QName::tokenizer().insert(ns, "LockMoveX") == NS_v11::LN_LockMoveX;
        flag&=QName::tokenizer().insert(ns, "LockMoveY") == NS_v11::LN_LockMoveY;
        flag&=QName::tokenizer().insert(ns, "LockAspect") == NS_v11::LN_LockAspect;
        flag&=QName::tokenizer().insert(ns, "LockDelete") == NS_v11::LN_LockDelete;
        flag&=QName::tokenizer().insert(ns, "LockBegin") == NS_v11::LN_LockBegin;
        flag&=QName::tokenizer().insert(ns, "LockEnd") == NS_v11::LN_LockEnd;
        flag&=QName::tokenizer().insert(ns, "LockRotate") == NS_v11::LN_LockRotate;
        flag&=QName::tokenizer().insert(ns, "LockCrop") == NS_v11::LN_LockCrop;
        flag&=QName::tokenizer().insert(ns, "LockVtxEdit") == NS_v11::LN_LockVtxEdit;
        flag&=QName::tokenizer().insert(ns, "LockTextEdit") == NS_v11::LN_LockTextEdit;
        flag&=QName::tokenizer().insert(ns, "LockFormat") == NS_v11::LN_LockFormat;
        flag&=QName::tokenizer().insert(ns, "LockGroup") == NS_v11::LN_LockGroup;
        flag&=QName::tokenizer().insert(ns, "LockCalcWH") == NS_v11::LN_LockCalcWH;
        flag&=QName::tokenizer().insert(ns, "LockSelect") == NS_v11::LN_LockSelect;
        flag&=QName::tokenizer().insert(ns, "LockCustProp") == NS_v11::LN_LockCustProp;
        flag&=QName::tokenizer().insert(ns, "HelpTopic") == NS_v11::LN_HelpTopic;
        flag&=QName::tokenizer().insert(ns, "Copyright") == NS_v11::LN_Copyright;
        flag&=QName::tokenizer().insert(ns, "NoObjHandles") == NS_v11::LN_NoObjHandles;
        flag&=QName::tokenizer().insert(ns, "NonPrinting") == NS_v11::LN_NonPrinting;
        flag&=QName::tokenizer().insert(ns, "NoCtlHandles") == NS_v11::LN_NoCtlHandles;
        flag&=QName::tokenizer().insert(ns, "NoAlignBox") == NS_v11::LN_NoAlignBox;
        flag&=QName::tokenizer().insert(ns, "UpdateAlignBox") == NS_v11::LN_UpdateAlignBox;
        flag&=QName::tokenizer().insert(ns, "HideText") == NS_v11::LN_HideText;
        flag&=QName::tokenizer().insert(ns, "DynFeedback") == NS_v11::LN_DynFeedback;
        flag&=QName::tokenizer().insert(ns, "GlueType") == NS_v11::LN_GlueType;
        flag&=QName::tokenizer().insert(ns, "WalkPreference") == NS_v11::LN_WalkPreference;
        flag&=QName::tokenizer().insert(ns, "BegTrigger") == NS_v11::LN_BegTrigger;
        flag&=QName::tokenizer().insert(ns, "EndTrigger") == NS_v11::LN_EndTrigger;
        flag&=QName::tokenizer().insert(ns, "ObjType") == NS_v11::LN_ObjType;
        flag&=QName::tokenizer().insert(ns, "Comment") == NS_v11::LN_Comment;
        flag&=QName::tokenizer().insert(ns, "IsDropSource") == NS_v11::LN_IsDropSource;
        flag&=QName::tokenizer().insert(ns, "NoLiveDynamics") == NS_v11::LN_NoLiveDynamics;
        flag&=QName::tokenizer().insert(ns, "LocalizeMerge") == NS_v11::LN_LocalizeMerge;
        flag&=QName::tokenizer().insert(ns, "Calendar") == NS_v11::LN_Calendar;
        flag&=QName::tokenizer().insert(ns, "LangID") == NS_v11::LN_LangID;
        flag&=QName::tokenizer().insert(ns, "ShapeKeywords") == NS_v11::LN_ShapeKeywords;
        flag&=QName::tokenizer().insert(ns, "DropOnPageScale") == NS_v11::LN_DropOnPageScale;
        flag&=QName::tokenizer().insert(ns, "XRulerDensity") == NS_v11::LN_XRulerDensity;
        flag&=QName::tokenizer().insert(ns, "YRulerDensity") == NS_v11::LN_YRulerDensity;
        flag&=QName::tokenizer().insert(ns, "XRulerOrigin") == NS_v11::LN_XRulerOrigin;
        flag&=QName::tokenizer().insert(ns, "YRulerOrigin") == NS_v11::LN_YRulerOrigin;
        flag&=QName::tokenizer().insert(ns, "XGridDensity") == NS_v11::LN_XGridDensity;
        flag&=QName::tokenizer().insert(ns, "YGridDensity") == NS_v11::LN_YGridDensity;
        flag&=QName::tokenizer().insert(ns, "XGridSpacing") == NS_v11::LN_XGridSpacing;
        flag&=QName::tokenizer().insert(ns, "YGridSpacing") == NS_v11::LN_YGridSpacing;
        flag&=QName::tokenizer().insert(ns, "XGridOrigin") == NS_v11::LN_XGridOrigin;
        flag&=QName::tokenizer().insert(ns, "YGridOrigin") == NS_v11::LN_YGridOrigin;
        flag&=QName::tokenizer().insert(ns, "OutputFormat") == NS_v11::LN_OutputFormat;
        flag&=QName::tokenizer().insert(ns, "LockPreview") == NS_v11::LN_LockPreview;
        flag&=QName::tokenizer().insert(ns, "AddMarkup") == NS_v11::LN_AddMarkup;
        flag&=QName::tokenizer().insert(ns, "ViewMarkup") == NS_v11::LN_ViewMarkup;
        flag&=QName::tokenizer().insert(ns, "PreviewQuality") == NS_v11::LN_PreviewQuality;
        flag&=QName::tokenizer().insert(ns, "PreviewScope") == NS_v11::LN_PreviewScope;
        flag&=QName::tokenizer().insert(ns, "DocLangID") == NS_v11::LN_DocLangID;
        flag&=QName::tokenizer().insert(ns, "Gamma") == NS_v11::LN_Gamma;
        flag&=QName::tokenizer().insert(ns, "Contrast") == NS_v11::LN_Contrast;
        flag&=QName::tokenizer().insert(ns, "Brightness") == NS_v11::LN_Brightness;
        flag&=QName::tokenizer().insert(ns, "Sharpen") == NS_v11::LN_Sharpen;
        flag&=QName::tokenizer().insert(ns, "Blur") == NS_v11::LN_Blur;
        flag&=QName::tokenizer().insert(ns, "Denoise") == NS_v11::LN_Denoise;
        flag&=QName::tokenizer().insert(ns, "Transparency") == NS_v11::LN_Transparency;
        flag&=QName::tokenizer().insert(ns, "SelectMode") == NS_v11::LN_SelectMode;
        flag&=QName::tokenizer().insert(ns, "DisplayMode") == NS_v11::LN_DisplayMode;
        flag&=QName::tokenizer().insert(ns, "IsDropTarget") == NS_v11::LN_IsDropTarget;
        flag&=QName::tokenizer().insert(ns, "IsSnapTarget") == NS_v11::LN_IsSnapTarget;
        flag&=QName::tokenizer().insert(ns, "IsTextEditTarget") == NS_v11::LN_IsTextEditTarget;
        flag&=QName::tokenizer().insert(ns, "DontMoveChildren") == NS_v11::LN_DontMoveChildren;
        flag&=QName::tokenizer().insert(ns, "ShapePermeableX") == NS_v11::LN_ShapePermeableX;
        flag&=QName::tokenizer().insert(ns, "ShapePermeableY") == NS_v11::LN_ShapePermeableY;
        flag&=QName::tokenizer().insert(ns, "ShapePermeablePlace") == NS_v11::LN_ShapePermeablePlace;
        flag&=QName::tokenizer().insert(ns, "ShapeFixedCode") == NS_v11::LN_ShapeFixedCode;
        flag&=QName::tokenizer().insert(ns, "ShapePlowCode") == NS_v11::LN_ShapePlowCode;
        flag&=QName::tokenizer().insert(ns, "ShapeRouteStyle") == NS_v11::LN_ShapeRouteStyle;
        flag&=QName::tokenizer().insert(ns, "ConFixedCode") == NS_v11::LN_ConFixedCode;
        flag&=QName::tokenizer().insert(ns, "ConLineJumpCode") == NS_v11::LN_ConLineJumpCode;
        flag&=QName::tokenizer().insert(ns, "ConLineJumpStyle") == NS_v11::LN_ConLineJumpStyle;
        flag&=QName::tokenizer().insert(ns, "ConLineJumpDirX") == NS_v11::LN_ConLineJumpDirX;
        flag&=QName::tokenizer().insert(ns, "ConLineJumpDirY") == NS_v11::LN_ConLineJumpDirY;
        flag&=QName::tokenizer().insert(ns, "ShapePlaceFlip") == NS_v11::LN_ShapePlaceFlip;
        flag&=QName::tokenizer().insert(ns, "ConLineRouteExt") == NS_v11::LN_ConLineRouteExt;
        flag&=QName::tokenizer().insert(ns, "ShapeSplit") == NS_v11::LN_ShapeSplit;
        flag&=QName::tokenizer().insert(ns, "ShapeSplittable") == NS_v11::LN_ShapeSplittable;
        flag&=QName::tokenizer().insert(ns, "ResizePage") == NS_v11::LN_ResizePage;
        flag&=QName::tokenizer().insert(ns, "EnableGrid") == NS_v11::LN_EnableGrid;
        flag&=QName::tokenizer().insert(ns, "DynamicsOff") == NS_v11::LN_DynamicsOff;
        flag&=QName::tokenizer().insert(ns, "CtrlAsInput") == NS_v11::LN_CtrlAsInput;
        flag&=QName::tokenizer().insert(ns, "PlaceStyle") == NS_v11::LN_PlaceStyle;
        flag&=QName::tokenizer().insert(ns, "RouteStyle") == NS_v11::LN_RouteStyle;
        flag&=QName::tokenizer().insert(ns, "PlaceDepth") == NS_v11::LN_PlaceDepth;
        flag&=QName::tokenizer().insert(ns, "PlowCode") == NS_v11::LN_PlowCode;
        flag&=QName::tokenizer().insert(ns, "LineJumpCode") == NS_v11::LN_LineJumpCode;
        flag&=QName::tokenizer().insert(ns, "LineJumpStyle") == NS_v11::LN_LineJumpStyle;
        flag&=QName::tokenizer().insert(ns, "PageLineJumpDirX") == NS_v11::LN_PageLineJumpDirX;
        flag&=QName::tokenizer().insert(ns, "PageLineJumpDirY") == NS_v11::LN_PageLineJumpDirY;
        flag&=QName::tokenizer().insert(ns, "LineToNodeX") == NS_v11::LN_LineToNodeX;
        flag&=QName::tokenizer().insert(ns, "LineToNodeY") == NS_v11::LN_LineToNodeY;
        flag&=QName::tokenizer().insert(ns, "BlockSizeX") == NS_v11::LN_BlockSizeX;
        flag&=QName::tokenizer().insert(ns, "BlockSizeY") == NS_v11::LN_BlockSizeY;
        flag&=QName::tokenizer().insert(ns, "AvenueSizeX") == NS_v11::LN_AvenueSizeX;
        flag&=QName::tokenizer().insert(ns, "AvenueSizeY") == NS_v11::LN_AvenueSizeY;
        flag&=QName::tokenizer().insert(ns, "LineToLineX") == NS_v11::LN_LineToLineX;
        flag&=QName::tokenizer().insert(ns, "LineToLineY") == NS_v11::LN_LineToLineY;
        flag&=QName::tokenizer().insert(ns, "LineJumpFactorX") == NS_v11::LN_LineJumpFactorX;
        flag&=QName::tokenizer().insert(ns, "LineJumpFactorY") == NS_v11::LN_LineJumpFactorY;
        flag&=QName::tokenizer().insert(ns, "LineAdjustFrom") == NS_v11::LN_LineAdjustFrom;
        flag&=QName::tokenizer().insert(ns, "LineAdjustTo") == NS_v11::LN_LineAdjustTo;
        flag&=QName::tokenizer().insert(ns, "PlaceFlip") == NS_v11::LN_PlaceFlip;
        flag&=QName::tokenizer().insert(ns, "LineRouteExt") == NS_v11::LN_LineRouteExt;
        flag&=QName::tokenizer().insert(ns, "PageShapeSplit") == NS_v11::LN_PageShapeSplit;
        flag&=QName::tokenizer().insert(ns, "PageLeftMargin") == NS_v11::LN_PageLeftMargin;
        flag&=QName::tokenizer().insert(ns, "PageRightMargin") == NS_v11::LN_PageRightMargin;
        flag&=QName::tokenizer().insert(ns, "PageTopMargin") == NS_v11::LN_PageTopMargin;
        flag&=QName::tokenizer().insert(ns, "PageBottomMargin") == NS_v11::LN_PageBottomMargin;
        flag&=QName::tokenizer().insert(ns, "ScaleX") == NS_v11::LN_ScaleX;
        flag&=QName::tokenizer().insert(ns, "ScaleY") == NS_v11::LN_ScaleY;
        flag&=QName::tokenizer().insert(ns, "PagesX") == NS_v11::LN_PagesX;
        flag&=QName::tokenizer().insert(ns, "PagesY") == NS_v11::LN_PagesY;
        flag&=QName::tokenizer().insert(ns, "CenterX") == NS_v11::LN_CenterX;
        flag&=QName::tokenizer().insert(ns, "CenterY") == NS_v11::LN_CenterY;
        flag&=QName::tokenizer().insert(ns, "OnPage") == NS_v11::LN_OnPage;
        flag&=QName::tokenizer().insert(ns, "PrintGrid") == NS_v11::LN_PrintGrid;
        flag&=QName::tokenizer().insert(ns, "PrintPageOrientation") == NS_v11::LN_PrintPageOrientation;
        flag&=QName::tokenizer().insert(ns, "PaperKind") == NS_v11::LN_PaperKind;
        flag&=QName::tokenizer().insert(ns, "PaperSource") == NS_v11::LN_PaperSource;
        flag&=QName::tokenizer().insert(ns, "Font") == NS_v11::LN_Font;
        flag&=QName::tokenizer().insert(ns, "Color") == NS_v11::LN_Color;
        flag&=QName::tokenizer().insert(ns, "Style") == NS_v11::LN_Style;
        flag&=QName::tokenizer().insert(ns, "Case") == NS_v11::LN_Case;
        flag&=QName::tokenizer().insert(ns, "Pos") == NS_v11::LN_Pos;
        flag&=QName::tokenizer().insert(ns, "FontScale") == NS_v11::LN_FontScale;
        flag&=QName::tokenizer().insert(ns, "Locale") == NS_v11::LN_Locale;
        flag&=QName::tokenizer().insert(ns, "Size") == NS_v11::LN_Size;
        flag&=QName::tokenizer().insert(ns, "DblUnderline") == NS_v11::LN_DblUnderline;
        flag&=QName::tokenizer().insert(ns, "Overline") == NS_v11::LN_Overline;
        flag&=QName::tokenizer().insert(ns, "Strikethru") == NS_v11::LN_Strikethru;
        flag&=QName::tokenizer().insert(ns, "Highlight") == NS_v11::LN_Highlight;
        flag&=QName::tokenizer().insert(ns, "Perpendicular") == NS_v11::LN_Perpendicular;
        flag&=QName::tokenizer().insert(ns, "DoubleStrikethrough") == NS_v11::LN_DoubleStrikethrough;
        flag&=QName::tokenizer().insert(ns, "RTLText") == NS_v11::LN_RTLText;
        flag&=QName::tokenizer().insert(ns, "UseVertical") == NS_v11::LN_UseVertical;
        flag&=QName::tokenizer().insert(ns, "Letterspace") == NS_v11::LN_Letterspace;
        flag&=QName::tokenizer().insert(ns, "ColorTrans") == NS_v11::LN_ColorTrans;
        flag&=QName::tokenizer().insert(ns, "AsianFont") == NS_v11::LN_AsianFont;
        flag&=QName::tokenizer().insert(ns, "ComplexScriptFont") == NS_v11::LN_ComplexScriptFont;
        flag&=QName::tokenizer().insert(ns, "LocalizeFont") == NS_v11::LN_LocalizeFont;
        flag&=QName::tokenizer().insert(ns, "ComplexScriptSize") == NS_v11::LN_ComplexScriptSize;
        flag&=QName::tokenizer().insert(ns, "IndFirst") == NS_v11::LN_IndFirst;
        flag&=QName::tokenizer().insert(ns, "IndLeft") == NS_v11::LN_IndLeft;
        flag&=QName::tokenizer().insert(ns, "IndRight") == NS_v11::LN_IndRight;
        flag&=QName::tokenizer().insert(ns, "SpLine") == NS_v11::LN_SpLine;
        flag&=QName::tokenizer().insert(ns, "SpBefore") == NS_v11::LN_SpBefore;
        flag&=QName::tokenizer().insert(ns, "SpAfter") == NS_v11::LN_SpAfter;
        flag&=QName::tokenizer().insert(ns, "HorzAlign") == NS_v11::LN_HorzAlign;
        flag&=QName::tokenizer().insert(ns, "Bullet") == NS_v11::LN_Bullet;
        flag&=QName::tokenizer().insert(ns, "BulletStr") == NS_v11::LN_BulletStr;
        flag&=QName::tokenizer().insert(ns, "BulletFont") == NS_v11::LN_BulletFont;
        flag&=QName::tokenizer().insert(ns, "LocalizeBulletFont") == NS_v11::LN_LocalizeBulletFont;
        flag&=QName::tokenizer().insert(ns, "BulletFontSize") == NS_v11::LN_BulletFontSize;
        flag&=QName::tokenizer().insert(ns, "TextPosAfterBullet") == NS_v11::LN_TextPosAfterBullet;
        flag&=QName::tokenizer().insert(ns, "Flags") == NS_v11::LN_Flags;
        flag&=QName::tokenizer().insert(ns, "Tab") == NS_v11::LN_Tab;
        flag&=QName::tokenizer().insert(ns, "Position") == NS_v11::LN_Position;
        flag&=QName::tokenizer().insert(ns, "Alignment") == NS_v11::LN_Alignment;
        flag&=QName::tokenizer().insert(ns, "Leader") == NS_v11::LN_Leader;
        flag&=QName::tokenizer().insert(ns, "X") == NS_v11::LN_X;
        flag&=QName::tokenizer().insert(ns, "Y") == NS_v11::LN_Y;
        flag&=QName::tokenizer().insert(ns, "A") == NS_v11::LN_A;
        flag&=QName::tokenizer().insert(ns, "B") == NS_v11::LN_B;
        flag&=QName::tokenizer().insert(ns, "C") == NS_v11::LN_C;
        flag&=QName::tokenizer().insert(ns, "D") == NS_v11::LN_D;
        flag&=QName::tokenizer().insert(ns, "DirX") == NS_v11::LN_DirX;
        flag&=QName::tokenizer().insert(ns, "DirY") == NS_v11::LN_DirY;
        flag&=QName::tokenizer().insert(ns, "Type") == NS_v11::LN_Type;
        flag&=QName::tokenizer().insert(ns, "AutoGen") == NS_v11::LN_AutoGen;
        flag&=QName::tokenizer().insert(ns, "Prompt") == NS_v11::LN_Prompt;
        flag&=QName::tokenizer().insert(ns, "Value") == NS_v11::LN_Value;
        flag&=QName::tokenizer().insert(ns, "EditMode") == NS_v11::LN_EditMode;
        flag&=QName::tokenizer().insert(ns, "Format") == NS_v11::LN_Format;
        flag&=QName::tokenizer().insert(ns, "UICat") == NS_v11::LN_UICat;
        flag&=QName::tokenizer().insert(ns, "UICod") == NS_v11::LN_UICod;
        flag&=QName::tokenizer().insert(ns, "UIFmt") == NS_v11::LN_UIFmt;
        flag&=QName::tokenizer().insert(ns, "ObjectKind") == NS_v11::LN_ObjectKind;
        flag&=QName::tokenizer().insert(ns, "XDyn") == NS_v11::LN_XDyn;
        flag&=QName::tokenizer().insert(ns, "YDyn") == NS_v11::LN_YDyn;
        flag&=QName::tokenizer().insert(ns, "XCon") == NS_v11::LN_XCon;
        flag&=QName::tokenizer().insert(ns, "YCon") == NS_v11::LN_YCon;
        flag&=QName::tokenizer().insert(ns, "CanGlue") == NS_v11::LN_CanGlue;
        flag&=QName::tokenizer().insert(ns, "NoFill") == NS_v11::LN_NoFill;
        flag&=QName::tokenizer().insert(ns, "NoLine") == NS_v11::LN_NoLine;
        flag&=QName::tokenizer().insert(ns, "NoShow") == NS_v11::LN_NoShow;
        flag&=QName::tokenizer().insert(ns, "NoSnap") == NS_v11::LN_NoSnap;
        flag&=QName::tokenizer().insert(ns, "MoveTo") == NS_v11::LN_MoveTo;
        flag&=QName::tokenizer().insert(ns, "LineTo") == NS_v11::LN_LineTo;
        flag&=QName::tokenizer().insert(ns, "ArcTo") == NS_v11::LN_ArcTo;
        flag&=QName::tokenizer().insert(ns, "InfiniteLine") == NS_v11::LN_InfiniteLine;
        flag&=QName::tokenizer().insert(ns, "Ellipse") == NS_v11::LN_Ellipse;
        flag&=QName::tokenizer().insert(ns, "EllipticalArcTo") == NS_v11::LN_EllipticalArcTo;
        flag&=QName::tokenizer().insert(ns, "SplineStart") == NS_v11::LN_SplineStart;
        flag&=QName::tokenizer().insert(ns, "SplineKnot") == NS_v11::LN_SplineKnot;
        flag&=QName::tokenizer().insert(ns, "PolylineTo") == NS_v11::LN_PolylineTo;
        flag&=QName::tokenizer().insert(ns, "NURBSTo") == NS_v11::LN_NURBSTo;
        flag&=QName::tokenizer().insert(ns, "E") == NS_v11::LN_E;
        flag&=QName::tokenizer().insert(ns, "Menu") == NS_v11::LN_Menu;
        flag&=QName::tokenizer().insert(ns, "Action") == NS_v11::LN_Action;
        flag&=QName::tokenizer().insert(ns, "Checked") == NS_v11::LN_Checked;
        flag&=QName::tokenizer().insert(ns, "Disabled") == NS_v11::LN_Disabled;
        flag&=QName::tokenizer().insert(ns, "ReadOnly") == NS_v11::LN_ReadOnly;
        flag&=QName::tokenizer().insert(ns, "Invisible") == NS_v11::LN_Invisible;
        flag&=QName::tokenizer().insert(ns, "BeginGroup") == NS_v11::LN_BeginGroup;
        flag&=QName::tokenizer().insert(ns, "TagName") == NS_v11::LN_TagName;
        flag&=QName::tokenizer().insert(ns, "ButtonFace") == NS_v11::LN_ButtonFace;
        flag&=QName::tokenizer().insert(ns, "SortKey") == NS_v11::LN_SortKey;
        flag&=QName::tokenizer().insert(ns, "Name") == NS_v11::LN_Name;
        flag&=QName::tokenizer().insert(ns, "Status") == NS_v11::LN_Status;
        flag&=QName::tokenizer().insert(ns, "Visible") == NS_v11::LN_Visible;
        flag&=QName::tokenizer().insert(ns, "Print") == NS_v11::LN_Print;
        flag&=QName::tokenizer().insert(ns, "Active") == NS_v11::LN_Active;
        flag&=QName::tokenizer().insert(ns, "Lock") == NS_v11::LN_Lock;
        flag&=QName::tokenizer().insert(ns, "Snap") == NS_v11::LN_Snap;
        flag&=QName::tokenizer().insert(ns, "Glue") == NS_v11::LN_Glue;
        flag&=QName::tokenizer().insert(ns, "NameUniv") == NS_v11::LN_NameUniv;
        flag&=QName::tokenizer().insert(ns, "Label") == NS_v11::LN_Label;
        flag&=QName::tokenizer().insert(ns, "Verify") == NS_v11::LN_Verify;
        flag&=QName::tokenizer().insert(ns, "Description") == NS_v11::LN_Description;
        flag&=QName::tokenizer().insert(ns, "Address") == NS_v11::LN_Address;
        flag&=QName::tokenizer().insert(ns, "SubAddress") == NS_v11::LN_SubAddress;
        flag&=QName::tokenizer().insert(ns, "ExtraInfo") == NS_v11::LN_ExtraInfo;
        flag&=QName::tokenizer().insert(ns, "Frame") == NS_v11::LN_Frame;
        flag&=QName::tokenizer().insert(ns, "NewWindow") == NS_v11::LN_NewWindow;
        flag&=QName::tokenizer().insert(ns, "Default") == NS_v11::LN_Default;
        flag&=QName::tokenizer().insert(ns, "Initials") == NS_v11::LN_Initials;
        flag&=QName::tokenizer().insert(ns, "ReviewerID") == NS_v11::LN_ReviewerID;
        flag&=QName::tokenizer().insert(ns, "CurrentIndex") == NS_v11::LN_CurrentIndex;
        flag&=QName::tokenizer().insert(ns, "MarkerIndex") == NS_v11::LN_MarkerIndex;
        flag&=QName::tokenizer().insert(ns, "Date") == NS_v11::LN_Date;
        flag&=QName::tokenizer().insert(ns, "XJustify") == NS_v11::LN_XJustify;
        flag&=QName::tokenizer().insert(ns, "YJustify") == NS_v11::LN_YJustify;
        flag&=QName::tokenizer().insert(ns, "Title") == NS_v11::LN_Title;
        flag&=QName::tokenizer().insert(ns, "Subject") == NS_v11::LN_Subject;
        flag&=QName::tokenizer().insert(ns, "Creator") == NS_v11::LN_Creator;
        flag&=QName::tokenizer().insert(ns, "Manager") == NS_v11::LN_Manager;
        flag&=QName::tokenizer().insert(ns, "Company") == NS_v11::LN_Company;
        flag&=QName::tokenizer().insert(ns, "Category") == NS_v11::LN_Category;
        flag&=QName::tokenizer().insert(ns, "Keywords") == NS_v11::LN_Keywords;
        flag&=QName::tokenizer().insert(ns, "Desc") == NS_v11::LN_Desc;
        flag&=QName::tokenizer().insert(ns, "HyperlinkBase") == NS_v11::LN_HyperlinkBase;
        flag&=QName::tokenizer().insert(ns, "AlternateNames") == NS_v11::LN_AlternateNames;
        flag&=QName::tokenizer().insert(ns, "Template") == NS_v11::LN_Template;
        flag&=QName::tokenizer().insert(ns, "BuildNumberCreated") == NS_v11::LN_BuildNumberCreated;
        flag&=QName::tokenizer().insert(ns, "BuildNumberEdited") == NS_v11::LN_BuildNumberEdited;
        flag&=QName::tokenizer().insert(ns, "PreviewPicture") == NS_v11::LN_PreviewPicture;
        flag&=QName::tokenizer().insert(ns, "CustomProps") == NS_v11::LN_CustomProps;
        flag&=QName::tokenizer().insert(ns, "TimeCreated") == NS_v11::LN_TimeCreated;
        flag&=QName::tokenizer().insert(ns, "TimeSaved") == NS_v11::LN_TimeSaved;
        flag&=QName::tokenizer().insert(ns, "TimeEdited") == NS_v11::LN_TimeEdited;
        flag&=QName::tokenizer().insert(ns, "TimePrinted") == NS_v11::LN_TimePrinted;
        flag&=QName::tokenizer().insert(ns, "CustomProp") == NS_v11::LN_CustomProp;
        flag&=QName::tokenizer().insert(ns, "GlueSettings") == NS_v11::LN_GlueSettings;
        flag&=QName::tokenizer().insert(ns, "SnapSettings") == NS_v11::LN_SnapSettings;
        flag&=QName::tokenizer().insert(ns, "SnapExtensions") == NS_v11::LN_SnapExtensions;
        flag&=QName::tokenizer().insert(ns, "SnapAngles") == NS_v11::LN_SnapAngles;
        flag&=QName::tokenizer().insert(ns, "DynamicGridEnabled") == NS_v11::LN_DynamicGridEnabled;
        flag&=QName::tokenizer().insert(ns, "ProtectStyles") == NS_v11::LN_ProtectStyles;
        flag&=QName::tokenizer().insert(ns, "ProtectShapes") == NS_v11::LN_ProtectShapes;
        flag&=QName::tokenizer().insert(ns, "ProtectMasters") == NS_v11::LN_ProtectMasters;
        flag&=QName::tokenizer().insert(ns, "ProtectBkgnds") == NS_v11::LN_ProtectBkgnds;
        flag&=QName::tokenizer().insert(ns, "CustomMenusFile") == NS_v11::LN_CustomMenusFile;
        flag&=QName::tokenizer().insert(ns, "CustomToolbarsFile") == NS_v11::LN_CustomToolbarsFile;
        flag&=QName::tokenizer().insert(ns, "AttachedToolbars") == NS_v11::LN_AttachedToolbars;
        flag&=QName::tokenizer().insert(ns, "SnapAngle") == NS_v11::LN_SnapAngle;
        flag&=QName::tokenizer().insert(ns, "ColorEntry") == NS_v11::LN_ColorEntry;
        flag&=QName::tokenizer().insert(ns, "PaperSize") == NS_v11::LN_PaperSize;
        flag&=QName::tokenizer().insert(ns, "PaperWidth") == NS_v11::LN_PaperWidth;
        flag&=QName::tokenizer().insert(ns, "PaperHeight") == NS_v11::LN_PaperHeight;
        flag&=QName::tokenizer().insert(ns, "PrintCenteredH") == NS_v11::LN_PrintCenteredH;
        flag&=QName::tokenizer().insert(ns, "PrintCenteredV") == NS_v11::LN_PrintCenteredV;
        flag&=QName::tokenizer().insert(ns, "PrintFitOnPages") == NS_v11::LN_PrintFitOnPages;
        flag&=QName::tokenizer().insert(ns, "PrintLandscape") == NS_v11::LN_PrintLandscape;
        flag&=QName::tokenizer().insert(ns, "PrintPagesAcross") == NS_v11::LN_PrintPagesAcross;
        flag&=QName::tokenizer().insert(ns, "PrintPagesDown") == NS_v11::LN_PrintPagesDown;
        flag&=QName::tokenizer().insert(ns, "PrintScale") == NS_v11::LN_PrintScale;
        flag&=QName::tokenizer().insert(ns, "FontEntry") == NS_v11::LN_FontEntry;
        flag&=QName::tokenizer().insert(ns, "FaceName") == NS_v11::LN_FaceName;
        flag&=QName::tokenizer().insert(ns, "StyleSheet") == NS_v11::LN_StyleSheet;
        flag&=QName::tokenizer().insert(ns, "Master") == NS_v11::LN_Master;
        flag&=QName::tokenizer().insert(ns, "MasterShortcut") == NS_v11::LN_MasterShortcut;
        flag&=QName::tokenizer().insert(ns, "PageSheet") == NS_v11::LN_PageSheet;
        flag&=QName::tokenizer().insert(ns, "Shapes") == NS_v11::LN_Shapes;
        flag&=QName::tokenizer().insert(ns, "Icon") == NS_v11::LN_Icon;
        flag&=QName::tokenizer().insert(ns, "Connects") == NS_v11::LN_Connects;
        flag&=QName::tokenizer().insert(ns, "Shape") == NS_v11::LN_Shape;
        flag&=QName::tokenizer().insert(ns, "Connect") == NS_v11::LN_Connect;
        flag&=QName::tokenizer().insert(ns, "Page") == NS_v11::LN_Page;
        flag&=QName::tokenizer().insert(ns, "Window") == NS_v11::LN_Window;
        flag&=QName::tokenizer().insert(ns, "StencilGroup") == NS_v11::LN_StencilGroup;
        flag&=QName::tokenizer().insert(ns, "StencilGroupPos") == NS_v11::LN_StencilGroupPos;
        flag&=QName::tokenizer().insert(ns, "ShowRulers") == NS_v11::LN_ShowRulers;
        flag&=QName::tokenizer().insert(ns, "ShowGrid") == NS_v11::LN_ShowGrid;
        flag&=QName::tokenizer().insert(ns, "ShowPageBreaks") == NS_v11::LN_ShowPageBreaks;
        flag&=QName::tokenizer().insert(ns, "ShowGuides") == NS_v11::LN_ShowGuides;
        flag&=QName::tokenizer().insert(ns, "ShowConnectionPoints") == NS_v11::LN_ShowConnectionPoints;
        flag&=QName::tokenizer().insert(ns, "TabSplitterPos") == NS_v11::LN_TabSplitterPos;
        flag&=QName::tokenizer().insert(ns, "EventItem") == NS_v11::LN_EventItem;
        flag&=QName::tokenizer().insert(ns, "HeaderMargin") == NS_v11::LN_HeaderMargin;
        flag&=QName::tokenizer().insert(ns, "FooterMargin") == NS_v11::LN_FooterMargin;
        flag&=QName::tokenizer().insert(ns, "HeaderLeft") == NS_v11::LN_HeaderLeft;
        flag&=QName::tokenizer().insert(ns, "HeaderCenter") == NS_v11::LN_HeaderCenter;
        flag&=QName::tokenizer().insert(ns, "HeaderRight") == NS_v11::LN_HeaderRight;
        flag&=QName::tokenizer().insert(ns, "FooterLeft") == NS_v11::LN_FooterLeft;
        flag&=QName::tokenizer().insert(ns, "FooterCenter") == NS_v11::LN_FooterCenter;
        flag&=QName::tokenizer().insert(ns, "FooterRight") == NS_v11::LN_FooterRight;
        flag&=QName::tokenizer().insert(ns, "HeaderFooterFont") == NS_v11::LN_HeaderFooterFont;
        flag&=QName::tokenizer().insert(ns, "DocumentProperties") == NS_v11::LN_DocumentProperties;
        flag&=QName::tokenizer().insert(ns, "DocumentSettings") == NS_v11::LN_DocumentSettings;
        flag&=QName::tokenizer().insert(ns, "Colors") == NS_v11::LN_Colors;
        flag&=QName::tokenizer().insert(ns, "PrintSetup") == NS_v11::LN_PrintSetup;
        flag&=QName::tokenizer().insert(ns, "Fonts") == NS_v11::LN_Fonts;
        flag&=QName::tokenizer().insert(ns, "FaceNames") == NS_v11::LN_FaceNames;
        flag&=QName::tokenizer().insert(ns, "StyleSheets") == NS_v11::LN_StyleSheets;
        flag&=QName::tokenizer().insert(ns, "DocumentSheet") == NS_v11::LN_DocumentSheet;
        flag&=QName::tokenizer().insert(ns, "Masters") == NS_v11::LN_Masters;
        flag&=QName::tokenizer().insert(ns, "Pages") == NS_v11::LN_Pages;
        flag&=QName::tokenizer().insert(ns, "Windows") == NS_v11::LN_Windows;
        flag&=QName::tokenizer().insert(ns, "EventList") == NS_v11::LN_EventList;
        flag&=QName::tokenizer().insert(ns, "HeaderFooter") == NS_v11::LN_HeaderFooter;
        flag&=QName::tokenizer().insert(ns, "VBProjectData") == NS_v11::LN_VBProjectData;
        flag&=QName::tokenizer().insert(ns, "EmailRoutingData") == NS_v11::LN_EmailRoutingData;
        CPPUNIT_ASSERT_MESSAGE("test for namespace v11 failed", flag);}
    void test_sst12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/sst";
        flag&=QName::tokenizer().insert(ns, "Sst") == NS_sst12::LN_Sst;
        CPPUNIT_ASSERT_MESSAGE("test for namespace sst12 failed", flag);}
    void test_swb12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/sharedworkbook";
        flag&=QName::tokenizer().insert(ns, "revisions") == NS_swb12::LN_revisions;
        flag&=QName::tokenizer().insert(ns, "info") == NS_swb12::LN_info;
        flag&=QName::tokenizer().insert(ns, "header") == NS_swb12::LN_header;
        flag&=QName::tokenizer().insert(ns, "rwCol") == NS_swb12::LN_rwCol;
        flag&=QName::tokenizer().insert(ns, "move") == NS_swb12::LN_move;
        flag&=QName::tokenizer().insert(ns, "userView") == NS_swb12::LN_userView;
        flag&=QName::tokenizer().insert(ns, "sheetName") == NS_swb12::LN_sheetName;
        flag&=QName::tokenizer().insert(ns, "sheet") == NS_swb12::LN_sheet;
        flag&=QName::tokenizer().insert(ns, "cell") == NS_swb12::LN_cell;
        flag&=QName::tokenizer().insert(ns, "formatting") == NS_swb12::LN_formatting;
        flag&=QName::tokenizer().insert(ns, "autofmt") == NS_swb12::LN_autofmt;
        flag&=QName::tokenizer().insert(ns, "name") == NS_swb12::LN_name;
        flag&=QName::tokenizer().insert(ns, "comment") == NS_swb12::LN_comment;
        flag&=QName::tokenizer().insert(ns, "qtField") == NS_swb12::LN_qtField;
        flag&=QName::tokenizer().insert(ns, "conflict") == NS_swb12::LN_conflict;
        flag&=QName::tokenizer().insert(ns, "revdata") == NS_swb12::LN_revdata;
        flag&=QName::tokenizer().insert(ns, "tabidMap") == NS_swb12::LN_tabidMap;
        flag&=QName::tokenizer().insert(ns, "tabId") == NS_swb12::LN_tabId;
        flag&=QName::tokenizer().insert(ns, "undoinfo") == NS_swb12::LN_undoinfo;
        flag&=QName::tokenizer().insert(ns, "cellOld") == NS_swb12::LN_cellOld;
        flag&=QName::tokenizer().insert(ns, "cellNew") == NS_swb12::LN_cellNew;
        flag&=QName::tokenizer().insert(ns, "dxfOld") == NS_swb12::LN_dxfOld;
        flag&=QName::tokenizer().insert(ns, "dxfNew") == NS_swb12::LN_dxfNew;
        flag&=QName::tokenizer().insert(ns, "dxfFmt") == NS_swb12::LN_dxfFmt;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_swb12::LN_formula;
        flag&=QName::tokenizer().insert(ns, "formulaOld") == NS_swb12::LN_formulaOld;
        CPPUNIT_ASSERT_MESSAGE("test for namespace swb12 failed", flag);}
    void test_pic12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/officeart/2005/8/picture";
        flag&=QName::tokenizer().insert(ns, "spPr") == NS_pic12::LN_spPr;
        CPPUNIT_ASSERT_MESSAGE("test for namespace pic12 failed", flag);}
    void test_vml(){
        bool flag=true;
        static const char ns[] = "urn:schemas-microsoft-com:vml";
        flag&=QName::tokenizer().insert(ns, "shape") == NS_vml::LN_shape;
        flag&=QName::tokenizer().insert(ns, "group") == NS_vml::LN_group;
        flag&=QName::tokenizer().insert(ns, "shapetype") == NS_vml::LN_shapetype;
        flag&=QName::tokenizer().insert(ns, "VMLFrame") == NS_vml::LN_VMLFrame;
        flag&=QName::tokenizer().insert(ns, "background") == NS_vml::LN_background;
        flag&=QName::tokenizer().insert(ns, "path") == NS_vml::LN_path;
        flag&=QName::tokenizer().insert(ns, "formulas") == NS_vml::LN_formulas;
        flag&=QName::tokenizer().insert(ns, "f") == NS_vml::LN_f;
        flag&=QName::tokenizer().insert(ns, "handles") == NS_vml::LN_handles;
        flag&=QName::tokenizer().insert(ns, "h") == NS_vml::LN_h;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_vml::LN_fill;
        flag&=QName::tokenizer().insert(ns, "stroke") == NS_vml::LN_stroke;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_vml::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "textbox") == NS_vml::LN_textbox;
        flag&=QName::tokenizer().insert(ns, "textpath") == NS_vml::LN_textpath;
        flag&=QName::tokenizer().insert(ns, "imagedata") == NS_vml::LN_imagedata;
        flag&=QName::tokenizer().insert(ns, "line") == NS_vml::LN_line;
        flag&=QName::tokenizer().insert(ns, "Behavior") == NS_vml::LN_Behavior;
        flag&=QName::tokenizer().insert(ns, "Sequence") == NS_vml::LN_Sequence;
        flag&=QName::tokenizer().insert(ns, "Action") == NS_vml::LN_Action;
        flag&=QName::tokenizer().insert(ns, "arc") == NS_vml::LN_arc;
        flag&=QName::tokenizer().insert(ns, "curve") == NS_vml::LN_curve;
        flag&=QName::tokenizer().insert(ns, "polyline") == NS_vml::LN_polyline;
        flag&=QName::tokenizer().insert(ns, "rect") == NS_vml::LN_rect;
        flag&=QName::tokenizer().insert(ns, "roundrect") == NS_vml::LN_roundrect;
        flag&=QName::tokenizer().insert(ns, "oval") == NS_vml::LN_oval;
        flag&=QName::tokenizer().insert(ns, "image") == NS_vml::LN_image;
        CPPUNIT_ASSERT_MESSAGE("test for namespace vml failed", flag);}
    void test_ax12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/activeX/2005/ax";
        flag&=QName::tokenizer().insert(ns, "ocx") == NS_ax12::LN_ocx;
        flag&=QName::tokenizer().insert(ns, "ocxPr") == NS_ax12::LN_ocxPr;
        flag&=QName::tokenizer().insert(ns, "ocxMBed") == NS_ax12::LN_ocxMBed;
        CPPUNIT_ASSERT_MESSAGE("test for namespace ax12 failed", flag);}
    void test_ws12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/worksheet";
        flag&=QName::tokenizer().insert(ns, "workSheet") == NS_ws12::LN_workSheet;
        flag&=QName::tokenizer().insert(ns, "macroSheet") == NS_ws12::LN_macroSheet;
        flag&=QName::tokenizer().insert(ns, "dialogSheet") == NS_ws12::LN_dialogSheet;
        flag&=QName::tokenizer().insert(ns, "chartSheet") == NS_ws12::LN_chartSheet;
        flag&=QName::tokenizer().insert(ns, "wsPr") == NS_ws12::LN_wsPr;
        flag&=QName::tokenizer().insert(ns, "dimension") == NS_ws12::LN_dimension;
        flag&=QName::tokenizer().insert(ns, "wsViews") == NS_ws12::LN_wsViews;
        flag&=QName::tokenizer().insert(ns, "userShViews") == NS_ws12::LN_userShViews;
        flag&=QName::tokenizer().insert(ns, "wsFmtInfo") == NS_ws12::LN_wsFmtInfo;
        flag&=QName::tokenizer().insert(ns, "colInfos") == NS_ws12::LN_colInfos;
        flag&=QName::tokenizer().insert(ns, "sheetData") == NS_ws12::LN_sheetData;
        flag&=QName::tokenizer().insert(ns, "calcChains") == NS_ws12::LN_calcChains;
        flag&=QName::tokenizer().insert(ns, "autofilter") == NS_ws12::LN_autofilter;
        flag&=QName::tokenizer().insert(ns, "sortState") == NS_ws12::LN_sortState;
        flag&=QName::tokenizer().insert(ns, "rwBrks") == NS_ws12::LN_rwBrks;
        flag&=QName::tokenizer().insert(ns, "colBrks") == NS_ws12::LN_colBrks;
        flag&=QName::tokenizer().insert(ns, "pageMargins") == NS_ws12::LN_pageMargins;
        flag&=QName::tokenizer().insert(ns, "printOptions") == NS_ws12::LN_printOptions;
        flag&=QName::tokenizer().insert(ns, "pageSetup") == NS_ws12::LN_pageSetup;
        flag&=QName::tokenizer().insert(ns, "headerFooter") == NS_ws12::LN_headerFooter;
        flag&=QName::tokenizer().insert(ns, "drawing") == NS_ws12::LN_drawing;
        flag&=QName::tokenizer().insert(ns, "fsb") == NS_ws12::LN_fsb;
        flag&=QName::tokenizer().insert(ns, "phonetic") == NS_ws12::LN_phonetic;
        flag&=QName::tokenizer().insert(ns, "dataConsolidation") == NS_ws12::LN_dataConsolidation;
        flag&=QName::tokenizer().insert(ns, "mergeCells") == NS_ws12::LN_mergeCells;
        flag&=QName::tokenizer().insert(ns, "dVals") == NS_ws12::LN_dVals;
        flag&=QName::tokenizer().insert(ns, "smartTags") == NS_ws12::LN_smartTags;
        flag&=QName::tokenizer().insert(ns, "hyperlinks") == NS_ws12::LN_hyperlinks;
        flag&=QName::tokenizer().insert(ns, "conditionalFormatting") == NS_ws12::LN_conditionalFormatting;
        flag&=QName::tokenizer().insert(ns, "legacyDrawing") == NS_ws12::LN_legacyDrawing;
        flag&=QName::tokenizer().insert(ns, "scenarios") == NS_ws12::LN_scenarios;
        flag&=QName::tokenizer().insert(ns, "cellWatches") == NS_ws12::LN_cellWatches;
        flag&=QName::tokenizer().insert(ns, "sheetProtection") == NS_ws12::LN_sheetProtection;
        flag&=QName::tokenizer().insert(ns, "protectedRanges") == NS_ws12::LN_protectedRanges;
        flag&=QName::tokenizer().insert(ns, "picture") == NS_ws12::LN_picture;
        flag&=QName::tokenizer().insert(ns, "webPubItems") == NS_ws12::LN_webPubItems;
        flag&=QName::tokenizer().insert(ns, "row") == NS_ws12::LN_row;
        flag&=QName::tokenizer().insert(ns, "chain") == NS_ws12::LN_chain;
        flag&=QName::tokenizer().insert(ns, "c") == NS_ws12::LN_c;
        flag&=QName::tokenizer().insert(ns, "colInfo") == NS_ws12::LN_colInfo;
        flag&=QName::tokenizer().insert(ns, "f") == NS_ws12::LN_f;
        flag&=QName::tokenizer().insert(ns, "v") == NS_ws12::LN_v;
        flag&=QName::tokenizer().insert(ns, "rst") == NS_ws12::LN_rst;
        flag&=QName::tokenizer().insert(ns, "tabColor") == NS_ws12::LN_tabColor;
        flag&=QName::tokenizer().insert(ns, "wsView") == NS_ws12::LN_wsView;
        flag&=QName::tokenizer().insert(ns, "pane") == NS_ws12::LN_pane;
        flag&=QName::tokenizer().insert(ns, "selection") == NS_ws12::LN_selection;
        flag&=QName::tokenizer().insert(ns, "brk") == NS_ws12::LN_brk;
        flag&=QName::tokenizer().insert(ns, "afdd") == NS_ws12::LN_afdd;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_ws12::LN_filter;
        flag&=QName::tokenizer().insert(ns, "DateGroupItem") == NS_ws12::LN_DateGroupItem;
        flag&=QName::tokenizer().insert(ns, "criteria") == NS_ws12::LN_criteria;
        flag&=QName::tokenizer().insert(ns, "criteriaColor") == NS_ws12::LN_criteriaColor;
        flag&=QName::tokenizer().insert(ns, "criteriaCellIcon") == NS_ws12::LN_criteriaCellIcon;
        flag&=QName::tokenizer().insert(ns, "dataRefs") == NS_ws12::LN_dataRefs;
        flag&=QName::tokenizer().insert(ns, "dataRef") == NS_ws12::LN_dataRef;
        flag&=QName::tokenizer().insert(ns, "mergeCell") == NS_ws12::LN_mergeCell;
        flag&=QName::tokenizer().insert(ns, "cellSmartTags") == NS_ws12::LN_cellSmartTags;
        flag&=QName::tokenizer().insert(ns, "cellSmartTag") == NS_ws12::LN_cellSmartTag;
        flag&=QName::tokenizer().insert(ns, "cellSmartTagPr") == NS_ws12::LN_cellSmartTagPr;
        flag&=QName::tokenizer().insert(ns, "userShView") == NS_ws12::LN_userShView;
        flag&=QName::tokenizer().insert(ns, "dVal") == NS_ws12::LN_dVal;
        flag&=QName::tokenizer().insert(ns, "formula1") == NS_ws12::LN_formula1;
        flag&=QName::tokenizer().insert(ns, "formula2") == NS_ws12::LN_formula2;
        flag&=QName::tokenizer().insert(ns, "cfRule") == NS_ws12::LN_cfRule;
        flag&=QName::tokenizer().insert(ns, "formula") == NS_ws12::LN_formula;
        flag&=QName::tokenizer().insert(ns, "colorScale") == NS_ws12::LN_colorScale;
        flag&=QName::tokenizer().insert(ns, "databar") == NS_ws12::LN_databar;
        flag&=QName::tokenizer().insert(ns, "iconset") == NS_ws12::LN_iconset;
        flag&=QName::tokenizer().insert(ns, "hyperlink") == NS_ws12::LN_hyperlink;
        flag&=QName::tokenizer().insert(ns, "cfvo") == NS_ws12::LN_cfvo;
        flag&=QName::tokenizer().insert(ns, "color") == NS_ws12::LN_color;
        flag&=QName::tokenizer().insert(ns, "oddHeader") == NS_ws12::LN_oddHeader;
        flag&=QName::tokenizer().insert(ns, "oddFooter") == NS_ws12::LN_oddFooter;
        flag&=QName::tokenizer().insert(ns, "evenHeader") == NS_ws12::LN_evenHeader;
        flag&=QName::tokenizer().insert(ns, "evenFooter") == NS_ws12::LN_evenFooter;
        flag&=QName::tokenizer().insert(ns, "firstHeader") == NS_ws12::LN_firstHeader;
        flag&=QName::tokenizer().insert(ns, "firstFooter") == NS_ws12::LN_firstFooter;
        flag&=QName::tokenizer().insert(ns, "sortCondition") == NS_ws12::LN_sortCondition;
        flag&=QName::tokenizer().insert(ns, "sct") == NS_ws12::LN_sct;
        flag&=QName::tokenizer().insert(ns, "rangeProtection") == NS_ws12::LN_rangeProtection;
        flag&=QName::tokenizer().insert(ns, "slc") == NS_ws12::LN_slc;
        flag&=QName::tokenizer().insert(ns, "cellWatch") == NS_ws12::LN_cellWatch;
        flag&=QName::tokenizer().insert(ns, "t") == NS_ws12::LN_t;
        flag&=QName::tokenizer().insert(ns, "r") == NS_ws12::LN_r;
        flag&=QName::tokenizer().insert(ns, "rt") == NS_ws12::LN_rt;
        flag&=QName::tokenizer().insert(ns, "rubyBase") == NS_ws12::LN_rubyBase;
        flag&=QName::tokenizer().insert(ns, "rPr") == NS_ws12::LN_rPr;
        flag&=QName::tokenizer().insert(ns, "font") == NS_ws12::LN_font;
        flag&=QName::tokenizer().insert(ns, "charset") == NS_ws12::LN_charset;
        flag&=QName::tokenizer().insert(ns, "family") == NS_ws12::LN_family;
        flag&=QName::tokenizer().insert(ns, "b") == NS_ws12::LN_b;
        flag&=QName::tokenizer().insert(ns, "i") == NS_ws12::LN_i;
        flag&=QName::tokenizer().insert(ns, "strike") == NS_ws12::LN_strike;
        flag&=QName::tokenizer().insert(ns, "outline") == NS_ws12::LN_outline;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_ws12::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "condense") == NS_ws12::LN_condense;
        flag&=QName::tokenizer().insert(ns, "extend") == NS_ws12::LN_extend;
        flag&=QName::tokenizer().insert(ns, "sz") == NS_ws12::LN_sz;
        flag&=QName::tokenizer().insert(ns, "u") == NS_ws12::LN_u;
        flag&=QName::tokenizer().insert(ns, "vertAlign") == NS_ws12::LN_vertAlign;
        flag&=QName::tokenizer().insert(ns, "ruby") == NS_ws12::LN_ruby;
        flag&=QName::tokenizer().insert(ns, "sstItem") == NS_ws12::LN_sstItem;
        flag&=QName::tokenizer().insert(ns, "webPubItem") == NS_ws12::LN_webPubItem;
        flag&=QName::tokenizer().insert(ns, "csViews") == NS_ws12::LN_csViews;
        flag&=QName::tokenizer().insert(ns, "csView") == NS_ws12::LN_csView;
        CPPUNIT_ASSERT_MESSAGE("test for namespace ws12 failed", flag);}
    void test_wddr12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/officeart/2005/8/wddr";
        flag&=QName::tokenizer().insert(ns, "extent") == NS_wddr12::LN_extent;
        flag&=QName::tokenizer().insert(ns, "effectExtent") == NS_wddr12::LN_effectExtent;
        flag&=QName::tokenizer().insert(ns, "docPr") == NS_wddr12::LN_docPr;
        flag&=QName::tokenizer().insert(ns, "graphic") == NS_wddr12::LN_graphic;
        flag&=QName::tokenizer().insert(ns, "start") == NS_wddr12::LN_start;
        flag&=QName::tokenizer().insert(ns, "lineTo") == NS_wddr12::LN_lineTo;
        flag&=QName::tokenizer().insert(ns, "wrapPolygon") == NS_wddr12::LN_wrapPolygon;
        flag&=QName::tokenizer().insert(ns, "wrapNone") == NS_wddr12::LN_wrapNone;
        flag&=QName::tokenizer().insert(ns, "wrapSquare") == NS_wddr12::LN_wrapSquare;
        flag&=QName::tokenizer().insert(ns, "wrapTight") == NS_wddr12::LN_wrapTight;
        flag&=QName::tokenizer().insert(ns, "wrapThrough") == NS_wddr12::LN_wrapThrough;
        flag&=QName::tokenizer().insert(ns, "wrapTopAndBottom") == NS_wddr12::LN_wrapTopAndBottom;
        flag&=QName::tokenizer().insert(ns, "align") == NS_wddr12::LN_align;
        flag&=QName::tokenizer().insert(ns, "offset") == NS_wddr12::LN_offset;
        flag&=QName::tokenizer().insert(ns, "simplePos") == NS_wddr12::LN_simplePos;
        flag&=QName::tokenizer().insert(ns, "positionH") == NS_wddr12::LN_positionH;
        flag&=QName::tokenizer().insert(ns, "positionV") == NS_wddr12::LN_positionV;
        CPPUNIT_ASSERT_MESSAGE("test for namespace wddr12 failed", flag);}
    void test_a12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/officeart/2005/8/oartml";
        flag&=QName::tokenizer().insert(ns, "spcPct") == NS_a12::LN_spcPct;
        flag&=QName::tokenizer().insert(ns, "spcPts") == NS_a12::LN_spcPts;
        flag&=QName::tokenizer().insert(ns, "tab") == NS_a12::LN_tab;
        flag&=QName::tokenizer().insert(ns, "lineSpc") == NS_a12::LN_lineSpc;
        flag&=QName::tokenizer().insert(ns, "spcBef") == NS_a12::LN_spcBef;
        flag&=QName::tokenizer().insert(ns, "spcAft") == NS_a12::LN_spcAft;
        flag&=QName::tokenizer().insert(ns, "tabLst") == NS_a12::LN_tabLst;
        flag&=QName::tokenizer().insert(ns, "defRPr") == NS_a12::LN_defRPr;
        flag&=QName::tokenizer().insert(ns, "br") == NS_a12::LN_br;
        flag&=QName::tokenizer().insert(ns, "field") == NS_a12::LN_field;
        flag&=QName::tokenizer().insert(ns, "math") == NS_a12::LN_math;
        flag&=QName::tokenizer().insert(ns, "t") == NS_a12::LN_t;
        flag&=QName::tokenizer().insert(ns, "st") == NS_a12::LN_st;
        flag&=QName::tokenizer().insert(ns, "end") == NS_a12::LN_end;
        flag&=QName::tokenizer().insert(ns, "audioCd") == NS_a12::LN_audioCd;
        flag&=QName::tokenizer().insert(ns, "audioFile") == NS_a12::LN_audioFile;
        flag&=QName::tokenizer().insert(ns, "videoFile") == NS_a12::LN_videoFile;
        flag&=QName::tokenizer().insert(ns, "dark1") == NS_a12::LN_dark1;
        flag&=QName::tokenizer().insert(ns, "light1") == NS_a12::LN_light1;
        flag&=QName::tokenizer().insert(ns, "dark2") == NS_a12::LN_dark2;
        flag&=QName::tokenizer().insert(ns, "light2") == NS_a12::LN_light2;
        flag&=QName::tokenizer().insert(ns, "accent1") == NS_a12::LN_accent1;
        flag&=QName::tokenizer().insert(ns, "accent2") == NS_a12::LN_accent2;
        flag&=QName::tokenizer().insert(ns, "accent3") == NS_a12::LN_accent3;
        flag&=QName::tokenizer().insert(ns, "accent4") == NS_a12::LN_accent4;
        flag&=QName::tokenizer().insert(ns, "accent5") == NS_a12::LN_accent5;
        flag&=QName::tokenizer().insert(ns, "accent6") == NS_a12::LN_accent6;
        flag&=QName::tokenizer().insert(ns, "hlink") == NS_a12::LN_hlink;
        flag&=QName::tokenizer().insert(ns, "folHlink") == NS_a12::LN_folHlink;
        flag&=QName::tokenizer().insert(ns, "custClr") == NS_a12::LN_custClr;
        flag&=QName::tokenizer().insert(ns, "latin") == NS_a12::LN_latin;
        flag&=QName::tokenizer().insert(ns, "highAnsi") == NS_a12::LN_highAnsi;
        flag&=QName::tokenizer().insert(ns, "complexScript") == NS_a12::LN_complexScript;
        flag&=QName::tokenizer().insert(ns, "eastAsian") == NS_a12::LN_eastAsian;
        flag&=QName::tokenizer().insert(ns, "majorFont") == NS_a12::LN_majorFont;
        flag&=QName::tokenizer().insert(ns, "minorFont") == NS_a12::LN_minorFont;
        flag&=QName::tokenizer().insert(ns, "fillStyle") == NS_a12::LN_fillStyle;
        flag&=QName::tokenizer().insert(ns, "line") == NS_a12::LN_line;
        flag&=QName::tokenizer().insert(ns, "lineStyle") == NS_a12::LN_lineStyle;
        flag&=QName::tokenizer().insert(ns, "scene3d") == NS_a12::LN_scene3d;
        flag&=QName::tokenizer().insert(ns, "sp3d") == NS_a12::LN_sp3d;
        flag&=QName::tokenizer().insert(ns, "effectStyle") == NS_a12::LN_effectStyle;
        flag&=QName::tokenizer().insert(ns, "bgFillStyle") == NS_a12::LN_bgFillStyle;
        flag&=QName::tokenizer().insert(ns, "fillStyleLst") == NS_a12::LN_fillStyleLst;
        flag&=QName::tokenizer().insert(ns, "lineStyleLst") == NS_a12::LN_lineStyleLst;
        flag&=QName::tokenizer().insert(ns, "effectStyleLst") == NS_a12::LN_effectStyleLst;
        flag&=QName::tokenizer().insert(ns, "bgFillStyleLst") == NS_a12::LN_bgFillStyleLst;
        flag&=QName::tokenizer().insert(ns, "clrScheme") == NS_a12::LN_clrScheme;
        flag&=QName::tokenizer().insert(ns, "custClrLst") == NS_a12::LN_custClrLst;
        flag&=QName::tokenizer().insert(ns, "fontScheme") == NS_a12::LN_fontScheme;
        flag&=QName::tokenizer().insert(ns, "effectScheme") == NS_a12::LN_effectScheme;
        flag&=QName::tokenizer().insert(ns, "sx") == NS_a12::LN_sx;
        flag&=QName::tokenizer().insert(ns, "sy") == NS_a12::LN_sy;
        flag&=QName::tokenizer().insert(ns, "off") == NS_a12::LN_off;
        flag&=QName::tokenizer().insert(ns, "ext") == NS_a12::LN_ext;
        flag&=QName::tokenizer().insert(ns, "chOff") == NS_a12::LN_chOff;
        flag&=QName::tokenizer().insert(ns, "chExt") == NS_a12::LN_chExt;
        flag&=QName::tokenizer().insert(ns, "tint") == NS_a12::LN_tint;
        flag&=QName::tokenizer().insert(ns, "shade") == NS_a12::LN_shade;
        flag&=QName::tokenizer().insert(ns, "comp") == NS_a12::LN_comp;
        flag&=QName::tokenizer().insert(ns, "inv") == NS_a12::LN_inv;
        flag&=QName::tokenizer().insert(ns, "gray") == NS_a12::LN_gray;
        flag&=QName::tokenizer().insert(ns, "alpha") == NS_a12::LN_alpha;
        flag&=QName::tokenizer().insert(ns, "alphaOff") == NS_a12::LN_alphaOff;
        flag&=QName::tokenizer().insert(ns, "alphaMod") == NS_a12::LN_alphaMod;
        flag&=QName::tokenizer().insert(ns, "hue") == NS_a12::LN_hue;
        flag&=QName::tokenizer().insert(ns, "hueOff") == NS_a12::LN_hueOff;
        flag&=QName::tokenizer().insert(ns, "hueMod") == NS_a12::LN_hueMod;
        flag&=QName::tokenizer().insert(ns, "sat") == NS_a12::LN_sat;
        flag&=QName::tokenizer().insert(ns, "satOff") == NS_a12::LN_satOff;
        flag&=QName::tokenizer().insert(ns, "satMod") == NS_a12::LN_satMod;
        flag&=QName::tokenizer().insert(ns, "lum") == NS_a12::LN_lum;
        flag&=QName::tokenizer().insert(ns, "lumOff") == NS_a12::LN_lumOff;
        flag&=QName::tokenizer().insert(ns, "lumMod") == NS_a12::LN_lumMod;
        flag&=QName::tokenizer().insert(ns, "red") == NS_a12::LN_red;
        flag&=QName::tokenizer().insert(ns, "redOff") == NS_a12::LN_redOff;
        flag&=QName::tokenizer().insert(ns, "redMod") == NS_a12::LN_redMod;
        flag&=QName::tokenizer().insert(ns, "green") == NS_a12::LN_green;
        flag&=QName::tokenizer().insert(ns, "greenOff") == NS_a12::LN_greenOff;
        flag&=QName::tokenizer().insert(ns, "greenMod") == NS_a12::LN_greenMod;
        flag&=QName::tokenizer().insert(ns, "blue") == NS_a12::LN_blue;
        flag&=QName::tokenizer().insert(ns, "blueOff") == NS_a12::LN_blueOff;
        flag&=QName::tokenizer().insert(ns, "blueMod") == NS_a12::LN_blueMod;
        flag&=QName::tokenizer().insert(ns, "scrgbClr") == NS_a12::LN_scrgbClr;
        flag&=QName::tokenizer().insert(ns, "srgbClr") == NS_a12::LN_srgbClr;
        flag&=QName::tokenizer().insert(ns, "hslClr") == NS_a12::LN_hslClr;
        flag&=QName::tokenizer().insert(ns, "sysClr") == NS_a12::LN_sysClr;
        flag&=QName::tokenizer().insert(ns, "schemeClr") == NS_a12::LN_schemeClr;
        flag&=QName::tokenizer().insert(ns, "prstClr") == NS_a12::LN_prstClr;
        flag&=QName::tokenizer().insert(ns, "spLock") == NS_a12::LN_spLock;
        flag&=QName::tokenizer().insert(ns, "cxnSpLock") == NS_a12::LN_cxnSpLock;
        flag&=QName::tokenizer().insert(ns, "stCxn") == NS_a12::LN_stCxn;
        flag&=QName::tokenizer().insert(ns, "endCxn") == NS_a12::LN_endCxn;
        flag&=QName::tokenizer().insert(ns, "gspLock") == NS_a12::LN_gspLock;
        flag&=QName::tokenizer().insert(ns, "graphicFrameLock") == NS_a12::LN_graphicFrameLock;
        flag&=QName::tokenizer().insert(ns, "bounds") == NS_a12::LN_bounds;
        flag&=QName::tokenizer().insert(ns, "graphicData") == NS_a12::LN_graphicData;
        flag&=QName::tokenizer().insert(ns, "graphic") == NS_a12::LN_graphic;
        flag&=QName::tokenizer().insert(ns, "igx") == NS_a12::LN_igx;
        flag&=QName::tokenizer().insert(ns, "chart") == NS_a12::LN_chart;
        flag&=QName::tokenizer().insert(ns, "bldIgx") == NS_a12::LN_bldIgx;
        flag&=QName::tokenizer().insert(ns, "bldChart") == NS_a12::LN_bldChart;
        flag&=QName::tokenizer().insert(ns, "txBody") == NS_a12::LN_txBody;
        flag&=QName::tokenizer().insert(ns, "xfrm") == NS_a12::LN_xfrm;
        flag&=QName::tokenizer().insert(ns, "useSpRect") == NS_a12::LN_useSpRect;
        flag&=QName::tokenizer().insert(ns, "anvPr") == NS_a12::LN_anvPr;
        flag&=QName::tokenizer().insert(ns, "anvSpPr") == NS_a12::LN_anvSpPr;
        flag&=QName::tokenizer().insert(ns, "nvSpPr") == NS_a12::LN_nvSpPr;
        flag&=QName::tokenizer().insert(ns, "spPr") == NS_a12::LN_spPr;
        flag&=QName::tokenizer().insert(ns, "txsp") == NS_a12::LN_txsp;
        flag&=QName::tokenizer().insert(ns, "style") == NS_a12::LN_style;
        flag&=QName::tokenizer().insert(ns, "anvCxnSpPr") == NS_a12::LN_anvCxnSpPr;
        flag&=QName::tokenizer().insert(ns, "nvCxnSpPr") == NS_a12::LN_nvCxnSpPr;
        flag&=QName::tokenizer().insert(ns, "anvGraphicFramePr") == NS_a12::LN_anvGraphicFramePr;
        flag&=QName::tokenizer().insert(ns, "nvGraphicFramePr") == NS_a12::LN_nvGraphicFramePr;
        flag&=QName::tokenizer().insert(ns, "anvGspPr") == NS_a12::LN_anvGspPr;
        flag&=QName::tokenizer().insert(ns, "nvGspPr") == NS_a12::LN_nvGspPr;
        flag&=QName::tokenizer().insert(ns, "gspPr") == NS_a12::LN_gspPr;
        flag&=QName::tokenizer().insert(ns, "sp") == NS_a12::LN_sp;
        flag&=QName::tokenizer().insert(ns, "cxnsp") == NS_a12::LN_cxnsp;
        flag&=QName::tokenizer().insert(ns, "graphicFrame") == NS_a12::LN_graphicFrame;
        flag&=QName::tokenizer().insert(ns, "gsp") == NS_a12::LN_gsp;
        flag&=QName::tokenizer().insert(ns, "spid") == NS_a12::LN_spid;
        flag&=QName::tokenizer().insert(ns, "name") == NS_a12::LN_name;
        flag&=QName::tokenizer().insert(ns, "descr") == NS_a12::LN_descr;
        flag&=QName::tokenizer().insert(ns, "rot") == NS_a12::LN_rot;
        flag&=QName::tokenizer().insert(ns, "camera") == NS_a12::LN_camera;
        flag&=QName::tokenizer().insert(ns, "lightRig") == NS_a12::LN_lightRig;
        flag&=QName::tokenizer().insert(ns, "backdrop") == NS_a12::LN_backdrop;
        flag&=QName::tokenizer().insert(ns, "anchor") == NS_a12::LN_anchor;
        flag&=QName::tokenizer().insert(ns, "normal") == NS_a12::LN_normal;
        flag&=QName::tokenizer().insert(ns, "up") == NS_a12::LN_up;
        flag&=QName::tokenizer().insert(ns, "bevelT") == NS_a12::LN_bevelT;
        flag&=QName::tokenizer().insert(ns, "bevelB") == NS_a12::LN_bevelB;
        flag&=QName::tokenizer().insert(ns, "extrusionClr") == NS_a12::LN_extrusionClr;
        flag&=QName::tokenizer().insert(ns, "contourClr") == NS_a12::LN_contourClr;
        flag&=QName::tokenizer().insert(ns, "flatTx") == NS_a12::LN_flatTx;
        flag&=QName::tokenizer().insert(ns, "clrFrom") == NS_a12::LN_clrFrom;
        flag&=QName::tokenizer().insert(ns, "clrTo") == NS_a12::LN_clrTo;
        flag&=QName::tokenizer().insert(ns, "fillToRect") == NS_a12::LN_fillToRect;
        flag&=QName::tokenizer().insert(ns, "gs") == NS_a12::LN_gs;
        flag&=QName::tokenizer().insert(ns, "gsList") == NS_a12::LN_gsList;
        flag&=QName::tokenizer().insert(ns, "lin") == NS_a12::LN_lin;
        flag&=QName::tokenizer().insert(ns, "path") == NS_a12::LN_path;
        flag&=QName::tokenizer().insert(ns, "tileRect") == NS_a12::LN_tileRect;
        flag&=QName::tokenizer().insert(ns, "fillRect") == NS_a12::LN_fillRect;
        flag&=QName::tokenizer().insert(ns, "alphaBiLevel") == NS_a12::LN_alphaBiLevel;
        flag&=QName::tokenizer().insert(ns, "alphaCeiling") == NS_a12::LN_alphaCeiling;
        flag&=QName::tokenizer().insert(ns, "alphaFloor") == NS_a12::LN_alphaFloor;
        flag&=QName::tokenizer().insert(ns, "alphaInv") == NS_a12::LN_alphaInv;
        flag&=QName::tokenizer().insert(ns, "alphaModFix") == NS_a12::LN_alphaModFix;
        flag&=QName::tokenizer().insert(ns, "alphaRepl") == NS_a12::LN_alphaRepl;
        flag&=QName::tokenizer().insert(ns, "biLevel") == NS_a12::LN_biLevel;
        flag&=QName::tokenizer().insert(ns, "blur") == NS_a12::LN_blur;
        flag&=QName::tokenizer().insert(ns, "clrBal") == NS_a12::LN_clrBal;
        flag&=QName::tokenizer().insert(ns, "clrChange") == NS_a12::LN_clrChange;
        flag&=QName::tokenizer().insert(ns, "clrRepl") == NS_a12::LN_clrRepl;
        flag&=QName::tokenizer().insert(ns, "duotone") == NS_a12::LN_duotone;
        flag&=QName::tokenizer().insert(ns, "fillOverlay") == NS_a12::LN_fillOverlay;
        flag&=QName::tokenizer().insert(ns, "grayscl") == NS_a12::LN_grayscl;
        flag&=QName::tokenizer().insert(ns, "hsl") == NS_a12::LN_hsl;
        flag&=QName::tokenizer().insert(ns, "luminance") == NS_a12::LN_luminance;
        flag&=QName::tokenizer().insert(ns, "sharpen") == NS_a12::LN_sharpen;
        flag&=QName::tokenizer().insert(ns, "blip") == NS_a12::LN_blip;
        flag&=QName::tokenizer().insert(ns, "srcRect") == NS_a12::LN_srcRect;
        flag&=QName::tokenizer().insert(ns, "tile") == NS_a12::LN_tile;
        flag&=QName::tokenizer().insert(ns, "stretch") == NS_a12::LN_stretch;
        flag&=QName::tokenizer().insert(ns, "fgClr") == NS_a12::LN_fgClr;
        flag&=QName::tokenizer().insert(ns, "bgClr") == NS_a12::LN_bgClr;
        flag&=QName::tokenizer().insert(ns, "noFill") == NS_a12::LN_noFill;
        flag&=QName::tokenizer().insert(ns, "solidFill") == NS_a12::LN_solidFill;
        flag&=QName::tokenizer().insert(ns, "gradFill") == NS_a12::LN_gradFill;
        flag&=QName::tokenizer().insert(ns, "blipFill") == NS_a12::LN_blipFill;
        flag&=QName::tokenizer().insert(ns, "pattFill") == NS_a12::LN_pattFill;
        flag&=QName::tokenizer().insert(ns, "grpFill") == NS_a12::LN_grpFill;
        flag&=QName::tokenizer().insert(ns, "cont") == NS_a12::LN_cont;
        flag&=QName::tokenizer().insert(ns, "effect") == NS_a12::LN_effect;
        flag&=QName::tokenizer().insert(ns, "alphaOutset") == NS_a12::LN_alphaOutset;
        flag&=QName::tokenizer().insert(ns, "blend") == NS_a12::LN_blend;
        flag&=QName::tokenizer().insert(ns, "fill") == NS_a12::LN_fill;
        flag&=QName::tokenizer().insert(ns, "glow") == NS_a12::LN_glow;
        flag&=QName::tokenizer().insert(ns, "innerShdw") == NS_a12::LN_innerShdw;
        flag&=QName::tokenizer().insert(ns, "outerShdw") == NS_a12::LN_outerShdw;
        flag&=QName::tokenizer().insert(ns, "prstShdw") == NS_a12::LN_prstShdw;
        flag&=QName::tokenizer().insert(ns, "reflection") == NS_a12::LN_reflection;
        flag&=QName::tokenizer().insert(ns, "relOff") == NS_a12::LN_relOff;
        flag&=QName::tokenizer().insert(ns, "sftEdg") == NS_a12::LN_sftEdg;
        flag&=QName::tokenizer().insert(ns, "effectLst") == NS_a12::LN_effectLst;
        flag&=QName::tokenizer().insert(ns, "effectDag") == NS_a12::LN_effectDag;
        flag&=QName::tokenizer().insert(ns, "gd") == NS_a12::LN_gd;
        flag&=QName::tokenizer().insert(ns, "pos") == NS_a12::LN_pos;
        flag&=QName::tokenizer().insert(ns, "ahXY") == NS_a12::LN_ahXY;
        flag&=QName::tokenizer().insert(ns, "ahPolar") == NS_a12::LN_ahPolar;
        flag&=QName::tokenizer().insert(ns, "cxn") == NS_a12::LN_cxn;
        flag&=QName::tokenizer().insert(ns, "rect") == NS_a12::LN_rect;
        flag&=QName::tokenizer().insert(ns, "pt") == NS_a12::LN_pt;
        flag&=QName::tokenizer().insert(ns, "close") == NS_a12::LN_close;
        flag&=QName::tokenizer().insert(ns, "moveTo") == NS_a12::LN_moveTo;
        flag&=QName::tokenizer().insert(ns, "lineTo") == NS_a12::LN_lineTo;
        flag&=QName::tokenizer().insert(ns, "arcTo") == NS_a12::LN_arcTo;
        flag&=QName::tokenizer().insert(ns, "quadBezTo") == NS_a12::LN_quadBezTo;
        flag&=QName::tokenizer().insert(ns, "cubicBezTo") == NS_a12::LN_cubicBezTo;
        flag&=QName::tokenizer().insert(ns, "avLst") == NS_a12::LN_avLst;
        flag&=QName::tokenizer().insert(ns, "gdLst") == NS_a12::LN_gdLst;
        flag&=QName::tokenizer().insert(ns, "ahLst") == NS_a12::LN_ahLst;
        flag&=QName::tokenizer().insert(ns, "cxnLst") == NS_a12::LN_cxnLst;
        flag&=QName::tokenizer().insert(ns, "rectLst") == NS_a12::LN_rectLst;
        flag&=QName::tokenizer().insert(ns, "pathLst") == NS_a12::LN_pathLst;
        flag&=QName::tokenizer().insert(ns, "custGeom") == NS_a12::LN_custGeom;
        flag&=QName::tokenizer().insert(ns, "prstGeom") == NS_a12::LN_prstGeom;
        flag&=QName::tokenizer().insert(ns, "round") == NS_a12::LN_round;
        flag&=QName::tokenizer().insert(ns, "bevel") == NS_a12::LN_bevel;
        flag&=QName::tokenizer().insert(ns, "miter") == NS_a12::LN_miter;
        flag&=QName::tokenizer().insert(ns, "prstDash") == NS_a12::LN_prstDash;
        flag&=QName::tokenizer().insert(ns, "custDash") == NS_a12::LN_custDash;
        flag&=QName::tokenizer().insert(ns, "headEnd") == NS_a12::LN_headEnd;
        flag&=QName::tokenizer().insert(ns, "tailEnd") == NS_a12::LN_tailEnd;
        flag&=QName::tokenizer().insert(ns, "ds") == NS_a12::LN_ds;
        flag&=QName::tokenizer().insert(ns, "noLine") == NS_a12::LN_noLine;
        flag&=QName::tokenizer().insert(ns, "solidLine") == NS_a12::LN_solidLine;
        flag&=QName::tokenizer().insert(ns, "gradLine") == NS_a12::LN_gradLine;
        flag&=QName::tokenizer().insert(ns, "pattLine") == NS_a12::LN_pattLine;
        flag&=QName::tokenizer().insert(ns, "font") == NS_a12::LN_font;
        flag&=QName::tokenizer().insert(ns, "txbPr") == NS_a12::LN_txbPr;
        flag&=QName::tokenizer().insert(ns, "txLstStyle") == NS_a12::LN_txLstStyle;
        flag&=QName::tokenizer().insert(ns, "spDef") == NS_a12::LN_spDef;
        flag&=QName::tokenizer().insert(ns, "lnDef") == NS_a12::LN_lnDef;
        flag&=QName::tokenizer().insert(ns, "txbxDef") == NS_a12::LN_txbxDef;
        flag&=QName::tokenizer().insert(ns, "masterClrMapping") == NS_a12::LN_masterClrMapping;
        flag&=QName::tokenizer().insert(ns, "overrideClrMapping") == NS_a12::LN_overrideClrMapping;
        flag&=QName::tokenizer().insert(ns, "baseStyles") == NS_a12::LN_baseStyles;
        flag&=QName::tokenizer().insert(ns, "objectDefaults") == NS_a12::LN_objectDefaults;
        flag&=QName::tokenizer().insert(ns, "masterClrScheme") == NS_a12::LN_masterClrScheme;
        flag&=QName::tokenizer().insert(ns, "overrideClrScheme") == NS_a12::LN_overrideClrScheme;
        flag&=QName::tokenizer().insert(ns, "masterFontScheme") == NS_a12::LN_masterFontScheme;
        flag&=QName::tokenizer().insert(ns, "overrideFontScheme") == NS_a12::LN_overrideFontScheme;
        flag&=QName::tokenizer().insert(ns, "masterStyleMatrix") == NS_a12::LN_masterStyleMatrix;
        flag&=QName::tokenizer().insert(ns, "overrideStyleMatrix") == NS_a12::LN_overrideStyleMatrix;
        flag&=QName::tokenizer().insert(ns, "clrMap") == NS_a12::LN_clrMap;
        flag&=QName::tokenizer().insert(ns, "lnL") == NS_a12::LN_lnL;
        flag&=QName::tokenizer().insert(ns, "lnR") == NS_a12::LN_lnR;
        flag&=QName::tokenizer().insert(ns, "lnT") == NS_a12::LN_lnT;
        flag&=QName::tokenizer().insert(ns, "lnB") == NS_a12::LN_lnB;
        flag&=QName::tokenizer().insert(ns, "lnTlToBr") == NS_a12::LN_lnTlToBr;
        flag&=QName::tokenizer().insert(ns, "lnBlToTr") == NS_a12::LN_lnBlToTr;
        flag&=QName::tokenizer().insert(ns, "cell3D") == NS_a12::LN_cell3D;
        flag&=QName::tokenizer().insert(ns, "gridCol") == NS_a12::LN_gridCol;
        flag&=QName::tokenizer().insert(ns, "txbx") == NS_a12::LN_txbx;
        flag&=QName::tokenizer().insert(ns, "tcPr") == NS_a12::LN_tcPr;
        flag&=QName::tokenizer().insert(ns, "tc") == NS_a12::LN_tc;
        flag&=QName::tokenizer().insert(ns, "tableStyle") == NS_a12::LN_tableStyle;
        flag&=QName::tokenizer().insert(ns, "tableStyleId") == NS_a12::LN_tableStyleId;
        flag&=QName::tokenizer().insert(ns, "tblPr") == NS_a12::LN_tblPr;
        flag&=QName::tokenizer().insert(ns, "tblGrid") == NS_a12::LN_tblGrid;
        flag&=QName::tokenizer().insert(ns, "tr") == NS_a12::LN_tr;
        flag&=QName::tokenizer().insert(ns, "unthemedFill") == NS_a12::LN_unthemedFill;
        flag&=QName::tokenizer().insert(ns, "themedFill") == NS_a12::LN_themedFill;
        flag&=QName::tokenizer().insert(ns, "ln") == NS_a12::LN_ln;
        flag&=QName::tokenizer().insert(ns, "unthemedLine") == NS_a12::LN_unthemedLine;
        flag&=QName::tokenizer().insert(ns, "themedLine") == NS_a12::LN_themedLine;
        flag&=QName::tokenizer().insert(ns, "unthemedFonts") == NS_a12::LN_unthemedFonts;
        flag&=QName::tokenizer().insert(ns, "themedFonts") == NS_a12::LN_themedFonts;
        flag&=QName::tokenizer().insert(ns, "fonts") == NS_a12::LN_fonts;
        flag&=QName::tokenizer().insert(ns, "b") == NS_a12::LN_b;
        flag&=QName::tokenizer().insert(ns, "i") == NS_a12::LN_i;
        flag&=QName::tokenizer().insert(ns, "left") == NS_a12::LN_left;
        flag&=QName::tokenizer().insert(ns, "right") == NS_a12::LN_right;
        flag&=QName::tokenizer().insert(ns, "top") == NS_a12::LN_top;
        flag&=QName::tokenizer().insert(ns, "bottom") == NS_a12::LN_bottom;
        flag&=QName::tokenizer().insert(ns, "insideH") == NS_a12::LN_insideH;
        flag&=QName::tokenizer().insert(ns, "insideV") == NS_a12::LN_insideV;
        flag&=QName::tokenizer().insert(ns, "tl2br") == NS_a12::LN_tl2br;
        flag&=QName::tokenizer().insert(ns, "tr2bl") == NS_a12::LN_tr2bl;
        flag&=QName::tokenizer().insert(ns, "tcBorders") == NS_a12::LN_tcBorders;
        flag&=QName::tokenizer().insert(ns, "rPr") == NS_a12::LN_rPr;
        flag&=QName::tokenizer().insert(ns, "tableBg") == NS_a12::LN_tableBg;
        flag&=QName::tokenizer().insert(ns, "wholeTable") == NS_a12::LN_wholeTable;
        flag&=QName::tokenizer().insert(ns, "band1Horz") == NS_a12::LN_band1Horz;
        flag&=QName::tokenizer().insert(ns, "band2Horz") == NS_a12::LN_band2Horz;
        flag&=QName::tokenizer().insert(ns, "band1Vert") == NS_a12::LN_band1Vert;
        flag&=QName::tokenizer().insert(ns, "band2Vert") == NS_a12::LN_band2Vert;
        flag&=QName::tokenizer().insert(ns, "lastCol") == NS_a12::LN_lastCol;
        flag&=QName::tokenizer().insert(ns, "firstCol") == NS_a12::LN_firstCol;
        flag&=QName::tokenizer().insert(ns, "lastRow") == NS_a12::LN_lastRow;
        flag&=QName::tokenizer().insert(ns, "seCell") == NS_a12::LN_seCell;
        flag&=QName::tokenizer().insert(ns, "swCell") == NS_a12::LN_swCell;
        flag&=QName::tokenizer().insert(ns, "firstRow") == NS_a12::LN_firstRow;
        flag&=QName::tokenizer().insert(ns, "neCell") == NS_a12::LN_neCell;
        flag&=QName::tokenizer().insert(ns, "nwCell") == NS_a12::LN_nwCell;
        flag&=QName::tokenizer().insert(ns, "topBaseline") == NS_a12::LN_topBaseline;
        flag&=QName::tokenizer().insert(ns, "bottomBaseline") == NS_a12::LN_bottomBaseline;
        flag&=QName::tokenizer().insert(ns, "pPr") == NS_a12::LN_pPr;
        flag&=QName::tokenizer().insert(ns, "endParaRPr") == NS_a12::LN_endParaRPr;
        flag&=QName::tokenizer().insert(ns, "defPPr") == NS_a12::LN_defPPr;
        flag&=QName::tokenizer().insert(ns, "outline1pPr") == NS_a12::LN_outline1pPr;
        flag&=QName::tokenizer().insert(ns, "outline2pPr") == NS_a12::LN_outline2pPr;
        flag&=QName::tokenizer().insert(ns, "lvl1pPr") == NS_a12::LN_lvl1pPr;
        flag&=QName::tokenizer().insert(ns, "lvl2pPr") == NS_a12::LN_lvl2pPr;
        flag&=QName::tokenizer().insert(ns, "lvl3pPr") == NS_a12::LN_lvl3pPr;
        flag&=QName::tokenizer().insert(ns, "lvl4pPr") == NS_a12::LN_lvl4pPr;
        flag&=QName::tokenizer().insert(ns, "lvl5pPr") == NS_a12::LN_lvl5pPr;
        flag&=QName::tokenizer().insert(ns, "lvl6pPr") == NS_a12::LN_lvl6pPr;
        flag&=QName::tokenizer().insert(ns, "lvl7pPr") == NS_a12::LN_lvl7pPr;
        flag&=QName::tokenizer().insert(ns, "lvl8pPr") == NS_a12::LN_lvl8pPr;
        flag&=QName::tokenizer().insert(ns, "lvl9pPr") == NS_a12::LN_lvl9pPr;
        flag&=QName::tokenizer().insert(ns, "noAutofit") == NS_a12::LN_noAutofit;
        flag&=QName::tokenizer().insert(ns, "normAutofit") == NS_a12::LN_normAutofit;
        flag&=QName::tokenizer().insert(ns, "shapeAutofit") == NS_a12::LN_shapeAutofit;
        flag&=QName::tokenizer().insert(ns, "textWarp") == NS_a12::LN_textWarp;
        flag&=QName::tokenizer().insert(ns, "prot") == NS_a12::LN_prot;
        flag&=QName::tokenizer().insert(ns, "bodyPr") == NS_a12::LN_bodyPr;
        flag&=QName::tokenizer().insert(ns, "lstStyle") == NS_a12::LN_lstStyle;
        flag&=QName::tokenizer().insert(ns, "p") == NS_a12::LN_p;
        flag&=QName::tokenizer().insert(ns, "buColorTx") == NS_a12::LN_buColorTx;
        flag&=QName::tokenizer().insert(ns, "buColor") == NS_a12::LN_buColor;
        flag&=QName::tokenizer().insert(ns, "buSzTx") == NS_a12::LN_buSzTx;
        flag&=QName::tokenizer().insert(ns, "buSzPct") == NS_a12::LN_buSzPct;
        flag&=QName::tokenizer().insert(ns, "buSzPts") == NS_a12::LN_buSzPts;
        flag&=QName::tokenizer().insert(ns, "buFontTx") == NS_a12::LN_buFontTx;
        flag&=QName::tokenizer().insert(ns, "buFont") == NS_a12::LN_buFont;
        flag&=QName::tokenizer().insert(ns, "buBlip") == NS_a12::LN_buBlip;
        flag&=QName::tokenizer().insert(ns, "noBullet") == NS_a12::LN_noBullet;
        flag&=QName::tokenizer().insert(ns, "autonumberBullet") == NS_a12::LN_autonumberBullet;
        flag&=QName::tokenizer().insert(ns, "charBullet") == NS_a12::LN_charBullet;
        flag&=QName::tokenizer().insert(ns, "blipBullet") == NS_a12::LN_blipBullet;
        flag&=QName::tokenizer().insert(ns, "shadow") == NS_a12::LN_shadow;
        flag&=QName::tokenizer().insert(ns, "outline") == NS_a12::LN_outline;
        flag&=QName::tokenizer().insert(ns, "engrave") == NS_a12::LN_engrave;
        flag&=QName::tokenizer().insert(ns, "emboss") == NS_a12::LN_emboss;
        flag&=QName::tokenizer().insert(ns, "highlight") == NS_a12::LN_highlight;
        flag&=QName::tokenizer().insert(ns, "formatting") == NS_a12::LN_formatting;
        flag&=QName::tokenizer().insert(ns, "u") == NS_a12::LN_u;
        flag&=QName::tokenizer().insert(ns, "ascii") == NS_a12::LN_ascii;
        flag&=QName::tokenizer().insert(ns, "ea") == NS_a12::LN_ea;
        flag&=QName::tokenizer().insert(ns, "cs") == NS_a12::LN_cs;
        flag&=QName::tokenizer().insert(ns, "hAnsi") == NS_a12::LN_hAnsi;
        flag&=QName::tokenizer().insert(ns, "sym") == NS_a12::LN_sym;
        flag&=QName::tokenizer().insert(ns, "link") == NS_a12::LN_link;
        flag&=QName::tokenizer().insert(ns, "acc") == NS_a12::LN_acc;
        flag&=QName::tokenizer().insert(ns, "bar") == NS_a12::LN_bar;
        flag&=QName::tokenizer().insert(ns, "box") == NS_a12::LN_box;
        flag&=QName::tokenizer().insert(ns, "d") == NS_a12::LN_d;
        flag&=QName::tokenizer().insert(ns, "eqArray") == NS_a12::LN_eqArray;
        flag&=QName::tokenizer().insert(ns, "f") == NS_a12::LN_f;
        flag&=QName::tokenizer().insert(ns, "func") == NS_a12::LN_func;
        flag&=QName::tokenizer().insert(ns, "groupChr") == NS_a12::LN_groupChr;
        flag&=QName::tokenizer().insert(ns, "lLow") == NS_a12::LN_lLow;
        flag&=QName::tokenizer().insert(ns, "lUp") == NS_a12::LN_lUp;
        flag&=QName::tokenizer().insert(ns, "m") == NS_a12::LN_m;
        flag&=QName::tokenizer().insert(ns, "nary") == NS_a12::LN_nary;
        flag&=QName::tokenizer().insert(ns, "phant") == NS_a12::LN_phant;
        flag&=QName::tokenizer().insert(ns, "rad") == NS_a12::LN_rad;
        flag&=QName::tokenizer().insert(ns, "sPre") == NS_a12::LN_sPre;
        flag&=QName::tokenizer().insert(ns, "sSub") == NS_a12::LN_sSub;
        flag&=QName::tokenizer().insert(ns, "sSubSup") == NS_a12::LN_sSubSup;
        flag&=QName::tokenizer().insert(ns, "sSup") == NS_a12::LN_sSup;
        flag&=QName::tokenizer().insert(ns, "r") == NS_a12::LN_r;
        flag&=QName::tokenizer().insert(ns, "chr") == NS_a12::LN_chr;
        flag&=QName::tokenizer().insert(ns, "accPr") == NS_a12::LN_accPr;
        flag&=QName::tokenizer().insert(ns, "e") == NS_a12::LN_e;
        flag&=QName::tokenizer().insert(ns, "barPr") == NS_a12::LN_barPr;
        flag&=QName::tokenizer().insert(ns, "begChr") == NS_a12::LN_begChr;
        flag&=QName::tokenizer().insert(ns, "sepChr") == NS_a12::LN_sepChr;
        flag&=QName::tokenizer().insert(ns, "endChr") == NS_a12::LN_endChr;
        flag&=QName::tokenizer().insert(ns, "grow") == NS_a12::LN_grow;
        flag&=QName::tokenizer().insert(ns, "shp") == NS_a12::LN_shp;
        flag&=QName::tokenizer().insert(ns, "dPr") == NS_a12::LN_dPr;
        flag&=QName::tokenizer().insert(ns, "baseJc") == NS_a12::LN_baseJc;
        flag&=QName::tokenizer().insert(ns, "maxDist") == NS_a12::LN_maxDist;
        flag&=QName::tokenizer().insert(ns, "objDist") == NS_a12::LN_objDist;
        flag&=QName::tokenizer().insert(ns, "eqArrPr") == NS_a12::LN_eqArrPr;
        flag&=QName::tokenizer().insert(ns, "type") == NS_a12::LN_type;
        flag&=QName::tokenizer().insert(ns, "numJc") == NS_a12::LN_numJc;
        flag&=QName::tokenizer().insert(ns, "denJc") == NS_a12::LN_denJc;
        flag&=QName::tokenizer().insert(ns, "fPr") == NS_a12::LN_fPr;
        flag&=QName::tokenizer().insert(ns, "num") == NS_a12::LN_num;
        flag&=QName::tokenizer().insert(ns, "den") == NS_a12::LN_den;
        flag&=QName::tokenizer().insert(ns, "fName") == NS_a12::LN_fName;
        flag&=QName::tokenizer().insert(ns, "groupChrPr") == NS_a12::LN_groupChrPr;
        flag&=QName::tokenizer().insert(ns, "lim") == NS_a12::LN_lim;
        flag&=QName::tokenizer().insert(ns, "count") == NS_a12::LN_count;
        flag&=QName::tokenizer().insert(ns, "mcJc") == NS_a12::LN_mcJc;
        flag&=QName::tokenizer().insert(ns, "mcPr") == NS_a12::LN_mcPr;
        flag&=QName::tokenizer().insert(ns, "mc") == NS_a12::LN_mc;
        flag&=QName::tokenizer().insert(ns, "plcHide") == NS_a12::LN_plcHide;
        flag&=QName::tokenizer().insert(ns, "mcs") == NS_a12::LN_mcs;
        flag&=QName::tokenizer().insert(ns, "mrJc") == NS_a12::LN_mrJc;
        flag&=QName::tokenizer().insert(ns, "mrPr") == NS_a12::LN_mrPr;
        flag&=QName::tokenizer().insert(ns, "mPr") == NS_a12::LN_mPr;
        flag&=QName::tokenizer().insert(ns, "mr") == NS_a12::LN_mr;
        flag&=QName::tokenizer().insert(ns, "limLoc") == NS_a12::LN_limLoc;
        flag&=QName::tokenizer().insert(ns, "subHide") == NS_a12::LN_subHide;
        flag&=QName::tokenizer().insert(ns, "supHide") == NS_a12::LN_supHide;
        flag&=QName::tokenizer().insert(ns, "naryPr") == NS_a12::LN_naryPr;
        flag&=QName::tokenizer().insert(ns, "sub") == NS_a12::LN_sub;
        flag&=QName::tokenizer().insert(ns, "sup") == NS_a12::LN_sup;
        flag&=QName::tokenizer().insert(ns, "width") == NS_a12::LN_width;
        flag&=QName::tokenizer().insert(ns, "asc") == NS_a12::LN_asc;
        flag&=QName::tokenizer().insert(ns, "desc") == NS_a12::LN_desc;
        flag&=QName::tokenizer().insert(ns, "phantPr") == NS_a12::LN_phantPr;
        flag&=QName::tokenizer().insert(ns, "degHide") == NS_a12::LN_degHide;
        flag&=QName::tokenizer().insert(ns, "radPr") == NS_a12::LN_radPr;
        flag&=QName::tokenizer().insert(ns, "deg") == NS_a12::LN_deg;
        CPPUNIT_ASSERT_MESSAGE("test for namespace a12 failed", flag);}
    void test_svg(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0";
        flag&=QName::tokenizer().insert(ns, "path") == NS_svg::LN_path;
        flag&=QName::tokenizer().insert(ns, "origin") == NS_svg::LN_origin;
        flag&=QName::tokenizer().insert(ns, "type") == NS_svg::LN_type;
        flag&=QName::tokenizer().insert(ns, "font-family") == NS_svg::LN_font_family;
        flag&=QName::tokenizer().insert(ns, "font-style") == NS_svg::LN_font_style;
        flag&=QName::tokenizer().insert(ns, "font-variant") == NS_svg::LN_font_variant;
        flag&=QName::tokenizer().insert(ns, "font-weight") == NS_svg::LN_font_weight;
        flag&=QName::tokenizer().insert(ns, "font-stretch") == NS_svg::LN_font_stretch;
        flag&=QName::tokenizer().insert(ns, "font-size") == NS_svg::LN_font_size;
        flag&=QName::tokenizer().insert(ns, "unicode-range") == NS_svg::LN_unicode_range;
        flag&=QName::tokenizer().insert(ns, "units-per-em") == NS_svg::LN_units_per_em;
        flag&=QName::tokenizer().insert(ns, "panose-1") == NS_svg::LN_panose_1;
        flag&=QName::tokenizer().insert(ns, "stemv") == NS_svg::LN_stemv;
        flag&=QName::tokenizer().insert(ns, "stemh") == NS_svg::LN_stemh;
        flag&=QName::tokenizer().insert(ns, "slope") == NS_svg::LN_slope;
        flag&=QName::tokenizer().insert(ns, "cap-height") == NS_svg::LN_cap_height;
        flag&=QName::tokenizer().insert(ns, "x-height") == NS_svg::LN_x_height;
        flag&=QName::tokenizer().insert(ns, "accent-height") == NS_svg::LN_accent_height;
        flag&=QName::tokenizer().insert(ns, "ascent") == NS_svg::LN_ascent;
        flag&=QName::tokenizer().insert(ns, "descent") == NS_svg::LN_descent;
        flag&=QName::tokenizer().insert(ns, "widths") == NS_svg::LN_widths;
        flag&=QName::tokenizer().insert(ns, "bbox") == NS_svg::LN_bbox;
        flag&=QName::tokenizer().insert(ns, "ideographic") == NS_svg::LN_ideographic;
        flag&=QName::tokenizer().insert(ns, "alphabetic") == NS_svg::LN_alphabetic;
        flag&=QName::tokenizer().insert(ns, "mathematical") == NS_svg::LN_mathematical;
        flag&=QName::tokenizer().insert(ns, "hanging") == NS_svg::LN_hanging;
        flag&=QName::tokenizer().insert(ns, "v-ideographic") == NS_svg::LN_v_ideographic;
        flag&=QName::tokenizer().insert(ns, "v-alphabetic") == NS_svg::LN_v_alphabetic;
        flag&=QName::tokenizer().insert(ns, "v-mathematical") == NS_svg::LN_v_mathematical;
        flag&=QName::tokenizer().insert(ns, "v-hanging") == NS_svg::LN_v_hanging;
        flag&=QName::tokenizer().insert(ns, "underline-position") == NS_svg::LN_underline_position;
        flag&=QName::tokenizer().insert(ns, "underline-thickness") == NS_svg::LN_underline_thickness;
        flag&=QName::tokenizer().insert(ns, "strikethrough-position") == NS_svg::LN_strikethrough_position;
        flag&=QName::tokenizer().insert(ns, "strikethrough-thickness") == NS_svg::LN_strikethrough_thickness;
        flag&=QName::tokenizer().insert(ns, "overline-position") == NS_svg::LN_overline_position;
        flag&=QName::tokenizer().insert(ns, "overline-thickness") == NS_svg::LN_overline_thickness;
        flag&=QName::tokenizer().insert(ns, "font-face-src") == NS_svg::LN_font_face_src;
        flag&=QName::tokenizer().insert(ns, "font-face-uri") == NS_svg::LN_font_face_uri;
        flag&=QName::tokenizer().insert(ns, "font-face-format") == NS_svg::LN_font_face_format;
        flag&=QName::tokenizer().insert(ns, "string") == NS_svg::LN_string;
        flag&=QName::tokenizer().insert(ns, "font-face-name") == NS_svg::LN_font_face_name;
        flag&=QName::tokenizer().insert(ns, "name") == NS_svg::LN_name;
        flag&=QName::tokenizer().insert(ns, "definition-src") == NS_svg::LN_definition_src;
        flag&=QName::tokenizer().insert(ns, "linearGradient") == NS_svg::LN_linearGradient;
        flag&=QName::tokenizer().insert(ns, "radialGradient") == NS_svg::LN_radialGradient;
        flag&=QName::tokenizer().insert(ns, "fx") == NS_svg::LN_fx;
        flag&=QName::tokenizer().insert(ns, "fy") == NS_svg::LN_fy;
        flag&=QName::tokenizer().insert(ns, "stop") == NS_svg::LN_stop;
        flag&=QName::tokenizer().insert(ns, "offset") == NS_svg::LN_offset;
        flag&=QName::tokenizer().insert(ns, "stop-color") == NS_svg::LN_stop_color;
        flag&=QName::tokenizer().insert(ns, "stop-opacity") == NS_svg::LN_stop_opacity;
        flag&=QName::tokenizer().insert(ns, "gradientUnits") == NS_svg::LN_gradientUnits;
        flag&=QName::tokenizer().insert(ns, "gradientTransform") == NS_svg::LN_gradientTransform;
        flag&=QName::tokenizer().insert(ns, "spreadMethod") == NS_svg::LN_spreadMethod;
        flag&=QName::tokenizer().insert(ns, "stroke-width") == NS_svg::LN_stroke_width;
        flag&=QName::tokenizer().insert(ns, "stroke-color") == NS_svg::LN_stroke_color;
        flag&=QName::tokenizer().insert(ns, "stroke-opacity") == NS_svg::LN_stroke_opacity;
        flag&=QName::tokenizer().insert(ns, "fill-rule") == NS_svg::LN_fill_rule;
        flag&=QName::tokenizer().insert(ns, "x1") == NS_svg::LN_x1;
        flag&=QName::tokenizer().insert(ns, "y1") == NS_svg::LN_y1;
        flag&=QName::tokenizer().insert(ns, "x2") == NS_svg::LN_x2;
        flag&=QName::tokenizer().insert(ns, "y2") == NS_svg::LN_y2;
        flag&=QName::tokenizer().insert(ns, "d") == NS_svg::LN_d;
        flag&=QName::tokenizer().insert(ns, "cx") == NS_svg::LN_cx;
        flag&=QName::tokenizer().insert(ns, "cy") == NS_svg::LN_cy;
        flag&=QName::tokenizer().insert(ns, "r") == NS_svg::LN_r;
        flag&=QName::tokenizer().insert(ns, "rx") == NS_svg::LN_rx;
        flag&=QName::tokenizer().insert(ns, "ry") == NS_svg::LN_ry;
        flag&=QName::tokenizer().insert(ns, "y") == NS_svg::LN_y;
        flag&=QName::tokenizer().insert(ns, "x") == NS_svg::LN_x;
        flag&=QName::tokenizer().insert(ns, "width") == NS_svg::LN_width;
        flag&=QName::tokenizer().insert(ns, "height") == NS_svg::LN_height;
        flag&=QName::tokenizer().insert(ns, "viewBox") == NS_svg::LN_viewBox;
        flag&=QName::tokenizer().insert(ns, "desc") == NS_svg::LN_desc;
        CPPUNIT_ASSERT_MESSAGE("test for namespace svg failed", flag);}
    void test_on12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/onenote/2004/import";
        flag&=QName::tokenizer().insert(ns, "Import") == NS_on12::LN_Import;
        flag&=QName::tokenizer().insert(ns, "EnsurePage") == NS_on12::LN_EnsurePage;
        flag&=QName::tokenizer().insert(ns, "PlaceObjects") == NS_on12::LN_PlaceObjects;
        flag&=QName::tokenizer().insert(ns, "Object") == NS_on12::LN_Object;
        flag&=QName::tokenizer().insert(ns, "Delete") == NS_on12::LN_Delete;
        flag&=QName::tokenizer().insert(ns, "Position") == NS_on12::LN_Position;
        flag&=QName::tokenizer().insert(ns, "Image") == NS_on12::LN_Image;
        flag&=QName::tokenizer().insert(ns, "Ink") == NS_on12::LN_Ink;
        flag&=QName::tokenizer().insert(ns, "Outline") == NS_on12::LN_Outline;
        flag&=QName::tokenizer().insert(ns, "Html") == NS_on12::LN_Html;
        flag&=QName::tokenizer().insert(ns, "File") == NS_on12::LN_File;
        flag&=QName::tokenizer().insert(ns, "Data") == NS_on12::LN_Data;
        CPPUNIT_ASSERT_MESSAGE("test for namespace on12 failed", flag);}
    void test_form(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:form:1.0";
        flag&=QName::tokenizer().insert(ns, "detail-fields") == NS_form::LN_detail_fields;
        flag&=QName::tokenizer().insert(ns, "escape-processing") == NS_form::LN_escape_processing;
        flag&=QName::tokenizer().insert(ns, "filter") == NS_form::LN_filter;
        flag&=QName::tokenizer().insert(ns, "ignore-result") == NS_form::LN_ignore_result;
        flag&=QName::tokenizer().insert(ns, "navigation-mode") == NS_form::LN_navigation_mode;
        flag&=QName::tokenizer().insert(ns, "order") == NS_form::LN_order;
        flag&=QName::tokenizer().insert(ns, "tab-cycle") == NS_form::LN_tab_cycle;
        flag&=QName::tokenizer().insert(ns, "connection-resource") == NS_form::LN_connection_resource;
        flag&=QName::tokenizer().insert(ns, "text") == NS_form::LN_text;
        flag&=QName::tokenizer().insert(ns, "textarea") == NS_form::LN_textarea;
        flag&=QName::tokenizer().insert(ns, "password") == NS_form::LN_password;
        flag&=QName::tokenizer().insert(ns, "echo-char") == NS_form::LN_echo_char;
        flag&=QName::tokenizer().insert(ns, "file") == NS_form::LN_file;
        flag&=QName::tokenizer().insert(ns, "formatted-text") == NS_form::LN_formatted_text;
        flag&=QName::tokenizer().insert(ns, "max-value") == NS_form::LN_max_value;
        flag&=QName::tokenizer().insert(ns, "min-value") == NS_form::LN_min_value;
        flag&=QName::tokenizer().insert(ns, "validation") == NS_form::LN_validation;
        flag&=QName::tokenizer().insert(ns, "number") == NS_form::LN_number;
        flag&=QName::tokenizer().insert(ns, "value") == NS_form::LN_value;
        flag&=QName::tokenizer().insert(ns, "current-value") == NS_form::LN_current_value;
        flag&=QName::tokenizer().insert(ns, "date") == NS_form::LN_date;
        flag&=QName::tokenizer().insert(ns, "time") == NS_form::LN_time;
        flag&=QName::tokenizer().insert(ns, "fixed-text") == NS_form::LN_fixed_text;
        flag&=QName::tokenizer().insert(ns, "multi-line") == NS_form::LN_multi_line;
        flag&=QName::tokenizer().insert(ns, "combobox") == NS_form::LN_combobox;
        flag&=QName::tokenizer().insert(ns, "auto-complete") == NS_form::LN_auto_complete;
        flag&=QName::tokenizer().insert(ns, "item") == NS_form::LN_item;
        flag&=QName::tokenizer().insert(ns, "listbox") == NS_form::LN_listbox;
        flag&=QName::tokenizer().insert(ns, "multiple") == NS_form::LN_multiple;
        flag&=QName::tokenizer().insert(ns, "xforms-list-source") == NS_form::LN_xforms_list_source;
        flag&=QName::tokenizer().insert(ns, "option") == NS_form::LN_option;
        flag&=QName::tokenizer().insert(ns, "button") == NS_form::LN_button;
        flag&=QName::tokenizer().insert(ns, "default-button") == NS_form::LN_default_button;
        flag&=QName::tokenizer().insert(ns, "toggle") == NS_form::LN_toggle;
        flag&=QName::tokenizer().insert(ns, "focus-on-click") == NS_form::LN_focus_on_click;
        flag&=QName::tokenizer().insert(ns, "xforms-submission") == NS_form::LN_xforms_submission;
        flag&=QName::tokenizer().insert(ns, "image") == NS_form::LN_image;
        flag&=QName::tokenizer().insert(ns, "checkbox") == NS_form::LN_checkbox;
        flag&=QName::tokenizer().insert(ns, "current-state") == NS_form::LN_current_state;
        flag&=QName::tokenizer().insert(ns, "is-tristate") == NS_form::LN_is_tristate;
        flag&=QName::tokenizer().insert(ns, "state") == NS_form::LN_state;
        flag&=QName::tokenizer().insert(ns, "radio") == NS_form::LN_radio;
        flag&=QName::tokenizer().insert(ns, "frame") == NS_form::LN_frame;
        flag&=QName::tokenizer().insert(ns, "image-frame") == NS_form::LN_image_frame;
        flag&=QName::tokenizer().insert(ns, "hidden") == NS_form::LN_hidden;
        flag&=QName::tokenizer().insert(ns, "grid") == NS_form::LN_grid;
        flag&=QName::tokenizer().insert(ns, "column") == NS_form::LN_column;
        flag&=QName::tokenizer().insert(ns, "text-style-name") == NS_form::LN_text_style_name;
        flag&=QName::tokenizer().insert(ns, "value-range") == NS_form::LN_value_range;
        flag&=QName::tokenizer().insert(ns, "step-size") == NS_form::LN_step_size;
        flag&=QName::tokenizer().insert(ns, "page-step-size") == NS_form::LN_page_step_size;
        flag&=QName::tokenizer().insert(ns, "delay-for-repeat") == NS_form::LN_delay_for_repeat;
        flag&=QName::tokenizer().insert(ns, "orientation") == NS_form::LN_orientation;
        flag&=QName::tokenizer().insert(ns, "generic-control") == NS_form::LN_generic_control;
        flag&=QName::tokenizer().insert(ns, "name") == NS_form::LN_name;
        flag&=QName::tokenizer().insert(ns, "control-implementation") == NS_form::LN_control_implementation;
        flag&=QName::tokenizer().insert(ns, "button-type") == NS_form::LN_button_type;
        flag&=QName::tokenizer().insert(ns, "id") == NS_form::LN_id;
        flag&=QName::tokenizer().insert(ns, "current-selected") == NS_form::LN_current_selected;
        flag&=QName::tokenizer().insert(ns, "disabled") == NS_form::LN_disabled;
        flag&=QName::tokenizer().insert(ns, "dropdown") == NS_form::LN_dropdown;
        flag&=QName::tokenizer().insert(ns, "for") == NS_form::LN_for;
        flag&=QName::tokenizer().insert(ns, "image-data") == NS_form::LN_image_data;
        flag&=QName::tokenizer().insert(ns, "label") == NS_form::LN_label;
        flag&=QName::tokenizer().insert(ns, "max-length") == NS_form::LN_max_length;
        flag&=QName::tokenizer().insert(ns, "printable") == NS_form::LN_printable;
        flag&=QName::tokenizer().insert(ns, "readonly") == NS_form::LN_readonly;
        flag&=QName::tokenizer().insert(ns, "selected") == NS_form::LN_selected;
        flag&=QName::tokenizer().insert(ns, "size") == NS_form::LN_size;
        flag&=QName::tokenizer().insert(ns, "tab-index") == NS_form::LN_tab_index;
        flag&=QName::tokenizer().insert(ns, "tab-stop") == NS_form::LN_tab_stop;
        flag&=QName::tokenizer().insert(ns, "title") == NS_form::LN_title;
        flag&=QName::tokenizer().insert(ns, "visual-effect") == NS_form::LN_visual_effect;
        flag&=QName::tokenizer().insert(ns, "image-position") == NS_form::LN_image_position;
        flag&=QName::tokenizer().insert(ns, "image-align") == NS_form::LN_image_align;
        flag&=QName::tokenizer().insert(ns, "bound-column") == NS_form::LN_bound_column;
        flag&=QName::tokenizer().insert(ns, "convert-empty-to-null") == NS_form::LN_convert_empty_to_null;
        flag&=QName::tokenizer().insert(ns, "data-field") == NS_form::LN_data_field;
        flag&=QName::tokenizer().insert(ns, "list-source") == NS_form::LN_list_source;
        flag&=QName::tokenizer().insert(ns, "list-source-type") == NS_form::LN_list_source_type;
        flag&=QName::tokenizer().insert(ns, "properties") == NS_form::LN_properties;
        flag&=QName::tokenizer().insert(ns, "property") == NS_form::LN_property;
        flag&=QName::tokenizer().insert(ns, "property-name") == NS_form::LN_property_name;
        flag&=QName::tokenizer().insert(ns, "list-property") == NS_form::LN_list_property;
        flag&=QName::tokenizer().insert(ns, "list-value") == NS_form::LN_list_value;
        flag&=QName::tokenizer().insert(ns, "automatic-focus") == NS_form::LN_automatic_focus;
        flag&=QName::tokenizer().insert(ns, "apply-design-mode") == NS_form::LN_apply_design_mode;
        flag&=QName::tokenizer().insert(ns, "form") == NS_form::LN_form;
        flag&=QName::tokenizer().insert(ns, "method") == NS_form::LN_method;
        flag&=QName::tokenizer().insert(ns, "enctype") == NS_form::LN_enctype;
        flag&=QName::tokenizer().insert(ns, "allow-deletes") == NS_form::LN_allow_deletes;
        flag&=QName::tokenizer().insert(ns, "allow-inserts") == NS_form::LN_allow_inserts;
        flag&=QName::tokenizer().insert(ns, "allow-updates") == NS_form::LN_allow_updates;
        flag&=QName::tokenizer().insert(ns, "apply-filter") == NS_form::LN_apply_filter;
        flag&=QName::tokenizer().insert(ns, "command-type") == NS_form::LN_command_type;
        flag&=QName::tokenizer().insert(ns, "command") == NS_form::LN_command;
        flag&=QName::tokenizer().insert(ns, "datasource") == NS_form::LN_datasource;
        flag&=QName::tokenizer().insert(ns, "master-fields") == NS_form::LN_master_fields;
        CPPUNIT_ASSERT_MESSAGE("test for namespace form failed", flag);}
    void test_script(){
        bool flag=true;
        static const char ns[] = "urn:oasis:names:tc:opendocument:xmlns:script:1.0";
        flag&=QName::tokenizer().insert(ns, "event-listener") == NS_script::LN_event_listener;
        flag&=QName::tokenizer().insert(ns, "macro-name") == NS_script::LN_macro_name;
        flag&=QName::tokenizer().insert(ns, "language") == NS_script::LN_language;
        flag&=QName::tokenizer().insert(ns, "event-name") == NS_script::LN_event_name;
        CPPUNIT_ASSERT_MESSAGE("test for namespace script failed", flag);}
    void test_aml12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/aml/2001/core";
        flag&=QName::tokenizer().insert(ns, "annotation") == NS_aml12::LN_annotation;
        flag&=QName::tokenizer().insert(ns, "arc") == NS_aml12::LN_arc;
        flag&=QName::tokenizer().insert(ns, "content") == NS_aml12::LN_content;
        flag&=QName::tokenizer().insert(ns, "context") == NS_aml12::LN_context;
        flag&=QName::tokenizer().insert(ns, "property") == NS_aml12::LN_property;
        CPPUNIT_ASSERT_MESSAGE("test for namespace aml12 failed", flag);}
    void test_dc(){
        bool flag=true;
        static const char ns[] = "http://purl.org/dc/elements/1.1/";
        flag&=QName::tokenizer().insert(ns, "title") == NS_dc::LN_title;
        flag&=QName::tokenizer().insert(ns, "description") == NS_dc::LN_description;
        flag&=QName::tokenizer().insert(ns, "subject") == NS_dc::LN_subject;
        flag&=QName::tokenizer().insert(ns, "creator") == NS_dc::LN_creator;
        flag&=QName::tokenizer().insert(ns, "date") == NS_dc::LN_date;
        flag&=QName::tokenizer().insert(ns, "language") == NS_dc::LN_language;
        CPPUNIT_ASSERT_MESSAGE("test for namespace dc failed", flag);}
    void test_wa11(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/word/2003/auxHint";
        flag&=QName::tokenizer().insert(ns, "sect") == NS_wa11::LN_sect;
        flag&=QName::tokenizer().insert(ns, "sub-section") == NS_wa11::LN_sub_section;
        flag&=QName::tokenizer().insert(ns, "allowEmptyCollapse") == NS_wa11::LN_allowEmptyCollapse;
        flag&=QName::tokenizer().insert(ns, "font") == NS_wa11::LN_font;
        flag&=QName::tokenizer().insert(ns, "sym") == NS_wa11::LN_sym;
        flag&=QName::tokenizer().insert(ns, "top") == NS_wa11::LN_top;
        flag&=QName::tokenizer().insert(ns, "left") == NS_wa11::LN_left;
        flag&=QName::tokenizer().insert(ns, "bottom") == NS_wa11::LN_bottom;
        flag&=QName::tokenizer().insert(ns, "right") == NS_wa11::LN_right;
        flag&=QName::tokenizer().insert(ns, "between") == NS_wa11::LN_between;
        flag&=QName::tokenizer().insert(ns, "bar") == NS_wa11::LN_bar;
        flag&=QName::tokenizer().insert(ns, "width") == NS_wa11::LN_width;
        flag&=QName::tokenizer().insert(ns, "height") == NS_wa11::LN_height;
        flag&=QName::tokenizer().insert(ns, "jc") == NS_wa11::LN_jc;
        flag&=QName::tokenizer().insert(ns, "vertFromText") == NS_wa11::LN_vertFromText;
        flag&=QName::tokenizer().insert(ns, "horizFromText") == NS_wa11::LN_horizFromText;
        flag&=QName::tokenizer().insert(ns, "borders") == NS_wa11::LN_borders;
        flag&=QName::tokenizer().insert(ns, "margin-left") == NS_wa11::LN_margin_left;
        flag&=QName::tokenizer().insert(ns, "margin-right") == NS_wa11::LN_margin_right;
        flag&=QName::tokenizer().insert(ns, "apo") == NS_wa11::LN_apo;
        flag&=QName::tokenizer().insert(ns, "shd") == NS_wa11::LN_shd;
        flag&=QName::tokenizer().insert(ns, "pBdrGroup") == NS_wa11::LN_pBdrGroup;
        flag&=QName::tokenizer().insert(ns, "t") == NS_wa11::LN_t;
        flag&=QName::tokenizer().insert(ns, "uiName") == NS_wa11::LN_uiName;
        CPPUNIT_ASSERT_MESSAGE("test for namespace wa11 failed", flag);}
    void test_swbu12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/excel/2005/8/sharedworkbookusers";
        flag&=QName::tokenizer().insert(ns, "users") == NS_swbu12::LN_users;
        flag&=QName::tokenizer().insert(ns, "usrinfo") == NS_swbu12::LN_usrinfo;
        CPPUNIT_ASSERT_MESSAGE("test for namespace swbu12 failed", flag);}
    void test_c12(){
        bool flag=true;
        static const char ns[] = "http://schemas.microsoft.com/office/2004/7/core";
        flag&=QName::tokenizer().insert(ns, "SmartTagType") == NS_c12::LN_SmartTagType;
        flag&=QName::tokenizer().insert(ns, "schemaRef") == NS_c12::LN_schemaRef;
        CPPUNIT_ASSERT_MESSAGE("test for namespace c12 failed", flag);}


    void test_context1()
    {
        std::auto_ptr<QNameTokenizerContext> context=QName::tokenizer().createQNameTokenizerContext();
        context->addBinding("xmlns:test", "http://sample.org/test1");
        CPPUNIT_ASSERT(rtl_str_compare(context->resolvePrefix("test"), "http://sample.org/test1")==0);
    }

    void test_context2()
    {
        std::auto_ptr<QNameTokenizerContext> context=QName::tokenizer().createQNameTokenizerContext();
        context->addBinding("xmlns:test", "http://sample.org/test1");
        context->enterScope();
        CPPUNIT_ASSERT(rtl_str_compare(context->resolvePrefix("test"), "http://sample.org/test1")==0);
        context->leaveScope();
        CPPUNIT_ASSERT(rtl_str_compare(context->resolvePrefix("test"), "http://sample.org/test1")==0);
    }

    void test_context3()
    {
        std::auto_ptr<QNameTokenizerContext> context=QName::tokenizer().createQNameTokenizerContext();
        context->addBinding("xmlns:test", "http://sample.org/test1");
        context->enterScope();
        context->addBinding("xmlns:test", "http://sample.org/test2");
        CPPUNIT_ASSERT(rtl_str_compare(context->resolvePrefix("test"), "http://sample.org/test2")==0);
        context->leaveScope();
        CPPUNIT_ASSERT(rtl_str_compare(context->resolvePrefix("test"), "http://sample.org/test1")==0);
    }

    void test_resolver1()
    {
        QName_t qn;
        std::auto_ptr<QNameTokenizerContext> context=QName::tokenizer().createQNameTokenizerContext();
        context->addBinding("xmlns:text", "urn:oasis:names:tc:opendocument:xmlns:text:1.0");
        CPPUNIT_ASSERT(QName::tokenizer().insert("text:span", *context, &qn));
        CPPUNIT_ASSERT(qn== NS_text::LN_span);
    }

    void test_serializer1()
    {
        std::auto_ptr<QNameSerializerContext> context=QName::tokenizer().createQNameSerializerContext();
        {
            context->addPrefix("text", "urn:oasis:names:tc:opendocument:xmlns:text:1.0");
            CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:text:1.0"), "text")==0);
        }
    }

    void test_serializer2()
    {
        std::auto_ptr<QNameSerializerContext> context=QName::tokenizer().createQNameSerializerContext();
        {
            CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://sample.org/test1"), "_p0")==0);
            CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://sample.org/test2"), "_p1")==0);
        }
    }


    void test_prefix()
    {
        std::auto_ptr<QNameSerializerContext> context=QName::tokenizer().createQNameSerializerContext();
        CPPUNIT_ASSERT(context->getPrefix("")==QName::EMPTY_STRING);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:animation:1.0"), "anium")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://www.w3.org/1998/Math/MathML"), "math")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:word"), "w10")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:chart:1.0"), "chart")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/singlecells"), "sc12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:odc"), "odc12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:component:spreadsheet"), "c11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://relaxng.org/ns/structure/1.0"), "rng")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:meta:1.0"), "meta")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/data/udc/soap"), "udcs")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://www.w3.org/1999/xlink"), "xlink")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:style:1.0"), "style")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0"), "fo")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://www.w3.org/2002/xforms"), "xforms")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:drawing:1.0"), "draw")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/project"), "prj12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/package/2005/06/metadata/core-properties"), "mdc12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:table:1.0"), "table")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:rowset"), "rs11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/sxref"), "sxr12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/officeart/2005/8/xldr"), "xldr12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/omml/2004/12/core"), "m12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0"), "number")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/package/2005/06/content-types"), "p12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/voldeps"), "voldeps12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/pivot"), "pt12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://www.w3.org/XML/1998/namespace"), "xml")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/word/2003/wordml"), "w11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/schemaLibrary/2003/core"), "sl12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/styles"), "s12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/data/udc"), "udc")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0"), "dr3d")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://www.w3.org/1999/02/22-rdf-syntax-ns#"), "rdf")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/supbook"), "sb12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/workbook"), "wb12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("#RowsetSchema"), "RowsetSchema")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/externalconnection"), "ec12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/querytable"), "qt12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:office"), "o12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("uuid:BDC6E3F0-6DA3-11d1-A2A3-00AA00C14882"), "s11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:excel"), "x11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:office:spreadsheet"), "ss11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:rtf:1.7"), "rtf")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:smil-compatible:1.0"), "smil")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/word/2005/8/25/wordml"), "w12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:rtf:schema"), "rtfs")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2003/xml"), "x2")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/infopath/2003/solutionDefinition"), "ip12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/package/2005/06/relationships"), "rel12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:config:1.0"), "config")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:text:1.0"), "text")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/table"), "tbl12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:presentation:1.0"), "pr")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/data/udc/xmlfile"), "udcxf11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:office:1.0"), "office")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/metadata"), "md12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/powerpoint/2005/8/pml"), "pp12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/visio/2003/core"), "v11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/sst"), "sst12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/sharedworkbook"), "swb12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/officeart/2005/8/picture"), "pic12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:schemas-microsoft-com:vml"), "vml")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/activeX/2005/ax"), "ax12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/worksheet"), "ws12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/officeart/2005/8/wddr"), "wddr12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/officeart/2005/8/oartml"), "a12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0"), "svg")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/onenote/2004/import"), "on12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:form:1.0"), "form")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("urn:oasis:names:tc:opendocument:xmlns:script:1.0"), "script")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/aml/2001/core"), "aml12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://purl.org/dc/elements/1.1/"), "dc")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/word/2003/auxHint"), "wa11")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/excel/2005/8/sharedworkbookusers"), "swbu12")==0);
        CPPUNIT_ASSERT(rtl_str_compare(context->getPrefix("http://schemas.microsoft.com/office/2004/7/core"), "c12")==0);
    }


    CPPUNIT_TEST_SUITE(TestQName);
    CPPUNIT_TEST(test);
//	CPPUNIT_TEST(testperf);
    CPPUNIT_TEST(testinsert);
//	CPPUNIT_TEST(testdos);
//	CPPUNIT_TEST(testperf);

    CPPUNIT_TEST(test_anium);
    CPPUNIT_TEST(test_math);
    CPPUNIT_TEST(test_w10);
    CPPUNIT_TEST(test_chart);
    CPPUNIT_TEST(test_sc12);
    CPPUNIT_TEST(test_odc12);
    CPPUNIT_TEST(test_c11);
    CPPUNIT_TEST(test_rng);
    CPPUNIT_TEST(test_meta);
    CPPUNIT_TEST(test_udcs);
    CPPUNIT_TEST(test_xlink);
    CPPUNIT_TEST(test_style);
    CPPUNIT_TEST(test_fo);
    CPPUNIT_TEST(test_xforms);
    CPPUNIT_TEST(test_draw);
    CPPUNIT_TEST(test_prj12);
    CPPUNIT_TEST(test_mdc12);
    CPPUNIT_TEST(test_table);
    CPPUNIT_TEST(test_rs11);
    CPPUNIT_TEST(test_sxr12);
    CPPUNIT_TEST(test_xldr12);
    CPPUNIT_TEST(test_m12);
    CPPUNIT_TEST(test_number);
    CPPUNIT_TEST(test_p12);
    CPPUNIT_TEST(test_voldeps12);
    CPPUNIT_TEST(test_pt12);
    CPPUNIT_TEST(test_xml);
    CPPUNIT_TEST(test_w11);
    CPPUNIT_TEST(test_sl12);
    CPPUNIT_TEST(test_s12);
    CPPUNIT_TEST(test_udc);
    CPPUNIT_TEST(test_dr3d);
    CPPUNIT_TEST(test_rdf);
    CPPUNIT_TEST(test_sb12);
    CPPUNIT_TEST(test_wb12);
    CPPUNIT_TEST(test_RowsetSchema);
    CPPUNIT_TEST(test_ec12);
    CPPUNIT_TEST(test_qt12);
    CPPUNIT_TEST(test_o12);
    CPPUNIT_TEST(test_s11);
    CPPUNIT_TEST(test_x11);
    CPPUNIT_TEST(test_ss11);
    CPPUNIT_TEST(test_rtf);
    CPPUNIT_TEST(test_smil);
    CPPUNIT_TEST(test_w12);
    CPPUNIT_TEST(test_rtfs);
    CPPUNIT_TEST(test_x2);
    CPPUNIT_TEST(test_ip12);
    CPPUNIT_TEST(test_rel12);
    CPPUNIT_TEST(test_NONE);
    CPPUNIT_TEST(test_config);
    CPPUNIT_TEST(test_text);
    CPPUNIT_TEST(test_tbl12);
    CPPUNIT_TEST(test_pr);
    CPPUNIT_TEST(test_udcxf11);
    CPPUNIT_TEST(test_office);
    CPPUNIT_TEST(test_md12);
    CPPUNIT_TEST(test_pp12);
    CPPUNIT_TEST(test_v11);
    CPPUNIT_TEST(test_sst12);
    CPPUNIT_TEST(test_swb12);
    CPPUNIT_TEST(test_pic12);
    CPPUNIT_TEST(test_vml);
    CPPUNIT_TEST(test_ax12);
    CPPUNIT_TEST(test_ws12);
    CPPUNIT_TEST(test_wddr12);
    CPPUNIT_TEST(test_a12);
    CPPUNIT_TEST(test_svg);
    CPPUNIT_TEST(test_on12);
    CPPUNIT_TEST(test_form);
    CPPUNIT_TEST(test_script);
    CPPUNIT_TEST(test_aml12);
    CPPUNIT_TEST(test_dc);
    CPPUNIT_TEST(test_wa11);
    CPPUNIT_TEST(test_swbu12);
    CPPUNIT_TEST(test_c12);

    CPPUNIT_TEST(test_context1);
    CPPUNIT_TEST(test_context2);
    CPPUNIT_TEST(test_context3);

    CPPUNIT_TEST(test_resolver1);

    CPPUNIT_TEST(test_serializer1);
    CPPUNIT_TEST(test_serializer2);

    CPPUNIT_TEST(test_prefix);

    CPPUNIT_TEST_SUITE_END();
};

//#####################################
// register test suites
CPPUNIT_TEST_SUITE_NAMED_REGISTRATION(TestQName, "TestQName");

NOADDITIONAL;

