/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: sharedstringsbuffer.cxx,v $
 *
 *  $Revision: 1.1.2.13 $
 *
 *  last change: $Author: dr $ $Date: 2007/08/24 09:07:42 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "oox/xls/sharedstringsbuffer.hxx"
#include <rtl/ustrbuf.hxx>
#include <com/sun/star/text/XText.hpp>
#include "oox/core/propertyset.hxx"
#include "oox/xls/biffinputstream.hxx"

using ::rtl::OString;
using ::rtl::OUString;
using ::rtl::OUStringBuffer;
using ::com::sun::star::uno::Reference;
using ::com::sun::star::text::XText;
using ::com::sun::star::text::XTextRange;
using ::oox::core::AttributeList;
using ::oox::core::PropertySet;

namespace oox {
namespace xls {

// ============================================================================

bool operator==( const BiffRichStringFontId& rLeft, const BiffRichStringFontId& rRight )
{
    return (rLeft.mnPos == rRight.mnPos) && (rLeft.mnFontId == rRight.mnFontId);
}

bool operator<( const BiffRichStringFontId& rLeft, const BiffRichStringFontId& rRight )
{
    return (rLeft.mnPos < rRight.mnPos) || ((rLeft.mnPos == rRight.mnPos) && (rLeft.mnFontId < rRight.mnFontId));
}

// ============================================================================

RichStringPortion::RichStringPortion( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData ),
    mnFontId( -1 )
{
}

void RichStringPortion::setText( const OUString& rText )
{
    maText = rText;
}

FontRef RichStringPortion::importFont( const AttributeList& )
{
    mxFont.reset( new Font( getGlobalData() ) );
    return mxFont;
}

void RichStringPortion::setFontId( sal_Int32 nFontId )
{
    mnFontId = nFontId;
}

void RichStringPortion::finalizeImport()
{
    if( mxFont.get() )
        mxFont->finalizeImport();
    else if( mnFontId >= 0 )
        mxFont = getStyles().getFont( mnFontId );
}

void RichStringPortion::convert( const Reference< XText >& rxText, sal_Int32 nXfId )
{
    Reference< XTextRange > xRange = rxText->getEnd();
    xRange->setString( maText );
    if( mxFont.get() )
    {
        PropertySet aPropSet( xRange );
        mxFont->writeToPropertySet( aPropSet, FONT_PROPTYPE_RICHTEXT );
    }
    if( const Font* pFont = getStyles().getFontFromCellXf( nXfId ).get() )
    {
        if( pFont->needsRichTextFormat() )
        {
            PropertySet aPropSet( xRange );
            pFont->writeToPropertySet( aPropSet, FONT_PROPTYPE_RICHTEXT );
        }
    }
}

// ============================================================================

RichStringPhoneticPortion::RichStringPhoneticPortion( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData )
{
}

void RichStringPhoneticPortion::setText( const OUString& rText )
{
    maText = rText;
}

// ============================================================================

RichString::RichString( const GlobalDataHelper& rGlobalData ) :
    GlobalDataHelper( rGlobalData )
{
}

RichStringPortionRef RichString::importText( const AttributeList& )
{
    return createPortion();
}

RichStringPortionRef RichString::importPortion( const AttributeList& )
{
    return createPortion();
}

void RichString::appendFontId( BiffRichStringFontIdVec& orFontIds, sal_Int32 nPos, sal_Int32 nFontId )
{
    // #i33341# real life -- same character index may occur several times
    OSL_ENSURE( orFontIds.empty() || (orFontIds.back().mnPos <= nPos), "RichString::appendFontId - wrong char order" );
    if( orFontIds.empty() || (orFontIds.back().mnPos < nPos) )
        orFontIds.push_back( BiffRichStringFontId( nPos, nFontId ) );
    else
        orFontIds.back().mnFontId = nFontId;
}

void RichString::importFontIds( BiffRichStringFontIdVec& orFontIds, BiffInputStream& rStrm, sal_uInt16 nCount, bool b16Bit )
{
    orFontIds.clear();
    orFontIds.reserve( nCount );
    /*  #i33341# real life -- same character index may occur several times
        -> use appendFontId() to validate string position. */
    if( b16Bit )
    {
        for( sal_uInt16 nIndex = 0; rStrm.isValid() && (nIndex < nCount); ++nIndex )
        {
            sal_uInt16 nPos, nFontId;
            rStrm >> nPos >> nFontId;
            appendFontId( orFontIds, nPos, nFontId );
        }
    }
    else
    {
        for( sal_uInt16 nIndex = 0; rStrm.isValid() && (nIndex < nCount); ++nIndex )
        {
            sal_uInt8 nPos, nFontId;
            rStrm >> nPos >> nFontId;
            appendFontId( orFontIds, nPos, nFontId );
        }
    }
}

void RichString::importFontIds( BiffRichStringFontIdVec& orFontIds, BiffInputStream& rStrm, bool b16Bit )
{
    sal_uInt16 nCount = b16Bit ? rStrm.readuInt16() : rStrm.readuInt8();
    importFontIds( orFontIds, rStrm, nCount, b16Bit );
}

void RichString::importByteString( BiffInputStream& rStrm, rtl_TextEncoding eDefaultTextEnc, BiffStringFlags nFlags )
{
    OSL_ENSURE( !getFlag( nFlags, BIFF_STR_KEEPFONTS ), "RichString::importString - keep fonts not implemented" );
    OSL_ENSURE( !getFlag( nFlags, static_cast< BiffStringFlags >( ~(BIFF_STR_8BITLENGTH | BIFF_STR_EXTRAFONTS) ) ), "RichString::importByteString - unknown flag" );
    bool b8BitLength = getFlag( nFlags, BIFF_STR_8BITLENGTH );

    OString aString = rStrm.readByteString( !b8BitLength );
    BiffRichStringFontIdVec aFontIds;
    if( getFlag( nFlags, BIFF_STR_EXTRAFONTS ) )
        importFontIds( aFontIds, rStrm, false );

    // create string portions
    maPortions.clear();
    sal_Int32 nStrLen = aString.getLength();
    if( nStrLen > 0 )
    {
        // add leading and trailing string position to ease the following loop
        if( aFontIds.empty() || (aFontIds.front().mnPos > 0) )
            aFontIds.insert( aFontIds.begin(), BiffRichStringFontId( 0, -1 ) );
        if( aFontIds.back().mnPos < nStrLen )
            aFontIds.push_back( BiffRichStringFontId( nStrLen, -1 ) );

        // create all string portions according to the font id vector
        for( BiffRichStringFontIdVec::const_iterator aIt = aFontIds.begin(); aIt->mnPos < nStrLen; ++aIt )
        {
            sal_Int32 nPortionLen = (aIt + 1)->mnPos - aIt->mnPos;
            if( nPortionLen > 0 )
            {
                // convert byte string to unicode string, using current font encoding
                FontRef xFont = getStyles().getFont( aIt->mnFontId );
                rtl_TextEncoding eTextEnc = xFont.get() ? xFont->getFontEncoding() : eDefaultTextEnc;
                OUString aUniStr = OStringToOUString( aString.copy( aIt->mnPos, nPortionLen ), eTextEnc );
                // create string portion
                RichStringPortionRef xPortion = createPortion();
                xPortion->setText( aUniStr );
                xPortion->setFontId( aIt->mnFontId );
            }
        }
    }
}

void RichString::importUniString( BiffInputStream& rStrm, BiffStringFlags nFlags )
{
    OSL_ENSURE( !getFlag( nFlags, BIFF_STR_KEEPFONTS ), "RichString::importUniString - keep fonts not implemented" );
    OSL_ENSURE( !getFlag( nFlags, static_cast< BiffStringFlags >( ~(BIFF_STR_8BITLENGTH | BIFF_STR_SMARTFLAGS) ) ), "RichString::importUniString - unknown flag" );
    bool b8BitLength = getFlag( nFlags, BIFF_STR_8BITLENGTH );

    // --- string header ---
    sal_uInt16 nChars = b8BitLength ? rStrm.readuInt8() : rStrm.readuInt16();
    sal_uInt8 nFlagField = 0;
    if( (nChars > 0) || !getFlag( nFlags, BIFF_STR_SMARTFLAGS ) )
        rStrm >> nFlagField;

    bool b16Bit, bFonts, bPhonetic;
    sal_uInt16 nFontCount;
    sal_uInt32 nPhoneticSize;
    rStrm.readExtendedUniStringHeader( b16Bit, bFonts, bPhonetic, nFontCount, nPhoneticSize, nFlagField );

    // --- character array ---
    OUString aString = rStrm.readRawUniString( nChars, b16Bit );

    // --- formatting ---
    // #122185# bRich flag may be set, but format runs may be missing
    BiffRichStringFontIdVec aFontIds;
    if( nFontCount > 0 )
    {
        // reserve 2 elements more for leading and trailing index added below
        aFontIds.reserve( nFontCount + 2 );
        importFontIds( aFontIds, rStrm, nFontCount, true );
    }

    // --- Asian phonetic information ---
    // #122185# bPhonetic flag may be set, but phonetic info may be missing
    rStrm.ignore( nPhoneticSize );

    // create string portions
    maPortions.clear();
    sal_Int32 nStrLen = aString.getLength();
    if( nStrLen > 0 )
    {
        // add leading and trailing string position to ease the following loop
        if( aFontIds.empty() || (aFontIds.front().mnPos > 0) )
            aFontIds.insert( aFontIds.begin(), BiffRichStringFontId( 0, -1 ) );
        if( aFontIds.back().mnPos < nStrLen )
            aFontIds.push_back( BiffRichStringFontId( nStrLen, -1 ) );

        // create all string portions according to the font id vector
        for( BiffRichStringFontIdVec::const_iterator aIt = aFontIds.begin(); aIt->mnPos < nStrLen; ++aIt )
        {
            sal_Int32 nPortionLen = (aIt + 1)->mnPos - aIt->mnPos;
            if( nPortionLen > 0 )
            {
                RichStringPortionRef xPortion = createPortion();
                xPortion->setText( aString.copy( aIt->mnPos, nPortionLen ) );
                xPortion->setFontId( aIt->mnFontId );
            }
        }
    }
}

void RichString::finalizeImport()
{
    maPortions.forEachMem( &RichStringPortion::finalizeImport );
}

void RichString::convert( const Reference< XText >& rxText, sal_Int32 nXfId ) const
{
    for( PortionVec::const_iterator aIt = maPortions.begin(), aEnd = maPortions.end(); aIt != aEnd; ++aIt )
    {
        (*aIt)->convert( rxText, nXfId );
        nXfId = -1;
    }
}

RichStringPortionRef RichString::createPortion()
{
    RichStringPortionRef xPortion( new RichStringPortion( getGlobalData() ) );
    maPortions.push_back( xPortion );
    return xPortion;
}

// ============================================================================

SharedStringsBuffer::SharedStringsBuffer( const GlobalDataHelper& rGlobalData ) :
     GlobalDataHelper( rGlobalData )
{
}

RichStringRef SharedStringsBuffer::importString( const AttributeList& )
{
    RichStringRef xString( new RichString( getGlobalData() ) );
    maStrings.push_back( xString );
    return xString;
}

void SharedStringsBuffer::importSst( BiffInputStream& rStrm )
{
    sal_Int32 nStringCount;
    rStrm.ignore( 4 );
    rStrm >> nStringCount;
    if( nStringCount > 0 )
    {
        maStrings.clear();
        maStrings.reserve( static_cast< size_t >( nStringCount ) );
        for( ; rStrm.isValid() && (nStringCount > 0); --nStringCount )
        {
            RichStringRef xString( new RichString( getGlobalData() ) );
            maStrings.push_back( xString );
            xString->importUniString( rStrm );
        }
    }
}

void SharedStringsBuffer::finalizeImport()
{
    maStrings.forEachMem( &RichString::finalizeImport );
}

void SharedStringsBuffer::convertString( const Reference< XText >& rxText, sal_Int32 nIndex, sal_Int32 nXfId ) const
{
    if( rxText.is() )
        if( const RichString* pString = maStrings.get( nIndex ).get() )
            pString->convert( rxText, nXfId );
}

// ============================================================================

} // namespace xls
} // namespace oox

