
/* line clipping algorithm, pilfered from GGI sources ************************/

/* $Id: draw_line.c,v 1.1 2003/12/23 01:15:02 agraef Exp $
******************************************************************************

   Graphics library for GGI.

   Copyright (C) 1998 Alexander Larsson   [alla@lysator.liu.se]
   
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/ggi-dl.h>

/*
  This is a line-clipper using the algorithm by cohen-sutherland.

  It is modified to do pixel-perfect clipping. This means that it
  will generate the same endpoints that would be drawn if an ordinary
  bresenham line-drawer where used and only visible pixels drawn.

  It can be used with a bresenham-like linedrawer if it is modified to
  start with a correct error-term.
*/

#define OC_LEFT 1
#define OC_RIGHT 2
#define OC_TOP 4
#define OC_BOTTOM 8

/* Outcodes:
+-> x
|       |      | 
V  0101 | 0100 | 0110
y ---------------------
   0001 | 0000 | 0010
  ---------------------
   1001 | 1000 | 1010
        |      | 
 */
#define outcode(code, xx, yy) \
{\
  code = 0;\
  if ((xx)<(LIBGGI_GC(vis)->cliptl.x))\
    code |= OC_LEFT;\
  else if ((xx)>=(LIBGGI_GC(vis)->clipbr.x))\
    code |= OC_RIGHT;\
  if ((yy)<(LIBGGI_GC(vis)->cliptl.y))\
    code |= OC_TOP;\
  else if ((yy)>=(LIBGGI_GC(vis)->clipbr.y))\
    code |= OC_BOTTOM;\
}

/*
  Calculates |_ a/b _| with mathematically correct floor
  */
static int FloorDiv(int a, int b)
{
  int _floor;
  if (b>0) {
    if (a>0) {
      return a /b;
    } else {
      _floor = -((-a)/b);
      if ((-a)%b != 0)
	_floor--;
    }
    return _floor;
  } else {
    if (a>0) {
      _floor = -(a/(-b));
      if (a%(-b) != 0)
	_floor--;
      return _floor;
    } else {
      return (-a)/(-b);
    }
  }
}
/*
  Calculates |^ a/b ^| with mathamatically correct floor
  */
static int CeilDiv(int a,int b)
{
  if (b>0)
    return FloorDiv(a-1,b)+1;
  else
    return FloorDiv(-a-1,-b)+1;
}

static int _ggi_clip2d(ggi_visual *vis,int *_x0, int *_y0, int *_x1, int *_y1,
		       int *clip_first, int *clip_last)
{
  int first,last, code;
  int x0,y0,x1,y1;
  int x,y;
  int dx,dy;
  int xmajor;
  int slope;

  *clip_first = first = 0;
  *clip_last = last = 0;
  outcode(first,*_x0,*_y0);
  outcode(last,*_x1,*_y1);

  if ((first | last) == 0) {
    return 1; /* Trivially accepted! */
  }
  if ((first & last) != 0) {
    return 0; /* Trivially rejected! */
  }

  x0=*_x0; y0=*_y0;
  x1=*_x1; y1=*_y1;

  dx = x1 - x0;
  dy = y1 - y0;
  
  xmajor = (abs(dx) > abs(dy));
  slope = ((dx>=0) && (dy>=0)) || ((dx<0) && (dy<0));
  
  for (;;) {
    code = first;
    if (first==0)
      code = last;

    if (code&OC_LEFT) {
      x = LIBGGI_GC(vis)->cliptl.x;
      if (xmajor) {
	y = *_y0 +  FloorDiv(dy*(x - *_x0)*2 + dx,
			     2*dx);
      } else {
	if (slope) {
	  y = *_y0 + CeilDiv(dy*((x - *_x0)*2
				 - 1), 2*dx);
	} else {
	  y = *_y0 + FloorDiv(dy*((x - *_x0)*2
				  - 1), 2*dx);
	}
      }
    } else if (code&OC_RIGHT) {
      x = LIBGGI_GC(vis)->clipbr.x - 1;
      if (xmajor) {
	y = *_y0 +  FloorDiv(dy*(x - *_x0)*2 + dx, 2*dx);
      } else {
	if (slope) {
	  y = *_y0 + CeilDiv(dy*((x - *_x0)*2
				 + 1), 2*dx)-1;
	} else {
	  y = *_y0 + FloorDiv(dy*((x - *_x0)*2
				  + 1), 2*dx)+1;
	}
      }
    } else if (code&OC_TOP) {
      y = LIBGGI_GC(vis)->cliptl.y;
      if (xmajor) {
	if (slope) {
	  x = *_x0 + CeilDiv(dx*((y - *_y0)*2
				 - 1), 2*dy);
	} else {
	  x = *_x0 + FloorDiv(dx*((y - *_y0)*2
				  - 1), 2*dy);
	}
      } else {
	x = *_x0 +  FloorDiv( dx*(y - *_y0)*2 + dy,
			      2*dy);
      }
    } else { /* OC_BOTTOM */
      y = LIBGGI_GC(vis)->clipbr.y - 1;
      if (xmajor) {
	if (slope) {
	  x = *_x0 + CeilDiv(dx*((y - *_y0)*2
				 + 1), 2*dy)-1;
	} else {
	  x = *_x0 + FloorDiv(dx*((y - *_y0)*2
				  + 1), 2*dy)+1;
	}
      } else {
	x = *_x0 +  FloorDiv(dx*(y - *_y0)*2 + dy,
			     2*dy);
      }
    }

    if (first!=0) {
      x0 = x;
      y0 = y;
      outcode(first,x0,y0);
      *clip_first = 1;
    } else {
      x1 = x;
      y1 = y;
      last = code;
      outcode(last,x1,y1);
      *clip_last = 1;
    }

    if ((first & last) != 0) {
      return 0; /* Trivially rejected! */
    }
    if ((first | last) == 0) {
      *_x0=x0; *_y0=y0;
      *_x1=x1; *_y1=y1;
      return 1; /* Trivially accepted! */
    }
  }
}

/* line drawing algorithm from GGI sources, slightly modified for Q-GGI ******/

/* $Id: draw_line.c,v 1.1 2003/12/23 01:15:02 agraef Exp $
******************************************************************************

   Graphics library for GGI.

   Copyright (C) 1995 Andreas Beck        [becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan      [jmcc@ggi-project.org]
   Copyright (C) 1998 Alexander Larsson   [alla@lysator.liu.se]
   
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

typedef struct {
  ggi_visual_t vis;
} visual_t;

extern int draw_box(visual_t *v, int x, int y, int w, int h);

static inline int DrawHLine(visual_t *v, int x, int y, int w)
{
  return (draw_box(v,x,y,w,1) >= 0)?0:-1;
}

static inline int DrawVLine(visual_t *v, int x, int y, int h)
{
  return (draw_box(v,x,y,1,h) >= 0)?0:-1;
}

static inline int DrawPixel(visual_t *v, int x, int y)
{
  return (draw_box(v,x,y,1,1) >= 0)?0:-1;
}

int draw_line(visual_t *v, int orig_x1, int orig_y1,
	      int orig_x2, int orig_y2)
{
  ggi_visual *vis = v->vis;
  int orig_dx, orig_dy, sx, sy;
  int dx,dy;
  int i;
  int x1, y1, x2, y2;
  int clip_first, clip_last;

  x1 = orig_x1;
  y1 = orig_y1;
  x2 = orig_x2;
  y2 = orig_y2;

  /* clip x1,y1 and x2,y2. Set clip_first and clip_last if clipped */
  if (!_ggi_clip2d(vis, &x1,&y1,&x2,&y2,&clip_first,&clip_last)) {
    return 0; /* Clipped */
  }
  
  dy = y2 - y1;
  orig_dy = orig_y2 - orig_y1;
  sy = 1;
  if (orig_dy < 0) {
    orig_dy = -orig_dy;
    dy = -dy;
    sy = -1;
  }

  dx = x2-x1;
  orig_dx = orig_x2 - orig_x1;
  sx = 1;
  if (orig_dx < 0) {
    sx = -1;
    orig_dx = -orig_dx;
    dx = -dx;
  }

  if (dx == 0) {
    return (sy>0) ? DrawVLine(v,x1,y1,dy+1)
      : DrawVLine(v,x2,y2,dy+1);
  }

  if (dy == 0) {
    return (sx>0) ? DrawHLine(v,x1,y1,dx+1)
      : DrawHLine(v,x2,y2,dx+1);
  }

  if (orig_dx == orig_dy) {
    for (i=dx; i >= 0; i--) {
      DrawPixel(v, x1, y1);
      x1 += sx;
      y1 += sy;
    }
    return 0;
  }

  if (orig_dx >= orig_dy) { /* x major */
    int runlen,adjup,adjdown,e,len;
    int firstlen,lastlen;

    runlen = orig_dx/orig_dy;
    adjup = orig_dx%orig_dy;
    lastlen = firstlen = (runlen>>1) + 1;
    if (clip_first) { /* clipped, Adjust firstlen */
      int clip_dx = abs(x1 - orig_x1);
      int clip_dy = abs(y1 - orig_y1);
      int d = (2*clip_dy+1)*orig_dx; 
      firstlen = d/(2*orig_dy) - clip_dx + 1;
      e = d%(2*orig_dy);
      if ((e==0) && (sy>0)) {
	/* Special case, arbitrary choise.
	   Select lower pixel.(?) */
	firstlen--;
	e += 2*orig_dy;
      }
      e -= (orig_dy*2);
    } else {
      /* Not clipped, calculate start error term */
      e = adjup - (orig_dy<<1); /* initial errorterm
				   == half a step */
      if ((runlen&1) != 0) {
	e += orig_dy;
      }
    }
    if (clip_last) { /* Last endpoint clipped */
      int clip_dx = abs(x2 - orig_x2);
      int clip_dy = abs(y2 - orig_y2);
      int d = (1+2*clip_dy)*orig_dx; 
      lastlen = d/(2*orig_dy) - clip_dx + 1;
      if ((sy<0) && ((d%(2*orig_dy))==0)) {
	/* special arbitrary case */
	lastlen--; 
      }
    }
    adjup <<= 1;
    adjdown = orig_dy<<1;

    if (sy>0) {  /* line goes down */
      if ((adjup==0) && ((runlen&1)==0) && (!clip_first)) {
	firstlen--;
      }
      if (sx>0) { /* line goes right */
	DrawHLine(v,x1,y1,firstlen);
	x1 += firstlen; y1 ++;
	for (i=dy-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>0) {
	    len++;
	    e -= adjdown;
	  }
	  DrawHLine(v,x1,y1,len);
	  x1 += len; y1++;
	}
	DrawHLine(v,x1,y1,lastlen);
	return 0;
      } else { /* line goes left */
	x1++; /* because ggiDrawHLine draws right */
	x1 -= firstlen;
	DrawHLine(v,x1,y1,firstlen);
	y1++;
	for (i=dy-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>0) {
	    len++;
	    e -= adjdown;
	  }
	  x1 -= len;
	  DrawHLine(v,x1,y1,len);
	  y1 ++;
	} 
	x1 -= lastlen;
	DrawHLine(v,x1,y1,lastlen);
	return 0;
      }
    } else { /* line goes up */
      if ((adjup==0) && ((runlen&1)==0) && (!clip_last)) { 
	lastlen--;
      }
      if (sx>0) { /* line goes right */
	DrawHLine(v,x1,y1,firstlen);
	x1 += firstlen; y1--;
	for (i=dy-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>=0) {
	    len++;
	    e -= adjdown;
	  }
	  DrawHLine(v,x1,y1,len);
	  x1 += len; y1--;
	}
	DrawHLine(v,x1,y1,lastlen);
	return 0;
      } else { /* line goes left */
	x1++; /* because ggiDrawHLine draws right */
	x1 -= firstlen;
	DrawHLine(v,x1,y1,firstlen);
	y1--;
	for (i=dy-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>=0) {
	    len++;
	    e -= adjdown;
	  }
	  x1 -= len;
	  DrawHLine(v,x1,y1,len);
	  y1--;
	} 
	x1 -= lastlen;
	DrawHLine(v,x1,y1,lastlen);
	return 0;
      }
    }
  } else { /* y major */
    int runlen,adjup,adjdown,e,len;
    int firstlen,lastlen;

    runlen = orig_dy/orig_dx;
    adjup = orig_dy%orig_dx;
    
    lastlen = firstlen = (runlen>>1) + 1;
    if (clip_first) { /* clipped, Adjust firstlen */
      int clip_dx = abs(x1 - orig_x1);
      int clip_dy = abs(y1 - orig_y1);
      int d = (2*clip_dx+1)*orig_dy;
      firstlen = d/(2*orig_dx) - clip_dy + 1;
      e = d%(2*orig_dx);
      if ((e==0) && (sx>0)) {
	/* Special case, arbitrary choise.
	   Select lower pixel.(?) */
	firstlen--;
	e += 2*orig_dx;
      }
      e -= (orig_dx*2);
    } else {
      /* Not clipped, calculate start error term */
      e = adjup - (orig_dx<<1); /* initial errorterm
				   == half a step */
      if ((runlen&1) != 0) {
	e += orig_dx;
      }
    }
    if (clip_last) { /* Last endpoint clipped */
      int clip_dx = abs(x2 - orig_x2);
      int clip_dy = abs(y2 - orig_y2);
      int d = (1+2*clip_dx)*orig_dy; 
      lastlen = d/(2*orig_dx) - clip_dy + 1;
      if ((sx<0)  && ((d%(2*orig_dx))==0)) {
	/* special arbitrary case */
	lastlen--;
      }
    }
    adjup <<= 1;
    adjdown = orig_dx<<1;
    if (sy>0) { /* Line goes DOWN */
      if (sx>0) { /* line goes RIGHT */
	if ((adjup==0) && ((runlen&1)==0) &&
	    (!clip_first)) {
	  firstlen--;
	}
	DrawVLine(v,x1,y1,firstlen);
	y1 += firstlen; x1++;
	for (i=dx-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>0) {
	    len++;
	    e -= adjdown;
	  }
	  DrawVLine(v,x1,y1,len);
	  y1 += len; x1++;
	}
	DrawVLine(v,x1,y1,lastlen);
	return 0;
      } else { /* line goes LEFT */
	if ((adjup==0) && ((runlen&1)==0) &&
	    (!clip_last)) {
	  lastlen--;
	}
	DrawVLine(v,x1,y1,firstlen);
	y1 += firstlen; x1--;
	for (i=dx-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>=0) {
	    len++;
	    e -= adjdown;
	  }
	  DrawVLine(v,x1,y1,len);
	  y1 += len; x1--;
	}
	DrawVLine(v,x1,y1,lastlen);
	return 0;
      }
    } else { /* Line goes UP */
      y1++;
      if (sx>0) { /* line goes RIGHT */
	if ((adjup==0) && ((runlen&1)==0) &&
	    (!clip_first)) {
	  firstlen--;
	}
	y1 -= firstlen;
	DrawVLine(v,x1,y1,firstlen);
	x1++;
	for (i=dx-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>0) {
	    len++;
	    e -= adjdown;
	  }
	  y1 -= len;
	  DrawVLine(v,x1,y1,len);
	  x1++;
	}
	y1 -= lastlen;
	DrawVLine(v,x1,y1,lastlen);
	return 0;
      } else { /* line goes LEFT */
	if ((adjup==0) && ((runlen&1)==0) &&
	    (!clip_last)) {
	  lastlen--;
	}
	y1 -= firstlen; 
	DrawVLine(v,x1,y1,firstlen);
	x1--;
	for (i=dx-1; i>0; i--) {
	  len = runlen;
	  e += adjup;
	  if (e>=0) {
	    len++;
	    e -= adjdown;
	  }
	  y1 -= len; 
	  DrawVLine(v,x1,y1,len);
	  x1--;
	}
	y1 -= lastlen; 
	DrawVLine(v,x1,y1,lastlen);
	return 0;
      }
    }
  }
}

/*****************************************************************************/
