
/*  Q eQuational Programming System
    Copyright (c) 1991-2002 by Albert Graef
    <ag@muwiinfa.geschichte.uni-mainz.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "qbase.h"

char *prompt = PROMPT, *prompt2 = "> ", *prompt3 = ": ", *qpath = NULL,
  *histfile = HISTFILE, *initrc = INITRC, *exitrc = EXITRC,
  *default_codeset = NULL, *prelude = "prelude.q";
char fformat[30];
char version[] = VERSION, year[] = YEAR, sysinfo[] = SYSINFO,
  dirstr[] = DIRSTR, volstr[] = VOLSTR;
int histmax = HISTMAX,
  memmax = MEMMAX, stackmax = STACKMAX, cstackmax = CSTACKMAX,
  imode = 0, fmode = 0, fprec = 15;

char outid[MAXSTRLEN];
int mainno;
int codespsz;
int strspsz;
int limbspsz;
int hashtbsz = HASHTBSZ;
int symtbsz;
int statetbsz;
int transtbsz;
int roffstbsz;
int matchtbsz;
int inittbsz;
int modtbsz;
int imptbsz;

OPREC *codesp;
char *strsp;
mp_limb_t *limbsp;
int *hashtb;
SYMREC *symtb;
STATEREC *statetb;
TRANSREC *transtb;
int *roffstb;
int *matchtb;
int *inittb;
int *modtb;
int *fnametb;
int *imports;
int *imptb;
byte *impib;

byte *globs;

char           *
charstr(char *s, char c)
{
  s[0] = c;
  s[1] = '\0';
  return (s);
}

char           *
substr(char *s, char *t, int n)
{
  *s = 0;
  strncat(s, t, n);
  s[n] = 0;
  return (s);
}

/* Thread-safe, locale-independent routines to convert between double
   precision floating point numbers and strings. These were pilfered from
   GNOME glib, and massaged a little to make them compile without the glib
   infrastructure. Source: glib/gstrfuncs.c in the glib tarball, distributed
   under GPL V2, available at http://www.gnome.org. */

#if defined(HAVE_UNICODE) && defined(HAVE_LOCALE_H)

/* Functions like the ones in <ctype.h> that are not affected by locale. */
typedef enum {
  ASCII_ALNUM  = 1 << 0,
  ASCII_ALPHA  = 1 << 1,
  ASCII_CNTRL  = 1 << 2,
  ASCII_DIGIT  = 1 << 3,
  ASCII_GRAPH  = 1 << 4,
  ASCII_LOWER  = 1 << 5,
  ASCII_PRINT  = 1 << 6,
  ASCII_PUNCT  = 1 << 7,
  ASCII_SPACE  = 1 << 8,
  ASCII_UPPER  = 1 << 9,
  ASCII_XDIGIT = 1 << 10
} AsciiType;

static const unsigned short ascii_table[256] = {
  0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
  0x004, 0x104, 0x104, 0x004, 0x104, 0x104, 0x004, 0x004,
  0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
  0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
  0x140, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0,
  0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0,
  0x459, 0x459, 0x459, 0x459, 0x459, 0x459, 0x459, 0x459,
  0x459, 0x459, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0,
  0x0d0, 0x653, 0x653, 0x653, 0x653, 0x653, 0x653, 0x253,
  0x253, 0x253, 0x253, 0x253, 0x253, 0x253, 0x253, 0x253,
  0x253, 0x253, 0x253, 0x253, 0x253, 0x253, 0x253, 0x253,
  0x253, 0x253, 0x253, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x0d0,
  0x0d0, 0x473, 0x473, 0x473, 0x473, 0x473, 0x473, 0x073,
  0x073, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073,
  0x073, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073, 0x073,
  0x073, 0x073, 0x073, 0x0d0, 0x0d0, 0x0d0, 0x0d0, 0x004
  /* the upper 128 are all zeroes */
};

#define ascii_isalnum(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_ALNUM) != 0)

#define ascii_isalpha(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_ALPHA) != 0)

#define ascii_iscntrl(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_CNTRL) != 0)

#define ascii_isdigit(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_DIGIT) != 0)

#define ascii_isgraph(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_GRAPH) != 0)

#define ascii_islower(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_LOWER) != 0)

#define ascii_isprint(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_PRINT) != 0)

#define ascii_ispunct(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_PUNCT) != 0)

#define ascii_isspace(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_SPACE) != 0)

#define ascii_isupper(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_UPPER) != 0)

#define ascii_isxdigit(c) \
  ((ascii_table[(unsigned char) (c)] & ASCII_XDIGIT) != 0)

/*
 * This function behaves like the standard strtod() function does in the C
 * locale. It does this without actually changing the current locale, since
 * that would not be thread-safe.
 */
double
my_strtod (const char  *nptr,
	   char       **endptr)
{
  char *fail_pos;
  double val;
  struct lconv *locale_data;
  const char *decimal_point;
  int decimal_point_len;
  const char *p, *decimal_point_pos;
  const char *end = NULL; /* Silence gcc */
  int strtod_errno;

  fail_pos = NULL;

  locale_data = localeconv ();
  decimal_point = locale_data->decimal_point;
  decimal_point_len = strlen (decimal_point);

  /* assert (decimal_point_len != 0); */
  
  decimal_point_pos = NULL;
  end = NULL;

  if (decimal_point[0] != '.' || 
      decimal_point[1] != 0)
    {
      p = nptr;
      /* Skip leading space */
      while (ascii_isspace (*p))
	p++;
      
      /* Skip leading optional sign */
      if (*p == '+' || *p == '-')
	p++;
      
      if (p[0] == '0' && 
	  (p[1] == 'x' || p[1] == 'X'))
	{
	  p += 2;
	  /* HEX - find the (optional) decimal point */
	  
	  while (ascii_isxdigit (*p))
	    p++;
	  
	  if (*p == '.')
	    decimal_point_pos = p++;
	      
	  while (ascii_isxdigit (*p))
	    p++;
	  
	  if (*p == 'p' || *p == 'P')
	    p++;
	  if (*p == '+' || *p == '-')
	    p++;
	  while (ascii_isdigit (*p))
	    p++;

	  end = p;
	}
      else if (ascii_isdigit (*p) || *p == '.')
	{
	  while (ascii_isdigit (*p))
	    p++;
	  
	  if (*p == '.')
	    decimal_point_pos = p++;
	  
	  while (ascii_isdigit (*p))
	    p++;
	  
	  if (*p == 'e' || *p == 'E')
	    p++;
	  if (*p == '+' || *p == '-')
	    p++;
	  while (ascii_isdigit (*p))
	    p++;

	  end = p;
	}
      /* For the other cases, we need not convert the decimal point */
    }

  if (decimal_point_pos)
    {
      char *copy, *c;

      /* We need to convert the '.' to the locale specific decimal point */
      copy = malloc (end - nptr + 1 + decimal_point_len);

      /* assert (copy != NULL); */
      
      c = copy;
      memcpy (c, nptr, decimal_point_pos - nptr);
      c += decimal_point_pos - nptr;
      memcpy (c, decimal_point, decimal_point_len);
      c += decimal_point_len;
      memcpy (c, decimal_point_pos + 1, end - (decimal_point_pos + 1));
      c += end - (decimal_point_pos + 1);
      *c = 0;

      errno = 0;
      val = strtod (copy, &fail_pos);
      strtod_errno = errno;

      if (fail_pos)
	{
	  if (fail_pos - copy > decimal_point_pos - nptr)
	    fail_pos = (char *)nptr + (fail_pos - copy) - (decimal_point_len - 1);
	  else
	    fail_pos = (char *)nptr + (fail_pos - copy);
	}
      
      free (copy);
	  
    }
  else if (end)
    {
      char *copy;
      
      copy = malloc (end - (char *)nptr + 1);
      /* assert (copy != NULL); */
      memcpy (copy, nptr, end - nptr);
      *(copy + (end - (char *)nptr)) = 0;
      
      errno = 0;
      val = strtod (copy, &fail_pos);
      strtod_errno = errno;

      if (fail_pos)
	{
	  fail_pos = (char *)nptr + (fail_pos - copy);
	}
      
      free (copy);
    }
  else
    {
      errno = 0;
      val = strtod (nptr, &fail_pos);
      strtod_errno = errno;
    }

  if (endptr)
    *endptr = fail_pos;

  errno = strtod_errno;

  return val;
}

/*
 * Converts a double to a string, using the '.' as decimal point. To format
 * the number you pass in a printf()-style format string. Allowed conversion
 * specifiers are 'e', 'E', 'f', 'F', 'g' and 'G'.
 */
char *
my_formatd (char        *buffer,
	    const char  *format,
	    double       d)
{
  struct lconv *locale_data;
  const char *decimal_point;
  int decimal_point_len;
  char *p;
  int rest_len;
  char format_char;

  format_char = format[strlen (format) - 1];
  
  if (format[0] != '%')
    return NULL;

  if (strpbrk (format + 1, "'l%"))
    return NULL;

  if (!(format_char == 'e' || format_char == 'E' ||
	format_char == 'f' || format_char == 'F' ||
	format_char == 'g' || format_char == 'G'))
    return NULL;

  sprintf (buffer, format, d);

  locale_data = localeconv ();
  decimal_point = locale_data->decimal_point;
  decimal_point_len = strlen (decimal_point);

  /* assert (decimal_point_len != 0); */

  if (decimal_point[0] != '.' ||
      decimal_point[1] != 0)
    {
      p = buffer;

      if (*p == '+' || *p == '-')
	p++;

      while (ascii_isdigit ((unsigned char)*p))
	p++;

      if (strncmp (p, decimal_point, decimal_point_len) == 0)
	{
	  *p = '.';
	  p++;
	  if (decimal_point_len > 1) {
	    rest_len = strlen (p + (decimal_point_len-1));
	    memmove (p, p + (decimal_point_len-1),
		     rest_len);
	    p[rest_len] = 0;
	    
	  }
	}
    }
  
  return buffer;
}

#else

double
my_strtod (const char  *nptr,
	   char       **endptr)
{
  return strtod(nptr, endptr);
}

char *
my_formatd (char        *buffer,
	    const char  *format,
	    double       d)
{
  sprintf (buffer, format, d);
  return buffer;
}

#endif

#ifdef HAVE_UNICODE

#ifndef HAVE_LANGINFO_CODESET

/* simplistic emulation of nl_langinfo(CODESET) on POSIX systems which don't
   have it (like older FreeBSD releases), pilfered from
   http://www.cl.cam.ac.uk/~mgk25/ucs */

/*
 * This is a quick-and-dirty emulator of the nl_langinfo(CODESET)
 * function defined in the Single Unix Specification for those systems
 * (FreeBSD, etc.) that don't have one yet. It behaves as if it had
 * been called after setlocale(LC_CTYPE, ""), that is it looks at
 * the locale environment variables.
 *
 * http://www.opengroup.org/onlinepubs/7908799/xsh/langinfo.h.html
 *
 * Please extend it as needed and suggest improvements to the author.
 * This emulator will hopefully become redundant soon as
 * nl_langinfo(CODESET) becomes more widely implemented.
 *
 * Since the proposed Li18nux encoding name registry is still not mature,
 * the output follows the MIME registry where possible:
 *
 *   http://www.iana.org/assignments/character-sets
 *
 * A possible autoconf test for the availability of nl_langinfo(CODESET)
 * can be found in
 *
 *   http://www.cl.cam.ac.uk/~mgk25/unicode.html#activate
 *
 * Markus.Kuhn@cl.cam.ac.uk -- 2002-03-11
 * Permission to use, copy, modify, and distribute this software
 * for any purpose and without fee is hereby granted. The author
 * disclaims all warranties with regard to this software.
 *
 * Latest version:
 *
 *   http://www.cl.cam.ac.uk/~mgk25/ucs/langinfo.c
 */

typedef int nl_item;

#define MYCODESET ((nl_item) 1)

#define C_CODESET "US-ASCII"     /* Return this as the encoding of the
				  * C/POSIX locale. Could as well one day
				  * become "UTF-8". */

#define digit(x) ((x) >= '0' && (x) <= '9')

static char *my_nl_langinfo(nl_item item)
{
  static char buf[16];
  char *l, *p;
  
  if (item != MYCODESET)
    return NULL;
  
  if (((l = getenv("LC_ALL"))   && *l) ||
      ((l = getenv("LC_CTYPE")) && *l) ||
      ((l = getenv("LANG"))     && *l)) {
    /* check standardized locales */
    if (!strcmp(l, "C") || !strcmp(l, "POSIX"))
      return C_CODESET;
    /* check for encoding name fragment */
    if (strstr(l, "UTF") || strstr(l, "utf"))
      return "UTF-8";
    if ((p = strstr(l, "8859-"))) {
      memcpy(buf, "ISO-8859-\0\0", 12);
      p += 5;
      if (digit(*p)) {
	buf[9] = *p++;
	if (digit(*p)) buf[10] = *p++;
	return buf;
      }
    }
    if (strstr(l, "KOI8-R")) return "KOI8-R";
    if (strstr(l, "KOI8-U")) return "KOI8-U";
    if (strstr(l, "620")) return "TIS-620";
    if (strstr(l, "2312")) return "GB2312";
    if (strstr(l, "HKSCS")) return "Big5HKSCS";   /* no MIME charset */
    if (strstr(l, "Big5") || strstr(l, "BIG5")) return "Big5";
    if (strstr(l, "GBK")) return "GBK";           /* no MIME charset */
    if (strstr(l, "18030")) return "GB18030";     /* no MIME charset */
    if (strstr(l, "Shift_JIS") || strstr(l, "SJIS")) return "Shift_JIS";
    /* check for conclusive modifier */
    if (strstr(l, "euro")) return "ISO-8859-15";
    /* check for language (and perhaps country) codes */
    if (strstr(l, "zh_TW")) return "Big5";
    if (strstr(l, "zh_HK")) return "Big5HKSCS";   /* no MIME charset */
    if (strstr(l, "zh")) return "GB2312";
    if (strstr(l, "ja")) return "EUC-JP";
    if (strstr(l, "ko")) return "EUC-KR";
    if (strstr(l, "ru")) return "KOI8-R";
    if (strstr(l, "uk")) return "KOI8-U";
    if (strstr(l, "pl") || strstr(l, "hr") ||
	strstr(l, "hu") || strstr(l, "cs") ||
	strstr(l, "sk") || strstr(l, "sl")) return "ISO-8859-2";
    if (strstr(l, "eo") || strstr(l, "mt")) return "ISO-8859-3";
    if (strstr(l, "el")) return "ISO-8859-7";
    if (strstr(l, "he")) return "ISO-8859-8";
    if (strstr(l, "tr")) return "ISO-8859-9";
    if (strstr(l, "th")) return "TIS-620";      /* or ISO-8859-11 */
    if (strstr(l, "lt")) return "ISO-8859-13";
    if (strstr(l, "cy")) return "ISO-8859-14";
    if (strstr(l, "ro")) return "ISO-8859-2";   /* or ISO-8859-16 */
    if (strstr(l, "am") || strstr(l, "vi")) return "UTF-8";
    /* Send me further rules if you like, but don't forget that we are
     * *only* interested in locale naming conventions on platforms
     * that do not already provide an nl_langinfo(CODESET) implementation. */
    return "ISO-8859-1"; /* should perhaps be "UTF-8" instead */
  }
  return C_CODESET;
}

#endif

static inline long
u8decode(char *s, char **t)
{
  size_t n = 0;
  unsigned p = 0, q = 0;
  unsigned long c = 0;
  if (s[0] == 0)
    return -1;
  else if (s[1] == 0) {
    *t = s+1;
    return (unsigned char)s[0];
  }
  for (; n == 0 && *s; s++) {
    unsigned char uc = (unsigned char)*s;
    if (q == 0) {
      if (((signed char)uc) < 0) {
	switch (uc & 0xf0) {
	case 0xc0: case 0xd0:
	  q = 1;
	  c = uc & 0x1f;
	  break;
	case 0xe0:
	  q = 2;
	  c = uc & 0xf;
	  break;
	case 0xf0:
	  if ((uc & 0x8) == 0) {
	    q = 3;
	    c = uc & 0x7;
	  } else
	    c = uc;
	  break;
	default:
	  c = uc;
	  break;
	}
      } else
	c = uc;
      p = 0; if (q == 0) n++;
    } else if ((uc & 0xc0) == 0x80) {
      /* continuation byte */
      c = c << 6 | (uc & 0x3f);
      if (--q == 0)
	n++;
      else
	p++;
    } else {
      /* malformed char */
      return -1;
    }
  }
  if (n == 1) {
    *t = s;
    return c;
  } else
    return -1;
}

static inline char *
u8encode(char *t, unsigned long c)
{
  unsigned char *uc = (unsigned char*)t;
  if (c < 0x80) {
    uc[1] = 0;
    uc[0] = c;
  } else if (c < 0x800) {
    uc[2] = 0;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xc0 | c;
  } else if (c < 0x10000) {
    uc[3] = 0;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xe0 | c;
  } else {
    uc[4] = 0;
    uc[3] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[2] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[1] = 0x80 | c&0x3f;
    c = c >> 6;
    uc[0] = 0xf0 | c;
  }
  return t;
}

#ifdef HAVE_ICONV

#define CHUNKSZ 128

static inline char *
toutf8(char *s, char *codeset)
{
  iconv_t ic;
  if (!codeset || !*codeset)
    codeset = default_encoding();
  if (codeset && strcmp(codeset, "UTF-8"))
    ic = iconv_open("UTF-8", codeset);
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return NULL;

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

static inline char *
fromutf8(char *s, char *codeset)
{
  iconv_t ic;
  if (!codeset || !*codeset)
    codeset = default_encoding();
  if (codeset && strcmp(codeset, "UTF-8"))
    ic = iconv_open(codeset, "UTF-8");
  else
    ic = (iconv_t)-1;

  if (ic == (iconv_t)-1)
    return NULL;

  else {
    size_t l = strlen(s);
    char *t = malloc(l+1), *t1;
    char *inbuf = s, *outbuf = t;
    size_t inbytes = l, outbytes = l;

    while (iconv(ic, &inbuf, &inbytes, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }

    /* here we might have to deal with a stateful encoding, so make sure that
       we emit the closing shift sequence */

    while (iconv(ic, NULL, NULL, &outbuf, &outbytes) ==
	   (size_t)-1)
      if (errno == E2BIG) {
	/* try to enlarge the output buffer */
	size_t k = outbuf-t;
	if ((t1 = realloc(t, l+CHUNKSZ+1))) {
	  t = t1;
	  outbuf = t+k;
	  l += CHUNKSZ;
	  outbytes += CHUNKSZ;
	} else {
	  /* memory overflow */
	  free(t);
	  return NULL;
	}
      } else {
	/* conversion error */
	free(t);
	return NULL;
      }

    /* terminate the output string */
    *outbuf = 0;
    iconv_close(ic);

    if (!(t1 = realloc(t, strlen(t)+1)))
      /* this shouldn't happen */
      return t;
    else
      return t1;
  }
}

/* conversion between UTF-8 and wchar_t */

static iconv_t myic[2] = { (iconv_t)-1, (iconv_t)-1 };

static inline wchar_t *
ictowcs(wchar_t *t, char *s)
{
  if (myic[1] == (iconv_t)-1)
    myic[1] = iconv_open("WCHAR_T", "UTF-8");
  if (myic[1] == (iconv_t)-1)
    return NULL;
  else {
    size_t l = strlen(s);
    char *inbuf = s; wchar_t *outbuf = t;
    size_t inbytes = l, outbytes = l*sizeof(wchar_t);

    if (iconv(myic[1], &inbuf, &inbytes, (char**)&outbuf, &outbytes) ==
	(size_t)-1)
      return NULL;
    /* terminate the output string */
    *outbuf = 0;
    return t;
  }
}

#endif

#ifdef __STDC_ISO_10646__
#define MY_ISO_10646
#else
#ifndef HAVE_ICONV
#warning "wchar_t encoding unknown and iconv not available, assuming ISO 10646"
#define MY_ISO_10646
#endif
#endif

#ifdef MY_ISO_10646
#define towchar(c) ((wchar_t)(c))
#else
static wchar_t towchar(unsigned long c)
{
  char s[5]; /* single utf-8 char can have at most 4 bytes, plus terminal 0 */
  wchar_t t[5]; /* just to be safe; 2 should actually be enough */
  u8encode(s, c);
  if (ictowcs(t, s))
    return t[0];
  else
    /* Your system is so utterly broken that we can't even convert UTF-8 to
       wchar_t. You should probably configure with --without-unicode. But
       let's just pretend we have an ISO 10646 compatible encoding anyway. */
    return (wchar_t)c;
}
#endif

#endif

char *default_encoding(void)
{
#ifdef HAVE_UNICODE
#ifdef HAVE_LANGINFO_CODESET
  /* use nl_langinfo() if it's available */
  return nl_langinfo(CODESET);
#else
#ifdef _WIN32
  /* always use the ANSI codepage on Windows (this might cause lossage if your
     system uses a different OEM codepage!?) */
  static char buf[20];
  sprintf(buf, "cp%d", GetACP());
  return buf;
#else
  /* use our own emulation of nl_langinfo() */
  return my_nl_langinfo(CODESET);
#endif /* _WIN32 */
#endif /* HAVE_LANGINFO_CODESET */
#else
  /* no unicode support */
  return NULL;
#endif /* HAVE_UNICODE */
}

char *utf8_to_sys(char *s)
{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  static char *buf1 = NULL, *buf2 = NULL;
  char *tmp = fromutf8(s, NULL);
  if (tmp) {
    if (buf1) free(buf1);
    buf1 = buf2;
    buf2 = tmp;
    return tmp;
  } else
#endif
    return s;
}

char *sys_to_utf8(char *s)
{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  static char *buf1 = NULL, *buf2 = NULL;
  char *tmp = toutf8(s, NULL);
  if (tmp) {
    if (buf1) free(buf1);
    buf1 = buf2;
    buf2 = tmp;
    return tmp;
  } else
#endif
    return s;
}

char *utf8_to_sys_dup(char *s)
{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  char *tmp = fromutf8(s, NULL);
  if (tmp)
    return tmp;
  else
#endif
    return strdup(s);
}

char *sys_to_utf8_dup(char *s)
{
#if defined(HAVE_UNICODE) && defined(HAVE_ICONV)
  char *tmp = toutf8(s, NULL);
  if (tmp)
    return tmp;
  else
#endif
    return strdup(s);
}

static char *
pchar(char *s, long c, char d)
{
  switch (c) {
  case '\t':
    return (strcpy(s, "\\t"));
  case '\b':
    return (strcpy(s, "\\b"));
  case '\f':
    return (strcpy(s, "\\f"));
  case '\n':
    return (strcpy(s, "\\n"));
  case '\r':
    return (strcpy(s, "\\r"));
  case '\\':
    return (strcpy(s, "\\\\"));
  case '\"':
    return (strcpy(s, "\\\""));
  default:
#ifdef HAVE_UNICODE
    if (iswprint(towchar(c))) {
      u8encode(s, c);
#else
    if (isprint((char)c)) {
      s[0] = (char)c;
      s[1] = 0;
#endif
      return (s);
    } else {
      if (d != 0 && isxdigit(d))
	sprintf(s, "\\(0x%lx)", c);
      else
	sprintf(s, "\\0x%lx", c);
      return (s);
    }
  }
}

char           *
pstr(char *s1, char *s2)
{
  char           *s, *t;

  *s1 = '\0';
  for (s = s1; *s2; s2 = t) {
#ifdef HAVE_UNICODE
    long c = u8decode(s2, &t);
    if (c < 0) {
      c = (unsigned char)*s2;
      t = s2+1;
    }
#else
    long c = (unsigned char)*s2;
    t = s2+1;
#endif
    s += strlen(pchar(s, c, *t));
  }
  return (s1);
}

char *
pmpz(char *s, int imode, mpz_t z)
{
  bool neg = z->_mp_size < 0;
  char *_s = s;
  if (neg) {
    z->_mp_size = -z->_mp_size;
    *_s++ = '-';
  }
  switch (imode) {
  case 1:
    strcpy(_s, "0x");
    mpz_get_str(_s+2, 16, z);
    break;
  case 2:
    strcpy(_s, "0");
    mpz_get_str(_s+1, 8, z);
    break;
  default:
    mpz_get_str(_s, 10, z);
    break;
  }
  if (neg) z->_mp_size = -z->_mp_size;
  return s;
}

/* Quick and dirty checks for IEEE floating point infinities and NaNs. Thanks
   to John Cowan. */

static inline int is_nan(double f) {
  return !(f == f);
}

static inline int is_inf(double f) {
  return (!is_nan(f) && is_nan(f-f));
}

char *
pfloat(char *s, char *fformat, double f)
{
  char *t;
  if (is_inf(f))
    if (f > 0)
      strcpy(s, "inf");
    else
      strcpy(s, "-inf");
  else if (is_nan(f))
    strcpy(s, "nan");
  else
    my_formatd(s, fformat, f);
  if (!isalpha(s[0]) && (s[0] != '-' || !isalpha(s[1]))) {
    /* not inf or nan */
    if (!(t = strchr(s, 'e')) && !(t = strchr(s, 'E')) &&
	!(t = strchr(s, '.')))
      /* looks like an int, add decimal point */
      strcat(s, ".0");
  }
  return s;
}

char *
pname(char *s, int fno)
{
  if (fno < BINARY)
    strcpy(s, strsp + symtb[fno].pname);
  else if (visible(fno) && unique(fno))
    if (symtb[fno].prec != NONE)
      sprintf(s, "(%s)", strsp + symtb[fno].pname);
    else
      strcpy(s, strsp + symtb[fno].pname);
  else if (symtb[fno].modno == NONE)
    if (symtb[fno].prec != NONE)
      sprintf(s, "(::%s)", strsp + symtb[fno].pname);
    else
      sprintf(s, "::%s", strsp + symtb[fno].pname);
  else
    if (symtb[fno].prec != NONE)
      sprintf(s, "(%s::%s)", strsp + modtb[symtb[fno].modno],
	      strsp + symtb[fno].pname);
    else
      sprintf(s, "%s::%s", strsp + modtb[symtb[fno].modno],
	      strsp + symtb[fno].pname);
  return s;
}

#include "w3centities.c"

static int entcmp(const void *_x, const void *_y)
{
  const char *x = (const char*)_x;
  const Entity *y = (const Entity*)_y;
  return strcmp(x, y->name);
}

static inline long entity(const char *name)
{
  Entity *x = bsearch(name, entities, sizeof(entities) / sizeof(Entity),
		      sizeof(Entity), entcmp);
  return x?x->c:-1;
}

static long parse_entity(char *s, char **t)
{
  long c;
  char *p = strchr(s, ';');
  if (!p) return -1;
  *p = 0;
  c = entity(s);
  *p = ';';
  if (c < 0) return c;
  *t = p+1;
  return c;
}

static char *
scanchar(char *t, char **s, char **p)
{
  char            c;

  /* scan character at the head of *s, advance s accordingly: */

  *p = NULL;
  if (!**s)
    return NULL;
  else if ((c = *(*s)++) == '\\') {
    switch (c = *(*s)++) {
    case 't':
      *t++ = '\t';
      break;
    case 'b':
      *t++ = '\b';
      break;
    case 'f':
      *t++ = '\f';
      break;
    case 'n':
      *t++ = '\n';
      break;
    case 'r':
      *t++ = '\r';
      break;
    case '"': case '\\':
      *t++ = c;
      break;
    case '&': {
      char *r;
      long c = parse_entity(*s, s);
      if (c >= 0) {
#ifdef HAVE_UNICODE
	if (c >= 0x110000)
	  c %=  0x110000;
	u8encode(t, c);
	t += strlen(t);
#else
	*t++ = (char)c;
#endif
      } else {
	*p = *s-1;
	*t++ = c;
      }
      break;
    }
    case '(': {
      if ('0' <= **s && **s <= '9') {
	char *r;
	long c = strtol(*s, &r, 0);
	if (*r == ')') {
	  *s = r+1;
#ifdef HAVE_UNICODE
	  if (c >= 0x110000)
	    c %=  0x110000;
	  u8encode(t, c);
	  t += strlen(t);
#else
	  *t++ = (char)c;
#endif
	} else {
	  *p = *s-1;
	  *t++ = '(';
	}
      } else {
	*p = *s-1;
	*t++ = c;
      }
      break;
    }
    default:
      if ('0' <= c && c <= '9') {
	long c = strtol(--*s, s, 0);
#ifdef HAVE_UNICODE
	if (c >= 0x110000)
	  c %=  0x110000;
	u8encode(t, c);
	t += strlen(t);
#else
	*t++ = (char)c;
#endif
      } else {
	*p = *s-1;
	*t++ = c;
      }
      break;
    }
  } else
    *t++ = c;
  return t;
}

char           *
scanstr(char *s1, char *s2)
{
  char           *s = s1;

  while (*s2) {
    char *p;
    s = scanchar(s, &s2, &p);
    if (p) {
      p[-1] = 0;
      s1 = NULL;
    }
  }
  *s = 0;
  return s1;
}

static char	       *
home(void)
{
  static char *homedir = NULL;
  if (!homedir && !(homedir = getenv("HOME"))) {
    homedir = strdup("/");
    *homedir = *dirstr;
  }
  return homedir;
}

#define tilde(s) (s[0] == '~' && (!s[1] || strchr(dirstr, s[1]) && !strchr(volstr, s[1])))

int
absolute(char *s)
{
  char *t = s;
  if (!s || !*s)
    return 0;
  else if (tilde(s))
    return 1;
  else {
    while (*s && !strchr(dirstr, *s)) ++s;
    return *s && (s == t || strchr(volstr, *s));
  }
}

int
dirprefix(char *s, char *prefix)
{
  int l = strlen(prefix);
  return s && *s && strncmp(s, prefix, l) == 0 &&
    (!s[l] || strchr(dirstr, s[l]) && !strchr(volstr, s[l]));
}

char *
dirname(char *t, char *s)
{
  char *s1, *s2 = NULL;
  for (s1 = s; *s1; s1++)
    if (strchr(dirstr, *s1))
      s2 = s1+1;
  if (s2) {
    strncpy(t, s, s2-s);
    t[s2-s] = 0;
  } else
    *t = 0;
  return t;
}

char *
basename(char *t, char *s, char c)
{
  char *s1, *s2;
  for (s1 = s2 = s; *s1; s1++)
    if (strchr(dirstr, *s1))
      s2 = s1+1;
  if ((s1 = strchr(strcpy(t, s2), c)))
    *s1 = 0;
  return t;
}

char *
absname(char *t, char *s)
{
  if (absolute(s))
    strcpy(t, s);
  else {
    if (!getcwd(t, MAXSTRLEN))
      strcpy(t, s);
    else {
      int l = strlen(t);
      if (l <= 1 || !strchr(dirstr, t[l-1]))
	t[l++] = *dirstr;
      strcpy(t+l, s);
    }
  }
  return t;
}

char           *
expand(char *s1, char *s2)
{
  if (tilde(s2)) {
    char *h = home();
    int l = strlen(h);
    strcpy(s1, h);
    if (l > 0 && strchr(dirstr, h[l-1]))
      strcpy(s1+l, s2+2);
    else
      strcpy(s1+l, s2+1);
  } else
    strcpy(s1, s2);
  return s1;
}

int
chkfile(char *s)
{
  struct stat st;
  return !stat(s, &st) && !S_ISDIR(st.st_mode);
}

char           *
searchlib(char *s1, char *s2)
{
  char           *s, *t;

  if (tilde(s2))
    return expand(s1, s2);
  else if (absolute(s2) || dirprefix(s2, ".") || dirprefix(s2, ".."))
    return strcpy(s1, s2);
  for (s = qpath; *s; s = t) {
    int l;
    char p[MAXSTRLEN];
    if (!(t = strchr(s, PATHDELIM)))
      t = strchr(s, 0);
    if (s == t) goto next;
    if (s[0] == '.')
      if (t == s+1)
	s = t;
      else if (strchr(dirstr, s[1]) &&
	       !strchr(volstr, s[1]))
	s += 2;
    l = t-s;
    strncpy(p, s, l);
    p[l] = 0;
    expand(s1, p);
    l = strlen(s1);
    if (l > 0 && (!strchr(dirstr, s1[l-1]) || 
		  strchr(volstr, s1[l-1])))
      s1[l] = *dirstr, l++;
    strcpy(s1+l, s2);
    if (chkfile(s1))
      return s1;
  next:
    if (*t) t++;
  }
  return strcpy(s1, s2);
}

void
setpath(PATH *p, int i, int v)
{
  if (v)
    *p = *p | ((PATH) 1 << i);
  else
    *p = *p & ~((PATH) 1 << i);
}

int
getint(mpz_t z, int len, int l)
{
  int sz = (len>=0)?len:-len;
  mpz_init(z);
  if (z->_mp_d)
    if (sz > 0) {
      mpz_t z1;
      memcpy(z1, z, sizeof(mpz_t));
      if (_mpz_realloc(z, sz)) {
	memcpy(z->_mp_d, limbsp+l, sz*sizeof(mp_limb_t));
	z->_mp_size = len;
	return 1;
      } else {
	mpz_clear(z1);
	return 0;
      }
    } else
      return 1;
  else
    return 0;
}

void *
arealloc(void *ptr, int nelems, int newelems, int size)
{
  if (size == 0 || newelems == 0)
    return ptr;
  else if (nelems > INT_MAX - newelems || nelems + newelems > INT_MAX/size)
    return NULL;
  else if (ptr)
    return (void*) realloc(ptr, (nelems + newelems) * size);
  else
    return (void*) malloc((nelems + newelems) * size);
}

void init_qpath(char *s)
{
  if (!s) return;
  if (qpath) free(qpath);
  qpath = strdup(s);
}

void change_qpath(char *s)
{
  char *qpath1;
  int l;
  if (!s) return;
  if (!qpath) {
    init_qpath(s);
    return;
  }
  if (*s == PATHDELIM) {
    qpath1 = malloc(strlen(s)+strlen(qpath)+1);
    if (!qpath1) return;
    strcat(strcpy(qpath1, qpath), s);
    free(qpath);
    qpath = qpath1;
  }
  else if ((l = strlen(s)) > 0 && s[l-1] == PATHDELIM) {
    qpath1 = malloc(strlen(s)+strlen(qpath)+1);
    if (!qpath1) return;
    strcat(strcpy(qpath1, s), qpath);
    free(qpath);
    qpath = qpath1;
  } else
    init_qpath(s);
}
