/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include "rast/config.h"

#include "rast/local_db.h"

#ifdef HAVE_XMLRPC
#include "rast/xmlrpc_client.h"
#endif

static apr_pool_t *functions_pool = NULL;

static apr_array_header_t *open_functions = NULL;

static void
initialize_functions_pool()
{
    if (functions_pool == NULL) {
        apr_pool_create(&functions_pool, rast_get_global_pool());
    }
}

void
rast_add_open_function(rast_open_func_t *open_func)
{
    if (open_functions == NULL) {
        initialize_functions_pool();
        open_functions = apr_array_make(functions_pool, 1,
                                        sizeof(rast_open_func_t *));
    }

    *(rast_open_func_t **) apr_array_push(open_functions) = open_func;
}

rast_db_create_option_t *
rast_db_create_option_create(apr_pool_t *pool)
{
    rast_db_create_option_t *options;

    options = (rast_db_create_option_t *)
        apr_palloc(pool, sizeof(rast_db_create_option_t));
    options->encoding = RAST_DEFAULT_ENCODING;
    options->byte_order = RAST_NATIVE_ENDIAN;
    options->preserve_text = 0;
    options->properties = NULL;
    options->num_properties = 0;
    options->pos_block_size = 512;
    return options;
}

rast_db_open_option_t *
rast_db_open_option_create(apr_pool_t *pool)
{
    rast_db_open_option_t *options;
    options = (rast_db_open_option_t *)
        apr_palloc(pool, sizeof(rast_db_open_option_t));
    options->sync_threshold_chars = 100000;
    return options;
}

rast_db_optimize_option_t *
rast_db_optimize_option_create(apr_pool_t *pool)
{
    rast_db_optimize_option_t *options;

    options = (rast_db_optimize_option_t *)
        apr_palloc(pool, sizeof(rast_db_optimize_option_t));
    options->squeeze_doc_id = 0;
    return options;
}

rast_error_t *
rast_document_add_text(rast_document_t *doc, const char *text, int nbytes)
{
    if (doc->add_text == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return doc->add_text(doc, text, nbytes);
}

rast_error_t *
rast_document_set_property(rast_document_t *doc, const char *name,
                           const rast_value_t *value)
{
    if (doc->set_property == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return doc->set_property(doc, name, value);
}

rast_error_t *
rast_document_commit(rast_document_t *doc)
{
    if (doc->commit == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return doc->commit(doc);
}

rast_error_t *
rast_document_abort(rast_document_t *doc)
{
    if (doc->abort == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return doc->abort(doc);
}

rast_error_t *
rast_document_get_doc_id(rast_document_t *doc, rast_doc_id_t *doc_id)
{
    if (doc->get_doc_id == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return doc->get_doc_id(doc, doc_id);
}

rast_error_t *
rast_db_create(const char *name, rast_db_create_option_t *options,
               apr_pool_t *pool)
{
    return rast_local_db_create(name, options, pool);
}

rast_error_t *
rast_db_optimize(const char *name, const rast_db_optimize_option_t *options,
                 apr_pool_t *pool)
{
    return rast_local_db_optimize(name, options, pool);
}

rast_error_t *
rast_db_open(rast_db_t **db, const char *name, int flags,
             rast_db_open_option_t *options, apr_pool_t *pool)
{
    int i;
    rast_error_t *error;

    if (open_functions != NULL) {
        for (i = 0; i < open_functions->nelts; i++) {
            rast_open_func_t *open_func =
                ((rast_open_func_t **) open_functions->elts)[i];
            error = open_func(db, name, flags, options, pool);
            if (error == RAST_OK) {
                return RAST_OK;
            }
            if (error->type == RAST_ERROR_TYPE_RAST &&
                error->code == RAST_ERROR_UNSUPPORTED_SCHEME) {
                rast_error_destroy(error);
                continue;
            }
            return error;
        }
    }

    return rast_local_db_open(db, name, flags, options, pool);
}

rast_error_t *
rast_db_sync(rast_db_t *db)
{
    if (db->db_sync == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_sync(db);
}

rast_error_t *
rast_db_close(rast_db_t *db)
{
    if (db->db_close == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_close(db);
}

rast_error_t *
rast_db_register(rast_db_t *db,
                 const char *text, rast_size_t nbytes,
                 rast_value_t *properties, rast_doc_id_t *doc_id)
{
    if (db->db_register == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_register(db, text, nbytes, properties, doc_id);
}

rast_error_t *
rast_db_create_document(rast_db_t *db, rast_document_t **doc)
{
    if (db->db_create_document == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_create_document(db, doc);
}

rast_error_t *
rast_db_search(rast_db_t *db,
               const char *query, rast_search_option_t *options,
               rast_result_t **result, apr_pool_t *pool)
{
    if (db->db_search == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_search(db, query, options, result, pool);
}

rast_error_t *
rast_db_delete(rast_db_t *db, rast_doc_id_t doc_id)
{
    if (db->db_delete == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_delete(db, doc_id);
}

rast_error_t *
rast_db_update(rast_db_t *db,
               rast_doc_id_t old_doc_id,
               const char *text, rast_size_t nbytes,
               rast_value_t *properties, rast_doc_id_t *new_doc_id)
{
    if (db->db_update == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_update(db, old_doc_id, text, nbytes, properties, new_doc_id);
}

rast_error_t *
rast_db_get_text(rast_db_t *db,
                 rast_doc_id_t doc_id,
                 char **s, rast_size_t *nbytes, apr_pool_t *pool)
{
    if (db->db_get_text == NULL) {
        return rast_error(RAST_ERROR_NOT_IMPLEMENTED, NULL);
    }
    return db->db_get_text(db, doc_id, s, nbytes, pool);
}

rast_byte_order_e
rast_db_byte_order(rast_db_t *db)
{
    if (db->db_byte_order == NULL) {
        return RAST_UNKNOWN_ENDIAN;
    }
    return db->db_byte_order(db);
}

const char *
rast_db_encoding(rast_db_t *db)
{
    if (db->db_encoding == NULL) {
        return NULL;
    }
    return db->db_encoding(db);
}

const rast_property_t *
rast_db_properties(rast_db_t *db, int *num_properties)
{
    return db->db_properties(db, num_properties);
}

int
rast_db_sync_threshold_chars(rast_db_t *db)
{
    if (db->db_sync_threshold_chars == NULL) {
        return -1;
    }
    return db->db_sync_threshold_chars(db);
}
