# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "fileutils"
require "timeout"
require "bdb"
require "nkf"

require "rast_test"
require "test-utility"
require "rast/database-generatable"

module Rast
  class LocalDBTest < Test::Unit::TestCase
    include DatabaseGeneratable

    def test_create
      create_test_little_endian
      create_test_big_endian_and_multiple_property
      create_test_invalid_uint_flags
      create_test_with_flags
      create_test_property_name
      create_test_duplicate_properties
    end

    def create_test_little_endian
      options = {
        "byte_order" => Rast::LITTLE_ENDIAN,
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name, Rast::DB::RDONLY) do |db|
        properties = db.properties
        assert_equal(Rast::LITTLE_ENDIAN, db.byte_order)
        assert_equal("UTF-8", db.encoding)
        assert_equal("title", properties[0].name)
        assert_equal(Rast::PROPERTY_TYPE_STRING, properties[0].type)
        assert_equal(true, properties[0].search)
        assert_equal(true, properties[0].text_search)
        assert_equal(true, properties[0].full_text_search)
        assert_equal(false, properties[0].unique)
        assert_equal(1, properties.length)
      end
      assert_equal(true, btree_db_exist?(db_name, "text.ngm"))
      assert_equal(true, btree_db_exist?(db_name, "text.rng"))
      assert_equal(true, file_exist?(db_name, "text.pos"))
      assert_equal(true, file_exist?(db_name, "text.pfl"))
      assert_equal(true, btree_db_exist?(db_name, "properties.db"))
      assert_equal(true, property_btree_db_exist?(db_name, "title.inv"))
      assert_equal(true, property_btree_db_exist?(db_name, "title.ngm"))
      assert_equal(true, property_btree_db_exist?(db_name, "title.rng"))
      assert_equal(true, property_file_exist?(db_name, "title.pos"))
      assert_equal(true, property_file_exist?(db_name, "title.pfl"))
      assert_equal(true, recno_db_exist?(db_name, "text.db"))
      check_doc_info(db_name, 0, 0)
    end

    def create_test_big_endian_and_multiple_property
      options = {
        "byte_order" => Rast::BIG_ENDIAN,
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => true,
            "unique" => true,
          },
          {
            "name" => "filesize",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name, Rast::DB::RDONLY) do |db|
        properties = db.properties
        assert_equal(Rast::BIG_ENDIAN, db.byte_order)
        assert_equal("UTF-8", db.encoding)
        assert_equal("filename", properties[0].name)
        assert_equal(Rast::PROPERTY_TYPE_STRING, properties[0].type)
        assert_equal(true, properties[0].search)
        assert_equal(true, properties[0].text_search)
        assert_equal(true, properties[0].full_text_search)
        assert_equal(true, properties[0].unique)
        assert_equal("filesize", properties[1].name)
        assert_equal(Rast::PROPERTY_TYPE_UINT, properties[1].type)
        assert_equal(true, properties[1].search)
        assert_equal(false, properties[1].text_search)
        assert_equal(false, properties[1].full_text_search)
        assert_equal(false, properties[1].unique)
        assert_equal("date", properties[2].name)
        assert_equal(Rast::PROPERTY_TYPE_DATE, properties[2].type)
        assert_equal(true, properties[2].search)
        assert_equal(false, properties[2].text_search)
        assert_equal(false, properties[2].full_text_search)
        assert_equal(false, properties[2].unique)
        assert_equal(Rast::PROPERTY_TYPE_DATETIME, properties[3].type)
        assert_equal(true, properties[3].search)
        assert_equal(false, properties[3].text_search)
        assert_equal(false, properties[3].full_text_search)
        assert_equal(false, properties[3].unique)
        assert_equal(4, properties.length)
      end
      assert_equal(true, btree_db_exist?(db_name, "text.ngm"))
      assert_equal(true, btree_db_exist?(db_name, "text.rng"))
      assert_equal(true, file_exist?(db_name, "text.pos"))
      assert_equal(true, btree_db_exist?(db_name, "properties.db"))
      assert_equal(true, property_btree_db_exist?(db_name, "filename.inv"))
      assert_equal(true, property_btree_db_exist?(db_name, "filename.ngm"))
      assert_equal(true, property_btree_db_exist?(db_name, "filename.rng"))
      assert_equal(true, property_file_exist?(db_name, "filename.pos"))
      assert_equal(true, property_btree_db_exist?(db_name, "filesize.inv"))
      assert_equal(true, property_btree_db_exist?(db_name, "date.inv"))
      assert_equal(true, property_btree_db_exist?(db_name, "datetime.inv"))
      assert_equal(false, dir_exist?(db_name, "text"))
      check_doc_info(db_name, 0, 0)
    end

    def create_test_invalid_uint_flags
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filesize",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filesize",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "birth_day",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "birth_day",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filesize",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end
    end

    def create_test_with_flags
      options = {
        "byte_order" => Rast::LITTLE_ENDIAN,
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_path = generate_db_name
      Dir.chdir(File.dirname(db_path)) do
        db_name = File.basename(db_path)
        LocalDB.create(db_name, options)
        LocalDB.open(db_name, Rast::DB::RDONLY) do |db|
          properties = db.properties
          assert_equal(Rast::LITTLE_ENDIAN, db.byte_order)
          assert_equal("UTF-8", db.encoding)
          assert_equal("title", properties[0].name)
          assert_equal(Rast::PROPERTY_TYPE_STRING, properties[0].type)
          assert_equal(true, properties[0].search)
          assert_equal(true, properties[0].text_search)
          assert_equal(true, properties[0].full_text_search)
          assert_equal(false, properties[0].unique)
          assert_equal(1, properties.length)
        end
        assert_equal(true, btree_db_exist?(db_name, "text.ngm"))
        assert_equal(true, btree_db_exist?(db_name, "text.rng"))
        assert_equal(true, file_exist?(db_name, "text.pos"))
        assert_equal(true, file_exist?(db_name, "text.pfl"))
        assert_equal(true, btree_db_exist?(db_name, "properties.db"))
        assert_equal(true, property_btree_db_exist?(db_name, "title.inv"))
        assert_equal(true, property_btree_db_exist?(db_name, "title.ngm"))
        assert_equal(true, property_btree_db_exist?(db_name, "title.rng"))
        assert_equal(true, property_file_exist?(db_name, "title.pos"))
        assert_equal(true, property_file_exist?(db_name, "title.pfl"))
        assert_equal(true, recno_db_exist?(db_name, "text.db"))
        check_doc_info(db_name, 0, 0)
      end
    end

    def create_test_property_name
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "abcdefghijklmn_lopqrstuvwxyz",
            "type" => Rast::PROPERTY_TYPE_STRING,
          },
          {
            "name" => "ABCDEFGHIJKLMN-LOPQRSTUVWXYZ",
            "type" => Rast::PROPERTY_TYPE_UINT,
          },
          {
            "name" => "0123456789",
            "type" => Rast::PROPERTY_TYPE_DATE,
          },
        ],
      }
      db_name = generate_db_name
      assert_nothing_raised do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "title/../../abc",
            "type" => Rast::PROPERTY_TYPE_STRING,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end

      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "日付",
            "type" => Rast::PROPERTY_TYPE_DATE,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end
    end

    def create_test_duplicate_properties
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "uri",
            "type" => Rast::PROPERTY_TYPE_STRING,
          },
          {
            "name" => "uri",
            "type" => Rast::PROPERTY_TYPE_UINT,
          },
        ],
      }
      db_name = generate_db_name
      assert_raise(RastError) do
        LocalDB.create(db_name, options)
      end
    end

    def test_open
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        assert_nothing_raised do
          db.register("", {})
        end
      end

      LocalDB.open("file://" + db_name, Rast::DB::RDWR) do |db|
        assert_nothing_raised do
          db.register("", {})
        end
      end

      LocalDB.open(db_name, Rast::DB::RDONLY) do |db|
        assert_raise(RastError) do
          db.register("", {})
        end
      end
      options = {
        "sync_threshold_chars" => 10000
      }
      LocalDB.open(db_name, Rast::DB::RDWR, options) do |db|
        assert_equal(10000, db.sync_threshold_chars)
      end
    end

    def test_register_raw
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register_raw("本日は晴天なり",
                                        [
                                          "test",
                                          Date.new(2005, 4, 15),
                                          DateTime.new(2005, 6, 16, 16, 10, 7),
                                          10
                                        ]))
      end
    end

    def test_register
      register_test_empty
      register_test_no_text
      register_test_text_without_normalize
      register_test_text_with_no_change_byte_normalize
      register_test_text_with_change_byte_normalize
      register_test_text_with_null_character
      register_test_property_enable_all_search
      register_test_property_enable_search
      register_test_property_enable_text_search
      register_test_property_enable_full_text_search
      register_test_property_unique
      register_test_omit_property
      register_test_summary_text

      # todo: this test is failed on low speed machine.
      # register_test_too_many_text
    end

    def register_test_empty
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register("", {}))
      end
      BDB::Btree.open(File.join(db_name, "properties.db"), nil, 0) do |db|
        s = db[[1].pack("I")]
        delete_flag = s.slice!(0, 1).unpack("C")[0]
        num_chars = s.slice!(0, 4).unpack("I")[0]

        assert_equal(0, delete_flag)
        assert_equal(0, num_chars)
        assert_equal(0, s.length)
      end
      check_doc_info(db_name, 1, 1)
    end

    def register_test_no_text
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        properties = {
          "title" => "test",
          "modified_date" => "2005-06-16",
          "modified" => "2005-02-17T19:04:38+0900",
          "serial" => 10,
        }
        assert_equal(1, db.register("", properties))
      end
      BDB::Btree.open(File.join(db_name, "properties.db"), nil, 0) do |db|
        s = db[[1].pack("I")]
        delete_flag = s.slice!(0, 1).unpack("C")[0]
        num_chars = s.slice!(0, 4).unpack("I")[0]
        title_nbytes = s.slice!(0, 4).unpack("I")[0]
        title = s.slice!(0, title_nbytes)
        modified_date_nbytes = s.slice!(0, 4).unpack("I")[0]
        modified_date = s.slice!(0, modified_date_nbytes)
        modified_nbytes = s.slice!(0, 4).unpack("I")[0]
        modified = s.slice!(0, modified_nbytes)
        serial = s.slice!(0, 4).unpack("I")[0]

        assert_equal(0, delete_flag)
        assert_equal(0, num_chars)
        assert_equal("test", title)
        assert_equal("2005-06-16", modified_date)
        assert_equal("2005-02-17T19:04:38", modified)
        assert_equal(10, serial)
        assert_equal(0, s.length)
      end
      check_doc_info(db_name, 1, 1)
    end

    def register_test_text_without_normalize
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register("あいうえお", {}))
      end
      LocalDB.open(db_name) do |db|
        assert_equal("あいうえお", db.get_text(1))
      end
      BDB::Btree.open(File.join(db_name, "properties.db"), nil, 0) do |db|
        s = db[[1].pack("I")]
        delete_flag = s.slice!(0, 1).unpack("C")[0]
        num_chars = s.slice!(0, 4).unpack("I")[0]

        assert_equal(0, delete_flag)
        assert_equal(5, num_chars)
        assert_equal(0, s.length)
      end
      check_doc_info(db_name, 1, 1)
    end

    def register_test_text_with_change_byte_normalize
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register("あい  Ue\n\r\fO", {}))
      end
      LocalDB.open(db_name) do |db|
        assert_equal("あい Ue O", db.get_text(1))
      end
      check_doc_property(db_name, 1, 0, 7)
      check_doc_info(db_name, 1, 1)
    end

    def register_test_text_with_no_change_byte_normalize
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register("AIUEO", {}))
        assert_equal(2, db.register("QwErtY", {}))
        assert_equal(3, db.register("あいうZxCv", {}))
      end
      check_doc_property(db_name, 1, 0, 5)
      check_doc_property(db_name, 2, 0, 6)
      check_doc_property(db_name, 3, 0, 7)
      check_registered_text_index(File.join(db_name, "text"), "aiueo")
      check_registered_text_index(File.join(db_name, "text"), "qwerty")
      check_registered_text_index(File.join(db_name, "text"), "あいうzxcv")
      LocalDB.open(db_name) do |db|
        assert_equal("AIUEO", db.get_text(1))
        assert_equal("QwErtY", db.get_text(2))
        assert_equal("あいうZxCv", db.get_text(3))
      end
    end

    def register_test_text_with_null_character
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal(1, db.register("abc\0def", {}))
      end
      LocalDB.open(db_name) do |db|
        assert_equal("abc\0def", db.get_text(1))
      end
      check_doc_property(db_name, 1, 0, 7)
      check_doc_info(db_name, 1, 1)
    end

    def register_test_property_enable_all_search
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => true,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text = "aiueo"
        properties = {
          "string" => "abc  DE    F g",
          "date" => "2005-06-16",
          "modified" => "2005-01-01T12:00:00+0900",
          "number" => 100000,
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_property(db_name, 1, 0, 5)
      check_summary_text(db_name, 1, "aiueo")

      # search
      check_property_inv(db_name, "string", "abc  DE    F g", 1)
      check_property_inv(db_name, "date", "2005-06-16", 1)
      check_property_inv(db_name, "modified", "2005-01-01T12:00:00", 1)
      check_property_inv(db_name, "number", [100000].pack("I"), 1)

      # text_search
      check_registered_text_index(File.join(db_name, "properties", "string"), 
                                  "abc de f g")
      check_not_exist_text_index(db_name, File.join("properties", "date"))
      check_not_exist_text_index(db_name, File.join("properties", "number"))

      # full_text_search
      check_registered_text_index(File.join(db_name, "text"), "aiueo")
      check_registered_text_index(File.join(db_name, "text"), "abc de f g")
      check_not_registered_text_index(File.join(db_name, "text"), "eoa")
      check_not_registered_text_index(File.join(db_name, "text"),
                                      "20050101t12:00:00")
      check_not_registered_text_index(File.join(db_name, "text"), "100000")
    end

    def register_test_property_enable_search
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text = "aiueo"
        properties = {
          "string" => "abc  DE    F g",
          "date" => "2005-06-16",
          "datetime" => "2005-01-01T12:00:00+0900",
          "number" => 100000,
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_property(db_name, 1, 0, 5)
      check_summary_text(db_name, 1, "aiueo")

      # search
      check_property_inv(db_name, "string", "abc  DE    F g", 1)
      check_property_inv(db_name, "date", "2005-06-16", 1)
      check_property_inv(db_name, "datetime", "2005-01-01T12:00:00", 1)
      check_property_inv(db_name, "number", [100000].pack("I"), 1)

      # text_search
      check_not_exist_text_index(db_name, File.join("properties", "string"))
      check_not_exist_text_index(db_name, File.join("properties", "date"))
      check_not_exist_text_index(db_name, File.join("properties", "datetime"))
      check_not_exist_text_index(db_name, File.join("properties", "number"))

      # full_text_search
      check_registered_text_index(File.join(db_name, "text"), "aiueo")
      check_not_registered_text_index(File.join(db_name, "text"), "abc de f g")
      check_not_registered_text_index(File.join(db_name, "text"),
                                      "2005-06-16")
      check_not_registered_text_index(File.join(db_name, "text"),
                                      "2005-01-01T12:00:00")
      check_not_registered_text_index(File.join(db_name, "text"), "100000")
    end

    def register_test_property_enable_text_search
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text = "aiueo"
        properties = {
          "string" => "aBc  def",
          "date" => "2005-02-21",
          "datetime" => "2005-02-21T00:00:00+0900",
          "number" => 100,
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_property(db_name, 1, 0, 5)
      check_summary_text(db_name, 1, "aiueo")

      # search
      assert_equal(false, property_btree_db_exist?(db_name, "string"))
      assert_equal(false, property_btree_db_exist?(db_name, "date"))
      assert_equal(false, property_btree_db_exist?(db_name, "datetime"))
      assert_equal(false, property_btree_db_exist?(db_name, "number"))

      # text_search
      check_registered_text_index(File.join(db_name, "properties", "string"), 
                                  "abc def")

      # full_text_search
      check_registered_text_index(File.join(db_name, "text"), "aiueo")
      check_not_registered_text_index(File.join(db_name, "text"), "abc def")
      check_not_registered_text_index(File.join(db_name, "text"),
                                      "a b c")
      check_not_registered_text_index(File.join(db_name, "text"), "100")
    end

    def register_test_property_enable_full_text_search
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text = "aiueo"
        properties = {
          "string" => "aBc  def",
          "date" => "2005-01-01",
          "datetime" => "2005-01-01T12:00:00+0900",
          "number" => 100000,
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_property(db_name, 1, 0, 5)
      check_summary_text(db_name, 1, "aiueo")

      # search
      assert_equal(false, property_btree_db_exist?(db_name, "string"))
      assert_equal(false, property_btree_db_exist?(db_name, "date"))
      assert_equal(false, property_btree_db_exist?(db_name, "datetime"))
      assert_equal(false, property_btree_db_exist?(db_name, "number"))

      # text_search
      check_not_exist_text_index(db_name, File.join("properties", "string"))
      check_not_exist_text_index(db_name, File.join("properties", "date"))
      check_not_exist_text_index(db_name, File.join("properties", "datetime"))
      check_not_exist_text_index(db_name, File.join("properties", "number"))

      # full_text_search
      check_registered_text_index(File.join(db_name, "text"), "aiueo")
      check_registered_text_index(File.join(db_name, "text"), "abc")
      check_registered_text_index(File.join(db_name, "text"), "abc def")
      check_not_registered_text_index(File.join(db_name, "text"), "eoa")

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "string",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "number",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text = ".N ."
        properties = {
          "string" => "NET",
          "date" => "2005-01-01",
          "datetime" => "2005-01-01T12:00:00",
          "number" => 100000,
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_property(db_name, 1, 0, 4)
      check_summary_text(db_name, 1, ".N .")

      # search
      assert_equal(false, property_btree_db_exist?(db_name, "string"))
      assert_equal(false, property_btree_db_exist?(db_name, "date"))
      assert_equal(false, property_btree_db_exist?(db_name, "datetime"))
      assert_equal(false, property_btree_db_exist?(db_name, "number"))

      # text_search
      check_not_exist_text_index(db_name, File.join("properties", "string"))
      check_not_exist_text_index(db_name, File.join("properties", "date"))
      check_not_exist_text_index(db_name, File.join("properties", "datetime"))
      check_not_exist_text_index(db_name, File.join("properties", "number"))

      # full_text_search
      check_registered_text_index(File.join(db_name, "text"), ".n .")
      check_registered_text_index(File.join(db_name, "text"), "net")
      check_not_registered_text_index(File.join(db_name, "text"), ".net")
    end

    def register_test_omit_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
            "omit_property" => true,
          },
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "title" => "天気",
                      "filename" => "tomorrow.txt",
                    })
      end
      properties_db_name = File.join(db_name, "properties.db")
      BDB::Btree.open(properties_db_name, nil, 0) do |db|
        assert_equal(1 + 4 + 4 + "tomorrow.txt".length,
                     db[[1].pack("I")].length)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("title = 天気",
                           {"properties" => ["filename"]})
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(1, result.items[0].properties.length)
        assert_equal("tomorrow.txt", result.items[0].properties[0])
        assert_equal(1, result.items.length)

        assert_raise(RastError) do
          db.search("title = 天気", {"properties" => ["title"]})
        end
      end
    end

    def register_test_property_unique
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
         ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)

      LocalDB.open(db_name) do |db|
        text = "hello"
        properties = {
          "title" => "hello",
          "filename" => "hello.txt",
        }
        assert_equal(1, db.register(text, properties))
      end
      check_doc_info(db_name, 1, 1)
      check_doc_property(db_name, 1, 0, 5)
      check_summary_text(db_name, 1, "hello")

      # search
      check_property_inv(db_name, "title", "hello", 1)
      check_property_inv(db_name, "filename", "hello.txt", 1)

      LocalDB.open(db_name) do |db|
        text ="hello"
        properties = {
          "title" => "hello world",
          "filename" => "hello.txt",
        }
        assert_raise(RastError) do
          db.register(text, properties)
        end
      end
      check_doc_info(db_name, 1, 1)

      # search
      check_property_inv(db_name, "title", "hello", 1)
      check_property_inv(db_name, "filename", "hello.txt", 1)
      check_property_inv(db_name, "title", "hello world")

      LocalDB.open(db_name) do |db|
        text = "hello"
        properties = {
          "title" => "hello",
          "filename" => "hello2.txt",
        }
        db.register(text, properties)
      end
      check_doc_info(db_name, 2, 2)
      check_doc_property(db_name, 2, 0, 5)
      check_summary_text(db_name, 2, "hello")

      # search
      check_property_inv(db_name, "title", "hello", 1, 2)
      check_property_inv(db_name, "filename", "hello2.txt", 2)

      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        text ="hello"
        properties = {
          "title" => "hello",
          "filename" => "hello.txt",
          "date" => "2005-06-17",
          "datetime" => "2005-06-17T10:37:30",
        }
        assert_equal(1, db.register(text, properties))
        properties = {
          "title" => "ruby",
          "filename" => "hello.txt",
          "date" => "2005-06-16",
          "datetime" => "2005-06-17T10:38:50",
        }
        assert_raise(RastError) do
          db.register(text, properties)
        end
        properties = {
          "title" => "perl",
          "filename" => "bye.txt",
          "date" => "2005-06-17",
          "datetime" => "2005-06-17T10:38:50",
        }
        assert_raise(RastError) do
          db.register(text, properties)
        end
        properties = {
          "title" => "python",
          "filename" => "oops.txt",
          "date" => "2005-06-15",
          "datetime" => "2005-06-17T10:37:30+0900",
        }
        assert_raise(RastError) do
          db.register(text, properties)
        end
      end
      check_doc_info(db_name, 1, 1)

      # search
      check_property_inv(db_name, "title", "hello", 1)
      check_property_inv(db_name, "filename", "hello.txt", 1)
      check_property_inv(db_name, "date", "2005-06-17", 1)
      check_property_inv(db_name, "datetime", "2005-06-17T10:37:30", 1)

      check_property_inv(db_name, "title", "ruby")
      check_property_inv(db_name, "title", "perl")
      check_property_inv(db_name, "title", "python")
    end

    def register_test_summary_text
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      File.open(File.expand_path("doc_info", db_name), "w") do |f|
        f.write([0xAA, 0xAA].pack("I*"))
      end
      LocalDB.open(db_name) do |db|
        assert_equal(0xAA + 1, db.register("aiueo", {}))
      end
      check_doc_info(db_name, 0xAB, 0xAB)
      check_summary_text(db_name, 0xAB, "aiueo")
      File.open(File.expand_path("doc_info", db_name), "w") do |f|
        f.write([127, 127].pack("I*"))
      end
      LocalDB.open(db_name) do |db|
        assert_equal(127 + 1, db.register("aiueo", {}))
      end
      check_doc_info(db_name, 128, 128)
      check_summary_text(db_name, 128, "aiueo")
      LocalDB.open(db_name) do |db|
        assert_equal(127 + 2, db.register("aiueo", {}))
      end
      check_doc_info(db_name, 129, 129)
      check_summary_text(db_name, 129, "aiueo")
    end

    def register_test_too_many_text
      options = {
        "encoding" => "mecab_euc_jp",
        "preserve_text" => true,
        "properties" => [],
      }
      # text = NKF.nkf("-e", "本日は晴天なり") * 170000
      text = NKF.nkf("-e", "本日は晴天なり") * 180000
      # text = NKF.nkf("-e", "本日は晴天なり") * 185000
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      pid = fork do
        LocalDB.open(db_name, Rast::DB::RDWR) do |db|
          assert_raise(RastError) do
            db.register(text, {})
          end
        end
      end
      begin
        Timeout.timeout(5) do
          Process.waitpid(pid)
        end
      rescue Timeout::Error
        Process.kill(:KILL, pid)
        flunk("register big text failed")
      rescue Errno::ECHILD
      end
    end

    def test_sync
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = File.expand_path("testdb", @db_dir)
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
        db.sync
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
    end

    def test_search
      search_test_simple
      search_test_summary
      search_test_text_and_property
      search_test_property_and_summary
      search_test_sort_with_score
      search_test_calc_score
      search_test_calc_received_all_num_docs_score
      search_test_sort_with_property
      search_test_set_range
      search_test_with_datetime
    end

    def search_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {})
        db.register("本日は晴天なり", {})
        db.register("明日は雪っす", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(3, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)

        result = db.search('""')
        assert_equal(3, result.num_docs)
        assert_equal(0, result.hit_count)
        assert_equal("", result.terms[0].term)
        assert_equal(0, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(0, result.items.length)

        result = db.search("not-found-string")
        assert_equal("not-found-string", result.terms[0].term)
        assert_equal(0, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(0, result.items.length)
      end
    end

    def search_test_summary
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("0123456789", {"title" => "full text search property"})
        db.register("零一二三四五六七八九",
                    {"title" => "full text search property"})
        db.register("零1二3四5六7八9",
                    {"title" => "full text search property"})
      end
      LocalDB.open(db_name) do |db|
        search_options = {"need_summary" => true, "summary_nchars" => 6}
        result = db.search("5", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("345678", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("3四5六7八", result.items[1].summary)
        assert_equal(2, result.items.length)

        result = db.search("1", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("012345", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("零1二3四5", result.items[1].summary)
        assert_equal(2, result.items.length)

        result = db.search("8", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("456789", result.items[0].summary)
        assert_equal(1, result.items.length)

        result = db.search("五", search_options)
        assert_equal(2, result.items[0].doc_id)
        assert_equal("三四五六七八", result.items[0].summary)
        assert_equal(1, result.items.length)

        result = db.search("一", search_options)
        assert_equal(2, result.items[0].doc_id)
        assert_equal("零一二三四五", result.items[0].summary)
        assert_equal(1, result.items.length)

        result = db.search("八", search_options)
        assert_equal(2, result.items[0].doc_id)
        assert_equal("四五六七八九", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("四5六7八9", result.items[1].summary)
        assert_equal(2, result.items.length)

        result = db.search("full", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("012345", result.items[0].summary)
        assert_equal(2, result.items[1].doc_id)
        assert_equal("零一二三四五", result.items[1].summary)
        assert_equal(3, result.items[2].doc_id)
        assert_equal("零1二3四5", result.items[2].summary)
        assert_equal(3, result.items.length)

        result = db.search("12", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("012345", result.items[0].summary)
        assert_equal(1, result.items.length)

        result = db.search("五六", search_options)
        assert_equal(2, result.items[0].doc_id)
        assert_equal("三四五六七八", result.items[0].summary)
        assert_equal(1, result.items.length)

        result = db.search("0123456789", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("234567", result.items[0].summary)
        assert_equal(1, result.items.length)

        search_options = {"need_summary" => true, "summary_nchars" => 11}
        result = db.search("5", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("0123456789", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("零1二3四5六7八9", result.items[1].summary)
        assert_equal(2, result.items.length)

        search_options = {"need_summary" => true, "summary_nchars" => 12}
        result = db.search("5", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("0123456789", result.items[0].summary)
        assert_equal(3, result.items[1].doc_id)
        assert_equal("零1二3四5六7八9", result.items[1].summary)
        assert_equal(2, result.items.length)

        search_options = {"need_summary" => true, "summary_nchars" => 1}
        result = db.search("567", search_options)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("6", result.items[0].summary)
        assert_equal(1, result.items.length)
      end

      options = {
        "encoding" => "euc_jp",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("utf8 no preserve_text need_summary test", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("no", {"need_summary" => true})
        assert_equal(1, result.items[0].doc_id)
        assert_equal("", result.items[0].summary)
        assert_equal(1, result.items.length)
      end

      options = {
        "encoding" => "euc_jp",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("euc_jp no preserve_text need_summary test", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("no", {"need_summary" => true})
        assert_equal(1, result.items[0].doc_id)
        assert_equal("", result.items[0].summary)
        assert_equal(1, result.items.length)
      end
    end

    def search_test_text_and_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
          },
        ],
      }

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("bcxy abc", {"filename" => "xyz"})
        db.register("bcxy", {"filename" => "123"})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.hit_count)
        result = db.search("xyz")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.hit_count)
        result = db.search("abcxyz")
        assert_equal(0, result.hit_count)
        result = db.search('"abc xyz"')
        assert_equal(0, result.hit_count)
      end
    end

    def search_test_property_and_summary
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり", {"filename" => "ame.txt"})
        db.register("a" * 100 + "晴天" + "b" * 100,
                    {"filename" => "hare.txt"})
        db.register("明日は雪っす", {"filename" => "yuki.txt"})
        db.register("", {"filename" => "yuki.txt"})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天",
                           {
                             "properties" => ["filename"],
                             "need_summary" => true,
                           })
        assert_equal(1, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal("hare.txt", result.items[0].properties[0])
        assert_equal("hare.txt", result.items[0].properties.filename)
        assert_equal(1, result.items[0].properties.length)
        expected_summary = "a" * 49 + "晴天" + "b" * 49
        assert_equal(expected_summary, result.items[0].summary)
        assert_equal(1, result.items.length)
      end

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("1" + "abc" * 100, {"filename" => "abc.txt"})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("abc",
                           {
                             "summary_nchars" => 301,
                             "need_summary" => true,
                           })
        assert_equal(1, result.hit_count)
        assert_equal("abc", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("1" + "abc" * 100, result.items[0].summary)
        assert_equal(1, result.items.length)
      end
    end

    def search_test_sort_with_score
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("あい" * 99 + "晴天", {}) 
        db.register("晴天" * 100, {})
        db.register("あい" * 49 + "晴天", {}) 
      end
      LocalDB.open(db_name) do |db|
        # descend
        result = db.search("晴天")
        assert_equal(3, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(3, result.items[1].doc_id)
        assert_equal(1, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        # ascend
        search_options = {
          "sort_order" => Rast::SORT_ORDER_ASCENDING
        }
        result = db.search("晴天", search_options)
        assert_equal(3, result.hit_count)
        assert_equal("晴天", result.terms[0].term)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(3, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)
      end
    end

    def search_test_calc_score
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("aaa" * 100, {"filename" => "abc.txt"})
      end
      LocalDB.open(db_name) do |db|
        options = {
          "score_method" => Rast::SCORE_METHOD_NONE,
        }
        result = db.search("aaa", options)
        assert_equal(0, result.items[0].score)

        options = {
          "score_method" => Rast::SCORE_METHOD_TFIDF,
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)

        result = db.search("aaa", {})
        assert_operator(0, :<, result.items[0].score)

        options = {
          "sort_method" => Rast::SORT_METHOD_PROPERTY,
          "sort_property" => "filename"
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)

        options = {
          "score_method" => Rast::SCORE_METHOD_TFIDF,
          "sort_method" => Rast::SORT_METHOD_PROPERTY,
          "sort_property" => "filename"
        }
        result = db.search("aaa", options)
        assert_operator(0, :<, result.items[0].score)
      end
    end

    def search_test_calc_received_all_num_docs_score
      options = {
        "encoding" => "utf8",
        "properties" => [
        ],
      }
      db_name1 = generate_db_name
      LocalDB.create(db_name1, options)
      LocalDB.open(db_name1) do |db|
        db.register("qwerty abcdef", {})
        db.register("qwerty", {})
      end
      result1 = LocalDB.open(db_name1) do |db|
        db.search("qwe abc", {})
      end
      db_name2 = generate_db_name
      LocalDB.create(db_name2, options)
      LocalDB.open(db_name2) do |db|
        db.register("qwerty abcdef", {})
      end
      LocalDB.open(db_name2) do |db|
        terms = result1.terms.collect do |term|
          term.doc_count
        end
        result2 = db.search("qwe abc", {"all_num_docs" => 2, "terms" => terms})
        assert_equal(result1.items[0].score, result2.items[0].score)
      end
    end

    def search_test_sort_with_property
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("昨日は雨天なり",
                    {"filename" => "2.txt", "serial" => 10})
        db.register("本日は晴天なり",
                    {"filename" => "3.txt", "serial" => 5})
        db.register("明日は雪っすり",
                    {"filename" => "1.txt", "serial" => 15})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "sort_order" => Rast::SORT_ORDER_ASCENDING,
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "filename",
                             "sort_order" => Rast::SORT_ORDER_DESCENDING,
                             "properties" => ["filename"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "sort_order" => Rast::SORT_ORDER_ASCENDING,
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(3, result.items[2].doc_id)
        assert_equal(3, result.items.length)

        result = db.search("日は",
                           {
                             "sort_method" => Rast::SORT_METHOD_PROPERTY,
                             "sort_property" => "serial",
                             "sort_order" => Rast::SORT_ORDER_DESCENDING,
                             "properties" => ["serial"],
                           })
        assert_equal(3, result.hit_count)
        assert_equal(3, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items[1].doc_id)
        assert_equal(2, result.items[2].doc_id)
        assert_equal(3, result.items.length)
      end
    end

    def search_test_set_range
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
        db.register("1", {}) 
      end
      LocalDB.open(db_name) do |db|
        search_options = {
          "start_no" => 0,
          "num_items" => 3,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(3, result.items.length)

        search_options = {
          "start_no" => 1,
          "num_items" => 3,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(3, result.items.length)

        search_options = {
          "start_no" => 4,
          "num_items" => 10,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(1, result.items.length)

        search_options = {
          "start_no" => 20,
          "num_items" => 10,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(0, result.items.length)

        search_options = {
          "num_items" => 0,
        }
        result = db.search("1", search_options)
        assert_equal(5, result.hit_count)
        assert_equal(0, result.items.length)
      end
    end

    def search_test_with_datetime
      options = {
        "encoding" => "utf8",
        "properties" => [
          {
            "name" => "last_modified_date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "last_modified",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)

      date1 = Date.new(2005, 6, 16)
      date2 = DateTime.new(2005, 6, 10, 12, 59, 20)
      datetime1 = DateTime.new(2005, 6, 16, 16, 54, 19)
      datetime2 = Time.mktime(2005, 6, 16, 11, 20, 40)
      LocalDB.open(db_name) do |db|
        doc_id = db.register("date1",
                             {
                               "last_modified_date" => date1,
                               "last_modified" => datetime1,
                             })
        assert_equal(1, doc_id)
        doc_id = db.register("date2",
                             {
                               "last_modified_date" => date2,
                               "last_modified" => datetime2,
                             })
        assert_equal(2, doc_id)
      end

      LocalDB.open(db_name) do |db|
        result = db.search("date",
                           {
                             "properties" => [
                               "last_modified_date",
                               "last_modified",
                             ],
                           })
        assert_equal(date1.to_s, result.items[0].properties[0])
        assert_equal(datetime1.strftime("%FT%T"),
                     result.items[0].properties[1])
        assert_equal(date2.strftime("%F"), result.items[1].properties[0])
        assert_equal(datetime2.strftime("%FT%T"),
                     result.items[1].properties[1])

        result = db.search("date",
                           {
                             "properties" => [
                               "last_modified_date",
                               "last_modified",
                             ],
                             "parse_date" => true,
                           })
        assert_equal(date1, result.items[0].properties[0])
        assert_equal(datetime1, result.items[0].properties[1])
        assert_equal(Date.new(date2.year, date2.month, date2.day),
                     result.items[1].properties[0])
        assert_equal(DateTime.new(datetime2.year,
                                  datetime2.month,
                                  datetime2.day,
                                  datetime2.hour,
                                  datetime2.min,
                                  datetime2.sec),
                     result.items[1].properties[1])
      end
    end

    def test_delete
      delete_test_simple
      delete_test_property
      delete_test_omit_property
    end

    def delete_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        db.delete(1)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
      end

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("abcde", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        db.delete(1)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("abc", {"need_summary" => true})
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
      end
    end

    def delete_test_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "filename" => "1.txt",
                      "count" => 1,
                    })
        db.register("明日は雨天なり",
                    {
                      "filename" => "2.txt",
                      "count" => 1,
                    })
      end
      properties_db_name = File.join(db_name, "properties.db")
      property_nbytes = BDB::Btree.open(properties_db_name) do |db|
        db[[1].pack("I")].length
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
        db.delete(1)
        result = db.search("count = 1")
        assert_equal(1, result.hit_count)
      end
      BDB::Btree.open(properties_db_name) do |db|
        assert_operator(property_nbytes, :>, db[[1].pack("I")].length)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
      end
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "filename" => "1.txt",
                      "count" => 1,
                    })
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
      end
    end

    def delete_test_omit_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
            "omit_property" => true,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "filename" => "1.txt",
                      "serial" => 1,
                    })
        db.register("明日は雨天なり",
                    {
                      "filename" => "2.txt",
                      "serial" => 2,
                    })
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 1")
        assert_equal(1, result.hit_count)
        db.delete(1)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 1")
        assert_equal(0, result.hit_count)
      end

      LocalDB.open(db_name) do |db|
        result = db.search("filename = 2.txt")
        assert_equal(1, result.hit_count)
        db.delete(2)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("filename = 2.txt")
        assert_equal(0, result.hit_count)
      end
    end

    def test_update
      update_test_simple
      update_test_property
      update_test_omit_property
    end

    def update_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
        assert_equal(2, db.update(1, "明日は雪っす", {}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
      end
    end

    def update_test_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {"filename" => "a.txt", "count" => 10})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(1, result.hit_count)
        assert_equal(2, db.update(1, "明日は雪っす",
                                  {"filename" => "b.txt", "count" => 20}))
      end
      LocalDB.open(db_name) do |db|
        result = db.search("晴天")
        assert_equal(0, result.hit_count)
        result = db.search("雪っ")
        assert_equal(1, result.hit_count)
        result = db.search("count = 10")
        assert_equal(0, result.hit_count)
        result = db.search("count = 20")
        assert_equal(1, result.hit_count)
      end
    end

    def update_test_omit_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => true,
            "omit_property" => true,
          },
          {
            "name" => "serial",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "filename" => "1.txt",
                      "serial" => 1,
                    })
        db.register("明日は雨天なり",
                    {
                      "filename" => "2.txt",
                      "serial" => 2,
                    })
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 1")
        assert_equal(1, result.hit_count)
        db.update(1, "本日は晴天なり",
                  {"filename" => "3.txt", "serial" => 3})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("serial = 1")
        assert_equal(0, result.hit_count)
        result = db.search("serial = 3")
        assert_equal(1, result.hit_count)
      end

      LocalDB.open(db_name) do |db|
        result = db.search("filename = 2.txt")
        assert_equal(1, result.hit_count)
        db.update(2, "明日は雨天なり",
                  {"filename" => "4.txt", "serial" => 4})
      end
      LocalDB.open(db_name) do |db|
        result = db.search("filename = 2.txt")
        assert_equal(0, result.hit_count)
        result = db.search("filename = 4.txt")
        assert_equal(1, result.hit_count)
      end
    end

    def test_get_text
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      text = "本日は晴天なり"
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register(text, {})
      end
      LocalDB.open(db_name) do |db|
        assert_equal(text, db.get_text(1))
      end

      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      assert_equal(false, dir_exist?(db_name, "text"))
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
      end
      LocalDB.open(db_name) do |db|
        assert_equal("", db.get_text(1))
      end
    end

    def test_optimize
      optimize_test_simple
      optimize_test_simple_with_squeeze
      optimize_test_property
      optimize_test_segv
      optimize_test_rollback
    end

    def create_test_db
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }

      db_name = generate_db_name
      remove_doc_id = nil
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり", {})
        remove_doc_id = db.register("昨日は雨でした", {})
        db.register("明日は雪らしい", {})
      end
      LocalDB.open(db_name) do |db|
        db.delete(remove_doc_id)
      end
      return db_name
    end

    def optimize_test_simple
      db_name = create_test_db

      LocalDB.optimize(db_name)
      BDB::Btree.open(File.join(db_name, "properties.db")) do |db|
        keys = db.keys.collect do |key|
          key.unpack("I")[0]
        end
        assert_equal([1, 3], keys)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("日は")
        assert_equal(2, result.num_docs)
        assert_equal(2, result.hit_count)
#        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(3, result.items[1].doc_id)
        assert_equal(2, result.items.length)
      end
      check_doc_info(db_name, 3, 2)

      BDB::Recno.open(File.join(db_name, "text.db")) do |db|
        assert_equal("本日は晴天なり", db[1])
        assert_equal("明日は雪らしい", db[3])
        assert_equal(2, db.length)
      end

      assert_equal(false, file_exist?(db_name, "old_text.ngm"))
      assert_equal(false, file_exist?(db_name, "old_text.pos"))
      assert_equal(false, file_exist?(db_name, "old_text.rng"))
      assert_equal(false, file_exist?(db_name, "old_text.pfl"))
      assert_equal(false, file_exist?(db_name, "old_text.db"))
      assert_equal(false, file_exist?(db_name, "old_properties.db"))
      assert_equal(false, file_exist?(db_name, "new_text.ngm"))
      assert_equal(false, file_exist?(db_name, "new_text.pos"))
      assert_equal(false, file_exist?(db_name, "new_text.rng"))
      assert_equal(false, file_exist?(db_name, "new_text.pfl"))
      assert_equal(false, file_exist?(db_name, "new_text.db"))
      assert_equal(false, file_exist?(db_name, "new_properties.db"))

      LocalDB.open(db_name, DB::RDWR) do |db|
        doc = db.create_document
        doc.commit
      end
      assert_nothing_raised do
        LocalDB.optimize(db_name)
      end
    end

    def optimize_test_simple_with_squeeze
      db_name = create_test_db

      LocalDB.optimize(db_name, {"squeeze_doc_id" => true})
      BDB::Btree.open(File.join(db_name, "properties.db")) do |db|
        keys = db.keys.collect do |key|
          key.unpack("I")[0]
        end
        assert_equal([1, 2], keys)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("日は")
        assert_equal(2, result.num_docs)
        assert_equal(2, result.hit_count)
#        assert_equal(2, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(2, result.items[1].doc_id)
        assert_equal(2, result.items.length)
      end
      check_doc_info(db_name, 2, 2)

      BDB::Recno.open(File.join(db_name, "text.db")) do |db|
        assert_equal("本日は晴天なり", db[1])
        assert_equal("明日は雪らしい", db[2])
        assert_equal(2, db.length)
      end

      assert_equal(false, file_exist?(db_name, "old_text.ngm"))
      assert_equal(false, file_exist?(db_name, "old_text.pos"))
      assert_equal(false, file_exist?(db_name, "old_text.rng"))
      assert_equal(false, file_exist?(db_name, "old_text.pfl"))
      assert_equal(false, file_exist?(db_name, "old_text.db"))
      assert_equal(false, file_exist?(db_name, "old_properties.db"))
      assert_equal(false, file_exist?(db_name, "new_text.ngm"))
      assert_equal(false, file_exist?(db_name, "new_text.pos"))
      assert_equal(false, file_exist?(db_name, "new_text.rng"))
      assert_equal(false, file_exist?(db_name, "new_text.pfl"))
      assert_equal(false, file_exist?(db_name, "new_text.db"))
      assert_equal(false, file_exist?(db_name, "new_properties.db"))
    end

    def optimize_test_property
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "uri",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      remove_doc_id = nil
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("本日は晴天なり",
                    {
                      "uri" => "file:///home/rast/hare.txt",
                      "count" => 12345
                    })
        remove_doc_id = db.register("昨日は雨でした",
                                    {
                                      "uri" => "file:///home/rast/ame.txt",
                                      "count" => 98765
                                    })
        db.register("明日は雪らしい",
                    {
                      "uri" => "file:///home/rast/yuki.txt",
                      "count" => 65432
                    })
      end
      LocalDB.open(db_name) do |db|
        db.delete(remove_doc_id)
      end

      text_db_size = File.size(File.join(db_name, "text.db"))

      LocalDB.optimize(db_name, {"squeeze_doc_id" => true})

      LocalDB.open(db_name) do |db|
        result = db.search("count > 1",
                           {"properties" => ["uri", "count"]})
        assert_equal(2, result.num_docs)
        assert_equal(2, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal("file:///home/rast/hare.txt",
                     result.items[0].properties[0])
        assert_equal("file:///home/rast/hare.txt",
                     result.items[0].properties.uri)
        assert_equal(12345, result.items[0].properties[1])
        assert_equal(12345, result.items[0].properties.count)
        assert_equal(2, result.items[1].doc_id)
        assert_equal("file:///home/rast/yuki.txt",
                     result.items[1].properties[0])
        assert_equal("file:///home/rast/yuki.txt",
                     result.items[1].properties.uri)
        assert_equal(65432, result.items[1].properties[1])
        assert_equal(65432, result.items[1].properties.count)
        assert_equal(2, result.items.length)

        result = db.search("uri : txt")
        assert_equal(2, result.num_docs)
        assert_equal(2, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(2, result.items[1].doc_id)
        assert_equal(2, result.items.length)

        result = db.search("uri = file:///home/rast/ame.txt")
        assert_equal(2, result.num_docs)
        assert_equal(0, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(0, result.items.length)

        result = db.search("uri = file:///home/rast/hare.txt")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("uri = file:///home/rast/yuki.txt")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(0, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end

      assert_equal(false, file_exist?(db_name, "properties/old_uri.ngm"))
      assert_equal(false, file_exist?(db_name, "properties/old_uri.pos"))
      assert_equal(false, file_exist?(db_name, "properties/old_uri.rng"))
      assert_equal(false, file_exist?(db_name, "properties/old_uri.pfl"))
      assert_equal(false, file_exist?(db_name, "properties/old_uri.inv"))
      assert_equal(false, file_exist?(db_name, "properties/old_uri.db"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.ngm"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.pos"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.rng"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.pfl"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.inv"))
      assert_equal(false, file_exist?(db_name, "properties/new_uri.db"))
    end

    def optimize_test_segv
      options = {
        "encoding" => "utf8",
        "preserve_text" => false,
        "properties" => [],
      }
      db_name = generate_db_name
      remove_doc_id = nil
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("abc", {})
        remove_doc_id = db.register("def", {})
        db.register("ghi", {})
      end

      LocalDB.open(db_name) do |db|
        db.delete(remove_doc_id)
      end

      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("ghi")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end

      LocalDB.optimize(db_name, {"squeeze_doc_id" => true})

      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("ghi")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end

      options = {
        "encoding" => "utf8",
#        "preserve_text" => false,
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          }
        ]
      }

      db_name = generate_db_name
      remove_doc_id = nil
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("abc", {"filename" => "abc.txt"})
        remove_doc_id = db.register("def", {"filename" => "def.txt"})
        db.register("ghi", {"filename" => "ghi.txt"})
      end

      LocalDB.open(db_name) do |db|
        db.delete(remove_doc_id)
      end

      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("ghi")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(3, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end

      LocalDB.optimize(db_name, {"squeeze_doc_id" => true})

      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("ghi")
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end
    end

    def optimize_test_rollback
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "filename",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => true,
          }
        ]
      }

      db_name = generate_db_name
      remove_doc_id = nil
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        db.register("text-1", {"filename" => "1"})
        db.register("text-2", {"filename" => "2"})
      end
      LocalDB.open(db_name) do |db|
        db.delete(1)
      end

      begin
        FileUtils.mkdir(File.join(db_name, "new_text.rng"))
        assert_raise(Rast::BDBError) do
          LocalDB.optimize(db_name)
        end
      ensure
        FileUtils.rmdir(File.join(db_name, "new_text.rng"))
      end
      check_registered_text_index(File.join(db_name, "text"), "text-1")
      check_removed_files(db_name, "new_")
      check_removed_files(File.join(db_name, "properties"), "new_")

      begin
        FileUtils.mkdir(File.join(db_name, "old_text.rng"))
        assert_raise(Rast::AprError) do
          LocalDB.optimize(db_name)
        end
      ensure
        FileUtils.rmdir(File.join(db_name, "old_text.rng"))
      end
      check_registered_text_index(File.join(db_name, "text"), "text-1")
      check_removed_files(db_name, "new_")
      check_removed_files(File.join(db_name, "properties"), "new_")
      check_removed_files(db_name, "old_")
      check_removed_files(File.join(db_name, "properties"), "old_")
    end

    def test_encoding
      options = {
        "encoding" => "utf8",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal("UTF-8", db.encoding)
      end

      options = {
        "encoding" => "euc_jp",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal("EUC-JP", db.encoding)
      end

      options = {
        "encoding" => "mecab_euc_jp",
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name) do |db|
        assert_equal("EUC-JP", db.encoding)
      end
    end

    private

    def file_exist?(db_name, filename)
      return File.exist?(File.expand_path(filename, db_name))
    end

    def dir_exist?(db_name, dir_name)
      return File.directory?(File.expand_path(dir_name, db_name)) 
    end

    def bdb_db_exist?(db_name, filename, type)
      begin
        type.open(File.expand_path(filename, db_name)) do |db|
        end
        return true
      rescue
        return false
      end
    end

    def btree_db_exist?(db_name, filename)
      bdb_db_exist?(db_name, filename, BDB::Btree)
    end

    def hash_db_exist?(db_name, filename)
      bdb_db_exist?(db_name, filename, BDB::Hash)
    end

    def recno_db_exist?(db_name, filename)
      bdb_db_exist?(db_name, filename, BDB::Recno)
    end

    def property_btree_db_exist?(db_name, filename)
      return btree_db_exist?(db_name, File.join("properties", filename))
    end

    def property_file_exist?(db_name, filename)
      return file_exist?(db_name, File.join("properties", filename))
    end

    def check_exist_text_index(db_name, path, is_exist = true)
      assert_equal(is_exist, btree_db_exist?(db_name, path + ".ngm"))
      assert_equal(is_exist, btree_db_exist?(db_name, path + ".rng"))
      assert_equal(is_exist, file_exist?(db_name, path + ".pos"))
      assert_equal(is_exist, file_exist?(db_name, path + ".pfl"))
    end

    def check_not_exist_text_index(db_name, path)
      check_exist_text_index(db_name, path, false)
    end

    def check_doc_info(db_name, expected_max_doc_id, expected_num_doc)
      File.open(File.expand_path("doc_info", db_name)) do |f|
        max_doc_id, num_doc = *f.read.unpack("I*")
        assert_equal(expected_max_doc_id, max_doc_id)
        assert_equal(expected_num_doc, num_doc)
      end
    end

    def check_doc_property(db_name, doc_id,
                           expected_delete_flag, expected_num_chars)
      BDB::Btree.open(File.join(db_name, "properties.db"), nil, 0) do |db|
        s = db[[doc_id].pack("I")]
        delete_flag = s.slice!(0, 1).unpack("C")[0]
        num_chars = s.slice!(0, 4).unpack("I")[0]

        assert_equal(expected_delete_flag, delete_flag)
        assert_equal(expected_num_chars, num_chars)
      end
    end

    def check_registered_text_index(text_index_name, text, is_registered = true)
      text_index = TextIndex.new(text_index_name)
      res = text_index.search(text)
      assert_equal(!is_registered, res.candidates.empty?)
    end

    def check_not_registered_text_index(text_index_name, text)
      check_registered_text_index(text_index_name, text, false)
    end

    def check_property_inv(db_name, property_name, property_value,
                           *expected_doc_ids)
      path = File.join(db_name, "properties", property_name + ".inv")
      BDB::Btree.open(path, nil, BDB::RDONLY) do |db|
        c = db.cursor
        key, value = c.set(property_value)
        for doc_id in expected_doc_ids
          assert_equal(property_value, key, "property not found")
          assert_equal(doc_id, value.unpack("I*")[0])
          key, value = c.next
        end
        assert_not_equal(property_value, key, "too many hit")
      end
    end

    def check_summary_text(db_name, doc_id, text)
      LocalDB.open(db_name) do |db|
        assert_equal(text, db.get_text(doc_id))
      end
      path = File.join(db_name, "text.db")
      BDB::Recno.open(path) do |db|
        assert_equal(text, db[doc_id])
      end
    end

    def check_removed_files(dir, prefix)
      Dir.glob(File.join(dir, prefix) + "*") do |path|
        flunk("file exist: #{path}")
      end
    end
  end
end
