/*************************************************************************
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005-2008 by Kohei Yoshida.
 *    1039 Kingsway Dr., Apex, NC 27502, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _SCSOLVER_NUMERIC_FUNCOBJ_HXX_
#define _SCSOLVER_NUMERIC_FUNCOBJ_HXX_

#include <vector>
#include <string>
#include <memory>
#include <exception>

namespace scsolver { namespace numeric {

class SingleVarFuncObj;

class VarSizeException : public ::std::exception
{
public:
	virtual const char* what() const throw();
};

/**
 * Base class for function object to be used for NLP algorithms.
 */
class BaseFuncObj
{
public:
    BaseFuncObj();
    BaseFuncObj(const BaseFuncObj& r);
    virtual ~BaseFuncObj() = 0;

    virtual void getVars(::std::vector<double>& rVars) const = 0;
    virtual double getVar(size_t index) const = 0;
    virtual void setVars(const ::std::vector<double>& vars) const = 0;
    virtual void setVar(size_t index, double var) const = 0;
    virtual size_t getVarCount() const = 0;

    virtual double eval() const = 0;

    /**
     * Return a display-friendly function string. 
     */
    virtual const ::std::string getFuncString() const = 0;

    double operator()(const ::std::vector<double>& vars) const;
    SingleVarFuncObj& getSingleVarFuncObj(size_t varIndex);
    SingleVarFuncObj& getSingleVarFuncObjByRatio(const ::std::vector<double>& ratios);

private:
    ::std::auto_ptr<SingleVarFuncObj> m_pSVFuncObj;
};

// ----------------------------------------------------------------------------

class SimpleFuncObj : public BaseFuncObj
{
public:
    SimpleFuncObj();
    SimpleFuncObj(size_t varCount);
    SimpleFuncObj(const SimpleFuncObj& r);
    virtual ~SimpleFuncObj();

    virtual void getVars(::std::vector<double>& rVars) const;
    virtual double getVar(size_t index) const;
    virtual void setVars(const ::std::vector<double>& vars) const;
    virtual void setVar(size_t index, double var) const;
    virtual size_t getVarCount() const;

    virtual double eval() const = 0;

    /**
     * Return a display-friendly function string. 
     */
    virtual const ::std::string getFuncString() const = 0;

private:
    mutable ::std::vector<double> m_vars;
};

// ----------------------------------------------------------------------------

/** 
 * Non-linear function object that only has one variable.  Used for a
 * line-search algorithm.
 */
class SingleVarFuncObj
{
public:
    SingleVarFuncObj();
    virtual ~SingleVarFuncObj() = 0;

    virtual void setVar(double var) = 0;
    virtual double getVar() const = 0;
    virtual double eval() const = 0;

    /** 
     * Return a display-friendly function string (e.g. x^3 + 2*x^2 + 4). 
     */
    virtual const ::std::string getFuncString() const = 0;

    double operator()(double var);
};


}}

#endif

