/****************************************************************************
** Simplification et complétion de la classe QFtp
** Créé par Anacr0x : anacr0x@free.fr
**
** Sous licence GPL v2 ou plus
****************************************************************************/

#ifndef MYFTP_H
#define MYFTP_H

#include <QtCore>
#include <QtGui>
#include <QtNetwork>
#include "mytreewidget.h"
#include "myftptransfercontrol.h"
#include "mydd.h"

/*************************************************************************/
//! Sauvegarde la liste des fichiers ftp

/*! FileList est une petite liste chaînée conservant simplement tous les
QUrlInfo des fichiers/dossiers présent sur le ftp dans le répertoire
courant afin de conserver à tout instant la liste des fichiers.*/
/*************************************************************************/

struct FileList
{
	/*! Stock les infos de chaque fichiers */
	QUrlInfo ui;
	/*! Permet d'accéder au fichier précédent */
	FileList *before;
};


/*************************************************************************/
//! S'occupe de toute la gestion ftp

/*! MyFtp est la classe pivot de la gestion ftp. Elle utilise MyTreeWidget
et MyFtpTransferControl.\n
\n
Elle possède toute les fonctions de QFtp, de MyFtpTransferControl et en rajoute
certaines.\n
\n
Elle est basé sur une utilisation de MyTreeWidget et nombre de ses fonctions
en tire partie. L'affichage des fichiers ainsi que la récupération de la
sélection se fait souvent automatiquement.\n
\n
De plus, elle simplifie le système des identifiants de chaque action
 en listant et nommant toute action lancé a partir de MyFtp.\n
Elle conserve également la liste des fichiers d'un répertoire du ftp
(grâce à FileList). */
/*************************************************************************/

class MyFtp : public MyFtpTransferControl
{
	Q_OBJECT

private:
	/*! Stock le chemin complet du ftp */
	QString path;

public:
	/*! Pointeur sur le MyDD utilisé pour la gestion des fichiers locaux */
	MyDD *dd;

	/*! Les différentes actions possibles : None est l'état courant, les autres
	entraînent des actions différentes */
	enum FtpCurrentAction {None, ChangePath, DelDir, Download, DownloadDir, Upload, UploadDir, Abort};

	/*! Constructeur : initialise les variables et connecte les signaux utiles\n
	ftp = liste où sont affiché les fichiers ftp\n
	local = liste où sont affiché les fichiers locaux\n
	ftpPath = lineEdit où est affiché le path du ftp\n
	localPath = lineEdit où est affiché le path du qdir\n
	\warning les MyTreeWidget ne doivent pas être nulles, elles sont obligatoire pour
	l'affichage. */
	MyFtp(MyTreeWidget *ftp, MyTreeWidget *local, QLineEdit *ftpPath = NULL, QLineEdit *localPath = NULL);

	/*-----------Voici les fonctions utilisables dans un programme extérieur-------------*/
	/*-----------------------( ne pas s'occuper pas du reste )---------------------------*/

	/*! Lance la connexion ftp */
	void connectFtp(QString host, int port, QString username, QString password, bool showFiles = true);

	/*! Ferme la connexion ftp, réinitialise idList et détruit les listes de fichiers et de transferts */
	void disconnectFtp();

	/*! Retourne le chemin complet du ftp */
	inline const QString& getPath()
	{
		return path;
	}

	/*! Entre dans le dossier sélectionné de la listView, affiche les fichiers si display == true,
	puis modifie le chemin du QLineEdit */
	void changePath(bool display = true);

	/*! Entre dans le dossier spécifié en paramètre, affiche les fichiers si display == true,
	puis modifie le chemin du QLineEdit */
	void changePath(const QString &FtpPath, bool display = true);

	/*! Voir changePath(const QString &FtpPath, bool display) */
	inline void changePath(const char *FtpPath, bool display = true)
	{
		changePath(QString(FtpPath), display);
	};

	/*! Récupère la liste des fichiers du ftp et l'affiche si display == true */
	void getList(bool display = true);

	/*! Affiche la liste déjà obtenu par getList() des dossiers/fichiers */
	void displayList();

	/*! Permet de donné la liste complète des sous-dossiers et fichiers \n
	voir signal scanDirFinished(const QString &) pour récupérer la liste */
	void scanDir(QString FtpPath);

	/*! Permet de donné la liste complète des sous-dossiers et fichiers \n
	voir signal scanDirFinished(const QString &) pour récupérer la liste */
	void scanDirBis();

	/*! Trouve un fichier dans la liste Ftp et précise s'il existe \n
	Retourne un pointeur sur le fichier ou NULL s'il n'a pas été trouvé */
	FileList* isFileExist(const QString &file);

	/*! Télécharge toute la sélection :\n
	destDir = nom du dossier destinataire */
	void downloadSelection(const QString &destDir);

	/*! Télécharge un fichier :\n
	filename = nom ou chemin du fichier du ftp\n
	dest = nom ou chemin du fichier du disque dur\n
	wait = précise si l'on doit attendre avant de lancer le transfer\n
	Retourne true si le transfert est accepté*/
	bool downloadFile(const QString &filename, const QString &dest);

	/*! Télécharge tous les fichiers/dossiers du dossier passé en param*/
	void downloadDir(const QString &dir);

	/*! Envoi toute la sélection :\n
	destDir = nom du dossier destinataire */
	void uploadSelection(const QString &destDir);

	/*! Envoi un fichier :\n
	filename = nom ou chemin du fichier du disque dur\n
	dest = nom ou chemin du fichier du ftp\n
	wait = précise si l'on doit attendre avant de lancer le transfer\n
	Retourne true si le transfert est accepté */
	bool uploadFile(const QString &filename, const QString &dest);

	/*! Envoi sur le ftp tous les fichiers/dossiers du dossier passé en param */
	void uploadDir(const QString &absLocalDir);

	/*! Supprime du ftp le fichier/dossier (même non vide) sélectionné dans la listbox */
	void delSelection();

	/*! Supprime un dossier (vide ou non) */
	void delDir(const QString &dir);

	/*! Renomme le dossier/fichier sélectionné dans la listbox en newName.\n
	Si restoreExt==true l'extension est remise\n
	\warning Renomme UNIQUEMENT le 1er item sélectionné */
	void renameSelection(QString newName, bool restoreExt = true);

	/*! Retourne l'action en cours du ftp */
	MyFtp::FtpCurrentAction getCurrentAction();

	/*! \warning à ne pas utiliser à la va-vite */
	inline void changeCurrentAction(FtpCurrentAction state);

signals:
	/*! Signal envoyé lorsque le ftp change d'état */
	void currentActionChanged(MyFtp::FtpCurrentAction state);

	/*! Signal envoyé lorsque le scannage de dossier est fini \n
	Cette fonction renvoie la liste complète de tous les fichiers/dossiers */
	void scanDirFinished(const QString &list);

	/*! Signal envoyé lorsqu'un changement de dossier se termine */
	void changePathFinished();

	/*! Signal envoyé lorsqu'une commande list se termine */
	void listFinished(int id);

	/*! Signal envoyé lorsqu'un suppression de fichier ftp se termine */
	void delSelectionFinished();

	/*! Signal envoyé lorsqu'une erreur s'est produite */
	void detailledError(const QString &error);

	/*-----------------------------------------------------------------------------------*/

private:
	/*! Pointeur sur le MyTreeWidget utilisé pour l'affichage des fichiers ftp */
	MyTreeWidget *lvFtp;

	/*! Pointeur sur le QLineEdit où l'on affiche le chemin du ftp */
	QLineEdit *leFtpPath;

	/*! Permet de savoir si l'on doit afficher ou non les fichiers du ftp */
	bool bDisplayListOrNot;

	/*! Pointeur sur le dernier fichier répertorié */
	FileList *lastFile;

	/*! Evite de déconnecter le client quand inactif */
	QTimer noDeco;

	/*! Stock l'id de la commande list en cour */
	QList <int> idList;

	/*! Indique le mode d'écrasement des fichiers */
	QMessageBox::StandardButton eraseStateBtn;

	/*! Stock l'action en cour du MyFtp afin d'effectuer certaines actions spéciales */
	FtpCurrentAction currentAction;

	/*! Permet de savoir si un scannage est en cours */
	bool isScanning;

	/*! Stock tous les fichiers trouvés par scanDir(QString) */
	QString ScanDirList;

	/*! Stock le path du dossier scanné par scanDir(QString) */
	QString TmpPath;

	/*! Stock les paths de dossier à crée par uploadDir(const QString&) */
	QString dirToMake;

	/*! Stock les différentes commandes en attente */
	QString queueAction;

	/*! Même effet que getList() mais utilisable dans n'importe quel état (à ne pas utiliser
	directement)\n
	Retourne l'id de la fonction list ou -1 en cas d'échec*/
	int myListFunction(bool display = true, const QString &dir = QString::null);

	/*! Suite de downloadDir(const QString &) : après le scannage de dossier */
	void downloadDirBis(const QString &list);

	/*! Suite de uploadDir(const QString &) : après le scannage de dossier */
	void uploadDirBis();

	/*! Suite de delDir(const QString &) : après le scannage de dossier */
	void delDirBis(const QString &list);

	/*! Efface dans la liste le fichier pointé par file\n
	Retourne true si l'opération à réussi */
	bool deleteOneFile(FileList *file);

	/*! Efface dans la liste tous les fichiers */
	void deleteAllFile();


private slots:
	/*! Lance régulièrement un signal NOOP afin de ne pas être déconnecté automatiquement */
	void slotNoDecoTimer();

	/*! Lance la prochaine action en attente */
	void startNextQueueAction();

	/*! Utilisé pour récupérer la liste des fichiers du ftp (ne pas reconnecter le signal) */
	void slotListInfo(const QUrlInfo &i);

	/*! Utilisé pour savoir quand une action est commencée (ne pas reconnecter le signal) */
	void slotCommandStarted(int id);

	/*! Utilisé pour savoir quand une action est terminée (ne pas reconnecter le signal) */
	void slotCommandFinished(int id, bool error);

	/*! Utilisé pour connaître le chemin du ftp (ne pas reconnecter le signal) */
	void slotRawCommandReply(int replyCode, const QString &detail);
};

#endif
