/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to track operations.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgtrackerpluginwidget.h"
#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgobjectmodel.h"
#include "skgsortfilterproxymodel.h"
#include "skgdocumentbank.h"
#include "skgtrackerobject.h"
#include "skgtransactionmng.h"

#include <QDomDocument>
#include <QKeyEvent>

SKGTrackerPluginWidget::SKGTrackerPluginWidget ( SKGDocument* iDocument )
        :SKGTabPage ( iDocument )
{
    SKGTRACEIN ( 1, "SKGTrackerPluginWidget::SKGTrackerPluginWidget" );

    ui.setupUi ( this );

    ui.kAddButton->setIcon ( KIcon ( "list-add" ) );
    ui.kModifyButton->setIcon ( KIcon ( "dialog-ok-apply" ) );

    objectModel = new SKGObjectModel ( static_cast<SKGDocumentBank*>(getDocument()), "v_refund_display", "1=0", this, "", false );
    SKGSortFilterProxyModel* modelproxy = new SKGSortFilterProxyModel ( this );
    modelproxy->setSourceModel ( objectModel );
    modelproxy->setSortRole ( Qt::UserRole );
    modelproxy->setDynamicSortFilter ( true );
    ui.kView->setModel ( modelproxy );

    ui.kFilterEdit->setProxy ( modelproxy );

    //Add registered global action in contextual menu
    if ( SKGMainPanel::getMainPanel() )
    {
        ui.kView->insertGlobalAction ( "edit_delete" );
        ui.kView->insertGlobalAction ();
        ui.kView->insertGlobalAction ( "edit_find" );
        ui.kView->insertGlobalAction ( "open_report" );

        ui.kView->resizeColumnToContents ( 0 );
    }

    ui.kTitle->setPixmap ( KIcon ( "dialog-information" ).pixmap ( 22, 22 ), KTitleWidget::ImageLeft );

    connect ( ( const QObject* ) getDocument(), SIGNAL ( tableModified ( QString, int ) ), this, SLOT ( dataModified ( QString, int ) ), Qt::QueuedConnection );
    dataModified ( "", 0 );

    connect ( ui.kView, SIGNAL ( selectionChangedDelayed() ), this, SLOT ( onSelectionChanged() ) );

    connect ( objectModel, SIGNAL ( beforeReset() ), ui.kView, SLOT ( saveSelection() ) );
    connect ( objectModel, SIGNAL ( afterReset() ), ui.kView, SLOT ( resetSelection() ) );

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter ( this );
}

SKGTrackerPluginWidget::~SKGTrackerPluginWidget()
{
    SKGTRACEIN ( 1, "SKGTrackerPluginWidget::~SKGTrackerPluginWidget" );
    objectModel=NULL;
}

bool SKGTrackerPluginWidget::eventFilter ( QObject *object, QEvent *event )
{
    if ( event && event->type() == QEvent::KeyPress )
    {
        QKeyEvent *keyEvent = ( QKeyEvent * ) event;
        if ( ( keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter ) && object==this )
        {
            if ( QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddButton->isEnabled() )
            {
                ui.kAddButton->click();
            }
            else if ( QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyButton->isEnabled() )
            {
                ui.kModifyButton->click();
            }
        }
    }

    return false;
}

void SKGTrackerPluginWidget::dataModified ( const QString& iTableName, int iIdTransaction )
{
    SKGTRACEIN ( 1, "SKGTrackerPluginWidget::dataModified" );
    Q_UNUSED ( iIdTransaction );

    if ( iTableName=="refund" || iTableName.isEmpty() )
    {
        //Correction bug 2299394 vvv
        if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^
        onEditorModified();
    }
}

void SKGTrackerPluginWidget::onSelectionChanged()
{
    SKGTRACEIN ( 10, "SKGTrackerPluginWidget::onSelectionChanged" );

    SKGObjectBase::SKGListSKGObjectBase objs=getSelectedObjects();
    if ( objs.count() )
    {
        SKGTrackerObject obj=objs.at ( 0 );
        ui.kNameInput->setText ( obj.getName() );
        ui.kCommentEdit->setText ( obj.getComment() );
    }

    onEditorModified();
    emit selectionChanged();
}

SKGObjectBase::SKGListSKGObjectBase SKGTrackerPluginWidget::getSelectedObjects()
{
    return ui.kView->getSelectedObjects();
}

int SKGTrackerPluginWidget::getNbSelectedObjects()
{
    return ui.kView->getNbSelectedObjects();
}

QString SKGTrackerPluginWidget::getState()
{
    SKGTRACEIN ( 10, "SKGTrackerPluginWidget::getState" );
    QDomDocument doc ( "SKGML" );
    QDomElement root = doc.createElement ( "parameters" );
    doc.appendChild ( root );

    root.setAttribute ( "hideUseless", ui.kHideUseless->checkState() ==Qt::Checked ? "Y" : "N" );
    root.setAttribute ( "filter", ui.kFilterEdit->lineEdit()->text() );

    //Memorize table settings
    root.setAttribute ( "view", ui.kView->getState() );
    return doc.toString();
}

void SKGTrackerPluginWidget::setState ( const QString& iState )
{
    SKGTRACEIN ( 10, "SKGTrackerPluginWidget::setState" );
    QDomDocument doc ( "SKGML" );
    doc.setContent ( iState );
    QDomElement root = doc.documentElement();

    QString hideUseless=root.attribute ( "hideUseless" );
    QString filter=root.attribute ( "filter" );

    if ( !hideUseless.isEmpty() ) ui.kHideUseless->setCheckState ( hideUseless=="Y" ? Qt::Checked : Qt::Unchecked );
    if ( !filter.isEmpty() ) ui.kFilterEdit->setText ( filter );

    if ( objectModel )
    {
        bool previous=objectModel->blockRefresh ( true );
        onFilterChanged();
        objectModel->blockRefresh ( previous );
    }

    //!!! Must be done here after onFilterChanged
    ui.kView->setState ( root.attribute ( "view" ) );
}

QString SKGTrackerPluginWidget::getDefaultStateAttribute()
{
    return "SKGREFUND_DEFAULT_PARAMETERS";
}

QWidget* SKGTrackerPluginWidget::getWidgetForPrint()
{
    return ui.kView;
}

void SKGTrackerPluginWidget::onEditorModified()
{
    _SKGTRACEIN ( 10, "SKGTrackerPluginWidget::onEditorModified" );
    int nb=getNbSelectedObjects();
    ui.kModifyButton->setEnabled ( !ui.kNameInput->text().isEmpty() && nb==1 );
    ui.kAddButton->setEnabled ( !ui.kNameInput->text().isEmpty() && nb<=1 );
}

void SKGTrackerPluginWidget::onFilterChanged()
{
    SKGTRACEIN ( 10, "SKGTrackerPluginWidget::onFilterChanged" );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    //Compute where clause
    QString filter;

    if ( ui.kHideUseless->checkState() ==Qt::Checked )
    {
        filter=" t_close='N'";
    }

    //Update model
    if ( objectModel )
    {
        objectModel->setFilter ( filter );
        objectModel->refresh();

        //Correction bug 2299394 vvv
        if ( ui.kView->isAutoResized() ) ui.kView->resizeColumnsToContentsDelayed();
        //Correction bug 2299394 ^^^
    }

    QApplication::restoreOverrideCursor();
}

void SKGTrackerPluginWidget::onAddTracker()
{
    SKGError err;
    _SKGTRACEINRC ( 10, "SKGTrackerPluginWidget::onAddTracker", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

    QString name=ui.kNameInput->text();
    SKGTrackerObject tracker;
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Tracker creation '%1'", name )  , err );

        err=SKGTrackerObject::createTracker ( static_cast<SKGDocumentBank*>(getDocument()), name, tracker );
        if ( err.isSucceeded() ) err=tracker.setComment ( ui.kCommentEdit->text() );
        if ( err.isSucceeded() ) err=tracker.save();
    }
    //status bar
    if ( err.isSucceeded() )
    {
        err=SKGError ( 0, i18nc ( "Successful message after an user action", "Tracker '%1' created", name ) );
        ui.kView->selectObject ( tracker.getUniqueID() );
    }
    else err.addError ( ERR_FAIL, i18nc ( "Error message", "Tracker creation failed" ) );
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGTrackerPluginWidget::onModifyTracker()
{
    SKGError err;
    _SKGTRACEINRC ( 10, "SKGTrackerPluginWidget::onModifyTracker", err );
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();

    int nb=selection.count();
    if ( nb==1 )
    {
        SKGBEGINTRANSACTION ( *getDocument(), i18nc ( "Noun, name of the user action", "Tracker '%1' update", ui.kNameInput->text() )  , err );

        //Modification of category object
        SKGTrackerObject tracker=selection[0];
        err=tracker.setName ( ui.kNameInput->text() );
        if ( err.isSucceeded() ) err=tracker.setComment ( ui.kCommentEdit->text() );
        if ( err.isSucceeded() ) err=tracker.save();

        //status bar
        if ( err.isSucceeded() ) err=SKGError ( 0, i18nc ( "Successful message after an user action", "Tracker '%1' updated", ui.kNameInput->text() ) );
        else err.addError ( ERR_FAIL, i18nc ( "Error message", "Tracker update failed" ) );
    }
    QApplication::restoreOverrideCursor();

    //Display error
    SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
}

void SKGTrackerPluginWidget::onDoubleClick()
{
    _SKGTRACEIN ( 10, "SKGTrackerPluginWidget::onDoubleClick" );

    //Get selection
    SKGObjectBase::SKGListSKGObjectBase selection=getSelectedObjects();
    if ( selection.count() ==1 )
    {
        SKGTrackerObject tracker ( selection.at ( 0 ) );
        QString name=tracker.getName();

        //Build where clause and title
        QString wc="t_REALREFUND = '"+SKGServices::stringToSqlString ( name ) +'\'';
        QString title=i18nc ( "Noun, a list of items", "Sub operations followed by tracker '%1'",  name );

        //Open
        if ( QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier )
        {
            //Call debug plugin
            QDomDocument doc ( "SKGML" );
            QDomElement root = doc.createElement ( "parameters" );
            doc.appendChild ( root );
            root.setAttribute ( "sqlOrder", "SELECT * from v_operation_consolidated WHERE "+wc );

            SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Debug plugin" ), -1, doc.toString() );
        }
        else
        {
            //Call operation plugin
            QDomDocument doc ( "SKGML" );
            doc.setContent ( getDocument()->getParameter ( "SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS" ) );
            QDomElement root = doc.documentElement();
            if ( root.isNull() )
            {
                root=doc.createElement ( "parameters" );
                doc.appendChild ( root );
            }

            root.setAttribute ( "operationTable", "v_operation_consolidated" );
            root.setAttribute ( "operationWhereClause", wc );
            root.setAttribute ( "title", title );
            root.setAttribute ( "title_icon", "skrooge_tracker" );
            root.setAttribute ( "currentPage", "-1" );

            SKGMainPanel::getMainPanel()->openPage ( SKGMainPanel::getMainPanel()->getPluginByName ( "Skrooge operation plugin" ), -1, doc.toString() );
        }
    }
}

void SKGTrackerPluginWidget::cleanEditor()
{
    if ( getNbSelectedObjects() ==0 )
    {
        ui.kNameInput->setText ( "" );
        ui.kCommentEdit->setText ( "" );
    }
}

void SKGTrackerPluginWidget::activateEditor()
{
    ui.kNameInput->setFocus();
}

bool SKGTrackerPluginWidget::isEditor()
{
    return true;
}

QWidget* SKGTrackerPluginWidget::zoomableWidget() const
{
    return ui.kView;
}
#include "skgtrackerpluginwidget.moc"


