/*
 * stream-engine-main.c - startup and shutdown of stream-engine
 * Copyright (C) 2005 Collabora Ltd.
 * Copyright (C) 2005 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#define _GNU_SOURCE
#include <sched.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <sys/time.h>

#ifndef RLIMIT_RTTIME
# define RLIMIT_RTTIME 15
#endif

#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>
#include <glib/gstdio.h>

#include <gtk/gtk.h>

#include <dbus/dbus-glib.h>
#include <gst/gst.h>

#include <telepathy-glib/debug.h>
#include <telepathy-glib/errors.h>

#include "api/api.h"
#include "tp-stream-engine.h"

GSource *timeout = NULL;
GMainLoop *mainloop = NULL;
TpStreamEngine *stream_engine = NULL;
gboolean connections_exist = FALSE;

/* timeout_id is initialized to 0 to mean no timeout has been added.
 * g_timeout_add() returns an event source ID greater than 0. */
guint timeout_id = 0;

gboolean forced_exit_in_progress = FALSE;

#define DIE_TIME 5000

/* watchdog barks every 5 seconds, and if we're unresponsive, bites us in 30 */
#define WATCHDOG_BARK 5
#define WATCHDOG_BITE 30


static gboolean
kill_stream_engine (gpointer data G_GNUC_UNUSED)
{
  if (!g_getenv ("STREAM_ENGINE_PERSIST") && !connections_exist)
    {
      g_debug("no channels are being handled, and timed out");
      g_object_unref (stream_engine);
      g_main_loop_quit (mainloop);
    }

  return FALSE;
}

static void
handling_channel (TpStreamEngine *stream_engine G_GNUC_UNUSED)
{
  connections_exist = TRUE;
  if (timeout_id != 0)
    g_source_remove (timeout_id);
}

static void
no_more_channels (TpStreamEngine *stream_engine G_GNUC_UNUSED)
{
  if (timeout_id != 0)
    {
      g_source_remove (timeout_id);
    }
  connections_exist = FALSE;
  timeout_id = g_timeout_add(DIE_TIME, kill_stream_engine, NULL);
}

static void
shutdown (TpStreamEngine *stream_engine)
{
  g_debug ("Unrefing stream_engine and quitting");
  g_object_unref (stream_engine);
  g_main_loop_quit (mainloop);
}

#if 0
static void
dsp_crashed (gpointer dummy)
{
  if (stream_engine)
  {
    tp_stream_engine_error (stream_engine, TP_MEDIA_STREAM_ERROR_MEDIA_ERROR,
        "DSP Crash");
    g_object_unref (stream_engine);
    g_main_loop_quit (mainloop);
  }
}
#endif

static gboolean
dump_dot_file (gpointer data G_GNUC_UNUSED)
{
  TpStreamEngine *engine = tp_stream_engine_get ();
  GstElement *pipeline = NULL;

  g_object_get (engine, "pipeline", &pipeline, NULL);

  if (pipeline)
    {
      GST_DEBUG_BIN_TO_DOT_FILE_WITH_TS (GST_BIN (pipeline),
          GST_DEBUG_GRAPH_SHOW_ALL, "stream-engine-pipeline");
      gst_object_unref (pipeline);
    }

  return FALSE;
}

static void
got_sigusr1 (int i G_GNUC_UNUSED)
{
  g_idle_add (dump_dot_file, NULL);
}

/* every time the watchdog barks, schedule a bite */
static gboolean
watchdog_bark (gpointer data G_GNUC_UNUSED)
{
  alarm (WATCHDOG_BITE);
  return TRUE;
}

/* if it ever catches us, we're gone */
static void
watchdog_bite (int sig G_GNUC_UNUSED)
{
  printf ("telepathy-stream-engine: bitten by the watchdog, aborting!\n");
  abort ();
}

int main(int argc, char **argv)
{
  GOptionContext *optcontext;
  GError *error = NULL;

  uid_t uid;
  struct rlimit rl;

  rl.rlim_max = 15;
  rl.rlim_cur = 15;
  if (setrlimit(RLIMIT_RTPRIO, &rl) < 0)
    g_warning("setrlimit rtprio: %s", strerror (errno));

  rl.rlim_max = 15;
  rl.rlim_cur = 15;
  if (setrlimit(RLIMIT_NICE, &rl) < 0)
    g_warning("setrlimit rtprio: %s", strerror (errno));

  /* In nanoseconds */
  rl.rlim_max = rl.rlim_cur = 30*1000*1000;

  if (setrlimit(RLIMIT_RTTIME, &rl) < 0)
    g_warning("setrlimit rttime: %s", strerror (errno));

  uid = getuid();
  if (uid != 0 && geteuid() == 0) {
    gint ret;
    ret = setresuid(uid, uid, uid);
    g_assert (ret >= 0);
    g_assert (getuid() == uid);
    g_assert (geteuid() == uid);
  }

  g_thread_init (NULL);

  optcontext = g_option_context_new ("Telepathy Stream Engine");
  g_option_context_add_group (optcontext, gst_init_get_option_group ());
  g_option_context_add_group (optcontext, gtk_get_option_group (TRUE));

  if (g_option_context_parse (optcontext, &argc, &argv, &error) == FALSE) {
    g_print ("%s\nRun '%s --help' to see a full list of available command line options.\n",
        error->message, argv[0]);
    return 1;
  }
  g_option_context_free (optcontext);

  g_set_application_name("Maemo Telepathy Stream Engine");
  g_setenv("PULSE_PROP_media.role", "phone", TRUE);

  tp_debug_divert_messages (g_getenv ("STREAM_ENGINE_LOGFILE"));
  tp_debug_set_flags (g_getenv ("STREAM_ENGINE_DEBUG"));
  /* FIXME: switch this project to use DEBUG() too */

  if (g_getenv ("STREAM_ENGINE_TIMING") != NULL)
    g_log_set_default_handler (tp_debug_timestamped_log_handler, NULL);

  signal (SIGUSR1, got_sigusr1);




  {
    GLogLevelFlags fatal_mask;

    fatal_mask = g_log_set_always_fatal (G_LOG_FATAL_MASK);
    fatal_mask |= G_LOG_LEVEL_CRITICAL;
    g_log_set_always_fatal (fatal_mask);
  }

  g_set_prgname("telepathy-stream-engine");

  stream_engine_cli_init ();

  mainloop = g_main_loop_new (NULL, FALSE);

  stream_engine = tp_stream_engine_get ();

  g_signal_connect (stream_engine, "handling-channel", 
                    (GCallback) handling_channel, NULL);

  g_signal_connect (stream_engine, "no-more-channels", 
                    (GCallback) no_more_channels, NULL);

  g_signal_connect (stream_engine, "shutdown-requested",
                    (GCallback) shutdown, NULL);

  tp_stream_engine_register (stream_engine);

  timeout_id = g_timeout_add(DIE_TIME, kill_stream_engine, NULL);

  if (g_getenv ("STREAM_ENGINE_NO_DOG") == NULL)
    {
      g_timeout_add (WATCHDOG_BARK * 1000, watchdog_bark, NULL);
      signal (SIGALRM, watchdog_bite);
    }

  g_debug("started");
  g_main_loop_run (mainloop);
  g_debug("finished");

  return 0;
}
