/*
 * playlist.c
 * Thomas Nemeth, le 08.01.2002
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "defines.h"
#include <ncurses.h>
#include "xmem.h"
#include "printlog.h"
#include "tmms.h"
#include "playlist.h"
#include "modules.h"
#include "cda_mod.h"


static PlayList *playlist = NULL;
static ListID current = 0;
static char *playlist_name = NULL;
static PlayMode play_mode = NRM_PLAY;


void playlist_add_all(const char *entry, const char *length)
{
        PlayList *p, *prev = NULL;

        if (!playlist)
        {
                playlist = xmalloc(sizeof(PlayList));
                p = playlist;
        }
        else
        {
                p = playlist;
                while (p->next)
                {
                        p = p->next;
                }
                p->next = xmalloc(sizeof(PlayList));
                prev = p;
                p = p->next;
        }
        p->file = xstrdup(entry);
        p->length = length ? xstrdup(length) : NULL;
        p->sel = 0;
        p->played = 0;
        p->next = NULL;
        p->prev = prev;
}


void playlist_add(const char *entry)
{
        playlist_add_all(entry, NULL);
}


void playlist_add_sort(const char *entry)
{
        PlayList *p, *prev = NULL, *next = NULL;

        if (!playlist)
        {
                playlist = xmalloc(sizeof(PlayList));
                p = playlist;
        }
        else
        {
                p = playlist;
                while (p && (strcoll(entry, p->file) >= 0))
                {
                        prev = p;
                        p = p->next;
                }
                next = p;
                p = xmalloc(sizeof(PlayList));
        }
        p->file = xstrdup(entry);
        p->length = NULL;
        p->sel = 0;
        p->next = next;
        p->prev = prev;
        if (prev)
        {
                prev->next = p;
        }
        else
        {
                playlist = p;
        }
        if (next)
        {
                next->prev = p;
        }
}


void playlist_del(int position)
{
        PlayList *p = playlist;
        int n = 1;

        while (p && (n != position))
        {
                p = p->next;
                n++;
        }

        if (!p)
        {
                return;
        }

        if (p == playlist)
        {
                playlist = playlist->next;
                if (playlist)
                {
                        playlist->prev = NULL;
                }
        }
        else
        {
                p->prev->next = p->next;
                if (p->next)
                {
                        p->next->prev = p->prev;
                }
        }
        FREE(p->file);
        FREE(p->length);
        free(p);

}


static PlayList *playlist_element(ListID n)
{
        ListID i = 1;
        PlayList *p = playlist;

        while (p && (i != n))
        {
                p = p->next;
                i++;
        }
        return p;
}


char *playlist_get(ListID n)
{
        PlayList *p = playlist_element(n);
        return p ? p->file : NULL;
}


char *playlist_get_length(ListID n)
{
        PlayList *p = playlist_element(n);
        return p ? p->length : NULL;
}


int set_current_length(int min, int sec)
{
        int r = FALSE;

        if ((min > 0) || ((min == 0) && (sec > 0)))
        {
                PlayList *p = playlist_element(current);

                if (! p->length)
                {
                        char l[10];
                        memset(l, 0, 10);
                        snprintf(l, 10, "%02d:%02d", min, sec);
                        p->length = xstrdup(l);
                }
                r = TRUE;
        }
        return r;
}


static void playlist_mvup(PlayList *pf, PlayList *pt)
{
        pf->prev = pt->prev;
        pf->next = pt;
        pt->prev = pf;
        if (pf->prev)
        {
                pf->prev->next = pf;
        }
        if (pt == playlist)
        {
                playlist = pf;
        }
}


static void playlist_mvdn(PlayList *pf, PlayList *pt)
{
        if (pf == playlist)
        {
                playlist = playlist->next;
        }
        pf->prev = pt;
        pf->next = pt->next;
        pt->next = pf;
        if (pf->next)
        {
                pf->next->prev = pf;
        }
}


void playlist_move(ListID from, ListID to)
{
        PlayList *pf = playlist_element(from);
        PlayList *pt = playlist_element(to);

        if (!pf || !pt)
        {
                return;
        }

        if (from != to)
        {
                if (pf->prev)
                {
                        pf->prev->next = pf->next;
                }
                if (pf->next)
                {
                        pf->next->prev = pf->prev;
                }

                if (from > to)
                {
                        playlist_mvup(pf, pt);
                }
                else if (from < to)
                {
                        playlist_mvdn(pf, pt);
                }
        }
}


char *playlist_current()
{
        char *entry;

        pthread_mutex_lock(&mutex_current);
        entry = playlist_get(current);
        pthread_mutex_unlock(&mutex_current);

        return entry;
}


ListID playlist_max()
{
        ListID i = 0;
        PlayList *p = playlist;

        while (p)
        {
                i++;
                p = p->next;
        }
        return i;
}


static void renum_selections(ListID removed)
{
        PlayList *p = playlist;

        while (p)
        {
                if (p->sel > removed)
                {
                        p->sel--;
                }
                p = p->next;
        }
}


void playlist_sel(ListID n, int sel)
{
        PlayList *p = playlist_element(n);
        if (p)
        {
                if (sel)
                {
                        p->sel = playlist_nb_sel() + 1;
                        printlog(3, "Element %d is #%d selected.\n", n, p->sel);
                }
                else
                {
                        renum_selections(p->sel);
                        p->sel = 0;
                }
        }
}


void playlist_tsel(ListID n)
{
        if (playlist_get_sel(n) != 0)
        {
                playlist_sel(n, FALSE);
        }
        else
        {
                playlist_sel(n, TRUE);
        }
}


ListID playlist_get_sel(ListID n)
{
        PlayList *p = playlist_element(n);
        if (p)
        {
                return p->sel;
        }
        return 0;
}


ListID playlist_get_id_by_sel(ListID sel_n)
{
        ListID i = 1;
        PlayList *p = playlist;

        while (p)
        {
                if (p->sel == sel_n)
                {
                        return i;
                }
                p = p->next;
                i++;
        }
        return 0;
}


ListID playlist_nb_sel()
{
        ListID n = 0;
        PlayList *p = playlist;

        while (p)
        {
                if (p->sel != 0)
                {
                        n++;
                }
                p = p->next;
        }
        return n;
}


ListID playlist_first_sel()
{
        ListID n = 1;
        PlayList *p = playlist;

        while (p)
        {
                if (p->sel == 1)
                {
                        return n;
                }
                p = p->next;
                n++;
        }
        return 0;
}


ListID playlist_last_sel()
{
        ListID i = 1, n = 0, s = 0;
        PlayList *p = playlist;

        while (p)
        {
                if (p->sel > s)
                {
                        s = p->sel;
                        n = i;
                }
                p = p->next;
                i++;
        }
        return n;
}


static ListID playlist_nb_played()
{
        PlayList *p = playlist;
        ListID n = 0;

        while (p)
        {
                if (p->played)
                {
                        n++;
                }
                p = p->next;
        }
        return n;
}


static int playlist_next_sel(ListID cur_sel, int rotate, int dir)
{
        PlayList *p = playlist;
        ListID n = 1;
        int found = FALSE;

        if (cur_sel + dir == 0)
                p = NULL;
        while (p)
        {
                if (p->sel == cur_sel + dir)
                {
                        break;
                }
                p = p->next;
                n++;
        }
        if (p)
        {
                current = n;
                found = TRUE;
        }
        else if (rotate)
        {
                if (dir == 1)
                {
                        current = playlist_first_sel();
                }
                else
                {
                        current = playlist_last_sel();
                }
                if (current != 0)
                {
                        found = TRUE;
                }
        }
        return found;
}


static int playlist_next_ordered(int rotate)
{
        int found = FALSE;

        if (playlist_nb_sel() != 0)
        {
                PlayList *p = playlist_element(current);
                found = playlist_next_sel(p->sel, rotate, + 1);
        }
        else
        {
                found = TRUE;
                current++;
                if (current > playlist_max())
                {
                        if (rotate)
                        {
                                current = 1;
                        }
                        else
                        {
                                current--;
                                found = FALSE;
                        }
                }
        }
        return found;
}


static int playlist_next_rnd(int rotate)
{
        PlayList *p = NULL;
        ListID max;
        int played = playlist_nb_played();
        int steps = 1 + (int)(played * rand() / (RAND_MAX + 1.0));

        max = playlist_nb_sel() > 0 ? playlist_nb_sel() : playlist_max();
        if (max == playlist_nb_played())
        {
                PlayList *l = playlist;
                while (l)
                {
                        l->played = FALSE;
                        l = l->next;
                }
                if (! rotate)
                {
                        return FALSE;
                }
        }
        while (steps != 0)
        {
                playlist_next_ordered(TRUE);
                p = playlist_element(current);
                if (p && (! p->played))
                {
                        steps--;
                }
        }
        p->played = TRUE;
        return TRUE;
}


int playlist_next(int rotate)
{
        int found;

        pthread_mutex_lock(&mutex_current);
        printlog(5, "Current title : %d\n", current);
        if (play_mode == RND_PLAY)
        {
                found = playlist_next_rnd(rotate);
        }
        else
        {
                found = playlist_next_ordered(rotate);
        }
        printlog(5, "Next title    : %d (%s)\n", current, found ? "OK" : "Stop");
        pthread_mutex_unlock(&mutex_current);
        return found;
}


int playlist_prev(int rotate)
{
        int found = TRUE;

        pthread_mutex_lock(&mutex_current);
        if (playlist_nb_sel() != 0)
        {
                PlayList *p = playlist_element(current);
                found = playlist_next_sel(p->sel, rotate, -1);
        }
        else
        {
                current--;
                if (current == 0)
                {
                        if (rotate)
                        {
                                current = playlist_max();
                        }
                        else
                        {
                                current++;
                                found = FALSE;
                        }
                }
        }
        pthread_mutex_unlock(&mutex_current);
        return found;
}


void playlist_free()
{
        while (playlist)
        {
                PlayList *p = playlist->next;
                FREE(playlist->file);
                free(playlist);
                playlist = p;
        }
        set_current(0);
        FREE(playlist_name);
}


void load_dir(const char *dirname)
{
        DIR *dir;
        struct dirent *dir_ent;

        if ((dir = opendir(dirname)) == NULL)
        {
                return;
        }
        while ((dir_ent = readdir(dir)) != NULL)
        {
                if (get_format(dir_ent->d_name))
                {
                        int len = strlen(dirname) + strlen(dir_ent->d_name) + 2;
                        char *filename = xmalloc(len);
                        sprintf(filename, "%s/%s", dirname, dir_ent->d_name);
                        playlist_add_sort(filename);
                        free(filename);
                }
        }
        closedir(dir);
}


void load_file(const char *filename)
{
        FILE *file;

        if (! filename)
        {
                return;
        }
        file = fopen(filename, "r");
        if (file == NULL)
        {
                return;
        }

        while (! feof(file))
        {
                char line[MAXSTRLEN + 1];
                memset(line, 0, MAXSTRLEN + 1);
                fgets(line, MAXSTRLEN, file);
                line[strlen(line) - 1] = 0;
                if ((line[0] != 0) && (line[0] != '#'))
                {
                        if (get_format(line))
                        {
                                playlist_add(line);
                        }
                }
        }
        fclose(file);
}


void load_playlist(const char *dataloc)
{
        struct stat s;

        if (stat(dataloc, &s) == -1)
        {
                return;
        }
        if (S_ISDIR(s.st_mode))
        {
                load_dir(dataloc);
        }
        else if (S_ISREG(s.st_mode))
        {
                load_file(dataloc);
        }
        else if (S_ISBLK(s.st_mode))
        {
                load_tracks_list(dataloc);
        }
        else
        {
                fprintf(stderr,
                        PROGRAM": not a regular file or dir (%s).\n",
                        dataloc);
        }
        FREE(playlist_name);
        playlist_name = xstrdup(dataloc);
        set_current(1);
}


void save_playlist(const char *filename)
{
        FILE *file;
        PlayList *p = playlist;

        if ((file = fopen(filename, "w")) == NULL)
        {
                return;
        }
        while (p)
        {
                fprintf(file, "%s\n", p->file);
                p = p->next;
        }
        fclose(file);
}


ListID get_current()
{
        ListID i;

        pthread_mutex_lock(&mutex_current);
        i = current;
        pthread_mutex_unlock(&mutex_current);

        return i;
}


void set_current(int c)
{
        pthread_mutex_lock(&mutex_current);
        current = c;
        pthread_mutex_unlock(&mutex_current);
}


int is_playable(const char *entry)
{
        struct stat s;

        /* http stream ? */
        if (is_http(entry))
        {
                return TRUE;
        }

        /* audio CD ? */
        if ((stat(playlist_name, &s) != -1) && S_ISBLK(s.st_mode))
        {
                return TRUE;
        }

        /* file exists ? */
        if (stat(entry, &s) == -1)
        {
                return FALSE;
        }

        /* regular file ? */
        if (S_ISREG(s.st_mode) || S_ISLNK(s.st_mode))
        {
                /* known regular file ? */
                return get_format(entry);
        }

        /* junk ! */
        return FALSE;
}


char *get_playlist_name()
{
        return playlist_name;
}


void toggle_playmode()
{
        if (play_mode == NRM_PLAY)
        {
                play_mode = RND_PLAY;
        }
        else
        {
                play_mode = NRM_PLAY;
        }
}


PlayMode get_playmode()
{
        return play_mode;
}

