/***************************************************************************
@author:    Fabian Wuertz <xadras@sidux.com>
@copyright: Fabian Wuertz <xadras@sidux.com>
@license:   GPL2, GPL3

@project:   Waheela
@file:      waheela.cpp   
***************************************************************************/

#include <kapplication.h>
#include <klocale.h>
#include <kconfigskeleton.h>
#include <kconfigdialog.h>
#include <kstandarddirs.h>
#include <kglobal.h>
#include <KPasswordDialog>
#include <KDesktopFile>
#include <KColorDialog>

#include "waheela.h"
#include "playercontroller.h"


//------------------------------------------------------------------------------
//--- init ---------------------------------------------------------------------
//------------------------------------------------------------------------------


Waheela::Waheela()
{  
	// init gui
	setWindowIcon(KIcon("waheela"));
	setAcceptDrops(true);
	
	// get current theme
	KConfig config("waheela/waheela.config");
	KConfigGroup generalSettings( &config, "General" );
	theme = generalSettings.readEntry( "Theme", "Evening" );
	
	// get screen size
	QDesktopWidget* desktopWidget = QApplication::desktop();
	screensize = desktopWidget->screenGeometry();
	
	showFullScreen();
	
	initElements();
	
	setupElements();

	// init first song
	QVariantMap metadata = PlayerController::self()->getMetadata();
	updateMetadata(metadata);
	volumeSlider->setValue(PlayerController::self()->volume());
	setStatus(PlayerController::self()->getStatus().int1);
	setPosition(PlayerController::self()->getPosition(), metadata.value("mtime").toInt());


	// update slots
	connect(PlayerController::self(), SIGNAL(metadataChanged(const QVariantMap &)), this,SLOT(updateMetadata(const QVariantMap &)));
	connect(PlayerController::self(), SIGNAL(statusChanged(const int &)), this, SLOT(setStatus(const int &)));
	connect(PlayerController::self(), SIGNAL(positionChanged(const int& , const int& )), this ,SLOT(setPosition(const int&, const int&)));
	connect(volumeSlider, SIGNAL(sliderMoved(const int&)), PlayerController::self(), SLOT(setVolume(const int&)));

}


//------------------------------------------------------------------------------
//--- init elements ------------------------------------------------------------
//------------------------------------------------------------------------------


void Waheela::initElements() {
	// init Dialogs
	initThemeDialog();  
	initSettingsElements();

	// init wallpaper
	initWallpaper();
	
	// init volume slider
	volumeSlider   = new QSlider(this);
	
	// init cover
	coverLabel    = new QLabel(backgroundWidget);
	coverLabel   -> show();
	coverLabel   -> setAccessibleName("cover");
	
	// init metadata
	metadataLabel = new Metadata(backgroundWidget);
	
	// init date, time and progressBar
	initBottom();

	// init amarok
	initAmarok();

	// init timer
	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(hideButtons()));
	setMouseTracking(TRUE);
	
	// init buttons
	initPlayButtons();
	initControlButtons();
}

void Waheela::initSettingsElements() {

  	// display elements
	displaySettingsWidget = new QWidget();
	displayUi.setupUi(displaySettingsWidget);

	// amarok page
	amarokSettingsWidget = new QWidget();
	amarokUi.setupUi(amarokSettingsWidget);
	
	// password page
	passwordSettingsWidget = new QWidget();
	passwordUi.setupUi(passwordSettingsWidget);
}




void Waheela::initControlButtons() {

	controlButtonsWidget =  new QWidget(backgroundWidget);
	controlButtonsWidget -> setStyleSheet("border: 0");
	controlButtonsWidget -> setGeometry(screensize.width()-130, 10, 120, 27);
	
	KPushButton *settingsButton  = new KPushButton(KIcon("preferences-system"), "", controlButtonsWidget);
	settingsButton->setIconSize(QSize(24,24));
	settingsButton->setGeometry(0, 0, 30, 30);
	settingsButton->show();
	connect(settingsButton,SIGNAL(clicked()), this, SLOT(showSettingsDialog() ));  
	
	KPushButton *themeButton = new KPushButton(KIcon("applications-graphics"), "", controlButtonsWidget);
	themeButton->setIconSize(QSize(24,24));
	themeButton->setGeometry(28, 0, 30, 30);
	themeButton->show();
	connect(themeButton,SIGNAL(clicked()), themeDialog, SLOT(show()));  
	
	KPushButton *aboutButton = new KPushButton(KIcon("waheela"), "", controlButtonsWidget);
	aboutButton->setIconSize(QSize(24,24));
	aboutButton->setGeometry(56, 0, 30, 30);
	aboutButton->show();
	connect(aboutButton,SIGNAL(clicked()), this, SLOT(showAboutDialog()));
	
	KPushButton *exitButton = new KPushButton(KIcon("application-exit"), "", controlButtonsWidget);
	exitButton->setIconSize(QSize(24,24));
	exitButton->setGeometry(84, 0, 30, 30);
	exitButton->show();
	connect(exitButton,SIGNAL(clicked()), this, SLOT(close()));
}


void Waheela::initBottom() {
  
	bottomWidget    = new QWidget(backgroundWidget);
	bottomWidget   -> setGeometry(0, screensize.height()-90, screensize.width(), 90);
	bottomWidget   -> show();

	progressWidget  = new PlayProgress();
	progressWidget -> show();
	progressWidget -> setMinimumSize(QSize(0,30));
	
	QWidget *spacerWidget = new QWidget();
	spacerWidget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

	dateTimeWidget         = new DateTime();
	dateTimeWidget -> show();
	
	QVBoxLayout *layout = new QVBoxLayout;
	layout->setContentsMargins(0,0,0,0);
	layout->addWidget(spacerWidget);
	layout->addWidget(progressWidget);
	layout->addWidget(dateTimeWidget);
	bottomWidget->setLayout(layout);
}


void Waheela::initPlayButtons() {

 	playButtonsWidget   = new QWidget(this);
	playButtonsWidget  -> setStyleSheet("border: 0");
	playButtonsWidget  -> setGeometry(screensize.width()/2-80, 5, 160, 55);
	playButtonsWidget  -> show();

	KPushButton *prevButton     = new KPushButton(KIcon("media-skip-backward"), "",  playButtonsWidget );
	prevButton->setIconSize(QSize(48,48));
	prevButton->setGeometry(0, 0, 50, 50); 
	prevButton->show();
	connect(prevButton,SIGNAL(clicked()), this, SLOT(prevSong()));
	
	playButton = new KPushButton(KIcon("media-playback-start"), "",  playButtonsWidget);
	playButton->setIconSize(QSize(48,48));
	playButton->setGeometry(52, 0, 50, 50);
	playButton->show();
	connect(playButton,SIGNAL(clicked()), this, SLOT(playSong())); 
	
	KPushButton *nextButton     = new KPushButton(KIcon("media-skip-forward"), "",  playButtonsWidget);
	nextButton    -> setIconSize(QSize(48,48));
	nextButton    -> setGeometry(104, 0, 50, 50);
	nextButton->show();
	connect(nextButton,SIGNAL(clicked()), this, SLOT(nextSong()));
}


//------------------------------------------------------------------------------
//--- setup elements------------------------------------------------------------
//------------------------------------------------------------------------------


void Waheela::setupElements() {
 	// load waheela & theme config
	KConfig config("waheela/waheela.config");
	KConfig themeConfig("waheela/themes/"+theme+".config");


	setupWallpaper();

	setupCover( KConfigGroup( &themeConfig, "Cover" ) );

	setupMetadata( KConfigGroup(&themeConfig, "Metadata") );

	setupDateTime( KConfigGroup( &themeConfig, "DateTime" ), KConfigGroup( &config, "Format" ) );

	setupProgressBar( KConfigGroup( &themeConfig, "ProgressBar" ) );

	KConfigGroup volumeSettings( &config, "Volume" );

	setupPassword( KConfigGroup( &config, "Password" ) );
}

void Waheela::setupCover(KConfigGroup coverSettings) {
	bool coverIsVisible = coverSettings.readEntry( "Visible", TRUE );
	coverLabel->setVisible( coverIsVisible );
	// position
	QPoint coverPosition  = coverSettings.readEntry( "Position", QPoint(450,75) );
	coverLabel->setGeometry(coverPosition.x(),coverPosition.y(),1,1); 
}


void Waheela::setupMetadata(KConfigGroup settings) {
	// position
	QPoint metadataPosition  = settings.readEntry( "Position", QPoint(75,75) );
	metadataLabel->setGeometry(metadataPosition.x(),metadataPosition.y(),350,210);
	// settings
	metadataLabel->setup(settings, theme);
}


void Waheela::setupProgressBar(KConfigGroup progressBarSettings) {
	progressWidget->setup(progressBarSettings, theme);
}


void Waheela::setupDateTime(KConfigGroup dataTimeSettings, KConfigGroup formatSettings) {
	dateTimeWidget->setup(dataTimeSettings, formatSettings, theme);
}

void Waheela::setupWallpaper() {
	backgroundWidget->setGeometry(0, 0, screensize.width(), screensize.height());
	backgroundWidget->setBackground(theme);
}

//------------------------------------------------------------------------------
//--- more inits ---------------------------------------------------------------
//------------------------------------------------------------------------------


void Waheela::initAmarok() {
	KConfig config("waheela/waheela.config");
	KConfigGroup generalGroup( &config, "General" );
	KConfigGroup amarokSettings( &config, "Amarok" );
	bool useCustomAmarokBinPath  = amarokSettings.readEntry("useCustomAmarokBinPath", false);
	QString amarokBinPath        = amarokSettings.readEntry("customAmarokBinPath", "/usr/bin/amarok");

	amarokUi.customAmarokBinPath->setText(PlayerController::self()->amarokPath());
	amarokUi.customAmarokBinPath->setEnabled(PlayerController::self()->useCustomAmarokPath());
	amarokUi.useCustomAmarokBinPath->setChecked(PlayerController::self()->useCustomAmarokPath());	
	
	PlayerController::registerPlayWolfInstance();
	PlayerController::self()->setAmarokPath(useCustomAmarokBinPath, amarokBinPath);
	PlayerController::self()->updateInfos();
}

void Waheela::setupPassword(KConfigGroup passwordSettings) {
	usePassword = passwordSettings.readEntry( "UsePassword", false );
	password    = passwordSettings.readEntry( "Password", "" );

	passwordUi.usePasswordCheckBox->setChecked( usePassword );
	passwordUi.passwordLineEdit->setEnabled( usePassword );
	passwordUi.passwordLineEdit->setText( password );
}


void Waheela::initWallpaper() {
  
	// set wallpaper
	backgroundWidget = new Background(this);

  
	QStringList wallpaperMetafiles = KGlobal::dirs()->findAllResources("wallpaper", "*/*.desktop");
	
	int i = 0, j = 0;
	foreach(QString wallpaperMetafile, wallpaperMetafiles) {
		KDesktopFile desktopFile(wallpaperMetafile);
		QString wallpaperPath = wallpaperMetafile.replace("metadata.desktop", "");
		QStringList tmpList = wallpaperPath.split("/");
		QString wallpaperName = tmpList[tmpList.count()-2];
		QDir screenshotDir(wallpaperPath+"contents/");
		screenshotDir.setNameFilters(QStringList() << "*.jpeg" << "*.jpg" << "*.png");
		
		QListWidgetItem *newItem = new QListWidgetItem;
		newItem->setText(desktopFile.readName());
		newItem->setStatusTip(wallpaperName);
		if( screenshotDir.count() > 0 ) {
			newItem->setIcon(QIcon(wallpaperPath+"contents/"+screenshotDir[0]));
		}

		
		wallpaperUi.wallpaperListWidget->addItem(newItem);
		if( wallpaperName == theme) {
			 j = i;
		}
		i++;
	}
	
	wallpaperUi.wallpaperListWidget->setCurrentRow(j);
	wallpaperUi.wallpaperListWidget->sortItems();
	
}

//------------------------------------------------------------------------------
//--- settings -----------------------------------------------------------------
//------------------------------------------------------------------------------

void Waheela::showSettingsDialog()
{
	KPageDialog *settingsDialog = new KPageDialog(this);
	connect(settingsDialog, SIGNAL(okClicked()), this, SLOT(applySettings()));
	settingsDialog->setButtons( KDialog::Ok | KDialog::Cancel);
	
	KPageWidgetItem *page1 = new KPageWidgetItem( displaySettingsWidget, i18n( "Diplay" ) );
	page1->setIcon( KIcon( "preferences-desktop-display" ) );
	settingsDialog->addPage( page1 );
	displayUi.metadataIsVisibleCheckBox->setChecked( metadataLabel->isVisible() );
	displayUi.dateTimeIsVisibleCheckBox->setChecked( dateTimeWidget->isVisible() );
	displayUi.coverIsVisibleCheckBox->setChecked( coverLabel->isVisible() );
	
	KPageWidgetItem *page3 = new KPageWidgetItem( amarokSettingsWidget, i18n( "Amarok" ) );
	page3->setIcon( KIcon( "amarok" ) );
	settingsDialog->addPage( page3 );
	
	KPageWidgetItem *page4 = new KPageWidgetItem( passwordSettingsWidget, i18n( "Password" ) );
	page4->setIcon( KIcon( "document-encrypt" ) );
	settingsDialog->addPage( page4 );

	settingsDialog->exec();
}

void Waheela::applySettings()
{

	KConfig config("waheela/waheela.config");
	KConfig themeConfig("waheela/themes/"+theme+".config");
	
	// amarok page
	KConfigGroup amarokSettings( &config, "Amarok" );
	bool useCustom = amarokUi.useCustomAmarokBinPath->isChecked();
	amarokSettings.writeEntry("useCustomAmarokBinPath", useCustom);
	QString amarokBinPath = amarokUi.customAmarokBinPath->text();
	amarokSettings.writeEntry("customAmarokBinPath", amarokBinPath);

	// date & time page
	if( displayUi.metadataIsVisibleCheckBox->isChecked() ) {
		dateTimeWidget->show2();
	}
	else {
		dateTimeWidget->hide2();
	}

	// cover
	KConfigGroup coverSettings( &themeConfig, "Cover" );
	coverSettings.writeEntry( "Visible", displayUi.coverIsVisibleCheckBox->isChecked() );
	coverLabel->setHidden( !displayUi.coverIsVisibleCheckBox->isChecked() );

	// metadata
	if( displayUi.metadataIsVisibleCheckBox->isChecked() ) {
		metadataLabel->show2();
	}
	else {
		metadataLabel->hide2();
	}
	
	// password page
	KConfigGroup passwordSettings( &config, "Password" );
	password = passwordUi.passwordLineEdit->text();
	passwordSettings.writeEntry( "Password", password );
	usePassword = passwordUi.usePasswordCheckBox->isChecked();
	passwordSettings.writeEntry( "UsePassword", usePassword );

	updateMetadata(PlayerController::self()->getMetadata()); 
}


void Waheela::initThemeDialog()
{  
	themeDialog = new KPageDialog(this);
	connect(themeDialog, SIGNAL(okClicked()), this, SLOT(applyTheme()));
	themeDialog->setButtons( KDialog::Ok | KDialog::Cancel);
	QWidget *wallpapperWidget = new QWidget();
	wallpaperUi.setupUi(wallpapperWidget);
	themeDialog->addPage(wallpapperWidget, i18n("Theme"));
}


void Waheela::applyTheme()
{
	KConfig config("waheela/waheela.config");
	KConfigGroup generalSettings( &config, "General" );
	theme = wallpaperUi.wallpaperListWidget->currentItem()->statusTip();
	generalSettings.writeEntry( "Theme", theme );
	setupElements();
}


//------------------------------------------------------------------------------
//--- button mode --------------------------------------------------------------
//------------------------------------------------------------------------------

void Waheela::playSong() {
	PlayerController::self()->playPause();
}

void Waheela::stopSong() {
	PlayerController::self()->stop();
}

void Waheela::prevSong() {
	PlayerController::self()->prev();
}

void Waheela::nextSong() {
	PlayerController::self()->next();
}

void Waheela::setPosition(const int pos, const int total) {
	progressWidget->setValue(pos);
	progressWidget->setMaximum(total);
}

void Waheela::setStatus(const int &status) {
	if (status == 1)
		playButton->setIcon(KIcon("media-playback-start"));
	else if (status == 0)
		playButton->setIcon(KIcon("media-playback-pause"));
	else
		playButton->setIcon(KIcon("media-playback-start"));
}

void Waheela::hideButtons() {
	playButtonsWidget->hide();
	controlButtonsWidget->hide();
	volumeSlider->hide();
	dateTimeWidget->hideControlElements();
	progressWidget->hideButtons();
	metadataLabel->hideButtons();
	pwd = "";
}

void Waheela::updateMetadata(const QVariantMap& metadata) {
	metadataLabel->setMetadata(metadata);
	
	int i = metadata.value("location").toString().indexOf(":");
	QString mode = metadata.value("location").toString().left(i);

	if( !coverLabel->isHidden() ) {
		QString cover;
		if (metadata.value("arturl").toString() != QString()) {
			cover = metadata.value("arturl").toString();
			cover.replace("file://", "");
		}
		else if (mode == QString("lastfm")) {
			cover = "/usr/share/kde4/apps/amarok/images/lastfm-default-cover.png";
		}
		else {
			cover = "/usr/share/kde4/apps/amarok/images/nocover.png";
		}

		QPixmap coverPixmap = QPixmap(cover);
		if( ! coverPixmap.isNull() ) {
			coverPixmap = coverPixmap.scaledToWidth(200, Qt::SmoothTransformation);
		}
		coverLabel->setPixmap(coverPixmap);
		coverLabel->setFixedWidth(coverPixmap.width());
		coverLabel->setFixedHeight(coverPixmap.height());
	}
 	else {
		coverLabel->setPixmap(QPixmap());
	}
}

//------------------------------------------------------------------------------
//--- mouse events -------------------------------------------------------------
//------------------------------------------------------------------------------

void Waheela::wheelEvent(QWheelEvent *event) {
	if( usePassword and pwd != password ) {
		KPasswordDialog *dlg = new KPasswordDialog( this );
		dlg->setPrompt(i18n("Enter a password"));
		connect( dlg, SIGNAL( gotPassword( const QString& , bool ) )  , this, SLOT( setPassword( const QString &) ) );
		dlg->show();
		return;
	}

	if (event->delta() > 0)
		PlayerController::self()->volumeUp();
	else
		PlayerController::self()->volumeDown();
	volumeSlider->setValue(PlayerController::self()->volume());

	timer->start(10000);
}

void Waheela::mousePressEvent(QMouseEvent *event) {

	if( usePassword and pwd != password ) {
		KPasswordDialog *dlg = new KPasswordDialog( this );
		dlg->setPrompt(i18n("Enter a password"));
		connect( dlg, SIGNAL( gotPassword( const QString& , bool ) )  , this, SLOT( setPassword( const QString &) ) );
		dlg->show();
		return;
	}
	timer->start(10000);

	int cursorX = event->pos().x();
	int cursorY = event->pos().y();
	QRect bar = progressWidget->geometry(); 
	
	if ( cursorY > bar.y() and cursorY < bar.y()+bar.height() and cursorX > bar.x() and cursorX < bar.x()+bar.width()) {
		int value = (cursorX-bar.x())*progressWidget->maximum()/bar.width()+1;
		progressWidget->setValue(value); 
		PlayerController::self()->seek(value);
	}
	else {
		playButtonsWidget->show();
		controlButtonsWidget->show();
		volumeSlider->show();
		progressWidget->showButtons();
		metadataLabel->showButtons();
		dateTimeWidget->showControlElements();
	}

	// drag&drop 
	QLabel *child = static_cast<QLabel*>(childAt(event->pos()));

	if( !child ) {
		return;
	}
	if( child->accessibleName() != "metadata" and child->accessibleName() != "cover"  ) {
		return;
	}

	QPoint hotSpot = event->pos() - child->pos();

	QMimeData *mimeData = new QMimeData;
	mimeData->setText(child->accessibleName());
	mimeData->setData("application/x-hotspot",  QByteArray::number(hotSpot.x())+ " " + QByteArray::number(hotSpot.y()));

	QPixmap pixmap(child->size());
	child->render(&pixmap);
	
	QDrag *drag = new QDrag(this);
	drag->setMimeData(mimeData);
	drag->setPixmap(pixmap);
	drag->setHotSpot(hotSpot);

	Qt::DropAction dropAction = drag->exec(Qt::CopyAction | Qt::MoveAction, Qt::CopyAction);

	if (dropAction == Qt::MoveAction) {
		child->close();
	}
}

void Waheela::closeEvent(QCloseEvent *event)
{
	if( usePassword and pwd != password ) {
		event->ignore();
	}
}
  
void Waheela::setPassword(QString input) {
	pwd = input;
}


//------------------------------------------------------------------------------
//--- about --------------------------------------------------------------------
//------------------------------------------------------------------------------

void Waheela::showAboutDialog() {
	AboutDialog *aboutWidget = new AboutDialog();
	aboutWidget->exec();
}

//------------------------------------------------------------------------------
//--- drag&drop ----------------------------------------------------------------
//------------------------------------------------------------------------------

void Waheela::dragEnterEvent(QDragEnterEvent *event) {
	if (event->mimeData()->hasText()) {
		if (children().contains(event->source())) {
			event->setDropAction(Qt::MoveAction);
			event->accept();
		} else {
			event->acceptProposedAction();
		}
	} else {
		event->ignore();
	}
}

void Waheela::dropEvent(QDropEvent *event) {
  
	if (event->mimeData()->hasText()) {
	
		KConfig themeConfig("waheela/themes/"+theme+".config");
	
		const QMimeData *mime = event->mimeData();
		QStringList pieces = mime->text().split(QRegExp("\\s+"),
		QString::SkipEmptyParts);
		QPoint position = event->pos();
		QPoint hotSpot;

		QList<QByteArray> hotSpotPos = mime->data("application/x-hotspot").split(' ');
		if (hotSpotPos.size() == 2) {
			hotSpot.setX(hotSpotPos.first().toInt());
			hotSpot.setY(hotSpotPos.last().toInt());
		}

		if( mime->text() == "cover" ) {
			coverLabel->move(position - hotSpot);
			KConfigGroup coverSettings( &themeConfig, "Cover" );
			coverSettings.writeEntry( "Position", position - hotSpot );
		}
		else if( mime->text() == "metadata" ){
			metadataLabel->move(position - hotSpot);
			KConfigGroup metadataSettings( &themeConfig, "Metadata" );
			metadataSettings.writeEntry( "Position", position - hotSpot );
		}
		else if( mime->text() == "buttons" ){
			playButtonsWidget->move(position - hotSpot);
		}
	} 
	else {
		event->ignore();
	}
}
