/***************************************************************************
@author:    Fabian Wuertz <xadras@sidux.com>
@copyright: Fabian Wuertz <xadras@sidux.com>
@license:   GPL2, GPL3

@project:   Waheela
@file:      metadata.cpp    
***************************************************************************/

#include "metadata.h"

#include <KColorDialog>
#include <klocale.h>
#include <KPageDialog>


Metadata::Metadata (QWidget* parent, Qt::WFlags flags): QLabel (parent, flags)
{  
	setAccessibleName("metadata");
	
	settingsWidget = new QWidget();
	metadataUi.setupUi(settingsWidget);
	show();
	
	settingsButton  = new KPushButton(KIcon("applications-graphics"), "", this);
	settingsButton -> setIconSize(QSize(18,18));
	settingsButton -> setGeometry(6, 3, 18, 18);
	connect(settingsButton, SIGNAL(clicked()), this, SLOT(showSettingsDialog()));
	hideButtons();
}

//------------------------------------------------------------------------------
//--- metadata text ------------------------------------------------------------
//------------------------------------------------------------------------------

void Metadata::setMetadata(const QVariantMap& metadata) {
	title  = metadata.value("title").toString();
	artist = i18n("by")+" "+metadata.value("artist").toString();
	album  = i18n("on")+" "+metadata.value("album").toString();
	showMetadata();
}

void Metadata::showMetadata() {
	QString text;
	bool titleIsVisible  = metadataUi.titleCheckBox->isChecked();
	bool artistIsVisible = metadataUi.artistCheckBox->isChecked();
	bool albumIsVisible  = metadataUi.albumCheckBox->isChecked();
  	if( titleIsVisible ) {
		text = title + "\n\n";
	}
	if( artistIsVisible ) {
		text += artist+"\n\n";
	}
	if( albumIsVisible ) {
		text += album;
	}
	setText(text);
	if(titleIsVisible or artistIsVisible or albumIsVisible) {
		show();
	}
	else{
		hide();
	}
}

//------------------------------------------------------------------------------
//--- settings -----------------------------------------------------------------
//------------------------------------------------------------------------------

void Metadata::setup(KConfigGroup settings, QString currentTheme) {
	theme = currentTheme;
  
	// visible
	bool titleIsVisible = settings.readEntry( "TitleIsVisible", TRUE );
	metadataUi.titleCheckBox->setChecked( titleIsVisible );
	bool artistIsVisible = settings.readEntry( "ArtistIsVisible", TRUE );
	metadataUi.artistCheckBox->setChecked( artistIsVisible );
	bool albumIsVisible = settings.readEntry( "AlbumIsVisible", TRUE );
	metadataUi.albumCheckBox->setChecked( albumIsVisible );
	showMetadata();
	
	// font
	setFont(  settings.readEntry( "Font", QFont("Sans Serif", 18, QFont::Normal) ) );

	// font color
	QColor fontColor = settings.readEntry( "FontColor", QColor( 0, 0, 255, 127) );
	QPalette q = palette();
	q.setBrush(QPalette::WindowText, QColor(fontColor) );
	setPalette(q);
	

	// frame shape & shadow
	setFrameStyle( settings.readEntry( "Style", 0 ) );

	// background
	QString backgroundType = settings.readEntry( "BackgroundType", "None" );
	if( backgroundType == "None" ) {
		setAutoFillBackground(FALSE);
	}
	else {
		QColor backgroundColor = settings.readEntry( "BackgroundColor", QColor(0,0,255,125) );
		QPalette p = palette();
		QColor backgroundColor2 = QColor(backgroundColor);
		backgroundColor2.setAlpha(127);
		p.setBrush(QPalette::Background, backgroundColor2 );
		setPalette(p);
		setAutoFillBackground(TRUE);
	}

}


void Metadata::showSettingsDialog()
{
	KPageDialog *settingsDialog = new KPageDialog(this);
	connect(settingsDialog, SIGNAL(okClicked()), this, SLOT(applySettings()));
	settingsDialog->setButtons( KDialog::Ok | KDialog::Cancel);
	settingsDialog->addPage(settingsWidget, i18n("Metadata"));
	
	metadataUi.fontChooser->setFont(font());
	metadataUi.fontKcolorbutton->setColor(palette().color(QPalette::WindowText));
	
	metadataUi.backgroundKcolorbutton->setColor(palette().color(QPalette::Background));
	
	if( frameShape() == QFrame::NoFrame ) {
		metadataUi.shapeComboBox->setCurrentIndex(0);
	}
	else if( frameShape() == QFrame::Box ) {
		metadataUi.shapeComboBox->setCurrentIndex(1);
	}
	else if( frameShape() == QFrame::Panel  ) {
		metadataUi.shapeComboBox->setCurrentIndex(2);
	}
	else if( frameShape() == QFrame::WinPanel ) {
		metadataUi.shapeComboBox->setCurrentIndex(3);
	}
	else if( frameShape() == QFrame::StyledPanel ) {
		metadataUi.shapeComboBox->setCurrentIndex(4);
	}
	
	if( frameShadow() == QFrame::Plain ) {
		metadataUi.shadowComboBox->setCurrentIndex(0);
	}
	else if( frameShadow() == QFrame::Raised ) {
		metadataUi.shadowComboBox->setCurrentIndex(1);
	}
	else if( frameShadow() == QFrame::Sunken ) {
		metadataUi.shadowComboBox->setCurrentIndex(2);
	}
	
	if(autoFillBackground()) {
		metadataUi.backgroundComboBox->setCurrentIndex(1);
	}
	else {
		metadataUi.backgroundComboBox->setCurrentIndex(0);
	}
	
	settingsDialog->exec();
}


void Metadata::applySettings()
{
	
	KConfig themeConfig("waheela/themes/"+theme+".config");
	KConfigGroup settings( &themeConfig, "Metadata" );
	
	// visible
	settings.writeEntry( "TitleIsVisible", metadataUi.titleCheckBox->isChecked() );
	settings.writeEntry( "ArtistIsVisible", metadataUi.artistCheckBox->isChecked() );
	settings.writeEntry( "AlbumIsVisible", metadataUi.albumCheckBox->isChecked() );
	showMetadata();
	
	// font
	QFont font = metadataUi.fontChooser->font();
	settings.writeEntry( "Font", font );

	// font color
	QColor fontColor = metadataUi.fontKcolorbutton->color();
	settings.writeEntry( "FontColor", fontColor );
	
	// frame shape and shadow
	QFrame::Shape shape = QFrame::NoFrame;
	QString shapeType = metadataUi.shapeComboBox->currentText();
	if( shapeType == i18n("Box") ) {
		shape = QFrame::Box;
	}
	else if( shapeType == i18n("Panel") ) {
		shape = QFrame::Panel;
	}
	else if( shapeType == i18n("StyledPanel") ) {
		shape = QFrame::StyledPanel;
	}
	else if( shapeType == i18n("WinPanel") ) {
		shape = QFrame::WinPanel;
	}
	QFrame::Shadow shadow = QFrame::Plain;
	QString shadowType = metadataUi.shadowComboBox->currentText();
	if( shadowType == i18n("Raised") ) {
		shadow = QFrame::Raised;
	}
	else if( shadowType == i18n("Sunken") ) {
		shadow = QFrame::Sunken;
	}
	settings.writeEntry( "Style", (shape | shadow) );
	
	// background
	QString backgroundType = metadataUi.backgroundComboBox->currentText();
	if( backgroundType == i18n("None") ) {
		settings.writeEntry( "BackgroundType", "None" );
	}
	else {
		settings.writeEntry( "BackgroundType", "Color" );
		QColor backgroundColor = metadataUi.backgroundKcolorbutton->color();
		settings.writeEntry( "BackgroundColor", backgroundColor );
	}
	
	setup(settings, theme);
}

//------------------------------------------------------------------------------
//--- control elements ---------------------------------------------------------
//------------------------------------------------------------------------------

void Metadata::hideButtons() {
	settingsButton->hide(); 
}

void Metadata::showButtons() {
	settingsButton->show();  
}

//------------------------------------------------------------------------------
//--- show/hide ----------------------------------------------------------------
//------------------------------------------------------------------------------

void Metadata::show2() {
  	show();
	if( metadataUi.titleCheckBox->isChecked() or metadataUi.artistCheckBox->isChecked() or metadataUi.albumCheckBox->isChecked() ) {
		return;
	}
	KConfig themeConfig("waheela/themes/"+theme+".config");
	KConfigGroup settings( &themeConfig, "Metadata" );
	metadataUi.titleCheckBox->setChecked(TRUE);
	settings.writeEntry( "TitleIsVisible", TRUE );
	metadataUi.artistCheckBox->setChecked(TRUE);
	settings.writeEntry( "ArtistIsVisible", TRUE );
	metadataUi.albumCheckBox->setChecked(TRUE);
	settings.writeEntry( "AlbumIsVisible", TRUE );

	showMetadata();
}

void Metadata::hide2() {
	hide();
	KConfig themeConfig("waheela/themes/"+theme+".config");
	KConfigGroup settings( &themeConfig, "Metadata" );
	settings.writeEntry( "TitleIsVisible", FALSE );
	settings.writeEntry( "ArtistIsVisible", FALSE );
	settings.writeEntry( "AlbumIsVisible", FALSE );

	showMetadata();
}