#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>

#include "msr-index.h"

#include "xenner.h"
#include "processor.h"
#include "cpufeature.h"

/* ------------------------------------------------------------------ */

static void real_cpuid(struct kvm_cpuid_entry *entry)
{
    asm volatile("cpuid"
		 : "=a" (entry->eax),
		   "=b" (entry->ebx),
		   "=c" (entry->ecx),
		   "=d" (entry->edx)
		 : "a" (entry->function));
}

static void filter_cpuid(struct xenvm *xen, struct kvm_cpuid_entry *entry)
{
    switch (entry->function) {
    case 0x00000000:
	entry->eax = 0x02; /* max. valid entry */
	break;

    case 0x00000001:
	/* pv */
	entry->edx &= ~(1 << (X86_FEATURE_VME     %32));
	entry->edx &= ~(1 << (X86_FEATURE_DE      %32));
	entry->edx &= ~(1 << (X86_FEATURE_PSE     %32));

	entry->edx &= ~(1 << (X86_FEATURE_SEP     %32));
	entry->edx &= ~(1 << (X86_FEATURE_MTRR    %32));
	entry->edx &= ~(1 << (X86_FEATURE_PGE     %32));

	/* hvm */
	entry->ecx &= ~(1 << (X86_FEATURE_MWAIT   %32));
	entry->edx &= ~(1 << (X86_FEATURE_PSE36   %32));

	/* # of cpus */
	entry->ebx &= 0xffff;
	entry->ebx |= xen->vcpus << 16;
	break;

    case 0x80000000:
	entry->eax = 0x80000002; /* max. valid entry */
	break;

    case 0x80000001:
	/* pv */
	entry->edx &= ~(1 << (X86_FEATURE_RDTSCP  %32));

	/* hvm */
	entry->ecx &= ~(1 << (X86_FEATURE_LAHF_LM %32));
	if (xen->mode == XENMODE_64)
	    break;
	entry->edx &= ~(1 << (X86_FEATURE_LM      %32));
	entry->edx &= ~(1 << (X86_FEATURE_SYSCALL %32));
	break;
    }
}

/* FIXME: resync kvm headers after clocksource merge, then drop these lines */
#define KVM_CPUID_SIGNATURE     0x40000000
#define KVM_CPUID_FEATURES      0x40000001

void setup_cpuid(struct xenvm *xen)
{
    struct kvm_cpuid_entry list[] = {
	{ .function = 0x00000000 },
	{ .function = 0x00000001 },
	{ .function = 0x00000002 },

	{ .function = KVM_CPUID_SIGNATURE   },
	{ .function = KVM_CPUID_FEATURES    },

	{ .function = 0x80000000 },
	{ .function = 0x80000001 },
	{ .function = 0x80000002 },
    };
    uint32_t sig[3];
    int count = sizeof(list)/sizeof(list[0]);
    int i;

    for (i = 0; i < count; i++) {
	switch (list[i].function) {
	case KVM_CPUID_SIGNATURE:
	    memcpy(sig, "KVMKVMKVM", 12);
	    list[i].eax = KVM_CPUID_FEATURES;  /* max. valid entry */
	    list[i].ebx = sig[0];
	    list[i].ecx = sig[1];
	    list[i].edx = sig[2];
	    break;
	case KVM_CPUID_FEATURES:
	    list[i].eax = xen->features;
	    break;
	default:
	    /* copy from host */
	    real_cpuid(list+i);
	    filter_cpuid(xen, list+i);
	    break;
	}
	d2printf("%s: %08x: %08x %08x %08x %08x\n", __FUNCTION__,
		 list[i].function, list[i].eax, list[i].ebx, list[i].ecx, list[i].edx);
    }

    for (i = 0; i < xen->vcpus; i++)
	kvm_setup_cpuid(xen->kvm, i, count, list);
}

/* ------------------------------------------------------------------ */

static const char *msr_low[] = {
    [ MSR_IA32_P5_MC_ADDR              ] = "IA32_P5_MC_ADDR",
    [ MSR_IA32_TSC                     ] = "IA32_TSC",
    [ MSR_IA32_SYSENTER_CS             ] = "IA32_SYSENTER_CS",
    [ MSR_IA32_SYSENTER_ESP            ] = "IA32_SYSENTER_ESP",
    [ MSR_IA32_SYSENTER_EIP            ] = "IA32_SYSENTER_EIP",
};

static const char *msr_high[] = {
    [ MSR_EFER             - 0xc0000000 ] = "EFER",
    [ MSR_STAR             - 0xc0000000 ] = "STAR",
    [ MSR_LSTAR            - 0xc0000000 ] = "LSTAR",
    [ MSR_CSTAR            - 0xc0000000 ] = "CSTAR",
    [ MSR_SYSCALL_MASK     - 0xc0000000 ] = "SYSCALL_MASK",
    [ MSR_FS_BASE          - 0xc0000000 ] = "FS_BASE",
    [ MSR_GS_BASE          - 0xc0000000 ] = "GS_BASE",
    [ MSR_KERNEL_GS_BASE   - 0xc0000000 ] = "KERNEL_GS_BASE",
};

static const char *msr_name(uint32_t index)
{
    if (index > 0xc0000000)
	return GETNAME(msr_high, index - 0xc0000000);
    return GETNAME(msr_low, index);
}

void set_msr(struct xenvcpu *vcpu, uint32_t index, uint64_t data)
{
    struct xenvm *xen = vcpu->vm;
    struct kvm_msr_entry msr = {
	.index    = index,
	.reserved = 0,
	.data     = data,
    };

    d1printf("%s: %" PRIx32 " (%s) = %" PRIx64 "\n",
	     __FUNCTION__, index, msr_name(index), data);
    kvm_set_msrs(xen->kvm, vcpu->id, &msr, 1);
}

uint64_t get_msr(struct xenvcpu *vcpu, uint32_t index)
{
    struct xenvm *xen = vcpu->vm;
    struct kvm_msr_entry msr = {
	.index    = index,
	.reserved = 0,
    };

    kvm_get_msrs(xen->kvm, vcpu->id, &msr, 1);
    if (MSR_IA32_TSC != index)
        d3printf("%s: %" PRIx32 " (%s) = %llx\n",
                 __FUNCTION__, index, msr_name(index), msr.data);
    return msr.data;
}

void print_msrs(struct xenvcpu *vcpu)
{
    struct xenvm *xen = vcpu->vm;
    struct kvm_msr_list *msrs;
    struct kvm_msr_entry entry;
    int i;

    logprintf(xen, "--- msrs (nonzero) ---\n");
    msrs = kvm_get_msr_list(xen->kvm);
    for (i = 0; i < msrs->nmsrs; i++) {
	entry.index = msrs->indices[i];
	kvm_get_msrs(xen->kvm, vcpu->id, &entry, 1);
	if (0 == entry.data)
	    continue;
	logprintf(xen, "msr %08x = %016llx (%s)\n",
		entry.index, entry.data, msr_name(entry.index));
    }
}
