#include "msr-index.h"
#include "emu.h"
#include "emu-main.c"

/* --------------------------------------------------------------------- */
/* helper functions                                                      */

int bounce_trap(struct xen_cpu *cpu, struct regs_64 *regs, int trapno, int cbno)
{
    uint64_t *stack, rip = 0, rsp, stack_cs, stack_rflags;
    int error_code = 0;
    int interrupt = 0;
    int failsafe = 0;
    int k = 0;

    vminfo.faults[XEN_FAULT_BOUNCE_TRAP]++;

    if (trapno >= 0) {
	/* trap bounce */
	rip  = xentr[trapno].address;
	if (TI_GET_IF(&xentr[trapno]))
	    interrupt = 1;
	if (trapno < sizeof(trapinfo)/sizeof(trapinfo[0]))
	    error_code = trapinfo[trapno].ec;
	if (14 == trapno) /* page fault */
	    cpu->v.vcpu_info->arch.cr2 = read_cr2();
    }
    if (cbno >= 0) {
	/* callback */
	rip  = xencb[cbno];
	switch (cbno) {
	case CALLBACKTYPE_event:
 	    interrupt = 1;
	    break;
	}
    }

    if (!rip) {
	printk(0, "%s: cbno %d, trapno %d\n", __FUNCTION__, cbno, trapno);
	panic("no guest trap handler", regs);
    }

    /* set interrupt flag depending on event channel mask */
    stack_rflags = regs->rflags & ~X86_EFLAGS_IF;
    if (guest_irq_flag(cpu))
	stack_rflags |= X86_EFLAGS_IF;

    /* old evtchn_upcall_mask is saved in cs slot on the stack */
    stack_cs = regs->cs | ((uint64_t)cpu->v.vcpu_info->evtchn_upcall_mask << 32);
    if (interrupt)
	guest_cli(cpu);

    if (!is_kernel(cpu)) {
	/* user mode */
	switch_mode(cpu);
	rsp = cpu->kernel_sp;
    } else {
	/* kernel mode */
	stack_cs &= ~3;         /* signal kernel mode */
	rsp = regs->rsp & ~0xf; /* align stack */
    }
    stack = (void*)(rsp);

#if 0
    if ((stack_cs & 3)  ||  cbno == CALLBACKTYPE_syscall) /* from user mode */
	printk("%s: code %" PRIx64 " stack %" PRIx64 "\n",
	       __FUNCTION__, rip, rsp);
#endif

    stack[-(++k)] = regs->ss;        // push ss
    stack[-(++k)] = regs->rsp;       // push rsp
    stack[-(++k)] = stack_rflags;    // push rflags
    stack[-(++k)] = stack_cs;        // push cs
    stack[-(++k)] = regs->rip;       // push rip
    if (error_code)
	stack[-(++k)] = regs->error; // push error code

    if (failsafe) {
	/* push segment registers */;
    }

    stack[-(++k)] = regs->r11;       // push r11
    stack[-(++k)] = regs->rcx;       // push rcx

    /* prepare emu stack, so iret jumps to the kernels handler. */
    regs->rip     = rip;
    regs->cs      = FLAT_KERNEL_CS;
    regs->rflags &= EFLAGS_TRAPMASK;
    regs->rsp     = rsp - 8*k;
    regs->ss      = FLAT_KERNEL_SS;

#if 0
    if (cbno == CALLBACKTYPE_syscall)
	regs->rflags |= X86_EFLAGS_TF;
#endif

    return 0;
}

/* --------------------------------------------------------------------- */

static const struct kvm_segment xen64_cs0_64 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe008,
    .dpl      = 0,
    .type     = 0xb,
    .present  = 1,  .l = 1,  .s = 1,  .g = 1,
};
static const struct kvm_segment xen64_ds0_32 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe010,
    .dpl      = 0,
    .type     = 0x3,
    .present  = 1,  .db = 1,  .s = 1,  .g = 1,
};
static const struct kvm_segment xen64_cs3_32 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe023,
    .dpl      = 3,
    .type     = 0xb,
    .present  = 1,  .db = 1,  .s = 1,  .g = 1,
};
static const struct kvm_segment xen64_ds3_32 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe02b,
    .dpl      = 3,
    .type     = 0x3,
    .present  = 1,  .db = 1,  .s = 1,  .g = 1,
};
static const struct kvm_segment xen64_cs3_64 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe033,
    .dpl      = 3,
    .type     = 0xb,
    .present  = 1,  .l = 1,  .s = 1,  .g = 1,
};
static const struct kvm_segment xen64_cs0_32 = {
    .base     = 0,
    .limit    = 0xffffffff,
    .selector = 0xe038,
    .dpl      = 0,
    .type     = 0xb,
    .present  = 1,  .db = 1,  .s = 1,  .g = 1,
};

void gdt_init(struct xen_cpu *cpu)
{
    printk(2, "%s: cpu %d\n", __FUNCTION__, cpu->id);

    if (!cpu->gdt)
	cpu->gdt = get_pages(16, "gdt");

    gdt_set(cpu->gdt, &xen64_cs0_64);
    gdt_set(cpu->gdt, &xen64_ds0_32);
    gdt_set(cpu->gdt, &xen64_cs3_32);
    gdt_set(cpu->gdt, &xen64_ds3_32);
    gdt_set(cpu->gdt, &xen64_cs3_64);
    gdt_set(cpu->gdt, &xen64_cs0_32);
}

void tss_init(struct xen_cpu *cpu)
{
    struct descriptor_32 *gdt = cpu->gdt;
    int size, idx = tss(cpu);
    uint64_t base;

    printk(2, "%s: cpu %d\n", __FUNCTION__, cpu->id);

    cpu->tss.rsp0   = (uintptr_t)cpu->stack_high;
    cpu->tss.ist[0] = (uintptr_t)cpu->irqstack_high;

    base = (uintptr_t)(&cpu->tss);
    size = sizeof(cpu->tss)-1;
    gdt[ idx +0 ] = mkdesc32(base & 0xffffffff, size, 0x89, 0);
    gdt[ idx +1 ].a = base >> 32;
    gdt[ idx +1 ].b = 0;
}

void msrs_init(struct xen_cpu *cpu)
{
    printk(2, "%s: cpu %d\n", __FUNCTION__, cpu->id);

    /* syscall setup */
    wrmsrl(MSR_STAR, (((uint64_t)0xe023 << 48) |
		      ((uint64_t)0xe008 << 32)));
    wrmsrl(MSR_LSTAR, (uintptr_t)STACK_PTR(cpu, trampoline_start));
    wrmsrl(MSR_SYSCALL_MASK, X86_EFLAGS_VM | X86_EFLAGS_RF |
	   X86_EFLAGS_NT | X86_EFLAGS_DF | X86_EFLAGS_IF | X86_EFLAGS_TF);
}

void idt_init(void)
{
    intptr_t entry;
    int i, len;

    printk(2, "%s\n", __FUNCTION__);

    len = (irq_common - irq_entries) / 256;
    for (i = 0; i < 256; i++) {
	entry = (intptr_t)(irq_entries + i*len);
	xen_idt[i]  = mkgate64(0xe008, (uintptr_t)irq_entries + i*len, 0x8e, 1);
    }

    xen_idt[    0 ] = mkgate64(0xe008, (uintptr_t)division_by_zero,    0x8e, 0);
    xen_idt[    1 ] = mkgate64(0xe008, (uintptr_t)debug_int1,          0x8e, 0);
    xen_idt[    2 ] = mkgate64(0xe008, (uintptr_t)nmi,                 0x8e, 0);
    xen_idt[    3 ] = mkgate64(0xe008, (uintptr_t)debug_int3,          0xee, 0);
    xen_idt[    4 ] = mkgate64(0xe008, (uintptr_t)overflow,            0xee, 0);
    xen_idt[    5 ] = mkgate64(0xe008, (uintptr_t)bound_check,         0x8e, 0);
    xen_idt[    6 ] = mkgate64(0xe008, (uintptr_t)illegal_instruction, 0x8e, 0);
    xen_idt[    7 ] = mkgate64(0xe008, (uintptr_t)no_device,           0x8e, 0);
    xen_idt[    8 ] = mkgate64(0xe008, (uintptr_t)double_fault,        0x8e, 0);
    xen_idt[    9 ] = mkgate64(0xe008, (uintptr_t)coprocessor,         0x8e, 0);
    xen_idt[   10 ] = mkgate64(0xe008, (uintptr_t)invalid_tss,         0x8e, 0);
    xen_idt[   11 ] = mkgate64(0xe008, (uintptr_t)segment_not_present, 0x8e, 0);
    xen_idt[   12 ] = mkgate64(0xe008, (uintptr_t)stack_fault,         0x8e, 0);
    xen_idt[   13 ] = mkgate64(0xe008, (uintptr_t)general_protection,  0x8e, 0);
    xen_idt[   14 ] = mkgate64(0xe008, (uintptr_t)page_fault,          0x8e, 0);
    xen_idt[   16 ] = mkgate64(0xe008, (uintptr_t)floating_point,      0x8e, 0);
    xen_idt[   17 ] = mkgate64(0xe008, (uintptr_t)alignment,           0x8e, 0);
    xen_idt[   18 ] = mkgate64(0xe008, (uintptr_t)machine_check,       0x8e, 0);
    xen_idt[   19 ] = mkgate64(0xe008, (uintptr_t)simd_floating_point, 0x8e, 0);

    xen_idt[ VECTOR_FLUSH_TLB  ] =
	mkgate64(0xe008, (uintptr_t)smp_flush_tlb, 0x8e, 1);

    xen_idt[ 0x80 ] = mkgate64(0xe008, (uintptr_t)int_80,              0xee, 0);
}

void guest_regs_init(struct xen_cpu *cpu, struct regs_64 *regs)
{
    struct vcpu_guest_context *ctxt = cpu->init_ctxt;

    cpu->kernel_ss = ctxt->kernel_ss;
    cpu->kernel_sp = ctxt->kernel_sp;

    regs->rax    = ctxt->user_regs.rax;
    regs->rbx    = ctxt->user_regs.rbx;
    regs->rcx    = ctxt->user_regs.rcx;
    regs->rdx    = ctxt->user_regs.rdx;
    regs->rsi    = ctxt->user_regs.rsi;
    regs->rdi    = ctxt->user_regs.rdi;
    regs->rbp    = ctxt->user_regs.rbp;
    regs->rip    = ctxt->user_regs.rip;
    regs->cs     = ctxt->user_regs.cs;
    regs->rflags = ctxt->user_regs.rflags;
    regs->rsp    = ctxt->user_regs.rsp;
    regs->ss     = ctxt->user_regs.ss;

    asm volatile("mov %0, %%ds;\n" :: "r" (ctxt->user_regs.ds) : "memory");
    asm volatile("mov %0, %%es;\n" :: "r" (ctxt->user_regs.es) : "memory");
    asm volatile("mov %0, %%fs;\n" :: "r" (ctxt->user_regs.fs) : "memory");
    asm volatile("mov %0, %%gs;\n" :: "r" (ctxt->user_regs.gs) : "memory");
}

/* --------------------------------------------------------------------- */
/* called from assembler                                                 */

asmlinkage void do_page_fault(struct regs_64 *regs)
{
    struct xen_cpu *cpu = get_cpu();
    uint64_t cr2 = read_cr2();

    vminfo.faults[XEN_FAULT_PAGE_FAULT]++;

    if (context_is_emu(regs)) {
	if (fixup_extable(regs))
	    return;
	print_page_fault_info(0, cpu, regs, cr2);
        pgtable_walk(0, cr2, read_cr3_mfn(cpu));
	panic("ring0 (emu) page fault", regs);
    }

    /* fixup error code for kernel faults */
    if (context_is_kernel(cpu, regs))
        regs->error &= ~0x04;

    if (wrpt && regs->error == 3) {
	/* kernel write to r/o page */
        if (!cpu->user_cr3_mfn || !pgtable_is_present(cr2, cpu->user_cr3_mfn)) {
            /* is kernel page -> rw fixup for page tables */
            if (pgtable_fixup_flag(cpu, cr2, _PAGE_RW) > 0) {
                vminfo.faults[XEN_FAULT_PAGE_FAULT_FIX_RO]++;
                return;
            }
        }
    }

    if (regs->error & 0x01) {
	/* present */
	if (pgtable_fixup_flag(cpu, cr2, _PAGE_USER) > 0) {
	    vminfo.faults[XEN_FAULT_PAGE_FAULT_FIX_USER]++;
	    return;
	}
    }

#if 0 /* for debugging */
    print_page_fault_info(1, cpu, regs, cr2);
    pgtable_walk(1, cr2, read_cr3_mfn(cpu));
#endif
    vminfo.faults[XEN_FAULT_PAGE_FAULT_GUEST]++;
    bounce_trap(cpu, regs, 14, -1);
}

asmlinkage void do_int_80(struct regs_64 *regs)
{
    struct xen_cpu *cpu =get_cpu();

    vminfo.faults[XEN_FAULT_INT_80]++;
    bounce_trap(cpu, regs, 0x80, -1);
}
