#ifndef longmode
# error do not include this directly but via emu.h
#endif

struct regs_32 {
    /* pushed onto stack before calling into C code */
    uint32_t eax;
    uint32_t ebx;
    uint32_t ecx;
    uint32_t edx;
    uint32_t esi;
    uint32_t edi;
    uint32_t ds;
    uint32_t es;
    uint32_t ebp;
    uint32_t trapno;
    /* trap / fault / int created */
    uint32_t error;
    uint32_t eip;
    uint32_t cs;
    uint32_t eflags;
    uint32_t esp;
    uint32_t ss;
};

/* 32bit defines */
#define EMUNAME   "emu32"
#define regs      regs_32
#define fix_sel   fix_sel32
#define fix_desc  fix_desc32
#define ureg_t    uint32_t
#define sreg_t    int32_t
#define PRIxREG   PRIx32
#define rax       eax
#define rbx       ebx
#define rcx       ecx
#define rdx       edx
#define rsi       esi
#define rdi       edi
#define rbp       ebp
#define rsp       esp
#define rip       eip
#define rflags    eflags
#define tss(_v)   ((0xe000 >> 3) + 8 + (((_v)->id) << 1))
#define ldt(_v)   ((0xe000 >> 3) + 9 + (((_v)->id) << 1))

/* emu32.S */
extern uint32_t emu_pgd_32[];
extern uint64_t emu_pgd_pae[];
extern uint64_t emu_pmd_pae[];

/* emu-data.c */
#define MAPS_MAX 256
extern page_aligned uint32_t maps_32[];
extern page_aligned uint64_t maps_pae[];
extern uint32_t maps_refcnt[MAPS_MAX];

extern struct descriptor_32 page_aligned xen_idt[256];
extern uint32_t *m2p;

/* emu-main.c */
int is_pae(void);

/* emu-mm.c */
void *map_32(uint32_t maddr);
void *map_pae(uint64_t maddr);
void *map_page(uint64_t maddr);
void *fixmap_page(struct xen_cpu *cpu, uint64_t maddr);
void free_page(void *ptr);
uint32_t *find_pte_32_lpt(uint32_t va);
uint32_t *find_pte_32_map(struct xen_cpu *cpu, uint32_t va);
uint64_t *find_pte_pae_lpt(uint32_t va);
uint64_t *find_pte_pae_map(struct xen_cpu *cpu, uint32_t va);
void pgtable_walk_pae(struct xen_cpu *cpu, uint32_t va);

/* emu-hcall.c */
asmlinkage void do_hypercall(struct regs_32 *regs);

/* macros */
#define context_is_emu(_r)       (((_r)->cs & 0x03) == 0x00)
#define context_is_kernel(_v,_r) (((_r)->cs & 0x03) == 0x01)
#define context_is_user(_v,_r)   (((_r)->cs & 0x03) == 0x03)

/* inline asm bits */
static inline int wrmsr_safe(uint32_t msr, uint32_t ax, uint32_t dx)
{
    int ret;
    asm volatile("91:  wrmsr               \n"
		 "    xorl %0,%0           \n"
		 "92:  nop                 \n"
		 ".section .exfix, \"ax\"  \n"
		 "93:  mov $-1,%0          \n"
		 "    jmp 92b              \n"
		 ".previous                \n"
		 ".section .extab, \"a\"   \n"
		 "    .align 4             \n"
		 "    .long 91b,93b        \n"
		 ".previous                \n"
		 : "=r" (ret)
		 : "c" (msr), "a" (ax), "d" (dx));
    return ret;
}

static inline int memcpy_pf(void *dest, const void *src, size_t bytes)
{
    int ret;

    asm volatile("    cld                  \n"
		 "91: rep movsb            \n"
		 "    xor %[ret],%[ret]    \n"
		 "98:                      \n"

		 ".section .exfix, \"ax\"  \n"
		 "99: mov $-1, %[ret]      \n"
		 "    jmp 98b              \n"
		 ".previous                \n"

		 ".section .extab, \"a\"   \n"
		 "    .align 4             \n"
		 "    .long 91b,99b        \n"
		 ".previous                \n"
		 : [ ret ] "=r" (ret),
		   [ esi ] "+S" (src),
		   [ edi ] "+D" (dest),
		   [ ecx ] "+c" (bytes)
		 :
		 : "memory" );
    return ret;
}

static inline int copy32_pf(uint32_t *dest, const uint32_t *src)
{
    int ret;
    
    asm volatile("91: mov (%[src]),%%ebx   \n"
		 "92: mov %%ebx,(%[dst])   \n"
		 "    xor %[ret],%[ret]    \n"
		 "98:                      \n"

		 ".section .exfix, \"ax\"  \n"
		 "99: mov $-1, %[ret]      \n"
		 "    jmp 98b              \n"
		 ".previous                \n"

		 ".section .extab, \"a\"   \n"
		 "    .align 4             \n"
		 "    .long 91b,99b        \n"
		 "    .long 92b,99b        \n"
		 ".previous                \n"

		 : [ ret ] "=r" (ret)
		 : [ src ] "r" (src),
		   [ dst ] "r" (dest)
		 : "ebx", "memory" );
    return ret;
}

static inline int copy64_pf(uint64_t *dest, const uint64_t *src)
{
    int ret;
    
    asm volatile("91: mov (%[src]), %%ebx    \n"
		 "92: mov 4(%[src]), %%ecx   \n"
		 "93: mov (%[dst]), %%eax    \n"
		 "94: mov 4(%[dst]), %%edx   \n"
		 "95: cmpxchg8b (%[dst])     \n"
		 "    xor %[ret],%[ret]      \n"
		 "98:                        \n"

		 ".section .exfix, \"ax\"    \n"
		 "99: mov $-1, %[ret]        \n"
		 "    jmp 98b                \n"
		 ".previous                  \n"

		 ".section .extab, \"a\"     \n"
		 "    .align 4               \n"
		 "    .long 91b,99b          \n"
		 "    .long 92b,99b          \n"
		 "    .long 93b,99b          \n"
		 "    .long 94b,99b          \n"
		 "    .long 95b,99b          \n"
		 ".previous                  \n"
		 : [ ret ] "=r" (ret)
		 : [ src ] "r" (src),
		   [ dst ] "r" (dest)
		 : "eax", "ebx", "ecx", "edx", "memory");
    return ret;
}
