#ifndef __PROCESSOR_H__
#define __PROCESSOR_H__ 1

/*
 * x86 hardware specific structs and defines
 */

/* page size */
#undef PAGE_SHIFT
#undef PAGE_SIZE
#undef PAGE_MASK
#define PAGE_SHIFT             12
#define PAGE_SIZE              (1 << PAGE_SHIFT)
#define PAGE_MASK              (~(PAGE_SIZE-1))

#define addr_to_frame(addr)    ((addr) >> PAGE_SHIFT)
#define frame_to_addr(frame)   ((frame) << PAGE_SHIFT)
#define addr_offset(addr)      ((addr) & ~PAGE_MASK)

/* page flags */
#define _PAGE_PRESENT	0x001
#define _PAGE_RW	0x002
#define _PAGE_USER	0x004
#define _PAGE_PWT	0x008
#define _PAGE_PCD	0x010
#define _PAGE_ACCESSED	0x020
#define _PAGE_DIRTY	0x040
#define _PAGE_PSE	0x080	/* 4 MB (or 2MB) page, Pentium+, if present.. */
#define _PAGE_GLOBAL	0x100	/* Global TLB entry PPro+ */
#define _PAGE_NX	((uint64_t)1<<63)

/* 32-bit paging */
#define PGD_SHIFT_32      22
#define PTE_SHIFT_32      12

#define PGD_COUNT_32      1024
#define PTE_COUNT_32      1024

#define PGD_INDEX_32(va)  (((va) >> PGD_SHIFT_32) & (PGD_COUNT_32-1))
#define PTE_INDEX_32(va)  (((va) >> PTE_SHIFT_32) & (PTE_COUNT_32-1))

static inline uint32_t get_pgentry_32(uint32_t frame, uint32_t flags)
{
    return (frame << PAGE_SHIFT) | flags;
}
static inline uint32_t get_pgframe_32(uint32_t entry)
{
    return entry >> PAGE_SHIFT;
}
static inline uint32_t get_pgflags_32(uint32_t entry)
{
    return entry & ~PAGE_MASK;
}
static inline uint32_t test_pgflag_32(uint32_t entry, uint32_t flag)
{
    return entry & ~PAGE_MASK & flag;
}

/* 32-bit pae paging */
#define PGD_SHIFT_PAE     30
#define PMD_SHIFT_PAE     21
#define PTE_SHIFT_PAE     12

#define PGD_COUNT_PAE     4
#define PMD_COUNT_PAE     512
#define PTE_COUNT_PAE     512

#define PGD_INDEX_PAE(va) (((va) >> PGD_SHIFT_PAE) & (PGD_COUNT_PAE-1))
#define PMD_INDEX_PAE(va) (((va) >> PMD_SHIFT_PAE) & (PMD_COUNT_PAE-1))
#define PTE_INDEX_PAE(va) (((va) >> PTE_SHIFT_PAE) & (PTE_COUNT_PAE-1))

static inline uint64_t get_pgentry_pae(uint32_t frame, uint32_t flags)
{
    return (frame << PAGE_SHIFT) | flags;
}
static inline uint32_t get_pgframe_pae(uint64_t entry)
{
    return (entry & ~_PAGE_NX) >> PAGE_SHIFT;
}
static inline uint32_t get_pgflags_pae(uint64_t entry)
{
    return entry & ~PAGE_MASK;
}
static inline uint32_t test_pgflag_pae(uint64_t entry, uint32_t flag)
{
    return entry & ~PAGE_MASK & flag;
}

/* 64-bit paging */
#define PGD_SHIFT_64     39
#define PUD_SHIFT_64     30
#define PMD_SHIFT_64     21
#define PTE_SHIFT_64     12

#define PGD_COUNT_64     512
#define PUD_COUNT_64     512
#define PMD_COUNT_64     512
#define PTE_COUNT_64     512

#define PGD_INDEX_64(va) (((va) >> PGD_SHIFT_64) & (PGD_COUNT_64-1))
#define PUD_INDEX_64(va) (((va) >> PUD_SHIFT_64) & (PUD_COUNT_64-1))
#define PMD_INDEX_64(va) (((va) >> PMD_SHIFT_64) & (PMD_COUNT_64-1))
#define PTE_INDEX_64(va) (((va) >> PTE_SHIFT_64) & (PTE_COUNT_64-1))

static inline uint64_t get_pgentry_64(uint64_t frame, uint32_t flags)
{
    return (frame << PAGE_SHIFT) | flags;
}
static inline uint64_t get_pgframe_64(uint64_t entry)
{
    return (entry & ~_PAGE_NX) >> PAGE_SHIFT;
}
static inline uint32_t get_pgflags_64(uint64_t entry)
{
    return entry & ~PAGE_MASK;
}
static inline uint32_t test_pgflag_64(uint64_t entry, uint32_t flag)
{
    return entry & ~PAGE_MASK & flag;
}

/* ------------------------------------------------------------------ */

/*
 * EFLAGS bits
 */
#define X86_EFLAGS_CF	0x00000001 /* Carry Flag */
#define X86_EFLAGS_PF	0x00000004 /* Parity Flag */
#define X86_EFLAGS_AF	0x00000010 /* Auxillary carry Flag */
#define X86_EFLAGS_ZF	0x00000040 /* Zero Flag */
#define X86_EFLAGS_SF	0x00000080 /* Sign Flag */
#define X86_EFLAGS_TF	0x00000100 /* Trap Flag */
#define X86_EFLAGS_IF	0x00000200 /* Interrupt Flag */
#define X86_EFLAGS_DF	0x00000400 /* Direction Flag */
#define X86_EFLAGS_OF	0x00000800 /* Overflow Flag */
#define X86_EFLAGS_IOPL	0x00003000 /* IOPL mask */
#define X86_EFLAGS_NT	0x00004000 /* Nested Task */
#define X86_EFLAGS_RF	0x00010000 /* Resume Flag */
#define X86_EFLAGS_VM	0x00020000 /* Virtual Mode */
#define X86_EFLAGS_AC	0x00040000 /* Alignment Check */
#define X86_EFLAGS_VIF	0x00080000 /* Virtual Interrupt Flag */
#define X86_EFLAGS_VIP	0x00100000 /* Virtual Interrupt Pending */
#define X86_EFLAGS_ID	0x00200000 /* CPUID detection flag */

/*
 * Basic CPU control in CR0
 */
#define X86_CR0_PE	0x00000001 /* Protection Enable */
#define X86_CR0_MP	0x00000002 /* Monitor Coprocessor */
#define X86_CR0_EM	0x00000004 /* Emulation */
#define X86_CR0_TS	0x00000008 /* Task Switched */
#define X86_CR0_ET	0x00000010 /* Extension Type */
#define X86_CR0_NE	0x00000020 /* Numeric Error */
#define X86_CR0_WP	0x00010000 /* Write Protect */
#define X86_CR0_AM	0x00040000 /* Alignment Mask */
#define X86_CR0_NW	0x20000000 /* Not Write-through */
#define X86_CR0_CD	0x40000000 /* Cache Disable */
#define X86_CR0_PG	0x80000000 /* Paging */

/*
 * Paging options in CR3
 */
#define X86_CR3_PWT	0x00000008 /* Page Write Through */
#define X86_CR3_PCD	0x00000010 /* Page Cache Disable */

/*
 * Intel CPU features in CR4
 */
#define X86_CR4_VME	0x00000001 /* enable vm86 extensions */
#define X86_CR4_PVI	0x00000002 /* virtual interrupts flag enable */
#define X86_CR4_TSD	0x00000004 /* disable time stamp at ipl 3 */
#define X86_CR4_DE	0x00000008 /* enable debugging extensions */
#define X86_CR4_PSE	0x00000010 /* enable page size extensions */
#define X86_CR4_PAE	0x00000020 /* enable physical address extensions */
#define X86_CR4_MCE	0x00000040 /* Machine check enable */
#define X86_CR4_PGE	0x00000080 /* enable global pages */
#define X86_CR4_PCE	0x00000100 /* enable performance counters at ipl 3 */
#define X86_CR4_OSFXSR	0x00000200 /* enable fast FPU save and restore */
#define X86_CR4_OSXMMEXCPT 0x00000400 /* enable unmasked SSE exceptions */
#define X86_CR4_VMXE	0x00002000 /* enable VMX virtualization */

/* ------------------------------------------------------------------ */

struct tss_32 {
    /* hardware */
    uint16_t  back_link,__blh;
    uint32_t  esp0;
    uint16_t  ss0,__ss0h;
    uint32_t  esp1;
    uint16_t  ss1,__ss1h;
    uint32_t  esp2;
    uint16_t  ss2,__ss2h;
    uint32_t  __cr3;
    uint32_t  eip;
    uint32_t  eflags;
    uint32_t  eax,ecx,edx,ebx;
    uint32_t  esp;
    uint32_t  ebp;
    uint32_t  esi;
    uint32_t  edi;
    uint16_t  es, __esh;
    uint16_t  cs, __csh;
    uint16_t  ss, __ssh;
    uint16_t  ds, __dsh;
    uint16_t  fs, __fsh;
    uint16_t  gs, __gsh;
    uint16_t  ldt, __ldth;
    uint16_t  trace, io_bitmap_base;
} __attribute__((packed));

struct tss_64 {
    uint32_t reserved1;
    uint64_t rsp0;	
    uint64_t rsp1;
    uint64_t rsp2;
    uint64_t reserved2;
    uint64_t ist[7];
    uint32_t reserved3;
    uint32_t reserved4;
    uint16_t reserved5;
    uint16_t io_bitmap_base;
} __attribute__((packed));

/* ------------------------------------------------------------------ */

#define EFLAGS_TRAPMASK (~(X86_EFLAGS_VM|X86_EFLAGS_RF|X86_EFLAGS_NT|X86_EFLAGS_TF))

/* ------------------------------------------------------------------ */

struct descriptor_32 {
    uint32_t a,b;
};

struct idt_64 {
    uint32_t a,b,c,d;
};

#define DESC32(base,limit,type,flags) { \
	.a = ((base & 0xffff) << 16) | (limit & 0xffff), \
	.b = (base & 0xff000000) | ((base & 0xff0000) >> 16) |	\
	     (limit & 0x000f0000) | ((type & 0xff) << 8) | ((flags & 0xf) << 20) \
	}

#define GATE32(seg,addr,type) {	\
	.a = ((seg & 0xffff) << 16) | (addr & 0xffff), \
	.b = (addr & 0xffff0000) | ((type & 0xff) << 8) \
	}

#define GATE64(seg,addr,type,ist) {		       \
	.a = ((seg & 0xffff) << 16) | (addr & 0xffff), \
	.b = (addr & 0xffff0000) | ((type & 0xff) << 8) | (ist & 0x07),	\
	.c = ((addr >> 32) & 0xffffffff), \
	.d = 0 \
	}

static inline struct descriptor_32 mkdesc32(uint32_t base, uint32_t limit,
					    uint32_t type, uint32_t flags)
{
    struct descriptor_32 desc = DESC32(base, limit, type, flags);
    return desc;
}

static inline struct descriptor_32 mkgate32(uint32_t seg, uint32_t addr,
					    uint32_t type)
{
    struct descriptor_32 desc = GATE32(seg, addr, type);
    return desc;
}

static inline struct idt_64 mkgate64(uint32_t seg, uint64_t addr,
				     uint32_t type, uint32_t ist)
{
    struct idt_64 desc = GATE64(seg, addr, type, ist);
    return desc;
}

#endif /* __PROCESSOR_H__ */
