/* GLIB - Library of useful routines for C programming
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GLib Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GLib Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GLib at ftp://ftp.gtk.org/pub/gtk/. 
 */

/*
 * Modified by INOUE Seiichiro <inoue@ainet.or.jp>
 * for xyaku <http://www.ainet.or.jp/~inoue/software/xyaku/>.
 * Change points:
 *  Removed glib types such as guint, gpointer.
 *  Removed strict error checks like g_return_if_fail.
 *  Removed g_hash_global that was used as lock for MT.
 *  Removed g_hash_table_foreach_remove. I don't need this.
 *  Replace g_malloc etc. with xmalloc.
 *  Removed g_mem_chunk. Instead use a simple memory allocation.
 *  Take care of only string as key and value.
 *  Modified indent style to my preferred style.
 *  Moved g_str_equal and g_str_hash from gstring.c.
 *  Replaced TRUE and FALSE with 1 and 0.
 */

#include <stdlib.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif
#include "glib.h"
#include "mem.h"

#define HASH_TABLE_MIN_SIZE 11
#define HASH_TABLE_MAX_SIZE 13845163


typedef struct _GHashNode      GHashNode;

struct _GHashNode {
	char *key;
	char *value;
	GHashNode *next;
};

struct _GHashTable {
	int size;
	int nnodes;
	unsigned int frozen;
	GHashNode **nodes;
	GHashFunc hash_func;
	GCompareFunc key_compare_func;
};


static void		g_hash_table_resize	 (GHashTable	*hash_table);
static GHashNode**	g_hash_table_lookup_node (GHashTable	*hash_table,
											  const char *key);
static GHashNode*	g_hash_node_new		 (char *key,
										  char *value);
static void		g_hash_node_destroy	 (GHashNode	*hash_node);
static void		g_hash_nodes_destroy	 (GHashNode	*hash_node);

static int g_str_equal (const char *v1, const char *v2);
static unsigned int g_str_hash (const char *key);

static GHashNode *node_free_list = NULL;


GHashTable*
g_hash_table_new (GHashFunc    hash_func,
				  GCompareFunc key_compare_func)
{
	GHashTable *hash_table;
	int i;
  
	hash_table = (GHashTable*)xmalloc(sizeof(GHashTable));
	hash_table->size = HASH_TABLE_MIN_SIZE;
	hash_table->nnodes = 0;
	hash_table->frozen = 0;
	hash_table->hash_func = hash_func ? hash_func : g_str_hash;
	hash_table->key_compare_func = key_compare_func ? key_compare_func : g_str_equal;
	hash_table->nodes = (GHashNode**)xmalloc(hash_table->size * sizeof(GHashNode*));
  
	for (i = 0; i < hash_table->size; i++)
		hash_table->nodes[i] = NULL;
  
	return hash_table;
}

void
g_hash_table_destroy (GHashTable *hash_table)
{
	int i;

	for (i = 0; i < hash_table->size; i++)
		g_hash_nodes_destroy (hash_table->nodes[i]);
  
	xfree (hash_table->nodes);
	xfree (hash_table);
}

static inline GHashNode**
g_hash_table_lookup_node (GHashTable	*hash_table,
						  const char *key)
{
	GHashNode **node;
  
	node = &hash_table->nodes
		[(* hash_table->hash_func) (key) % hash_table->size];
  
	/* Hash table lookup needs to be fast.
	 *  We therefore remove the extra conditional of testing
	 *  whether to call the key_compare_func or not from
	 *  the inner loop.
	 */
	if (hash_table->key_compare_func)
		while (*node && !(*hash_table->key_compare_func) ((*node)->key, key))
			node = &(*node)->next;
	else
		while (*node && (*node)->key != key)
			node = &(*node)->next;
  
	return node;
}

const char*
g_hash_table_lookup (GHashTable	  *hash_table,
					 const char *key)
{
	GHashNode *node;
  
	node = *g_hash_table_lookup_node (hash_table, key);
  
	return node ? node->value : NULL;
}

void
g_hash_table_insert (GHashTable *hash_table,
					 char *key,
					 char *value)
{
	GHashNode **node;
  
	node = g_hash_table_lookup_node (hash_table, key);
  
	if (*node) {
		/* do not reset node->key in this place, keeping
		 * the old key might be intended.
		 * a g_hash_table_remove/g_hash_table_insert pair
		 * can be used otherwise.
		 *
		 * node->key = key; */
		(*node)->value = value;
    } else {
		*node = g_hash_node_new (key, value);
		hash_table->nnodes++;
		if (!hash_table->frozen)
			g_hash_table_resize (hash_table);
    }
}

void
g_hash_table_remove (GHashTable	     *hash_table,
					 const char *key)
{
	GHashNode **node, *dest;
  
	node = g_hash_table_lookup_node (hash_table, key);

	if (*node) {
		dest = *node;
		(*node) = dest->next;
		g_hash_node_destroy (dest);
		hash_table->nnodes--;
  
		if (!hash_table->frozen)
			g_hash_table_resize (hash_table);
    }
}

int
g_hash_table_lookup_extended (GHashTable	*hash_table,
							  const char *lookup_key,
							  const char **orig_key,
							  const char **value)
{
	GHashNode *node;
  
	node = *g_hash_table_lookup_node (hash_table, lookup_key);
  
	if (node) {
		if (orig_key)
			*orig_key = node->key;
		if (value)
			*value = node->value;
		return 1;
    } else
		return 0;
}

void
g_hash_table_freeze (GHashTable *hash_table)
{
	hash_table->frozen++;
}

void
g_hash_table_thaw (GHashTable *hash_table)
{
	if (hash_table->frozen)
		if (!(--hash_table->frozen))
			g_hash_table_resize (hash_table);
}

void
g_hash_table_foreach (GHashTable *hash_table,
					  GHFunc	  func,
					  void *user_data)
{
	GHashNode *node;
	int i;
  
	for (i = 0; i < hash_table->size; i++)
		for (node = hash_table->nodes[i]; node; node = node->next)
			(* func) (node->key, node->value, user_data);
}

/* Returns the number of elements contained in the hash table. */
unsigned int
g_hash_table_size (GHashTable *hash_table)
{
	return hash_table->nnodes;
}

static void
g_hash_table_resize (GHashTable *hash_table)
{
	GHashNode **new_nodes;
	GHashNode *node;
	GHashNode *next;
	float nodes_per_list;
	int hash_val;
	int new_size;
	int i;
  
	nodes_per_list = (float) hash_table->nnodes / (float) hash_table->size;
  
	if ((nodes_per_list > 0.3 || hash_table->size <= HASH_TABLE_MIN_SIZE) &&
		(nodes_per_list < 3.0 || hash_table->size >= HASH_TABLE_MAX_SIZE))
		return;
  
	new_size = CLAMP(g_spaced_primes_closest (hash_table->nnodes),
					 HASH_TABLE_MIN_SIZE,
					 HASH_TABLE_MAX_SIZE);
	new_nodes = (GHashNode**)xcalloc(new_size, sizeof(GHashNode*));
  
	for (i = 0; i < hash_table->size; i++)
		for (node = hash_table->nodes[i]; node; node = next) {
			next = node->next;

			hash_val = (* hash_table->hash_func) (node->key) % new_size;

			node->next = new_nodes[hash_val];
			new_nodes[hash_val] = node;
		}
  
	xfree (hash_table->nodes);
	hash_table->nodes = new_nodes;
	hash_table->size = new_size;
}

static GHashNode*
g_hash_node_new (char *key,
				 char *value)
{
	GHashNode *hash_node;
  
	if (node_free_list) {
		hash_node = node_free_list;
		node_free_list = node_free_list->next;
    } else {
		/* Original glib code used g_mem_chunk routines.
		   I don't want the dependency, so I replaced it with a simple malloc.
		   This might be more inefficient. */
		hash_node = (GHashNode*)xmalloc(sizeof(GHashNode));
    }
  
	hash_node->key = key;
	hash_node->value = value;
	hash_node->next = NULL;
  
	return hash_node;
}

static void
g_hash_node_destroy (GHashNode *hash_node)
{
	hash_node->next = node_free_list;
	node_free_list = hash_node;
}

static void
g_hash_nodes_destroy (GHashNode *hash_node)
{
	if (hash_node) {
		GHashNode *node = hash_node;
  
		while (node->next)
			node = node->next;
  
		node->next = node_free_list;
		node_free_list = hash_node;
    }
}

/* Picked up gstring.c <glib-1.2.8>
   I modified them, because original code was too general for my purpose. */
/* Hash Functions.
 */

static int
g_str_equal (const char *v1, const char *v2)
{
  return strcmp(v1, v2) == 0;
}

/* 31 bit hash function */
static unsigned int
g_str_hash (const char *key)
{
  const char *p = key;
  int h = *p;
  
  if (h)
    for (p += 1; *p != '\0'; p++)
      h = (h << 5) - h + *p;
  
  return h;
}
