/*
 * Xpopup window module
 *
 * 1998/04/25
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#if defined(HAVE_STRING_H)
#include <string.h>
#elif defined(HAVE_STRINGS_H)
#include <strings.h>
#endif

#include <X11/IntrinsicP.h>
#include <X11/Xatom.h>

#include "xyaku.h"
#include "xpopup.h"
#include "mem.h"
#include "cache.h"


/* Constant numbers */
#define DEFAULT_MAX_BUFSIZ	1024
#define DEFAULT_MAX_NLINES	16
#define DEFAULT_AUTOMODE_INTERVAL	1500	/* [millisecond] */
#define	VERTICAL_MARGIN		5
#define	HORIZONTAL_MARGIN	5
#define CACHE_INI_SIZE		256	/* currently, no meaning */

#define INCREMENT_BORDER	5

/* Data structure */
struct _XPopup {
	/* Xt Widget */
	Widget w;

	GC gc;
	XFontSet fontset;
	XFontSetExtents	*fontset_ext;
	Dimension max_width;

	/* String to draw. String is terminated with a null byte */
	uchar *pstr;/* pointer to string, which points to s_str or an entry in cache */

	/* Buffer to store string to draw */
	int cur_max_bufsiz;	/* set via xpopup_set_addin_cmd() */
	uchar *s_str;		/* string */

	/* The following vals work as kinda cache for drawing string */
	int cur_max_nlines;	/* set via xpopup_set_addin_cmd() */
	int *len_lines;		/* array of length of every line of string */
	int	total_nlines;	/* total number of display lines */

	/* Cache: selected string(key) and result string(value) pair is stored */
	int f_use_cache;/* flag */
	XyCache *cache;

	/* Add-in command and its arguments.
	   These are updated via xpopup_set_addin_cmd() */
	const char *addin_cmd;
	char **argv;

	/* to move the window, store the position */
	int	grab_flag;
	int	grab_x;
	int	grab_y;

	/* automode */
	XtIntervalId timer_id;/* timer_id > 0 implies automode */
	unsigned long interval; /* timer interval [millisecond] */
};


/* Action procedures */
static void Redisplay(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);
static void Grab(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);
static void UnGrab(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);
static void Move(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);
static void UnMap(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);
static void Selection(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont);

/* Internal functions */
static void show_result(XPopup *xpop);
static void popup_window(XPopup *xpop, int width, int height);
static void redraw_window(XPopup *xpop, int width, int height);
static const char* get_dsp_line_break(XFontSet xfs, const char *ptr, int lenb, int max_width);
static int adjust_pos_on_rootw(int start, int size, int root_size);
static void timeout_cb(XtPointer client_data, XtIntervalId *id);


/* initialization */
XPopup*
xpopup_new(Widget w, const AppResources appRes)
{
	XPopup *xpop;

	xpop = (XPopup*)xmalloc(sizeof(XPopup));
	XtRealizeWidget(w);
	xpop->w = w;
	
	xpop->gc = XCreateGC(dpy, XtWindow(w), 0, 0);
	XSetForeground(dpy, xpop->gc, appRes->foreground_pixel);
	xpop->fontset = appRes->fontset;
	xpop->fontset_ext = XExtentsOfFontSet(xpop->fontset);
	xpop->max_width = appRes->max_width;
	
	/* Add event handlers */
	XtAddEventHandler(w, NoEventMask, True, Selection, xpop);
	XtAddEventHandler(w, ExposureMask, False, Redisplay, xpop);
	XtAddEventHandler(w, LeaveWindowMask, False, UnMap, xpop);
	XtAddEventHandler(w, KeyPressMask, False, UnMap, xpop);
	XtAddEventHandler(w, ButtonPressMask, False, Grab, xpop);
	XtAddEventHandler(w, ButtonReleaseMask, False, UnGrab, xpop);
	/*obsolete	XtAddEventHandler(w, PointerMotionMask, False, Move, xpop);*/
	XtAddEventHandler(w, ButtonMotionMask, False, Move, xpop);

	/* initialize the internal data */
	xpop->pstr = NULL;
	xpop->cur_max_bufsiz = DEFAULT_MAX_BUFSIZ;
	xpop->s_str = (uchar*)xmalloc(sizeof(uchar) * xpop->cur_max_bufsiz);
	xpop->cur_max_nlines = DEFAULT_MAX_NLINES;
	xpop->len_lines = (int*)xmalloc(sizeof(int) * xpop->cur_max_nlines);;
	xpop->total_nlines = 0;
	
	/* cache */
	xpop->f_use_cache = 0;
	xpop->cache = NULL;
	
	xpop->addin_cmd = NULL;
	xpop->argv = NULL;

	xpop->grab_x = 30;
	xpop->grab_y = 30;
	xpop->grab_flag = False;

	xpop->timer_id = 0;
	xpop->interval = DEFAULT_AUTOMODE_INTERVAL;

	return xpop;
}

void
xpopup_destroy(XPopup *xpop)
{
	xfree(xpop->len_lines);
	xfree(xpop->s_str);
	if (xpop->cache)
		cache_destroy(xpop->cache);
	XFreeGC(dpy, xpop->gc);
	
	xfree(xpop);
}

Window
xpopup_get_xwin(const XPopup *xpop)
{
	return XtWindow(xpop->w);
}

Widget
xpopup_get_widget(const XPopup *xpop)
{
	return xpop->w;
}

/**
 * xpopup_set_addin_cmd:
 * Set the specified add-in command path, arguments, max width, max # of lines,
 * and max buffer size.
 * I don't copy the contents of addin_cmd and argv, because they are static data.
 * It implies XPopup depends on the external module's state.
 * It's not beautiful...
 * If max_nlines of max_bufsiz are given, enlarge two arrays, len_lines and s_str.
 * In that case, I don't need the old array, so free and malloc.
 * When the specified length is smaller than the curren one, nerve shrink them for efficiency.
 **/
void
xpopup_set_addin_cmd(XPopup *xpop, const char *addin_cmd, char **argv, int max_nlines, int max_bufsiz, int max_width)
{
	/* Just refer them */
	xpop->addin_cmd = addin_cmd;
	xpop->argv = argv;
	
	if (max_nlines) {
		/* If the current array is not big enough, enlarge it */
		if (xpop->cur_max_nlines < max_nlines) {
			xfree(xpop->len_lines);
			xpop->len_lines = (int*)xmalloc(sizeof(int) * max_nlines);
		}
		xpop->cur_max_nlines = max_nlines;
	} else {
		xpop->cur_max_nlines = DEFAULT_MAX_NLINES;
	}

	if (max_bufsiz) {
		/* If the current buffer is not big enough, enlarge it */
		if (xpop->cur_max_bufsiz < max_bufsiz) {
			xfree(xpop->s_str);
			xpop->s_str = (uchar*)xmalloc(sizeof(uchar) * max_bufsiz);
		}
		xpop->cur_max_bufsiz = max_bufsiz;
	} else {
		xpop->cur_max_bufsiz = DEFAULT_MAX_BUFSIZ;
	}

	xpop->max_width = max_width;/* override */
}

/**
 * xpopup_set_automode:
 * If f_automode == 1, enable automode. Or, disable automode.
 * Return 1 if automode on. Or, return 0.
 * Change the visibility(just border).
 **/
int
xpopup_set_automode(XPopup *xpop, int f_automode, unsigned long interval)
{
	XWindowAttributes attr;
	
	if (xpop->timer_id) {
		XtRemoveTimeOut(xpop->timer_id);
	}

	if (xpop->timer_id == 0 && f_automode) {
		XGetWindowAttributes(dpy, XtWindow(xpop->w), &attr);
		XSetWindowBorderWidth(dpy, XtWindow(xpop->w),
							  attr.border_width + INCREMENT_BORDER);
	} else if (xpop->timer_id && !f_automode) {
		XGetWindowAttributes(dpy, XtWindow(xpop->w), &attr);
		XSetWindowBorderWidth(dpy, XtWindow(xpop->w),
							  attr.border_width - INCREMENT_BORDER);
	}

	if (f_automode) {
		xpop->interval = (interval) ? interval : DEFAULT_AUTOMODE_INTERVAL;
		xpop->timer_id = XtAppAddTimeOut(XtWidgetToApplicationContext(xpop->w),
										 xpop->interval, timeout_cb, xpop);
		return 1;
		
	} else {
		xpop->timer_id = 0;
		return 0;
	}
}

/**
 * xpopup_set_use_cache:
 * If f_use_cache is true, enable cache.
 * Don't discard the cache even if f_use_cache is false, for efficiency.
 **/
int
xpopup_set_use_cache(XPopup *xpop, int f_use_cache)
{
	xpop->f_use_cache = f_use_cache;

	if (f_use_cache && xpop->cache == NULL) {
		xpop->cache = cache_new(CACHE_INI_SIZE);
	}
	return f_use_cache;
}

/**
 * xpopup_clear_cache:
 * Clear the contents of the current cache.
 **/
void
xpopup_clear_cache(XPopup *xpop)
{
	if (xpop->cache)
		cache_destroy(xpop->cache);
#ifdef DEBUG
	fprintf(stderr, "cache cleared\n");
#endif
	xpop->cache = cache_new(CACHE_INI_SIZE);
}


/* The followings are private functions. */
/* In the callback functions, xpop->w == w is always true. */
/*
 * Expose method
 * redisplay the string.
 */
static void
Redisplay(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;
	int line;
	const uchar	*p_line;/* pointer to a beginning of a line */
	int x, y;
	int height_line = xpop->fontset_ext->max_ink_extent.height;
	const int *len_lines = xpop->len_lines;
	int	total_nlines = xpop->total_nlines;

	if (!XtIsRealized(w))
		return;
	if (xpop->pstr == NULL)
		return;
	
	x = VERTICAL_MARGIN;
	y = height_line + HORIZONTAL_MARGIN;
	for (line = 0, p_line = xpop->pstr; line < total_nlines; line++) {
#ifdef DEBUG
		uchar *buf;

		buf = (uchar*)xmalloc(sizeof(char) * xpop->cur_max_bufsiz);
		memcpy(buf, p_line, len_lines[line]);
		buf[len_lines[line]] = '\0';
		fprintf(stderr, "LINE:%d:LEN=%d: %s\n", line, len_lines[line], buf);
		xfree(buf);
#endif
		if (len_lines[line]) {
			XmbDrawString(dpy, XtWindow(w), xpop->fontset,
						  xpop->gc, x, y, p_line, len_lines[line]);
			p_line += len_lines[line];
			p_line += strspn(p_line, "\r\n");
		}
		y += height_line;
	}
}


/*
 * Called when key pressed or mouse pointer left.
 * Unmap the window, except grabbed or automode.
 */
static void
UnMap(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;

	if (xpop->grab_flag == False && xpop->timer_id == 0) {
		XtUnmapWidget(w);
	}
}


/*
 * Called when click a button on popup window.
 * Keep the position to drag the popup window.
 */
static void
Grab(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;

	if (ev->type == ButtonPress) {
		xpop->grab_x = ev->xbutton.x;
		xpop->grab_y = ev->xbutton.y;
		xpop->grab_flag = True;
	}
}

/* Called when release mouse button. Unset grab flag. */
static void
UnGrab(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;

	xpop->grab_flag = False;
}


/* Called when a user drag popup window. */
static void
Move(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;

	if (xpop->grab_flag == True) {
		XMoveWindow(dpy, XtWindow(w),
					ev->xmotion.x_root - xpop->grab_x,
					ev->xmotion.y_root - xpop->grab_y);
	}
}

/*
 * SelectionNotify event method
 * Get the selected word and pass it to add-in commnads.
 * Then call popup_display().
 */
static void 
Selection(Widget w, XtPointer client_data, XEvent *ev, Boolean *cont)
{
	XPopup *xpop = client_data;
	uchar *selected_str;	/* terminated by a null byte */
	int lenb_selected_str;	/* length in byte of selected_str */
	int lenb_result;		/* length in byte of result string */
	uchar *old_pstr;		/* used to avoid useless redrawing */

	if (ev->type != SelectionNotify)
		return;
	if (xpop->addin_cmd == NULL)
		return;
	
	selected_str = get_selected_string(ev, XtWindow(w));
	if (selected_str == NULL)
		return;
	
	lenb_selected_str = strlen(selected_str);
	/* check the cache */
	old_pstr = xpop->pstr; /* used to avoid useless redrawing */
	xpop->pstr = NULL;
	if (xpop->f_use_cache) {
		xpop->pstr = (uchar*)cache_lookup(xpop->cache, selected_str, lenb_selected_str);
	}

	if (xpop->pstr) {
#ifdef DEBUG
		fprintf(stderr, "cache hit: %s\n", selected_str);
#endif
		if (xpop->pstr[0] != '\0') {
			if (xpop->timer_id) {
				if (xpop->pstr == old_pstr) {
					;/* No need to redraw */
				} else {
					show_result(xpop);
				}
			} else {
				show_result(xpop);
			}
		}
	} else {
		/* Execute add-in command. Result string is returned in xpop->s_str. */
		lenb_result = exec_addin_cmd(xpop->addin_cmd, xpop->argv, selected_str,
									 xpop->s_str, xpop->cur_max_bufsiz);
		xpop->s_str[lenb_result] = '\0';
		xpop->pstr = xpop->s_str;
#ifdef DEBUG
		fprintf(stderr, "len=%d, %s\n", lenb_result, xpop->s_str);
#endif
		if (xpop->f_use_cache) {
			if (!cache_insert(xpop->cache, selected_str, lenb_selected_str,
							  xpop->s_str, lenb_result))
				fprintf(stderr, "Warning: cache_insert error\n");
		}
		if (lenb_result > 0) {
			show_result(xpop);
		}
	}
	
	XFree(selected_str);
}


/* Internal functions */
/**
 * show_result:
 * Calculate the window size. Then, mapped the window.
 * Information such as len_lines are stored for a later use.
 * Note:
 * "Logical line" represents line separated by CR or LF in the buffer,
 * "Display line" represents line on the display.
 **/
static void
show_result(XPopup *xpop)
{
	Widget w = xpop->w;
	int nlines;/* display line number */
	const uchar	*ptr;
	const uchar	*p_line;/* pointer to the beginning of logical line */
	int total_width = 0;/* Indeed, it's maximum width. */
	int total_height = 0;
	int height_line;

	/* count the number of lines, and compute the window size */
	nlines = 0;
	ptr = p_line = xpop->pstr;
	do {
		int escapement;/* equals to width */
		int left_lenb;/* bytes left in the logical line. */
		
		ptr = strpbrk(p_line, "\r\n");/* point to the end of logical line */
		if (ptr == NULL)
			break;
		left_lenb = ptr - p_line;

		/* If a user specify max width of popup window, wrap display lines. */
		if (xpop->max_width > 0)
			ptr = get_dsp_line_break(xpop->fontset, p_line, left_lenb, xpop->max_width);

		/* Store the length of the display line */
		xpop->len_lines[nlines] = ptr - p_line;

		/* Compute the extent for one display line */
		escapement = XmbTextEscapement(xpop->fontset, p_line, xpop->len_lines[nlines]);

		if (escapement > total_width) {
			total_width = escapement;/* Total width is the max  */
		}

		left_lenb -= (ptr - p_line);
		if (left_lenb == 0) {
			/* point to the beginning of the next logical line */
			if (ptr[0] == '\r' && ptr[1] == '\n')
				ptr++;
			ptr++;/* skip one '\n' */
			
			while (ptr[0] != '\0' && (ptr[0] == '\r' || ptr[0] == '\n')) {
				if (ptr[0] == '\r' && ptr[1] == '\n')
					ptr++;
				ptr++;
				if (++nlines >= xpop->cur_max_nlines)
					break;
				xpop->len_lines[nlines] = 0;/* empty line */
			}
			p_line = ptr;
		} else {
			p_line = ptr;
		}
		nlines++;
	} while (nlines < xpop->cur_max_nlines);

	if (nlines >= xpop->cur_max_nlines)/* this could happen */
		xpop->total_nlines = xpop->cur_max_nlines - 1;
	else
		xpop->total_nlines = nlines;

	/* Calculate the height */
	height_line = xpop->fontset_ext->max_ink_extent.height;
	total_height = height_line * (xpop->total_nlines);

	if (xpop->timer_id == 0) {
		popup_window(xpop, total_width, total_height);
	} else {
		redraw_window(xpop, total_width, total_height);
	}

	XRaiseWindow(dpy, XtWindow(w));

	XtMapWidget(w);
}

/*
 * Pop up the window. This is called in non-automode.
 */
static void
popup_window(XPopup *xpop, int width, int height)
{
	Widget w = xpop->w;
	int pos_x, pos_y;
	int root_width, root_height;
	
	/* I assume the size of root window is the same as one of the screen. */
	root_width = WidthOfScreen(DefaultScreenOfDisplay(dpy));
	root_height = HeightOfScreen(DefaultScreenOfDisplay(dpy));

	/* Recalculate the position */
	pos_x = adjust_pos_on_rootw(w->core.x - xpop->grab_x,
								width + VERTICAL_MARGIN * 2, root_width);
	pos_y = adjust_pos_on_rootw(w->core.y - xpop->grab_y,
								height + HORIZONTAL_MARGIN * 2, root_height);
	XMoveResizeWindow(dpy, XtWindow(w), pos_x, pos_y,
					  width + VERTICAL_MARGIN * 2,
					  height + HORIZONTAL_MARGIN * 2);
}

/*
 * Redraw the window. This is called in automode.
 */
static void
redraw_window(XPopup *xpop, int width, int height)
{
	Widget w = xpop->w;
	XEvent xev;

#ifdef DEBUG
	fprintf(stderr, "redraw\n");
#endif
	XClearWindow(dpy, XtWindow(w));
	XResizeWindow(dpy, XtWindow(w),
				  width + VERTICAL_MARGIN * 2,
				  height + HORIZONTAL_MARGIN * 2);
	/* Send Expose event */
	xev.xexpose.type = Expose;
	xev.xexpose.display = dpy;
	xev.xexpose.window = XtWindow(w);
	xev.xexpose.x = 0;
	xev.xexpose.y = 0;
	xev.xexpose.width = width;
	xev.xexpose.height = height;
	xev.xexpose.count = 0;
	XSendEvent(dpy, XtWindow(w), FALSE, ExposureMask, &xev);
}


/**
 * get_dsp_line_break:
 * Get the display line break.
 * When max_width is specified, I have to wrap the display lines.
 * Returns the pointer of the beginning of the next display line.
 **/
static const char*
get_dsp_line_break(XFontSet xfs, const char *ptr, int lenb, int max_width)
{
#define DEFAULT_ARRAY_SZ	128	/* Normally, avoid dynamic memory allocation */
	const char *retp;
	XRectangle *p_ink_array;
	XRectangle ink_array[DEFAULT_ARRAY_SZ];
	XRectangle *p_logic_array;
	XRectangle logic_array[DEFAULT_ARRAY_SZ];
	int array_sz;
	int num_chars;
	Status sts;
	int n;
	int nc;
	
	array_sz = lenb;/* prepare maximum */
	if (array_sz > DEFAULT_ARRAY_SZ) {
		p_ink_array = xmalloc(sizeof(XRectangle) * array_sz);
		p_logic_array = xmalloc(sizeof(XRectangle) * array_sz);
	} else {
		p_ink_array = ink_array;
		p_logic_array = logic_array;
	}		

	sts = XmbTextPerCharExtents(xfs, ptr, lenb,
								p_ink_array, p_logic_array, array_sz,
								&num_chars, NULL, NULL);
	if (sts == 0) {/* error */
		/* I give up to wrap the text. */
		retp = ptr + lenb;
		goto DONE;
	}
	/* At first, check whether logical line doesn't exceed max_width */
	if ((p_ink_array[num_chars - 1].x + p_ink_array[num_chars - 1].width) < max_width) {
		retp = ptr + lenb;
		goto DONE;
	}

	/* Get the number of characters which exceeds max_width */
	for (nc = 0; nc < num_chars; nc++) {
		if ((p_ink_array[nc].x + p_ink_array[nc].width) > max_width)
			break;
	}
	if (nc == 0)
		perror("Can't display even one character in one line");

	/* retp will point to the nc'th character */
	retp = ptr;
	for (n = 0; n < nc; n++) {
		retp += mblen(retp, MB_CUR_MAX);
	}
	
 DONE:	
	if (array_sz > DEFAULT_ARRAY_SZ) {
		xfree(p_ink_array);
		xfree(p_logic_array);
	}
	return retp;
}


/*
 * Calculate the position on the root window(screen).
 * Especially, takes care of the case the window exceeds root window.
 */
static int
adjust_pos_on_rootw(int start, int size, int root_size)
{
	int ret;

	if (size > root_size) {
		ret = 0;
	} else if (start + size > root_size) {
		ret = root_size - size;
	} else if (start < 0) {
		ret = 0;
	} else {
		ret = start;
	}

	return ret;
}

static void
timeout_cb(XtPointer client_data, XtIntervalId *id)
{
	XPopup *xpop = client_data;

	/* Request for available list of selection data type.
	 * This implicitly calls SelectionNotify event handler of popup window. */
	req_selection(XA_PRIMARY, XA_TARGETS, XA_TARGETS,
				  XtWindow(xpop->w), CurrentTime);
	xpop->timer_id = XtAppAddTimeOut(XtWidgetToApplicationContext(xpop->w),
									 xpop->interval, timeout_cb, xpop);
}

