/**
 * Copyright (c) 2002-2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v20.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 */
package org.eclipse.emf.codegen.jet;


import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.codegen.CodeGenPlugin;
import org.eclipse.emf.common.util.BasicDiagnostic;
import org.eclipse.emf.common.util.Diagnostic;
import org.eclipse.emf.common.util.DiagnosticException;


/**
 * Base class for all exceptions generated by the JET engine.
 * Makes it convenient to catch just this at the top-level.
 */
public class JETException extends DiagnosticException
{
  private static final long serialVersionUID = 1L;

  private final JETMark start;

  private final JETMark stop;

  private final String problemKey;

  /**
   * @deprecated Use {@link #JETException(String, String, Throwable, JETMark, JETMark, int)}.
   */
  public JETException(String reason)
  {
    super(new BasicDiagnostic(Diagnostic.ERROR, CodeGenPlugin.INSTANCE.getSymbolicName(), 0, reason, null));
    start = null;
    stop = null;
    problemKey = JETProblemListener.UNKNOWN_PROBLEM;
  }

  /**
   * Creates a JETException with the embedded exception and the reason for throwing a JETException.
   * @deprecated Use {@link #JETException(String, String, Throwable, JETMark, JETMark, int)}.
   */
  public JETException(String reason, Throwable exception)
  {
    super(new BasicDiagnostic(Diagnostic.ERROR, CodeGenPlugin.INSTANCE.getSymbolicName(), 0, reason, new Object []{ exception }));
    start = null;
    stop = null;
    problemKey = JETProblemListener.UNKNOWN_PROBLEM;
  }

  /**
   * Can if the marks can be use to create an error marker in a resource visible in the workspace, that's done instead with this exception instead of showing it in a dialog.
   * @since 2.19
   */
  public JETException(String problemKey, String reason, Throwable exception, JETMark start, JETMark stop, int severity)
  {
    super(new BasicDiagnostic(severity, CodeGenPlugin.INSTANCE.getSymbolicName(), 0, reason, exception == null ? null : new Object []{ exception }));
    this.problemKey = problemKey;
    this.start = start;
    this.stop = stop;
  }

  /**
   * Creates a JETException with the embedded exception.
   */
  public JETException(Throwable exception)
  {
    super(BasicDiagnostic.toDiagnostic(exception));
    start = null;
    stop = null;
    problemKey = JETProblemListener.UNKNOWN_PROBLEM;
  }

  /**
   * @see JETProblemListener
   * @since 2.19
   */
  public String getProblemKey()
  {
    return problemKey;
  }

  public IStatus getStatus()
  {
    return BasicDiagnostic.toIStatus(getDiagnostic());
  }

  /**
   * @since 2.19
   */
  public JETMark getStart()
  {
    return start;
  }

  /**
   * @since 2.19
   */
  public JETMark getStop()
  {
    return stop;
  }

  protected static String getMessage(Throwable exception)
  {
    String result = exception.getLocalizedMessage();
    if (result == null)
    {
      result = "";
    }

    return result;
  }
}
