# -*- coding:utf-8 -*-

#  ************************** Copyrights and license ***************************
#
# This file is part of gcovr 5.2, a parsing and reporting tool for gcov.
# https://gcovr.com/en/stable
#
# _____________________________________________________________________________
#
# Copyright (c) 2013-2022 the gcovr authors
# Copyright (c) 2013 Sandia Corporation.
# Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
# the U.S. Government retains certain rights in this software.
#
# This software is distributed under the 3-clause BSD License.
# For more information, see the README.rst file.
#
# ****************************************************************************

import glob
import io
import logging
import os
import platform
import pytest
import re
import shutil
import subprocess
import sys
import difflib
import zipfile

from yaxmldiff import compare_xml

python_interpreter = sys.executable.replace(
    "\\", "/"
)  # use forward slash on windows as well
env = os.environ
env["GCOVR"] = python_interpreter + " -m gcovr"
for var in [
    "CPATH",
    "C_INCLUDE_PATH",
    "CPLUS_INCLUDE_PATH",
    "OBJC_INCLUDE_PATH",
    "CFLAGS",
    "CXXFLAGS",
    "LDFLAGS",
]:
    if var in env:
        env.pop(var)
# Override language for input files
env["LANG"] = "C.UTF-8"

basedir = os.path.split(os.path.abspath(__file__))[0]

skip_clean = None

CC = os.path.split(env["CC"])[1]
IS_CLANG = True if CC.startswith("clang") else False

IS_MACOS = platform.system() == "Darwin"
IS_WINDOWS = platform.system() == "Windows"
if IS_WINDOWS:
    import win32api
    import string

    used_drives = win32api.GetLogicalDriveStrings().split("\0")
    sys.stdout.write(f"Used drives: {used_drives}")
    free_drives = sorted(set(string.ascii_uppercase) - set(used_drives))
    sys.stdout.write(f"Free drives: {free_drives}")
    assert free_drives, "Must have at least one free drive letter"
    env["GCOVR_TEST_DRIVE_WINDOWS"] = f"{free_drives[-1]}:"

CC_REFERENCE = env.get("CC_REFERENCE", CC)

REFERENCE_DIRS = []
REFERENCE_DIR_VERSION_LIST = (
    ["gcc-5", "gcc-6", "gcc-8", "gcc-9", "gcc-10", "gcc-11"]
    if "gcc" in CC_REFERENCE
    else ["clang-10", "clang-13"]
)
for ref in REFERENCE_DIR_VERSION_LIST:
    REFERENCE_DIRS.append(os.path.join("reference", ref))
    if platform.system() != "Linux":
        REFERENCE_DIRS.append(f"{REFERENCE_DIRS[-1]}-{platform.system()}")
    if ref in CC_REFERENCE:
        break
REFERENCE_DIRS.reverse()

RE_DECIMAL = re.compile(r"(\d+\.\d+)")

RE_TXT_WHITESPACE = re.compile(r"[ ]+$", flags=re.MULTILINE)

RE_XML_ATTRS = re.compile(r'(timestamp)="[^"]*"')
RE_XML_GCOVR_VERSION = re.compile(r'version="gcovr [^"]+"')

RE_COVERALLS_CLEAN_KEYS = re.compile(
    r'"(commit_sha|repo_token|run_at|version)": "[^"]*"'
)
RE_COVERALLS_GIT = re.compile(
    r'"git": \{(?:"[^"]*": (?:"[^"]*"|\{[^\}]*\}|\[[^\]]*\])(?:, )?)+\}, '
)
RE_COVERALLS_GIT_PRETTY = re.compile(
    r'\s+"git": \{\s+"branch": "branch",\s+"head": \{(?:\s+"[^"]+":.+\n)+\s+\},\s+"remotes": \[[^\]]+\]\s+\},'
)

RE_HTML_ATTRS = re.compile('((timestamp)|(version))="[^"]*"')
RE_HTML_FOOTER_VERSION = re.compile(
    r"(Generated by: <a [^>]+>GCOVR \(Version) \d+\.\d+(\)</a>)"
)
RE_HTML_HEADER_DATE = re.compile(r"(<td)>\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d<(/td>)")


def scrub_txt(contents):
    return RE_TXT_WHITESPACE.sub("", contents)


def scrub_csv(contents):
    contents = contents.replace("\r", "")
    contents = contents.replace("\n\n", "\n")
    # Replace windows file separator for html reports generated in Windows
    contents = contents.replace("\\", "/")
    return contents


def scrub_xml(contents):
    contents = RE_DECIMAL.sub(lambda m: str(round(float(m.group(1)), 5)), contents)
    contents = RE_XML_ATTRS.sub(r'\1=""', contents)
    contents = RE_XML_GCOVR_VERSION.sub('version=""', contents)
    contents = contents.replace("\r", "")
    return contents


def scrub_html(contents):
    contents = RE_HTML_ATTRS.sub('\\1=""', contents)
    contents = RE_HTML_FOOTER_VERSION.sub("\\1 4.x\\2", contents)
    contents = RE_HTML_HEADER_DATE.sub("\\1>0000-00-00 00:00:00<\\2", contents)
    contents = contents.replace("\r", "")
    # Replace windows file separator for html reports generated in Windows
    contents = contents.replace("\\", "/")
    return contents


def scrub_coveralls(contents):
    contents += "\n"
    contents = RE_COVERALLS_CLEAN_KEYS.sub('"\\1": ""', contents)
    contents = RE_COVERALLS_GIT_PRETTY.sub("", contents)
    contents = RE_COVERALLS_GIT.sub("", contents)
    return contents


def findtests(basedir):
    for f in sorted(os.listdir(basedir)):
        if not os.path.isdir(os.path.join(basedir, f)):
            continue
        if f.startswith("."):
            continue
        if "pycache" in f:
            continue
        yield f


def assert_xml_equals(reference, coverage):
    diff = compare_xml(reference, coverage)
    if diff is None:
        return

    raise AssertionError(f"XML documents differed (-reference, +actual):\n{diff}")


def run(cmd, cwd=None):
    sys.stdout.write(f"STDOUT - START {cmd}\n")
    returncode = subprocess.call(cmd, stderr=subprocess.STDOUT, env=env, cwd=cwd)
    sys.stdout.write("STDOUT - END\n")
    return returncode == 0


def find_reference_files(output_pattern):
    seen_files = set([])
    for reference_dir in REFERENCE_DIRS:
        for pattern in output_pattern:
            if os.path.isdir(reference_dir):
                for file in glob.glob(os.path.join(reference_dir, pattern)):
                    if os.path.basename(file) not in seen_files:
                        coverage = os.path.basename(file)
                        seen_files.add(coverage)
                        yield coverage, file


@pytest.fixture(scope="module")
def compiled(request, name):
    path = os.path.join(basedir, name)
    assert run(["make", "clean"], cwd=path)
    assert run(["make", "all"], cwd=path)
    yield name
    if not skip_clean:
        assert run(["make", "clean"], cwd=path)


KNOWN_FORMATS = [
    "txt",
    "xml",
    "html",
    "sonarqube",
    "json",
    "json_summary",
    "csv",
    "coveralls",
]


def pytest_generate_tests(metafunc):
    """generate a list of all available integration tests."""

    global skip_clean
    skip_clean = metafunc.config.getoption("skip_clean")
    generate_reference = metafunc.config.getoption("generate_reference")
    update_reference = metafunc.config.getoption("update_reference")
    archive_differences = metafunc.config.getoption("archive_differences")

    collected_params = []

    if archive_differences:  # pragma: no cover
        diffs_zip = os.path.join(basedir, "diff.zip")
        # Create an empty ZIP
        zipfile.ZipFile(diffs_zip, mode="w").close()

    for name in findtests(basedir):
        targets = parse_makefile_for_available_targets(
            os.path.join(basedir, name, "Makefile")
        )

        # check that the "run" target lists no unknown formats
        target_run = targets.get("run", set())
        unknown_formats = target_run.difference(KNOWN_FORMATS)
        if unknown_formats:  # pragma: no cover
            raise ValueError(
                "{}/Makefile target 'run' references unknown format {}".format(
                    name, unknown_formats
                )
            )

        # check that all "run" targets are actually available
        unresolved_prereqs = target_run.difference(targets)
        if unresolved_prereqs:  # pragma: no cover
            raise ValueError(
                "{}/Makefile target 'run' has unresolved prerequisite {}".format(
                    name, unresolved_prereqs
                )
            )

        # check that all available known formats are also listed in the "run" target
        unreferenced_formats = (
            set(KNOWN_FORMATS).intersection(targets).difference(target_run)
        )
        if unreferenced_formats:  # pragma: no cover
            raise ValueError(
                "{}/Makefile target 'run' doesn't reference available target {}".format(
                    name, unreferenced_formats
                )
            )

        for format in KNOWN_FORMATS:

            # only test formats where the Makefile provides a target
            if format not in targets:
                continue

            marks = [
                pytest.mark.skipif(
                    name == "simple1-drive-subst" and not IS_WINDOWS,
                    reason="drive substitution only available on windows",
                ),
                pytest.mark.xfail(
                    name == "exclude-throw-branches"
                    and format == "html"
                    and IS_WINDOWS,
                    reason="branch coverage details seem to be platform-dependent",
                ),
                pytest.mark.xfail(
                    name == "rounding" and IS_WINDOWS,
                    reason="branch coverage seem to be platform-dependent",
                ),
                pytest.mark.xfail(
                    name == "html-source-encoding-cp1252" and IS_CLANG,
                    reason="clang doesnt understand -finput-charset=...",
                ),
                pytest.mark.xfail(
                    name == "html-source-encoding-cp1252" and IS_MACOS,
                    reason="On MacOS -finput-charset=cp1252 isn't supported",
                ),
                pytest.mark.xfail(
                    name in ["excl-branch", "exclude-throw-branches", "html-themes"]
                    and IS_MACOS,
                    reason="On MacOS the constructor is called twice",
                ),
                pytest.mark.xfail(
                    name == "gcc-abspath"
                    and (
                        not env["CC"].startswith("gcc-")
                        or int(env["CC"].replace("gcc-", "")) < 8
                    ),
                    reason="Option -fprofile-abs-path is supported since gcc-8",
                ),
            ]

            collected_params.append(
                pytest.param(
                    name,
                    format,
                    targets,
                    generate_reference,
                    update_reference,
                    archive_differences,
                    marks=marks,
                    id="-".join([name, format]),
                )
            )

    metafunc.parametrize(
        "name, format, available_targets, generate_reference, update_reference, archive_differences",
        collected_params,
        indirect=False,
        scope="module",
    )


def parse_makefile_for_available_targets(path):
    targets = {}
    with open(path) as makefile:
        for line in makefile:
            m = re.match(r"^(\w[\w -]*):([\s\w.-]*)$", line)
            if m:
                deps = m.group(2).split()
                for target in m.group(1).split():
                    targets.setdefault(target, set()).update(deps)
    return targets


def generate_reference_data(output_pattern):  # pragma: no cover
    for pattern in output_pattern:
        for generated_file in glob.glob(pattern):
            reference_file = os.path.join(REFERENCE_DIRS[0], generated_file)
            if os.path.isfile(reference_file):
                continue
            else:
                os.makedirs(REFERENCE_DIRS[0], exist_ok=True)
                logging.info(f"copying {generated_file} to {reference_file}")
                shutil.copyfile(generated_file, reference_file)


def update_reference_data(coverage_file, reference_file):  # pragma: no cover
    if CC_REFERENCE in reference_file:
        reference_dir = os.path.dirname(reference_file)
    else:
        reference_dir = os.path.join("reference", CC_REFERENCE)
        os.makedirs(reference_dir, exist_ok=True)
        reference_file = os.path.join(reference_dir, os.path.basename(reference_file))
    shutil.copyfile(coverage_file, reference_file)

    return reference_file


def archive_difference_data(name, coverage_file, reference_file):  # pragma: no cover
    diffs_zip = os.path.join("..", "diff.zip")
    with zipfile.ZipFile(diffs_zip, mode="a") as f:
        f.write(
            coverage_file,
            os.path.join(name, os.path.dirname(reference_file), coverage_file).replace(
                os.path.sep, "/"
            ),
        )


def remove_duplicate_data(
    encoding, scrub, coverage, coverage_file, reference_file
):  # pragma: no cover
    reference_dir = os.path.dirname(reference_file)
    # Loop over the other coverage data
    for reference_dir in REFERENCE_DIRS:  # pragma: no cover
        other_reference_file = os.path.join(reference_dir, coverage_file)
        # ... and unlink the current file if it's identical to the other one.
        if other_reference_file != reference_file and os.path.isfile(
            other_reference_file
        ):  # pragma: no cover
            with io.open(other_reference_file, encoding=encoding) as f:
                if coverage == scrub(f.read()):
                    os.unlink(reference_file)
            break


SCRUBBERS = dict(
    txt=scrub_txt,
    xml=scrub_xml,
    html=scrub_html,
    sonarqube=scrub_xml,
    json=lambda x: x,
    json_summary=lambda x: x,
    csv=scrub_csv,
    coveralls=scrub_coveralls,
)

OUTPUT_PATTERN = dict(
    txt=["coverage*.txt"],
    xml=["coverage*.xml"],
    html=["coverage*.html", "coverage*.css"],
    sonarqube=["sonarqube*.xml"],
    json=["coverage*.json"],
    json_summary=["summary_coverage*.json"],
    csv=["coverage*.csv"],
    coveralls=["coveralls*.json"],
)

ASSERT_EQUALS = dict(xml=assert_xml_equals, sonarqube=assert_xml_equals)


def test_build(
    compiled,
    format,
    available_targets,
    generate_reference,
    update_reference,
    archive_differences,
):
    name = compiled
    scrub = SCRUBBERS[format]
    output_pattern = OUTPUT_PATTERN[format]
    assert_equals = ASSERT_EQUALS.get(format, None)

    encoding = "utf8"
    if format == "html" and name.startswith("html-encoding-"):
        encoding = re.match("^html-encoding-(.*)$", name).group(1)

    os.chdir(os.path.join(basedir, name))
    assert run(["make", format])

    if generate_reference:  # pragma: no cover
        generate_reference_data(output_pattern)

    whole_diff_output = []
    for coverage_file, reference_file in find_reference_files(output_pattern):
        with io.open(coverage_file, encoding=encoding) as f:
            coverage = scrub(f.read())
        with io.open(reference_file, encoding=encoding) as f:
            reference = scrub(f.read())

        try:
            if assert_equals is not None:
                assert_equals(reference, coverage)
            else:
                diff_out = list(
                    difflib.unified_diff(
                        reference.splitlines(keepends=True),
                        coverage.splitlines(keepends=True),
                        fromfile=reference_file,
                        tofile=coverage_file,
                    )
                )
                diff_is_empty = len(diff_out) == 0
                assert diff_is_empty, "".join(diff_out)
        except Exception as e:  # pragma: no cover
            whole_diff_output += "  " + str(e) + "\n"
            if update_reference:
                reference_file = update_reference_data(coverage_file, reference_file)
            if archive_differences:
                archive_difference_data(name, coverage_file, reference_file)

        if generate_reference or update_reference:  # pragma: no cover
            remove_duplicate_data(
                encoding, scrub, coverage, coverage_file, reference_file
            )

    diff_is_empty = len(whole_diff_output) == 0
    assert diff_is_empty, "Diff output:\n" + "".join(whole_diff_output)

    # some tests require additional cleanup after each test
    if "clean-each" in available_targets:  # pragma: no cover
        assert run(["make", "clean-each"])

    os.chdir(basedir)
