/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SRC_TRACE_PROCESSOR_PERFETTO_SQL_INTRINSICS_TABLE_FUNCTIONS_STATIC_TABLE_FUNCTION_H_
#define SRC_TRACE_PROCESSOR_PERFETTO_SQL_INTRINSICS_TABLE_FUNCTIONS_STATIC_TABLE_FUNCTION_H_

#include <cstddef>
#include <cstdint>
#include <memory>
#include <string>
#include <utility>
#include <vector>

#include "perfetto/base/status.h"
#include "perfetto/trace_processor/basic_types.h"
#include "src/trace_processor/core/dataframe/dataframe.h"
#include "src/trace_processor/core/dataframe/specs.h"

namespace perfetto::trace_processor {

// Interface which can be subclassed to allow generation of tables dynamically
// at filter time.
//
// This class is used to implement table-valued functions and other similar
// tables.
class StaticTableFunction {
 public:
  class Cursor {
   public:
    virtual ~Cursor();

    // Executes the table function with the provided arguments.
    //
    // Returns true on success, false on failure.
    virtual bool Run(const std::vector<SqlValue>& arguments) = 0;

    // Returns the dataframe that was generated by the last call to Run.
    const dataframe::Dataframe* dataframe() const { return dataframe_; }

    // Returns the status of the last call to Run.
    const base::Status& status() const { return status_; }

   protected:
    [[nodiscard]] bool OnSuccess(const dataframe::Dataframe* df) {
      dataframe_ = df;
      return true;
    }
    [[nodiscard]] bool OnFailure(base::Status status) {
      status_ = std::move(status);
      return false;
    }
    const dataframe::Dataframe* dataframe_ = nullptr;
    base::Status status_;
  };

  virtual ~StaticTableFunction();

  // Makes a cursor to repeatedly call Run with different arguments.
  virtual std::unique_ptr<Cursor> MakeCursor() = 0;

  // Returns the schema of the table that will be returned by ComputeTable.
  virtual dataframe::DataframeSpec CreateSpec() = 0;

  // Returns the name of the dynamic table.
  // This will be used to register the table with SQLite.
  virtual std::string TableName() = 0;

  // Returns the number of arguments that the table function takes.
  virtual uint32_t GetArgumentCount() const = 0;
};

}  // namespace perfetto::trace_processor

#endif  // SRC_TRACE_PROCESSOR_PERFETTO_SQL_INTRINSICS_TABLE_FUNCTIONS_STATIC_TABLE_FUNCTION_H_
