/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*-  vi:set ts=8 sts=4 sw=4: */

/*
    Sonic Visualiser
    An audio file viewer and annotation editor.
    Centre for Digital Music, Queen Mary, University of London.
    This file copyright 2006-2018 Chris Cannam and QMUL.
    
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#include "LogNumericalScale.h"
#include "LayerGeometryProvider.h"
#include "CoordinateScale.h"

#include "base/LogRange.h"
#include "base/ScaleTickIntervals.h"

#include <QPainter>

#include <cmath>

namespace sv {

int
LogNumericalScale::getWidth(LayerGeometryProvider *,
                            QPainter &paint)
{
    return paint.fontMetrics().horizontalAdvance("-000.00") + 10;
}

void
LogNumericalScale::paintVertical(LayerGeometryProvider *v,
                                 const CoordinateScale &scale,
                                 QPainter &paint,
                                 int x0)
{
    int n = 10;
    double minf = scale.getDisplayMinimum();
    double maxf = scale.getDisplayMaximum();
    auto ticks = ScaleTickIntervals::logarithmic({ minf, maxf, n });
    n = int(ticks.size());

    int w = getWidth(v, paint) + x0;

    int prevy = -1;
                
    for (int i = 0; i < n; ++i) {

        int y, ty;
        bool drawText = true;

        if (i == n-1 &&
            v->getPaintHeight() < paint.fontMetrics().height() * (n*2)) {
            if (scale.getUnit() != "") drawText = false;
        }

        double val = ticks[i].value;
        QString label = QString::fromStdString(ticks[i].label);

        y = scale.getCoordForValueRounded(v, val);

        ty = y - paint.fontMetrics().height() + paint.fontMetrics().ascent() + 2;
        
        if (prevy >= 0 && (prevy - y) < paint.fontMetrics().height()) {
            continue;
        }

        paint.drawLine(w - 5, y, w, y);

        if (drawText) {
            paint.drawText(w - paint.fontMetrics().horizontalAdvance(label) - 6,
                           ty, label);
        }

        prevy = y;
    }
}

} // end namespace sv

