/*
** 2018-03-01
**
** The author disclaims copyright to this source code.  In place of
** a legal notice, here is a blessing:
**
**    May you do good and not evil.
**    May you find forgiveness for yourself and forgive others.
**    May you share freely, never taking more than you give.
**
*************************************************************************
**
** This file implements a program used for fuzz-testing the session
** module.
**
** Usage:
**
**      sessionfuzz setup         -- Generate starter test cases
**      sessionfuzz run FILE ...  -- Run a test fuzz on FILE
**      sesssiofuzz run SQLAR ... -- Run all test cases in the SQL Archive
**
** Compiling:
**
**    (1) Have a version of SQLite that supports SQLITE_ENABLE_MEMDB
**        in the local directory.
**    (2) Run:
**
**          gcc -Wall -O3 -o sessionfuzz sessionfuzz.c -lz
**
** Use with AFL (American Fuzzy Lop - http://lcamtuf.coredump.cx/afl/)
**
**    (1) ./afl-gcc -O3 -o sessionfuzz sessionfuzz.c -lz
**    (2) mkdir session-init session-run session-cases
**    (3) cd session-init; ../sessionfuzz setup; cd ..
**    (4) ./afl -i session-init -o session-run -- ./sessionfuzz run @@
**    ... let the previous step run for a while.  Weeks, maybe.
**    (5) ./afl-cmin -i session-run -o session-cases
**
** The afl-cmin command on step (5) writes a minimal set of test cases
** for coverage into the session-cases directory.  Gather the cases written
** there into an SQL Archive using a command like this:
**
**     sqlite3 session-cases.db -Ac session-cases
**
** Then repeat the test using:
**
**     ./sessionfuzz run session-cases.db
*/

/*
** We will import the entire SQLite source file to make compiling easier
*/
#ifdef SQLITE_DEBUG
#undef SQLITE_DEBUG
#endif

#ifdef SQLITE_THREADSAFE
#undef SQLITE_THREADSAFE
#endif

#define SQLITE_DEBUG 1
#define SQLITE_THREADSAFE 0
#define SQLITE_OMIT_LOAD_EXTENSION 0
#define SQLITE_ENABLE_SESSION 1
#define SQLITE_ENABLE_PREUPDATE_HOOK 1
#define SQLITE_ENABLE_DESERIALIZE 1
#include "sqlite3.c"

/* Create a test database.  This will be an in-memory database */
static const char zInitSql[] = 
  "CREATE TABLE t1(a INTEGER PRIMARY KEY,b,c,d);\n"
  "CREATE TABLE t2(e TEXT PRIMARY KEY NOT NULL,f,g);\n"
  "CREATE TABLE t3(w REAL PRIMARY KEY NOT NULL,x,y);\n"
  "CREATE TABLE t4(z PRIMARY KEY) WITHOUT ROWID;\n"
;

/* Code to populate the database */
static const char zFillSql[] = 
  "INSERT INTO t1(a,b,c,d) VALUES\n"
  "  (1,2,3,4),\n"
  "  (2,3.5,'four',x'556677'),\n"
  "  (3,null,'xyz',15),\n"
  "  (4,'bubba',0x80000000,0.0);\n"
  "INSERT INTO t1 SELECT a+4,c,d,b FROM t1;\n"
  "INSERT INTO t1 SELECT a+8,d,b,c FROM t1;\n"
  "INSERT INTO t1 SELECT a+16,d,c,b FROM t1;\n"
  "INSERT INTO t1 SELECT a+32,b,d,c FROM t1;\n"
  "INSERT INTO t2 SELECT printf('x%dy',a),b,c FROM t1;\n"
  "INSERT INTO t3 SELECT a*1.1,b,c FROM t1;\n"
  "INSERT INTO t4 SELECT a||','||quote(b) FROM t1;\n"
;

/* A database file created by running the two scripts above */
static const unsigned char aDbBytes[] = {
  83, 81, 76,105,116,101, 32,102,111,114,109, 97,116, 32, 51,  0,  2,  0,  1,
   1,  0, 64, 32, 32,  0,  0,  0, 13,  0,  0,  0, 22,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  5,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  13,  0, 46, 32,152, 13,  1,186,  0,  6,  0,176,  0,  1,194,  1, 84,  1,150,
   0,238,  1, 48,  0,176,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0, 60,  6,  6, 23, 17, 17,  1,101,116, 97, 98,108,101,116,
  52,116, 52,  7, 67, 82, 69, 65, 84, 69, 32, 84, 65, 66, 76, 69, 32,116, 52,
  40,122, 32, 80, 82, 73, 77, 65, 82, 89, 32, 75, 69, 89, 41, 32, 87, 73, 84,
  72, 79, 85, 84, 32, 82, 79, 87, 73, 68, 64,  4,  6, 23, 17, 17,  1,109,116,
  97, 98,108,101,116, 51,116, 51,  5, 67, 82, 69, 65, 84, 69, 32, 84, 65, 66,
  76, 69, 32,116, 51, 40,119, 32, 82, 69, 65, 76, 32, 80, 82, 73, 77, 65, 82,
  89, 32, 75, 69, 89, 32, 78, 79, 84, 32, 78, 85, 76, 76, 44,120, 44,121, 41,
  34,  5,  5, 23, 55, 17,  1,105,110,100,101,120,115,113,108,105,116,101, 95,
  97,117,116,111,105,110,100,101,120, 95,116, 51, 95, 49,116, 51,  6, 64,  2,
   6, 23, 17, 17,  1,109,116, 97, 98,108,101,116, 50,116, 50,  3, 67, 82, 69,
  65, 84, 69, 32, 84, 65, 66, 76, 69, 32,116, 50, 40,101, 32, 84, 69, 88, 84,
  32, 80, 82, 73, 77, 65, 82, 89, 32, 75, 69, 89, 32, 78, 79, 84, 32, 78, 85,
  76, 76, 44,102, 44,103, 41, 34,  3,  5, 23, 55, 17,  1,105,110,100,101,120,
 115,113,108,105,116,101, 95, 97,117,116,111,105,110,100,101,120, 95,116, 50,
  95, 49,116, 50,  4,  0,  0,  0,  8,  0,  0,  0,  0, 60,  1,  6, 23, 17, 17,
   1,101,116, 97, 98,108,101,116, 49,116, 49,  2, 67, 82, 69, 65, 84, 69, 32,
  84, 65, 66, 76, 69, 32,116, 49, 40, 97, 32, 73, 78, 84, 69, 71, 69, 82, 32,
  80, 82, 73, 77, 65, 82, 89, 32, 75, 69, 89, 44, 98, 44, 99, 44,100, 41,  5,
   0,  0,  0,  2,  1,246,  0,  0,  0,  0, 10,  1,251,  1,246,  1,177,  1,155,
   1,145,  1,119,  1,109,  1, 87,  1, 76,  1, 50,  1, 40,  1, 18,  1,  7,  0,
 237,  0,227,  0,205,  0,195,  0,169,  0,159,  0,137,  0,126,  0,100,  0, 90,
   0, 68,  0,  0,  0,  0,  0,  0,  0,  0, 20, 26,  5,  0, 21,  7, 18,102,111,
 117,114, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,119,  8, 25,  5,  0,  1,  1,
   1,  3,  2,  4, 24, 24,  5,  0, 23,  7,  5, 98,117, 98, 98, 97,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  9, 23,  5,  0,  0,  1, 19, 15,
 120,121,122, 20, 22,  5,  0,  7, 18, 21, 64, 12,  0,  0,  0,  0,  0,  0, 85,
 102,119,102,111,117,114,  8, 21,  5,  0,  1,  1,  1,  2,  4,  3, 24, 20,  5,
   0,  7,  5, 23,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0, 98,
 117, 98, 98, 97,  8, 19,  4,  0,  1, 19, 15,120,121,122, 20, 18,  5,  0, 18,
  21,  7, 85,102,119,102,111,117,114, 64, 12,  0,  0,  0,  0,  0,  0,  8, 17,
   5,  0,  1,  1,  1,  4,  3,  2, 24, 16,  5,  0, 23,  5,  7, 98,117, 98, 98,
  97,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  9, 15,  5,  0,
   0, 19,  1,120,121,122, 15, 20, 14,  5,  0,  7, 21, 18, 64, 12,  0,  0,  0,
   0,  0,  0,102,111,117,114, 85,102,119,  8, 13,  5,  0,  1,  1,  1,  2,  3,
   4, 24, 12,  5,  0,  7, 23,  5,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98,
  98, 97,  0,  0,128,  0,  0,  0,  9, 11,  5,  0,  1,  0, 19, 15,120,121,122,
  20, 10,  5,  0, 18,  7, 21, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,102,
 111,117,114,  8,  9,  5,  0,  1,  1,  1,  4,  2,  3, 24,  8,  5,  0,  5,  7,
  23,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98,
  97,  8,  7,  4,  0, 19,  1,120,121,122, 15, 20,  6,  5,  0, 21, 18,  7,102,
 111,117,114, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,  8,  5,  5,  0,  1,
   1,  1,  3,  4,  2, 24,  4,  5,  0, 23,  5,  7, 98,117, 98, 98, 97,  0,  0,
 128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  9,  3,  5,  0,  0, 19,  1,
 120,121,122, 15, 20,  2,  5,  0,  7, 21, 18, 64, 12,  0,  0,  0,  0,  0,  0,
 102,111,117,114, 85,102,119,  0,  0,  0,  9, 52,  0,  0,  0,  8, 26,  5,  0,
   0,  0,  2,  1,246,  0,  0,  0,  0, 13,  1,251,  1,246,  1,181,  1,165,  1,
 152,  1,129,  1,118,  1, 97,  1, 87,  1, 64,  1, 52,  1, 30,  1, 17,  0,252,
   0,240,  0,223,  0,209,  0,185,  0,173,  0,152,  0,141,  0,118,  0,106,  0,
  84,  0, 72,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 10, 27,  3, 21, 19,120,
  50, 55,121,120,121,122, 20, 26,  4, 21, 21,  7,120, 50, 54,121,102,111,117,
 114, 64, 12,  0,  0,  0,  0,  0,  0, 10, 25,  4, 21,  1,  1,120, 50, 53,121,
   3,  2, 21, 24,  4, 21, 23,  7,120, 50, 52,121, 98,117, 98, 98, 97,  0,  0,
   0,  0,  0,  0,  0,  0,  9, 23,  4, 21,  0,  1,120, 50, 51,121, 15, 19, 22,
   4, 21,  7, 18,120, 50, 50,121, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,119,
  10, 21,  4, 21,  1,  1,120, 50, 49,121,  2,  4, 22, 20,  4, 21,  7,  5,120,
  50, 48,121,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0, 12, 19,
   4, 21,  1, 19,120, 49, 57,121, 15,120,121,122, 15, 18,  4, 21, 18, 21,120,
  49, 56,121, 85,102,119,102,111,117,114, 10, 17,  4, 21,  1,  1,120, 49, 55,
 121,  4,  3, 19, 16,  4, 21, 23,  5,120, 49, 54,121, 98,117, 98, 98, 97,  0,
   0,128,  0,  0,  0, 11, 15,  4, 21,  0, 19,120, 49, 53,121,120,121,122, 20,
  14,  4, 21,  7, 21,120, 49, 52,121, 64, 12,  0,  0,  0,  0,  0,  0,102,111,
 117,114, 10, 13,  4, 21,  1,  1,120, 49, 51,121,  2,  3, 21, 12,  4, 21,  7,
  23,120, 49, 50,121,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  8,
  11,  3, 21,  1,120, 49, 49,121, 15, 19, 10,  4, 21, 18,  7,120, 49, 48,121,
  85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,  9,  9,  4, 19,  1,  1,120, 57,
 121,  4,  2, 21,  8,  4, 19,  5,  7,120, 56,121,  0,  0,128,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0, 11,  7,  4, 19, 19,  1,120, 55,121,120,121,122,
  15, 14,  6,  4, 19, 21, 18,120, 54,121,102,111,117,114, 85,102,119,  9,  5,
   4, 19,  1,  1,120, 53,121,  3,  4, 18,  4,  4, 19, 23,  5,120, 52,121, 98,
 117, 98, 98, 97,  0,  0,128,  0,  0,  0, 10,  3,  4, 19,  0, 19,120, 51,121,
 120,121,122, 19,  2,  4, 19,  7, 21,120, 50,121, 64, 12,  0,  0,  0,  0,  0,
   0,102,111,117,114,  9,  0,  0,  0, 12, 53,  0,  0,  0, 11, 27,  2,  0,  0,
   0,  1,  1,243,  0,  0,  0,  0, 15,  1,243,  1,220,  1,211,  1,202,  1,193,
   1,184,  1,175,  1,166,  1,159,  1,150,  1,141,  1,132,  1,123,  1,114,  1,
 105,  1, 96,  1, 87,  1, 78,  1, 69,  1, 61,  1, 52,  1, 43,  1, 34,  1, 25,
   1, 16,  1,  7,  0,254,  0,245,  0,236,  0,227,  0,219,  0,210,  0,201,  0,
 192,  0,183,  0,174,  0,165,  0,156,  0,147,  0,138,  0,129,  0,121,  0,112,
   0,103,  0,  0,  0,  8,  3, 21,  1,120, 53, 49,121, 51,  8,  3, 21,  1,120,
  53, 48,121, 50,  7,  3, 19,  1,120, 52,121,  4,  8,  3, 21,  1,120, 52, 57,
 121, 49,  8,  3, 21,  1,120, 52, 56,121, 48,  8,  3, 21,  1,120, 52, 55,121,
  47,  8,  3, 21,  1,120, 52, 54,121, 46,  8,  3, 21,  1,120, 52, 53,121, 45,
   8,  3, 21,  1,120, 52, 52,121, 44,  8,  3, 21,  1,120, 52, 51,121, 43,  8,
   3, 21,  1,120, 52, 50,121, 42,  8,  3, 21,  1,120, 52, 49,121, 41,  8,  3,
  21,  1,120, 52, 48,121, 40,  7,  3, 19,  1,120, 51,121,  3,  8,  3, 21,  1,
 120, 51, 57,121, 39,  8,  3, 21,  1,120, 51, 56,121, 38,  8,  3, 21,  1,120,
  51, 55,121, 37,  8,  3, 21,  1,120, 51, 54,121, 36,  8,  3, 21,  1,120, 51,
  53,121, 35,  8,  3, 21,  1,120, 51, 52,121, 34,  8,  3, 21,  1,120, 51, 51,
 121, 33,  8,  3, 21,  1,120, 51, 50,121, 32,  8,  3, 21,  1,120, 51, 49,121,
  31,  8,  3, 21,  1,120, 51, 48,121, 30,  7,  3, 19,  1,120, 50,121,  2,  8,
   3, 21,  1,120, 50, 57,121, 29,  8,  3, 21,  1,120, 50, 56,121, 28,  8,  3,
  21,  1,120, 50, 55,121, 27,  8,  3, 21,  1,120, 50, 54,121, 26,  8,  3, 21,
   1,120, 50, 53,121, 25,  8,  3, 21,  1,120, 50, 52,121, 24,  8,  3, 21,  1,
 120, 50, 51,121, 23,  8,  3, 21,  1,120, 50, 50,121, 22,  8,  3, 21,  1,120,
  50, 49,121, 21,  8,  3, 21,  1,120, 50, 48,121, 20,  6,  3, 19,  9,120, 49,
 121,  8,  3, 21,  1,120, 49, 57,121, 19,  8,  3, 21,  1,120, 49, 56,121, 18,
   8,  3, 21,  1,120, 49, 55,121, 17,  8,  3, 21,  1,120, 49, 54,121, 16,  8,
   3, 21,  1,120, 49, 53,121, 15,  8,  3, 21,  1,120, 49, 52,121, 14,  8,  3,
  21,  1,120, 49, 51,121, 13,  8,  3, 21,  1,120, 49, 50,121, 12,  8,  3, 21,
   1,120,  0,  0,  0, 14,  8,  3, 21,  1,120, 53, 49,121, 51,  5,  0,  0,  0,
   2,  1,246,  0,  0,  0,  0, 18,  1,251,  1,246,  1,156,  1,135,  1,117,  1,
  89,  1, 73,  1, 55,  1, 41,  1, 14,  0,254,  0,228,  0,211,  0,186,  0,170,
   0,149,  0,131,  0,110,  0, 94,  0, 69,  0, 54, 13, 23,  4,  7,  0,  1, 64,
  57, 76,204,204,204,204,205, 15, 23, 22,  4,  7,  7, 18, 64, 56, 51, 51, 51,
  51, 51, 52, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,119, 14, 21,  4,  7,  1,
   1, 64, 55, 25,153,153,153,153,154,  2,  4, 19, 20,  4,  1,  7,  5, 22,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0, 16, 19,  4,  7,  1, 19,
  64, 52,230,102,102,102,102,103, 15,120,121,122, 19, 18,  4,  7, 18, 21, 64,
  51,204,204,204,204,204,205, 85,102,119,102,111,117,114, 14, 17,  4,  7,  1,
   1, 64, 50,179, 51, 51, 51, 51, 52,  4,  3, 23, 16,  4,  7, 23,  5, 64, 49,
 153,153,153,153,153,154, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0, 15, 15,
   4,  7,  0, 19, 64, 48,128,  0,  0,  0,  0,  0,120,121,122, 24, 14,  4,  7,
   7, 21, 64, 46,204,204,204,204,204,206, 64, 12,  0,  0,  0,  0,  0,  0,102,
 111,117,114, 14, 13,  4,  7,  1,  1, 64, 44,153,153,153,153,153,154,  2,  3,
  25, 12,  4,  7,  7, 23, 64, 42,102,102,102,102,102,103,  0,  0,  0,  0,  0,
   0,  0,  0, 98,117, 98, 98, 97, 12, 11,  3,  7,  1, 64, 40, 51, 51, 51, 51,
  51, 52, 15, 16, 10,  4,  1, 18,  7, 11, 85,102,119, 64, 12,  0,  0,  0,  0,
   0,  0, 14,  9,  4,  7,  1,  1, 64, 35,204,204,204,204,204,205,  4,  2, 26,
   8,  4,  7,  5,  7, 64, 33,153,153,153,153,153,154,  0,  0,128,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0, 16,  7,  4,  7, 19,  1, 64, 30,204,204,204,
 204,204,206,120,121,122, 15, 19,  6,  4,  7, 21, 18, 64, 26,102,102,102,102,
 102,103,102,111,117,114, 85,102,119, 14,  5,  4,  7,  1,  1, 64, 22,  0,  0,
   0,  0,  0,  0,  3,  4, 23,  4,  4,  7, 23,  5, 64, 17,153,153,153,153,153,
 154, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0, 15,  3,  4,  7,  0, 19, 64,
  10,102,102,102,102,102,103,120,121,122, 24,  2,  4,  7,  7, 21, 64,  1,153,
 153,153,153,153,154, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114, 14,  1,
   4,  7,  1,  1,  0,  0,  0, 17, 45,  0,  0,  0, 16, 23,  2,  0,  0,  0,  1,
   1,239,  0,  0,  0,  0, 20,  1,239,  1,205,  1,192,  1,179,  1,166,  1,153,
   1,140,  1,134,  1,121,  1,108,  1, 95,  1, 82,  1, 69,  1, 56,  1, 43,  1,
  30,  1, 17,  1, 11,  0,254,  0,241,  0,228,  0,215,  0,202,  0,189,  0,176,
   0,163,  0,150,  0,144,  0,131,  0,118,  0,105,  0, 92,  0, 79,  0, 12,  3,
   7,  1, 64, 67, 64,  0,  0,  0,  0,  0, 35, 12,  3,  7,  1, 64, 66,179, 51,
  51, 51, 51, 52, 34, 12,  3,  7,  1, 64, 66, 38,102,102,102,102,103, 33, 12,
   3,  7,  1, 64, 65,153,153,153,153,153,154, 32, 12,  3,  7,  1, 64, 65, 12,
 204,204,204,204,205, 31,  5,  3,  1,  1, 33, 30, 12,  3,  7,  1, 64, 63,230,
 102,102,102,102,103, 29, 12,  3,  7,  1, 64, 62,204,204,204,204,204,206, 28,
  12,  3,  7,  1, 64, 61,179, 51, 51, 51, 51, 52, 27, 12,  3,  7,  1, 64, 60,
 153,153,153,153,153,154, 26, 12,  3,  7,  1, 64, 59,128,  0,  0,  0,  0,  1,
  25, 12,  3,  7,  1, 64, 58,102,102,102,102,102,103, 24, 12,  3,  7,  1, 64,
  57, 76,204,204,204,204,205, 23, 12,  3,  7,  1, 64, 56, 51, 51, 51, 51, 51,
  52, 22, 12,  3,  7,  1, 64, 55, 25,153,153,153,153,154, 21,  5,  3,  1,  1,
  22, 20, 12,  3,  7,  1, 64, 52,230,102,102,102,102,103, 19, 12,  3,  7,  1,
  64, 51,204,204,204,204,204,205, 18, 12,  3,  7,  1, 64, 50,179, 51, 51, 51,
  51, 52, 17, 12,  3,  7,  1, 64, 49,153,153,153,153,153,154, 16, 12,  3,  7,
   1, 64, 48,128,  0,  0,  0,  0,  0, 15, 12,  3,  7,  1, 64, 46,204,204,204,
 204,204,206, 14, 12,  3,  7,  1, 64, 44,153,153,153,153,153,154, 13, 12,  3,
   7,  1, 64, 42,102,102,102,102,102,103, 12, 12,  3,  7,  1, 64, 40, 51, 51,
  51, 51, 51, 52, 11,  5,  3,  1,  1, 11, 10, 12,  3,  7,  1, 64, 35,204,204,
 204,204,204,205,  9, 12,  3,  7,  1, 64, 33,153,153,153,153,153,154,  8, 12,
   3,  7,  1, 64, 30,204,204,204,204,204,206,  7, 12,  3,  7,  1, 64, 26,102,
 102,102,102,102,103,  6, 12,  3,  7,  1, 64, 22,  0,  0,  0,  0,  0,  0,  5,
  12,  3,  7,  1, 64, 17,153,153,153,153,153,154,  4, 12,  3,  7,  1, 64, 10,
 102,102,102,102,102,103,  3, 12,  3,  7,  1, 64,  1,153,153,  0,  0,  0, 19,
  12,  3,  7,  1, 64, 67, 64,  0,  0,  0,  0,  0, 35,  2,  0,  0,  0,  1,  1,
 242,  0,  0,  0,  0, 22,  1,242,  1,218,  1,211,  1,202,  1,192,  1,179,  1,
 172,  1,157,  1,149,  1,141,  1,132,  1,125,  1,116,  1,106,  1, 93,  1, 86,
   1, 74,  1, 63,  1, 47,  1, 40,  1, 31,  1, 16,  1,  8,  0,255,  0,248,  0,
 239,  0,229,  0,216,  0,209,  0,197,  0,186,  0,174,  0,158,  0,151,  0,136,
   0,128,  0,119,  0,112,  0,103,  0, 93,  0,  9,  2, 27, 52, 55, 44, 78, 85,
  76, 76,  8,  2, 25, 52, 54, 44, 51, 46, 53,  6,  2, 21, 52, 53, 44, 50,  8,
   2, 25, 52, 52, 44, 48, 46, 48,  7,  2, 23, 52, 51, 44, 49, 53, 14,  2, 37,
  52, 50, 44, 88, 39, 53, 53, 54, 54, 55, 55, 39,  6,  2, 21, 52, 49, 44, 52,
  15,  2, 39, 52, 48, 44, 50, 49, 52, 55, 52, 56, 51, 54, 52, 56, 11,  2, 31,
  52, 44, 39, 98,117, 98, 98, 97, 39, 10,  2, 29, 51, 57, 44, 39,120,121,122,
  39, 11,  2, 31, 51, 56, 44, 39,102,111,117,114, 39,  6,  2, 21, 51, 55, 44,
  51, 12,  2, 33, 51, 54, 44, 39, 98,117, 98, 98, 97, 39,  9,  2, 27, 51, 53,
  44, 78, 85, 76, 76,  8,  2, 25, 51, 52, 44, 51, 46, 53,  6,  2, 21, 51, 51,
  44, 50,  8,  2, 25, 51, 50, 44, 48, 46, 48,  7,  2, 23, 51, 49, 44, 49, 53,
  14,  2, 37, 51, 48, 44, 88, 39, 53, 53, 54, 54, 55, 55, 39,  8,  2, 25, 51,
  44, 78, 85, 76, 76,  6,  2, 21, 50, 57, 44, 52, 15,  2, 39, 50, 56, 44, 50,
  49, 52, 55, 52, 56, 51, 54, 52, 56, 10,  2, 29, 50, 55, 44, 39,120,121,122,
  39, 11,  2, 31, 50, 54, 44, 39,102,111,117,114, 39,  6,  2, 21, 50, 53, 44,
  51, 12,  2, 33, 50, 52, 44, 39, 98,117, 98, 98, 97, 39,  9,  2, 27, 50, 51,
  44, 78, 85, 76, 76,  8,  2, 25, 50, 50, 44, 51, 46, 53,  6,  2, 21, 50, 49,
  44, 50,  8,  2, 25, 50, 48, 44, 48, 46, 48,  7,  2, 23, 50, 44, 51, 46, 53,
   7,  2, 23, 49, 57, 44, 49, 53, 14,  2, 37, 49, 56, 44, 88, 39, 53, 53, 54,
  54, 55, 55, 39,  6,  2, 21, 49, 55, 44, 52, 12,  2, 33, 49, 54, 44, 39, 98,
 117, 98, 98, 97, 39,  9,  2, 27, 49, 53, 44, 78, 85, 76, 76,  8,  2, 25, 49,
  52, 44, 51, 46, 53,  6,  2, 21, 49, 51, 44, 50,  8,  2, 25, 49, 50, 44, 48,
  46, 48,  7,  2, 23, 49, 49, 44, 49, 53, 14,  2, 37, 49, 48, 44, 88,  0,  0,
   0, 21,  9,  2, 27, 52, 55, 44, 78, 85, 76, 76, 13,  0,  0,  0, 26,  0, 68,
   0,  1,246,  1,224,  1,213,  1,187,  1,177,  1,155,  1,145,  1,119,  1,109,
   1, 87,  1, 76,  1, 50,  1, 40,  1, 18,  1,  7,  0,237,  0,227,  0,205,  0,
 195,  0,169,  0,159,  0,137,  0,126,  0,100,  0, 90,  0, 68,  0,  0,  0,  0,
   0,  0,  0,  0, 20, 26,  5,  0, 21,  7, 18,102,111,117,114, 64, 12,  0,  0,
   0,  0,  0,  0, 85,102,119,  8, 25,  5,  0,  1,  1,  1,  3,  2,  4, 24, 24,
   5,  0, 23,  7,  5, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,128,  0,  0,  0,  9, 23,  5,  0,  0,  1, 19, 15,120,121,122, 20, 22,  5,
   0,  7, 18, 21, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,119,102,111,117,114,
   8, 21,  5,  0,  1,  1,  1,  2,  4,  3, 24, 20,  5,  0,  7,  5, 23,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0, 98,117, 98, 98, 97,  8, 19,
   4,  0,  1, 19, 15,120,121,122, 20, 18,  5,  0, 18, 21,  7, 85,102,119,102,
 111,117,114, 64, 12,  0,  0,  0,  0,  0,  0,  8, 17,  5,  0,  1,  1,  1,  4,
   3,  2, 24, 16,  5,  0, 23,  5,  7, 98,117, 98, 98, 97,  0,  0,128,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  9, 15,  5,  0,  0, 19,  1,120,121,122,
  15, 20, 14,  5,  0,  7, 21, 18, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,
 114, 85,102,119,  8, 13,  5,  0,  1,  1,  1,  2,  3,  4, 24, 12,  5,  0,  7,
  23,  5,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  0,  0,128,  0,
   0,  0,  9, 11,  5,  0,  1,  0, 19, 15,120,121,122, 20, 10,  5,  0, 18,  7,
  21, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114,  8,  9,  5,
   0,  1,  1,  1,  4,  2,  3, 24,  8,  5,  0,  5,  7, 23,  0,  0,128,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  8,  7,  4,  0, 19,
   1,120,121,122, 15, 20,  6,  5,  0, 21, 18,  7,102,111,117,114, 85,102,119,
  64, 12,  0,  0,  0,  0,  0,  0,  8,  5,  5,  0,  1,  1,  1,  3,  4,  2, 24,
   4,  5,  0, 23,  5,  7, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  9,  3,  5,  0,  0, 19,  1,120,121,122, 15, 20,  2,
   5,  0,  7, 21, 18, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114, 85,102,
 119,  8,  1,  5,  0,  1,  1,  1,  2,  3,  4, 13,  0,  0,  0, 26,  0, 63,  0,
   1,245,  1,219,  1,209,  1,187,  1,177,  1,151,  1,141,  1,119,  1,108,  1,
  82,  1, 72,  1, 50,  1, 39,  1, 13,  1,  3,  0,237,  0,227,  0,201,  0,191,
   0,169,  0,158,  0,132,  0,122,  0,100,  0, 89,  0, 63,  0,  0,  0, 24, 52,
   5,  0,  7, 23,  5,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  0,
   0,128,  0,  0,  0,  9, 51,  5,  0,  1,  0, 19, 15,120,121,122, 20, 50,  5,
   0, 18,  7, 21, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114,
   8, 49,  5,  0,  1,  1,  1,  4,  2,  3, 24, 48,  5,  0, 23,  7,  5, 98,117,
  98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  9, 47,
   5,  0,  0,  1, 19, 15,120,121,122, 20, 46,  5,  0,  7, 18, 21, 64, 12,  0,
   0,  0,  0,  0,  0, 85,102,119,102,111,117,114,  8, 45,  5,  0,  1,  1,  1,
   2,  4,  3, 24, 44,  5,  0,  7,  5, 23,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,128,  0,  0,  0, 98,117, 98, 98, 97,  8, 43,  4,  0,  1, 19, 15,120,121,
 122, 20, 42,  5,  0, 18, 21,  7, 85,102,119,102,111,117,114, 64, 12,  0,  0,
   0,  0,  0,  0,  8, 41,  5,  0,  1,  1,  1,  4,  3,  2, 24, 40,  5,  0,  5,
  23,  7,  0,  0,128,  0,  0,  0, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,
   0,  0,  9, 39,  5,  0, 19,  0,  1,120,121,122, 15, 20, 38,  5,  0, 21,  7,
  18,102,111,117,114, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,119,  8, 37,  5,
   0,  1,  1,  1,  3,  2,  4, 24, 36,  5,  0, 23,  7,  5, 98,117, 98, 98, 97,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,  9, 35,  5,  0,  0,
   1, 19, 15,120,121,122, 20, 34,  5,  0,  7, 18, 21, 64, 12,  0,  0,  0,  0,
   0,  0, 85,102,119,102,111,117,114,  8, 33,  5,  0,  1,  1,  1,  2,  4,  3,
  24, 32,  5,  0,  7,  5, 23,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,
   0,  0, 98,117, 98, 98, 97,  8, 31,  4,  0,  1, 19, 15,120,121,122, 20, 30,
   5,  0, 18, 21,  7, 85,102,119,102,111,117,114, 64, 12,  0,  0,  0,  0,  0,
   0,  8, 29,  5,  0,  1,  1,  1,  4,  3,  2, 24, 28,  5,  0,  5, 23,  7,  0,
   0,128,  0,  0,  0, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0,  9,
  27,  5,  0, 19,  0,  1,120,121,122, 15, 13,  0,  0,  0, 12,  1, 50,  0,  1,
 246,  1,224,  1,213,  1,187,  1,177,  1,155,  1,145,  1,119,  1,109,  1, 87,
   1, 76,  1, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 24, 64,  5,  0,  7, 23,  5,
   0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0,
   9, 63,  5,  0,  1,  0, 19, 15,120,121,122, 20, 62,  5,  0, 18,  7, 21, 85,
 102,119, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114,  8, 61,  5,  0,  1,
   1,  1,  4,  2,  3, 24, 60,  5,  0,  5,  7, 23,  0,  0,128,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  8, 59,  4,  0, 19,  1,120,
 121,122, 15, 20, 58,  5,  0, 21, 18,  7,102,111,117,114, 85,102,119, 64, 12,
   0,  0,  0,  0,  0,  0,  8, 57,  5,  0,  1,  1,  1,  3,  4,  2, 24, 56,  5,
   0, 23,  5,  7, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  9, 55,  5,  0,  0, 19,  1,120,121,122, 15, 20, 54,  5,  0,
   7, 21, 18, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114, 85,102,119,  8,
  53,  5,  0,  1,  1,  1,  2,  3,  4, 13,  0,  0,  0, 27,  0, 72,  0,  1,245,
   1,224,  1,212,  1,192,  1,181,  1,165,  1,152,  1,129,  1,118,  1, 97,  1,
  87,  1, 64,  1, 52,  1, 30,  1, 17,  0,252,  0,240,  0,223,  0,209,  0,185,
   0,173,  0,152,  0,141,  0,118,  0,106,  0, 84,  0, 72,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0, 10, 27,  3, 21, 19,120, 50, 55,121,120,121,122, 20, 26,
   4, 21, 21,  7,120, 50, 54,121,102,111,117,114, 64, 12,  0,  0,  0,  0,  0,
   0, 10, 25,  4, 21,  1,  1,120, 50, 53,121,  3,  2, 21, 24,  4, 21, 23,  7,
 120, 50, 52,121, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0,  9, 23,
   4, 21,  0,  1,120, 50, 51,121, 15, 19, 22,  4, 21,  7, 18,120, 50, 50,121,
  64, 12,  0,  0,  0,  0,  0,  0, 85,102,119, 10, 21,  4, 21,  1,  1,120, 50,
  49,121,  2,  4, 22, 20,  4, 21,  7,  5,120, 50, 48,121,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,128,  0,  0,  0, 12, 19,  4, 21,  1, 19,120, 49, 57,121,
  15,120,121,122, 15, 18,  4, 21, 18, 21,120, 49, 56,121, 85,102,119,102,111,
 117,114, 10, 17,  4, 21,  1,  1,120, 49, 55,121,  4,  3, 19, 16,  4, 21, 23,
   5,120, 49, 54,121, 98,117, 98, 98, 97,  0,  0,128,  0,  0,  0, 11, 15,  4,
  21,  0, 19,120, 49, 53,121,120,121,122, 20, 14,  4, 21,  7, 21,120, 49, 52,
 121, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114, 10, 13,  4, 21,  1,  1,
 120, 49, 51,121,  2,  3, 21, 12,  4, 21,  7, 23,120, 49, 50,121,  0,  0,  0,
   0,  0,  0,  0,  0, 98,117, 98, 98, 97,  8, 11,  3, 21,  1,120, 49, 49,121,
  15, 19, 10,  4, 21, 18,  7,120, 49, 48,121, 85,102,119, 64, 12,  0,  0,  0,
   0,  0,  0,  9,  9,  4, 19,  1,  1,120, 57,121,  4,  2, 21,  8,  4, 19,  5,
   7,120, 56,121,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 11,
   7,  4, 19, 19,  1,120, 55,121,120,121,122, 15, 14,  6,  4, 19, 21, 18,120,
  54,121,102,111,117,114, 85,102,119,  9,  5,  4, 19,  1,  1,120, 53,121,  3,
   4, 18,  4,  4, 19, 23,  5,120, 52,121, 98,117, 98, 98, 97,  0,  0,128,  0,
   0,  0, 10,  3,  4, 19,  0, 19,120, 51,121,120,121,122, 19,  2,  4, 19,  7,
  21,120, 50,121, 64, 12,  0,  0,  0,  0,  0,  0,102,111,117,114,  9,  1,  4,
  19,  1,  1,120, 49,121,  2,  3, 13,  0,  0,  0, 26,  0, 78,  0,  1,235,  1,
 223,  1,206,  1,192,  1,168,  1,156,  1,135,  1,124,  1,101,  1, 89,  1, 67,
   1, 55,  1, 34,  1, 22,  1,  5,  0,247,  0,223,  0,211,  0,190,  0,179,  0,
 156,  0,144,  0,123,  0,113,  0, 90,  0, 78,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 10, 53,  4, 21,  1,  1,120, 53, 51,
 121,  2,  3, 21, 52,  4, 21,  7, 23,120, 53, 50,121,  0,  0,  0,  0,  0,  0,
   0,  0, 98,117, 98, 98, 97,  8, 51,  3, 21,  1,120, 53, 49,121, 15, 19, 50,
   4, 21, 18,  7,120, 53, 48,121, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0,
  10, 49,  4, 21,  1,  1,120, 52, 57,121,  4,  2, 21, 48,  4, 21, 23,  7,120,
  52, 56,121, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0,  9, 47,  4,
  21,  0,  1,120, 52, 55,121, 15, 19, 46,  4, 21,  7, 18,120, 52, 54,121, 64,
  12,  0,  0,  0,  0,  0,  0, 85,102,119, 10, 45,  4, 21,  1,  1,120, 52, 53,
 121,  2,  4, 22, 44,  4, 21,  7,  5,120, 52, 52,121,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,128,  0,  0,  0, 12, 43,  4, 21,  1, 19,120, 52, 51,121, 15,
 120,121,122, 15, 42,  4, 21, 18, 21,120, 52, 50,121, 85,102,119,102,111,117,
 114, 10, 41,  4, 21,  1,  1,120, 52, 49,121,  4,  3, 19, 40,  4, 21,  5, 23,
 120, 52, 48,121,  0,  0,128,  0,  0,  0, 98,117, 98, 98, 97, 10, 39,  3, 21,
  19,120, 51, 57,121,120,121,122, 20, 38,  4, 21, 21,  7,120, 51, 56,121,102,
 111,117,114, 64, 12,  0,  0,  0,  0,  0,  0, 10, 37,  4, 21,  1,  1,120, 51,
  55,121,  3,  2, 21, 36,  4, 21, 23,  7,120, 51, 54,121, 98,117, 98, 98, 97,
   0,  0,  0,  0,  0,  0,  0,  0,  9, 35,  4, 21,  0,  1,120, 51, 53,121, 15,
  19, 34,  4, 21,  7, 18,120, 51, 52,121, 64, 12,  0,  0,  0,  0,  0,  0, 85,
 102,119, 10, 33,  4, 21,  1,  1,120, 51, 51,121,  2,  4, 22, 32,  4, 21,  7,
   5,120, 51, 50,121,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0,
  12, 31,  4, 21,  1, 19,120, 51, 49,121, 15,120,121,122, 15, 30,  4, 21, 18,
  21,120, 51, 48,121, 85,102,119,102,111,117,114, 10, 29,  4, 21,  1,  1,120,
  50, 57,121,  4,  3, 19, 28,  4, 21,  5, 23,120, 50, 56,121,  0,  0,128,  0,
   0,  0, 98,117, 98, 98, 97, 13,  0,  0,  0, 11,  1, 67,  0,  1,234,  1,221,
   1,200,  1,188,  1,171,  1,157,  1,133,  1,121,  1,100,  1, 90,  1, 67,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0, 21, 64,  4, 21,  7, 23,120, 54, 52,121,  0,  0,
   0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97,  8, 63,  3, 21,  1,120, 54, 51,
 121, 15, 19, 62,  4, 21, 18,  7,120, 54, 50,121, 85,102,119, 64, 12,  0,  0,
   0,  0,  0,  0, 10, 61,  4, 21,  1,  1,120, 54, 49,121,  4,  2, 22, 60,  4,
  21,  5,  7,120, 54, 48,121,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0, 12, 59,  4, 21, 19,  1,120, 53, 57,121,120,121,122, 15, 15, 58,  4,
  21, 21, 18,120, 53, 56,121,102,111,117,114, 85,102,119, 10, 57,  4, 21,  1,
   1,120, 53, 55,121,  3,  4, 19, 56,  4, 21, 23,  5,120, 53, 54,121, 98,117,
  98, 98, 97,  0,  0,128,  0,  0,  0, 11, 55,  4, 21,  0, 19,120, 53, 53,121,
 120,121,122, 20, 54,  4, 21,  7, 21,120, 53, 52,121, 64, 12,  0,  0,  0,  0,
   0,  0,102,111,117,114, 10,  0,  0,  0, 45,  0,112,  0,  1,247,  1,238,  1,
 229,  1,220,  1,211,  1,202,  1,193,  1,184,  1,175,  1,166,  1,159,  1,150,
   1,141,  1,132,  1,123,  1,114,  1,105,  1, 96,  1, 87,  1, 78,  1, 69,  1,
  61,  1, 52,  1, 43,  1, 34,  1, 25,  1, 16,  1,  7,  0,254,  0,245,  0,236,
   0,227,  0,219,  0,210,  0,201,  0,192,  0,183,  0,174,  0,165,  0,156,  0,
 147,  0,138,  0,129,  0,121,  0,112,  0,103,  0,  0,  0,  0,  0,  0,  9,120,
  53, 49,121, 51,  8,  3, 21,  1,120, 53, 48,121, 50,  7,  3, 19,  1,120, 52,
 121,  4,  8,  3, 21,  1,120, 52, 57,121, 49,  8,  3, 21,  1,120, 52, 56,121,
  48,  8,  3, 21,  1,120, 52, 55,121, 47,  8,  3, 21,  1,120, 52, 54,121, 46,
   8,  3, 21,  1,120, 52, 53,121, 45,  8,  3, 21,  1,120, 52, 52,121, 44,  8,
   3, 21,  1,120, 52, 51,121, 43,  8,  3, 21,  1,120, 52, 50,121, 42,  8,  3,
  21,  1,120, 52, 49,121, 41,  8,  3, 21,  1,120, 52, 48,121, 40,  7,  3, 19,
   1,120, 51,121,  3,  8,  3, 21,  1,120, 51, 57,121, 39,  8,  3, 21,  1,120,
  51, 56,121, 38,  8,  3, 21,  1,120, 51, 55,121, 37,  8,  3, 21,  1,120, 51,
  54,121, 36,  8,  3, 21,  1,120, 51, 53,121, 35,  8,  3, 21,  1,120, 51, 52,
 121, 34,  8,  3, 21,  1,120, 51, 51,121, 33,  8,  3, 21,  1,120, 51, 50,121,
  32,  8,  3, 21,  1,120, 51, 49,121, 31,  8,  3, 21,  1,120, 51, 48,121, 30,
   7,  3, 19,  1,120, 50,121,  2,  8,  3, 21,  1,120, 50, 57,121, 29,  8,  3,
  21,  1,120, 50, 56,121, 28,  8,  3, 21,  1,120, 50, 55,121, 27,  8,  3, 21,
   1,120, 50, 54,121, 26,  8,  3, 21,  1,120, 50, 53,121, 25,  8,  3, 21,  1,
 120, 50, 52,121, 24,  8,  3, 21,  1,120, 50, 51,121, 23,  8,  3, 21,  1,120,
  50, 50,121, 22,  8,  3, 21,  1,120, 50, 49,121, 21,  8,  3, 21,  1,120, 50,
  48,121, 20,  6,  3, 19,  9,120, 49,121,  8,  3, 21,  1,120, 49, 57,121, 19,
   8,  3, 21,  1,120, 49, 56,121, 18,  8,  3, 21,  1,120, 49, 55,121, 17,  8,
   3, 21,  1,120, 49, 54,121, 16,  8,  3, 21,  1,120, 49, 53,121, 15,  8,  3,
  21,  1,120, 49, 52,121, 14,  8,  3, 21,  1,120, 49, 51,121, 13,  8,  3, 21,
   1,120, 49, 50,121, 12,  8,  3, 21,  1,120, 49, 49,121, 11,  8,  3, 21,  1,
 120, 49, 48,121, 10, 10,  0,  0,  0, 18,  1, 99,  0,  1,247,  1,238,  1,229,
   1,220,  1,211,  1,202,  1,193,  1,184,  1,176,  1,167,  1,158,  1,149,  1,
 140,  1,131,  1,123,  1,115,  1,107,  1, 99,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,
   3, 19,  1,120, 57,121,  9,  7,  3, 19,  1,120, 56,121,  8,  7,  3, 19,  1,
 120, 55,121,  7,  7,  3, 19,  1,120, 54,121,  6,  8,  3, 21,  1,120, 54, 52,
 121, 64,  8,  3, 21,  1,120, 54, 51,121, 63,  8,  3, 21,  1,120, 54, 50,121,
  62,  8,  3, 21,  1,120, 54, 49,121, 61,  8,  3, 21,  1,120, 54, 48,121, 60,
   7,  3, 19,  1,120, 53,121,  5,  8,  3, 21,  1,120, 53, 57,121, 59,  8,  3,
  21,  1,120, 53, 56,121, 58,  8,  3, 21,  1,120, 53, 55,121, 57,  8,  3, 21,
   1,120, 53, 54,121, 56,  8,  3, 21,  1,120, 53, 53,121, 55,  8,  3, 21,  1,
 120, 53, 52,121, 54,  8,  3, 21,  1,120, 53, 51,121, 53,  8,  3, 21,  1,120,
  53, 50,121, 52, 13,  0,  0,  0, 23,  0, 54,  0,  1,240,  1,214,  1,197,  1,
 172,  1,156,  1,135,  1,117,  1, 89,  1, 73,  1, 55,  1, 41,  1, 14,  0,254,
   0,228,  0,211,  0,186,  0,170,  0,149,  0,131,  0,110,  0, 94,  0, 69,  0,
  54, 13, 23,  4,  7,  0,  1, 64, 57, 76,204,204,204,204,205, 15, 23, 22,  4,
   7,  7, 18, 64, 56, 51, 51, 51, 51, 51, 52, 64, 12,  0,  0,  0,  0,  0,  0,
  85,102,119, 14, 21,  4,  7,  1,  1, 64, 55, 25,153,153,153,153,154,  2,  4,
  19, 20,  4,  1,  7,  5, 22,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,
   0,  0, 16, 19,  4,  7,  1, 19, 64, 52,230,102,102,102,102,103, 15,120,121,
 122, 19, 18,  4,  7, 18, 21, 64, 51,204,204,204,204,204,205, 85,102,119,102,
 111,117,114, 14, 17,  4,  7,  1,  1, 64, 50,179, 51, 51, 51, 51, 52,  4,  3,
  23, 16,  4,  7, 23,  5, 64, 49,153,153,153,153,153,154, 98,117, 98, 98, 97,
   0,  0,128,  0,  0,  0, 15, 15,  4,  7,  0, 19, 64, 48,128,  0,  0,  0,  0,
   0,120,121,122, 24, 14,  4,  7,  7, 21, 64, 46,204,204,204,204,204,206, 64,
  12,  0,  0,  0,  0,  0,  0,102,111,117,114, 14, 13,  4,  7,  1,  1, 64, 44,
 153,153,153,153,153,154,  2,  3, 25, 12,  4,  7,  7, 23, 64, 42,102,102,102,
 102,102,103,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97, 12, 11,  3,
   7,  1, 64, 40, 51, 51, 51, 51, 51, 52, 15, 16, 10,  4,  1, 18,  7, 11, 85,
 102,119, 64, 12,  0,  0,  0,  0,  0,  0, 14,  9,  4,  7,  1,  1, 64, 35,204,
 204,204,204,204,205,  4,  2, 26,  8,  4,  7,  5,  7, 64, 33,153,153,153,153,
 153,154,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16,  7,  4,
   7, 19,  1, 64, 30,204,204,204,204,204,206,120,121,122, 15, 19,  6,  4,  7,
  21, 18, 64, 26,102,102,102,102,102,103,102,111,117,114, 85,102,119, 14,  5,
   4,  7,  1,  1, 64, 22,  0,  0,  0,  0,  0,  0,  3,  4, 23,  4,  4,  7, 23,
   5, 64, 17,153,153,153,153,153,154, 98,117, 98, 98, 97,  0,  0,128,  0,  0,
   0, 15,  3,  4,  7,  0, 19, 64, 10,102,102,102,102,102,103,120,121,122, 24,
   2,  4,  7,  7, 21, 64,  1,153,153,153,153,153,154, 64, 12,  0,  0,  0,  0,
   0,  0,102,111,117,114, 14,  1,  4,  7,  1,  1, 63,241,153,153,153,153,153,
 154,  2,  3, 13,  0,  0,  0, 22,  0, 68,  0,  1,229,  1,213,  1,187,  1,171,
   1,146,  1,130,  1,116,  1, 98,  1, 70,  1, 54,  1, 29,  1, 14,  0,243,  0,
 227,  0,201,  0,185,  0,167,  0,151,  0,130,  0,112,  0, 84,  0, 68,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 14, 45,  4,  7,  1,
   1, 64, 72,192,  0,  0,  0,  0,  1,  2,  4, 26, 44,  4,  7,  7,  5, 64, 72,
  51, 51, 51, 51, 51, 52,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,
   0, 16, 43,  4,  7,  1, 19, 64, 71,166,102,102,102,102,103, 15,120,121,122,
  19, 42,  4,  7, 18, 21, 64, 71, 25,153,153,153,153,154, 85,102,119,102,111,
 117,114, 14, 41,  4,  7,  1,  1, 64, 70,140,204,204,204,204,205,  4,  3, 16,
  40,  4,  1,  5, 23, 44,  0,  0,128,  0,  0,  0, 98,117, 98, 98, 97, 14, 39,
   3,  7, 19, 64, 69,115, 51, 51, 51, 51, 52,120,121,122, 24, 38,  4,  7, 21,
   7, 64, 68,230,102,102,102,102,103,102,111,117,114, 64, 12,  0,  0,  0,  0,
   0,  0, 14, 37,  4,  7,  1,  1, 64, 68, 89,153,153,153,153,154,  3,  2, 25,
  36,  4,  7, 23,  7, 64, 67,204,204,204,204,204,205, 98,117, 98, 98, 97,  0,
   0,  0,  0,  0,  0,  0,  0, 13, 35,  4,  7,  0,  1, 64, 67, 64,  0,  0,  0,
   0,  0, 15, 23, 34,  4,  7,  7, 18, 64, 66,179, 51, 51, 51, 51, 52, 64, 12,
   0,  0,  0,  0,  0,  0, 85,102,119, 14, 33,  4,  7,  1,  1, 64, 66, 38,102,
 102,102,102,103,  2,  4, 26, 32,  4,  7,  7,  5, 64, 65,153,153,153,153,153,
 154,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,128,  0,  0,  0, 16, 31,  4,  7,
   1, 19, 64, 65, 12,204,204,204,204,205, 15,120,121,122, 12, 30,  4,  1, 18,
  21, 33, 85,102,119,102,111,117,114, 14, 29,  4,  7,  1,  1, 64, 63,230,102,
 102,102,102,103,  4,  3, 23, 28,  4,  7,  5, 23, 64, 62,204,204,204,204,204,
 206,  0,  0,128,  0,  0,  0, 98,117, 98, 98, 97, 14, 27,  3,  7, 19, 64, 61,
 179, 51, 51, 51, 51, 52,120,121,122, 24, 26,  4,  7, 21,  7, 64, 60,153,153,
 153,153,153,154,102,111,117,114, 64, 12,  0,  0,  0,  0,  0,  0, 14, 25,  4,
   7,  1,  1, 64, 59,128,  0,  0,  0,  0,  1,  3,  2, 25, 24,  4,  7, 23,  7,
  64, 58,102,102,102,102,102,103, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,
   0,  0, 13,  0,  0,  0, 19,  0,121,  0,  1,231,  1,216,  1,189,  1,173,  1,
 148,  1,134,  1,107,  1, 91,  1, 65,  1, 48,  1, 23,  1,  7,  0,242,  0,224,
   0,203,  0,187,  0,162,  0,148,  0,121,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0, 25, 64,  4,  7,  7, 23, 64, 81,153,153,
 153,153,153,154,  0,  0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97, 12, 63,
   3,  7,  1, 64, 81, 83, 51, 51, 51, 51, 52, 15, 23, 62,  4,  7, 18,  7, 64,
  81, 12,204,204,204,204,205, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0, 14,
  61,  4,  7,  1,  1, 64, 80,198,102,102,102,102,103,  4,  2, 19, 60,  4,  1,
   5,  7, 66,  0,  0,128,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 16, 59,
   4,  7, 19,  1, 64, 80, 57,153,153,153,153,154,120,121,122, 15, 19, 58,  4,
   7, 21, 18, 64, 79,230,102,102,102,102,103,102,111,117,114, 85,102,119, 14,
  57,  4,  7,  1,  1, 64, 79, 89,153,153,153,153,154,  3,  4, 23, 56,  4,  7,
  23,  5, 64, 78,204,204,204,204,204,206, 98,117, 98, 98, 97,  0,  0,128,  0,
   0,  0, 15, 55,  4,  7,  0, 19, 64, 78, 64,  0,  0,  0,  0,  1,120,121,122,
  24, 54,  4,  7,  7, 21, 64, 77,179, 51, 51, 51, 51, 52, 64, 12,  0,  0,  0,
   0,  0,  0,102,111,117,114, 14, 53,  4,  7,  1,  1, 64, 77, 38,102,102,102,
 102,103,  2,  3, 25, 52,  4,  7,  7, 23, 64, 76,153,153,153,153,153,154,  0,
   0,  0,  0,  0,  0,  0,  0, 98,117, 98, 98, 97, 12, 51,  3,  7,  1, 64, 76,
  12,204,204,204,204,205, 15, 23, 50,  4,  7, 18,  7, 64, 75,128,  0,  0,  0,
   0,  1, 85,102,119, 64, 12,  0,  0,  0,  0,  0,  0, 14, 49,  4,  7,  1,  1,
  64, 74,243, 51, 51, 51, 51, 52,  4,  2, 25, 48,  4,  7, 23,  7, 64, 74,102,
 102,102,102,102,103, 98,117, 98, 98, 97,  0,  0,  0,  0,  0,  0,  0,  0, 13,
  47,  4,  7,  0,  1, 64, 73,217,153,153,153,153,154, 15, 23, 46,  4,  7,  7,
  18, 64, 73, 76,204,204,204,204,205, 64, 12,  0,  0,  0,  0,  0,  0, 85,102,
 119, 10,  0,  0,  0, 34,  0, 92,  0,  1,244,  1,231,  1,218,  1,205,  1,192,
   1,179,  1,166,  1,153,  1,140,  1,134,  1,121,  1,108,  1, 95,  1, 82,  1,
  69,  1, 56,  1, 43,  1, 30,  1, 17,  1, 11,  0,254,  0,241,  0,228,  0,215,
   0,202,  0,189,  0,176,  0,163,  0,150,  0,144,  0,131,  0,118,  0,105,  0,
  92,  0, 79,  0,  0,  0,  0, 13, 64, 67, 64,  0,  0,  0,  0,  0, 35, 12,  3,
   7,  1, 64, 66,179, 51, 51, 51, 51, 52, 34, 12,  3,  7,  1, 64, 66, 38,102,
 102,102,102,103, 33, 12,  3,  7,  1, 64, 65,153,153,153,153,153,154, 32, 12,
   3,  7,  1, 64, 65, 12,204,204,204,204,205, 31,  5,  3,  1,  1, 33, 30, 12,
   3,  7,  1, 64, 63,230,102,102,102,102,103, 29, 12,  3,  7,  1, 64, 62,204,
 204,204,204,204,206, 28, 12,  3,  7,  1, 64, 61,179, 51, 51, 51, 51, 52, 27,
  12,  3,  7,  1, 64, 60,153,153,153,153,153,154, 26, 12,  3,  7,  1, 64, 59,
 128,  0,  0,  0,  0,  1, 25, 12,  3,  7,  1, 64, 58,102,102,102,102,102,103,
  24, 12,  3,  7,  1, 64, 57, 76,204,204,204,204,205, 23, 12,  3,  7,  1, 64,
  56, 51, 51, 51, 51, 51, 52, 22, 12,  3,  7,  1, 64, 55, 25,153,153,153,153,
 154, 21,  5,  3,  1,  1, 22, 20, 12,  3,  7,  1, 64, 52,230,102,102,102,102,
 103, 19, 12,  3,  7,  1, 64, 51,204,204,204,204,204,205, 18, 12,  3,  7,  1,
  64, 50,179, 51, 51, 51, 51, 52, 17, 12,  3,  7,  1, 64, 49,153,153,153,153,
 153,154, 16, 12,  3,  7,  1, 64, 48,128,  0,  0,  0,  0,  0, 15, 12,  3,  7,
   1, 64, 46,204,204,204,204,204,206, 14, 12,  3,  7,  1, 64, 44,153,153,153,
 153,153,154, 13, 12,  3,  7,  1, 64, 42,102,102,102,102,102,103, 12, 12,  3,
   7,  1, 64, 40, 51, 51, 51, 51, 51, 52, 11,  5,  3,  1,  1, 11, 10, 12,  3,
   7,  1, 64, 35,204,204,204,204,204,205,  9, 12,  3,  7,  1, 64, 33,153,153,
 153,153,153,154,  8, 12,  3,  7,  1, 64, 30,204,204,204,204,204,206,  7, 12,
   3,  7,  1, 64, 26,102,102,102,102,102,103,  6, 12,  3,  7,  1, 64, 22,  0,
   0,  0,  0,  0,  0,  5, 12,  3,  7,  1, 64, 17,153,153,153,153,153,154,  4,
  12,  3,  7,  1, 64, 10,102,102,102,102,102,103,  3, 12,  3,  7,  1, 64,  1,
 153,153,153,153,153,154,  2, 11,  3,  7,  9, 63,241,153,153,153,153,153,154,
  10,  0,  0,  0, 29,  0,149,  0,  1,243,  1,230,  1,217,  1,204,  1,198,  1,
 185,  1,172,  1,159,  1,146,  1,133,  1,120,  1,107,  1, 94,  1, 81,  1, 68,
   1, 55,  1, 42,  1, 29,  1, 16,  1,  3,  0,246,  0,233,  0,220,  0,207,  0,
 201,  0,188,  0,175,  0,162,  0,149,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  3,  7,
   1, 64, 81,153,153,153,153,153,154, 64, 12,  3,  7,  1, 64, 81, 83, 51, 51,
  51, 51, 52, 63, 12,  3,  7,  1, 64, 81, 12,204,204,204,204,205, 62, 12,  3,
   7,  1, 64, 80,198,102,102,102,102,103, 61,  5,  3,  1,  1, 66, 60, 12,  3,
   7,  1, 64, 80, 57,153,153,153,153,154, 59, 12,  3,  7,  1, 64, 79,230,102,
 102,102,102,103, 58, 12,  3,  7,  1, 64, 79, 89,153,153,153,153,154, 57, 12,
   3,  7,  1, 64, 78,204,204,204,204,204,206, 56, 12,  3,  7,  1, 64, 78, 64,
   0,  0,  0,  0,  1, 55, 12,  3,  7,  1, 64, 77,179, 51, 51, 51, 51, 52, 54,
  12,  3,  7,  1, 64, 77, 38,102,102,102,102,103, 53, 12,  3,  7,  1, 64, 76,
 153,153,153,153,153,154, 52, 12,  3,  7,  1, 64, 76, 12,204,204,204,204,205,
  51, 12,  3,  7,  1, 64, 75,128,  0,  0,  0,  0,  1, 50, 12,  3,  7,  1, 64,
  74,243, 51, 51, 51, 51, 52, 49, 12,  3,  7,  1, 64, 74,102,102,102,102,102,
 103, 48, 12,  3,  7,  1, 64, 73,217,153,153,153,153,154, 47, 12,  3,  7,  1,
  64, 73, 76,204,204,204,204,205, 46, 12,  3,  7,  1, 64, 72,192,  0,  0,  0,
   0,  1, 45, 12,  3,  7,  1, 64, 72, 51, 51, 51, 51, 51, 52, 44, 12,  3,  7,
   1, 64, 71,166,102,102,102,102,103, 43, 12,  3,  7,  1, 64, 71, 25,153,153,
 153,153,154, 42, 12,  3,  7,  1, 64, 70,140,204,204,204,204,205, 41,  5,  3,
   1,  1, 44, 40, 12,  3,  7,  1, 64, 69,115, 51, 51, 51, 51, 52, 39, 12,  3,
   7,  1, 64, 68,230,102,102,102,102,103, 38, 12,  3,  7,  1, 64, 68, 89,153,
 153,153,153,154, 37, 12,  3,  7,  1, 64, 67,204,204,204,204,204,205, 36, 10,
   0,  0,  0, 41,  0,103,  0,  1,250,  1,235,  1,227,  1,218,  1,211,  1,202,
   1,192,  1,179,  1,172,  1,157,  1,149,  1,141,  1,132,  1,125,  1,116,  1,
 106,  1, 93,  1, 86,  1, 74,  1, 63,  1, 47,  1, 40,  1, 31,  1, 16,  1,  8,
   0,255,  0,248,  0,239,  0,229,  0,216,  0,209,  0,197,  0,186,  0,174,  0,
 158,  0,151,  0,136,  0,128,  0,119,  0,112,  0,103,  0, 93,  0,  0,  0,  0,
  10, 55, 44, 78, 85, 76, 76,  8,  2, 25, 52, 54, 44, 51, 46, 53,  6,  2, 21,
  52, 53, 44, 50,  8,  2, 25, 52, 52, 44, 48, 46, 48,  7,  2, 23, 52, 51, 44,
  49, 53, 14,  2, 37, 52, 50, 44, 88, 39, 53, 53, 54, 54, 55, 55, 39,  6,  2,
  21, 52, 49, 44, 52, 15,  2, 39, 52, 48, 44, 50, 49, 52, 55, 52, 56, 51, 54,
  52, 56, 11,  2, 31, 52, 44, 39, 98,117, 98, 98, 97, 39, 10,  2, 29, 51, 57,
  44, 39,120,121,122, 39, 11,  2, 31, 51, 56, 44, 39,102,111,117,114, 39,  6,
   2, 21, 51, 55, 44, 51, 12,  2, 33, 51, 54, 44, 39, 98,117, 98, 98, 97, 39,
   9,  2, 27, 51, 53, 44, 78, 85, 76, 76,  8,  2, 25, 51, 52, 44, 51, 46, 53,
   6,  2, 21, 51, 51, 44, 50,  8,  2, 25, 51, 50, 44, 48, 46, 48,  7,  2, 23,
  51, 49, 44, 49, 53, 14,  2, 37, 51, 48, 44, 88, 39, 53, 53, 54, 54, 55, 55,
  39,  8,  2, 25, 51, 44, 78, 85, 76, 76,  6,  2, 21, 50, 57, 44, 52, 15,  2,
  39, 50, 56, 44, 50, 49, 52, 55, 52, 56, 51, 54, 52, 56, 10,  2, 29, 50, 55,
  44, 39,120,121,122, 39, 11,  2, 31, 50, 54, 44, 39,102,111,117,114, 39,  6,
   2, 21, 50, 53, 44, 51, 12,  2, 33, 50, 52, 44, 39, 98,117, 98, 98, 97, 39,
   9,  2, 27, 50, 51, 44, 78, 85, 76, 76,  8,  2, 25, 50, 50, 44, 51, 46, 53,
   6,  2, 21, 50, 49, 44, 50,  8,  2, 25, 50, 48, 44, 48, 46, 48,  7,  2, 23,
  50, 44, 51, 46, 53,  7,  2, 23, 49, 57, 44, 49, 53, 14,  2, 37, 49, 56, 44,
  88, 39, 53, 53, 54, 54, 55, 55, 39,  6,  2, 21, 49, 55, 44, 52, 12,  2, 33,
  49, 54, 44, 39, 98,117, 98, 98, 97, 39,  9,  2, 27, 49, 53, 44, 78, 85, 76,
  76,  8,  2, 25, 49, 52, 44, 51, 46, 53,  6,  2, 21, 49, 51, 44, 50,  8,  2,
  25, 49, 50, 44, 48, 46, 48,  7,  2, 23, 49, 49, 44, 49, 53, 14,  2, 37, 49,
  48, 44, 88, 39, 53, 53, 54, 54, 55, 55, 39,  5,  2, 19, 49, 44, 50, 10,  0,
   0,  0, 22,  1, 32,  0,  1,243,  1,236,  1,230,  1,215,  1,207,  1,198,  1,
 191,  1,182,  1,172,  1,159,  1,152,  1,140,  1,129,  1,118,  1,102,  1, 95,
   1, 80,  1, 72,  1, 63,  1, 53,  1, 38,  1, 32,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
   0,  5,  2, 19, 57, 44, 52, 14,  2, 37, 56, 44, 50, 49, 52, 55, 52, 56, 51,
  54, 52, 56,  9,  2, 27, 55, 44, 39,120,121,122, 39,  8,  2, 25, 54, 52, 44,
  48, 46, 48,  7,  2, 23, 54, 51, 44, 49, 53, 14,  2, 37, 54, 50, 44, 88, 39,
  53, 53, 54, 54, 55, 55, 39,  6,  2, 21, 54, 49, 44, 52, 15,  2, 39, 54, 48,
  44, 50, 49, 52, 55, 52, 56, 51, 54, 52, 56, 10,  2, 29, 54, 44, 39,102,111,
 117,114, 39, 10,  2, 29, 53, 57, 44, 39,120,121,122, 39, 11,  2, 31, 53, 56,
  44, 39,102,111,117,114, 39,  6,  2, 21, 53, 55, 44, 51, 12,  2, 33, 53, 54,
  44, 39, 98,117, 98, 98, 97, 39,  9,  2, 27, 53, 53, 44, 78, 85, 76, 76,  8,
   2, 25, 53, 52, 44, 51, 46, 53,  6,  2, 21, 53, 51, 44, 50,  8,  2, 25, 53,
  50, 44, 48, 46, 48,  7,  2, 23, 53, 49, 44, 49, 53, 14,  2, 37, 53, 48, 44,
  88, 39, 53, 53, 54, 54, 55, 55, 39,  5,  2, 19, 53, 44, 51,  6,  2, 21, 52,
  57, 44, 52, 12,  2, 33, 52, 56, 44, 39, 98,117, 98, 98, 97, 39,
};

/* Help message */
static const char zHelp[] =
  "Usage:\n"
  "  sessionfuzz setup          -- Generate seed files c1.txt, c2.txt, etc.\n"
  "  sessionfuzz run FILE ...   -- Run against fuzzed changeset FILE\n"
  "  sessionfuzz run SQLAR ...  -- Run against all files in the SQL Archive\n"
;

#include <stdio.h>
#include <string.h>
#include <assert.h>
#ifndef OMIT_ZLIB
#include "zlib.h"
#endif

/*
** Implementation of the "sqlar_uncompress(X,SZ)" SQL function
**
** Parameter SZ is interpreted as an integer. If it is less than or
** equal to zero, then this function returns a copy of X. Or, if
** SZ is equal to the size of X when interpreted as a blob, also
** return a copy of X. Otherwise, decompress blob X using zlib
** utility function uncompress() and return the results (another
** blob).
*/
static void sqlarUncompressFunc(
  sqlite3_context *context,
  int argc,
  sqlite3_value **argv
){
#ifdef OMIT_ZLIB
  sqlite3_result_value(context, argv[0]);
#else
  uLong nData;
  uLongf sz;

  assert( argc==2 );
  sz = sqlite3_value_int(argv[1]);

  if( sz<=0 || sz==(nData = sqlite3_value_bytes(argv[0])) ){
    sqlite3_result_value(context, argv[0]);
  }else{
    const Bytef *pData= sqlite3_value_blob(argv[0]);
    Bytef *pOut = sqlite3_malloc(sz);
    if( Z_OK!=uncompress(pOut, &sz, pData, nData) ){
      sqlite3_result_error(context, "error in uncompress()", -1);
    }else{
      sqlite3_result_blob(context, pOut, sz, SQLITE_TRANSIENT);
    }
    sqlite3_free(pOut);
  }
#endif
}


/* Run a chunk of SQL.  If any errors happen, print an error message
** and exit.
*/
static void runSql(sqlite3 *db, const char *zSql){
  int rc;
  char *zErr = 0;
  rc = sqlite3_exec(db, zSql, 0, 0, &zErr);
  if( rc || zErr ){
    fprintf(stderr, "SQL failed: rc=%d zErr=[%s]\n", rc, zErr);
    fprintf(stderr, "SQL: [%s]\n", zSql);
    exit(1);
  }
}

/*
** Write buffer to disk
*/
static void writeFile(const char *zFilename, const void *pData, int nData){
  FILE *out;
  int n;
  out = fopen(zFilename, "wb");
  if( out==0 ){
    fprintf(stderr, "cannot open \"%s\" for writing\n", zFilename);
    exit(1);
  }
  n = (int)fwrite(pData, 1, nData, out);
  fclose(out);
  if( n!=nData ){
    fprintf(stderr, "only wrote %d of %d bytes to \"%s\"\n",n,nData,zFilename);
    exit(1);
  }
}

/*
** Generate a changeset from session pSess and write it to zFile
*/
static void makeChangeset(const char *zFile, sqlite3_session *pSess){
  void *pChg;
  int nChg;
  int rc;
  rc = sqlite3session_changeset(pSess, &nChg, &pChg);
  if( rc ){
    fprintf(stderr, "sqlite3session_changeset() returned %d\n", rc);
    exit(1);
  }
  writeFile(zFile, pChg, nChg);
  sqlite3_free(pChg);
}

/*
** Read a file from disk.  Space to hold the answer is obtained from
** sqlite3_malloc64().
*/
static void readFile(const char *zName, void **ppData, int *pnData){
  FILE *in = fopen(zName, "rb");
  long nIn;
  size_t nRead;
  char *pBuf;
  *ppData = 0;
  *pnData = 0;
  if( in==0 ){
    fprintf(stderr, "Cannot open \"%s\" for reading\n", zName);
    exit(1);
  }
  fseek(in, 0, SEEK_END);
  nIn = ftell(in);
  rewind(in);
  pBuf = sqlite3_malloc64( nIn+1 );
  if( pBuf==0 ){
    fprintf(stderr, "Failed to malloc %lld bytes\n", (sqlite3_int64)(nIn+1));
    exit(1);
  }
  nRead = fread(pBuf, 1, nIn, in);
  fclose(in);
  if( nRead!=(size_t)nIn ){
    fprintf(stderr, "Read only %d of %d bytes from %s\n", (int)nRead, (int)nIn,
                    zName);
    exit(1);
  }
  pBuf[nIn] = 0;
  *pnData = nIn;
  *ppData = pBuf;
}

/*
** The conflict callback
*/
static int conflictCall(
  void *NotUsed,
  int eConflict,
  sqlite3_changeset_iter *p
){
  (void)NotUsed;
  (void)p;
  return SQLITE_CHANGESET_OMIT;
}

/*
** Reset the database file
*/
static void db_reset(sqlite3 *db){
  unsigned char *pData;
  int nData;
  int rc;

  nData = sizeof(aDbBytes);
  pData = sqlite3_malloc64( nData );
  if( pData==0 ){
    fprintf(stderr, "could not allocate %d bytes\n", nData);
    exit(1);
  }
  memcpy(pData, aDbBytes, nData);
  rc = sqlite3_deserialize(db, 0, pData, nData, nData,
     SQLITE_DESERIALIZE_FREEONCLOSE | SQLITE_DESERIALIZE_RESIZEABLE);
  if( rc ){
    fprintf(stderr, "sqlite3_deserialize() failed with %d: %s\n",
            rc, sqlite3_errmsg(db));
    exit(1);
  }
}

/*
** Given a full file pathname, return a pointer to the tail.
** Example:
** 
**   input:    /home/drh/sqlite/abc.db
**   output:   abc.db
*/
static const char *fileTail(const char *z){
  const char *zOut = z;
  while( z[0] ){
    if( z[0]=='/' && z[1]!=0 ) zOut = &z[1];
    z++;
  }
  return zOut;
}

int main(int argc, char **argv){
  const char *zCmd;
  sqlite3 *db;
  int rc;
  sqlite3_session *pSess;
  sqlite3_stmt *pStmt;
  void *pChgset;
  int nChgset;
  int bVerbose = 0;

  if( argc<2 ){
    fprintf(stderr, "%s", zHelp);
    exit(1);
  }
  rc = sqlite3_open_v2(":memory:",&db,
                       SQLITE_OPEN_READWRITE|SQLITE_OPEN_CREATE, "memdb");
  if( rc ){
    fprintf(stderr, "Failed to open :memory: database: %s\n",
            sqlite3_errmsg(db));
    exit(1);
  }
  db_reset(db);
  zCmd = argv[1];
  if( strcmp(zCmd, "setup")==0 ){
    if( argc!=2 ){
      fprintf(stdout, "Wrong number of arguments.\n%s", zHelp);
      exit(1);
    }
    runSql(db, zFillSql);
    rc = sqlite3session_create(db, "main", &pSess);
    if( rc ){
      fprintf(stderr, "sqlite3session_create() returns %d\n", rc);
      exit(1);
    }
    rc = sqlite3session_attach(pSess, 0);
    if( rc ){
      fprintf(stderr, "sqlite3session_attach(db,0) returns %d\n", rc);
      exit(1);
    }
    runSql(db, "INSERT INTO t4(z) VALUES('');");
    makeChangeset("c1.txt", pSess);
    runSql(db, 
      "UPDATE t1 SET b=c, c=b WHERE a IN (5,7);\n"
      "DELETE FROM t2 WHERE rowid IN (8,2);\n"
      "INSERT OR IGNORE INTO t4 SELECT b FROM t1 WHERE b IS TRUE LIMIT 2;");
    makeChangeset("c2.txt", pSess);
    runSql(db, "UPDATE t3 SET x=y, y=NULL WHERE rowid IN (1,3);");
    makeChangeset("c3.txt", pSess);
    sqlite3session_delete(pSess);
  }else
  if( strcmp(zCmd, "run")==0 ){
    int i;
    if( argc<3 ){
      fprintf(stdout, "Wrong number of arguments.\n%s", zHelp);
      exit(1);
    }
    for(i=2; i<argc; i++){
      if( strcmp(argv[i],"-v")==0 ){
        bVerbose = 1;
        continue;
      }
      readFile(argv[i], &pChgset, &nChgset);
      if( nChgset >= 512 
       && memcmp(pChgset, "SQLite format 3", 16)==0 
      ){
        sqlite3 *db2;
        sqlite3_stmt *pStmt2;
        int nCase = 0;
        /* This file is an SQL Archive containing many changesets */
        if( !bVerbose ){ printf("%s: ", fileTail(argv[i])); fflush(stdout); }
        sqlite3_open_v2(":memory:", &db2, 
                        SQLITE_OPEN_CREATE|SQLITE_OPEN_READWRITE, "memdb");
        sqlite3_deserialize(db2, 0, pChgset, nChgset, nChgset,
              SQLITE_DESERIALIZE_READONLY | SQLITE_DESERIALIZE_FREEONCLOSE);
        sqlite3_create_function(db2, "sqlar_uncompress", 2, SQLITE_UTF8, 0,
                                 sqlarUncompressFunc, 0, 0);        
        rc = sqlite3_prepare_v2(db2, "SELECT name, sqlar_uncompress(data,sz)"
                                     "  FROM sqlar", -1, &pStmt2, 0);
        if( rc ){
          fprintf(stderr, "SQL error: %s\n", sqlite3_errmsg(db2));
          exit(1);
        }
        while( SQLITE_ROW==sqlite3_step(pStmt2) ){
          if( bVerbose ){
            printf("%s/%s:", fileTail(argv[i]), sqlite3_column_text(pStmt2,0));
            fflush(stdout);
          }
          runSql(db, "BEGIN");
          pChgset = (unsigned char*)sqlite3_column_blob(pStmt2, 1);
          nChgset = sqlite3_column_bytes(pStmt2, 1);
          rc = sqlite3changeset_apply(db, nChgset, pChgset, 0, conflictCall, 0);
          if( bVerbose ){
            printf(" Ok.  rc=%d\n", rc);
            fflush(stdout);
          }
          runSql(db, "ROLLBACK");
          nCase++;
        }
        sqlite3_finalize(pStmt2);
        sqlite3_close(db2);
        if( bVerbose ) printf("%s: ", fileTail(argv[i]));
        printf(" %d cases, 0 crashes\n", nCase);
        fflush(stdout);
      }else{
        /* The named file is just an ordinary changeset */
        printf("%s:", fileTail(argv[i]));
        fflush(stdout);
        runSql(db, "BEGIN");
        rc = sqlite3changeset_apply(db, nChgset, pChgset, 0, conflictCall, 0);
        printf(" %d\n", rc);
        fflush(stdout);
        runSql(db, "ROLLBACK");
        sqlite3_free(pChgset);
      }
    }
  }else
  {
    fprintf(stderr, "%s", zHelp);
    exit(1);
  }
  rc = sqlite3_prepare_v2(db, "PRAGMA integrity_check;", -1, &pStmt, 0);
  if( rc ){
    fprintf(stderr, "SQL error: %s\n", sqlite3_errmsg(db));
    exit(1);
  }
  if( sqlite3_step(pStmt)!=SQLITE_ROW
   || strcmp((const char*)sqlite3_column_text(pStmt,0),"ok")!=0
  ){
    fprintf(stderr, "Integrity check failed!\n");
    do{
      fprintf(stderr, "%s\n", sqlite3_column_text(pStmt,0));
    }while( sqlite3_step(pStmt)==SQLITE_ROW );
  }
  sqlite3_finalize(pStmt);
  sqlite3_close(db);
  if( sqlite3_memory_used()>0 ){
    fprintf(stderr, "memory leak of %lld bytes\n",
         sqlite3_memory_used());
    exit(1);
  }
  return 0;
}
