#include <iostream>
#include <fstream>
#include <vector>
#include <string>
#include <cstring>
#include <getopt.h>

#include "node.h"
#include "tree_reader.h"
#include "string_node_object.h"
#include "vector_node_object.h"
#include "tree.h"
#include "utils.h"
#include "tree_utils.h"
#include "log.h"
#include "citations.h"


void print_help ();
std::string get_version_line ();

void print_help () {
    std::cout << "Get the number of descendant tips of internal nodes specified by mrca statements." << std::endl;
    std::cout << "This will take a newick- or nexus-formatted tree from a file or STDIN," << std::endl;
    std::cout << "and an MRCA file with format:" << std::endl;
    std::cout << "MRCANAME = tip1 tip2 ..." << std::endl;
    std::cout << std::endl;
    std::cout << "Usage: pxmrca [OPTIONS]..." << std::endl;
    std::cout << std::endl;
    std::cout << "Options:" << std::endl;
    std::cout << " -t, --treef=FILE    input newick tree file, STDIN otherwise" << std::endl;
    std::cout << " -m, --mrca=FILE     file containing MRCA declarations" << std::endl;
    std::cout << " -o, --outf=FILE     output newick file, STOUT otherwise" << std::endl;
    std::cout << " -h, --help          display this help and exit" << std::endl;
    std::cout << " -V, --version       display version and exit" << std::endl;
    std::cout << " -C, --citation      display phyx citation and exit" << std::endl;
    std::cout << std::endl;
    std::cout << "Report bugs to: <https://github.com/FePhyFoFum/phyx/issues>" << std::endl;
    std::cout << "phyx home page: <https://github.com/FePhyFoFum/phyx>" << std::endl;
}
std::string get_version_line () {
    std::string vl = "pxmrca 1.3\n";
    vl += "Copyright (C) 2013-2021 FePhyFoFum\n";
    vl += "License GPLv3\n";
    vl += "Written by Stephen A. Smith (blackrim)";
    return vl;
}

static struct option const long_options[] =
{
    {"treef", required_argument, nullptr, 't'},
    {"outf", required_argument, nullptr, 'o'},
    {"mrca", required_argument, nullptr, 'm'},
    {"help", no_argument, nullptr, 'h'},
    {"version", no_argument, nullptr, 'V'},
    {"citation", no_argument, nullptr, 'C'},
    {nullptr, 0, nullptr, 0}
};

int main(int argc, char * argv[]) {
    
    log_call(argc, argv);
    
    char * outf = nullptr;
    char * treef = nullptr;
    char * mrcaf = nullptr;
    bool outfileset = false;
    bool fileset = false;
    bool mrcaset = false;
    
    while (true) {
        int oi = -1;
        int c = getopt_long(argc, argv, "t:o:m:hVC", long_options, &oi);
        if (c == -1) {
            break;
        }
        switch(c) {
            case 't':
                fileset = true;
                treef = strdup(optarg);
                check_file_exists(treef);
                break;
            case 'o':
                outfileset = true;
                outf = strdup(optarg);
                break;
            case 'm':
                mrcaset = true;
                mrcaf = strdup(optarg);
                break;
            case 'h':
                print_help();
                exit(0);
            case 'V':
                std::cout << get_version_line() << std::endl;
                exit(0);  
            case 'C':
                std::cout << get_phyx_citation() << std::endl;
                exit(0);
            default:
                print_error(*argv);
                exit(0);
        }
    }
    
    if (fileset && outfileset) {
        check_inout_streams_identical(treef, outf);
    }
    
    if (!mrcaset) {
        std::cout << "Must supply mrca file" << std::endl;
        exit(0);
    }
    
    std::istream * pios = nullptr;
    std::ostream * poos = nullptr;
    std::ifstream * fstr = nullptr;
    std::ofstream * ofstr = nullptr;
    
    if (outfileset) {
        ofstr = new std::ofstream(outf, std::ios::app);
        poos = ofstr;
    } else {
        poos = &std::cout;
    }
    if (fileset) {
        fstr = new std::ifstream(treef);
        pios = fstr;
    } else {
        pios = &std::cin;
        if (!check_for_input_to_stream()) {
            print_help();
            exit(1);
        }
    }
    
    std::ifstream inmrca(mrcaf);
    std::string mrcaline;
    std::map<std::string, std::vector<std::string> > mrcas;
    while (getline_safe(inmrca, mrcaline)) {
        if (mrcaline.empty()) {
            continue;
        }
        std::vector<std::string> searchtokens;
        tokenize(mrcaline, searchtokens, "=");
        std::string mrcaname = searchtokens[0];
        trim_spaces(mrcaname);
        searchtokens.erase(searchtokens.begin());
        searchtokens = tokenize(searchtokens[0]);
        mrcas[mrcaname] = searchtokens;
    }
    inmrca.close();
    
    std::string retstring;
    int ft = test_tree_filetype_stream(*pios, retstring);
    if (ft != 0 && ft != 1) {
        std::cerr << "Error: this really only works with nexus or newick. Exiting." << std::endl;
        exit(0);
    }
    
    bool going = true;
    std::map<std::string, std::vector<std::string> >::iterator it;
    
    if (ft == 1) {
        Tree * tree;
        while (going) {
            tree = read_next_tree_from_stream_newick(*pios, retstring, &going);
            if (tree != nullptr) {
                for (it=mrcas.begin(); it != mrcas.end(); ++it) {
                    //std::cout << "Dealing with clade '" << (*it).first << "'" << std::endl;
                    if (!check_names_against_tree(tree, (*it).second)) {
                        // allow more flexibility here
                        std::cerr << "Error: check mrca file for typos. Exiting." << std::endl;
                        exit(0);
                    }
                    Node * nd = tree->getMRCA((*it).second);
                    (*poos) << (*it).first << " " << nd->get_num_leaves() << " "
                        << nd->getName() << std::endl;
                }
                delete tree;
            }
        }
    } else if (ft == 0) { // Nexus. need to worry about possible translation tables
        std::map<std::string, std::string> translation_table;
        bool ttexists;
        ttexists = get_nexus_translation_table(*pios, &translation_table, &retstring);
        Tree * tree;
        while (going) {
            tree = read_next_tree_from_stream_nexus(*pios, retstring, ttexists,
                &translation_table, &going);
            if (tree != nullptr) {
                for (it=mrcas.begin(); it != mrcas.end(); ++it) {
                    //std::cout << "Dealing with clade '" << (*it).first << "'" << std::endl;
                    if (!check_names_against_tree(tree, (*it).second)) {
                        // allow more flexibility here
                        std::cerr << "Error: check mrca file for typos. Exiting." << std::endl;
                        exit(0);
                    }
                    Node * nd = tree->getMRCA((*it).second);
                    (*poos) << (*it).first << " " << nd->get_num_leaves() << " "
                        << nd->getName() << std::endl;
                }
                delete tree;
            }
        }
    }
    
    if (fileset) {
        fstr->close();
        delete pios;
    }
    if (outfileset) {
        ofstr->close();
        delete poos;
    }
    
    return EXIT_SUCCESS;
}
