/*	Value

PIRL CVS ID: Value.java,v 1.30 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.PVL;

//	The base class
import	javax.swing.tree.DefaultMutableTreeNode;
import	javax.swing.tree.MutableTreeNode;

import	java.lang.Math;
import	java.util.List;
import	java.util.Vector;
import	java.util.Arrays;
import	java.util.Iterator;
import	java.util.ListIterator;
import	java.util.NoSuchElementException;
import	java.io.OutputStream;
import	java.io.StringWriter;
import	java.io.IOException;


/**	A <I>Value</I> is a general-purpose value entity containing arbitrary
	data.
<P>
	A Value provides the data value for Parameter objects. It is based on
	the Parameter Value Language syntax:
<P>
<BLOCKQUOTE>
[ <B>(</B> | <B>{</B> ] <B><I>Datum</I></B> [ <B>< <I>Units</I> ></B> ] [ <B>, <I>Datum</I></B> [...] ] [ <B>)</B> | <B>}</B> [ <B>< <I>Units</I> ></B> ] ]
</BLOCKQUOTE>
<P>
	The type of <I>Datum</I> that a Value contains is indicated by its
	<I>Type</I> code. There are three basic types:
<P>
<DL>
<DT><CODE>NUMERIC</CODE>
	<DD>A number with two specific types:
	<BLOCKQUOTE>
		<CODE>INTEGER</CODE> (Long) with a variable radix <I>Base</I><BR>
		<CODE>REAL</CODE> (Double)<BR>
	</BLOCKQUOTE>
<DT><CODE>STRING</CODE>
	<DD>A String of characters with four specific types:
	<BLOCKQUOTE>
		<CODE>IDENTIFIER</CODE> - Unquoted<BR>
		<CODE>SYMBOL</CODE> - Single-quoted (')<BR>
		<CODE>TEXT</CODE> - Double-quoted (")<BR>
		<CODE>DATE_TIME</CODE> - Contains dash (-) or colon (:)<BR>
	</BLOCKQUOTE>
<DT><CODE>ARRAY</CODE>
	<DD>A Vector of zero or more Datum, with two specific types:
	<BLOCKQUOTE>
		<CODE>SET</CODE> - Enclosed by curly braces ('{' and '}')
			or no enclosing characters<BR>
		<CODE>SEQUENCE</CODE> - Enclosed by parentheses ('(' and ')')<BR>
	</BLOCKQUOTE>
</DL>
<P>
	There is also an <CODE>UNKNOWN</CODE> Type for a Value with no
	Datum. Note that any Datum may be an <CODE>ARRAY</CODE> Type, so
	Arrays of Arrays are supported. Arrays may be of mixed Types and
	non-rectangular sizes.
<P>
	Any Datum, including an Array, may have an optional <I>Units</I>
	String that provides an arbitrary description for the Datum.
<P>
	@see		Parameter
	@see		Parser

	@author		Bradford Castalia, UA/PIRL
	@version	1.30 
*/
public class Value
	extends DefaultMutableTreeNode
	implements Cloneable
{
/**	Class name and version identification.
*/
public static final String	
	ID = "PIRL.PVL.Value (1.30 2012/04/16 06:14:23)";

/*
	A DefaultMutableTreeNode contains a userObject that is displayed
	using its toString method in a JTree. For our purposes the
	userObject is the Value object itself.
*/
/*
	When a Value is single-valued (NUMERIC or STRING), _Datum_ is used
	to hold the object of the specific Type.

	However, when a Value is an ARRAY, then the children variable in
	the base class holds a Vector of Values.
*/
private Object
	_Datum_		= null;

private int
	_Type_		= UNKNOWN;

/*	Value _Type_ codes:

	There are three basic Value Types:

		_Type_		_Datum_ object class
		---------	-------------------------------------------
		NUMERIC:
		  INTEGER	Long
		  REAL		Double

		STRING:		String

			IDENTIFIER - Unquoted
			SYMBOL - Single-quoted (')
			TEXT - Double-quoted (")
			DATE_TIME - Contains dash (-) or colon (:)

		ARRAY:		null (children Vector contains Values)

			SET - Enclosed by curly braces ('{' and '}')
			      or no enclosing characters
			SEQUENCE - Enclosed by parentheses ('(' and ')')

			Each element of an ARRAY children Vector is a Value
			(which may include another ARRAY children Vector).

	In addition there is the UNKNOWN Type for empty Values.

	Every basic Type has specific Types. Only in the case of the
	INTEGER and REAL NUMERIC Types does this distinction have any
	significance here.

	Note that Value _Type_ codes are expected to be distinct from
	Parameter _Classification_ codes.
*/

/**	The lowest bit used by the Type code bit field.
*/
public static final int
	LOWEST_BIT	= 0;

/**	The highest bit used by the Type code bit field.
*/
public static final int
	HIGHEST_BIT	= LOWEST_BIT + 9;

/**	The classification of an empty Value.
*/
public static final int
	UNKNOWN		= 0;

/**	Non-specific Type for a Value with a numeric datum.
<P>
	The specific Type will be either <CODE>INTEGER</CODE> or
	<CODE>REAL</CODE>.
<P>
	@see	#INTEGER
	@see	#REAL
*/
public static final int
	NUMERIC		= 1 << LOWEST_BIT;

/**	Type for a Value with a Long datum.
*/
public static final int
	INTEGER		= NUMERIC | (NUMERIC << 1);

/**	Type for a Value with a Double datum.
*/
public static final int
	REAL		= NUMERIC | (NUMERIC << 2);

/**	Non-specific Type for a Value with a String datum.
<P>
	The specific Type will be either <CODE>IDENTIFIER</CODE>,
	<CODE>SYMBOL</CODE>, <CODE>TEXT</CODE>, or <CODE>DATE_TIME</CODE>.
<P>
	@see	#IDENTIFIER
	@see	#SYMBOL
	@see	#TEXT
	@see	#DATE_TIME
*/
public static final int
	STRING		= 1 << (LOWEST_BIT + 3);

/**	Type for a Value with a String datum.
<P>
	It will be written unadorned (no quoting).
*/
public static final int
	IDENTIFIER	= STRING | (STRING << 1);

/**	Type for a Value with a String datum.
<P>
	It will be written enclosed within {@link Parser#SYMBOL_DELIMITER
	<CODE>Parser.SYMBOL_DELIMITER</CODE>} characters.
*/
public static final int
	SYMBOL		= STRING | (STRING << 2);

/**	Type for a Value with a String datum.
<P>
	It will be written enclosed within {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} characters.
*/
public static final int
	TEXT		= STRING | (STRING << 3);

/**	Type for a Value with a String datum.
<P>
	It may represent a date or time (it contains one or more {@link
	Parser#DATE_TIME_DELIMITERS <CODE>Parser.DATE_TIME_DELIMITERS</CODE>}).
*/
public static final int
	DATE_TIME	= SYMBOL | TEXT;

/**	Non-specific Type for a Value with a Vector datum.
<P>
	The specific Type will be either <CODE>SET</CODE> or
	<CODE>SEQUENCE</CODE>.
<P>
	@see	#SET
	@see	#SEQUENCE
*/
public static final int
	ARRAY		= 1 << (LOWEST_BIT + 7);

/**	Type for a Value with a datum that is a Vector of Values.
<P>
	It will be written preceeded by a {@link Parser#SET_START_DELIMITER
	<CODE>Parser.SET_START_DELIMITER</CODE>} character and ending with
	a {@link Parser#SET_END_DELIMITER
	<CODE>Parser.SET_END_DELIMITER</CODE>} character.
*/
public static final int
	SET			= ARRAY | (ARRAY << 1);

/**	Type for a Value with a datum that is a Vector of Values.
<P>
	It will be written preceeded by a {@link
	Parser#SEQUENCE_START_DELIMITER
	<CODE>Parser.SEQUENCE_START_DELIMITER</CODE>} character and ending
	with a {@link Parser#SEQUENCE_END_DELIMITER
	<CODE>Parser.SEQUENCE_END_DELIMITER</CODE>} character.
*/
public static final int
	SEQUENCE	= ARRAY | (ARRAY << 2);

private static int
	Default_Array_Type	= SET;

/**	Selects only the Type code bit fields recognized by the Value
	class.
<P>
	Logical AND with another int variable will ensure that only Value
	Type bits remain.
*/
public static final int
	MASK		= 0x3FF;

/*	INETEGER _Datum_ may be represented in any (valid) base.

	Non-decimal base integers are representated with the syntax:

		base#value#

	where "base" is the decimal representation of the number base for
	the value, and "value" is represented using the corresponding
	number base digits. Valid base values are in the range
	Character.MIN_RADIX to Character.MAX_RADIX.
*/
private int
	_Base_		= 10;

//	The units string is application dependent.
private String
	_Units_		= null;


//	When throwing an exception is inappropriate.
private PVL_Exception
	_First_Warning_			= null,
	_Last_Warning_			= null;	
private boolean
	_Use_First_Warning_		= true;


//	DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_DATA		= 1 << 1,
	DEBUG_MODIFY	= 1 << 2,
	DEBUG_TYPE		= 1 << 3,
	DEBUG_WRITE		= 1 << 4,
	DEBUG_MATCH		= 1 << 5,
	DEBUG_FIND		= 1 << 6,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Creates an empty Value, with Type <CODE>UNKNOWN</CODE>.
*/
public Value ()
{userObject = this;}

/**	Creates an <CODE>IDENTIFIER</CODE> Value from the primitive
	char value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(char)
*/
public Value (char   value) {this (); Data (value);}

/**	Creates an <CODE>INTEGER</CODE> Value from the primitive
	byte value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(byte)
*/
public Value (byte   value) {this (); Data (value);}

/**	Creates an <CODE>INTEGER</CODE> Value from the primitive
	short value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(short)
*/
public Value (short  value) {this (); Data (value);}

/**	Creates an <CODE>INTEGER</CODE> Value from the primitive
	int value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(int)
*/
public Value (int    value) {this (); Data (value);}

/**	Creates an <CODE>INTEGER</CODE> Value from the primitive
	long value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(long)
*/
public Value (long   value) {this (); Data (value);}

/**	Creates a <CODE>REAL</CODE> Value from the primitive
	float value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(float)
*/
public Value (float  value) {this (); Data (value);}

/**	Creates a <CODE>REAL</CODE> Value from the primitive
	double value.
<P>
	@param	value	The data used to construct the Value.
	@see	#Data(double)
*/
public Value (double value) {this (); Data (value);}

/**	Creates a Value whose Type depends on the class of the
	value object.
<P>
	@param	value	The object used to construct the Value.
	@throws	PVL_Exception	From the <CODE>Data</CODE> method.
	@see	#Data(Object)
*/
public Value (Object value)
	throws PVL_Exception    {this (); Data (value);}

/**	A Parser is used to get a new Value.
<P>
	@param	parser	The Parser source for PVL syntax.
	@throws	PVL_Exception	When the Parser encounters an
		unrecoverable error. The Parser may also generate a
		Warning which will associated with the new Parameter.
	@see	Parser
*/
public Value
	(
	Parser		parser
	)
throws PVL_Exception
{
this ();
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Value construction from a Parser");
parser.Reset_Warning ();	//	Start clean.
//	Graft the Value from the parser into this Value.
graft (parser.Get_Value ());
_First_Warning_ = _Last_Warning_ = parser.Warning ();
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		("<<< Value construction from Parser completed.");
}

/**	A Value is constructed from a copy of another Value.
<P>
	The contents of the original Value are copied into the new Value.
	This includes its datum, radix base and units String. When the
	original Value is an Array, each datum is recursively copied and
	added to the new Value in the same order. The warning status from
	the original is not copied.
<P>
	@param	value	The original Value to be copied.
	@throws PVL_Exception
		<DL>
		<DT>BAD_ARGUMENT
			<DD>An inappropriate object was found as the element of an
			Array.
		</DL>
	@see #set_type(int)
	@see #Add(Value)
*/
public Value
	(
	Value		value
	)
throws PVL_Exception
{
this ();
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> Value: Constructing from " + value);
_Base_  = value._Base_;
_Units_ = value._Units_;
set_type (value._Type_);

if (value.Is_Array ())
	{
	ListIterator
		list = value.listIterator ();
	while (list.hasNext ())
		{
		Object
			data = list.next ();
		if (! Is_Value (data))
			throw new PVL_Exception
				(
				ID, PVL_Exception.BAD_ARGUMENT,
				"While constructing a new Value\n"
					+ " from the " + value.Type_Name () + " Value,\n"
				+ "  element " + list.previousIndex ()
					+ " is an object of class "
					+ data.getClass ().getName () + "."
				);
		/*	Add a copy of the Value from the list.
			This will recursively replicate any ARRAY Values.
		*/
		if ((DEBUG & DEBUG_SETUP) != 0)
			System.out.println ("Value: Add " + (Value)data);
		Add (new Value ((Value)data));
		}
	}
else
	//	All _Datum_ are final classes that can be assigned directly.
	_Datum_ = value._Datum_;

//	Don't copy the warning status.
}

/*==============================================================================
	Accessors
*/
/*------------------------------------------------------------------------------
	Datum:
*/
/*..............................................................................
	Get
*/
/**	Gets a Long object from the Value.
<P>
	If the Value is Type <CODE>INTEGER</CODE> this is the unmodified
	datum, unless the datum is null in which case a Long with a 0 value
	is created. If the Value is Type <CODE>REAL</CODE> then a Long will
	be created from the Double datum by casting (which may result in
	loss of accuracy). If the Value is a <CODE>STRING</CODE> Type, then
	a Long will be constructed from the Value's <CODE>long_Data</CODE>
	method value. <B>Note</B>: The current radix base for the Value
	will be used when converting a String to a Long.
<P>
	@return	A Long object from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is not a <CODE>NUMERIC</CODE> Type,
				or a <CODE>STRING</CODE> Type could not be converted
				to a long.
		</DL>
	@see	#long_Data()
*/
public Long Long_Data ()
	throws PVL_Exception
{
if (Is_Numeric ())
	{
	if (_Datum_ == null)
		return new Long (0L);
	if (Is_Integer ())
		return (Long)_Datum_;
	return new Long (((Double)_Datum_).longValue ());
	}
if (Is_String ())
	return new Long (long_Data ());
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a Long from a value of Type " + Type_Name () + "."
	);
}

/**	Gets a primitive long value from the Value.
<P>
	If the Value is Type <CODE>INTEGER</CODE> this is the long value
	contained in the Long datum, unless the datum is null in which case
	a value of 0L is returned. If the Value is Type <CODE>REAL</CODE>
	then a long value will be created from the Double datum by casting
	(which may result in loss of accuracy). If the Value is a
	<CODE>STRING</CODE> Type, then an attempt will be made to interpret
	the String as a long value. <B>Note</B>: The current radix base for
	the Value will be used when converting a String to a long.
<P>
	@return	A long value from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is not a <CODE>NUMERIC</CODE> Type,
				or a <CODE>STRING</CODE> Type could not be converted
				to a long.
		</DL>
	@see	Long#parseLong(String, int)
*/
public long long_Data ()
	throws PVL_Exception
{
if (Is_Numeric ())
	{
	if (_Datum_ == null)
		return 0L;
	if (Is_Integer ())
		return ((Long)_Datum_).longValue ();
	return ((Double)_Datum_).longValue ();
	}
if (Is_String ())
	{
	if (_Datum_ == null)
		return 0L;
	try
		{
		long
			number = Long.parseLong((String)_Datum_, Math.abs (_Base_));
		if (_Base_ < 0)
			number = -number;
		return number;
		}
	catch (NumberFormatException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.ILLEGAL_SYNTAX,
			"Can't create a long value from the " + Type_Name ()
				+ " Value \"" + (String)_Datum_ + "\""
			);
		}
	}
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a long value from a Value of Type " + Type_Name () + "."
	);
}

/**	Gets a Double object from the Value.
<P>
	If the Value is Type <CODE>REAL</CODE> this is the unmodified
	datum, unless the datum is null in which case a Double with a 0.0
	value is created. If the Value is Type <CODE>INTEGER</CODE> then a
	Double will be created from the Long datum by casting. If the Value
	is a <CODE>STRING</CODE> Type, then an attempt will be made to
	interpret the String as a Double value.
<P>
	@return	A Double object from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is not a <CODE>NUMERIC</CODE> Type,
				or a <CODE>STRING</CODE> Type could not be converted
				to a Double.
		</DL>
	@see	Double#valueOf(String)
*/
public Double Double_Data ()
	throws PVL_Exception
{
if (Is_Numeric ())
	{
	if (_Datum_ == null)
		return new Double (0.0);
	if (Is_Real ())
		return (Double)_Datum_;
	return new Double (((Long)_Datum_).doubleValue ());
	}
if (Is_String ())
	{
	if (_Datum_ == null)
		return new Double (0.0);
	try {return Double.valueOf((String)_Datum_);}
	catch (NumberFormatException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.ILLEGAL_SYNTAX,
			"Can't create a Double from the " + Type_Name ()
				+ " Value \"" + (String)_Datum_ + "\""
			);
		}
	}
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a Double from a Value of Type " + Type_Name () + "."
	);
}

/**	Gets a primitive double value from the Value.
<P>
	If the Value is Type <CODE>REAL</CODE> this is the double value
	contained in the Double datum, unless the datum is null in which
	case a value of 0.0 is returned. If the Value is Type
	<CODE>INTEGER</CODE> then a double will be created from the Long
	datum by casting. If the Value is a <CODE>STRING</CODE> Type, then
	an attempt will be made to interpret the String as a double value.
<P>
	@return	A double value from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is not a <CODE>NUMERIC</CODE> Type,
				or a <CODE>STRING</CODE> Type could not be converted
				to a double.
		</DL>
	@see	Double#parseDouble(String)
*/
public double double_Data ()
	throws PVL_Exception
{
if (Is_Numeric ())
	{
	if (_Datum_ == null)
		return 0.0;
	if (Is_Real ())
		return ((Double)_Datum_).doubleValue ();
	return ((Long)_Datum_).doubleValue ();
	}
if (Is_String ())
	{
	if (_Datum_ == null)
		return 0.0;
	try {return Double.parseDouble((String)_Datum_);}
	catch (NumberFormatException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.ILLEGAL_SYNTAX,
			"Can't create a double value from the " + Type_Name ()
				+ " Value \"" + (String)_Datum_ + "\""
			);
		}
	}
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a double value from a Value of Type " + Type_Name () + "."
	);
}

/**	Gets a String object representing the Value datum.
<P>
	If the Value is a <CODE>STRING</CODE> Type then the unmodified
	datum is returned, unless the datum is null in which case an empty
	String is returned. No special enclosing characters associated with
	the PVL syntax for specific <CODE>STRING</CODE> Types are added
	(use the <CODE>toString</CODE> method for this format).
<P>
	If the Value is a <CODE>NUMERIC</CODE> Type it is converted to its
	String representation. <B>Note</B>: For an <CODE>INTEGER</CODE>
	Type Value it's current radix base is used in the conversion (the
	representation is preceeded by a negative sign if the datum value
	is negative, which avoids machine precision and byte ordering
	issues). The radix base value and PVL syntax characters are not
	included (use the <CODE>toString</CODE> method for this format).
<P>
	@return	A String object from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is an <CODE>ARRAY</CODE> Type.
		</DL>
	@see	#toString()
	@see	Long#toString(long, int)
	@see	Double#toString()
*/
public String String_Data ()
	throws PVL_Exception
{
if (Is_String ())
	return (_Datum_ == null) ?
		"" :  (String)_Datum_;
if (Is_Integer ())
	{
	long
		number = (_Datum_ == null) ?
			0 : ((Long)_Datum_).longValue ();
	int
		base = _Base_;
	if (base < 0)
		{
		base = -base;
		number = -number;
		}
	return Long.toString (number, base);
	}
if (Is_Real ())
	return (_Datum_ == null) ?
		"0.0" : ((Double)_Datum_).toString ();
if (Is_Unknown ())
	return "";
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a String from a Value of Type " + Type_Name () + "\n"
	+ "  (toString will always return a String)."
	);
}

/**	Gets the Value at the specified index of the Array.
<P>
	@param	index	The Array index of the Value to get.
	@return	A Value. 
		If this Value is not an Array, or has no data Vector, null is
		returned. If this Value is not an Array a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> Warning status is
		set.
	@see	Vector#remove(int)
*/
public Value Get
	(
	int			index
	)
{
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println
		(">>> Value.Get: from " + toString () + " at index " + index);
Value
	got = null;
if (! Is_Array ())
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Get an indexed Value from the Value of Type " + Type_Name () + "."
		);
if (children != null &&
	index < children.size () &&
	index >= 0)
	//	Get the entry from the data Vector.
	got = (Value)children.get (index);
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println
		("<<< Value.Get - "
			+ ((got == null) ? "(nothing)" : got.toString ()));
return got;
}

/**	Gets the data Vector of an Array Value.
<P>
	@return	The data Vector from the Value.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Value is not <CODE>ARRAY</CODE> Type.
		</DL>
*/
public Vector Vector_Data ()
	throws PVL_Exception
{
if (Is_Array ())
	{
	if (children == null)
		children = new Vector (0);
	return children;
	}
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a Vector from a Value of Type " + Type_Name () + "."
	);
}

/**	Gets the number of elements in an Array Value.
<P>
	@return	The number of Values in the Array. If the
		Value is not an Array or has no elements, zero will be
		returned; i.e. returning zero is not determinate that the
		Value is an Array with an empty list. <b>N.B.</b>: This
		value does not include the number of Values in any
		Arrays Values this Array may contain.
*/
public int Array_Size ()
{
if (Is_Array () && children != null)
	return children.size ();
return 0;
}

/**	Gets the datum object from the Value.
<P>
	If the Value is an <CODE>ARRAY</CODE> Type, the object will be a
	Vector. The object will be a Long for an <CODE>INTEGER</CODE> Type,
	a Double for a <CODE>REAL</CODE>, or a String for a
	<CODE>STRING</CODE>.
<P>
	@return	An object of the appropriate class for the Value.
*/
public Object Data ()
{
if (Is_Array ())
	return children;
return _Datum_;
}

/*..............................................................................
	Set
*/
/**	Sets the datum of the Value to the Long object, and sets its Type
	to <CODE>INTEGER</CODE>. <B>Note</B>: A null datum is acceptable.
<P>
	@param	value	The new datum for the Value.
	@return	This Value.
	@see	#set_type(int)
	@see	#set_data(Object)
*/
public Value Data
	(
	Long		value
	)
{
set_type (INTEGER);
try {set_data (value);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Sets the datum of the Value to the primitive byte value,
	and sets its Type to <CODE>INTEGER</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Long)
*/
public Value Data (byte  value) {return Data (new Long ((long)value));}

/**	Sets the datum of the Value to the primitive short value,
	and sets its Type to <CODE>INTEGER</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Long)
*/
public Value Data (short value) {return Data (new Long ((long)value));}

/**	Sets the datum of the Value to the primitive int value,
	and sets its Type to <CODE>INTEGER</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Long)
*/
public Value Data (int   value) {return Data (new Long ((long)value));}

/**	Sets the datum of the Value to the primitive long value,
	and sets its Type to <CODE>INTEGER</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Long)
*/
public Value Data (long  value) {return Data (new Long (value));}


/**	Sets the datum of the Value to the Double object, and sets its Type
	to <CODE>REAL</CODE>. <B>Note</B>: A null datum is acceptable.
<P>
	@param	value	The new datum for the Value.
	@return	This Value.
	@see	#set_type(int)
	@see	#set_data(Object)
*/
public Value Data
	(
	Double		value
	)
{
set_type (REAL);
try {set_data (value);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Sets the datum of the Value to the primitive float value,
	and sets its Type to <CODE>REAL</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Double)
*/
public Value Data (float  value) {return Data (new Double ((double)value));}

/**	Sets the datum of the Value to the primitive double value,
	and sets its Type to <CODE>REAL</CODE>.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(Double)
*/
public Value Data (double value) {return Data (new Double (value));}


/**	Sets the datum of the Value to the String object.
<P>
	If the String begins with the {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} character then the Value Type
	is set to <CODE>TEXT</CODE>. If the String begins with the {@link
	Parser#SYMBOL_DELIMITER <CODE>Parser.SYMBOL_DELIMITER</CODE>}
	character then the Value Type is set to <CODE>SYMBOL</CODE>. In
	either of these cases the leading and trailing (if present)
	delimiter characters are removed from the String before it is
	assigned to the datum. Otherwise, if the String contains any {@link
	Parser#DATE_TIME_DELIMITERS
	<CODE>Parser.DATE_TIME_DELIMITERS</CODE>} the Value Type is set to
	<CODE>DATE_TIME</CODE>. By default the Value Type is set to
	<CODE>IDENTIFIER</CODE>. <B>Note</B>: A null datum is acceptable.
<P>
	If the String (after removing any enclosing delimiters) contains
	any of the {@link Parser#RESERVED_CHARACTERS
	<CODE>Parser.RESERVED_CHARACTERS</CODE>} then a warning status is
	set. These characters have special meaning in the PVL syntax so
	writing values containing them could confuse a PVL parser that
	reads them.
<P>
	@param	value	The new datum for the Value.
	@return	This Value.
	@see	#set_type(int)
	@see	#set_data(Object)
*/
public Value Data
	(
	String		value
	)
{
set_type (IDENTIFIER);	//	Even if the String value is null.
if (value != null && value.length () > 0)
	{
	int
		index;
	try
		{
		if (value.charAt (0) == Parser.TEXT_DELIMITER)
			{
			set_type (TEXT);
			value = value.substring (1);
			if (value.charAt (value.length () - 1) == Parser.TEXT_DELIMITER)
				value = value.substring (0, value.length () - 1);
			}
		else if (value.charAt (0) == Parser.SYMBOL_DELIMITER)
			{
			set_type (SYMBOL);
			value = value.substring (1);
			if (value.charAt (value.length () - 1) == Parser.SYMBOL_DELIMITER)
				value = value.substring (0, value.length () - 1);
			}
		else if (value.length () > 0)
			{
			for (index = 0;
				 index < Parser.DATE_TIME_DELIMITERS.length ();
				 index++)
				{
				if (value.indexOf (Parser.DATE_TIME_DELIMITERS.charAt (index)) >= 0)
					{
					set_type (DATE_TIME);
					break;
					}
				}
			}

		//	Check for reserved characters in the String.
		if ((index = Parser.Bad_Character (value)) >= 0)
			Warning
				(
				PVL_Exception.RESERVED_CHARACTER,
				"At character " + index + " of the Value Data String."
				);
		}
	catch (IndexOutOfBoundsException exception) {/* No problem */}
	}
try {set_data (value);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	The primitive char value is used to  constructs a single character
	String which is then set to the datum of the Value.
<P>
	The Type of the Value is set according to the rules of the {@link
	#Data(String) <CODE>Data (String)</CODE>} method.
<P>
	@param	value	The new value for the Value.
	@return	This Value.
	@see	#Data(String)
*/
public Value Data
	(
	char		value
	)
{
char character[] = {value};
return Data (new String (character));
}

/**	Sets the Value data as an Array of Values constructed from the List.
<P>
	The Value is classified as the {@link #Default_Array_Type() default
	Array Type}. A new data Vector is assembled containing Values
	constructed from the Objects in the list. If any of these Objects are
	Values they are copied. An Object that is a List results in a new
	Array Value being recursivley constructed and added to the new data
	Vector.
<P>
	If this Value previously had data, it is replaced with the new Vector
	of Values. In the case where the previous data was a Vector of
	Values, all Values of the old Vector are orphaned (they are
	marked as having no parent).
<P>
	@param	list	The List of values to be adopted.
	@return	This Parameter.
	@throws	PVL_Exception	From set_data.
	@see	#set_type(int)
	@see	#set_data(Object)
*/
public Value Data
	(
	List		list
	)
	throws PVL_Exception
{
set_type (Default_Array_Type);
if (list != null)
	{
	Iterator
		entries = list.iterator ();
	list = new Vector (list.size ());
	while (entries.hasNext ())
		list.add (new Value (entries.next ()));
	}
set_data (list);
return this;
}

/**	Copies the contents of a Value into this Value.
<P>
	@param	value	The Value to be copied into this Value.
	@return	This Value.
	@throws	PVL_Exception	From making a copy of the value.
	@see	#Value(Value)
	@see	#graft(Value)
*/
public Value Data
	(
	Value		value
	)
	throws PVL_Exception
{
//	Graft a deep copy of the original Value.
graft (new Value (value));
return this;
}

/**	Sets the datum for the Value based on the class of the
	specified Object.
<P>
	There are numerous possibilities:
<P>
<DL>
<DT>Value
	<DD>The Value object is copied into this Value using the
	{@link #Data(Value) <CODE>Data (Value)</CODE>} method.
<DT><CODE>ARRAY</CODE>
	<DD>The Value is assigned a Vector of Values using the
	{@link #Data(List) <CODE>Data (List)</CODE>} method.
	The object classes for this case are:
		<UL TYPE="DISC">
			<LI>List - The List object is used.
			<LI>Value[] - A new Vector is constructed from the array of Values.
		</UL>
<DT><CODE>INTEGER</CODE>
	<DD>The Value is assigned a Long using the
	{@link #Data(Long) <CODE>Data (Long)</CODE>} method.
	The object classes for this case are:
		<UL TYPE="DISC">
			<LI>Long - The Long object itself is used.
			<LI>Byte - A new Long is constructed from the object's
				<CODE>longValue</CODE> method.
			<LI>Short - A new Long is constructed from the object's
				<CODE>longValue</CODE> method.
			<LI>Integer - A new Long is constructed from the object's
				<CODE>longValue</CODE> method.
		</UL>
<DT><CODE>REAL</CODE>
	<DD>The Value is assigned a Double using the
	{@link #Data(Double) <CODE>Data (Double)</CODE>} method.
	The object classes for this case are:
		<UL TYPE="DISC">
			<LI>Double - The Double object itself is used.
			<LI>Float - A new Double is constructed from the object's
				<CODE>doubleValue</CODE> method.
		</UL>
<DT><CODE>STRING</CODE>
	<DD>The Value is assigned a String using the
	{@link #Data(String) <CODE>Data (String)</CODE>} method.
	The object classes for this case are:
		<UL TYPE="DISC">
			<LI>String - The String object itself is used.
			<LI>Character - A String is obtained from the object's
				<CODE>toString</CODE> method.
			<LI>char[ ] - A new String is constructed from the object.
			<LI>byte[ ] - A new String is constructed from the object.
		</UL>
<DT><CODE>UNKNOWN</CODE>
	<DD>The object is null so the Value datum will be null.
</DL>
<P>
	@param	object	The Object to be considered a the Value datum.
	@return	This Value.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>The Object is not suitable for a Value datum.
		</DT>
*/
public Value Data
	(
	Object		object
	)
	throws PVL_Exception
{
if (object == null)
	{
	set_type (UNKNOWN);
	set_data (object);
	}

//	Value:
else if (Is_Value (object))
	Data ((Value)object);

//	ARRAY:
else if (object instanceof List)
	Data ((List)object);
else if (object instanceof Value[])
	Data (new Vector (Arrays.asList ((Value[])object)));
else if (object instanceof Object[])
	Data (new Vector (Arrays.asList ((Object[])object)));

//	NUMERIC:
else if (object instanceof Long)
	Data ((Long)object);
else if (object instanceof Byte)
	Data (new Long (((Byte)object).longValue ()));
else if (object instanceof Short)
	Data (new Long (((Short)object).longValue ()));
else if (object instanceof Integer)
	Data (new Long (((Integer)object).longValue ()));

else if (object instanceof Double)
	Data ((Double)object);
else if (object instanceof Float)
	Data (new Double (((Float)object).doubleValue ()));

//	STRING:
else if (object instanceof String)
	Data ((String)object);
else if (object instanceof Character)
	Data (((Character)object).toString ());
else if (object instanceof char[])
	Data (new String ((char[])object));
else if (object instanceof byte[])
	Data (new String ((byte[])object));

else
	throw new PVL_Exception
		(
		ID, PVL_Exception.BAD_ARGUMENT,
		"Invalid Data Object for a Value: "
			+ object.getClass ().getName () + "."
		);
return this;
}

/*..............................................................................
	Modify
*/
/**	Inserts a Value into the Array at the specified index.
<P>
	If the Value is not already an Array Type, and does not have a datum,
	it is given the {@link #Default_Array_Type() default Array Type}. The
	specified Value is inserted at the specified index in this Value's
	Array. Any Values in the Array at, or beyond the index are moved to
	the next index. After the specified Value is inserted its parent is
	set to this Value. <B>Note</B>: The inserted Value is not removed
	from any other Array it may be in.
<P>
	@param	value	The Value object to be inserted.
	@param	index	The location in the Array for the insertion.
	@return	This Value.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>The Value has a non-Array datum.
		</DL>
	@see	#Type(int)
	@see	#Remove(Value)
	@see	Vector#insertElementAt(Object, int)
*/
public Value Insert
	(
	Value		value,
	int			index
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println
		(">>> Value.Insert: "
			+ value + " into " + toString () + " at index " + index);
if ((Is_Numeric () || Is_String ()) && _Datum_ != null)
	throw new PVL_Exception
		(
		ID, PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Insert into the " + Type_Name () + " Value (" + toString () + ")."
		);
if (value != null)
	{
	if (! Is_Array ())
		Type (Default_Array_Type);

	//	Adopt it as one of the children.
	if (children == null)
		children = new Vector ();
	try {children.insertElementAt (value, index);}
	catch (RuntimeException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.SYSTEM_ERROR,
			"During Insert at index " + index
				+ " of the " + value.Type_Name ()
				+ " Value (" + value + ")\n"
			+ exception.getMessage ()
			);
		}
	value.parent = this;
	}
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println ("<<< Value.Insert");
return this;
}

/**	Insert an Object constructed as a new Value at the specified
	index of the Array.
<P>
	If the Object is a Value it is not copied.
<P>
	@param	object	The Object whose Value is to be inserted.
	@param	index	The location in the Array for the insertion.
	@return	This Value.
	@throws	PVL_Exception	From Insert(Value, int).
	@see	#Value(Object)
	@see	#Insert(Value, int)
*/
public Value Insert
	(
	Object		object,
	int			index
	)
	throws PVL_Exception
{
if (Is_Value (object))
	return Insert ((Value)object, index);
return Insert (new Value (object), index);
}

/**	Inserts a List of Objects into the Value's Array starting at the
	specified index.
<P>
	The elements of the List are <CODE>Insert</CODE>ed in the order
	they occur from the List's listIterator at sequential locations
	starting with the specified index.
<P>
	@param	list	The List of Objects to Insert.
	@param	index	The first of sequential Insert locations.
	@return	This Value.
	@throws	PVL_Exception	From the construction of a new Value from
		an Object.
	@see	#Insert(Object, int)
	@see	List#listIterator()
*/
public Value Insert
	(
	List		list,
	int			index
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println (">>> Value.Insert: List at index " + index);
if (list != null)
	{
	ListIterator
		value_list = list.listIterator ();
	while (value_list.hasNext ())
		{
		/*	Try to convert the object into a Value,
			or make a copy if it is already a Value.
		*/
		if ((DEBUG & DEBUG_MODIFY) != 0)
			System.out.println ("Value.Insert: List object element " + index);
		Insert (value_list.next (), index++);
		}
	}
return this;
}

/**	Adds a Value to the end of this Array.
<P>
	If this Value is not currently an Array an attempt will be made to
	convert it to the {@link #Default_Array_Type() default Array Type}.
	The <CODE>Insert</CODE> method is used with the index at the end of
	the Array.
<P>
	<B>N.B.</B>: The {@link #Array_Size() Array size} will only be
	increased by one even if the Value being added is an Array; in this
	case the Array Value being added becomes a sub-Array whose parent is
	this Value. To append the entries of a Value Array to this Value's
	list of entries add the Value Array's {@link #Vector_Data() list} to
	this Value.
<P>
	@param	value	The Value to be added.
	@return	This Value.
	@throws	PVL_Exception	If <CODE>Type</CODE> could not convert the
		Value to an Array, or there was a problem with the
		<CODE>Insert</CODE> of the value at the end of the Array.
	@see	#Type(int)
	@see	#Insert(Value, int)
*/
public Value Add
	(
	Value		value
	)
	throws PVL_Exception
{
if (! Is_Array ())
	Type (Default_Array_Type);		//	Try to become an Array.
return Insert (value, getChildCount ());
}

/**	Adds an Object constructed as a new Value to the end of this Array.
<P>
	If the Object is a Value it is not copied.
<P>
	@param	object	The Object whose Value is to be inserted.
	@return	This Value.
	@throws	PVL_Exception	From <CODE>Add(Value)</CODE>.
	@see	#Value(Object)
	@see	#Add(Value)
*/
public Value Add
	(
	Object		object
	)
	throws PVL_Exception
{
if (Is_Value (object))
	return Add ((Value)object);
return Add (new Value (object));
}

/**	Adds a List of Objects to the end of the Array.
<P>
	If this Value is not currently an Array an attempt will be made to
	convert it to the {@link #Default_Array_Type() default Array Type}.
	The <CODE>Insert</CODE> method is used with the index at the end of
	the Array.
<P>
	@param	list	The List of Objects to Insert.
	@return	This Value.
	@throws	PVL_Exception	If <CODE>Type</CODE> could not convert the
		Value to an Array, or there was a problem with the
		<CODE>Insert</CODE> of the list at the end of the Array.
	@see	#Type(int)
	@see	#Insert(List, int)

*/
public Value Add
	(
	List		list
	)
	throws PVL_Exception
{
if (! Is_Array ())
	Type (Default_Array_Type);		//	Try to become an Array.
return Insert (list, getChildCount ());
}

/**	Removes the Value at the specified index from the Array.
<P>
	After being removed from the Array the Value is marked as having no
	parent.
<P>
	@param	index	The Array index of the Value to be removed.
	@return	The Value removed. 
		If this Value is not an Array, or has no Array, null is
		returned. If this Value is not an Array a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> Warning status is
		set.
	@see	Vector#remove(int)
*/
public Value Remove
	(
	int			index
	)
{
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println
		(">>> Value.Remove: from " + toString () + " at index " + index);
Value
	removed = null;
if (! Is_Array ())
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Remove from the Value of Type " + Type_Name () + "."
		);
if (children != null)
	{
	//	Remove the entry from the data Vector.
	removed = (Value)children.remove (index);
	//	Make it an orphan.
	removed.parent = null;
	if ((DEBUG & DEBUG_MODIFY) != 0)
		System.out.println
			("Value.Remove - ");
	}
if ((DEBUG & DEBUG_MODIFY) != 0)
	System.out.println
		("<<< Value.Remove - "
			+ ((removed == null) ? "(nothing)" : removed.toString ()));
return removed;
}

/**	Removes the specified Value from the Array.
<P>
	The first occurance of the Value in the Array is removed. The Array,
	if there is one, is searched regardless of the Type of this Value.
	The search is not recursive: Only this Value's Array data Vector is
	searched, not the Array Vectors of any Array Values in the Array. The
	Value is found in the Array by equality of Value object reference,
	not by the equals method. To remove a Value that equals another
	Value, use <CODE>Find</CODE> and then remove the found Value from its
	<CODE>Parent</CODE>.
<P>
	@param	value	The Value object to be removed.
	@return	The Value removed. 
		If this Value is not an Array, or has no Array, null is
		returned. If this Value is not an Array a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> Warning status is
		set.
	@see	#Remove(int)
*/
public Value Remove
	(
	Value		value
	)
{
if (value != null &&
	children != null)
	for (int index = 0;
		 index < children.size ();
		 index++)
		//	Remove the first occurances.
		if (children.get (index) == value)
			return Remove (index);
return null;
}

/**	Empties the Value's Array.
<P>
	The Array data Vector, if it exists, is removed regardless of the Type
	of this Value. All Values in the Array (but not sub-Arrays) are
	marked as not having a parent.
<P>
	@return	This Value.
*/
public Value Remove_All ()
{
orphan (children);
children = null;
return this;
}

/*..............................................................................
*/
/**	Set the appropriate Data variables.
<P>
	The <CODE>Type</CODE> of the data object determines how the
	internal data variables are set. Consistency of the Value's data is
	ensured by always setting it with this method.
<P>
	For Vector data, the data becomes the base class children data
	Vector. Its contents are adopted which confirms that each element
	is a Value and its parent is set to this Value. The local datum is
	set to null.
<P>
	For non-<CODE>ARRAY</CODE> data, the data becomes the datum of the
	Value. The children data Vector is orphaned which sets all of its
	Value elements to have no parent, and then set to null.
<P>
	<B>>>> WARNING <<<</B> The Type of the Value is not updated.
	Accordingly, this method should only be used by methods associated
	with this package that will also set the Type appropriately.
<P>
	@param	data	The Object to become Value's data.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If the data object is not a valid class for a Value
				(null is acceptable), or the data object is a Vector and
				one of its elements is not a Value.
		</DL>
	@see	#Type(Object)
	@see	#adopt(Vector)
	@see	#orphan(Vector)
*/
protected void set_data
	(
	Object		data
	)
	throws PVL_Exception
{
switch (Type (data))
	{
	case UNKNOWN:	//	The data is null.
	case INTEGER:
	case REAL:
	case STRING:
		orphan (children);
		children = null;
		_Datum_ = data;
		if ((DEBUG & DEBUG_DATA) != 0)
			System.out.println ("Value.set_data: _Datum_");
		break;
	case ARRAY:
		/*
			N.B.: Each DefaultMutableTreeNode contains a parent
			reference which must be updated when it is moved into a new
			parent. So transfering a Value list into this Value
			requires setting the parent of each Value in the list.
		*/
		children = (Vector)data;
		adopt (children);
		_Datum_ = null;
		if ((DEBUG & DEBUG_DATA) != 0)
			System.out.println ("Value.set_data: ARRAY");
		break;
	default:
	throw new PVL_Exception
		(
		ID, PVL_Exception.BAD_ARGUMENT,
		"Can't set_data of a Value to an object of class "
			+ data.getClass ().getName () + "."
		);
	}
}

/**	Grafts the contents of a Value into this Value.
<P>
	In effect, this Value becomes the specified Value.
<P>
	The datum reference is set (not copied) to the specified Value's
	datum reference along with the base, units, and warning status. The
	Array (children) data Vector reference is set to the specified
	Value's Vector and if it has contents they are adopted by this Value
	(the specified Value's children data Vector is set to null to avoid
	any possible confusion). The Type is set to be the same as the
	specified Value.
<P>
	@param	value	The Value object to graft into this Value.
	@throws	PVL_Exception	From adopt.
	@see	#adopt(Vector)
	@see	#set_type(int)
*/
protected void graft
	(
	Value		value
	)
	throws PVL_Exception
{
if (value != null)
	{
	//	Graft the contents of the Value into this Value.
	_Datum_ = value._Datum_;
	if ((children = value.children) != null)
		{
		//	Become the parent of the children.
		value.children = null;
		adopt (children);
		}
	set_type (value._Type_);
	_Base_ = value._Base_;
	_Units_ = value._Units_;
	_First_Warning_ = value._First_Warning_;
	_Last_Warning_ = value._Last_Warning_;
	_Use_First_Warning_ = value._Use_First_Warning_;
	}
}

/**	Sets the parents of a Vector of Values to this Value.
<P>
	<B>>>> WARNING <<<</B>
<P>
	Only the parent references for the Values in the Vector are set to
	this Value. The Vector itself may still be referenced by some other
	Value. In this case the adoption of the Vector will not be complete
	until the reference in the former parent is changed (e.g. to null).
<P>
	@param	vector	The Vector of Parameters to adopt.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an entry in the list is not a Value.
		</DL>
*/
protected void adopt
	(
	Vector		vector
	)
	throws PVL_Exception
{
Object
	value;
if (vector != null)
	{
	for (int index = vector.size ();
		   --index >= 0;)
		{
		if (Is_Value (value = vector.get (index)))
			//	Set the child's parent to this value.
			((Value)value).parent = this;
		else
			throw new PVL_Exception
				(
				ID, PVL_Exception.BAD_ARGUMENT,
				"Can't adopt an object from a List\n"
				+ "  element " + index + " is an object of class "
					+ value.getClass ().getName () + "."
				);
		}
	}
}

/**	Sets a Vector of Values to have no parent.
<P>
	<B>>>> WARNING <<<</B>
<P>
	Only the parent references for the Values in the Vector are set to
	null. The Vector itself may still be referenced by some other
	Value. In this case the orphaning of the Vector will not be
	complete until the reference in the former parent is changed (e.g.
	to null).
<P>
	@param	vector	The Vector of Values to orphan.
*/
static protected void orphan
	(
	Vector		vector
	)
{
Object
	value;
if (vector != null)
	for (int index = vector.size ();
		   --index >= 0;)
		if (Is_Value (value = vector.get (index)))
			((Value)value).parent = null;
}

/*------------------------------------------------------------------------------
	Type:
*/
/**	Gets the Value's Type code.
<P>
	@return	The Type code.
*/
public int Type ()
{return _Type_;}

/**	Determine the Value Type of an Object.
<P>
	The Object is considered to be potentially the datum for a Value and
	is tested to see if it is an instance of an acceptable Java Class.
	The object must be an instance of a class suitable for being assigned
	directly as the datum or data Vector of a Value. Only a basic Type
	can be determined by examining an object, except for the specific
	<CODE>NUMERIC</CODE> Types. <B>Note</B>: If this method does not
	provide a valid Type that does not necessarily mean that the object
	can not be used by the <CODE>Data (Object)</CODE> method.
<P>
	@param	object	The Object to be examined.
	@return	The int basic Type code associated with a Value in
		which the object is its datum. Long objects return the
		<CODE>INTEGER</CODE> code, Double objects the <CODE>REAL</CODE>
		code, String objects the <CODE>STRING</CODE> code, and List
		objects return the <CODE>ARRAY</CODE> code. When the object is
		null, the <CODE>UNKNOWN</CODE> code is returned. If an invalid
		object is specified, then the return value is -1.
	@see	#Data(Object)
*/
public static int Type
	(
	Object	object
	)
{
if (object instanceof Long)
	return INTEGER;
else if (object instanceof Double)
	return REAL;
else if (object instanceof String)
	return STRING;
else if (object instanceof List)
	return ARRAY;
else if (object == null)
	return UNKNOWN;
return -1;
}

/**	Sets the Type of the Value.
<P>
	When the Value does not have any data (neither a datum nor a
	data Vector), or the current Type is <CODE>UNKNOWN</CODE>, then the
	Value is simply set to the specifed Type. When the new Type is
	<CODE>UNKNOWN</CODE> then any existing data is removed with the
	{@link #set_data(Object) <CODE>set_data (null)</CODE>} method
	and then the Type is set.
<P>
	When the current Value Type is the same basic Type as the new Type
	- i.e. they are both <CODE>NUMERIC</CODE>, <CODE>STRING</CODE> or
	<CODE>ARRAY</CODE> Types - then the Value is set to the new Type.
	In the case of a Type change from <CODE>INTEGER</CODE> to
	<CODE>DOUBLE</CODE>, or vice versa, the datum is cast to the new
	Type.
<P>
	When the Value is a <CODE>STRING</CODE> Type and the new Type is
	<CODE>NUMERIC</CODE> an attempt is made to parse the String using
	the Value's <CODE>Double_Data</CODE> method to produce a
	<CODE>REAL</CODE> Type datum, or the <CODE>Long_Data</CODE> method
	to produce an <CODE>INTEGER</CODE> Type datum. In the latter case
	the current radix base of the Value will be used in the conversion.
	If the String datum is not a valid representation of the
	appropriate Type of number then a
	<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> will be thrown.
<P>
	When the Value is a <CODE>NUMERIC</CODE> Type and the new Type is
	<CODE>STRING</CODE> the datum is converted to its String
	representation using the Value's <CODE>String_Data</CODE> method.
	For an <CODE>INTEGER</CODE> Type the current radix base of the
	Value will be used in the conversion.
<P>
	When the Value is to be converted to an <CODE>ARRAY</CODE> Type the
	current Value is {@link #clone() <CODE>clone</CODE>}d, the current
	Value is set to the (empty) {@link #Default_Array_Type() default
	Array Type}, and the copy of the original Value is {@link
	#Insert(Value, int) <CODE>Insert</CODE>}ed as the first element of
	the Array. In effect the original Value is "pushed down" as the first
	and only element of what has become an Array Value; i.e. the new
	Array contains a copy of the original Value.
<P>
	When the Value is an <CODE>ARRAY</CODE> Type and the new Type is
	not, this conversion can only be accomplished if the current Array
	contains less than two Values. If the Array contains no Values then
	the new Type is simply set. If the Array contains two or more
	Values then a <CODE>PVL_Exception.INCOMPATIBLE_TYPES</CODE> will be
	thrown. For an Array with just one Value an attempt is made to
	convert that Value to the new Type by recursively calling this
	<CODE>Type</CODE> method. If this succeeds then the converted value
	is {@link #graft(Value) <CODE>graft</CODE>}ed into this Value to
	become the new Type. In effect a single Value contained in an Array
	is "pulled up" into this Value; i.e. the Array's Value becomes this
	Value with the new Type.
<P>
	@param	type	The new Type for the Value.
	@return	This Value.
	@throws	PVL_Exception
		<DL>
		<DT>INCOMPATIBLE_TYPES
			<DD>An Array Value could not be converted to a non-Array
				value because the Array containes more than one Value,
				or the Array data Vector unexpectedly contains a non-Value
				entry.
		<DT>ILLEGAL_SYNTAX
			<DD>A String datum could not be converted to a number because
				the String is not a valid represenation of the specified
				numerical Type.
		<DT>BAD_ARGUMENT
			<DD>Conversion to or from an unsupported Value Type.
		</DL>
	@see	#NUMERIC
	@see	#STRING
	@see	#ARRAY
	@see	#set_type(int)
	@see	Long#doubleValue()
	@see	Double#longValue()
	@see	#clone()
	@see	#graft(Value)
	@see	#Long_Data()
	@see	#Double_Data()
	@see	#String_Data()
*/
public Value Type
	(
	int		type
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_TYPE) != 0)
	System.out.println
		(">>> Value.Type: From " + Type_Name () + " to " + Type_Name (type));
if (Data () == null)
	//	There's no data, so just set the new Type.
	set_type (type);
	
else if (_Type_ != type)
	{
	//	No change to the basic Type:

	if ((Is_Array () && Is_Array (type)) ||
		(Is_String () && Is_String (type)))
		//	Decorative change.
		set_type (type);
	else if (Is_Numeric () && Is_Numeric (type))
		{
		//	Type casting.
		if (Is_Integer ())	//	then type is REAL.
			_Datum_ = new Double (((Long)_Datum_).doubleValue ());
		else				//	then type is INTEGER.
			_Datum_ = new Long (((Double)_Datum_).longValue ());
		if ((DEBUG & DEBUG_TYPE) != 0)
			System.out.println
				("Type: cast");
		set_type (type);
		}

	//	Basic Type change:

	else if (Is_Unknown ())
		//	From nothing to something.
		set_type (type);
	else if (Is_Unknown (type))
		{
		//	From something to nothing.
		set_data (null);	//	Empty the Value.
		set_type (type);
		}
	else if (Is_Array (type))
		{
		//	From a single Value into an Array.
		Value
			value = (Value)clone ();
		orphan (children);
		children = null;
		set_type (type);
		Insert (value, 0);
		if ((DEBUG & DEBUG_TYPE) != 0)
			System.out.println
				("Type: cloned into new Array");
		}
	else if (Is_Array ())
		{
		//	From an Array into a single Value.
		if (children.size () == 0)
			{
			//	Just toss the empty Array.
			children = null;
			set_type (type);
			}
		else
			{
			if (children.size () == 1)
				{
				//	The Array has only one Value.
				Object
					value = children.get (0);
				if (! Is_Value (value))
					throw new PVL_Exception
						(
						ID, PVL_Exception.INCOMPATIBLE_TYPES,
						"During Value Type change from " + Type_Name ()
							+ " to " + Type_Name (type) + "\n"
						+ "  an object of class " + value.getClass ().getName ()
							+ " was found!"
						);
				/*	Convert it to the new non-Array Type.
					This will delve into any sub-Arrays.
				*/
				((Value)value).Type (type);
				orphan (children);
				children = null;		//	The child has grown up.
				graft ((Value)value);	//	It becomes the parent.
				if ((DEBUG & DEBUG_TYPE) != 0)
					System.out.println
						("Type: grafted the Array value");
				}
			else
				throw new PVL_Exception
					(
					ID, PVL_Exception.INCOMPATIBLE_TYPES,
					"Can't convert the ARRAY Value to Type "
						+ Type_Name (type) + "\n"
					+ "  because it contains more than one Value."
					);
			}
		}
	else if (Is_String ())
		{
		//	From a String to the number it represents.
		if (Is_Integer (type))
			_Datum_ = Long_Data ();
		else
			_Datum_ = Double_Data ();
		set_type (type);
		if ((DEBUG & DEBUG_TYPE) != 0)
			System.out.println
				("Type: converted to a number");
		}
	else if (Is_Numeric ())
		{
		//	From a number to its String representation.
		_Datum_ = String_Data ();
		set_type (type);
		if ((DEBUG & DEBUG_TYPE) != 0)
			System.out.println
				("Type: converted to a string");
		}
	else
		//	This shouldn't happen if all the combinations are covered.
		throw new PVL_Exception
			(
			ID, PVL_Exception.BAD_ARGUMENT,
			"Can't convert from a Value of Type " + Type_Name ()
				+ " to Type " + Type_Name (type)
			);
	}
if ((DEBUG & DEBUG_TYPE) != 0)
	System.out.println
		("<<< Type");
return this;
}

/**	Sets the appropriate Type variables.
<P>
	Consistency of the Value's Type is ensured by always setting it
	with this method. If the Type code is for an <CODE>ARRAY</CODE>,
	then the allowsChildren variable of the DefaultMutableTreeNode base
	class will be set to true; otherwise it will be set to false.
<P>
	<B>>>> WARNING <<<</B> The data of the Value is not checked for
	consistency with the Type. Accordingly, this method should only be
	used by methods associated with this package that will also set the
	data appropriately.
<P>
	@param	type	The int Type code for the Value.
*/
protected void set_type
	(
	int			type
	)
{
if ((DEBUG & DEBUG_TYPE) != 0)
	System.out.println
		("Value.set_type: from " + Type_Name () + " to " + Type_Name (type));
_Type_ = type & MASK;
if (Is_Array ())
	allowsChildren = true;
else
	allowsChildren = false;
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("Value.set_type: allowsChildren = " + allowsChildren);
}

/**	Gets the name String associated with the Type code.
<P>
	Type names are identical to the names of the Type code constants.
<P>
	@param	type	The Type code int.
	@return	The appropriate Type name String. An invalid type code will
		return a null.
*/
public static String Type_Name
	(
	int		type
	)
{
switch (type)
	{
	case INTEGER:		return "INTEGER";
	case REAL:			return "REAL";
	case STRING:		return "STRING";
	case TEXT:			return "TEXT";
	case IDENTIFIER:	return "IDENTIFIER";
	case SYMBOL:		return "SYMBOL";
	case DATE_TIME:		return "DATE_TIME";
	case ARRAY:			return "ARRAY";
	case SET:			return "SET";
	case SEQUENCE:		return "SEQUENCE";
	case UNKNOWN:		return "UNKNOWN";
	default:			return null;
	}
}

/**	Gets the name String associated with the Value's Type.
<P>
	@return	The appropriate Type name String.
*/
public String Type_Name ()
{return Type_Name (_Type_);}

/**	Sets the default Array Value Type to use when an Array is
	automatically constructed.
<p>
	@param	type	The Array Type to use by default. If this is not an
		{@link #Is_Array(int) Array Type} nothing is done.
	@return	The previous value.
*/
public static int Default_Array_Type
	(
	int		type
	)
{
int
	previous_type = Default_Array_Type;
if (Is_Array (type))
	Default_Array_Type = type;
return previous_type;
}

/**	Gets the default Array Value Type to use when an Array is
	automatically constructed.
<p>
	@return	The Array Type to use by default.
	@see	#Default_Array_Type(int)
*/
public static int Default_Array_Type ()
{return Default_Array_Type;}

/*..............................................................................
	Type checkers:
*/
/**	Tests if an Object is a Value.
<P>
	@param	object	The Object to be tested.
	@return	true if the object is an instance of the Value class;
		otherwise false (also false if the object is null).
*/
public static boolean Is_Value (Object object)
{return (object instanceof Value);}

/**	Tests if an Object is a List suitable for an Array Value.
<P>
	<B>Note</B>: The contents of the List are recursively checked to all
	be Values. null data Vectors are acceptable Values.
<P>
	@param	object	The Object to be tested.
	@return	true if the object is an instance of List and all its
		elements are Values, or the object is null; otherwise false.
	@see	#Is_Value(Object)
	@see	#Is_Array()
*/
public static boolean Is_Value_Array (Object object)
{
if (object == null)
	return true;
if (object instanceof List)
	{
	Object
		value, list;
	for (int index = ((List)object).size ();
		   --index >= 0;)
		{
		if (! Is_Value (value = ((List)object).get (index)))
			return false;
		if (((Value)value).Is_Array () &&
			(((Value)value).children != null) &&
			! Is_Value_Array (((Value)value).children))
			return false;
		}
	return true;
	}
return false;
}

/**	Tests if the Value is one of the <CODE>NUMERIC</CODE> Types.
<P>
	<CODE>NUMERIC</CODE> Types include <CODE>INTEGER</CODE> and
	<CODE>REAL</CODE> Types.
<P>
	@return	true if the Value has a <CODE>NUMERIC</CODE> Type.
	@see	#NUMERIC
*/
public boolean Is_Numeric ()
{return ((_Type_ & NUMERIC) != 0);}

/**	Tests if the Value is the <CODE>INTEGER</CODE> Type.
<P>
	@return	true if the Value is the <CODE>INTEGER</CODE> Type.
	@see	#INTEGER
*/
public boolean Is_Integer ()
{return (_Type_ == INTEGER);}

/**	Tests if the Value is the <CODE>REAL</CODE> Type.
<P>
	@return	true if the Value is the <CODE>REAL</CODE> Type.
	@see	#REAL
*/
public boolean Is_Real ()
{return (_Type_ == REAL);}

/**	Tests if the Value is one of the <CODE>STRING</CODE> Types.
<P>
	<CODE>STRING</CODE> Types include <CODE>IDENTIFIER</CODE>,
	<CODE>SYMBOL</CODE>, <CODE>TEXT</CODE> and <CODE>DATE_TIME</CODE>
	Types.
<P>
	@return	true if the Value has a <CODE>STRING</CODE> Type.
	@see	#STRING
*/
public boolean Is_String ()
{return ((_Type_ & STRING) != 0);}

/**	Tests if the Value is the <CODE>IDENTIFIER</CODE> Type.
<P>
	@return	true if the Value is the <CODE>IDENTIFIER</CODE> Type.
	@see	#IDENTIFIER
*/
public boolean Is_Identifier ()
{return (_Type_ == IDENTIFIER);}

/**	Tests if the Value is the <CODE>SYMBOL</CODE> Type.
<P>
	@return	true if the Value is the <CODE>SYMBOL</CODE> Type.
	@see	#SYMBOL
*/
public boolean Is_Symbol ()
{return (_Type_ == SYMBOL);}

/**	Tests if the Value is the <CODE>TEXT</CODE> Type.
<P>
	@return	true if the Value is the <CODE>TEXT</CODE> Type.
	@see	#TEXT
*/
public boolean Is_Text ()
{return (_Type_ == TEXT);}

/**	Tests if the Value is the <CODE>DATE_TIME</CODE> Type.
<P>
	@return	true if the Value is the <CODE>DATE_TIME</CODE> Type.
	@see	#DATE_TIME
*/
public boolean Is_Date_Time ()
{return (_Type_ == DATE_TIME);}

/**	Tests if the Value is one of the <CODE>ARRAY</CODE> Types.
<P>
	<CODE>ARRAY</CODE> Types include <CODE>SET</CODE> and
	<CODE>SEQUENCE</CODE> Types.
<P>
	@return	true if the Value has an <CODE>ARRAY</CODE> Type.
	@see	#ARRAY
*/
public boolean Is_Array ()
{return ((_Type_ & ARRAY) != 0);}

/**	Tests if the Value is the <CODE>SET</CODE> Type.
<P>
	@return	true if the Value is the <CODE>SET</CODE> Type.
	@see	#SET
*/
public boolean Is_Set ()
{return (_Type_ == SET);}

/**	Tests if the Value is the <CODE>SEQUENCE</CODE> Type.
<P>
	@return	true if the Value is the <CODE>SEQUENCE</CODE> Type.
	@see	#SEQUENCE
*/
public boolean Is_Sequence ()
{return (_Type_ == SEQUENCE);}

/**	Tests if the Value Type is <CODE>UNKNOWN</CODE>
	(an empty Value).
<P>
	@return	true if the Value Type is <CODE>UNKNOWN</CODE>.
	@see	#UNKNOWN
*/
public boolean Is_Unknown ()
{return (_Type_ == UNKNOWN);}


//	Static versions:

/**	Tests for a <CODE>NUMERIC</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is for a <CODE>NUMERIC</CODE>.
	@see	#Is_Numeric()
*/
public static boolean Is_Numeric (int type)
{return ((type & NUMERIC) != 0);}

/**	Tests for the <CODE>INTEGER</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>INTEGER</CODE>.
	@see	#INTEGER
*/
public static boolean Is_Integer (int type)
{return (type == INTEGER);}

/**	Tests for the <CODE>REAL</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>REAL</CODE>.
	@see	#REAL
*/
public static boolean Is_Real (int type)
{return (type == REAL);}

/**	Tests for a <CODE>STRING</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is for a <CODE>STRING</CODE>.
	@see	#Is_String()
*/
public static boolean Is_String (int type)
{return ((type & STRING) != 0);}

/**	Tests for the <CODE>IDENTIFIER</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>IDENTIFIER</CODE>.
	@see	#IDENTIFIER
*/
public static boolean Is_Identifier (int type)
{return (type == IDENTIFIER);}

/**	Tests for the <CODE>SYMBOL</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>SYMBOL</CODE>.
	@see	#SYMBOL
*/
public static boolean Is_Symbol (int type)
{return (type == SYMBOL);}

/**	Tests for the <CODE>TEXT</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>TEXT</CODE>.
	@see	#TEXT
*/
public static boolean Is_Text (int type)
{return (type == TEXT);}

/**	Tests for the <CODE>DATE_TIME</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>DATE_TIME</CODE>.
	@see	#DATE_TIME
*/
public static boolean Is_Date_Time (int type)
{return (type == DATE_TIME);}

/**	Tests for an <CODE>ARRAY</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is for an <CODE>ARRAY</CODE>.
	@see	#Is_Array()
*/
public static boolean Is_Array (int type)
{return ((type & ARRAY) != 0);}

/**	Tests for the <CODE>SET</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>SET</CODE>.
	@see	#SET
*/
public static boolean Is_Set (int type)
{return (type == SET);}

/**	Tests for the <CODE>SEQUENCE</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>SEQUENCE</CODE>.
	@see	#SEQUENCE
*/
public static boolean Is_Sequence (int type)
{return (type == SEQUENCE);}

/**	Tests for the <CODE>UNKNOWN</CODE> Type.
<P>
	@param	type	The Type code int.
	@return	true if the Type is <CODE>UNKNOWN</CODE>.
	@see	#UNKNOWN
*/
public static boolean Is_Unknown (int type)
{return (type == UNKNOWN);}

/*------------------------------------------------------------------------------
	Base:
*/
/**	Gets the radix base that will be applied if this Value is an
	<CODE>INTEGER</CODE> Type.
<P>
	@return	The current radix base int associated with this Value.
	@see	#Base(int)
*/
public int Base ()
{return _Base_;}


/**	Sets the radix base to be applied if this Value is an
	<CODE>INTEGER</CODE> Type during a conversion between
	(to or from) a Long and String representation.
<P>
	The radix base is associated with the Value regardless of whether
	it is currently an <CODE>INTEGER</CODE> Type and remains associated
	with the Value regarless of Type changes. Valid radix base values
	are in the range Character.MIN_RADIX <= |base| <=
	Character.MAX_RADIX. The sign of the radix base value will be
	applied to the Value datum when the radix base is used, but is
	otherwise ignored. For example, if a negative radix base is applied
	to positive <CODE>INTEGER</CODE> Value, then <CODE>long_Data</CODE>
	will return a positive long value but <CODE>String_Data</CODE> will
	return the represenation of the negative of the long value since a
	conversion has been applied. Using negative radix base numbers can
	be a source of confusion and probably should be avoided.
<P>
	@param	base	The radix base int to be used.
	@return	This Value.
	@throws	PVL_Exception
		<DL>
		<DT>VALUE_OVERFLOW
			<DD>The value of the radix base out of range.
		</DL>
	@see	Character#MIN_RADIX
	@see	Character#MAX_RADIX
*/
public Value Base
	(
	int		base
	)
	throws PVL_Exception
{
if (Math.abs (base) < Character.MIN_RADIX ||
	Math.abs (base) > Character.MAX_RADIX)
	throw new PVL_Exception
		(
		ID, PVL_Exception.VALUE_OVERFLOW,
		"Invalid Value Base " + base + ".\n"
		+ "  Acceptable range: "
			+ Character.MIN_RADIX + " <= |Base| <= "
			+ Character.MAX_RADIX
		);
_Base_ = base;
return this;
}

/*------------------------------------------------------------------------------
	Units:
*/
/**	Get the units description String.
<P>
	@return	The units description String, which may be null.
*/
public String Units ()
{return _Units_;}

/**	Set the units description String.
<P>
	This is an arbitrary String that is expected to provide some form
	of real world units description for the Value. It may be null to
	remove any units description.
<P>
	@param	units	The units description String.
	@return	This Value.
*/
public Value Units
	(
	String		units
	)
{
_Units_ = units;
return this;
}

/*------------------------------------------------------------------------------
	Parent:
*/
/**	Gets the Value that contains this Value as an Array element.
<P>
	@return	The Array that contains this Value in its data Vector, or
		null if this Value is not contained in an Array.
*/
public Value Parent ()
{return (Value)getParent ();}

/*------------------------------------------------------------------------------
	Warning:
*/
/**	Gets the current warning status.
<P>
	When conditions are encountered that are unusual enough to warrant
	attention, but not an error condition that would prevent successful
	processing which would cause an exception to be thrown, a warning
	condition is registered. The warning is in the form of a
	PVL_Exception that was not thrown. The current warning status is
	either the {@link #First_Warning(boolean)
	<CODE>First_Warning</CODE>} or the {@link #Last_Warning(boolean)
	<CODE>Last_Warning</CODE>} since a {@link #Reset_Warning()
	<CODE>Reset_Warning</CODE>}.
<P>
	@return	The current warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#First_Warning(boolean)
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Warning ()
{
if (_Use_First_Warning_)
	return _First_Warning_;
return _Last_Warning_;
}

/**	Clears any warning status so that the <CODE>Warning</CODE> method
	will return null until the next warning condition occurs.
<P>
	@return	This Value.
	@see	#Warning()
*/
public Value Reset_Warning ()
{
_First_Warning_ = null;
_Last_Warning_ = null;
return this;
}

/**	Enables or disables returning the first warning that occurs as the
	current warning status.
<P>
	The first warning is one that occurs when the current warning
	status is null.
<P>
	@param	first	true to enable returning the first warning status;
		false to return the last warning that occurred as the current
		warning status.
	@return	This Value.
	@see	#Warning()
	@see	#First_Warning()
	@see	#Reset_Warning()
*/
public Value First_Warning
	(
	boolean		first
	)
{
_Use_First_Warning_ = first;
return this;
}

/**	Returns the first warning since the last <CODE>Reset_Warning</CODE>.
<P>
	@return	The first warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#First_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception First_Warning ()
{return _First_Warning_;}

/**	Enables or disables returning the last warning that occurs as the
	current warning status.
<P>
	The last warning is the most recent one regarless of any previous
	warning conditions that may have occured without an intervening
	<CODE>Reset_Warning</CODE>.
<P>
	@param	last	true to enable returning the last warning status;
		false to return the first warning condition that occurred as
		the current warning status.
	@return	This Value.
	@see	#Warning()
	@see	#Last_Warning()
	@see	#Reset_Warning()
*/
public Value Last_Warning
	(
	boolean		last
	)
{
_Use_First_Warning_ = ! last;
return this;
}

/**	Returns the last warning since a <CODE>Reset_Warning</CODE>.
<P>
	@return	The last warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Last_Warning ()
{return _Last_Warning_;}

/*==============================================================================
	General Methods
*/
/**	Makes a deep copy of the Value.
<P>
	@return	An Object of class Value that is a copy of this Value.
	@see	#Value(Value)
	@see	Object#clone()
	@see	Cloneable
*/
public Object clone ()
{
try {return new Value (this);}
catch (PVL_Exception execption) {return null;}
}

/**	Gets a String representation of the Value.
<P>
	For an <CODE>INTEGER</CODE> Type Value with a decimal radix base
	the Long datum's <CODE>toString</CODE> representation is returned.
	If the datum is null a 0L value is provided. When the radix base is
	not decimal then the radix base is incuded in the representation
	using PVL syntax:
<P>
	<BLOCKQUOTE>
	[ <B><I>sign</I></B> ] <B><I>base</I>#<I>value</I>#</B>
	</BLOCKQUOTE>
<P>
	Where the negative sign is used if either the radix base or the
	datum value (but not both) are negative, the radix base is
	represented in decimal notation, and the datum value is represented
	in the character set corresponding to the radix base. The
	crosshatch characters are the {@link Parser#NUMBER_BASE_DELIMITER
	<CODE>Parser.NUMBER_BASE_DELIMITER</CODE>} character.
<P>
	For a <CODE>REAL</CODE> Type Value the Double datum's
	<CODE>toString</CODE> representation is returned. If the datum is
	null a "0.0" representation is returned.
<P>
	For <CODE>STRING</CODE> Types the String datum is returned (the
	empy string is used for a null datum) enclosed in any PVL syntax
	characters appropriate for its specific Type. For the
	<CODE>TEXT</CODE> Type the {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} character starts and ends the
	String, while for the <CODE>SYMBOL</CODE> Type the {@link
	Parser#SYMBOL_DELIMITER <CODE>Parser.SYMBOL_DELIMITER</CODE>} is
	used.
<P>
	For Values with an <CODE>ARRAY</CODE> Type, the
	<CODE>Type_Name</CODE> is used as a place holder (use the
	<CODE>Write</CODE> method to get all of the individual Array
	entries). For the <CODE>SET</CODE> Type this is enclosed in {@link
	Parser#SET_START_DELIMITER <CODE>Parser.SET_START_DELIMITER</CODE>}
	and {@link Parser#SET_END_DELIMITER
	<CODE>Parser.SET_END_DELIMITER</CODE>} characters. For the
	<CODE>SEQUENCE</CODE> Type the {@link
	Parser#SEQUENCE_START_DELIMITER
	<CODE>Parser.SEQUENCE_START_DELIMITER</CODE>} and {@link
	Parser#SEQUENCE_END_DELIMITER
	<CODE>Parser.SEQUENCE_END_DELIMITER</CODE>} characters enclose the
	name.
<P>
	For Values with the <CODE>UNKNOWN</CODE> Type just the
	<CODE>Type_Name</CODE> is provided.
<P>
	@return	A String description of the Value datum plus any units.
	@see	#Write(OutputStream, int, boolean)
	@see	#Type_Name()
	@see	Long#toString()
	@see	Double#toString()
*/
public String toString ()
{
switch (_Type_)
	{
	case INTEGER:
		int
			sign = (_Base_ < 0) ? -1 : 1,
			base = Math.abs (_Base_);
		long
			number =
				(_Datum_ == null) ? 0 : ((Long)_Datum_).longValue ();
		if (number < 0)
			{
			//	Always use a positive number.
			sign   = -sign;
			number = -number;
			}
		if (base == 10)
			return
				Long.toString (sign * number);
		else
			return
				String.valueOf (sign * base) +
				Parser.NUMBER_BASE_DELIMITER +
				Long.toString (number, base).toUpperCase () +
				Parser.NUMBER_BASE_DELIMITER;

	case REAL:
		return
			(_Datum_ == null) ? "0.0" : ((Double)_Datum_).toString ();

	case STRING:
	case IDENTIFIER:
	case DATE_TIME:
		return
			(_Datum_ == null) ? "" : (String)_Datum_;
	case TEXT:
		return
			Parser.TEXT_DELIMITER +
			((_Datum_ == null) ? "" : (String)_Datum_) +
			Parser.TEXT_DELIMITER;
	case SYMBOL:
		return
			Parser.SYMBOL_DELIMITER +
			((_Datum_ == null) ? "" : (String)_Datum_) +
			Parser.SYMBOL_DELIMITER;

	case ARRAY:
		return Type_Name ();
	case SET:
		return
			Parser.SET_START_DELIMITER +
			Type_Name () +
			Parser.SET_END_DELIMITER;
	case SEQUENCE:
		return
			Parser.SEQUENCE_START_DELIMITER +
			Type_Name () +
			Parser.SEQUENCE_END_DELIMITER;

	case UNKNOWN:
		return Type_Name ();

	default:
		return null;
	}
}

/**	Gets a PVL description of the Value using a specified Lister.
<p>
	The Value is {@link #Write(OutputStream) Write}n to a
	StringWriter on the specified Lister and the resultant String is
	returned. If the Write method produces an exception, the exception
	message is returned. The Lister will have its Writer {@link
	Lister#Set_Writer(Writer) set} to the StringWriter used to write
	the PVL syntax. If no Lister is specified (the argument is null)
	then a default Lister will be provided.
<P>
	@param	lister	The Lister to use when writing the PVL syntax.
	@return	The String describing the Value in PVL syntax.
*/
public String Description
	(
	Lister	lister
	)
{
StringWriter
	writer = new StringWriter ();
if (lister == null)
	lister = new Lister (writer);
else
	lister.Set_Writer (writer);
try {lister.Write (this);}
catch (Exception exception)
	{writer.write ("\n" + exception.getMessage ());}
String
	description = writer.toString ();
if (description.charAt (description.length () - 1) == '\n')
	description = description.substring (0, description.length () - 1);
return description;
}

/**	Gets a PVL description of the Value.
<p>
	The Value is {@link #Write(OutputStream) Write}n to a
	StringWriter on a default Lister and the resultant String is
	returned. If the Write method produces an exception, the exception
	message is returned.
<P>
	@return	The String describing the Value in PVL syntax.
	@see	#Description(Lister)
*/
public String Description ()
{return Description (null);}

/**	Tests if a Value matches this Value using the specified criteria.
<P>
	@param	value	The Value to be compared to this Value.
	@param	criteria	The Selector providing the comparison criteria.
	@return	true if there is a criteria match; false otherwise.
	@see	Selector
*/
public boolean Match
	(
	Value		value,
	Selector	criteria
	)
{
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		(">>> Value.Match");
boolean
	match = criteria.Values_Match (this, value);
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		("<<< Value.Match: " + match);
return match;
}

/**	Tests if a Value matches this Value using the specified criteria.
<P>
	If the Value is an Array matching is done recursively on all
	Values of the Array. The match succeeds if all Values match.
<P>
	<B>N.B.</B>: To avoid unnecessary redundant recursive comparisons of
	Array data values if the {@link Selector#Data(boolean) Data}
	criteria is included in the criteria the {@link
	Selector#Data_Match(Value, Value) data comparison} is only done for
	non-Array Values.
<P>
	@param	value	The Value to be compared to this Value.
	@param	criteria	The Selector providing the comparison criteria.
	@return	true if there is a criteria match; false otherwise.
	@see	Selector
	@see	Selection
*/
public boolean Match_Depth
	(
	Value		value,
	Selector	criteria
	)
{
boolean
	matches,
	match_data = criteria.Data ();
if (match_data)
	//	Defer Data_Match to non-Array values.
	criteria.Data (false);

matches = match_depth (value, criteria, match_data);

if (match_data)
		criteria.Data (true);
return matches;
}


private boolean match_depth
	(
	Value		value,
	Selector	criteria,
	boolean		match_data
	)
{
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		(">>> Value.match_depth:\n"
		+"     this - " + this.toString () + '\n'
		+"    value - " + value + '\n'
		+"    " + criteria + '\n'
		+"    match_data - " + match_data);
if (value == null ||
	! criteria.Values_Match (this, value))
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("<<< Value.match_depth: false");
	return false;
	}

if ( this.Is_Array () &&
	value.Is_Array ())
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("    Comparing Array Lists ...");
	try
		{
		if (this.Vector_Data ().size () != value.Vector_Data ().size ())
			{
			if ((DEBUG & DEBUG_MATCH) != 0)
				System.out.println
					("    this list size " + this.Vector_Data ().size ()
						+ " != value list size " + value.Vector_Data ().size () + '\n'
					+"<<< Value.match_depth: false");
			return false;
			}
		}
	catch (PVL_Exception exception)
		{
		//	Shouldn't happen since they're both valid Arrays.
		return false;
		}
	Iterator
		these_values =  this.iterator (),
		those_values = value.iterator ();
	while (these_values.hasNext () &&
		   those_values.hasNext ())
		{
		if (! ((Value)these_values.next ()).match_depth
			  ((Value)those_values.next (), criteria, match_data))
			{
			if ((DEBUG & DEBUG_MATCH) != 0)
				System.out.println
					("<<< Value.match_depth: false");
			return false;
			}
		}
	}
else if (match_data)
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("    Comparing data values ...");
	boolean
		matches = criteria.Data_Match (this, value);
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("<<< Value.match_depth: " + matches);
	return matches;
	}
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		("<<< Value.match_depth: true");
return true;
}

/**	Tests if an Object is equal to this Value.
<P>
	An Object is equal to this Value when it is also a Value object and
	all of its fields are equal to this Value's fields. A Selection
	object with <CODE>VALUE_MATCH</CODE> criteria is used with the
	<CODE>Match_Depth</CODE> method.
<P>
	@param	object	The Object to be compared to this Value.
	@return	true if the object equals this Value, false otherwise.
	@see	#Match_Depth(Value, Selector)
	@see	Selection
	@see	Selector#VALUE_MATCH
*/
public boolean equals
	(
	Object		object
	)
{
if (! Is_Value (object))
	return false;
return Match_Depth
	((Value)object, new Selection (Selector.VALUE_MATCH));
}

/**	Tests if a Value is equivalent to this Value, ignoring certain
	cases where field values are not equal.
<P>
	Unlike <CODE>equals</CODE>, for a Value to be equivalent to this
	Value the fields need only be logically equivalent, rather than
	containing the same value. A Selection object with
	<CODE>VALUE_MATCH</CODE> but <CODE>Specific (false)</CODE> criteria
	is used with the <CODE>Match_Depth</CODE> method.
<P>
	@param	value	The Value to be compared to this Value.
	@return	true if the Value is logically equivalent to this Value,
		false otherwise.
	@see	#Match_Depth(Value, Selector)
	@see	Selection
	@see	Selector#VALUE_MATCH
	@see	Selection#Specific(boolean)
*/
public boolean equalsIgnoreCase
	(
	Value		value
	)
{return Match_Depth
	(value, new Selection (Selector.VALUE_MATCH).Specific (false));}

/*------------------------------------------------------------------------------
*/
/**	Writes the Value in PVL syntax using a PVL Lister.
<P>
	The Lister is created with the output stream as its list
	destination using the specified indent level and syntax strictness.
<P>
	@param	output	The OutputStream to receive what is written.
		If output is null <CODE>System.out</CODE> is used.
	@param	level	The indent level at which to start the listing.
		If the level is negative, indenting is disabled.
	@param	strict	Use strict PVL syntax if true; otherwise use an
		easier to read format.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an Array is not a Value.
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>If a PVL reserved character was found in an unenclosed
				<CODE>STRING</CODE> datum during strict writing.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	Lister#Write(Value)
*/
public int Write
	(
	OutputStream	output,
	int				level,
	boolean			strict
	)
	throws PVL_Exception, IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println ("\n>>> Value.Write");
Lister
	lister = new Lister (output)
		.Indent_Level (level)
		.Strict (strict);
int
	total = lister.Write (this);
Warning (lister.Warning ());
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println ("<<< Value.Write: total written = " + total);
return total;
}

/**	Writes the Value in PVL syntax to <CODE>System.out</CODE>
	starting at level 0, not strict.
<P>
	@return	The total number of bytes written.
	@throws	PVL_Exception	From the foundation <CODE>Write</CODE> method.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write ()
	throws PVL_Exception, IOException
{return Write (System.out, 0, false);}

/**	Writes the Value in PVL syntax to <CODE>System.out</CODE>
	with or without indenting, not strict.
<P>
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@return	The total number of bytes written.
	@throws	PVL_Exception	From the foundation <CODE>Write</CODE> method.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	boolean			indent
	)
	throws PVL_Exception, IOException
{return Write (System.out, (indent ? 0 : -1), false);}

/**	Writes the Value in PVL syntax to output <CODE>System.out</CODE>
	with indenting and strict PVL enabled or disabled.
<P>
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@param	strict	Use strict PVL syntax if true; otherwise use an
		easier to read format.
	@return	The total number of bytes written.
	@throws	PVL_Exception	From the foundation <CODE>Write</CODE> method.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	boolean			indent,
	boolean			strict
	)
	throws PVL_Exception, IOException
{return Write (System.out, (indent ? 0 : -1), strict);}

/**	Writes the Value in PVL syntax to the output starting at level 0,
	not strict.
<P>
	@param	output	The OutputStream to receive what is written.
	@return	The total number of bytes written.
	@throws	PVL_Exception	From the foundation <CODE>Write</CODE> method.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	OutputStream	output
	)
	throws PVL_Exception, IOException
{return Write (output, 0, false);}

/**	Writes the Value in PVL syntax to the output with or without
	indenting, not strict.
<P>
	@param	output	The OutputStream to receive what is written.
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@return	The total number of bytes written.
	@throws	PVL_Exception	From the foundation <CODE>Write</CODE> method.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	OutputStream	output,
	boolean			indent
	)
	throws PVL_Exception, IOException
{return Write (output, (indent ? 0 : -1), false);}

/*------------------------------------------------------------------------------
	Iterators:
*/
/**	Provides an Iterator for Values contained in an Array positioned so
	that the next element returned will be from the specified index of
	the Array.
<P>
	This Iterator does not descend into nested Array Values. The
	{@link DefaultMutableTreeNode DefaultMutableTreeNode} base class
	provdes various Enumeration objects that will traverse the
	hierarchy in different orders.
<P>
	For non-Array Values an "empty" Iterator will be provided. This
	will act the same as an Iterator on an empty data Vector.
<P>
	@param	index	The starting list element.
	@return	An Iterator object.
	@see	Iterator
	@see	DefaultMutableTreeNode
*/
public Iterator iterator
	(
	int			index
	)
{
if (Is_Array ())
	return new Value_Iterator (index);
return Parameter.EMPTY_LIST_ITERATOR;
}

/**	Provides an Iterator starting at the first element of the list.
<P>
	@return	An Iterator object.
	@see	#iterator(int)
*/
public Iterator iterator ()
{return iterator (0);}

/**	Provides a ListIterator for Values contained in an Array positioned
	so that the next element returned will be from the specified index of
	the list.
<P>
	The functionality of the Value Iterator is provided (which it
	extends) with the ListIterator interface implemented.
<P>
	<B>Note</B>: When the Value is an <CODE>ARRAY</CODE> Type, but has
	no data Vector, an empty one is provided so the <CODE>add</CODE>
	method will be able to insert new Values into the Array.
<P>
	@param	index	The starting list element.
	@return	A ListIterator object.
	@see	#iterator()
	@see	ListIterator
*/
public ListIterator listIterator
	(
	int			index
	)
{
if (Is_Array ())
	return new Value_ListIterator (index);
return Parameter.EMPTY_LIST_ITERATOR;
}

/**	Provides a ListIterator starting at the first element of the list.
<P>
	@return	A ListIterator object.
	@see	#listIterator(int)
*/
public ListIterator listIterator ()
{return listIterator (0);}

/*.............................................................................
*/
private class Value_Iterator
	implements Iterator
{
ListIterator
	list_iterator;
Value
	last = null;


private Value_Iterator ()
{this (0);}


private Value_Iterator
	(
	int			index
	)
{
if (children == null)
	children = new Vector ();	//	May need to add to this aggregate.
list_iterator = children.listIterator (index);
}


public boolean hasNext ()
{return list_iterator.hasNext ();}


public Object next ()
{
if (hasNext ())
	return (last = (Value)list_iterator.next ());
throw new NoSuchElementException (ID);
}


public void remove ()
{
if (last != null)
	{
	list_iterator.remove ();
	last.parent = null;
	last = null;
	return;
	}
throw new IllegalStateException (ID);
}
}	//	End of Value_Iterator class


private class Value_ListIterator
	extends Value_Iterator
	implements ListIterator
{
private Value_ListIterator
	(
	int			index
	)
{super (index);}


public boolean hasPrevious ()
{return list_iterator.hasPrevious ();}


public Object previous ()
{return (last = (Value)list_iterator.previous ());}


public int nextIndex ()
{return list_iterator.nextIndex ();}


public int previousIndex ()
{return list_iterator.previousIndex ();}


public void set
	(
	Object		object
	)
{
if (! Is_Value (object))
	throw new ClassCastException
		(ID + "\nThe Object to set is not a Value.");
if (last != null)
	{
	list_iterator.set (object);
	((Value)object).parent = last.parent;
	last.parent = null;
	last = (Value)object;
	return;
	}
throw new IllegalStateException (ID);
}


public void add
	(
	Object		object
	)
{
if (! Is_Value (object))
	throw new ClassCastException
		(ID + "\nThe Object to set is not a Parameter.");
list_iterator.add (object);
((Value)object).parent = Value.this;
last = null;
}


public void insert
	(
	Value	value
	)
{add (value);}
}	//	End of Value_ListIterator class

/*------------------------------------------------------------------------------
	Find
*/
/**	Searches for a Value that matches a test Value according
	the Selector criteria, optionally after some specific Value
	object has been located first.
<P>
	A search is made of the Values contained in this Array, and,
	recursively, any Arrays it contains. The search is depth-wise: each
	Array Value is searched when it is encountered.
<P>
	The search must first find the last_value (having the same object
	reference) before proceeding with the search for the test_value
	beginning with the Value following the last_value. If the
	last_value is null, then the search for the test_value begins with
	the first Value contained in this Array.
<P>
	The search compares each value encountered against the test_value
	using the Selector criteria's <CODE>Values_Match</CODE> method. The
	first Value that matches the test_value is returned. If the
	last_value or the test_value is not found, null is returned. If the
	last_value could not be found a
	<CODE>PVL_Exception.NO_LAST_LOCATION</CODE> warning status is
	registered with this Value. If the Value in which to
	<CODE>Find</CODE> is not an <CODE>ARRAY</CODE> Type, null will be
	returned and a <CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning
	status will be set.
<P>
	The test_value may be any parameter, pre-existing or created ad hoc
	as a template for the Selector criteria. A Selector offers numerous
	criteria that may be conditionally applied when comparing the
	test_value with Array members. <B>Note</B>: The PIRL.PVL
	package provides a Selection class that implements the Selector
	interface.
<P>
	@param	test_value	A Value object to use for testing Values in
		this Array for a match.
	@param	criteria	A Selector object providing the methods to
		determine if Values match.
	@param	last_value	A Value object to be found within the Array
		before search comparisons begin. If null, begin the search at
		the first element of the Array.
	@return	The Value object that was found, or null if not found. If
		the Value in which to <CODE>Find</CODE> is not an
		<CODE>ARRAY</CODE> Type, null will be returned and a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning status will
		be set. If the last_value could not be found a
		<CODE>PVL_Exception.NO_LAST_LOCATION</CODE> warning status is
		registered with this Value.
	@see	Selector
	@see	Selection#Values_Match(Value, Value)
*/
public Value Find
	(
	Value		test_value,
	Selector	criteria,
	Value		last_value
	)
{
if (! find_check (test_value))
	return null;
Value[]
	Last_Value = {last_value};

if ((DEBUG & DEBUG_FIND) != 0)
	{
	System.out.println
		(
		">>> Value Find: " + test_value.Type_Name () + " " + test_value + "\n"
		+ "  criteria = " + Integer.toString (criteria.Criteria (), 2) + "\n"
		+ "    Data     = " + criteria.Data () + "\n"
		+ "    Type     = " + criteria.Type () + "\n"
		+ "    Base     = " + criteria.Base () + "\n"
		+ "    Units    = " + criteria.Units () + "\n"
		+ "    Specific = " + criteria.Specific () + "\n"
		+ "    AND      = " + criteria.And ()
		);
	if (last_value != null)
		System.out.println
			("  last_value is " + last_value);
	}

test_value = find (test_value, criteria, Last_Value);

if (test_value == null &&
	Last_Value[0] != null &&
	Warning () == null)
	Warning
		(
		PVL_Exception.NO_LAST_LOCATION,
		"Can't Find the last Value (" + last_value + ")."
		);
return test_value;
}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(byte)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (byte value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(short)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (short value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(int)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (int value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(long)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (long value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(float)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (float value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the primitive value.
<P>
	A test_value is constructed from the primitive value and a
	selection criteria requiring a data match is provided.
<P>
	@param	value	The primitive value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@see	#Value(double)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (double value)
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds a Value having the datum equal to the value of the Object.
<P>
	A test_value is constructed from the object and a selection
	criteria requiring a data match is provided.
<P>
	@param	value	The object having the value for the datum being sought.
	@return	The Value object that was found, or null if not found.
	@throws	PVL_Exception	If a Value could not be constructed from
		the Object.
	@see	#Value(Object)
	@see	#Find(Value, Selector, Value)
*/
public Value Find (Object value)
	throws PVL_Exception
{return Find (new Value (value), (new Selection ()).Data (true));}

/**	Finds the Value of the specified Type, beginning after the
	specified Value object is found.
<P>
	The Type matching is specific. For a general Type match - i.e.
	where any two <CODE>NUMERIC</CODE>, <CODE>STRING</CODE> or
	<CODE>ARRAY</CODE> Values will match - use:
<P>
<PRE>Find (new Value ().Type (type),
&nbsp;&nbsp;&nbsp;&nbsp;  new Selection ().Type (true).Specific (false),
&nbsp;&nbsp;&nbsp;&nbsp;  last_value);
</PRE>
<P>
	@param	type	The Type code to search for.
	@param	last_value	A Value object to be found within the Array
		before search comparisons begin. If null, begin the search at
		the first element of the Array.
	@return	The Value object that was found, or null if not found.
	@see	#Find(int)
	@see	#Find(Value, Selector, Value)
	@see	Selection#Type(boolean)
	@see	Selection#Specific(boolean)
*/
public Value Find
	(
	int			type,
	Value		last_value
	)
{
Value
	value = new Value ();
value.set_type (type);
return Find
	(
	value,
	new Selection ().Type (true).Specific (true),
	last_value
	);
}

/**	Finds the Value that completely matches a test Value.
<P>
	A complete match requires an exact match of the Value data and its
	Type (as well as the numeric base for integers), and any units
	String. <B>Note</B>: Value contents, not Value object references,
	are compared.
<P>
	@param	test_value	The Value to use when testing for a match.
	@return	The Value object that was found, or null if not found.
	@see	#Find(Value, Selector, Value)
*/
public Value Find
	(
	Value	test_value
	)
{
return Find
	(
	test_value,
	new Selection ().Criteria (Selector.VALUE_MATCH),
	null
	);
}

/**	Finds the Value that completely matches a test Value, beginning
	after the specified Value within the Array.
<P>
	@param	test_value	A Value object to use for testing Values in
		this Array for a match.
	@param	last_value	A Value object to be found within the Array
		before search comparisons begin. If null, begin the search at
		the first element of the Array.
	@return	The Value object that was found, or null if not found.
	@see	#Find(Value)
	@see	#Find(Value, Selector, Value)
*/
public Value Find
	(
	Value	test_value,
	Value	last_value
	)
{
return Find
	(
	test_value,
	new Selection ().Criteria (Selector.VALUE_MATCH),
	last_value
	);
}

/**	Finds the Value that matches a test Value according the Selector
	criteria.
<P>
	@param	test_value	A Value to use for testing Values in this Array
		for a match.
	@param	criteria	A Selector object providing the methods to
		determine if Values match.
	@return	The Value object that was found, or null if not found.
	@see	#Find(Value, Selector, Value)
*/
public Value Find
	(
	Value		test_value,
	Selector	criteria
	)
{return Find (test_value, criteria, null);}


private boolean find_check
	(
	Value		test_value
	)
{
//	So last value not found will be noticed.
Reset_Warning ();

if (! Is_Array ())
	{
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Find in a Value of Type " + Type_Name () + "."
		);
	return false;
	}
if (test_value == null ||
	children == null)
	return false;
return true;
}


private Value find
	(
	Value		Test_Value,
	Selector	Criteria,
	Value[]		Last_Value
	)
{
/*	An array is used for Last_Value to provide a pass-back value
	when the Last_Value[0] has been located.
*/
if ((DEBUG & DEBUG_FIND) != 0)
	System.out.println (">>> find");

Value
	value,
	found_value = null;


/*	Find the Test_Value.

	Note: The algorithm of this function searches depth-wise;
	i.e. each value encountered in the Array is
	searched (conditionally) when it is encountered.
*/
Iterator
	values = iterator ();
while (values.hasNext ())
	{
	value = (Value)values.next ();
	if ((DEBUG & DEBUG_FIND) != 0)
		System.out.println ("find: " + value.Type_Name () + " " + value);

	if (Last_Value[0] == null)
		{
		/*	Compare the value with the Test_Value using the Criteria.

			Note that the Test_Value comparison only occurs after the
			Last_Value has been found (or isn't being used).
		*/
		if (Criteria.Values_Match (value, Test_Value))
			{
			found_value = value;
			if ((DEBUG & DEBUG_FIND) != 0)
				System.out.println ("      Matched");
			break;
			}
		}
	else
		{
		//	Just searching for the Last_Value.
		if (value == Last_Value[0])
			Last_Value[0] = null;	//	Found it.
		}
		
	if (value.Is_Array () &&
		(found_value =
			value.find
				(
				Test_Value,
				Criteria,
				Last_Value
				)) != null)
		break;
	}
if ((DEBUG & DEBUG_FIND) != 0) System.out.println
	("<<< Value find: found_value is " + (found_value != null));
return found_value;
}

/*==============================================================================
	Methods that override base class methods:
*/
/*	Errors are reported as a Warning.

	It would be better if there were a way to tell the caller directly
	about problems. Unfortunately, the flawed design of the base class
	methods does not provide any means to safely report back to the
	caller, not even a function return value.
*/
/**	Overrides the base class method to ensure Value integrity.
<P>
	@param	object	Since the <CODE>userObject</CODE> of a Value is a
		reference to itself, it is illegal to use this method. A
		<CODE>PVL_Exception.BAD_ARGUMENT</CODE> warning status is set.
	@see	DefaultMutableTreeNode#setUserObject(Object)
*/
public void setUserObject
	(
	Object		object
	)
{
Warning
	(
	PVL_Exception.ILLEGAL_SYNTAX,
	"Can't setUserObject of a Value."
	);
}

/**	Overrides the base class method to ensure Value integrity.
<P>
	@param	allows	Since the ability of a Value to have children is
		totally determined by its Type (only Arrays may have a children
		data Vector), this argument is ignored and a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning status is
		set.
	@see	DefaultMutableTreeNode#setAllowsChildren(boolean)
*/
public void setAllowsChildren
	(
	boolean		allows
	)
{
Warning
	(
	PVL_Exception.ILLEGAL_SYNTAX,
	"Can't setAllowsChildren of a Value.\n"
	+ "  This is determined by the Value's Type."
	);
}

/**	Invokes the Value's <CODE>Add</CODE> method.
<P>
	@param	value	A MutableTreeNode that must be a Value object.
	@throws	IllegalArgumentException	If the value argument is not a
		Value object, or the <CODE>Add</CODE> method threw a
		PVL_Exception (in which case it's message is used for the
		exception message).
	@see	#Add(Value)
	@see	DefaultMutableTreeNode#add(MutableTreeNode)
*/
public void add
	(
	MutableTreeNode	value
	)
	throws IllegalArgumentException
{
if (! Is_Value (value))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to add is not a Value."
		);
try {Add ((Value)value);}
catch (PVL_Exception exception)
	{throw new IllegalArgumentException (exception.getMessage ());}
}

/**	Invokes the Value's <CODE>Insert</CODE> method.
<P>
	@param	value	A MutableTreeNode that must be a Value object.
	@param	index		The index in the Array where the Value is to be
		inserted.
	@throws	IllegalArgumentException	If the value argument is not
		a Value object, or the <CODE>Insert</CODE> method threw a
		PVL_Exception (in which case it's message is used for the
		exception message).
	@see	#Insert(Value, int)
	@see	DefaultMutableTreeNode#insert(MutableTreeNode, int)
*/
public void insert
	(
	MutableTreeNode	value,
	int				index
	)
	throws IllegalArgumentException
{
if (! Is_Value (value))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to insert is not a Value."
		);
try {Insert ((Value)value, index);}
catch (PVL_Exception exception)
	{throw new IllegalArgumentException (exception.getMessage ());}
}

/**	Invokes the Value's <CODE>Remove</CODE> method.
<P>
	@param	index	The index of the element in the Array to remove.
	@see	#Remove(int)
	@see	DefaultMutableTreeNode#remove(int)
*/
public void remove
	(
	int			index
	)
{Remove (index);}

/**	Invokes the Value's <CODE>Remove</CODE> method.
<P>
	@param	value	A MutableTreeNode that must be a Value object.
	@throws	IllegalArgumentException	If the value argument is not
		a Value object.
	@see	#Remove(Value)
	@see	DefaultMutableTreeNode#remove(MutableTreeNode)
*/
public void remove
	(
	MutableTreeNode	value
	)
	throws IllegalArgumentException
{
if (! Is_Value (value))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to remove is not a Value."
		);
Remove ((Value)value);
}

/*------------------------------------------------------------------------------
*/
private void Warning
	(
	String		description,
	String		explanation
	)
{
Warning
	(
	new PVL_Exception
		(
		ID,
		description,
		explanation
		)
	);
}


private void Warning
	(
	PVL_Exception	warning
	)
{
if (warning != null)
	{
	_Last_Warning_ = warning;
	if (_First_Warning_ == null)
		_First_Warning_ = _Last_Warning_;
	}
}


}	//	End of class
