///////////////////////////////////////////////////////////////
//  Copyright 2011 John Maddock. Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at https://www.boost.org/LICENSE_1_0.txt
//

#include <boost/detail/lightweight_test.hpp>
#include <boost/math/special_functions/sign.hpp>
#include <complex>

#ifdef _MSC_VER
#define _SCL_SECURE_NO_WARNINGS
#endif

#if !defined(TEST_CPP_BIN_FLOAT) && !defined(TEST_DOUBLE) && !defined(TEST_MPC)
#define TEST_CPP_BIN_FLOAT
#define TEST_DOUBLE
#define TEST_MPC
#endif


#ifdef TEST_CPP_BIN_FLOAT
#include <boost/multiprecision/cpp_complex.hpp>
#endif
#ifdef TEST_MPC
#include <boost/multiprecision/mpc.hpp>
#endif

template <class T>
void test()
{
   using std::real;
   using std::imag;
   using std::signbit;
   using std::isnan;

   using value_type = typename T::value_type;

   T result;
   //
   // Muliply the permutations which give at least one zero:
   //
   static constexpr std::array<std::array<double, 6>, 256> mul_data = 
   { {
      {{+0.000, +0.000, +0.000, +0.000, +0.000, +0.000}},
      {{+0.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
      {{+0.000, +0.000, +0.000, +2.000, +0.000, +0.000}},
      {{+0.000, +0.000, +0.000, -2.000, +0.000, +0.000}},
      {{+0.000, +0.000, -0.000, +0.000, -0.000, +0.000}},
      {{+0.000, +0.000, -0.000, -0.000, +0.000, -0.000}},
      {{+0.000, +0.000, -0.000, +2.000, -0.000, +0.000}},
      {{+0.000, +0.000, -0.000, -2.000, +0.000, -0.000}},
      {{+0.000, +0.000, +2.000, +0.000, +0.000, +0.000}},
      {{+0.000, +0.000, +2.000, -0.000, +0.000, +0.000}},
      {{+0.000, +0.000, +2.000, +2.000, +0.000, +0.000}},
      {{+0.000, +0.000, +2.000, -2.000, +0.000, +0.000}},
      {{+0.000, +0.000, -2.000, +0.000, -0.000, +0.000}},
      {{+0.000, +0.000, -2.000, -0.000, +0.000, -0.000}},
      {{+0.000, +0.000, -2.000, +2.000, -0.000, +0.000}},
      {{+0.000, +0.000, -2.000, -2.000, +0.000, -0.000}},
      {{+0.000, -0.000, +0.000, +0.000, +0.000, +0.000}},
      {{+0.000, -0.000, +0.000, -0.000, +0.000, -0.000}},
      {{+0.000, -0.000, +0.000, +2.000, +0.000, +0.000}},
      {{+0.000, -0.000, +0.000, -2.000, +0.000, -0.000}},
      {{+0.000, -0.000, -0.000, +0.000, +0.000, +0.000}},
      {{+0.000, -0.000, -0.000, -0.000, -0.000, +0.000}},
      {{+0.000, -0.000, -0.000, +2.000, +0.000, +0.000}},
      {{+0.000, -0.000, -0.000, -2.000, -0.000, +0.000}},
      {{+0.000, -0.000, +2.000, +0.000, +0.000, +0.000}},
      {{+0.000, -0.000, +2.000, -0.000, +0.000, -0.000}},
      {{+0.000, -0.000, +2.000, +2.000, +0.000, +0.000}},
      {{+0.000, -0.000, +2.000, -2.000, +0.000, -0.000}},
      {{+0.000, -0.000, -2.000, +0.000, +0.000, +0.000}},
      {{+0.000, -0.000, -2.000, -0.000, -0.000, +0.000}},
      {{+0.000, -0.000, -2.000, +2.000, +0.000, +0.000}},
      {{+0.000, -0.000, -2.000, -2.000, -0.000, +0.000}},
      {{+0.000, +2.000, +0.000, +0.000, +0.000, +0.000}},
      {{+0.000, +2.000, +0.000, -0.000, +0.000, +0.000}},
      {{+0.000, +2.000, +0.000, +2.000, -4.000, +0.000}},
      {{+0.000, +2.000, +0.000, -2.000, +4.000, +0.000}},
      {{+0.000, +2.000, -0.000, +0.000, -0.000, +0.000}},
      {{+0.000, +2.000, -0.000, -0.000, +0.000, -0.000}},
      {{+0.000, +2.000, -0.000, +2.000, -4.000, +0.000}},
      {{+0.000, +2.000, -0.000, -2.000, +4.000, -0.000}},
      {{+0.000, +2.000, +2.000, +0.000, +0.000, +4.000}},
      {{+0.000, +2.000, +2.000, -0.000, +0.000, +4.000}},
      {{+0.000, +2.000, +2.000, +2.000, -4.000, +4.000}},
      {{+0.000, +2.000, +2.000, -2.000, +4.000, +4.000}},
      {{+0.000, +2.000, -2.000, +0.000, -0.000, -4.000}},
      {{+0.000, +2.000, -2.000, -0.000, +0.000, -4.000}},
      {{+0.000, +2.000, -2.000, +2.000, -4.000, -4.000}},
      {{+0.000, +2.000, -2.000, -2.000, +4.000, -4.000}},
      {{+0.000, -2.000, +0.000, +0.000, +0.000, +0.000}},
      {{+0.000, -2.000, +0.000, -0.000, +0.000, -0.000}},
      {{+0.000, -2.000, +0.000, +2.000, +4.000, +0.000}},
      {{+0.000, -2.000, +0.000, -2.000, -4.000, -0.000}},
      {{+0.000, -2.000, -0.000, +0.000, +0.000, +0.000}},
      {{+0.000, -2.000, -0.000, -0.000, -0.000, +0.000}},
      {{+0.000, -2.000, -0.000, +2.000, +4.000, +0.000}},
      {{+0.000, -2.000, -0.000, -2.000, -4.000, +0.000}},
      {{+0.000, -2.000, +2.000, +0.000, +0.000, -4.000}},
      {{+0.000, -2.000, +2.000, -0.000, +0.000, -4.000}},
      {{+0.000, -2.000, +2.000, +2.000, +4.000, -4.000}},
      {{+0.000, -2.000, +2.000, -2.000, -4.000, -4.000}},
      {{+0.000, -2.000, -2.000, +0.000, +0.000, +4.000}},
      {{+0.000, -2.000, -2.000, -0.000, -0.000, +4.000}},
      {{+0.000, -2.000, -2.000, +2.000, +4.000, +4.000}},
      {{+0.000, -2.000, -2.000, -2.000, -4.000, +4.000}},
      {{-0.000, +0.000, +0.000, +0.000, -0.000, +0.000}},
      {{-0.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
      {{-0.000, +0.000, +0.000, +2.000, -0.000, +0.000}},
      {{-0.000, +0.000, +0.000, -2.000, +0.000, +0.000}},
      {{-0.000, +0.000, -0.000, +0.000, +0.000, -0.000}},
      {{-0.000, +0.000, -0.000, -0.000, +0.000, +0.000}},
      {{-0.000, +0.000, -0.000, +2.000, +0.000, -0.000}},
      {{-0.000, +0.000, -0.000, -2.000, +0.000, +0.000}},
      {{-0.000, +0.000, +2.000, +0.000, -0.000, +0.000}},
      {{-0.000, +0.000, +2.000, -0.000, +0.000, +0.000}},
      {{-0.000, +0.000, +2.000, +2.000, -0.000, +0.000}},
      {{-0.000, +0.000, +2.000, -2.000, +0.000, +0.000}},
      {{-0.000, +0.000, -2.000, +0.000, +0.000, -0.000}},
      {{-0.000, +0.000, -2.000, -0.000, +0.000, +0.000}},
      {{-0.000, +0.000, -2.000, +2.000, +0.000, -0.000}},
      {{-0.000, +0.000, -2.000, -2.000, +0.000, +0.000}},
      {{-0.000, -0.000, +0.000, +0.000, +0.000, -0.000}},
      {{-0.000, -0.000, +0.000, -0.000, -0.000, +0.000}},
      {{-0.000, -0.000, +0.000, +2.000, +0.000, -0.000}},
      {{-0.000, -0.000, +0.000, -2.000, -0.000, +0.000}},
      {{-0.000, -0.000, -0.000, +0.000, +0.000, +0.000}},
      {{-0.000, -0.000, -0.000, -0.000, +0.000, +0.000}},
      {{-0.000, -0.000, -0.000, +2.000, +0.000, +0.000}},
      {{-0.000, -0.000, -0.000, -2.000, +0.000, +0.000}},
      {{-0.000, -0.000, +2.000, +0.000, +0.000, -0.000}},
      {{-0.000, -0.000, +2.000, -0.000, -0.000, +0.000}},
      {{-0.000, -0.000, +2.000, +2.000, +0.000, -0.000}},
      {{-0.000, -0.000, +2.000, -2.000, -0.000, +0.000}},
      {{-0.000, -0.000, -2.000, +0.000, +0.000, +0.000}},
      {{-0.000, -0.000, -2.000, -0.000, +0.000, +0.000}},
      {{-0.000, -0.000, -2.000, +2.000, +0.000, +0.000}},
      {{-0.000, -0.000, -2.000, -2.000, +0.000, +0.000}},
      {{-0.000, +2.000, +0.000, +0.000, -0.000, +0.000}},
      {{-0.000, +2.000, +0.000, -0.000, +0.000, +0.000}},
      {{-0.000, +2.000, +0.000, +2.000, -4.000, +0.000}},
      {{-0.000, +2.000, +0.000, -2.000, +4.000, +0.000}},
      {{-0.000, +2.000, -0.000, +0.000, +0.000, -0.000}},
      {{-0.000, +2.000, -0.000, -0.000, +0.000, +0.000}},
      {{-0.000, +2.000, -0.000, +2.000, -4.000, -0.000}},
      {{-0.000, +2.000, -0.000, -2.000, +4.000, +0.000}},
      {{-0.000, +2.000, +2.000, +0.000, -0.000, +4.000}},
      {{-0.000, +2.000, +2.000, -0.000, +0.000, +4.000}},
      {{-0.000, +2.000, +2.000, +2.000, -4.000, +4.000}},
      {{-0.000, +2.000, +2.000, -2.000, +4.000, +4.000}},
      {{-0.000, +2.000, -2.000, +0.000, +0.000, -4.000}},
      {{-0.000, +2.000, -2.000, -0.000, +0.000, -4.000}},
      {{-0.000, +2.000, -2.000, +2.000, -4.000, -4.000}},
      {{-0.000, +2.000, -2.000, -2.000, +4.000, -4.000}},
      {{-0.000, -2.000, +0.000, +0.000, +0.000, -0.000}},
      {{-0.000, -2.000, +0.000, -0.000, -0.000, +0.000}},
      {{-0.000, -2.000, +0.000, +2.000, +4.000, -0.000}},
      {{-0.000, -2.000, +0.000, -2.000, -4.000, +0.000}},
      {{-0.000, -2.000, -0.000, +0.000, +0.000, +0.000}},
      {{-0.000, -2.000, -0.000, -0.000, +0.000, +0.000}},
      {{-0.000, -2.000, -0.000, +2.000, +4.000, +0.000}},
      {{-0.000, -2.000, -0.000, -2.000, -4.000, +0.000}},
      {{-0.000, -2.000, +2.000, +0.000, +0.000, -4.000}},
      {{-0.000, -2.000, +2.000, -0.000, -0.000, -4.000}},
      {{-0.000, -2.000, +2.000, +2.000, +4.000, -4.000}},
      {{-0.000, -2.000, +2.000, -2.000, -4.000, -4.000}},
      {{-0.000, -2.000, -2.000, +0.000, +0.000, +4.000}},
      {{-0.000, -2.000, -2.000, -0.000, +0.000, +4.000}},
      {{-0.000, -2.000, -2.000, +2.000, +4.000, +4.000}},
      {{-0.000, -2.000, -2.000, -2.000, -4.000, +4.000}},
      {{+2.000, +0.000, +0.000, +0.000, +0.000, +0.000}},
      {{+2.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
      {{+2.000, +0.000, +0.000, +2.000, +0.000, +4.000}},
      {{+2.000, +0.000, +0.000, -2.000, +0.000, -4.000}},
      {{+2.000, +0.000, -0.000, +0.000, -0.000, +0.000}},
      {{+2.000, +0.000, -0.000, -0.000, +0.000, -0.000}},
      {{+2.000, +0.000, -0.000, +2.000, -0.000, +4.000}},
      {{+2.000, +0.000, -0.000, -2.000, +0.000, -4.000}},
      {{+2.000, +0.000, +2.000, +0.000, +4.000, +0.000}},
      {{+2.000, +0.000, +2.000, -0.000, +4.000, +0.000}},
      {{+2.000, +0.000, +2.000, +2.000, +4.000, +4.000}},
      {{+2.000, +0.000, +2.000, -2.000, +4.000, -4.000}},
      {{+2.000, +0.000, -2.000, +0.000, -4.000, +0.000}},
      {{+2.000, +0.000, -2.000, -0.000, -4.000, -0.000}},
      {{+2.000, +0.000, -2.000, +2.000, -4.000, +4.000}},
      {{+2.000, +0.000, -2.000, -2.000, -4.000, -4.000}},
      {{+2.000, -0.000, +0.000, +0.000, +0.000, +0.000}},
      {{+2.000, -0.000, +0.000, -0.000, +0.000, -0.000}},
      {{+2.000, -0.000, +0.000, +2.000, +0.000, +4.000}},
      {{+2.000, -0.000, +0.000, -2.000, +0.000, -4.000}},
      {{+2.000, -0.000, -0.000, +0.000, +0.000, +0.000}},
      {{+2.000, -0.000, -0.000, -0.000, -0.000, +0.000}},
      {{+2.000, -0.000, -0.000, +2.000, +0.000, +4.000}},
      {{+2.000, -0.000, -0.000, -2.000, -0.000, -4.000}},
      {{+2.000, -0.000, +2.000, +0.000, +4.000, +0.000}},
      {{+2.000, -0.000, +2.000, -0.000, +4.000, -0.000}},
      {{+2.000, -0.000, +2.000, +2.000, +4.000, +4.000}},
      {{+2.000, -0.000, +2.000, -2.000, +4.000, -4.000}},
      {{+2.000, -0.000, -2.000, +0.000, -4.000, +0.000}},
      {{+2.000, -0.000, -2.000, -0.000, -4.000, +0.000}},
      {{+2.000, -0.000, -2.000, +2.000, -4.000, +4.000}},
      {{+2.000, -0.000, -2.000, -2.000, -4.000, -4.000}},
      {{+2.000, +2.000, +0.000, +0.000, +0.000, +0.000}},
      {{+2.000, +2.000, +0.000, -0.000, +0.000, +0.000}},
      {{+2.000, +2.000, +0.000, +2.000, -4.000, +4.000}},
      {{+2.000, +2.000, +0.000, -2.000, +4.000, -4.000}},
      {{+2.000, +2.000, -0.000, +0.000, -0.000, +0.000}},
      {{+2.000, +2.000, -0.000, -0.000, +0.000, -0.000}},
      {{+2.000, +2.000, -0.000, +2.000, -4.000, +4.000}},
      {{+2.000, +2.000, -0.000, -2.000, +4.000, -4.000}},
      {{+2.000, +2.000, +2.000, +0.000, +4.000, +4.000}},
      {{+2.000, +2.000, +2.000, -0.000, +4.000, +4.000}},
      {{+2.000, +2.000, +2.000, +2.000, +0.000, +8.000}},
      {{+2.000, +2.000, +2.000, -2.000, +8.000, +0.000}},
      {{+2.000, +2.000, -2.000, +0.000, -4.000, -4.000}},
      {{+2.000, +2.000, -2.000, -0.000, -4.000, -4.000}},
      {{+2.000, +2.000, -2.000, +2.000, -8.000, +0.000}},
      {{+2.000, +2.000, -2.000, -2.000, +0.000, -8.000}},
      {{+2.000, -2.000, +0.000, +0.000, +0.000, +0.000}},
      {{+2.000, -2.000, +0.000, -0.000, +0.000, -0.000}},
      {{+2.000, -2.000, +0.000, +2.000, +4.000, +4.000}},
      {{+2.000, -2.000, +0.000, -2.000, -4.000, -4.000}},
      {{+2.000, -2.000, -0.000, +0.000, +0.000, +0.000}},
      {{+2.000, -2.000, -0.000, -0.000, -0.000, +0.000}},
      {{+2.000, -2.000, -0.000, +2.000, +4.000, +4.000}},
      {{+2.000, -2.000, -0.000, -2.000, -4.000, -4.000}},
      {{+2.000, -2.000, +2.000, +0.000, +4.000, -4.000}},
      {{+2.000, -2.000, +2.000, -0.000, +4.000, -4.000}},
      {{+2.000, -2.000, +2.000, +2.000, +8.000, +0.000}},
      {{+2.000, -2.000, +2.000, -2.000, +0.000, -8.000}},
      {{+2.000, -2.000, -2.000, +0.000, -4.000, +4.000}},
      {{+2.000, -2.000, -2.000, -0.000, -4.000, +4.000}},
      {{+2.000, -2.000, -2.000, +2.000, +0.000, +8.000}},
      {{+2.000, -2.000, -2.000, -2.000, -8.000, +0.000}},
      {{-2.000, +0.000, +0.000, +0.000, -0.000, +0.000}},
      {{-2.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
      {{-2.000, +0.000, +0.000, +2.000, -0.000, -4.000}},
      {{-2.000, +0.000, +0.000, -2.000, +0.000, +4.000}},
      {{-2.000, +0.000, -0.000, +0.000, +0.000, -0.000}},
      {{-2.000, +0.000, -0.000, -0.000, +0.000, +0.000}},
      {{-2.000, +0.000, -0.000, +2.000, +0.000, -4.000}},
      {{-2.000, +0.000, -0.000, -2.000, +0.000, +4.000}},
      {{-2.000, +0.000, +2.000, +0.000, -4.000, +0.000}},
      {{-2.000, +0.000, +2.000, -0.000, -4.000, +0.000}},
      {{-2.000, +0.000, +2.000, +2.000, -4.000, -4.000}},
      {{-2.000, +0.000, +2.000, -2.000, -4.000, +4.000}},
      {{-2.000, +0.000, -2.000, +0.000, +4.000, -0.000}},
      {{-2.000, +0.000, -2.000, -0.000, +4.000, +0.000}},
      {{-2.000, +0.000, -2.000, +2.000, +4.000, -4.000}},
      {{-2.000, +0.000, -2.000, -2.000, +4.000, +4.000}},
      {{-2.000, -0.000, +0.000, +0.000, +0.000, -0.000}},
      {{-2.000, -0.000, +0.000, -0.000, -0.000, +0.000}},
      {{-2.000, -0.000, +0.000, +2.000, +0.000, -4.000}},
      {{-2.000, -0.000, +0.000, -2.000, -0.000, +4.000}},
      {{-2.000, -0.000, -0.000, +0.000, +0.000, +0.000}},
      {{-2.000, -0.000, -0.000, -0.000, +0.000, +0.000}},
      {{-2.000, -0.000, -0.000, +2.000, +0.000, -4.000}},
      {{-2.000, -0.000, -0.000, -2.000, +0.000, +4.000}},
      {{-2.000, -0.000, +2.000, +0.000, -4.000, -0.000}},
      {{-2.000, -0.000, +2.000, -0.000, -4.000, +0.000}},
      {{-2.000, -0.000, +2.000, +2.000, -4.000, -4.000}},
      {{-2.000, -0.000, +2.000, -2.000, -4.000, +4.000}},
      {{-2.000, -0.000, -2.000, +0.000, +4.000, +0.000}},
      {{-2.000, -0.000, -2.000, -0.000, +4.000, +0.000}},
      {{-2.000, -0.000, -2.000, +2.000, +4.000, -4.000}},
      {{-2.000, -0.000, -2.000, -2.000, +4.000, +4.000}},
      {{-2.000, +2.000, +0.000, +0.000, -0.000, +0.000}},
      {{-2.000, +2.000, +0.000, -0.000, +0.000, +0.000}},
      {{-2.000, +2.000, +0.000, +2.000, -4.000, -4.000}},
      {{-2.000, +2.000, +0.000, -2.000, +4.000, +4.000}},
      {{-2.000, +2.000, -0.000, +0.000, +0.000, -0.000}},
      {{-2.000, +2.000, -0.000, -0.000, +0.000, +0.000}},
      {{-2.000, +2.000, -0.000, +2.000, -4.000, -4.000}},
      {{-2.000, +2.000, -0.000, -2.000, +4.000, +4.000}},
      {{-2.000, +2.000, +2.000, +0.000, -4.000, +4.000}},
      {{-2.000, +2.000, +2.000, -0.000, -4.000, +4.000}},
      {{-2.000, +2.000, +2.000, +2.000, -8.000, +0.000}},
      {{-2.000, +2.000, +2.000, -2.000, +0.000, +8.000}},
      {{-2.000, +2.000, -2.000, +0.000, +4.000, -4.000}},
      {{-2.000, +2.000, -2.000, -0.000, +4.000, -4.000}},
      {{-2.000, +2.000, -2.000, +2.000, +0.000, -8.000}},
      {{-2.000, +2.000, -2.000, -2.000, +8.000, +0.000}},
      {{-2.000, -2.000, +0.000, +0.000, +0.000, -0.000}},
      {{-2.000, -2.000, +0.000, -0.000, -0.000, +0.000}},
      {{-2.000, -2.000, +0.000, +2.000, +4.000, -4.000}},
      {{-2.000, -2.000, +0.000, -2.000, -4.000, +4.000}},
      {{-2.000, -2.000, -0.000, +0.000, +0.000, +0.000}},
      {{-2.000, -2.000, -0.000, -0.000, +0.000, +0.000}},
      {{-2.000, -2.000, -0.000, +2.000, +4.000, -4.000}},
      {{-2.000, -2.000, -0.000, -2.000, -4.000, +4.000}},
      {{-2.000, -2.000, +2.000, +0.000, -4.000, -4.000}},
      {{-2.000, -2.000, +2.000, -0.000, -4.000, -4.000}},
      {{-2.000, -2.000, +2.000, +2.000, +0.000, -8.000}},
      {{-2.000, -2.000, +2.000, -2.000, -8.000, +0.000}},
      {{-2.000, -2.000, -2.000, +0.000, +4.000, +4.000}},
      {{-2.000, -2.000, -2.000, -0.000, +4.000, +4.000}},
      {{-2.000, -2.000, -2.000, +2.000, +8.000, +0.000}},
      {{-2.000, -2.000, -2.000, -2.000, +0.000, +8.000}},
   }};

   for (unsigned i = 0; i < mul_data.size(); ++i)
   {
      result = T(mul_data[i][0], mul_data[i][1]) * T(mul_data[i][2], mul_data[i][3]);
      BOOST_TEST(result.real() == mul_data[i][4]);
      BOOST_TEST(result.imag() == mul_data[i][5]);
      BOOST_TEST((bool)signbit(result.real()) == signbit(mul_data[i][4]));
      BOOST_TEST((bool)signbit(result.imag()) == signbit(mul_data[i][5]));
   }

   //
   // Divide the permutations which give at least one zero:
   //
   static constexpr std::array<std::array<double, 6>, 256> div_data = 
   { {
      {{ +0.000, +0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, +0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, +0.000, +0.000, +2.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, +0.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, +0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, +0.000, -0.000, +2.000, +0.000, -0.000 }},
      {{ +0.000, +0.000, -0.000, -2.000, -0.000, +0.000 }},
      {{ +0.000, +0.000, +2.000, +0.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, +2.000, -0.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, +2.000, +2.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, +2.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, +0.000, -2.000, +0.000, +0.000, -0.000 }},
      {{ +0.000, +0.000, -2.000, -0.000, -0.000, +0.000 }},
      {{ +0.000, +0.000, -2.000, +2.000, +0.000, -0.000 }},
      {{ +0.000, +0.000, -2.000, -2.000, -0.000, +0.000 }},
      {{ +0.000, -0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, -0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, -0.000, +0.000, +2.000, +0.000, -0.000 }},
      {{ +0.000, -0.000, +0.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, -0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, -0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +0.000, -0.000, -0.000, +2.000, -0.000, +0.000 }},
      {{ +0.000, -0.000, -0.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, -0.000, +2.000, +0.000, +0.000, -0.000 }},
      {{ +0.000, -0.000, +2.000, -0.000, +0.000, +0.000 }},
      {{ +0.000, -0.000, +2.000, +2.000, +0.000, -0.000 }},
      {{ +0.000, -0.000, +2.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, -0.000, -2.000, +0.000, -0.000, +0.000 }},
      {{ +0.000, -0.000, -2.000, -0.000, +0.000, +0.000 }},
      {{ +0.000, -0.000, -2.000, +2.000, -0.000, +0.000 }},
      {{ +0.000, -0.000, -2.000, -2.000, +0.000, +0.000 }},
      {{ +0.000, +2.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ +0.000, +2.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ +0.000, +2.000, +0.000, +2.000, +1.000, +0.000 }},
      {{ +0.000, +2.000, +0.000, -2.000, -1.000, +0.000 }},
      {{ +0.000, +2.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ +0.000, +2.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ +0.000, +2.000, -0.000, +2.000, +1.000, -0.000 }},
      {{ +0.000, +2.000, -0.000, -2.000, -1.000, +0.000 }},
      {{ +0.000, +2.000, +2.000, +0.000, +0.000, +1.000 }},
      {{ +0.000, +2.000, +2.000, -0.000, +0.000, +1.000 }},
      {{ +0.000, +2.000, +2.000, +2.000, +0.500, +0.500 }},
      {{ +0.000, +2.000, +2.000, -2.000, -0.500, +0.500 }},
      {{ +0.000, +2.000, -2.000, +0.000, +0.000, -1.000 }},
      {{ +0.000, +2.000, -2.000, -0.000, -0.000, -1.000 }},
      {{ +0.000, +2.000, -2.000, +2.000, +0.500, -0.500 }},
      {{ +0.000, +2.000, -2.000, -2.000, -0.500, -0.500 }},
      {{ +0.000, -2.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ +0.000, -2.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ +0.000, -2.000, +0.000, +2.000, -1.000, -0.000 }},
      {{ +0.000, -2.000, +0.000, -2.000, +1.000, +0.000 }},
      {{ +0.000, -2.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ +0.000, -2.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ +0.000, -2.000, -0.000, +2.000, -1.000, +0.000 }},
      {{ +0.000, -2.000, -0.000, -2.000, +1.000, +0.000 }},
      {{ +0.000, -2.000, +2.000, +0.000, +0.000, -1.000 }},
      {{ +0.000, -2.000, +2.000, -0.000, +0.000, -1.000 }},
      {{ +0.000, -2.000, +2.000, +2.000, -0.500, -0.500 }},
      {{ +0.000, -2.000, +2.000, -2.000, +0.500, -0.500 }},
      {{ +0.000, -2.000, -2.000, +0.000, -0.000, +1.000 }},
      {{ +0.000, -2.000, -2.000, -0.000, +0.000, +1.000 }},
      {{ +0.000, -2.000, -2.000, +2.000, -0.500, +0.500 }},
      {{ +0.000, -2.000, -2.000, -2.000, +0.500, +0.500 }},
      {{ -0.000, +0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, +0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, +0.000, +0.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, +0.000, -2.000, -0.000, +0.000 }},
      {{ -0.000, +0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, +0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, +0.000, -0.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, -0.000, -2.000, +0.000, -0.000 }},
      {{ -0.000, +0.000, +2.000, +0.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, +2.000, -0.000, -0.000, +0.000 }},
      {{ -0.000, +0.000, +2.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, +2.000, -2.000, -0.000, +0.000 }},
      {{ -0.000, +0.000, -2.000, +0.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, -2.000, -0.000, +0.000, -0.000 }},
      {{ -0.000, +0.000, -2.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, +0.000, -2.000, -2.000, +0.000, -0.000 }},
      {{ -0.000, -0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, -0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, -0.000, +0.000, +2.000, -0.000, +0.000 }},
      {{ -0.000, -0.000, +0.000, -2.000, +0.000, -0.000 }},
      {{ -0.000, -0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, -0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -0.000, -0.000, -0.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, -0.000, -0.000, -2.000, +0.000, +0.000 }},
      {{ -0.000, -0.000, +2.000, +0.000, -0.000, +0.000 }},
      {{ -0.000, -0.000, +2.000, -0.000, +0.000, -0.000 }},
      {{ -0.000, -0.000, +2.000, +2.000, -0.000, +0.000 }},
      {{ -0.000, -0.000, +2.000, -2.000, +0.000, -0.000 }},
      {{ -0.000, -0.000, -2.000, +0.000, +0.000, +0.000 }},
      {{ -0.000, -0.000, -2.000, -0.000, +0.000, +0.000 }},
      {{ -0.000, -0.000, -2.000, +2.000, +0.000, +0.000 }},
      {{ -0.000, -0.000, -2.000, -2.000, +0.000, +0.000 }},
      {{ -0.000, +2.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ -0.000, +2.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ -0.000, +2.000, +0.000, +2.000, +1.000, +0.000 }},
      {{ -0.000, +2.000, +0.000, -2.000, -1.000, +0.000 }},
      {{ -0.000, +2.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ -0.000, +2.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ -0.000, +2.000, -0.000, +2.000, +1.000, +0.000 }},
      {{ -0.000, +2.000, -0.000, -2.000, -1.000, -0.000 }},
      {{ -0.000, +2.000, +2.000, +0.000, +0.000, +1.000 }},
      {{ -0.000, +2.000, +2.000, -0.000, -0.000, +1.000 }},
      {{ -0.000, +2.000, +2.000, +2.000, +0.500, +0.500 }},
      {{ -0.000, +2.000, +2.000, -2.000, -0.500, +0.500 }},
      {{ -0.000, +2.000, -2.000, +0.000, +0.000, -1.000 }},
      {{ -0.000, +2.000, -2.000, -0.000, +0.000, -1.000 }},
      {{ -0.000, +2.000, -2.000, +2.000, +0.500, -0.500 }},
      {{ -0.000, +2.000, -2.000, -2.000, -0.500, -0.500 }},
      {{ -0.000, -2.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ -0.000, -2.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity() }},
      {{ -0.000, -2.000, +0.000, +2.000, -1.000, +0.000 }},
      {{ -0.000, -2.000, +0.000, -2.000, +1.000, -0.000 }},
      {{ -0.000, -2.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ -0.000, -2.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity() }},
      {{ -0.000, -2.000, -0.000, +2.000, -1.000, +0.000 }},
      {{ -0.000, -2.000, -0.000, -2.000, +1.000, +0.000 }},
      {{ -0.000, -2.000, +2.000, +0.000, -0.000, -1.000 }},
      {{ -0.000, -2.000, +2.000, -0.000, +0.000, -1.000 }},
      {{ -0.000, -2.000, +2.000, +2.000, -0.500, -0.500 }},
      {{ -0.000, -2.000, +2.000, -2.000, +0.500, -0.500 }},
      {{ -0.000, -2.000, -2.000, +0.000, +0.000, +1.000 }},
      {{ -0.000, -2.000, -2.000, -0.000, +0.000, +1.000 }},
      {{ -0.000, -2.000, -2.000, +2.000, -0.500, +0.500 }},
      {{ -0.000, -2.000, -2.000, -2.000, +0.500, +0.500 }},
      {{ +2.000, +0.000, +0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, +0.000, +0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, +0.000, +0.000, +2.000, +0.000, -1.000 }},
      {{ +2.000, +0.000, +0.000, -2.000, +0.000, +1.000 }},
      {{ +2.000, +0.000, -0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, +0.000, -0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, +0.000, -0.000, +2.000, +0.000, -1.000 }},
      {{ +2.000, +0.000, -0.000, -2.000, -0.000, +1.000 }},
      {{ +2.000, +0.000, +2.000, +0.000, +1.000, +0.000 }},
      {{ +2.000, +0.000, +2.000, -0.000, +1.000, +0.000 }},
      {{ +2.000, +0.000, +2.000, +2.000, +0.500, -0.500 }},
      {{ +2.000, +0.000, +2.000, -2.000, +0.500, +0.500 }},
      {{ +2.000, +0.000, -2.000, +0.000, -1.000, -0.000 }},
      {{ +2.000, +0.000, -2.000, -0.000, -1.000, +0.000 }},
      {{ +2.000, +0.000, -2.000, +2.000, -0.500, -0.500 }},
      {{ +2.000, +0.000, -2.000, -2.000, -0.500, +0.500 }},
      {{ +2.000, -0.000, +0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, -0.000, +0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, -0.000, +0.000, +2.000, +0.000, -1.000 }},
      {{ +2.000, -0.000, +0.000, -2.000, +0.000, +1.000 }},
      {{ +2.000, -0.000, -0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, -0.000, -0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ +2.000, -0.000, -0.000, +2.000, -0.000, -1.000 }},
      {{ +2.000, -0.000, -0.000, -2.000, +0.000, +1.000 }},
      {{ +2.000, -0.000, +2.000, +0.000, +1.000, -0.000 }},
      {{ +2.000, -0.000, +2.000, -0.000, +1.000, +0.000 }},
      {{ +2.000, -0.000, +2.000, +2.000, +0.500, -0.500 }},
      {{ +2.000, -0.000, +2.000, -2.000, +0.500, +0.500 }},
      {{ +2.000, -0.000, -2.000, +0.000, -1.000, +0.000 }},
      {{ +2.000, -0.000, -2.000, -0.000, -1.000, +0.000 }},
      {{ +2.000, -0.000, -2.000, +2.000, -0.500, -0.500 }},
      {{ +2.000, -0.000, -2.000, -2.000, -0.500, +0.500 }},
      {{ +2.000, +2.000, +0.000, +0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ +2.000, +2.000, +0.000, -0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ +2.000, +2.000, +0.000, +2.000, +1.000, -1.000 }},
      {{ +2.000, +2.000, +0.000, -2.000, -1.000, +1.000 }},
      {{ +2.000, +2.000, -0.000, +0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ +2.000, +2.000, -0.000, -0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ +2.000, +2.000, -0.000, +2.000, +1.000, -1.000 }},
      {{ +2.000, +2.000, -0.000, -2.000, -1.000, +1.000 }},
      {{ +2.000, +2.000, +2.000, +0.000, +1.000, +1.000 }},
      {{ +2.000, +2.000, +2.000, -0.000, +1.000, +1.000 }},
      {{ +2.000, +2.000, +2.000, +2.000, +1.000, +0.000 }},
      {{ +2.000, +2.000, +2.000, -2.000, +0.000, +1.000 }},
      {{ +2.000, +2.000, -2.000, +0.000, -1.000, -1.000 }},
      {{ +2.000, +2.000, -2.000, -0.000, -1.000, -1.000 }},
      {{ +2.000, +2.000, -2.000, +2.000, +0.000, -1.000 }},
      {{ +2.000, +2.000, -2.000, -2.000, -1.000, +0.000 }},
      {{ +2.000, -2.000, +0.000, +0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ +2.000, -2.000, +0.000, -0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ +2.000, -2.000, +0.000, +2.000, -1.000, -1.000 }},
      {{ +2.000, -2.000, +0.000, -2.000, +1.000, +1.000 }},
      {{ +2.000, -2.000, -0.000, +0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ +2.000, -2.000, -0.000, -0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ +2.000, -2.000, -0.000, +2.000, -1.000, -1.000 }},
      {{ +2.000, -2.000, -0.000, -2.000, +1.000, +1.000 }},
      {{ +2.000, -2.000, +2.000, +0.000, +1.000, -1.000 }},
      {{ +2.000, -2.000, +2.000, -0.000, +1.000, -1.000 }},
      {{ +2.000, -2.000, +2.000, +2.000, +0.000, -1.000 }},
      {{ +2.000, -2.000, +2.000, -2.000, +1.000, +0.000 }},
      {{ +2.000, -2.000, -2.000, +0.000, -1.000, +1.000 }},
      {{ +2.000, -2.000, -2.000, -0.000, -1.000, +1.000 }},
      {{ +2.000, -2.000, -2.000, +2.000, -1.000, +0.000 }},
      {{ +2.000, -2.000, -2.000, -2.000, +0.000, +1.000 }},
      {{ -2.000, +0.000, +0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, +0.000, +0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, +0.000, +0.000, +2.000, +0.000, +1.000 }},
      {{ -2.000, +0.000, +0.000, -2.000, -0.000, -1.000 }},
      {{ -2.000, +0.000, -0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, +0.000, -0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, +0.000, -0.000, +2.000, +0.000, +1.000 }},
      {{ -2.000, +0.000, -0.000, -2.000, +0.000, -1.000 }},
      {{ -2.000, +0.000, +2.000, +0.000, -1.000, +0.000 }},
      {{ -2.000, +0.000, +2.000, -0.000, -1.000, +0.000 }},
      {{ -2.000, +0.000, +2.000, +2.000, -0.500, +0.500 }},
      {{ -2.000, +0.000, +2.000, -2.000, -0.500, -0.500 }},
      {{ -2.000, +0.000, -2.000, +0.000, +1.000, +0.000 }},
      {{ -2.000, +0.000, -2.000, -0.000, +1.000, -0.000 }},
      {{ -2.000, +0.000, -2.000, +2.000, +0.500, +0.500 }},
      {{ -2.000, +0.000, -2.000, -2.000, +0.500, -0.500 }},
      {{ -2.000, -0.000, +0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, -0.000, +0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, -0.000, +0.000, +2.000, -0.000, +1.000 }},
      {{ -2.000, -0.000, +0.000, -2.000, +0.000, -1.000 }},
      {{ -2.000, -0.000, -0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, -0.000, -0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN() }},
      {{ -2.000, -0.000, -0.000, +2.000, +0.000, +1.000 }},
      {{ -2.000, -0.000, -0.000, -2.000, +0.000, -1.000 }},
      {{ -2.000, -0.000, +2.000, +0.000, -1.000, +0.000 }},
      {{ -2.000, -0.000, +2.000, -0.000, -1.000, -0.000 }},
      {{ -2.000, -0.000, +2.000, +2.000, -0.500, +0.500 }},
      {{ -2.000, -0.000, +2.000, -2.000, -0.500, -0.500 }},
      {{ -2.000, -0.000, -2.000, +0.000, +1.000, +0.000 }},
      {{ -2.000, -0.000, -2.000, -0.000, +1.000, +0.000 }},
      {{ -2.000, -0.000, -2.000, +2.000, +0.500, +0.500 }},
      {{ -2.000, -0.000, -2.000, -2.000, +0.500, -0.500 }},
      {{ -2.000, +2.000, +0.000, +0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ -2.000, +2.000, +0.000, -0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ -2.000, +2.000, +0.000, +2.000, +1.000, +1.000 }},
      {{ -2.000, +2.000, +0.000, -2.000, -1.000, -1.000 }},
      {{ -2.000, +2.000, -0.000, +0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ -2.000, +2.000, -0.000, -0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ -2.000, +2.000, -0.000, +2.000, +1.000, +1.000 }},
      {{ -2.000, +2.000, -0.000, -2.000, -1.000, -1.000 }},
      {{ -2.000, +2.000, +2.000, +0.000, -1.000, +1.000 }},
      {{ -2.000, +2.000, +2.000, -0.000, -1.000, +1.000 }},
      {{ -2.000, +2.000, +2.000, +2.000, +0.000, +1.000 }},
      {{ -2.000, +2.000, +2.000, -2.000, -1.000, +0.000 }},
      {{ -2.000, +2.000, -2.000, +0.000, +1.000, -1.000 }},
      {{ -2.000, +2.000, -2.000, -0.000, +1.000, -1.000 }},
      {{ -2.000, +2.000, -2.000, +2.000, +1.000, +0.000 }},
      {{ -2.000, +2.000, -2.000, -2.000, +0.000, -1.000 }},
      {{ -2.000, -2.000, +0.000, +0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ -2.000, -2.000, +0.000, -0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity() }},
      {{ -2.000, -2.000, +0.000, +2.000, -1.000, +1.000 }},
      {{ -2.000, -2.000, +0.000, -2.000, +1.000, -1.000 }},
      {{ -2.000, -2.000, -0.000, +0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ -2.000, -2.000, -0.000, -0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity() }},
      {{ -2.000, -2.000, -0.000, +2.000, -1.000, +1.000 }},
      {{ -2.000, -2.000, -0.000, -2.000, +1.000, -1.000 }},
      {{ -2.000, -2.000, +2.000, +0.000, -1.000, -1.000 }},
      {{ -2.000, -2.000, +2.000, -0.000, -1.000, -1.000 }},
      {{ -2.000, -2.000, +2.000, +2.000, -1.000, +0.000 }},
      {{ -2.000, -2.000, +2.000, -2.000, +0.000, -1.000 }},
      {{ -2.000, -2.000, -2.000, +0.000, +1.000, +1.000 }},
      {{ -2.000, -2.000, -2.000, -0.000, +1.000, +1.000 }},
      {{ -2.000, -2.000, -2.000, +2.000, +0.000, +1.000 }},
      {{ -2.000, -2.000, -2.000, -2.000, +1.000, +0.000 }},   
   }};

   for (unsigned i = 0; i < div_data.size(); ++i)
   {
      if (!(isnan)(div_data[i][4]) && !(isnan)(div_data[i][5]))
      {
         result = T(div_data[i][0], div_data[i][1]) / T(div_data[i][2], div_data[i][3]);
         BOOST_TEST(result.real() == div_data[i][4]);
         BOOST_TEST(result.imag() == div_data[i][5]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(div_data[i][4]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(div_data[i][5]));
      }
   }


   //
   // Add the permutations:
   //
   static constexpr std::array<std::array<double, 6>, 256> add_data =
   { {
       {{+0.000, +0.000, +0.000, +0.000, +0.000, +0.000}},
       {{+0.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
       {{+0.000, +0.000, +0.000, +2.000, +0.000, +2.000}},
       {{+0.000, +0.000, +0.000, -2.000, +0.000, -2.000}},
       {{+0.000, +0.000, -0.000, +0.000, +0.000, +0.000}},
       {{+0.000, +0.000, -0.000, -0.000, +0.000, +0.000}},
       {{+0.000, +0.000, -0.000, +2.000, +0.000, +2.000}},
       {{+0.000, +0.000, -0.000, -2.000, +0.000, -2.000}},
       {{+0.000, +0.000, +2.000, +0.000, +2.000, +0.000}},
       {{+0.000, +0.000, +2.000, -0.000, +2.000, +0.000}},
       {{+0.000, +0.000, +2.000, +2.000, +2.000, +2.000}},
       {{+0.000, +0.000, +2.000, -2.000, +2.000, -2.000}},
       {{+0.000, +0.000, -2.000, +0.000, -2.000, +0.000}},
       {{+0.000, +0.000, -2.000, -0.000, -2.000, +0.000}},
       {{+0.000, +0.000, -2.000, +2.000, -2.000, +2.000}},
       {{+0.000, +0.000, -2.000, -2.000, -2.000, -2.000}},
       {{+0.000, -0.000, +0.000, +0.000, +0.000, +0.000}},
       {{+0.000, -0.000, +0.000, -0.000, +0.000, -0.000}},
       {{+0.000, -0.000, +0.000, +2.000, +0.000, +2.000}},
       {{+0.000, -0.000, +0.000, -2.000, +0.000, -2.000}},
       {{+0.000, -0.000, -0.000, +0.000, +0.000, +0.000}},
       {{+0.000, -0.000, -0.000, -0.000, +0.000, -0.000}},
       {{+0.000, -0.000, -0.000, +2.000, +0.000, +2.000}},
       {{+0.000, -0.000, -0.000, -2.000, +0.000, -2.000}},
       {{+0.000, -0.000, +2.000, +0.000, +2.000, +0.000}},
       {{+0.000, -0.000, +2.000, -0.000, +2.000, -0.000}},
       {{+0.000, -0.000, +2.000, +2.000, +2.000, +2.000}},
       {{+0.000, -0.000, +2.000, -2.000, +2.000, -2.000}},
       {{+0.000, -0.000, -2.000, +0.000, -2.000, +0.000}},
       {{+0.000, -0.000, -2.000, -0.000, -2.000, -0.000}},
       {{+0.000, -0.000, -2.000, +2.000, -2.000, +2.000}},
       {{+0.000, -0.000, -2.000, -2.000, -2.000, -2.000}},
       {{+0.000, +2.000, +0.000, +0.000, +0.000, +2.000}},
       {{+0.000, +2.000, +0.000, -0.000, +0.000, +2.000}},
       {{+0.000, +2.000, +0.000, +2.000, +0.000, +4.000}},
       {{+0.000, +2.000, +0.000, -2.000, +0.000, +0.000}},
       {{+0.000, +2.000, -0.000, +0.000, +0.000, +2.000}},
       {{+0.000, +2.000, -0.000, -0.000, +0.000, +2.000}},
       {{+0.000, +2.000, -0.000, +2.000, +0.000, +4.000}},
       {{+0.000, +2.000, -0.000, -2.000, +0.000, +0.000}},
       {{+0.000, +2.000, +2.000, +0.000, +2.000, +2.000}},
       {{+0.000, +2.000, +2.000, -0.000, +2.000, +2.000}},
       {{+0.000, +2.000, +2.000, +2.000, +2.000, +4.000}},
       {{+0.000, +2.000, +2.000, -2.000, +2.000, +0.000}},
       {{+0.000, +2.000, -2.000, +0.000, -2.000, +2.000}},
       {{+0.000, +2.000, -2.000, -0.000, -2.000, +2.000}},
       {{+0.000, +2.000, -2.000, +2.000, -2.000, +4.000}},
       {{+0.000, +2.000, -2.000, -2.000, -2.000, +0.000}},
       {{+0.000, -2.000, +0.000, +0.000, +0.000, -2.000}},
       {{+0.000, -2.000, +0.000, -0.000, +0.000, -2.000}},
       {{+0.000, -2.000, +0.000, +2.000, +0.000, +0.000}},
       {{+0.000, -2.000, +0.000, -2.000, +0.000, -4.000}},
       {{+0.000, -2.000, -0.000, +0.000, +0.000, -2.000}},
       {{+0.000, -2.000, -0.000, -0.000, +0.000, -2.000}},
       {{+0.000, -2.000, -0.000, +2.000, +0.000, +0.000}},
       {{+0.000, -2.000, -0.000, -2.000, +0.000, -4.000}},
       {{+0.000, -2.000, +2.000, +0.000, +2.000, -2.000}},
       {{+0.000, -2.000, +2.000, -0.000, +2.000, -2.000}},
       {{+0.000, -2.000, +2.000, +2.000, +2.000, +0.000}},
       {{+0.000, -2.000, +2.000, -2.000, +2.000, -4.000}},
       {{+0.000, -2.000, -2.000, +0.000, -2.000, -2.000}},
       {{+0.000, -2.000, -2.000, -0.000, -2.000, -2.000}},
       {{+0.000, -2.000, -2.000, +2.000, -2.000, +0.000}},
       {{+0.000, -2.000, -2.000, -2.000, -2.000, -4.000}},
       {{-0.000, +0.000, +0.000, +0.000, +0.000, +0.000}},
       {{-0.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
       {{-0.000, +0.000, +0.000, +2.000, +0.000, +2.000}},
       {{-0.000, +0.000, +0.000, -2.000, +0.000, -2.000}},
       {{-0.000, +0.000, -0.000, +0.000, -0.000, +0.000}},
       {{-0.000, +0.000, -0.000, -0.000, -0.000, +0.000}},
       {{-0.000, +0.000, -0.000, +2.000, -0.000, +2.000}},
       {{-0.000, +0.000, -0.000, -2.000, -0.000, -2.000}},
       {{-0.000, +0.000, +2.000, +0.000, +2.000, +0.000}},
       {{-0.000, +0.000, +2.000, -0.000, +2.000, +0.000}},
       {{-0.000, +0.000, +2.000, +2.000, +2.000, +2.000}},
       {{-0.000, +0.000, +2.000, -2.000, +2.000, -2.000}},
       {{-0.000, +0.000, -2.000, +0.000, -2.000, +0.000}},
       {{-0.000, +0.000, -2.000, -0.000, -2.000, +0.000}},
       {{-0.000, +0.000, -2.000, +2.000, -2.000, +2.000}},
       {{-0.000, +0.000, -2.000, -2.000, -2.000, -2.000}},
       {{-0.000, -0.000, +0.000, +0.000, +0.000, +0.000}},
       {{-0.000, -0.000, +0.000, -0.000, +0.000, -0.000}},
       {{-0.000, -0.000, +0.000, +2.000, +0.000, +2.000}},
       {{-0.000, -0.000, +0.000, -2.000, +0.000, -2.000}},
       {{-0.000, -0.000, -0.000, +0.000, -0.000, +0.000}},
       {{-0.000, -0.000, -0.000, -0.000, -0.000, -0.000}},
       {{-0.000, -0.000, -0.000, +2.000, -0.000, +2.000}},
       {{-0.000, -0.000, -0.000, -2.000, -0.000, -2.000}},
       {{-0.000, -0.000, +2.000, +0.000, +2.000, +0.000}},
       {{-0.000, -0.000, +2.000, -0.000, +2.000, -0.000}},
       {{-0.000, -0.000, +2.000, +2.000, +2.000, +2.000}},
       {{-0.000, -0.000, +2.000, -2.000, +2.000, -2.000}},
       {{-0.000, -0.000, -2.000, +0.000, -2.000, +0.000}},
       {{-0.000, -0.000, -2.000, -0.000, -2.000, -0.000}},
       {{-0.000, -0.000, -2.000, +2.000, -2.000, +2.000}},
       {{-0.000, -0.000, -2.000, -2.000, -2.000, -2.000}},
       {{-0.000, +2.000, +0.000, +0.000, +0.000, +2.000}},
       {{-0.000, +2.000, +0.000, -0.000, +0.000, +2.000}},
       {{-0.000, +2.000, +0.000, +2.000, +0.000, +4.000}},
       {{-0.000, +2.000, +0.000, -2.000, +0.000, +0.000}},
       {{-0.000, +2.000, -0.000, +0.000, -0.000, +2.000}},
       {{-0.000, +2.000, -0.000, -0.000, -0.000, +2.000}},
       {{-0.000, +2.000, -0.000, +2.000, -0.000, +4.000}},
       {{-0.000, +2.000, -0.000, -2.000, -0.000, +0.000}},
       {{-0.000, +2.000, +2.000, +0.000, +2.000, +2.000}},
       {{-0.000, +2.000, +2.000, -0.000, +2.000, +2.000}},
       {{-0.000, +2.000, +2.000, +2.000, +2.000, +4.000}},
       {{-0.000, +2.000, +2.000, -2.000, +2.000, +0.000}},
       {{-0.000, +2.000, -2.000, +0.000, -2.000, +2.000}},
       {{-0.000, +2.000, -2.000, -0.000, -2.000, +2.000}},
       {{-0.000, +2.000, -2.000, +2.000, -2.000, +4.000}},
       {{-0.000, +2.000, -2.000, -2.000, -2.000, +0.000}},
       {{-0.000, -2.000, +0.000, +0.000, +0.000, -2.000}},
       {{-0.000, -2.000, +0.000, -0.000, +0.000, -2.000}},
       {{-0.000, -2.000, +0.000, +2.000, +0.000, +0.000}},
       {{-0.000, -2.000, +0.000, -2.000, +0.000, -4.000}},
       {{-0.000, -2.000, -0.000, +0.000, -0.000, -2.000}},
       {{-0.000, -2.000, -0.000, -0.000, -0.000, -2.000}},
       {{-0.000, -2.000, -0.000, +2.000, -0.000, +0.000}},
       {{-0.000, -2.000, -0.000, -2.000, -0.000, -4.000}},
       {{-0.000, -2.000, +2.000, +0.000, +2.000, -2.000}},
       {{-0.000, -2.000, +2.000, -0.000, +2.000, -2.000}},
       {{-0.000, -2.000, +2.000, +2.000, +2.000, +0.000}},
       {{-0.000, -2.000, +2.000, -2.000, +2.000, -4.000}},
       {{-0.000, -2.000, -2.000, +0.000, -2.000, -2.000}},
       {{-0.000, -2.000, -2.000, -0.000, -2.000, -2.000}},
       {{-0.000, -2.000, -2.000, +2.000, -2.000, +0.000}},
       {{-0.000, -2.000, -2.000, -2.000, -2.000, -4.000}},
       {{+2.000, +0.000, +0.000, +0.000, +2.000, +0.000}},
       {{+2.000, +0.000, +0.000, -0.000, +2.000, +0.000}},
       {{+2.000, +0.000, +0.000, +2.000, +2.000, +2.000}},
       {{+2.000, +0.000, +0.000, -2.000, +2.000, -2.000}},
       {{+2.000, +0.000, -0.000, +0.000, +2.000, +0.000}},
       {{+2.000, +0.000, -0.000, -0.000, +2.000, +0.000}},
       {{+2.000, +0.000, -0.000, +2.000, +2.000, +2.000}},
       {{+2.000, +0.000, -0.000, -2.000, +2.000, -2.000}},
       {{+2.000, +0.000, +2.000, +0.000, +4.000, +0.000}},
       {{+2.000, +0.000, +2.000, -0.000, +4.000, +0.000}},
       {{+2.000, +0.000, +2.000, +2.000, +4.000, +2.000}},
       {{+2.000, +0.000, +2.000, -2.000, +4.000, -2.000}},
       {{+2.000, +0.000, -2.000, +0.000, +0.000, +0.000}},
       {{+2.000, +0.000, -2.000, -0.000, +0.000, +0.000}},
       {{+2.000, +0.000, -2.000, +2.000, +0.000, +2.000}},
       {{+2.000, +0.000, -2.000, -2.000, +0.000, -2.000}},
       {{+2.000, -0.000, +0.000, +0.000, +2.000, +0.000}},
       {{+2.000, -0.000, +0.000, -0.000, +2.000, -0.000}},
       {{+2.000, -0.000, +0.000, +2.000, +2.000, +2.000}},
       {{+2.000, -0.000, +0.000, -2.000, +2.000, -2.000}},
       {{+2.000, -0.000, -0.000, +0.000, +2.000, +0.000}},
       {{+2.000, -0.000, -0.000, -0.000, +2.000, -0.000}},
       {{+2.000, -0.000, -0.000, +2.000, +2.000, +2.000}},
       {{+2.000, -0.000, -0.000, -2.000, +2.000, -2.000}},
       {{+2.000, -0.000, +2.000, +0.000, +4.000, +0.000}},
       {{+2.000, -0.000, +2.000, -0.000, +4.000, -0.000}},
       {{+2.000, -0.000, +2.000, +2.000, +4.000, +2.000}},
       {{+2.000, -0.000, +2.000, -2.000, +4.000, -2.000}},
       {{+2.000, -0.000, -2.000, +0.000, +0.000, +0.000}},
       {{+2.000, -0.000, -2.000, -0.000, +0.000, -0.000}},
       {{+2.000, -0.000, -2.000, +2.000, +0.000, +2.000}},
       {{+2.000, -0.000, -2.000, -2.000, +0.000, -2.000}},
       {{+2.000, +2.000, +0.000, +0.000, +2.000, +2.000}},
       {{+2.000, +2.000, +0.000, -0.000, +2.000, +2.000}},
       {{+2.000, +2.000, +0.000, +2.000, +2.000, +4.000}},
       {{+2.000, +2.000, +0.000, -2.000, +2.000, +0.000}},
       {{+2.000, +2.000, -0.000, +0.000, +2.000, +2.000}},
       {{+2.000, +2.000, -0.000, -0.000, +2.000, +2.000}},
       {{+2.000, +2.000, -0.000, +2.000, +2.000, +4.000}},
       {{+2.000, +2.000, -0.000, -2.000, +2.000, +0.000}},
       {{+2.000, +2.000, +2.000, +0.000, +4.000, +2.000}},
       {{+2.000, +2.000, +2.000, -0.000, +4.000, +2.000}},
       {{+2.000, +2.000, +2.000, +2.000, +4.000, +4.000}},
       {{+2.000, +2.000, +2.000, -2.000, +4.000, +0.000}},
       {{+2.000, +2.000, -2.000, +0.000, +0.000, +2.000}},
       {{+2.000, +2.000, -2.000, -0.000, +0.000, +2.000}},
       {{+2.000, +2.000, -2.000, +2.000, +0.000, +4.000}},
       {{+2.000, +2.000, -2.000, -2.000, +0.000, +0.000}},
       {{+2.000, -2.000, +0.000, +0.000, +2.000, -2.000}},
       {{+2.000, -2.000, +0.000, -0.000, +2.000, -2.000}},
       {{+2.000, -2.000, +0.000, +2.000, +2.000, +0.000}},
       {{+2.000, -2.000, +0.000, -2.000, +2.000, -4.000}},
       {{+2.000, -2.000, -0.000, +0.000, +2.000, -2.000}},
       {{+2.000, -2.000, -0.000, -0.000, +2.000, -2.000}},
       {{+2.000, -2.000, -0.000, +2.000, +2.000, +0.000}},
       {{+2.000, -2.000, -0.000, -2.000, +2.000, -4.000}},
       {{+2.000, -2.000, +2.000, +0.000, +4.000, -2.000}},
       {{+2.000, -2.000, +2.000, -0.000, +4.000, -2.000}},
       {{+2.000, -2.000, +2.000, +2.000, +4.000, +0.000}},
       {{+2.000, -2.000, +2.000, -2.000, +4.000, -4.000}},
       {{+2.000, -2.000, -2.000, +0.000, +0.000, -2.000}},
       {{+2.000, -2.000, -2.000, -0.000, +0.000, -2.000}},
       {{+2.000, -2.000, -2.000, +2.000, +0.000, +0.000}},
       {{+2.000, -2.000, -2.000, -2.000, +0.000, -4.000}},
       {{-2.000, +0.000, +0.000, +0.000, -2.000, +0.000}},
       {{-2.000, +0.000, +0.000, -0.000, -2.000, +0.000}},
       {{-2.000, +0.000, +0.000, +2.000, -2.000, +2.000}},
       {{-2.000, +0.000, +0.000, -2.000, -2.000, -2.000}},
       {{-2.000, +0.000, -0.000, +0.000, -2.000, +0.000}},
       {{-2.000, +0.000, -0.000, -0.000, -2.000, +0.000}},
       {{-2.000, +0.000, -0.000, +2.000, -2.000, +2.000}},
       {{-2.000, +0.000, -0.000, -2.000, -2.000, -2.000}},
       {{-2.000, +0.000, +2.000, +0.000, +0.000, +0.000}},
       {{-2.000, +0.000, +2.000, -0.000, +0.000, +0.000}},
       {{-2.000, +0.000, +2.000, +2.000, +0.000, +2.000}},
       {{-2.000, +0.000, +2.000, -2.000, +0.000, -2.000}},
       {{-2.000, +0.000, -2.000, +0.000, -4.000, +0.000}},
       {{-2.000, +0.000, -2.000, -0.000, -4.000, +0.000}},
       {{-2.000, +0.000, -2.000, +2.000, -4.000, +2.000}},
       {{-2.000, +0.000, -2.000, -2.000, -4.000, -2.000}},
       {{-2.000, -0.000, +0.000, +0.000, -2.000, +0.000}},
       {{-2.000, -0.000, +0.000, -0.000, -2.000, -0.000}},
       {{-2.000, -0.000, +0.000, +2.000, -2.000, +2.000}},
       {{-2.000, -0.000, +0.000, -2.000, -2.000, -2.000}},
       {{-2.000, -0.000, -0.000, +0.000, -2.000, +0.000}},
       {{-2.000, -0.000, -0.000, -0.000, -2.000, -0.000}},
       {{-2.000, -0.000, -0.000, +2.000, -2.000, +2.000}},
       {{-2.000, -0.000, -0.000, -2.000, -2.000, -2.000}},
       {{-2.000, -0.000, +2.000, +0.000, +0.000, +0.000}},
       {{-2.000, -0.000, +2.000, -0.000, +0.000, -0.000}},
       {{-2.000, -0.000, +2.000, +2.000, +0.000, +2.000}},
       {{-2.000, -0.000, +2.000, -2.000, +0.000, -2.000}},
       {{-2.000, -0.000, -2.000, +0.000, -4.000, +0.000}},
       {{-2.000, -0.000, -2.000, -0.000, -4.000, -0.000}},
       {{-2.000, -0.000, -2.000, +2.000, -4.000, +2.000}},
       {{-2.000, -0.000, -2.000, -2.000, -4.000, -2.000}},
       {{-2.000, +2.000, +0.000, +0.000, -2.000, +2.000}},
       {{-2.000, +2.000, +0.000, -0.000, -2.000, +2.000}},
       {{-2.000, +2.000, +0.000, +2.000, -2.000, +4.000}},
       {{-2.000, +2.000, +0.000, -2.000, -2.000, +0.000}},
       {{-2.000, +2.000, -0.000, +0.000, -2.000, +2.000}},
       {{-2.000, +2.000, -0.000, -0.000, -2.000, +2.000}},
       {{-2.000, +2.000, -0.000, +2.000, -2.000, +4.000}},
       {{-2.000, +2.000, -0.000, -2.000, -2.000, +0.000}},
       {{-2.000, +2.000, +2.000, +0.000, +0.000, +2.000}},
       {{-2.000, +2.000, +2.000, -0.000, +0.000, +2.000}},
       {{-2.000, +2.000, +2.000, +2.000, +0.000, +4.000}},
       {{-2.000, +2.000, +2.000, -2.000, +0.000, +0.000}},
       {{-2.000, +2.000, -2.000, +0.000, -4.000, +2.000}},
       {{-2.000, +2.000, -2.000, -0.000, -4.000, +2.000}},
       {{-2.000, +2.000, -2.000, +2.000, -4.000, +4.000}},
       {{-2.000, +2.000, -2.000, -2.000, -4.000, +0.000}},
       {{-2.000, -2.000, +0.000, +0.000, -2.000, -2.000}},
       {{-2.000, -2.000, +0.000, -0.000, -2.000, -2.000}},
       {{-2.000, -2.000, +0.000, +2.000, -2.000, +0.000}},
       {{-2.000, -2.000, +0.000, -2.000, -2.000, -4.000}},
       {{-2.000, -2.000, -0.000, +0.000, -2.000, -2.000}},
       {{-2.000, -2.000, -0.000, -0.000, -2.000, -2.000}},
       {{-2.000, -2.000, -0.000, +2.000, -2.000, +0.000}},
       {{-2.000, -2.000, -0.000, -2.000, -2.000, -4.000}},
       {{-2.000, -2.000, +2.000, +0.000, +0.000, -2.000}},
       {{-2.000, -2.000, +2.000, -0.000, +0.000, -2.000}},
       {{-2.000, -2.000, +2.000, +2.000, +0.000, +0.000}},
       {{-2.000, -2.000, +2.000, -2.000, +0.000, -4.000}},
       {{-2.000, -2.000, -2.000, +0.000, -4.000, -2.000}},
       {{-2.000, -2.000, -2.000, -0.000, -4.000, -2.000}},
       {{-2.000, -2.000, -2.000, +2.000, -4.000, +0.000}},
       {{-2.000, -2.000, -2.000, -2.000, -4.000, -4.000}},
    }};
   for (unsigned i = 0; i < add_data.size(); ++i)
   {
      if (!(isnan)(add_data[i][4]) && !(isnan)(add_data[i][5]))
      {
         result = T(add_data[i][0], add_data[i][1]) + T(add_data[i][2], add_data[i][3]);
         BOOST_TEST(result.real() == add_data[i][4]);
         BOOST_TEST(result.imag() == add_data[i][5]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(add_data[i][4]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(add_data[i][5]));
      }
   }

   //
   // Subtract the permutations:
   //
   static constexpr std::array<std::array<double, 6>, 256> sub_data =
   { {
       {{+0.000, +0.000, +0.000, +0.000, +0.000, +0.000}},
       {{+0.000, +0.000, +0.000, -0.000, +0.000, +0.000}},
       {{+0.000, +0.000, +0.000, +2.000, +0.000, -2.000}},
       {{+0.000, +0.000, +0.000, -2.000, +0.000, +2.000}},
       {{+0.000, +0.000, -0.000, +0.000, +0.000, +0.000}},
       {{+0.000, +0.000, -0.000, -0.000, +0.000, +0.000}},
       {{+0.000, +0.000, -0.000, +2.000, +0.000, -2.000}},
       {{+0.000, +0.000, -0.000, -2.000, +0.000, +2.000}},
       {{+0.000, +0.000, +2.000, +0.000, -2.000, +0.000}},
       {{+0.000, +0.000, +2.000, -0.000, -2.000, +0.000}},
       {{+0.000, +0.000, +2.000, +2.000, -2.000, -2.000}},
       {{+0.000, +0.000, +2.000, -2.000, -2.000, +2.000}},
       {{+0.000, +0.000, -2.000, +0.000, +2.000, +0.000}},
       {{+0.000, +0.000, -2.000, -0.000, +2.000, +0.000}},
       {{+0.000, +0.000, -2.000, +2.000, +2.000, -2.000}},
       {{+0.000, +0.000, -2.000, -2.000, +2.000, +2.000}},
       {{+0.000, -0.000, +0.000, +0.000, +0.000, -0.000}},
       {{+0.000, -0.000, +0.000, -0.000, +0.000, +0.000}},
       {{+0.000, -0.000, +0.000, +2.000, +0.000, -2.000}},
       {{+0.000, -0.000, +0.000, -2.000, +0.000, +2.000}},
       {{+0.000, -0.000, -0.000, +0.000, +0.000, -0.000}},
       {{+0.000, -0.000, -0.000, -0.000, +0.000, +0.000}},
       {{+0.000, -0.000, -0.000, +2.000, +0.000, -2.000}},
       {{+0.000, -0.000, -0.000, -2.000, +0.000, +2.000}},
       {{+0.000, -0.000, +2.000, +0.000, -2.000, -0.000}},
       {{+0.000, -0.000, +2.000, -0.000, -2.000, +0.000}},
       {{+0.000, -0.000, +2.000, +2.000, -2.000, -2.000}},
       {{+0.000, -0.000, +2.000, -2.000, -2.000, +2.000}},
       {{+0.000, -0.000, -2.000, +0.000, +2.000, -0.000}},
       {{+0.000, -0.000, -2.000, -0.000, +2.000, +0.000}},
       {{+0.000, -0.000, -2.000, +2.000, +2.000, -2.000}},
       {{+0.000, -0.000, -2.000, -2.000, +2.000, +2.000}},
       {{+0.000, +2.000, +0.000, +0.000, +0.000, +2.000}},
       {{+0.000, +2.000, +0.000, -0.000, +0.000, +2.000}},
       {{+0.000, +2.000, +0.000, +2.000, +0.000, +0.000}},
       {{+0.000, +2.000, +0.000, -2.000, +0.000, +4.000}},
       {{+0.000, +2.000, -0.000, +0.000, +0.000, +2.000}},
       {{+0.000, +2.000, -0.000, -0.000, +0.000, +2.000}},
       {{+0.000, +2.000, -0.000, +2.000, +0.000, +0.000}},
       {{+0.000, +2.000, -0.000, -2.000, +0.000, +4.000}},
       {{+0.000, +2.000, +2.000, +0.000, -2.000, +2.000}},
       {{+0.000, +2.000, +2.000, -0.000, -2.000, +2.000}},
       {{+0.000, +2.000, +2.000, +2.000, -2.000, +0.000}},
       {{+0.000, +2.000, +2.000, -2.000, -2.000, +4.000}},
       {{+0.000, +2.000, -2.000, +0.000, +2.000, +2.000}},
       {{+0.000, +2.000, -2.000, -0.000, +2.000, +2.000}},
       {{+0.000, +2.000, -2.000, +2.000, +2.000, +0.000}},
       {{+0.000, +2.000, -2.000, -2.000, +2.000, +4.000}},
       {{+0.000, -2.000, +0.000, +0.000, +0.000, -2.000}},
       {{+0.000, -2.000, +0.000, -0.000, +0.000, -2.000}},
       {{+0.000, -2.000, +0.000, +2.000, +0.000, -4.000}},
       {{+0.000, -2.000, +0.000, -2.000, +0.000, +0.000}},
       {{+0.000, -2.000, -0.000, +0.000, +0.000, -2.000}},
       {{+0.000, -2.000, -0.000, -0.000, +0.000, -2.000}},
       {{+0.000, -2.000, -0.000, +2.000, +0.000, -4.000}},
       {{+0.000, -2.000, -0.000, -2.000, +0.000, +0.000}},
       {{+0.000, -2.000, +2.000, +0.000, -2.000, -2.000}},
       {{+0.000, -2.000, +2.000, -0.000, -2.000, -2.000}},
       {{+0.000, -2.000, +2.000, +2.000, -2.000, -4.000}},
       {{+0.000, -2.000, +2.000, -2.000, -2.000, +0.000}},
       {{+0.000, -2.000, -2.000, +0.000, +2.000, -2.000}},
       {{+0.000, -2.000, -2.000, -0.000, +2.000, -2.000}},
       {{+0.000, -2.000, -2.000, +2.000, +2.000, -4.000}},
       {{+0.000, -2.000, -2.000, -2.000, +2.000, +0.000}},
       {{-0.000, +0.000, +0.000, +0.000, -0.000, +0.000}},
       {{-0.000, +0.000, +0.000, -0.000, -0.000, +0.000}},
       {{-0.000, +0.000, +0.000, +2.000, -0.000, -2.000}},
       {{-0.000, +0.000, +0.000, -2.000, -0.000, +2.000}},
       {{-0.000, +0.000, -0.000, +0.000, +0.000, +0.000}},
       {{-0.000, +0.000, -0.000, -0.000, +0.000, +0.000}},
       {{-0.000, +0.000, -0.000, +2.000, +0.000, -2.000}},
       {{-0.000, +0.000, -0.000, -2.000, +0.000, +2.000}},
       {{-0.000, +0.000, +2.000, +0.000, -2.000, +0.000}},
       {{-0.000, +0.000, +2.000, -0.000, -2.000, +0.000}},
       {{-0.000, +0.000, +2.000, +2.000, -2.000, -2.000}},
       {{-0.000, +0.000, +2.000, -2.000, -2.000, +2.000}},
       {{-0.000, +0.000, -2.000, +0.000, +2.000, +0.000}},
       {{-0.000, +0.000, -2.000, -0.000, +2.000, +0.000}},
       {{-0.000, +0.000, -2.000, +2.000, +2.000, -2.000}},
       {{-0.000, +0.000, -2.000, -2.000, +2.000, +2.000}},
       {{-0.000, -0.000, +0.000, +0.000, -0.000, -0.000}},
       {{-0.000, -0.000, +0.000, -0.000, -0.000, +0.000}},
       {{-0.000, -0.000, +0.000, +2.000, -0.000, -2.000}},
       {{-0.000, -0.000, +0.000, -2.000, -0.000, +2.000}},
       {{-0.000, -0.000, -0.000, +0.000, +0.000, -0.000}},
       {{-0.000, -0.000, -0.000, -0.000, +0.000, +0.000}},
       {{-0.000, -0.000, -0.000, +2.000, +0.000, -2.000}},
       {{-0.000, -0.000, -0.000, -2.000, +0.000, +2.000}},
       {{-0.000, -0.000, +2.000, +0.000, -2.000, -0.000}},
       {{-0.000, -0.000, +2.000, -0.000, -2.000, +0.000}},
       {{-0.000, -0.000, +2.000, +2.000, -2.000, -2.000}},
       {{-0.000, -0.000, +2.000, -2.000, -2.000, +2.000}},
       {{-0.000, -0.000, -2.000, +0.000, +2.000, -0.000}},
       {{-0.000, -0.000, -2.000, -0.000, +2.000, +0.000}},
       {{-0.000, -0.000, -2.000, +2.000, +2.000, -2.000}},
       {{-0.000, -0.000, -2.000, -2.000, +2.000, +2.000}},
       {{-0.000, +2.000, +0.000, +0.000, -0.000, +2.000}},
       {{-0.000, +2.000, +0.000, -0.000, -0.000, +2.000}},
       {{-0.000, +2.000, +0.000, +2.000, -0.000, +0.000}},
       {{-0.000, +2.000, +0.000, -2.000, -0.000, +4.000}},
       {{-0.000, +2.000, -0.000, +0.000, +0.000, +2.000}},
       {{-0.000, +2.000, -0.000, -0.000, +0.000, +2.000}},
       {{-0.000, +2.000, -0.000, +2.000, +0.000, +0.000}},
       {{-0.000, +2.000, -0.000, -2.000, +0.000, +4.000}},
       {{-0.000, +2.000, +2.000, +0.000, -2.000, +2.000}},
       {{-0.000, +2.000, +2.000, -0.000, -2.000, +2.000}},
       {{-0.000, +2.000, +2.000, +2.000, -2.000, +0.000}},
       {{-0.000, +2.000, +2.000, -2.000, -2.000, +4.000}},
       {{-0.000, +2.000, -2.000, +0.000, +2.000, +2.000}},
       {{-0.000, +2.000, -2.000, -0.000, +2.000, +2.000}},
       {{-0.000, +2.000, -2.000, +2.000, +2.000, +0.000}},
       {{-0.000, +2.000, -2.000, -2.000, +2.000, +4.000}},
       {{-0.000, -2.000, +0.000, +0.000, -0.000, -2.000}},
       {{-0.000, -2.000, +0.000, -0.000, -0.000, -2.000}},
       {{-0.000, -2.000, +0.000, +2.000, -0.000, -4.000}},
       {{-0.000, -2.000, +0.000, -2.000, -0.000, +0.000}},
       {{-0.000, -2.000, -0.000, +0.000, +0.000, -2.000}},
       {{-0.000, -2.000, -0.000, -0.000, +0.000, -2.000}},
       {{-0.000, -2.000, -0.000, +2.000, +0.000, -4.000}},
       {{-0.000, -2.000, -0.000, -2.000, +0.000, +0.000}},
       {{-0.000, -2.000, +2.000, +0.000, -2.000, -2.000}},
       {{-0.000, -2.000, +2.000, -0.000, -2.000, -2.000}},
       {{-0.000, -2.000, +2.000, +2.000, -2.000, -4.000}},
       {{-0.000, -2.000, +2.000, -2.000, -2.000, +0.000}},
       {{-0.000, -2.000, -2.000, +0.000, +2.000, -2.000}},
       {{-0.000, -2.000, -2.000, -0.000, +2.000, -2.000}},
       {{-0.000, -2.000, -2.000, +2.000, +2.000, -4.000}},
       {{-0.000, -2.000, -2.000, -2.000, +2.000, +0.000}},
       {{+2.000, +0.000, +0.000, +0.000, +2.000, +0.000}},
       {{+2.000, +0.000, +0.000, -0.000, +2.000, +0.000}},
       {{+2.000, +0.000, +0.000, +2.000, +2.000, -2.000}},
       {{+2.000, +0.000, +0.000, -2.000, +2.000, +2.000}},
       {{+2.000, +0.000, -0.000, +0.000, +2.000, +0.000}},
       {{+2.000, +0.000, -0.000, -0.000, +2.000, +0.000}},
       {{+2.000, +0.000, -0.000, +2.000, +2.000, -2.000}},
       {{+2.000, +0.000, -0.000, -2.000, +2.000, +2.000}},
       {{+2.000, +0.000, +2.000, +0.000, +0.000, +0.000}},
       {{+2.000, +0.000, +2.000, -0.000, +0.000, +0.000}},
       {{+2.000, +0.000, +2.000, +2.000, +0.000, -2.000}},
       {{+2.000, +0.000, +2.000, -2.000, +0.000, +2.000}},
       {{+2.000, +0.000, -2.000, +0.000, +4.000, +0.000}},
       {{+2.000, +0.000, -2.000, -0.000, +4.000, +0.000}},
       {{+2.000, +0.000, -2.000, +2.000, +4.000, -2.000}},
       {{+2.000, +0.000, -2.000, -2.000, +4.000, +2.000}},
       {{+2.000, -0.000, +0.000, +0.000, +2.000, -0.000}},
       {{+2.000, -0.000, +0.000, -0.000, +2.000, +0.000}},
       {{+2.000, -0.000, +0.000, +2.000, +2.000, -2.000}},
       {{+2.000, -0.000, +0.000, -2.000, +2.000, +2.000}},
       {{+2.000, -0.000, -0.000, +0.000, +2.000, -0.000}},
       {{+2.000, -0.000, -0.000, -0.000, +2.000, +0.000}},
       {{+2.000, -0.000, -0.000, +2.000, +2.000, -2.000}},
       {{+2.000, -0.000, -0.000, -2.000, +2.000, +2.000}},
       {{+2.000, -0.000, +2.000, +0.000, +0.000, -0.000}},
       {{+2.000, -0.000, +2.000, -0.000, +0.000, +0.000}},
       {{+2.000, -0.000, +2.000, +2.000, +0.000, -2.000}},
       {{+2.000, -0.000, +2.000, -2.000, +0.000, +2.000}},
       {{+2.000, -0.000, -2.000, +0.000, +4.000, -0.000}},
       {{+2.000, -0.000, -2.000, -0.000, +4.000, +0.000}},
       {{+2.000, -0.000, -2.000, +2.000, +4.000, -2.000}},
       {{+2.000, -0.000, -2.000, -2.000, +4.000, +2.000}},
       {{+2.000, +2.000, +0.000, +0.000, +2.000, +2.000}},
       {{+2.000, +2.000, +0.000, -0.000, +2.000, +2.000}},
       {{+2.000, +2.000, +0.000, +2.000, +2.000, +0.000}},
       {{+2.000, +2.000, +0.000, -2.000, +2.000, +4.000}},
       {{+2.000, +2.000, -0.000, +0.000, +2.000, +2.000}},
       {{+2.000, +2.000, -0.000, -0.000, +2.000, +2.000}},
       {{+2.000, +2.000, -0.000, +2.000, +2.000, +0.000}},
       {{+2.000, +2.000, -0.000, -2.000, +2.000, +4.000}},
       {{+2.000, +2.000, +2.000, +0.000, +0.000, +2.000}},
       {{+2.000, +2.000, +2.000, -0.000, +0.000, +2.000}},
       {{+2.000, +2.000, +2.000, +2.000, +0.000, +0.000}},
       {{+2.000, +2.000, +2.000, -2.000, +0.000, +4.000}},
       {{+2.000, +2.000, -2.000, +0.000, +4.000, +2.000}},
       {{+2.000, +2.000, -2.000, -0.000, +4.000, +2.000}},
       {{+2.000, +2.000, -2.000, +2.000, +4.000, +0.000}},
       {{+2.000, +2.000, -2.000, -2.000, +4.000, +4.000}},
       {{+2.000, -2.000, +0.000, +0.000, +2.000, -2.000}},
       {{+2.000, -2.000, +0.000, -0.000, +2.000, -2.000}},
       {{+2.000, -2.000, +0.000, +2.000, +2.000, -4.000}},
       {{+2.000, -2.000, +0.000, -2.000, +2.000, +0.000}},
       {{+2.000, -2.000, -0.000, +0.000, +2.000, -2.000}},
       {{+2.000, -2.000, -0.000, -0.000, +2.000, -2.000}},
       {{+2.000, -2.000, -0.000, +2.000, +2.000, -4.000}},
       {{+2.000, -2.000, -0.000, -2.000, +2.000, +0.000}},
       {{+2.000, -2.000, +2.000, +0.000, +0.000, -2.000}},
       {{+2.000, -2.000, +2.000, -0.000, +0.000, -2.000}},
       {{+2.000, -2.000, +2.000, +2.000, +0.000, -4.000}},
       {{+2.000, -2.000, +2.000, -2.000, +0.000, +0.000}},
       {{+2.000, -2.000, -2.000, +0.000, +4.000, -2.000}},
       {{+2.000, -2.000, -2.000, -0.000, +4.000, -2.000}},
       {{+2.000, -2.000, -2.000, +2.000, +4.000, -4.000}},
       {{+2.000, -2.000, -2.000, -2.000, +4.000, +0.000}},
       {{-2.000, +0.000, +0.000, +0.000, -2.000, +0.000}},
       {{-2.000, +0.000, +0.000, -0.000, -2.000, +0.000}},
       {{-2.000, +0.000, +0.000, +2.000, -2.000, -2.000}},
       {{-2.000, +0.000, +0.000, -2.000, -2.000, +2.000}},
       {{-2.000, +0.000, -0.000, +0.000, -2.000, +0.000}},
       {{-2.000, +0.000, -0.000, -0.000, -2.000, +0.000}},
       {{-2.000, +0.000, -0.000, +2.000, -2.000, -2.000}},
       {{-2.000, +0.000, -0.000, -2.000, -2.000, +2.000}},
       {{-2.000, +0.000, +2.000, +0.000, -4.000, +0.000}},
       {{-2.000, +0.000, +2.000, -0.000, -4.000, +0.000}},
       {{-2.000, +0.000, +2.000, +2.000, -4.000, -2.000}},
       {{-2.000, +0.000, +2.000, -2.000, -4.000, +2.000}},
       {{-2.000, +0.000, -2.000, +0.000, +0.000, +0.000}},
       {{-2.000, +0.000, -2.000, -0.000, +0.000, +0.000}},
       {{-2.000, +0.000, -2.000, +2.000, +0.000, -2.000}},
       {{-2.000, +0.000, -2.000, -2.000, +0.000, +2.000}},
       {{-2.000, -0.000, +0.000, +0.000, -2.000, -0.000}},
       {{-2.000, -0.000, +0.000, -0.000, -2.000, +0.000}},
       {{-2.000, -0.000, +0.000, +2.000, -2.000, -2.000}},
       {{-2.000, -0.000, +0.000, -2.000, -2.000, +2.000}},
       {{-2.000, -0.000, -0.000, +0.000, -2.000, -0.000}},
       {{-2.000, -0.000, -0.000, -0.000, -2.000, +0.000}},
       {{-2.000, -0.000, -0.000, +2.000, -2.000, -2.000}},
       {{-2.000, -0.000, -0.000, -2.000, -2.000, +2.000}},
       {{-2.000, -0.000, +2.000, +0.000, -4.000, -0.000}},
       {{-2.000, -0.000, +2.000, -0.000, -4.000, +0.000}},
       {{-2.000, -0.000, +2.000, +2.000, -4.000, -2.000}},
       {{-2.000, -0.000, +2.000, -2.000, -4.000, +2.000}},
       {{-2.000, -0.000, -2.000, +0.000, +0.000, -0.000}},
       {{-2.000, -0.000, -2.000, -0.000, +0.000, +0.000}},
       {{-2.000, -0.000, -2.000, +2.000, +0.000, -2.000}},
       {{-2.000, -0.000, -2.000, -2.000, +0.000, +2.000}},
       {{-2.000, +2.000, +0.000, +0.000, -2.000, +2.000}},
       {{-2.000, +2.000, +0.000, -0.000, -2.000, +2.000}},
       {{-2.000, +2.000, +0.000, +2.000, -2.000, +0.000}},
       {{-2.000, +2.000, +0.000, -2.000, -2.000, +4.000}},
       {{-2.000, +2.000, -0.000, +0.000, -2.000, +2.000}},
       {{-2.000, +2.000, -0.000, -0.000, -2.000, +2.000}},
       {{-2.000, +2.000, -0.000, +2.000, -2.000, +0.000}},
       {{-2.000, +2.000, -0.000, -2.000, -2.000, +4.000}},
       {{-2.000, +2.000, +2.000, +0.000, -4.000, +2.000}},
       {{-2.000, +2.000, +2.000, -0.000, -4.000, +2.000}},
       {{-2.000, +2.000, +2.000, +2.000, -4.000, +0.000}},
       {{-2.000, +2.000, +2.000, -2.000, -4.000, +4.000}},
       {{-2.000, +2.000, -2.000, +0.000, +0.000, +2.000}},
       {{-2.000, +2.000, -2.000, -0.000, +0.000, +2.000}},
       {{-2.000, +2.000, -2.000, +2.000, +0.000, +0.000}},
       {{-2.000, +2.000, -2.000, -2.000, +0.000, +4.000}},
       {{-2.000, -2.000, +0.000, +0.000, -2.000, -2.000}},
       {{-2.000, -2.000, +0.000, -0.000, -2.000, -2.000}},
       {{-2.000, -2.000, +0.000, +2.000, -2.000, -4.000}},
       {{-2.000, -2.000, +0.000, -2.000, -2.000, +0.000}},
       {{-2.000, -2.000, -0.000, +0.000, -2.000, -2.000}},
       {{-2.000, -2.000, -0.000, -0.000, -2.000, -2.000}},
       {{-2.000, -2.000, -0.000, +2.000, -2.000, -4.000}},
       {{-2.000, -2.000, -0.000, -2.000, -2.000, +0.000}},
       {{-2.000, -2.000, +2.000, +0.000, -4.000, -2.000}},
       {{-2.000, -2.000, +2.000, -0.000, -4.000, -2.000}},
       {{-2.000, -2.000, +2.000, +2.000, -4.000, -4.000}},
       {{-2.000, -2.000, +2.000, -2.000, -4.000, +0.000}},
       {{-2.000, -2.000, -2.000, +0.000, +0.000, -2.000}},
       {{-2.000, -2.000, -2.000, -0.000, +0.000, -2.000}},
       {{-2.000, -2.000, -2.000, +2.000, +0.000, -4.000}},
       {{-2.000, -2.000, -2.000, -2.000, +0.000, +0.000}},
   }};
   for (unsigned i = 0; i < sub_data.size(); ++i)
   {
      if (!(isnan)(sub_data[i][4]) && !(isnan)(sub_data[i][5]))
      {
         result = T(sub_data[i][0], sub_data[i][1]) - T(sub_data[i][2], sub_data[i][3]);
         BOOST_TEST(result.real() == sub_data[i][4]);
         BOOST_TEST(result.imag() == sub_data[i][5]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(sub_data[i][4]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(sub_data[i][5]));
      }
   }

   //
   // Multiplication by scalar:
   //
   static constexpr std::array<std::array<double, 5>, 64> scalar_mul_data =
       {{
           {{+0.000, +0.000, +0.000, +0.000, +0.000}},
           {{+0.000, +0.000, -0.000, -0.000, -0.000}},
           {{+0.000, +0.000, +2.000, +0.000, +0.000}},
           {{+0.000, +0.000, -2.000, -0.000, -0.000}},
           {{+0.000, -0.000, +0.000, +0.000, -0.000}},
           {{+0.000, -0.000, -0.000, -0.000, +0.000}},
           {{+0.000, -0.000, +2.000, +0.000, -0.000}},
           {{+0.000, -0.000, -2.000, -0.000, +0.000}},
           {{+0.000, +2.000, +0.000, +0.000, +0.000}},
           {{+0.000, +2.000, -0.000, -0.000, -0.000}},
           {{+0.000, +2.000, +2.000, +0.000, +4.000}},
           {{+0.000, +2.000, -2.000, -0.000, -4.000}},
           {{+0.000, -2.000, +0.000, +0.000, -0.000}},
           {{+0.000, -2.000, -0.000, -0.000, +0.000}},
           {{+0.000, -2.000, +2.000, +0.000, -4.000}},
           {{+0.000, -2.000, -2.000, -0.000, +4.000}},
           {{-0.000, +0.000, +0.000, -0.000, +0.000}},
           {{-0.000, +0.000, -0.000, +0.000, -0.000}},
           {{-0.000, +0.000, +2.000, -0.000, +0.000}},
           {{-0.000, +0.000, -2.000, +0.000, -0.000}},
           {{-0.000, -0.000, +0.000, -0.000, -0.000}},
           {{-0.000, -0.000, -0.000, +0.000, +0.000}},
           {{-0.000, -0.000, +2.000, -0.000, -0.000}},
           {{-0.000, -0.000, -2.000, +0.000, +0.000}},
           {{-0.000, +2.000, +0.000, -0.000, +0.000}},
           {{-0.000, +2.000, -0.000, +0.000, -0.000}},
           {{-0.000, +2.000, +2.000, -0.000, +4.000}},
           {{-0.000, +2.000, -2.000, +0.000, -4.000}},
           {{-0.000, -2.000, +0.000, -0.000, -0.000}},
           {{-0.000, -2.000, -0.000, +0.000, +0.000}},
           {{-0.000, -2.000, +2.000, -0.000, -4.000}},
           {{-0.000, -2.000, -2.000, +0.000, +4.000}},
           {{+2.000, +0.000, +0.000, +0.000, +0.000}},
           {{+2.000, +0.000, -0.000, -0.000, -0.000}},
           {{+2.000, +0.000, +2.000, +4.000, +0.000}},
           {{+2.000, +0.000, -2.000, -4.000, -0.000}},
           {{+2.000, -0.000, +0.000, +0.000, -0.000}},
           {{+2.000, -0.000, -0.000, -0.000, +0.000}},
           {{+2.000, -0.000, +2.000, +4.000, -0.000}},
           {{+2.000, -0.000, -2.000, -4.000, +0.000}},
           {{+2.000, +2.000, +0.000, +0.000, +0.000}},
           {{+2.000, +2.000, -0.000, -0.000, -0.000}},
           {{+2.000, +2.000, +2.000, +4.000, +4.000}},
           {{+2.000, +2.000, -2.000, -4.000, -4.000}},
           {{+2.000, -2.000, +0.000, +0.000, -0.000}},
           {{+2.000, -2.000, -0.000, -0.000, +0.000}},
           {{+2.000, -2.000, +2.000, +4.000, -4.000}},
           {{+2.000, -2.000, -2.000, -4.000, +4.000}},
           {{-2.000, +0.000, +0.000, -0.000, +0.000}},
           {{-2.000, +0.000, -0.000, +0.000, -0.000}},
           {{-2.000, +0.000, +2.000, -4.000, +0.000}},
           {{-2.000, +0.000, -2.000, +4.000, -0.000}},
           {{-2.000, -0.000, +0.000, -0.000, -0.000}},
           {{-2.000, -0.000, -0.000, +0.000, +0.000}},
           {{-2.000, -0.000, +2.000, -4.000, -0.000}},
           {{-2.000, -0.000, -2.000, +4.000, +0.000}},
           {{-2.000, +2.000, +0.000, -0.000, +0.000}},
           {{-2.000, +2.000, -0.000, +0.000, -0.000}},
           {{-2.000, +2.000, +2.000, -4.000, +4.000}},
           {{-2.000, +2.000, -2.000, +4.000, -4.000}},
           {{-2.000, -2.000, +0.000, -0.000, -0.000}},
           {{-2.000, -2.000, -0.000, +0.000, +0.000}},
           {{-2.000, -2.000, +2.000, -4.000, -4.000}},
           {{-2.000, -2.000, -2.000, +4.000, +4.000}},
       }};
   for (unsigned i = 0; i < scalar_mul_data.size(); ++i)
   {
      if (!(isnan)(scalar_mul_data[i][3]) && !(isnan)(scalar_mul_data[i][4]))
      {
         result = T(scalar_mul_data[i][0], scalar_mul_data[i][1]) * value_type(scalar_mul_data[i][2]);
         BOOST_TEST(result.real() == scalar_mul_data[i][3]);
         BOOST_TEST(result.imag() == scalar_mul_data[i][4]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(scalar_mul_data[i][3]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(scalar_mul_data[i][4]));
      }
   }
   //
   // Division by scalar:
   //
   static constexpr std::array<std::array<double, 5>, 64> scalar_div_data =
       {{
           {{+0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{+0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{+0.000, +0.000, +2.000, +0.000, +0.000}},
           {{+0.000, +0.000, -2.000, -0.000, -0.000}},
           {{+0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{+0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{+0.000, -0.000, +2.000, +0.000, -0.000}},
           {{+0.000, -0.000, -2.000, -0.000, +0.000}},
           {{+0.000, +2.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity()}},
           {{+0.000, +2.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity()}},
           {{+0.000, +2.000, +2.000, +0.000, +1.000}},
           {{+0.000, +2.000, -2.000, -0.000, -1.000}},
           {{+0.000, -2.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity()}},
           {{+0.000, -2.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity()}},
           {{+0.000, -2.000, +2.000, +0.000, -1.000}},
           {{+0.000, -2.000, -2.000, -0.000, +1.000}},
           {{-0.000, +0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{-0.000, +0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{-0.000, +0.000, +2.000, -0.000, +0.000}},
           {{-0.000, +0.000, -2.000, +0.000, -0.000}},
           {{-0.000, -0.000, +0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{-0.000, -0.000, -0.000, std::numeric_limits<double>::quiet_NaN(), std::numeric_limits<double>::quiet_NaN()}},
           {{-0.000, -0.000, +2.000, -0.000, -0.000}},
           {{-0.000, -0.000, -2.000, +0.000, +0.000}},
           {{-0.000, +2.000, +0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity()}},
           {{-0.000, +2.000, -0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity()}},
           {{-0.000, +2.000, +2.000, -0.000, +1.000}},
           {{-0.000, +2.000, -2.000, +0.000, -1.000}},
           {{-0.000, -2.000, +0.000, std::numeric_limits<double>::quiet_NaN(), -std::numeric_limits<double>::infinity()}},
           {{-0.000, -2.000, -0.000, std::numeric_limits<double>::quiet_NaN(), +std::numeric_limits<double>::infinity()}},
           {{-0.000, -2.000, +2.000, -0.000, -1.000}},
           {{-0.000, -2.000, -2.000, +0.000, +1.000}},
           {{+2.000, +0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{+2.000, +0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{+2.000, +0.000, +2.000, +1.000, +0.000}},
           {{+2.000, +0.000, -2.000, -1.000, -0.000}},
           {{+2.000, -0.000, +0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{+2.000, -0.000, -0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{+2.000, -0.000, +2.000, +1.000, -0.000}},
           {{+2.000, -0.000, -2.000, -1.000, +0.000}},
           {{+2.000, +2.000, +0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity()}},
           {{+2.000, +2.000, -0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity()}},
           {{+2.000, +2.000, +2.000, +1.000, +1.000}},
           {{+2.000, +2.000, -2.000, -1.000, -1.000}},
           {{+2.000, -2.000, +0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity()}},
           {{+2.000, -2.000, -0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity()}},
           {{+2.000, -2.000, +2.000, +1.000, -1.000}},
           {{+2.000, -2.000, -2.000, -1.000, +1.000}},
           {{-2.000, +0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{-2.000, +0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{-2.000, +0.000, +2.000, -1.000, +0.000}},
           {{-2.000, +0.000, -2.000, +1.000, -0.000}},
           {{-2.000, -0.000, +0.000, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{-2.000, -0.000, -0.000, +std::numeric_limits<double>::infinity(), std::numeric_limits<double>::quiet_NaN()}},
           {{-2.000, -0.000, +2.000, -1.000, -0.000}},
           {{-2.000, -0.000, -2.000, +1.000, +0.000}},
           {{-2.000, +2.000, +0.000, -std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity()}},
           {{-2.000, +2.000, -0.000, +std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity()}},
           {{-2.000, +2.000, +2.000, -1.000, +1.000}},
           {{-2.000, +2.000, -2.000, +1.000, -1.000}},
           {{-2.000, -2.000, +0.000, -std::numeric_limits<double>::infinity(), -std::numeric_limits<double>::infinity()}},
           {{-2.000, -2.000, -0.000, +std::numeric_limits<double>::infinity(), +std::numeric_limits<double>::infinity()}},
           {{-2.000, -2.000, +2.000, -1.000, -1.000}},
           {{-2.000, -2.000, -2.000, +1.000, +1.000}},
       }};
   for (unsigned i = 0; i < scalar_div_data.size(); ++i)
   {
      if (!(isnan)(scalar_div_data[i][3]) && !(isnan)(scalar_div_data[i][4]))
      {
         result = T(scalar_div_data[i][0], scalar_div_data[i][1]) / value_type(scalar_div_data[i][2]);
         BOOST_TEST(result.real() == scalar_div_data[i][3]);
         BOOST_TEST(result.imag() == scalar_div_data[i][4]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(scalar_div_data[i][3]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(scalar_div_data[i][4]));
      }
   }


   //
   // Addition of scalar:
   //
   static constexpr std::array<std::array<double, 5>, 64> scalar_add_data =
   {{
           {{+0.000, +0.000, +0.000, +0.000, +0.000}},
           {{+0.000, +0.000, -0.000, +0.000, +0.000}},
           {{+0.000, +0.000, +2.000, +2.000, +0.000}},
           {{+0.000, +0.000, -2.000, -2.000, +0.000}},
           {{+0.000, -0.000, +0.000, +0.000, -0.000}},
           {{+0.000, -0.000, -0.000, +0.000, -0.000}},
           {{+0.000, -0.000, +2.000, +2.000, -0.000}},
           {{+0.000, -0.000, -2.000, -2.000, -0.000}},
           {{+0.000, +2.000, +0.000, +0.000, +2.000}},
           {{+0.000, +2.000, -0.000, +0.000, +2.000}},
           {{+0.000, +2.000, +2.000, +2.000, +2.000}},
           {{+0.000, +2.000, -2.000, -2.000, +2.000}},
           {{+0.000, -2.000, +0.000, +0.000, -2.000}},
           {{+0.000, -2.000, -0.000, +0.000, -2.000}},
           {{+0.000, -2.000, +2.000, +2.000, -2.000}},
           {{+0.000, -2.000, -2.000, -2.000, -2.000}},
           {{-0.000, +0.000, +0.000, +0.000, +0.000}},
           {{-0.000, +0.000, -0.000, -0.000, +0.000}},
           {{-0.000, +0.000, +2.000, +2.000, +0.000}},
           {{-0.000, +0.000, -2.000, -2.000, +0.000}},
           {{-0.000, -0.000, +0.000, +0.000, -0.000}},
           {{-0.000, -0.000, -0.000, -0.000, -0.000}},
           {{-0.000, -0.000, +2.000, +2.000, -0.000}},
           {{-0.000, -0.000, -2.000, -2.000, -0.000}},
           {{-0.000, +2.000, +0.000, +0.000, +2.000}},
           {{-0.000, +2.000, -0.000, -0.000, +2.000}},
           {{-0.000, +2.000, +2.000, +2.000, +2.000}},
           {{-0.000, +2.000, -2.000, -2.000, +2.000}},
           {{-0.000, -2.000, +0.000, +0.000, -2.000}},
           {{-0.000, -2.000, -0.000, -0.000, -2.000}},
           {{-0.000, -2.000, +2.000, +2.000, -2.000}},
           {{-0.000, -2.000, -2.000, -2.000, -2.000}},
           {{+2.000, +0.000, +0.000, +2.000, +0.000}},
           {{+2.000, +0.000, -0.000, +2.000, +0.000}},
           {{+2.000, +0.000, +2.000, +4.000, +0.000}},
           {{+2.000, +0.000, -2.000, +0.000, +0.000}},
           {{+2.000, -0.000, +0.000, +2.000, -0.000}},
           {{+2.000, -0.000, -0.000, +2.000, -0.000}},
           {{+2.000, -0.000, +2.000, +4.000, -0.000}},
           {{+2.000, -0.000, -2.000, +0.000, -0.000}},
           {{+2.000, +2.000, +0.000, +2.000, +2.000}},
           {{+2.000, +2.000, -0.000, +2.000, +2.000}},
           {{+2.000, +2.000, +2.000, +4.000, +2.000}},
           {{+2.000, +2.000, -2.000, +0.000, +2.000}},
           {{+2.000, -2.000, +0.000, +2.000, -2.000}},
           {{+2.000, -2.000, -0.000, +2.000, -2.000}},
           {{+2.000, -2.000, +2.000, +4.000, -2.000}},
           {{+2.000, -2.000, -2.000, +0.000, -2.000}},
           {{-2.000, +0.000, +0.000, -2.000, +0.000}},
           {{-2.000, +0.000, -0.000, -2.000, +0.000}},
           {{-2.000, +0.000, +2.000, +0.000, +0.000}},
           {{-2.000, +0.000, -2.000, -4.000, +0.000}},
           {{-2.000, -0.000, +0.000, -2.000, -0.000}},
           {{-2.000, -0.000, -0.000, -2.000, -0.000}},
           {{-2.000, -0.000, +2.000, +0.000, -0.000}},
           {{-2.000, -0.000, -2.000, -4.000, -0.000}},
           {{-2.000, +2.000, +0.000, -2.000, +2.000}},
           {{-2.000, +2.000, -0.000, -2.000, +2.000}},
           {{-2.000, +2.000, +2.000, +0.000, +2.000}},
           {{-2.000, +2.000, -2.000, -4.000, +2.000}},
           {{-2.000, -2.000, +0.000, -2.000, -2.000}},
           {{-2.000, -2.000, -0.000, -2.000, -2.000}},
           {{-2.000, -2.000, +2.000, +0.000, -2.000}},
           {{-2.000, -2.000, -2.000, -4.000, -2.000}},
       }};
   for (unsigned i = 0; i < scalar_add_data.size(); ++i)
   {
      if (!(isnan)(scalar_add_data[i][3]) && !(isnan)(scalar_add_data[i][4]))
      {
         result = T(scalar_add_data[i][0], scalar_add_data[i][1]) + value_type(scalar_add_data[i][2]);
         BOOST_TEST(result.real() == scalar_add_data[i][3]);
         BOOST_TEST(result.imag() == scalar_add_data[i][4]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(scalar_add_data[i][3]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(scalar_add_data[i][4]));
      }
   }
   //
   // Subtraction of scalar:
   //
   static constexpr std::array<std::array<double, 5>, 64> scalar_sub_data =
   {{
           {{+0.000, +0.000, +0.000, +0.000, +0.000}},
           {{+0.000, +0.000, -0.000, +0.000, +0.000}},
           {{+0.000, +0.000, +2.000, -2.000, +0.000}},
           {{+0.000, +0.000, -2.000, +2.000, +0.000}},
           {{+0.000, -0.000, +0.000, +0.000, -0.000}},
           {{+0.000, -0.000, -0.000, +0.000, -0.000}},
           {{+0.000, -0.000, +2.000, -2.000, -0.000}},
           {{+0.000, -0.000, -2.000, +2.000, -0.000}},
           {{+0.000, +2.000, +0.000, +0.000, +2.000}},
           {{+0.000, +2.000, -0.000, +0.000, +2.000}},
           {{+0.000, +2.000, +2.000, -2.000, +2.000}},
           {{+0.000, +2.000, -2.000, +2.000, +2.000}},
           {{+0.000, -2.000, +0.000, +0.000, -2.000}},
           {{+0.000, -2.000, -0.000, +0.000, -2.000}},
           {{+0.000, -2.000, +2.000, -2.000, -2.000}},
           {{+0.000, -2.000, -2.000, +2.000, -2.000}},
           {{-0.000, +0.000, +0.000, -0.000, +0.000}},
           {{-0.000, +0.000, -0.000, +0.000, +0.000}},
           {{-0.000, +0.000, +2.000, -2.000, +0.000}},
           {{-0.000, +0.000, -2.000, +2.000, +0.000}},
           {{-0.000, -0.000, +0.000, -0.000, -0.000}},
           {{-0.000, -0.000, -0.000, +0.000, -0.000}},
           {{-0.000, -0.000, +2.000, -2.000, -0.000}},
           {{-0.000, -0.000, -2.000, +2.000, -0.000}},
           {{-0.000, +2.000, +0.000, -0.000, +2.000}},
           {{-0.000, +2.000, -0.000, +0.000, +2.000}},
           {{-0.000, +2.000, +2.000, -2.000, +2.000}},
           {{-0.000, +2.000, -2.000, +2.000, +2.000}},
           {{-0.000, -2.000, +0.000, -0.000, -2.000}},
           {{-0.000, -2.000, -0.000, +0.000, -2.000}},
           {{-0.000, -2.000, +2.000, -2.000, -2.000}},
           {{-0.000, -2.000, -2.000, +2.000, -2.000}},
           {{+2.000, +0.000, +0.000, +2.000, +0.000}},
           {{+2.000, +0.000, -0.000, +2.000, +0.000}},
           {{+2.000, +0.000, +2.000, +0.000, +0.000}},
           {{+2.000, +0.000, -2.000, +4.000, +0.000}},
           {{+2.000, -0.000, +0.000, +2.000, -0.000}},
           {{+2.000, -0.000, -0.000, +2.000, -0.000}},
           {{+2.000, -0.000, +2.000, +0.000, -0.000}},
           {{+2.000, -0.000, -2.000, +4.000, -0.000}},
           {{+2.000, +2.000, +0.000, +2.000, +2.000}},
           {{+2.000, +2.000, -0.000, +2.000, +2.000}},
           {{+2.000, +2.000, +2.000, +0.000, +2.000}},
           {{+2.000, +2.000, -2.000, +4.000, +2.000}},
           {{+2.000, -2.000, +0.000, +2.000, -2.000}},
           {{+2.000, -2.000, -0.000, +2.000, -2.000}},
           {{+2.000, -2.000, +2.000, +0.000, -2.000}},
           {{+2.000, -2.000, -2.000, +4.000, -2.000}},
           {{-2.000, +0.000, +0.000, -2.000, +0.000}},
           {{-2.000, +0.000, -0.000, -2.000, +0.000}},
           {{-2.000, +0.000, +2.000, -4.000, +0.000}},
           {{-2.000, +0.000, -2.000, +0.000, +0.000}},
           {{-2.000, -0.000, +0.000, -2.000, -0.000}},
           {{-2.000, -0.000, -0.000, -2.000, -0.000}},
           {{-2.000, -0.000, +2.000, -4.000, -0.000}},
           {{-2.000, -0.000, -2.000, +0.000, -0.000}},
           {{-2.000, +2.000, +0.000, -2.000, +2.000}},
           {{-2.000, +2.000, -0.000, -2.000, +2.000}},
           {{-2.000, +2.000, +2.000, -4.000, +2.000}},
           {{-2.000, +2.000, -2.000, +0.000, +2.000}},
           {{-2.000, -2.000, +0.000, -2.000, -2.000}},
           {{-2.000, -2.000, -0.000, -2.000, -2.000}},
           {{-2.000, -2.000, +2.000, -4.000, -2.000}},
           {{-2.000, -2.000, -2.000, +0.000, -2.000}},
       }};
   for (unsigned i = 0; i < scalar_sub_data.size(); ++i)
   {
      if (!(isnan)(scalar_sub_data[i][3]) && !(isnan)(scalar_sub_data[i][4]))
      {
         result = T(scalar_sub_data[i][0], scalar_sub_data[i][1]) - value_type(scalar_sub_data[i][2]);
         BOOST_TEST(result.real() == scalar_sub_data[i][3]);
         BOOST_TEST(result.imag() == scalar_sub_data[i][4]);
         BOOST_TEST((bool)signbit(result.real()) == signbit(scalar_sub_data[i][3]));
         BOOST_TEST((bool)signbit(result.imag()) == signbit(scalar_sub_data[i][4]));
      }
   }
}

int main()
{
#ifdef TEST_DOUBLE
   //
   // This does not actually pass these tests:
   // 
   //test<std::complex<double>>();
#endif
#ifdef TEST_CPP_BIN_FLOAT
   test<boost::multiprecision::cpp_complex_50>();
   test<boost::multiprecision::number<boost::multiprecision::cpp_complex_50::backend_type, boost::multiprecision::et_on> >();
#endif
#ifdef TEST_MPC
   test<boost::multiprecision::mpc_complex_50>();
#endif
   return boost::report_errors();
}
