#!/usr/bin/env python3
"""Apply a gschema/dconf default via the edubuntu-installer-helper.

Usage:
    apply-default.py <label>                    # global default
    apply-default.py --user <username> <label>  # per-user default

This thin wrapper holds the override content so the JS frontend does
not need to embed large multi-line config strings.
"""

import subprocess
import sys
import textwrap

HELPER = "/usr/sbin/edubuntu-installer-helper"

SCHEMA_DIR = "/usr/share/glib-2.0/schemas"

GSCHEMA_TARGETS = {
    "Preschool": SCHEMA_DIR + "/96-edubuntu-preschool.gschema.override",
    "Primary/Elementary": SCHEMA_DIR + "/96-edubuntu-primary.gschema.override",
    "Secondary/Middle/High School": SCHEMA_DIR + "/96-edubuntu-secondary.gschema.override",
}

GSCHEMA_CONTENTS = {
    "Preschool": textwrap.dedent("""\
        [org.gnome.desktop.background:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-preschool.png'
        picture-uri-dark = 'file:///usr/share/backgrounds/edubuntu_default-preschool.png'

        [org.gnome.desktop.screensaver:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-preschool.png'

        [org.gnome.shell:ubuntu]
        favorite-apps = [ 'firefox_firefox.desktop', 'org.kde.gcompris.desktop', 'tuxpaint.desktop', 'org.kde.klettres.desktop', 'org.gnome.Calculator.desktop', 'yelp.desktop' ]
    """),
    "Primary/Elementary": textwrap.dedent("""\
        [org.gnome.desktop.background:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-primary.png'
        picture-uri-dark = 'file:///usr/share/backgrounds/edubuntu_default-primary.png'

        [org.gnome.desktop.screensaver:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-primary.png'

        [org.gnome.shell:ubuntu]
        favorite-apps = [ 'firefox_firefox.desktop', 'org.gnome.Calendar.desktop', 'org.gnome.Nautilus.desktop', 'tuxmath.desktop', 'org.kde.ktouch.desktop', 'org.gnome.Calculator.desktop', 'artha.desktop', 'libreoffice-writer.desktop', 'yelp.desktop' ]
    """),
    "Secondary/Middle/High School": textwrap.dedent("""\
        [org.gnome.desktop.background:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-secondary.png'
        picture-uri-dark = 'file:///usr/share/backgrounds/edubuntu_default-secondary.png'

        [org.gnome.desktop.screensaver:ubuntu]
        picture-uri = 'file:///usr/share/backgrounds/edubuntu_default-secondary.png'

        [org.gnome.shell:ubuntu]
        favorite-apps = [ 'firefox_firefox.desktop', 'thunderbird_thunderbird.desktop', 'org.gnome.Calendar.desktop', 'org.gnome.Nautilus.desktop', 'org.gnome.Calculator.desktop', 'org.gnome.Gnote.desktop', 'libreoffice-writer.desktop', 'yelp.desktop' ]
    """),
}

DCONF_CONTENTS = {
    "Preschool": textwrap.dedent("""\
[org/gnome/desktop/background]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-preschool.png'
picture-uri-dark='file:///usr/share/backgrounds/edubuntu_default-preschool.png'

[org/gnome/desktop/screensaver]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-preschool.png'

[org/gnome/shell]
favorite-apps=['firefox_firefox.desktop', 'org.kde.gcompris.desktop', 'tuxpaint.desktop', 'org.kde.klettres.desktop', 'org.gnome.Calculator.desktop', 'yelp.desktop']
"""),
    "Primary/Elementary": textwrap.dedent("""\
[org/gnome/desktop/background]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-primary.png'
picture-uri-dark='file:///usr/share/backgrounds/edubuntu_default-primary.png'

[org/gnome/desktop/screensaver]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-primary.png'

[org/gnome/shell]
favorite-apps=['firefox_firefox.desktop', 'org.gnome.Calendar.desktop', 'org.gnome.Nautilus.desktop', 'tuxmath.desktop', 'org.kde.ktouch.desktop', 'org.gnome.Calculator.desktop', 'artha.desktop', 'libreoffice-writer.desktop', 'yelp.desktop']
"""),
    "Secondary/Middle/High School": textwrap.dedent("""\
[org/gnome/desktop/background]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-secondary.png'
picture-uri-dark='file:///usr/share/backgrounds/edubuntu_default-secondary.png'

[org/gnome/desktop/screensaver]
picture-uri='file:///usr/share/backgrounds/edubuntu_default-secondary.png'

[org/gnome/shell]
favorite-apps=['firefox_firefox.desktop', 'thunderbird_thunderbird.desktop', 'org.gnome.Calendar.desktop', 'org.gnome.Nautilus.desktop', 'org.gnome.Calculator.desktop', 'org.gnome.Gnote.desktop', 'libreoffice-writer.desktop', 'yelp.desktop']
"""),
}

TERTIARY_LABEL = "Tertiary/College/University (Default)"


def main():
    args = sys.argv[1:]
    user = None
    if args and args[0] == "--user":
        user = args[1]
        label = args[2]
    else:
        label = args[0]

    if user:
        if label == TERTIARY_LABEL:
            rc = subprocess.run([HELPER, "usertertdefault", user]).returncode
        elif label in DCONF_CONTENTS:
            rc = subprocess.run(
                [HELPER, "userdefault", user, DCONF_CONTENTS[label], label],
            ).returncode
        else:
            print("Unknown label: " + label, file=sys.stderr)
            sys.exit(1)
    else:
        if label == TERTIARY_LABEL:
            rc = subprocess.run([HELPER, "tertdefault"]).returncode
        elif label in GSCHEMA_CONTENTS:
            target = GSCHEMA_TARGETS[label]
            rc = subprocess.run(
                [HELPER, "newdefault", target, GSCHEMA_CONTENTS[label]],
            ).returncode
        else:
            print("Unknown label: " + label, file=sys.stderr)
            sys.exit(1)

    sys.exit(rc)


if __name__ == "__main__":
    main()
