/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

const { Visitor, walk } = require("resource://devtools/shared/heapsnapshot/CensusUtils.js");

const DEFAULT_MAX_DEPTH = 4;
const DEFAULT_MAX_SIBLINGS = 15;

/**
 * A single node in a dominator tree.
 *
 * @param {NodeId} nodeId
 * @param {NodeSize} retainedSize
 */
function DominatorTreeNode(nodeId, label, shallowSize, retainedSize) {
  // The id of this node.
  this.nodeId = nodeId;

  // The label structure generated by describing the given node.
  this.label = label;

  // The shallow size of this node.
  this.shallowSize = shallowSize;

  // The retained size of this node.
  this.retainedSize = retainedSize;

  // The id of this node's parent or undefined if this node is the root.
  this.parentId = undefined;

  // An array of immediately dominated child `DominatorTreeNode`s, or undefined.
  this.children = undefined;

  // True iff the `children` property does not contain every immediately
  // dominated node.
  //
  // * If children is an array and this property is true: the array does not
  //   contain the complete set of immediately dominated children.
  // * If children is an array and this property is false: the array contains
  //   the complete set of immediately dominated children.
  // * If children is undefined and this property is true: there exist
  //   immediately dominated children for this node, but they have not been
  //   loaded yet.
  // * If children is undefined and this property is false: this node does not
  //   dominate any others and therefore has no children.
  this.moreChildrenAvailable = true;
}

DominatorTreeNode.prototype = null;

module.exports = DominatorTreeNode;

/**
 * Add `child` to the `parent`'s set of children.
 *
 * @param {DominatorTreeNode} parent
 * @param {DominatorTreeNode} child
 */
DominatorTreeNode.addChild = function (parent, child) {
  if (parent.children === undefined) {
    parent.children = [];
  }

  parent.children.push(child);
  child.parentId = parent.nodeId;
};

/**
 * A Visitor that is used to generate a label for a node in the heap snapshot
 * and get its shallow size as well while we are at it.
 */
function LabelAndShallowSizeVisitor() {
  // As we walk the description, we accumulate edges in this array.
  this._labelPieces = [];

  // Once we reach the non-zero count leaf node in the description, we move the
  // labelPieces here to signify that we no longer need to accumulate edges.
  this._label = undefined;

  // Once we reach the non-zero count leaf node in the description, we grab the
  // shallow size and place it here.
  this._shallowSize = 0;
}

DominatorTreeNode.LabelAndShallowSizeVisitor = LabelAndShallowSizeVisitor;

LabelAndShallowSizeVisitor.prototype = Object.create(Visitor);

/**
 * @overrides Visitor.prototype.enter
 */
LabelAndShallowSizeVisitor.prototype.enter = function (breakdown, report, edge) {
  if (this._labelPieces && edge) {
    this._labelPieces.push(edge);
  }
};

/**
 * @overrides Visitor.prototype.exit
 */
LabelAndShallowSizeVisitor.prototype.exit = function (breakdown, report, edge) {
  if (this._labelPieces && edge) {
    this._labelPieces.pop();
  }
};

/**
 * @overrides Visitor.prototype.count
 */
LabelAndShallowSizeVisitor.prototype.count = function (breakdown, report, edge) {
  if (report.count === 0) {
    return;
  }

  this._label = this._labelPieces;
  this._labelPieces = undefined;

  this._shallowSize = report.bytes;
};

/**
 * Get the generated label structure accumulated by this visitor.
 *
 * @returns {Object}
 */
LabelAndShallowSizeVisitor.prototype.label = function () {
  return this._label;
};

/**
 * Get the shallow size of the node this visitor visited.
 *
 * @returns {Number}
 */
LabelAndShallowSizeVisitor.prototype.shallowSize = function () {
  return this._shallowSize;
};

/**
 * Generate a label structure for the node with the given id and grab its
 * shallow size.
 *
 * What is a "label" structure? HeapSnapshot.describeNode essentially takes a
 * census of a single node rather than the whole heap graph. The resulting
 * report has only one count leaf that is non-zero. The label structure is the
 * path in this report from the root to the non-zero count leaf.
 *
 * @param {Number} nodeId
 * @param {HeapSnapshot} snapshot
 * @param {Object} breakdown
 *
 * @returns {Object}
 *          An object with the following properties:
 *          - {Number} shallowSize
 *          - {Object} label
 */
DominatorTreeNode.getLabelAndShallowSize = function (nodeId,
                                                     snapshot,
                                                     breakdown) {
  const description = snapshot.describeNode(breakdown, nodeId);

  const visitor = new LabelAndShallowSizeVisitor();
  walk(breakdown, description, visitor);

  return {
    label: visitor.label(),
    shallowSize: visitor.shallowSize(),
  };
};

/**
 * Do a partial traversal of the given dominator tree and convert it into a tree
 * of `DominatorTreeNode`s. Dominator trees have a node for every node in the
 * snapshot's heap graph, so we must not allocate a JS object for every node. It
 * would be way too many and the node count is effectively unbounded.
 *
 * Go no deeper down the tree than `maxDepth` and only consider at most
 * `maxSiblings` within any single node's children.
 *
 * @param {DominatorTree} dominatorTree
 * @param {Number} maxDepth
 * @param {Number} maxSiblings
 *
 * @returns {DominatorTreeNode}
 */
DominatorTreeNode.partialTraversal = function (dominatorTree,
                                               snapshot,
                                               breakdown,
                                               maxDepth = DEFAULT_MAX_DEPTH,
                                               maxSiblings = DEFAULT_MAX_SIBLINGS) {
  function dfs(nodeId, depth) {
    const { label, shallowSize } =
      DominatorTreeNode.getLabelAndShallowSize(nodeId, snapshot, breakdown);
    const retainedSize = dominatorTree.getRetainedSize(nodeId);
    const node = new DominatorTreeNode(nodeId, label, shallowSize, retainedSize);
    const childNodeIds = dominatorTree.getImmediatelyDominated(nodeId);

    const newDepth = depth + 1;
    if (newDepth < maxDepth) {
      const endIdx = Math.min(childNodeIds.length, maxSiblings);
      for (let i = 0; i < endIdx; i++) {
        DominatorTreeNode.addChild(node, dfs(childNodeIds[i], newDepth));
      }
      node.moreChildrenAvailable = childNodeIds.length < endIdx;
    } else {
      node.moreChildrenAvailable = childNodeIds.length > 0;
    }

    return node;
  }

  return dfs(dominatorTree.root, 0);
};
