#!/usr/bin/env python

import os

import locale
from collections import defaultdict
locale.setlocale(locale.LC_ALL, 'en_US.UTF-8')

header = """
#
#   #####   #######           #     #     #     #     #     #
# ##     #  #     #          # #    #  #  #    # #     #   #
# ##        #     #         #   #   #  #  #   #   #     # #
# ##  ####  #     #        #     #  #  #  #  #     #     #
# ##     #  #     #        #######  #  #  #  #######     #      ###
# ##     #  #     #        #     #  #  #  #  #     #     #      ###
# # #####   #######        #     #   ## ##   #     #     #      ###
#
# Seriously. You shouldn't even be looking at this file unless you're
# debugging generate_mozbuild.py.
#
# DO NOT MODIFY THIS FILE IT IS AUTOGENERATED.
#
"""

footer = """

# We allow warnings for third-party code that can be updated from upstream.
ALLOW_COMPILER_WARNINGS = True

FINAL_LIBRARY = 'gkmedias'
LOCAL_INCLUDES += [
    'skia/include/c',
    'skia/include/config',
    'skia/include/core',
    'skia/include/effects',
    'skia/include/gpu',
    'skia/include/images',
    'skia/include/pathops',
    'skia/include/ports',
    'skia/include/private',
    'skia/include/utils',
    'skia/include/utils/mac',
    'skia/include/utils/win',
    'skia/include/views',
    'skia/src/core',
    'skia/src/gpu',
    'skia/src/gpu/effects',
    'skia/src/gpu/gl',
    'skia/src/image',
    'skia/src/lazy',
    'skia/src/opts',
    'skia/src/sfnt',
    'skia/src/utils',
    'skia/src/utils/mac',
    'skia/src/utils/win',
]

if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('android', 'gonk'):
    DEFINES['SK_FONTHOST_CAIRO_STANDALONE'] = 0

if CONFIG['MOZ_WIDGET_TOOLKIT'] in {
    'android',
    'cocoa',
    'gtk2',
    'gtk3',
    'uikit',
    'gonk',
    'qt',
  }:
    DEFINES['SK_FONTHOST_DOES_NOT_USE_FONTMGR'] = 1

if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'windows':
    DEFINES['UNICODE'] = True
    DEFINES['_UNICODE'] = True

# We should autogenerate these SSE related flags.

if CONFIG['_MSC_VER']:
    # MSVC doesn't need special compiler flags, but Skia needs to be told that these files should
    # be built with the required SSE level or it will simply compile in stubs and cause runtime crashes
    SOURCES['skia/src/opts/SkBitmapFilter_opts_SSE2.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSE2.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSSE3.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=31']
    SOURCES['skia/src/opts/SkBlitRow_opts_SSE2.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkOpts_sse2.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=20']
    SOURCES['skia/src/opts/SkOpts_ssse3.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=31']
    SOURCES['skia/src/opts/SkOpts_sse41.cpp'].flags += ['/arch:SSE2 -DSK_CPU_SSE_LEVEL=41']
if CONFIG['INTEL_ARCHITECTURE'] and CONFIG['GNU_CC']:
    SOURCES['skia/src/opts/SkBitmapFilter_opts_SSE2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSE2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSSE3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkBlitRow_opts_SSE2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkOpts_sse2.cpp'].flags += CONFIG['SSE2_FLAGS']
    SOURCES['skia/src/opts/SkOpts_ssse3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkOpts_sse41.cpp'].flags += ['-msse4.1']
elif CONFIG['CPU_ARCH'] == 'arm' and CONFIG['GNU_CC'] and CONFIG['BUILD_ARM_NEON']:
    DEFINES['SK_ARM_HAS_OPTIONAL_NEON'] = 1
elif CONFIG['CLANG_CL']:
    SOURCES['skia/src/opts/SkBitmapProcState_opts_SSSE3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkOpts_ssse3.cpp'].flags += ['-mssse3']
    SOURCES['skia/src/opts/SkOpts_sse41.cpp'].flags += ['-msse4.1']

if CONFIG['GNU_CXX'] and CONFIG['CPU_ARCH'] == 'arm':
    SOURCES['skia/src/opts/SkBlitRow_opts_arm.cpp'].flags += ['-fomit-frame-pointer']

DEFINES['SKIA_IMPLEMENTATION'] = 1

if not CONFIG['MOZ_ENABLE_SKIA_GPU']:
    DEFINES['SK_SUPPORT_GPU'] = 0

# Suppress warnings in third-party code.
if CONFIG['GNU_CXX'] or CONFIG['CLANG_CL']:
    CXXFLAGS += [
        '-Wno-deprecated-declarations',
        '-Wno-overloaded-virtual',
        '-Wno-sign-compare',
        '-Wno-unused-function',
    ]
if CONFIG['GNU_CXX'] and not CONFIG['CLANG_CXX'] and not CONFIG['CLANG_CL']:
    CXXFLAGS += [
        '-Wno-logical-op',
        '-Wno-maybe-uninitialized',
    ]
if CONFIG['CLANG_CXX'] or CONFIG['CLANG_CL']:
    CXXFLAGS += [
        '-Wno-implicit-fallthrough',
        '-Wno-inconsistent-missing-override',
        '-Wno-macro-redefined',
        '-Wno-unused-private-field',
    ]

if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('gtk2', 'gtk3', 'android', 'gonk', 'qt'):
    CXXFLAGS += CONFIG['MOZ_CAIRO_CFLAGS']
    CXXFLAGS += CONFIG['CAIRO_FT_CFLAGS']

if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('gtk2', 'gtk3', 'qt'):
    CXXFLAGS += CONFIG['MOZ_PANGO_CFLAGS']
"""

import json

platforms = ['linux', 'mac', 'android', 'win']

def generate_opt_sources():
  opt_sources = {'opts': {''}}
  for root, dirs, files in os.walk('skia/src/opts'):
    for name in files:
      if name.endswith('.cpp'):
        opt_sources['opts'].add(os.path.join(root, name))

  return opt_sources

def generate_platform_sources():
  sources = {}

  for plat in platforms:
    if os.system("cd skia && GYP_GENERATORS=dump_mozbuild ./gyp_skia -D OS=%s -D host_os=linux gyp/skia_lib.gyp" % plat) != 0:
      print 'Failed to generate sources for ' + plat
      continue


    f = open('skia/sources.json');
    sources[plat] = set(v.replace('../', 'skia/') for v in json.load(f));
    f.close()

  return dict(sources.items() + generate_opt_sources().items())


def generate_separated_sources(platform_sources):
  blacklist = [
    'experimental',
    'SkXML',
    'GrGLCreateNativeInterface',
    'fontconfig',
    'SkThreadUtils_pthread_',
    'SkFontConfig',
    'SkFontMgr_android',
    'SkFontMgr_custom',
    'SkFontHost_FreeType.cpp',
    'SkForceLinking',
    'Movie',
    'ImageEncoder',
    'skia/src/ports/SkImageGenerator',
    'SkBitmapRegion',
    'codec',
    'SkWGL',
    'SkMemory_malloc',
    'SkOpts_',
    'opts_check_x86',
    'third_party',
  ]

  def isblacklisted(value):
    for item in blacklist:
      if value.find(item) >= 0:
        return True

    return False

  separated = defaultdict(set, {
    'common': {
      'skia/src/gpu/gl/GrGLCreateNativeInterface_none.cpp',
      'skia/src/ports/SkDiscardableMemory_none.cpp',
      'skia/src/ports/SkMemory_mozalloc.cpp',
      'skia/src/ports/SkImageEncoder_none.cpp',
      'skia/src/ports/SkImageGenerator_none.cpp',
    },
    'android': {
      # 'skia/src/ports/SkDebug_android.cpp',
      'skia/src/ports/SkFontHost_cairo.cpp',
      # 'skia/src/ports/SkFontHost_FreeType.cpp',
      # 'skia/src/ports/SkFontHost_FreeType_common.cpp',
      # 'skia/src/ports/SkTime_Unix.cpp',
      # 'skia/src/utils/SkThreadUtils_pthread.cpp',
    },
    'linux': {
      'skia/src/ports/SkFontHost_cairo.cpp',
    },
    'intel': {
      # There is currently no x86-specific opt for SkTextureCompression
      'skia/src/opts/opts_check_x86.cpp',
      'skia/src/opts/SkOpts_sse2.cpp',
      'skia/src/opts/SkOpts_ssse3.cpp',
      'skia/src/opts/SkOpts_sse41.cpp',
    },
    'arm': {
      'skia/src/core/SkUtilsArm.cpp',
    },
    'neon': {
      'skia/src/opts/SkOpts_neon.cpp',
      'skia/src/opts/SkBitmapProcState_arm_neon.cpp',
    },
    'none': set(),
    'gpu': set()
  })

  for plat in platform_sources.keys():
    for value in platform_sources[plat]:
      if isblacklisted(value):
        continue

      if value in separated['common']:
        continue

      key = plat

      if '_SSE' in value or '_SSSE' in value:
        key = 'intel'
      elif '_neon' in value:
        key = 'neon'
      elif '_arm' in value:
        key = 'arm'
      elif '_none' in value:
        key = 'none'
      elif 'gpu' in value or 'Gpu' in value:
        key = 'gpu'
      elif all(value in platform_sources.get(p, {})
               for p in platforms if p != plat):
        key = 'common'

      separated[key].add(value)

  return separated

def uniq(seq):
  seen = set()
  seen_add = seen.add
  return [ x for x in seq if x not in seen and not seen_add(x)]

def write_cflags(f, values, subsearch, cflag, indent):
  def write_indent(indent):
    for _ in range(indent):
        f.write(' ')

  val_list = uniq(sorted(values, key=lambda x: x.lower()))

  if len(val_list) == 0:
    return

  for val in val_list:
    if val.find(subsearch) > 0:
      write_indent(indent)
      f.write("SOURCES[\'" + val + "\'].flags += [\'" + cflag + "\']\n")

def write_sources(f, values, indent):

  # Unfortunately for now the gpu and pathops directories are
  # non-unifiable. Keep track of this and fix it.
  unified_blacklist = [
    '_SSE',
    '_SSSE',
    '_neon',
    'FontHost',
    'SkAdvancedTypefaceMetrics',
    'SkBitmapProcState_matrixProcs.cpp',
    'SkBlitter_A8.cpp',
    'SkBlitter_ARGB32.cpp',
    'SkBlitter_RGB16.cpp',
    'SkBlitter_Sprite.cpp',
    'SkBlitRow_opts_arm.cpp',
    'SkScan_Antihair.cpp',
    'SkParse.cpp',
    'SkPictureData.cpp',
    'SkMatrixConvolutionImageFilter.cpp',
    'opts_check_x86.cpp',
    'GrDrawContext',
    'GrResourceCache',
    'GrAA',
    'GrGL',
    'GrBatchAtlas.cpp',
    'SkArithmeticMode_gpu.cpp',
    'SkImage_Gpu.cpp',
    'SkPathOpsDebug.cpp',
    'SkParsePath.cpp',
    'SkOpts',
    'SkRecorder.cpp',
    'SkXfermode',
  ]

  def isblacklisted(value):
    for item in unified_blacklist:
      if value.find(item) >= 0:
        return True

    return False

  sources = {}
  sources['nonunified'] = set()
  sources['unified'] = set()

  for item in values:
    if isblacklisted(item):
      sources['nonunified'].add(item)
    else:
      sources['unified'].add(item)

  write_list(f, "UNIFIED_SOURCES", sources['unified'], indent)
  write_list(f, "SOURCES", sources['nonunified'], indent)
  
def write_list(f, name, values, indent):
  def write_indent(indent):
    for _ in range(indent):
        f.write(' ')

  val_list = uniq(sorted(values, key=lambda x: x.lower()))

  if len(val_list) == 0:
    return

  write_indent(indent)
  f.write(name + ' += [\n')
  for val in val_list:
    write_indent(indent + 4)
    f.write('\'' + val + '\',\n')

  write_indent(indent)
  f.write(']\n')

def write_mozbuild(sources):
  filename = 'moz.build'
  f = open(filename, 'w')

  f.write(header)

  write_sources(f, sources['common'], 0)

  f.write("if CONFIG['MOZ_ENABLE_SKIA_GPU']:\n")
  write_sources(f, sources['gpu'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] in ('android', 'gonk'):\n")
  write_sources(f, sources['android'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] in {'cocoa', 'uikit'}:\n")
  write_sources(f, sources['mac'], 4)

  f.write("if 'gtk' in CONFIG['MOZ_WIDGET_TOOLKIT']:\n")
  write_sources(f, sources['linux'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'qt':\n")
  write_sources(f, sources['linux'], 4)

  f.write("if CONFIG['MOZ_WIDGET_TOOLKIT'] == 'windows':\n")
  # Windows-specific files don't get unification because of nasty headers.
  # Luckily there are not many files in this.
  write_list(f, "SOURCES", sources['win'], 4)

  f.write("if CONFIG['INTEL_ARCHITECTURE']:\n")
  write_sources(f, sources['intel'], 4)

  f.write("elif CONFIG['CPU_ARCH'] == 'arm' and CONFIG['GNU_CC']:\n")
  write_sources(f, sources['arm'], 4)

  f.write("    if CONFIG['BUILD_ARM_NEON']:\n")
  write_list(f, 'SOURCES', sources['neon'], 8)
  write_cflags(f, sources['neon'], 'neon', '-mfpu=neon', 8)

  f.write("else:\n")
  write_sources(f, sources['none'], 4)

  f.write(footer)

  f.close()

  print 'Wrote ' + filename

def main():
  platform_sources = generate_platform_sources()
  separated_sources = generate_separated_sources(platform_sources)
  write_mozbuild(separated_sources)


if __name__ == '__main__':
  main()
