/*	$KAME$	*/

/*
 * Copyright (C) 1999, 2000 and 2001 WIDE Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 *	Copyright(c) 1999
 *	Keisuke UEHARA(kei@wide.ad.jp)
 *	All rights reserved.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <net/bpf.h>
#include <sys/socket.h>
#include <net/if.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <signal.h>
#include <md5.h>
#ifdef HAVE_NET_ETHERNET_H
#include <net/ethernet.h>
#endif
#ifdef HAVE_NET_IF_ETHER_H
#include <net/if_ether.h>
#endif


#define	NIFB			10
#define	NNODE			32
#define	BUFSIZE			4096
#define	NKEEP			8
#define	TIMEOUT			2
#define	ETHERHDR_SRC_OFFSET	6


struct ifb {
	char ifname[IFNAMSIZ];
	int fd;
	char mac[ETHER_ADDR_LEN];
} ifb[NIFB];


struct nodes {
	int inuse;
	int if_num;
	int timeout;
	unsigned char mac[ETHER_ADDR_LEN];
} nodes[NNODE];

MD5_CTX	md5context;

unsigned char digests[NKEEP][16];
int digests_p = 0;

int debug = 0;

int getmac(char *, char *);


void
usage(int e)
{
	fprintf(stderr, "Usage: bridge [-d] ifname1 ifname2 [...]\n");

	if (e)
		exit(-1);
}


int
bpfopen(void)
{
	int i, fd;
	char bpf[10];

	for (i=0; i<10; i++) {
		snprintf(bpf, sizeof(bpf), "/dev/bpf%d", i);
		fd = open(bpf, O_RDWR);
		if (fd >= 0) return(fd);

		if (errno != EBUSY) {
			perror(bpf);
			return(-1);
		}
	}

	fprintf(stderr, "No more bpfs\n");
	return(-1);
}


int
bpfbind(int fd, char *ifname)
{
	int blen=BUFSIZE, immediate=1, s;
	struct ifreq ifr;
	struct bpf_program prog;
	char mac[ETHER_ADDR_LEN];
	unsigned short h;
	unsigned long l;

	if (ioctl(fd, BIOCSBLEN, &blen) < 0) {
		return(-1);
	}

	bzero(&ifr, sizeof(struct ifreq));
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
	if ((s=socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		return(-1);
	}
	if (ioctl(s, SIOCGIFFLAGS, &ifr) < 0) {
		return(-1);
	}
	if ((ifr.ifr_flags & IFF_UP) == 0) {
		ifr.ifr_flags |= IFF_UP;
		if (ioctl(s, SIOCSIFFLAGS, &ifr) < 0) {
			return(-1);
		}
	}
	close(s);
	if (ioctl(fd, BIOCSETIF, &ifr) < 0) {
		return(-1);
	}

	if (ioctl(fd, BIOCPROMISC, &immediate) < 0) {
		return(-1);
	}

	if (ioctl(fd, BIOCIMMEDIATE, &immediate) < 0) {
		return(-1);
	}

	if (ioctl(fd, BIOCFLUSH) < 0) {
		return(-1);
	}

	return(0);
}


int
bdglist(unsigned char *p)
{
	int i;

	for (i=0; i<NNODE; i++) {
		if (nodes[i].inuse && !bcmp(p, nodes[i].mac, ETHER_ADDR_LEN)) {
			return nodes[i].if_num;
		}
	}
	return -1;
}


void
bdglist_expire(int dummy)
{
	int i;

	for (i=0; i<NNODE; i++) {
		if (nodes[i].inuse && --nodes[i].timeout < 0) {
			nodes[i].inuse = 0;
			if (debug) {
				fprintf(stderr,
					"del [%2d] %02x:%02x:%02x:%02x:%02x:%02x <%d>\n",
					i,
					nodes[i].mac[0], nodes[i].mac[1],
					nodes[i].mac[2], nodes[i].mac[3],
					nodes[i].mac[4], nodes[i].mac[5],
					nodes[i].if_num);
			}
		}
	}
	alarm(60);
}


void
bdglist_add(unsigned char *p, int n)
{
	int i;

	for (i=0; i<NNODE; i++) {
		if (nodes[i].inuse && !bcmp(p, nodes[i].mac, ETHER_ADDR_LEN)) {
			nodes[i].timeout = TIMEOUT;
			return;
		}
	}
	
	for (i=0; i<NNODE; i++) {
		if (!nodes[i].inuse) {
			if (debug) {
				fprintf(stderr,
					"add [%2d] %02x:%02x:%02x:%02x:%02x:%02x <%d>\n",
					i, p[0], p[1], p[2], p[3], p[4], p[5], n);
			}
			nodes[i].inuse = 1;
			nodes[i].timeout = TIMEOUT;
			nodes[i].if_num = n;
			bcopy(p, nodes[i].mac, ETHER_ADDR_LEN);
			return;
		}
	}

	if (debug) {
		fprintf(stderr, "WARNING: No node table entry available\n");
	}
}


int
duppkt(unsigned char *p, int len)
{
	int i;
	unsigned char digest[16];
	
	MD5Init(&md5context);
	MD5Update(&md5context, p, len);
	MD5Final(digest, &md5context);

	for (i=0; i<NKEEP; i++) {
		if (!bcmp(digest, digests[i], 16)) {
			return 1;
		}
	}

	bcopy(digest, digests[digests_p], 16);
	digests_p = (digests_p + 1) % NKEEP;
	return 0;
}


int
main(int argc, char *argv[])
{
	int i, j, fd, n=0, nfds=0, c, ifn;
	int len;
	fd_set fds;
	fd_set tfds;
	static unsigned char *buf, *p, macsave[ETHER_ADDR_LEN];
	struct bpf_hdr *bh;

	for (i=1; i<argc; i++) {
		if (!strncmp(argv[i], "-d", strlen("-d"))) {
			debug = 1;
		} else {
			if (n >= NIFB) {
				fprintf(stderr, "Too many interfaces\n");
				exit(0);
			}

			if ((fd = bpfopen()) < 0) {
				exit(0);
			}

			if (bpfbind(fd, argv[i]) < 0) {
				perror(argv[i]);
				exit(-1);
			}

			if (getmac(argv[i], ifb[n].mac) >= 0) {
				bzero(ifb[n].mac, ETHER_ADDR_LEN);
			}

			strncpy(ifb[n].ifname, argv[i], IFNAMSIZ);
			ifb[n].fd = fd;
			++n;
		}
	}

	if (n < 2) {
		usage(1);
	}


	for (i=0; i<NNODE; i++) {
		nodes[i].inuse = 0;
	}
	bzero(digests, sizeof(digests));

	buf = malloc(BUFSIZE);

	nfds = getdtablesize();
	FD_ZERO(&fds);
	for (i=0; i<n; i++) {
		if (debug) {
			fprintf(stderr, "interface %d: %s\n",
					i, ifb[i].ifname, ifb[i].fd);
		}
		FD_SET(ifb[i].fd, &fds);
		if (ioctl(ifb[i].fd, BIOCFLUSH) < 0) {
			perror("BIOCFLUSH");
			exit(-1);
		}
	}

	signal(SIGALRM, bdglist_expire);
	alarm(60);

	for (;;) {
		bcopy((char *) &fds, (char *) &tfds, sizeof(fd_set));
	
		if (select(nfds, &tfds, NULL, NULL, NULL) < 0) {
			if (errno == EINTR) continue;
			perror("select");
			exit(-1);
		}

		for (i=0; i<n; i++) {
			if (!FD_ISSET(ifb[i].fd, &tfds)) continue;

			if ((c=read(ifb[i].fd, buf, BUFSIZE)) < 0) {
				perror("read");
				exit(-1);
			}
			bh = (struct bpf_hdr *) buf;
			p = ((char *) bh) + bh->bh_hdrlen;
			len = (bh->bh_caplen);
			if (len > 1514) len = 1514;

			bdglist_add(&p[ETHERHDR_SRC_OFFSET], i);
			bcopy(&(p[ETHERHDR_SRC_OFFSET]), macsave,
							ETHER_ADDR_LEN);
			bzero(&(p[ETHERHDR_SRC_OFFSET]), ETHER_ADDR_LEN);
			if (duppkt(p, len)) continue;
			bcopy(macsave, &(p[ETHERHDR_SRC_OFFSET]),
							ETHER_ADDR_LEN);
			ifn = bdglist(p);

			for (j=0; j<n; j++) {
				if (j==i) continue;
				if (ifn >= 0 && ifn != j) continue;
				if (write(ifb[j].fd, p, len) < 0) {
					if (debug) {
						perror("write");
					}
				}
			}
		}
	}
}
