use forgejo_api::structs::{IssueListLabelsQuery, Label};

use crate::actions::GeneralArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;

use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::types::output::OutputMode;

use clap::Parser;

/// List all labels in the current repository
#[derive(Parser, Debug)]
pub struct ListLabelsArgs {
    /// Number of labels to be displayed
    #[arg(short, long, value_name = "N", default_value_t = 5)]
    pub count: usize,
}

impl ListLabelsArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;
        let (_, labels_list) = spin_until_ready(ctx.client.issue_list_labels(
            owner.as_str(),
            repo.as_str(),
            IssueListLabelsQuery::default(),
        ))
        .await?;

        match general_args.output_mode {
            OutputMode::Pretty => {
                present_labels_list(&ctx, labels_list);
            }
            OutputMode::Json => labels_list.print_json()?,
        }

        Ok(())
    }
}

fn present_labels_list(ctx: &BergContext<ListLabelsArgs>, labels: Vec<Label>) {
    let labels_empty = labels.is_empty();

    let mut table = ctx.make_table();

    table
        .set_header(vec![format!(
            "Labels{}",
            if labels_empty {
                " (empty)"
            } else {
                Default::default()
            }
        )])
        .add_rows(
            labels
                .into_iter()
                .map(|label| vec![option_display(&label.name)]),
        );

    println!("{table}", table = table.show());
}
