use crate::actions::GeneralArgs;
use crate::actions::text_manipulation::{input_prompt_for, select_prompt_for};
use crate::render::json::JsonToStdout;
use crate::render::ui::{fuzzy_select_with_key, multi_fuzzy_select_with_key};
use crate::types::api::privacy_type::Privacy;
use crate::types::context::BergContext;
use crate::types::output::OutputMode;
use clap::Parser;
use forgejo_api::structs::CreateRepoOption;
use strum::{Display, VariantArray};

/// Create a new repository
#[derive(Parser, Debug)]
pub struct RepoCreateArgs {
    /// Main branch to init repository with (usually "main")
    #[arg(id = "default-branch", long)]
    pub default_branch: Option<String>,

    /// Repository description
    #[arg(short, long)]
    pub description: Option<String>,

    /// Organization name
    #[arg(long)]
    pub org: Option<String>,

    /// Repository name
    #[arg(short, long)]
    pub name: Option<String>,

    /// Repository visibility
    #[arg(short, long, value_enum, value_name = "VISIBILITY")]
    pub private: Option<Privacy>,
}

#[derive(Display, PartialEq, Eq, VariantArray)]
enum CreatableFields {
    DefaultBranch,
    Description,
    Private,
}

impl RepoCreateArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;

        let options = create_options(&ctx).await?;
        let pull_request = if let Some(org) = ctx.args.org.as_ref() {
            ctx.client.create_org_repo(org, options).await
        } else {
            ctx.client.create_current_user_repo(options).await
        }?;
        match general_args.output_mode {
            OutputMode::Pretty => {
                tracing::debug!("{pull_request:?}");
            }
            OutputMode::Json => pull_request.print_json()?,
        }
        Ok(())
    }
}

async fn create_options(ctx: &BergContext<RepoCreateArgs>) -> anyhow::Result<CreateRepoOption> {
    let name = match ctx.args.name.as_ref() {
        Some(name) => name.clone(),
        None => inquire::Text::new(input_prompt_for("New Repository Name").as_str()).prompt()?,
    };

    let mut options = CreateRepoOption {
        name,
        auto_init: None,
        default_branch: ctx.args.default_branch.clone(),
        description: ctx.args.description.clone(),
        gitignores: None,
        issue_labels: None,
        license: None,
        private: ctx.args.private.map(|p| match p {
            Privacy::Private => true,
            Privacy::Public => false,
        }),
        readme: None,
        template: None,
        trust_model: None,
        object_format_name: None,
    };

    let optional_data = {
        use CreatableFields::*;
        [
            (DefaultBranch, ctx.args.default_branch.is_none()),
            (Description, ctx.args.description.is_none()),
            (Private, ctx.args.private.is_none()),
        ]
        .into_iter()
        .filter_map(|(name, missing)| missing.then_some(name))
        .collect::<Vec<_>>()
    };

    if !optional_data.is_empty() && !ctx.general_args.non_interactive {
        let chosen_optionals = multi_fuzzy_select_with_key(
            &optional_data,
            "Choose optional properties",
            |_| false,
            |o| o.to_string(),
        )?;

        {
            use CreatableFields::*;
            options.default_branch.replace(repo_default_branch(
                ctx,
                chosen_optionals.contains(&&DefaultBranch),
            )?);
            options.private = repo_private(ctx, chosen_optionals.contains(&&Private)).await?;
            options.description =
                repo_description(ctx, chosen_optionals.contains(&&Description)).await?;
        }
    }

    Ok(options)
}

fn repo_default_branch(
    ctx: &BergContext<RepoCreateArgs>,
    interactive: bool,
) -> anyhow::Result<String> {
    let branch = match ctx.args.default_branch.as_ref() {
        Some(branch) => branch.clone(),
        None => {
            if !interactive {
                return Ok(String::from("main"));
            }
            inquire::Text::new(input_prompt_for("Default Branch Name").as_str()).prompt()?
        }
    };
    Ok(branch)
}

async fn repo_private(
    ctx: &BergContext<RepoCreateArgs>,
    interactive: bool,
) -> anyhow::Result<Option<bool>> {
    let privacy = match ctx.args.private {
        Some(privacy) => match privacy {
            Privacy::Private => true,
            Privacy::Public => false,
        },
        None => {
            if !interactive {
                return Ok(None);
            }
            fuzzy_select_with_key(
                &[true, false],
                select_prompt_for("repo privacy"),
                |private| {
                    if *private {
                        String::from("Private")
                    } else {
                        String::from("Public")
                    }
                },
            )
            .copied()?
        }
    };
    Ok(Some(privacy))
}

async fn repo_description(
    ctx: &BergContext<RepoCreateArgs>,
    interactive: bool,
) -> anyhow::Result<Option<String>> {
    let description = match ctx.args.description.as_ref() {
        Some(desc) => desc.clone(),
        None => {
            if !interactive {
                return Ok(None);
            }
            ctx.editor_for("a description", "Enter Repository description")?
        }
    };
    Ok(Some(description))
}
