/*
 * Unit tests for API functions of libssiconv
 *
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

/* Check for XSI extension on POSIX systems older than POSIX.1-2001 */
#if (defined _POSIX_C_SOURCE) && (_POSIX_C_SOURCE < 200112L)
#include <unistd.h>
#if _XOPEN_VERSION >= 500
#define _XOPEN_SOURCE  _XOPEN_VERSION
#endif  /* _XOPEN_VERSION >= 500 */
#endif  /* (defined _POSIX_C_SOURCE) && (_POSIX_C_SOURCE < 200112L) */

/* System headers */
#include <errno.h>
#include <stdio.h>
#include <stddef.h>
#include <string.h>

/* libssiconv */
#include "libssiconv-0/iconv.h"
#include "libssiconv-0/iconv_table_ni.h"

/* CHEAT unit test framework */
#ifndef __BASE_FILE__
#define __BASE_FILE__  SSIC0_UTFILE
#endif  /* __BASE_FILE__ */
#include <cheat.h>


CHEAT_DECLARE(
    static size_t rv CHEAT_COMMA inlen CHEAT_COMMA outlen;

    /* Input buffer */
    static char inbuf[256];

    /* Output buffer (four times larger, sufficient for UTF-8) */
    static char outbuf[1024];

    /* Fill input buffer with octet values representing the index */
    void PREPARE_INBUF(void)
    {
        size_t i;

        for (i = 0; 256 > i; ++i)
            inbuf[i] = i;
    }
)


/* Check return value and print debug output after error */
CHEAT_DECLARE(
    void CHECK(size_t rv, size_t expected)
    {
        if (expected != rv)
        {
            if ((size_t)-1 == rv)
                fprintf(stderr, "rv: -1, ", (unsigned long int)rv);
            else
                fprintf(stderr, "rv: %lu, ", (unsigned long int)rv);

            fprintf(stderr, "inlen: %lu, outlen: %lu\n",
                    (unsigned long int)inlen, (unsigned long int)outlen);

            /* Print human readable string for errno */
            perror("");
        }

        cheat_assert(expected == rv);
    }
)


/* Compare conversion output with reference file */
CHEAT_DECLARE(
    void COMPARE_WITH_REFERENCE(const char *buf, size_t len, const char* pn)
    {
        FILE *fp = fopen(pn, "rb");

        cheat_assert(NULL != fp);
        if (NULL != fp)
        {
            int    rv = EOF;
            size_t i  = 0;

            /* Compare with reference data */
            for ( ; len > i; ++i)
            {
                unsigned int chk = (unsigned char)buf[i];

                rv = fgetc(fp);
                cheat_assert(EOF != rv);
                {
                    unsigned int ref = (unsigned char)rv;

                    if (chk != (unsigned char)rv)
                    {

                        fprintf( stderr, "Mismatch at index %u (0x%02X): "
                                 "Checked 0x%02X against Reference 0x%02X\n",
                                 (unsigned int) i, (unsigned int) i, chk, ref );
                    }
                    cheat_assert(chk == ref);
                    cheat_yield();
                }
            }

            /* Check for EOF in reference data file */
            rv = fgetc(fp);
            cheat_assert(EOF == rv);

            rv = fclose(fp);
            cheat_assert(0U == rv);
        }
    }
)


/* Print CHEAT version */
CHEAT_TEST(version,
    (void)cheat_print(stdout, "%s", 1, "Unit test framework: ");
    cheat_print_version();
    /* Yellow foreground color is unreadable with white background */
    (void)cheat_print(stdout,
                      "%s", 1, "Yellow foreground color patched out.\n\n");
)


CHEAT_TEST(empty_name,
    size_t rv;

    (void)fputs("Testing ssic0_iconvstr() with empty name ... \n", stdout);

    inbuf[0] = 0x20;
    inlen    = 1;
    outlen   = 1024;
    rv = ssic0_iconvstr("", "Us-AscII", inbuf, &inlen, outbuf, &outlen, 0);
    CHECK(rv, (size_t)-1);
    cheat_assert(EBADF == errno);
)


CHEAT_TEST(invalid_name,
    (void)fputs("Testing ssic0_iconvstr() with invalid name ... \n", stdout);

    inbuf[0] = 0x20;
    inlen    = 1;
    outlen   = 1024;
    rv = ssic0_iconvstr("INVAL", "US-ASCII", inbuf, &inlen, outbuf, &outlen, 0);
    CHECK(rv, (size_t)-1);
    cheat_assert(EBADF == errno);
)


CHEAT_TEST(buffer_size,
    (void)fputs("Testing ssic0_iconvstr() "
                "with insufficient output buffer ... \n", stdout);

    inbuf[0] = 0x41;
    inbuf[1] = 0x42;
    inbuf[2] = 0x43;
    inlen    = 3;
    outlen   = 2;
    rv = ssic0_iconvstr("uS-aSCII", "Us-AscII", inbuf, &inlen, outbuf, &outlen,
                        0);
    CHECK(rv, (size_t)-1);
    cheat_assert(E2BIG == errno);
)


CHEAT_TEST(control_nul,
    (void)fputs("Testing ssic0_iconvstr() with NUL control character ... \n",
                  stdout);

    inbuf[0] = 0x20;
    inbuf[1] = 0x00;
    inbuf[2] = 0x20;
    inlen    = 3;
    outlen   = 1024;
    rv = ssic0_iconvstr("uS-aSCII", "Us-AscII", inbuf, &inlen, outbuf, &outlen,
                        0);
    CHECK(rv, 0);
    cheat_assert(2U == inlen);
)


CHEAT_TEST(non_ident,
    (void)fputs("Testing ssic0_iconvstr() with non-identical conversion ... \n",
                stdout);

    inbuf[0] = 0x80;
    inlen    = 1;
    outlen   = 1024;
    rv = ssic0_iconvstr("UtF-8", "Us-AscII", inbuf, &inlen, outbuf, &outlen, 0);
    CHECK(rv, (size_t)-1);
    cheat_yield();

    inlen    = 1;
    outlen   = 1024;
    rv = ssic0_iconvstr("UtF-8", "Us-AscII", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 1);
)


CHEAT_TEST(us_ascii,
    size_t i;

    (void)fputs("Testing ssic0_iconvstr() with codepage US-ASCII ... \n",
                stdout);

    for (i = 0; 128 > i; ++i)
        inbuf[i] = i;
    inlen  = 128;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Us-AscII", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_assert((1024 - 128) == outlen);
    cheat_yield();
    for (i = 0; 128 > i; ++i)
        cheat_assert(inbuf[i] == outbuf[i]);
    cheat_yield();

    /* (Nonexistent) upper half of 8-bit codepage is mapped to replacement */
    for (i = 0; 256 > i; ++i)
        inbuf[i] = i;
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UTF-8", "US-ASCII", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 128);
    cheat_assert(0U == inlen);
    cheat_assert(128U + (128U * strlen(SSIC0_I_NI)) == outlen);
    cheat_yield();
    for (i = 0; 128 > i; ++i)
        cheat_assert(inbuf[i] == outbuf[i]);
)


CHEAT_TEST(iso_8859_1,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-1 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-1", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-1.utf8");
)


CHEAT_TEST(iso_8859_2,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-2 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-2", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-2.utf8");
)


CHEAT_TEST(iso_8859_3,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-3 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-3", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 7);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-3.utf8");
)


CHEAT_TEST(iso_8859_4,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-4 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-4", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-4.utf8");
)


CHEAT_TEST(iso_8859_5,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-5 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-5", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-5.utf8");
)


CHEAT_TEST(iso_8859_6,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-6 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-6", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 45);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-6.utf8");
)


CHEAT_TEST(iso_8859_7,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-7 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-7", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 3);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-7.utf8");
)


CHEAT_TEST(iso_8859_8,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-7 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-8", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 36);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-8.utf8");
)


CHEAT_TEST(iso_8859_9,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-9 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-9", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-9.utf8");
)


CHEAT_TEST(iso_8859_10,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-10 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-10", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-10.utf8");
)


CHEAT_TEST(iso_8859_11,
    (void)fputs("Testing ssic0_iconvstr() with codepage TIS-620 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "TIS-620", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 8);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-11.utf8");
)


CHEAT_TEST(iso_8859_13,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-13 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-13", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-13.utf8");
)


CHEAT_TEST(iso_8859_14,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-14 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-14", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-14.utf8");
)


CHEAT_TEST(iso_8859_15,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-14 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-14", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-14.utf8");
)


CHEAT_TEST(iso_8859_16,
    (void)fputs("Testing ssic0_iconvstr() with codepage ISO-8859-16 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IsO-8859-16", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/ISO-8859-16.utf8");
)


CHEAT_TEST(koi8_r,
    (void)fputs("Testing ssic0_iconvstr() with codepage KOI8-R ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "KoI8-r", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/KOI8-R.utf8");
)


CHEAT_TEST(koi8_u,
    (void)fputs("Testing ssic0_iconvstr() with codepage KOI8-U ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("uTf-8", "KoI8-U", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/KOI8-U.utf8");
)


CHEAT_TEST(windows_1250,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1250 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1250", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 5);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1250.utf8");
)


CHEAT_TEST(windows_1251,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1251 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1251", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 1);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1251.utf8");
)


CHEAT_TEST(windows_1252,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1252 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1252", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 5);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1252.utf8");
)


CHEAT_TEST(windows_1253,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1253 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1253", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 17);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1253.utf8");
)


CHEAT_TEST(windows_1254,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1254 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1254", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 7);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1254.utf8");
)


CHEAT_TEST(windows_1255,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1255 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1255", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 22);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1255.utf8");
)


CHEAT_TEST(windows_1256,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1256 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1256", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1256.utf8");
)


CHEAT_TEST(windows_1257,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1257 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1257", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 12);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1257.utf8");
)


CHEAT_TEST(windows_1258,
    (void)fputs("Testing ssic0_iconvstr() with codepage Windows-1258 ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Windows-1258", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 9);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/WINDOWS-1258.utf8");
)


CHEAT_TEST(macintosh,
    (void)fputs("Testing ssic0_iconvstr() with codepage Macintosh ... \n",
                stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "Macintosh", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL | SSIC0_ICONV_REPLACE_INVALID);
    CHECK(rv, 1);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/MACINTOSH.utf8");
)


CHEAT_TEST(ibm_437,
    (void)fputs("Testing ssic0_iconvstr() with codepage IBM437 ... \n", stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IBM437", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/IBM437.utf8");
)


CHEAT_TEST(ibm_775,
    (void)fputs("Testing ssic0_iconvstr() with codepage IBM775 ... \n", stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IBM775", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0 == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/IBM775.utf8");
)


CHEAT_TEST(ibm_850,
    (void)fputs("Testing ssic0_iconvstr() with codepage IBM850 ... \n", stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IBM850", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/IBM850.utf8");
)


CHEAT_TEST(ibm_852,
    (void)fputs("Testing ssic0_iconvstr() with codepage IBM852 ... \n", stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IBM852", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/IBM852.utf8");
)


CHEAT_TEST(ibm_858,
    (void)fputs("Testing ssic0_iconvstr() with codepage IBM858 ... \n", stdout);

    PREPARE_INBUF();
    inlen  = 256;
    outlen = 1024;
    rv = ssic0_iconvstr("UtF-8", "IBM00858", inbuf, &inlen, outbuf, &outlen,
                        SSIC0_ICONV_IGNORE_NULL);
    CHECK(rv, 0);
    cheat_assert(0U == inlen);
    cheat_yield();
    COMPARE_WITH_REFERENCE(outbuf, 1024U - outlen, "reference/IBM00858.utf8");
)
