package search_test

import (
	"testing"

	"github.com/vulncheck-oss/go-exploit/search"
)

var htmlTestOIDC = `<html><head><meta http-equiv='X-UA-Compatible' content='IE=edge' /><base target='_self'/></head><html><form method='post' action='https://xp0.sc/identity/signin'><input type='hidden' name='code' value='23431FCD6CA15658D8267B3A8013D2F013AA32CF38B20E59EA9C1529DFAF44FD-1' />
<input type='hidden' name='id_token' value='eyJhbGciOiJSUzI1NiIsImtpZCI6IjI4OEI4MEQ5RDMzRDZDNkY2MDgzMjY2MENCMzdEREJCRDdGNDFFMjVSUzI1NiIsIng1dCI6IktJdUEyZE05Ykc5Z2d5Wmd5emZkdTlmMEhpVSIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJodHRwczovL3hwMC5pZGVudGl0eXNlcnZlciIsIm5iZiI6MTc1MDcxOTYzMywiaWF0IjoxNzUwNzE5NjMzLCJleHAiOjE3NTA3MjExMzMsImF1ZCI6IlNpdGVjb3JlIiwiYW1yIjpbInB3ZCJdLCJub25jZSI6IjYzODg2MzE0ODM4NzM0MDYwMy5ZVFl3TnpFMk5UZ3RNR1V3WXkwMFl6TXpMVGcwTURZdE1tVXdZVFJqT1dJNFlUVTNZbVZoWkdNM05qSXRNamRrWVMwMFlUWmlMV0V3TXprdE1qYzJNRFU0T1dObU1UZzEiLCJhdF9oYXNoIjoiTlFnT0xERm5EQWZCYXJ3RS1hQzIzQSIsImNfaGFzaCI6IlZRdVhKZ1hsdE8zUlZYWXFWZnJ4V1EiLCJzaWQiOiJFOTU2RjYyNEFEMUNDMjVGQUY1MEM4NjNDNEEzQ0ZBQSIsInN1YiI6IjQ4MjZkOGY3NTZkNDQ1ZGM5YTE5MmE5MzVhYjA4MjQ4IiwiYXV0aF90aW1lIjoxNzUwNzE5NDkwLCJpZHAiOiJsb2NhbCJ9.QJTHf8jiXVkVY95NS_pV2dZcbFuohQyQNEIU7MM4VPxFNTN7toAoOGLCMuncGJHfHu2IA0lnj0OMK11eS-4ONiJ4Qzq08M2hP-5kv0XFiSgJeoAG-8AEEzTdO6Ag_IznEpStKmGxoq7ojrDUZrsgg5e7FSxnJiFtWinvUJrGvrjQ0XIvMMMTLgxlXlSXf3dy6t93Kge8CI3tVUKqfQ_EfTxf7CJ2dm2vcDgRxDj0qc7edIuSq9_w55Aj6o0mToNfyqlhx3q8emSfQSutGj4Hp3zHYWeoD6OhrHhn5lB1OJ4Jq8zvh_SUC5pL1mCpTL7C2crlvywQBTYCv4smaxLz1Q' />
<input type='hidden' name='access_token' value='eyJhbGciOiJSUzI1NiIsImtpZCI6IjI4OEI4MEQ5RDMzRDZDNkY2MDgzMjY2MENCMzdEREJCRDdGNDFFMjVSUzI1NiIsIng1dCI6IktJdUEyZE05Ykc5Z2d5Wmd5emZkdTlmMEhpVSIsInR5cCI6ImF0K2p3dCJ9.eyJpc3MiOiJodHRwczovL3hwMC5pZGVudGl0eXNlcnZlciIsIm5iZiI6MTc1MDcxOTYzMywiaWF0IjoxNzUwNzE5NjMzLCJleHAiOjE3NTA3MjMyMzMsImF1ZCI6Imh0dHBzOi8veHAwLmlkZW50aXR5c2VydmVyL3Jlc291cmNlcyIsInNjb3BlIjpbIm9wZW5pZCIsInNpdGVjb3JlLnByb2ZpbGUiXSwiYW1yIjpbInB3ZCJdLCJjbGllbnRfaWQiOiJTaXRlY29yZSIsInN1YiI6IjQ4MjZkOGY3NTZkNDQ1ZGM5YTE5MmE5MzVhYjA4MjQ4IiwiYXV0aF90aW1lIjoxNzUwNzE5NDkwLCJpZHAiOiJsb2NhbCIsInNpZCI6IkU5NTZGNjI0QUQxQ0MyNUZBRjUwQzg2M0M0QTNDRkFBIiwianRpIjoiMjcxMEY1MDRBMTZBMjY1OUVDNUQzMDhDQ0RFQUM2RDgifQ.KRYVImxWFfCG482guoXBi86EuirC6g4HuqZP4mJrug0Z7fTgnXL6RuDkJ-AwR3ok9o5kDI71y5Eo7IVx50VQnhvsgeelHIF_XN1_oOrPg3wB5Aj7VWSiimHEAb1Nf5iMDzZJVMeyiRKcv-AYizR7b9dpePoQNb6xiRHClELWK5_gS1sLh28matOhvnB9aYte2ycdUxMbcwi8TaKPtrvFitp4LSmQbJXDfAAV3KId2OwJ8t6Y3LN8PxPMMjG1y1wl3fI1o-y09X9mQ-9UPnNTViyPMy9Q-TP9GzirEro6TlK2i0lkeuaFldsfVT0I-xGCrECKT0yXF4YkYESG5pY2sg' />
<input type='hidden' name='token_type' value='Bearer' />
<input type='hidden' name='expires_in' value='3600' />
<input type='hidden' name='scope' value='openid sitecore.profile' />
<input type='hidden' name='state' value='OpenIdConnect.AuthenticationProperties=POHfd7aul1EaHTEMbw6KaGMlXt3HbsjWv5zXk0cC6JXcLuCVeyp-dN8jqrWF7GM976vk64kCVEQ1hitngF-m6_qOnywZuEwq67v2Li3WLXcZA4uH8CpUAF5xhYAVKU0E0tx6Wtd6gbrT_s-oQc8-RgfZuE7uTSzS7lyjA4P3uBCIOt0kbQ3IRFBeYVUPDe2RyOzAPtghNVdFsPGXgv2SNoZj4rSX27uvCkcalg0tuRI' />
<input type='hidden' name='session_state' value='OH1VzekymvxjsmB3e2EckAKOYVApIi5E-5KeXY4gc0U.6D0FC8F5AC27A7E35229745BB9FE1108' />
<noscript><button>Click to continue</button></noscript></form><script>window.addEventListener('load', function(){document.forms[0].submit();});</script></html>`

func TestCheckSemVer_Full(t *testing.T) {
	if !search.CheckSemVer("1.0.0", "<= 1.0.0") {
		t.Error("Constraint should have passed")
	}
	if search.CheckSemVer("1.0.0", "> 1.0.0") {
		t.Error("Constraint should not have passed")
	}
}

func TestCheckSemVer_BadVersion(t *testing.T) {
	if search.CheckSemVer("uwu", "<= 1.0.0") {
		t.Error("Version was invalid, should not have passed")
	}
	if search.CheckSemVer("1.0.0 ", "<= 1.0.0") {
		t.Error("Version was invalid, should not have passed")
	}
}

func TestCheckSemVer_BadConstraint(t *testing.T) {
	if search.CheckSemVer("1.0.0", "<== 1.0.0") {
		t.Error("Constraint was invalid, should not have passed")
	}
	if search.CheckSemVer("1.0.0", "xp") {
		t.Error("Constraint was invalid, should not have passed")
	}
}

func TestXPath_Node(t *testing.T) {
	c, ok := search.XPath(htmlTestOIDC, `//script`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if c != `window.addEventListener('load', function(){document.forms[0].submit();});` {
		t.Error("XPath node value did not match")
	}
}

func TestXPath_NodeMultiple(t *testing.T) {
	c, ok := search.XPath(htmlTestOIDC, `//input/@value`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if c != `23431FCD6CA15658D8267B3A8013D2F013AA32CF38B20E59EA9C1529DFAF44FD-1` {
		t.Error("XPath node value did not match")
	}
}

func TestXPathAll_NodeMultiple(t *testing.T) {
	c, ok := search.XPathAll(htmlTestOIDC, `//input/@value`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if len(c) != 8 {
		t.Error("Unexpected amount of matched nodes")
	}
}

func TestXPath_Attributes(t *testing.T) {
	var c string
	var ok bool
	c, ok = search.XPath(htmlTestOIDC, `//input[@name="code"]/@value`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if c != `23431FCD6CA15658D8267B3A8013D2F013AA32CF38B20E59EA9C1529DFAF44FD-1` {
		t.Error("XPath `code` did not match")
	}
	c, ok = search.XPath(htmlTestOIDC, `//input[@name="id_token"]/@value`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if c != `eyJhbGciOiJSUzI1NiIsImtpZCI6IjI4OEI4MEQ5RDMzRDZDNkY2MDgzMjY2MENCMzdEREJCRDdGNDFFMjVSUzI1NiIsIng1dCI6IktJdUEyZE05Ykc5Z2d5Wmd5emZkdTlmMEhpVSIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJodHRwczovL3hwMC5pZGVudGl0eXNlcnZlciIsIm5iZiI6MTc1MDcxOTYzMywiaWF0IjoxNzUwNzE5NjMzLCJleHAiOjE3NTA3MjExMzMsImF1ZCI6IlNpdGVjb3JlIiwiYW1yIjpbInB3ZCJdLCJub25jZSI6IjYzODg2MzE0ODM4NzM0MDYwMy5ZVFl3TnpFMk5UZ3RNR1V3WXkwMFl6TXpMVGcwTURZdE1tVXdZVFJqT1dJNFlUVTNZbVZoWkdNM05qSXRNamRrWVMwMFlUWmlMV0V3TXprdE1qYzJNRFU0T1dObU1UZzEiLCJhdF9oYXNoIjoiTlFnT0xERm5EQWZCYXJ3RS1hQzIzQSIsImNfaGFzaCI6IlZRdVhKZ1hsdE8zUlZYWXFWZnJ4V1EiLCJzaWQiOiJFOTU2RjYyNEFEMUNDMjVGQUY1MEM4NjNDNEEzQ0ZBQSIsInN1YiI6IjQ4MjZkOGY3NTZkNDQ1ZGM5YTE5MmE5MzVhYjA4MjQ4IiwiYXV0aF90aW1lIjoxNzUwNzE5NDkwLCJpZHAiOiJsb2NhbCJ9.QJTHf8jiXVkVY95NS_pV2dZcbFuohQyQNEIU7MM4VPxFNTN7toAoOGLCMuncGJHfHu2IA0lnj0OMK11eS-4ONiJ4Qzq08M2hP-5kv0XFiSgJeoAG-8AEEzTdO6Ag_IznEpStKmGxoq7ojrDUZrsgg5e7FSxnJiFtWinvUJrGvrjQ0XIvMMMTLgxlXlSXf3dy6t93Kge8CI3tVUKqfQ_EfTxf7CJ2dm2vcDgRxDj0qc7edIuSq9_w55Aj6o0mToNfyqlhx3q8emSfQSutGj4Hp3zHYWeoD6OhrHhn5lB1OJ4Jq8zvh_SUC5pL1mCpTL7C2crlvywQBTYCv4smaxLz1Q` {
		t.Error("XPath `id_token` did not match")
	}
	c, ok = search.XPath(htmlTestOIDC, `//input[@name="access_token"]/@value`)
	if !ok {
		t.Error("Could not find HTML attribute")
	}
	if c != `eyJhbGciOiJSUzI1NiIsImtpZCI6IjI4OEI4MEQ5RDMzRDZDNkY2MDgzMjY2MENCMzdEREJCRDdGNDFFMjVSUzI1NiIsIng1dCI6IktJdUEyZE05Ykc5Z2d5Wmd5emZkdTlmMEhpVSIsInR5cCI6ImF0K2p3dCJ9.eyJpc3MiOiJodHRwczovL3hwMC5pZGVudGl0eXNlcnZlciIsIm5iZiI6MTc1MDcxOTYzMywiaWF0IjoxNzUwNzE5NjMzLCJleHAiOjE3NTA3MjMyMzMsImF1ZCI6Imh0dHBzOi8veHAwLmlkZW50aXR5c2VydmVyL3Jlc291cmNlcyIsInNjb3BlIjpbIm9wZW5pZCIsInNpdGVjb3JlLnByb2ZpbGUiXSwiYW1yIjpbInB3ZCJdLCJjbGllbnRfaWQiOiJTaXRlY29yZSIsInN1YiI6IjQ4MjZkOGY3NTZkNDQ1ZGM5YTE5MmE5MzVhYjA4MjQ4IiwiYXV0aF90aW1lIjoxNzUwNzE5NDkwLCJpZHAiOiJsb2NhbCIsInNpZCI6IkU5NTZGNjI0QUQxQ0MyNUZBRjUwQzg2M0M0QTNDRkFBIiwianRpIjoiMjcxMEY1MDRBMTZBMjY1OUVDNUQzMDhDQ0RFQUM2RDgifQ.KRYVImxWFfCG482guoXBi86EuirC6g4HuqZP4mJrug0Z7fTgnXL6RuDkJ-AwR3ok9o5kDI71y5Eo7IVx50VQnhvsgeelHIF_XN1_oOrPg3wB5Aj7VWSiimHEAb1Nf5iMDzZJVMeyiRKcv-AYizR7b9dpePoQNb6xiRHClELWK5_gS1sLh28matOhvnB9aYte2ycdUxMbcwi8TaKPtrvFitp4LSmQbJXDfAAV3KId2OwJ8t6Y3LN8PxPMMjG1y1wl3fI1o-y09X9mQ-9UPnNTViyPMy9Q-TP9GzirEro6TlK2i0lkeuaFldsfVT0I-xGCrECKT0yXF4YkYESG5pY2sg` {
		t.Error("XPath `access_token` did not match")
	}
}
